#!/usr/bin/env perl

use strict;
use warnings;
use Test::More;
use Test::Exception;
use FindBin;
use lib "$FindBin::Bin/../lib";

# 确保模块可以加载
use_ok('PDK::Config::Context::Static');

# 测试编码检测和解码
subtest '编码检测和解码' => sub {
    # 测试包含中文字符的配置（UTF-8编码）
    my @utf8_config = (
        "interface GigabitEthernet0/0",
        " description test description",
        " ip address 192.168.1.1 255.255.255.0",
        " no shutdown"
    );
    
    my $config_obj = PDK::Config::Context::Static->new(
        id     => 1,
        name   => 'test-firewall',
        vendor => 'test',
        config => \@utf8_config
    );
    
    # 测试逐行读取
    my @lines = ();
    while (my $line = $config_obj->nextUnParsedLine) {
        push @lines, $line;
    }
    
    # 验证内容正确读取
    is(scalar(@lines), 4, '正确读取了所有4行配置');
    is($lines[0], "interface GigabitEthernet0/0", '第一行正确');
    is($lines[1], " description test description", 'UTF-8字符正确解码');
    is($lines[2], " ip address 192.168.1.1 255.255.255.0", '第三行正确');
    is($lines[3], " no shutdown", '第四行正确');
    
    done_testing();
};

# 测试特殊字符和符号
subtest '特殊字符处理' => sub {
    my @special_config = (
        "line with symbols: !@#$%^&*()",
        "line with numbers: 1234567890",
        "line with spaces:   multiple   spaces  ",
        "line with tabs:\t\ttabs\there"
    );
    
    my $config_obj = PDK::Config::Context::Static->new(
        id     => 1,
        name   => 'test-firewall',
        vendor => 'test',
        config => \@special_config
    );
    
    my @lines = ();
    while (my $line = $config_obj->nextUnParsedLine) {
        push @lines, $line;
    }
    
    is(scalar(@lines), 4, '正确读取了所有4行配置');
    is($lines[0], "line with symbols: !@#$%^&*()", '符号字符正确');
    is($lines[1], "line with numbers: 1234567890", '数字字符正确');
    is($lines[2], "line with spaces:   multiple   spaces  ", '空格字符正确保留');
    is($lines[3], "line with tabs:\t\ttabs\there", '制表符正确保留');
    
    done_testing();
};

# 测试空行处理
subtest '空行处理' => sub {
    my @empty_config = (
        "first line",
        "",
        "   ",  # 只有空格的行
        "\t",   # 只有制表符的行
        "second line",
        "",
        "third line"
    );
    
    my $config_obj = PDK::Config::Context::Static->new(
        id     => 1,
        name   => 'test-firewall',
        vendor => 'test',
        config => \@empty_config
    );
    
    my @lines = ();
    while (my $line = $config_obj->nextUnParsedLine) {
        push @lines, $line;
    }
    
    # 应该跳过空行，只返回有内容的行
    is(scalar(@lines), 3, '正确跳过了空行，只返回3行有内容的配置');
    is($lines[0], "first line", '第一行正确');
    is($lines[1], "second line", '第二行正确');
    is($lines[2], "third line", '第三行正确');
    
    done_testing();
};

# 测试编码检测的健壮性
subtest '编码检测健壮性' => sub {
    # 测试纯ASCII内容
    my @ascii_config = (
        "interface FastEthernet0/1",
        " ip address 10.0.0.1 255.255.255.0",
        " no shutdown"
    );
    
    my $config_obj = PDK::Config::Context::Static->new(
        id     => 1,
        name   => 'test-firewall',
        vendor => 'test',
        config => \@ascii_config
    );
    
    my @lines = ();
    while (my $line = $config_obj->nextUnParsedLine) {
        push @lines, $line;
    }
    
    is(scalar(@lines), 3, '正确读取了所有3行ASCII配置');
    is($lines[0], "interface FastEthernet0/1", 'ASCII第一行正确');
    is($lines[1], " ip address 10.0.0.1 255.255.255.0", 'ASCII第二行正确');
    is($lines[2], " no shutdown", 'ASCII第三行正确');
    
    done_testing();
};

done_testing();
