/*
 * xs_jit_builder.c - C implementation for XS::JIT code builder
 *
 * This provides a fluent C API for generating C code strings dynamically.
 */

#define PERL_NO_GET_CONTEXT
#include "xs_jit_builder.h"
#include <stdarg.h>

/* ============================================
 * Internal helpers
 * ============================================ */

static void add_indent(pTHX_ XS_JIT_Builder* b) {
    int i;
    if (b->use_tabs) {
        for (i = 0; i < b->indent; i++) {
            sv_catpvs(b->code, "\t");
        }
    } else {
        int spaces = b->indent * b->indent_width;
        for (i = 0; i < spaces; i++) {
            sv_catpvs(b->code, " ");
        }
    }
}

/* ============================================
 * Lifecycle
 * ============================================ */

XS_JIT_Builder* xs_jit_builder_new(pTHX) {
    XS_JIT_Builder* b;
    Newxz(b, 1, XS_JIT_Builder);
    b->code = newSVpvs("");
    b->indent = 0;
    b->indent_width = 4;
    b->use_tabs = 0;
    b->in_function = 0;
    return b;
}

void xs_jit_builder_free(pTHX_ XS_JIT_Builder* b) {
    /* Note: does NOT free code SV - caller should get it first if needed */
    if (b) {
        SvREFCNT_dec(b->code);
        Safefree(b);
    }
}

SV* xs_jit_builder_code(pTHX_ XS_JIT_Builder* b) {
    SvREFCNT_inc(b->code);
    return b->code;
}

const char* xs_jit_builder_cstr(pTHX_ XS_JIT_Builder* b) {
    return SvPV_nolen(b->code);
}

void xs_jit_builder_reset(pTHX_ XS_JIT_Builder* b) {
    SvREFCNT_dec(b->code);
    b->code = newSVpvs("");
    b->indent = 0;
    b->in_function = 0;
}

/* ============================================
 * Low-level output
 * ============================================ */

void xs_jit_line(pTHX_ XS_JIT_Builder* b, const char* fmt, ...) {
    va_list args;
    
    add_indent(aTHX_ b);
    
    va_start(args, fmt);
    sv_vcatpvf(b->code, fmt, &args);
    va_end(args);
    
    sv_catpvs(b->code, "\n");
}

void xs_jit_raw(pTHX_ XS_JIT_Builder* b, const char* fmt, ...) {
    va_list args;
    
    va_start(args, fmt);
    sv_vcatpvf(b->code, fmt, &args);
    va_end(args);
}

void xs_jit_comment(pTHX_ XS_JIT_Builder* b, const char* text) {
    add_indent(aTHX_ b);
    sv_catpvf(b->code, "/* %s */\n", text);
}

void xs_jit_blank(pTHX_ XS_JIT_Builder* b) {
    sv_catpvs(b->code, "\n");
}

/* ============================================
 * Indentation control
 * ============================================ */

void xs_jit_indent(XS_JIT_Builder* b) {
    b->indent++;
}

void xs_jit_dedent(XS_JIT_Builder* b) {
    if (b->indent > 0) {
        b->indent--;
    }
}

void xs_jit_set_indent_width(XS_JIT_Builder* b, int width) {
    b->indent_width = width;
}

void xs_jit_set_use_tabs(XS_JIT_Builder* b, int use_tabs) {
    b->use_tabs = use_tabs;
}

/* ============================================
 * Blocks and structure
 * ============================================ */

void xs_jit_block_start(pTHX_ XS_JIT_Builder* b) {
    xs_jit_line(aTHX_ b, "{");
    b->indent++;
}

void xs_jit_block_end(pTHX_ XS_JIT_Builder* b) {
    b->indent--;
    xs_jit_line(aTHX_ b, "}");
}

/* ============================================
 * XS Function structure
 * ============================================ */

void xs_jit_xs_function(pTHX_ XS_JIT_Builder* b, const char* name) {
    xs_jit_line(aTHX_ b, "XS_EUPXS(%s) {", name);
    b->indent++;
    b->in_function = 1;
}

void xs_jit_xs_preamble(pTHX_ XS_JIT_Builder* b) {
    xs_jit_line(aTHX_ b, "dVAR; dXSARGS;");
    xs_jit_line(aTHX_ b, "PERL_UNUSED_VAR(cv);");
}

void xs_jit_xs_end(pTHX_ XS_JIT_Builder* b) {
    b->indent--;
    xs_jit_line(aTHX_ b, "}");
    b->in_function = 0;
}

void xs_jit_xs_return(pTHX_ XS_JIT_Builder* b, int count) {
    xs_jit_line(aTHX_ b, "XSRETURN(%d);", count);
}

void xs_jit_xs_return_undef(pTHX_ XS_JIT_Builder* b) {
    xs_jit_line(aTHX_ b, "ST(0) = &PL_sv_undef;");
    xs_jit_line(aTHX_ b, "XSRETURN(1);");
}

void xs_jit_croak_usage(pTHX_ XS_JIT_Builder* b, const char* usage) {
    xs_jit_line(aTHX_ b, "croak_xs_usage(cv, \"%s\");", usage);
}

/* ============================================
 * Control flow
 * ============================================ */

void xs_jit_if(pTHX_ XS_JIT_Builder* b, const char* cond) {
    xs_jit_line(aTHX_ b, "if (%s) {", cond);
    b->indent++;
}

void xs_jit_elsif(pTHX_ XS_JIT_Builder* b, const char* cond) {
    b->indent--;
    xs_jit_line(aTHX_ b, "} else if (%s) {", cond);
    b->indent++;
}

void xs_jit_else(pTHX_ XS_JIT_Builder* b) {
    b->indent--;
    xs_jit_line(aTHX_ b, "} else {");
    b->indent++;
}

void xs_jit_endif(pTHX_ XS_JIT_Builder* b) {
    b->indent--;
    xs_jit_line(aTHX_ b, "}");
}

void xs_jit_for(pTHX_ XS_JIT_Builder* b, const char* init, const char* cond, const char* incr) {
    xs_jit_line(aTHX_ b, "for (%s; %s; %s) {", init, cond, incr);
    b->indent++;
}

void xs_jit_while(pTHX_ XS_JIT_Builder* b, const char* cond) {
    xs_jit_line(aTHX_ b, "while (%s) {", cond);
    b->indent++;
}

void xs_jit_endloop(pTHX_ XS_JIT_Builder* b) {
    b->indent--;
    xs_jit_line(aTHX_ b, "}");
}

/* ============================================
 * Variable declarations
 * ============================================ */

void xs_jit_declare(pTHX_ XS_JIT_Builder* b, const char* type, const char* name, const char* value) {
    if (value && *value) {
        xs_jit_line(aTHX_ b, "%s %s = %s;", type, name, value);
    } else {
        xs_jit_line(aTHX_ b, "%s %s;", type, name);
    }
}

void xs_jit_declare_sv(pTHX_ XS_JIT_Builder* b, const char* name, const char* value) {
    xs_jit_declare(aTHX_ b, "SV*", name, value);
}

void xs_jit_declare_hv(pTHX_ XS_JIT_Builder* b, const char* name, const char* value) {
    xs_jit_declare(aTHX_ b, "HV*", name, value);
}

void xs_jit_declare_av(pTHX_ XS_JIT_Builder* b, const char* name, const char* value) {
    xs_jit_declare(aTHX_ b, "AV*", name, value);
}

void xs_jit_declare_int(pTHX_ XS_JIT_Builder* b, const char* name, const char* value) {
    xs_jit_declare(aTHX_ b, "int", name, value);
}

void xs_jit_declare_iv(pTHX_ XS_JIT_Builder* b, const char* name, const char* value) {
    xs_jit_declare(aTHX_ b, "IV", name, value);
}

void xs_jit_declare_nv(pTHX_ XS_JIT_Builder* b, const char* name, const char* value) {
    xs_jit_declare(aTHX_ b, "NV", name, value);
}

void xs_jit_declare_pv(pTHX_ XS_JIT_Builder* b, const char* name, const char* value) {
    xs_jit_declare(aTHX_ b, "const char*", name, value);
}

void xs_jit_assign(pTHX_ XS_JIT_Builder* b, const char* var, const char* value) {
    xs_jit_line(aTHX_ b, "%s = %s;", var, value);
}

/* ============================================
 * Common XS patterns
 * ============================================ */

void xs_jit_get_self(pTHX_ XS_JIT_Builder* b) {
    xs_jit_declare_sv(aTHX_ b, "self", "ST(0)");
}

void xs_jit_get_self_hv(pTHX_ XS_JIT_Builder* b) {
    xs_jit_declare_sv(aTHX_ b, "self", "ST(0)");
    xs_jit_declare_hv(aTHX_ b, "hv", "(HV*)SvRV(self)");
}

void xs_jit_get_self_av(pTHX_ XS_JIT_Builder* b) {
    xs_jit_declare_sv(aTHX_ b, "self", "ST(0)");
    xs_jit_declare_av(aTHX_ b, "av", "(AV*)SvRV(self)");
}

/* ============================================
 * Hash operations
 * ============================================ */

void xs_jit_hv_fetch(pTHX_ XS_JIT_Builder* b, const char* hv, const char* key, STRLEN len, const char* result_var) {
    xs_jit_line(aTHX_ b, "SV** %s = hv_fetch(%s, \"%s\", %lu, 0);", result_var, hv, key, (unsigned long)len);
}

void xs_jit_hv_fetch_sv(pTHX_ XS_JIT_Builder* b, const char* hv, const char* key_expr, const char* len_expr, const char* result_var) {
    xs_jit_line(aTHX_ b, "SV** %s = hv_fetch(%s, %s, %s, 0);", result_var, hv, key_expr, len_expr);
}

void xs_jit_hv_store(pTHX_ XS_JIT_Builder* b, const char* hv, const char* key, STRLEN len, const char* value) {
    xs_jit_line(aTHX_ b, "(void)hv_store(%s, \"%s\", %lu, %s, 0);", hv, key, (unsigned long)len, value);
}

void xs_jit_hv_store_sv(pTHX_ XS_JIT_Builder* b, const char* hv, const char* key_expr, const char* len_expr, const char* value) {
    xs_jit_line(aTHX_ b, "(void)hv_store(%s, %s, %s, %s, 0);", hv, key_expr, len_expr, value);
}

void xs_jit_hv_fetch_return(pTHX_ XS_JIT_Builder* b, const char* hv, const char* key, STRLEN len) {
    xs_jit_line(aTHX_ b, "SV** valp = hv_fetch(%s, \"%s\", %lu, 0);", hv, key, (unsigned long)len);
    xs_jit_line(aTHX_ b, "ST(0) = (valp && *valp) ? *valp : &PL_sv_undef;");
    xs_jit_line(aTHX_ b, "XSRETURN(1);");
}

/* ============================================
 * Array operations
 * ============================================ */

void xs_jit_av_fetch(pTHX_ XS_JIT_Builder* b, const char* av, const char* index, const char* result_var) {
    xs_jit_line(aTHX_ b, "SV** %s = av_fetch(%s, %s, 0);", result_var, av, index);
}

void xs_jit_av_store(pTHX_ XS_JIT_Builder* b, const char* av, const char* index, const char* value) {
    xs_jit_line(aTHX_ b, "av_store(%s, %s, %s);", av, index, value);
}

void xs_jit_av_push(pTHX_ XS_JIT_Builder* b, const char* av, const char* value) {
    xs_jit_line(aTHX_ b, "av_push(%s, %s);", av, value);
}

void xs_jit_av_len(pTHX_ XS_JIT_Builder* b, const char* av, const char* result_var) {
    xs_jit_line(aTHX_ b, "I32 %s = av_len(%s);", result_var, av);
}

/* ============================================
 * SV creation
 * ============================================ */

void xs_jit_new_sv_iv(pTHX_ XS_JIT_Builder* b, const char* result_var, const char* value) {
    xs_jit_line(aTHX_ b, "SV* %s = newSViv(%s);", result_var, value);
}

void xs_jit_new_sv_nv(pTHX_ XS_JIT_Builder* b, const char* result_var, const char* value) {
    xs_jit_line(aTHX_ b, "SV* %s = newSVnv(%s);", result_var, value);
}

void xs_jit_new_sv_pv(pTHX_ XS_JIT_Builder* b, const char* result_var, const char* str, STRLEN len) {
    xs_jit_line(aTHX_ b, "SV* %s = newSVpvn(\"%s\", %lu);", result_var, str, (unsigned long)len);
}

void xs_jit_mortal(pTHX_ XS_JIT_Builder* b, const char* sv) {
    xs_jit_line(aTHX_ b, "%s = sv_2mortal(%s);", sv, sv);
}

/* ============================================
 * Type checking
 * ============================================ */

void xs_jit_check_items(pTHX_ XS_JIT_Builder* b, int min, int max, const char* usage) {
    char cond[128];
    if (min == max) {
        snprintf(cond, sizeof(cond), "items != %d", min);
    } else if (max < 0) {
        snprintf(cond, sizeof(cond), "items < %d", min);
    } else {
        snprintf(cond, sizeof(cond), "items < %d || items > %d", min, max);
    }
    xs_jit_if(aTHX_ b, cond);
    xs_jit_croak_usage(aTHX_ b, usage);
    xs_jit_endif(aTHX_ b);
}

void xs_jit_check_defined(pTHX_ XS_JIT_Builder* b, const char* sv, const char* error_msg) {
    char cond[128];
    snprintf(cond, sizeof(cond), "!SvOK(%s)", sv);
    xs_jit_if(aTHX_ b, cond);
    xs_jit_croak(aTHX_ b, error_msg);
    xs_jit_endif(aTHX_ b);
}

void xs_jit_check_ref_type(pTHX_ XS_JIT_Builder* b, const char* sv, const char* type, const char* error_msg) {
    char cond[256];
    snprintf(cond, sizeof(cond), "!SvROK(%s) || SvTYPE(SvRV(%s)) != %s", sv, sv, type);
    xs_jit_if(aTHX_ b, cond);
    xs_jit_croak(aTHX_ b, error_msg);
    xs_jit_endif(aTHX_ b);
}

void xs_jit_check_hashref(pTHX_ XS_JIT_Builder* b, const char* sv, const char* error_msg) {
    xs_jit_check_ref_type(aTHX_ b, sv, "SVt_PVHV", error_msg);
}

void xs_jit_check_arrayref(pTHX_ XS_JIT_Builder* b, const char* sv, const char* error_msg) {
    xs_jit_check_ref_type(aTHX_ b, sv, "SVt_PVAV", error_msg);
}

/* ============================================
 * SV conversion (extract values from SV)
 * ============================================ */

void xs_jit_sv_to_iv(pTHX_ XS_JIT_Builder* b, const char* result_var, const char* sv) {
    xs_jit_line(aTHX_ b, "IV %s = SvIV(%s);", result_var, sv);
}

void xs_jit_sv_to_uv(pTHX_ XS_JIT_Builder* b, const char* result_var, const char* sv) {
    xs_jit_line(aTHX_ b, "UV %s = SvUV(%s);", result_var, sv);
}

void xs_jit_sv_to_nv(pTHX_ XS_JIT_Builder* b, const char* result_var, const char* sv) {
    xs_jit_line(aTHX_ b, "NV %s = SvNV(%s);", result_var, sv);
}

void xs_jit_sv_to_pv(pTHX_ XS_JIT_Builder* b, const char* result_var, const char* len_var, const char* sv) {
    if (len_var && *len_var) {
        xs_jit_line(aTHX_ b, "STRLEN %s;", len_var);
        xs_jit_line(aTHX_ b, "const char* %s = SvPV(%s, %s);", result_var, sv, len_var);
    } else {
        xs_jit_line(aTHX_ b, "const char* %s = SvPV_nolen(%s);", result_var, sv);
    }
}

void xs_jit_sv_to_bool(pTHX_ XS_JIT_Builder* b, const char* result_var, const char* sv) {
    xs_jit_line(aTHX_ b, "int %s = SvTRUE(%s);", result_var, sv);
}

/* ============================================
 * Return helpers
 * ============================================ */

void xs_jit_return_iv(pTHX_ XS_JIT_Builder* b, const char* value) {
    xs_jit_line(aTHX_ b, "ST(0) = sv_2mortal(newSViv(%s));", value);
    xs_jit_xs_return(aTHX_ b, 1);
}

void xs_jit_return_uv(pTHX_ XS_JIT_Builder* b, const char* value) {
    xs_jit_line(aTHX_ b, "ST(0) = sv_2mortal(newSVuv(%s));", value);
    xs_jit_xs_return(aTHX_ b, 1);
}

void xs_jit_return_nv(pTHX_ XS_JIT_Builder* b, const char* value) {
    xs_jit_line(aTHX_ b, "ST(0) = sv_2mortal(newSVnv(%s));", value);
    xs_jit_xs_return(aTHX_ b, 1);
}

void xs_jit_return_pv(pTHX_ XS_JIT_Builder* b, const char* str, const char* len) {
    /* str is used as-is - caller must quote literal strings with \" */
    if (len && *len) {
        xs_jit_line(aTHX_ b, "ST(0) = sv_2mortal(newSVpvn(%s, %s));", str, len);
    } else {
        xs_jit_line(aTHX_ b, "ST(0) = sv_2mortal(newSVpv(%s, 0));", str);
    }
    xs_jit_xs_return(aTHX_ b, 1);
}

void xs_jit_return_sv(pTHX_ XS_JIT_Builder* b, const char* sv) {
    xs_jit_line(aTHX_ b, "ST(0) = %s;", sv);
    xs_jit_xs_return(aTHX_ b, 1);
}

void xs_jit_return_yes(pTHX_ XS_JIT_Builder* b) {
    xs_jit_line(aTHX_ b, "ST(0) = &PL_sv_yes;");
    xs_jit_xs_return(aTHX_ b, 1);
}

void xs_jit_return_no(pTHX_ XS_JIT_Builder* b) {
    xs_jit_line(aTHX_ b, "ST(0) = &PL_sv_no;");
    xs_jit_xs_return(aTHX_ b, 1);
}

void xs_jit_return_self(pTHX_ XS_JIT_Builder* b) {
    xs_jit_line(aTHX_ b, "ST(0) = self;");
    xs_jit_xs_return(aTHX_ b, 1);
}

/* ============================================
 * Common method patterns
 * ============================================ */

void xs_jit_method_start(pTHX_ XS_JIT_Builder* b, const char* func_name, int min_args, int max_args, const char* usage) {
    xs_jit_xs_function(aTHX_ b, func_name);
    xs_jit_xs_preamble(aTHX_ b);
    if (min_args > 0 || max_args >= 0) {
        xs_jit_check_items(aTHX_ b, min_args, max_args, usage);
    }
    xs_jit_get_self_hv(aTHX_ b);
}

/* Predicate (returns true/false based on attribute existence) */
void xs_jit_predicate(pTHX_ XS_JIT_Builder* b, const char* func_name, const char* attr_name, STRLEN attr_len) {
    xs_jit_xs_function(aTHX_ b, func_name);
    xs_jit_xs_preamble(aTHX_ b);
    xs_jit_get_self_hv(aTHX_ b);
    xs_jit_line(aTHX_ b, "SV** valp = hv_fetch(hv, \"%s\", %lu, 0);", attr_name, (unsigned long)attr_len);
    xs_jit_if(aTHX_ b, "valp != NULL");
    xs_jit_return_yes(aTHX_ b);
    xs_jit_else(aTHX_ b);
    xs_jit_return_no(aTHX_ b);
    xs_jit_endif(aTHX_ b);
    xs_jit_xs_end(aTHX_ b);
}

/* Clearer (deletes an attribute) */
void xs_jit_clearer(pTHX_ XS_JIT_Builder* b, const char* func_name, const char* attr_name, STRLEN attr_len) {
    xs_jit_xs_function(aTHX_ b, func_name);
    xs_jit_xs_preamble(aTHX_ b);
    xs_jit_get_self_hv(aTHX_ b);
    xs_jit_line(aTHX_ b, "(void)hv_delete(hv, \"%s\", %lu, G_DISCARD);", attr_name, (unsigned long)attr_len);
    xs_jit_return_self(aTHX_ b);
    xs_jit_xs_end(aTHX_ b);
}

/* ============================================
 * Error handling
 * ============================================ */

void xs_jit_croak(pTHX_ XS_JIT_Builder* b, const char* message) {
    xs_jit_line(aTHX_ b, "croak(\"%s\");", message);
}

void xs_jit_warn(pTHX_ XS_JIT_Builder* b, const char* message) {
    xs_jit_line(aTHX_ b, "warn(\"%s\");", message);
}

/* ============================================
 * Prebuilt patterns
 * ============================================ */

void xs_jit_ro_accessor(pTHX_ XS_JIT_Builder* b, const char* func_name, const char* attr_name, STRLEN attr_len) {
    xs_jit_xs_function(aTHX_ b, func_name);
    xs_jit_xs_preamble(aTHX_ b);
    
    xs_jit_if(aTHX_ b, "items > 1");
    xs_jit_croak(aTHX_ b, "Read only attribute cannot be set");
    xs_jit_endif(aTHX_ b);
    
    xs_jit_get_self_hv(aTHX_ b);
    xs_jit_hv_fetch_return(aTHX_ b, "hv", attr_name, attr_len);
    
    xs_jit_xs_end(aTHX_ b);
}

void xs_jit_rw_accessor(pTHX_ XS_JIT_Builder* b, const char* func_name, const char* attr_name, STRLEN attr_len) {
    xs_jit_xs_function(aTHX_ b, func_name);
    xs_jit_xs_preamble(aTHX_ b);
    
    xs_jit_get_self_hv(aTHX_ b);
    
    xs_jit_if(aTHX_ b, "items > 1");
    xs_jit_line(aTHX_ b, "SV* new_val = newSVsv(ST(1));");
    xs_jit_hv_store(aTHX_ b, "hv", attr_name, attr_len, "new_val");
    xs_jit_line(aTHX_ b, "ST(0) = new_val;");
    xs_jit_line(aTHX_ b, "XSRETURN(1);");
    xs_jit_endif(aTHX_ b);
    
    xs_jit_hv_fetch_return(aTHX_ b, "hv", attr_name, attr_len);
    
    xs_jit_xs_end(aTHX_ b);
}

void xs_jit_constructor(pTHX_ XS_JIT_Builder* b, const char* func_name, const char* class_name, XS_JIT_Attr* attrs, int num_attrs) {
    int i;
    
    xs_jit_xs_function(aTHX_ b, func_name);
    xs_jit_xs_preamble(aTHX_ b);
    
    xs_jit_comment(aTHX_ b, "Get class name");
    xs_jit_declare_sv(aTHX_ b, "class_sv", "ST(0)");
    xs_jit_line(aTHX_ b, "const char* classname = SvPV_nolen(class_sv);");
    xs_jit_blank(aTHX_ b);
    
    xs_jit_comment(aTHX_ b, "Parse args hash if provided");
    xs_jit_declare_hv(aTHX_ b, "args", "NULL");
    xs_jit_if(aTHX_ b, "items > 1 && SvROK(ST(1)) && SvTYPE(SvRV(ST(1))) == SVt_PVHV");
    xs_jit_line(aTHX_ b, "args = (HV*)SvRV(ST(1));");
    xs_jit_endif(aTHX_ b);
    xs_jit_blank(aTHX_ b);
    
    xs_jit_comment(aTHX_ b, "Create new hash for object");
    xs_jit_declare_hv(aTHX_ b, "hv", "newHV()");
    xs_jit_blank(aTHX_ b);
    
    xs_jit_comment(aTHX_ b, "Process attributes");
    for (i = 0; i < num_attrs; i++) {
        const char* name = attrs[i].name;
        STRLEN len = attrs[i].len;
        
        xs_jit_line(aTHX_ b, "{");
        b->indent++;
        xs_jit_line(aTHX_ b, "SV** %s_valp = args ? hv_fetch(args, \"%s\", %lu, 0) : NULL;", name, name, (unsigned long)len);
        /* Build condition string for if */
        SV* cond_sv = newSVpvf("%s_valp && SvOK(*%s_valp)", name, name);
        xs_jit_if(aTHX_ b, SvPV_nolen(cond_sv));
        SvREFCNT_dec(cond_sv);
        xs_jit_line(aTHX_ b, "(void)hv_store(hv, \"%s\", %lu, newSVsv(*%s_valp), 0);", name, (unsigned long)len, name);
        xs_jit_endif(aTHX_ b);
        b->indent--;
        xs_jit_line(aTHX_ b, "}");
    }
    xs_jit_blank(aTHX_ b);
    
    xs_jit_comment(aTHX_ b, "Bless and return");
    xs_jit_declare_sv(aTHX_ b, "self", "newRV_noinc((SV*)hv)");
    xs_jit_line(aTHX_ b, "sv_bless(self, gv_stashpv(classname, GV_ADD));");
    xs_jit_line(aTHX_ b, "ST(0) = sv_2mortal(self);");
    xs_jit_line(aTHX_ b, "XSRETURN(1);");
    
    xs_jit_xs_end(aTHX_ b);
}
