# session_info

`session_info` outputs version information for modules loaded in the current session,
Python, the OS, and the CPU. It is designed as a minimum measure to increase
reproducibility and provides similar information as `sessionInfo` and
`devtools::session_info` in R.

## Motivation

`session_info` is particularly useful when conducting exploratory data analysis in
Jupyter notebooks. Listing the version numbers of all loaded modules after
importing them is a simple way to ensure a minimum level of reproducibility
while requiring little additional effort. This practice is useful both when
revisiting notebooks and when sharing them with colleagues. `session_info` is meant to
complement more robust practices such as frozen virtual environments,
containers, and binder.

## Installation

`session_info` can be installed via `pip install session_info`. It does not depend on a package
manager to find version numbers since it fetches them from the module's version
string. Its only dependency is `stdlib_list`, which is used to distinguish
between standard library and third party modules.

## Usage

`session_info` can be used from a script like so:

```python
import math

import natsort
import pandas
import session_info


session_info.show()
```


Output:

```
Session information:
-----
natsort             7.1.1
pandas              1.2.2
session_info        1.0.0
-----
IPython             7.23.0
jupyter_client      6.1.12
jupyter_core        4.7.1
-----
Python 3.9.2 | packaged by conda-forge | (default, Feb 21 2021, 05:02:46) [GCC 9.3.0]
Linux-5.11.13-arch1-1-x86_64-with-glibc2.33
-----
Session information updated at 2021-05-06 09:59
```

The default behavior is to only output modules not in the standard library,
which is why the `math` module is omitted above (it can be included by
specifying `std_lib=True`). To include not only the explicitly imported
modules, but also any dependencies they import internally, specify `dependencies=True`.

When `session_info` is invoked from a Jupyter Notebook,
the output is concealed in `<details>` tags
and will only show when clicked.
Since this saves visual real estate,
any modules imported indirectly as dependencies
will be included by default
and it looks like this:

<details>
<summary>Click to view session information</summary>
<pre>
-----
natsort             7.1.1
pandas              1.2.2
session_info        1.0.0
-----
</pre>
<details>
<summary>Click to view modules imported as dependencies</summary>
<pre>
backcall            0.2.0
cython_runtime      NA
dateutil            2.8.1
decorator           5.0.7
ipykernel           5.5.3
ipython_genutils    0.2.0
jedi                0.18.0
numpy               1.20.2
parso               0.8.2
pexpect             4.8.0
pickleshare         0.7.5
prompt_toolkit      3.0.18
ptyprocess          0.7.0
pygments            2.8.1
pytz                2021.1
six                 1.15.0
storemagic          NA
tornado             6.1
traitlets           5.0.5
wcwidth             0.2.5
zmq                 22.0.3
</pre>
</details>
<pre>
-----
IPython             7.23.0
jupyter_client      6.1.12
jupyter_core        4.7.1
-----
Python 3.9.2 | packaged by conda-forge | (default, Feb 21 2021, 05:02:46) [GCC 9.3.0]
Linux-5.11.13-arch1-1-x86_64-with-glibc2.33
-----
Session information updated at 2021-05-06 09:59
</pre>
</details>

If you prefer to show the session information without the HTML tags,
you can use `session_info.show(html=False)` in the notebook
to get the same output as in the first example above.

`session_info` can also write the module dependencies
to a `requirements.txt` file,
which is compatible with `pip install -r /path/to/requirements.txt`.

[View the docstring for complete parameter information](https://gitlab.com/joelostblom/session_info/-/blob/master/sinfo/main.py#L73-118).

## Background

`session_info` started as minor modifications of `py_session`, and as it grew it
became convenient to create a new package. `session_info` was built with the help of
information provided in stackoverflow answers and existing similar packages,
including

- https://github.com/fbrundu/py_session
- https://github.com/jrjohansson/version_information
- https://github.com/rasbt/watermark
- https://stackoverflow.com/a/4858123/2166823
- https://stackoverflow.com/a/40690954/2166823
- https://stackoverflow.com/a/52187331/2166823
