/*-
 * util.c
 *
 * SPDX-License-Identifier: BSD-2-Clause
 *
 * Copyright (c) 2001 Maksim Yevmenkin <m_evmenkin@yahoo.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: util.c,v 1.2 2003/05/19 17:29:29 max Exp $
 */
 
#include <sys/param.h>
#define L2CAP_SOCKET_CHECKED
#include <bluetooth.h>
#include <stdio.h>
#include <string.h>

#define SIZE(x) (sizeof((x))/sizeof((x)[0]))

char const *
hci_link2str(int link_type)
{
	static char const * const	t[] = {
		/* NG_HCI_LINK_SCO */ "SCO",
		/* NG_HCI_LINK_ACL */ "ACL"
	};

	return (link_type >= SIZE(t)? "?" : t[link_type]);
} /* hci_link2str */

char const *
hci_pin2str(int type)
{
	static char const * const	t[] = {
		/* 0x00 */ "Variable PIN",
		/* 0x01 */ "Fixed PIN"
	};

	return (type >= SIZE(t)? "?" : t[type]);
} /* hci_pin2str */

char const *
hci_scan2str(int scan)
{
	static char const * const	t[] = {
		/* 0x00 */ "No Scan enabled",
		/* 0x01 */ "Inquiry Scan enabled. Page Scan disabled",
		/* 0x02 */ "Inquiry Scan disabled. Page Scan enabled",
		/* 0x03 */ "Inquiry Scan enabled. Page Scan enabled"
	};

	return (scan >= SIZE(t)? "?" : t[scan]);
} /* hci_scan2str */

char const *
hci_encrypt2str(int encrypt, int brief)
{
	static char const * const	t[] = {
		/* 0x00 */ "Disabled",
		/* 0x01 */ "Only for point-to-point packets",
		/* 0x02 */ "Both point-to-point and broadcast packets"
	};

	static char const * const	t1[] = {
		/* NG_HCI_ENCRYPTION_MODE_NONE */ "NONE",
		/* NG_HCI_ENCRYPTION_MODE_P2P */  "P2P",
		/* NG_HCI_ENCRYPTION_MODE_ALL */  "ALL",
	};

	if (brief)
		return (encrypt >= SIZE(t1)? "?" : t1[encrypt]);

	return (encrypt >= SIZE(t)? "?" : t[encrypt]);
} /* hci_encrypt2str */

char const *
hci_coding2str(int coding)
{
	static char const * const	t[] = {
		/* 0x00 */ "Linear",
		/* 0x01 */ "u-law",
		/* 0x02 */ "A-law",
		/* 0x03 */ "Reserved"
	};

	return (coding >= SIZE(t)? "?" : t[coding]);
} /* hci_coding2str */

char const *
hci_vdata2str(int data)
{
	static char const * const	t[] = {
		/* 0x00 */ "1's complement",
		/* 0x01 */ "2's complement",
		/* 0x02 */ "Sign-Magnitude",
		/* 0x03 */ "Reserved"
	};

	return (data >= SIZE(t)? "?" : t[data]);
} /* hci_vdata2str */

char const *
hci_hmode2str(int mode, char *buffer, int size)
{
	static char const * const	t[] = {
		/* 0x01 */ "Suspend Page Scan ",
		/* 0x02 */ "Suspend Inquiry Scan ",
		/* 0x04 */ "Suspend Periodic Inquiries "
        };

	if (buffer != NULL && size > 0) {
		int	n;

		memset(buffer, 0, size);
		size--;
		for (n = 0; n < SIZE(t); n++) {
			int	len = strlen(buffer);

			if (len >= size)
				break;
			if (mode & (1 << n))
				strncat(buffer, t[n], size - len);
		}
	}

	return (buffer);
} /* hci_hmode2str */

char const *
hci_ver2str(int ver)
{
	static char const * const	t[] = {
		/* 0x00 */ "Bluetooth HCI Specification 1.0B",
		/* 0x01 */ "Bluetooth HCI Specification 1.1",
		/* 0x02 */ "Bluetooth HCI Specification 1.2",
		/* 0x03 */ "Bluetooth HCI Specification 2.0",
		/* 0x04 */ "Bluetooth HCI Specification 2.1",
		/* 0x05 */ "Bluetooth HCI Specification 3.0",
		/* 0x06 */ "Bluetooth HCI Specification 4.0",
		/* 0x07 */ "Bluetooth HCI Specification 4.1",
		/* 0x08 */ "Bluetooth HCI Specification 4.2",
		/* 0x09 */ "Bluetooth HCI Specification 5.0",
		/* 0x0a */ "Bluetooth HCI Specification 5.1",
		/* 0x0b */ "Bluetooth HCI Specification 5.2"
	};

	return (ver >= SIZE(t)? "?" : t[ver]);
} /* hci_ver2str */

char const *
hci_lmpver2str(int ver)
{
	static char const * const	t[] = {
		/* 0x00 */ "Bluetooth LMP 1.0",
		/* 0x01 */ "Bluetooth LMP 1.1",
		/* 0x02 */ "Bluetooth LMP 1.2",
		/* 0x03 */ "Bluetooth LMP 2.0",
		/* 0x04 */ "Bluetooth LMP 2.1",
		/* 0x05 */ "Bluetooth LMP 3.0",
		/* 0x06 */ "Bluetooth LMP 4.0",
		/* 0x07 */ "Bluetooth LMP 4.1",
		/* 0x08 */ "Bluetooth LMP 4.2",
		/* 0x09 */ "Bluetooth LMP 5.0",
		/* 0x0a */ "Bluetooth LMP 5.1",
		/* 0x0b */ "Bluetooth LMP 5.2"
	};

	return (ver >= SIZE(t)? "?" : t[ver]);
} /* hci_lmpver2str */

char const *
hci_manufacturer2str(int m)
{
	static char const * const	t[] = {
		/* 0000 */ "Ericsson Technology Licensing",
		/* 0001 */ "Nokia Mobile Phones",
		/* 0002 */ "Intel Corp.",
		/* 0003 */ "IBM Corp.",
		/* 0004 */ "Toshiba Corp.",
		/* 0005 */ "3Com",
		/* 0006 */ "Microsoft",
		/* 0007 */ "Lucent",
		/* 0008 */ "Motorola",
		/* 0009 */ "Infineon Technologies AG",
		/* 0010 */ "Qualcomm Technologies International, Ltd. (QTIL)",
		/* 0011 */ "Silicon Wave",
		/* 0012 */ "Digianswer A/S",
		/* 0013 */ "Texas Instruments Inc.",
		/* 0014 */ "Parthus Technologies Inc.",
		/* 0015 */ "Broadcom Corporation",
		/* 0016 */ "Mitel Semiconductor",
		/* 0017 */ "Widcomm, Inc.",
		/* 0018 */ "Zeevo, Inc.",
		/* 0019 */ "Atmel Corporation",
		/* 0020 */ "Mitsubishi Electric Corporation",
		/* 0021 */ "RTX Telecom A/S",
		/* 0022 */ "KC Technology Inc.",
		/* 0023 */ "Newlogic",
		/* 0024 */ "Transilica, Inc.",
		/* 0025 */ "Rohde & Schwarz GmbH & Co. KG",
		/* 0026 */ "TTPCom Limited",
		/* 0027 */ "Signia Technologies, Inc.",
		/* 0028 */ "Conexant Systems Inc.",
		/* 0029 */ "Qualcomm",
		/* 0030 */ "Inventel",
		/* 0031 */ "AVM Berlin",
		/* 0032 */ "BandSpeed, Inc.",
		/* 0033 */ "Mansella Ltd",
		/* 0034 */ "NEC Corporation",
		/* 0035 */ "WavePlus Technology Co., Ltd.",
		/* 0036 */ "Alcatel",
		/* 0037 */ "NXP Semiconductors (formerly Philips Semiconductors)",
		/* 0038 */ "C Technologies",
		/* 0039 */ "Open Interface",
		/* 0040 */ "R F Micro Devices",
		/* 0041 */ "Hitachi Ltd",
		/* 0042 */ "Symbol Technologies, Inc.",
		/* 0043 */ "Tenovis",
		/* 0044 */ "Macronix International Co. Ltd.",
		/* 0045 */ "GCT Semiconductor",
		/* 0046 */ "Norwood Systems",
		/* 0047 */ "MewTel Technology Inc.",
		/* 0048 */ "ST Microelectronics",
		/* 0049 */ "Synopsys, Inc.",
		/* 0050 */ "Red-M (Communications) Ltd",
		/* 0051 */ "Commil Ltd",
		/* 0052 */ "Computer Access Technology Corporation (CATC)",
		/* 0053 */ "Eclipse (HQ Espana) S.L.",
		/* 0054 */ "Renesas Electronics Corporation",
		/* 0055 */ "Mobilian Corporation",
		/* 0056 */ "Syntronix Corporation",
		/* 0057 */ "Integrated System Solution Corp.",
		/* 0058 */ "Panasonic Corporation (formerly Matsushita Electric Industrial Co., Ltd.)",
		/* 0059 */ "Gennum Corporation",
		/* 0060 */ "BlackBerry Limited (formerly Research In Motion)",
		/* 0061 */ "IPextreme, Inc.",
		/* 0062 */ "Systems and Chips, Inc",
		/* 0063 */ "Bluetooth SIG, Inc",
		/* 0064 */ "Seiko Epson Corporation",
		/* 0065 */ "Integrated Silicon Solution Taiwan, Inc.",
		/* 0066 */ "CONWISE Technology Corporation Ltd",
		/* 0067 */ "PARROT AUTOMOTIVE SAS",
		/* 0068 */ "Socket Mobile",
		/* 0069 */ "Atheros Communications, Inc.",
		/* 0070 */ "MediaTek, Inc.",
		/* 0071 */ "Bluegiga",
		/* 0072 */ "Marvell Technology Group Ltd.",
		/* 0073 */ "3DSP Corporation",
		/* 0074 */ "Accel Semiconductor Ltd.",
		/* 0075 */ "Continental Automotive Systems",
		/* 0076 */ "Apple, Inc.",
		/* 0077 */ "Staccato Communications, Inc.",
		/* 0078 */ "Avago Technologies",
		/* 0079 */ "APT Ltd.",
		/* 0080 */ "SiRF Technology, Inc.",
		/* 0081 */ "Tzero Technologies, Inc.",
		/* 0082 */ "J&M Corporation",
		/* 0083 */ "Free2move AB",
		/* 0084 */ "3DiJoy Corporation",
		/* 0085 */ "Plantronics, Inc.",
		/* 0086 */ "Sony Ericsson Mobile Communications",
		/* 0087 */ "Harman International Industries, Inc.",
		/* 0088 */ "Vizio, Inc.",
		/* 0089 */ "Nordic Semiconductor ASA",
		/* 0090 */ "EM Microelectronic-Marin SA",
		/* 0091 */ "Ralink Technology Corporation",
		/* 0092 */ "Belkin International, Inc.",
		/* 0093 */ "Realtek Semiconductor Corporation",
		/* 0094 */ "Stonestreet One, LLC",
		/* 0095 */ "Wicentric, Inc.",
		/* 0096 */ "RivieraWaves S.A.S",
		/* 0097 */ "RDA Microelectronics",
		/* 0098 */ "Gibson Guitars",
		/* 0099 */ "MiCommand Inc.",
		/* 0100 */ "Band XI International, LLC",
		/* 0101 */ "Hewlett-Packard Company",
		/* 0102 */ "9Solutions Oy",
		/* 0103 */ "GN Netcom A/S",
		/* 0104 */ "General Motors",
		/* 0105 */ "A&D Engineering, Inc.",
		/* 0106 */ "MindTree Ltd.",
		/* 0107 */ "Polar Electro OY",
		/* 0108 */ "Beautiful Enterprise Co., Ltd.",
		/* 0109 */ "BriarTek, Inc",
		/* 0110 */ "Summit Data Communications, Inc.",
		/* 0111 */ "Sound ID",
		/* 0112 */ "Monster, LLC",
		/* 0113 */ "connectBlue AB",
		/* 0114 */ "ShangHai Super Smart Electronics Co. Ltd.",
		/* 0115 */ "Group Sense Ltd.",
		/* 0116 */ "Zomm, LLC",
		/* 0117 */ "Samsung Electronics Co. Ltd.",
		/* 0118 */ "Creative Technology Ltd.",
		/* 0119 */ "Laird Technologies",
		/* 0120 */ "Nike, Inc.",
		/* 0121 */ "lesswire AG",
		/* 0122 */ "MStar Semiconductor, Inc.",
		/* 0123 */ "Hanlynn Technologies",
		/* 0124 */ "A & R Cambridge",
		/* 0125 */ "Seers Technology Co., Ltd.",
		/* 0126 */ "Sports Tracking Technologies Ltd.",
		/* 0127 */ "Autonet Mobile",
		/* 0128 */ "DeLorme Publishing Company, Inc.",
		/* 0129 */ "WuXi Vimicro",
		/* 0130 */ "Sennheiser Communications A/S",
		/* 0131 */ "TimeKeeping Systems, Inc.",
		/* 0132 */ "Ludus Helsinki Ltd.",
		/* 0133 */ "BlueRadios, Inc.",
		/* 0134 */ "Equinux AG",
		/* 0135 */ "Garmin International, Inc.",
		/* 0136 */ "Ecotest",
		/* 0137 */ "GN ReSound A/S",
		/* 0138 */ "Jawbone",
		/* 0139 */ "Topcon Positioning Systems, LLC",
		/* 0140 */ "Gimbal Inc. (formerly Qualcomm Labs, Inc. and Qualcomm Retail Solutions, Inc.)",
		/* 0141 */ "Zscan Software",
		/* 0142 */ "Quintic Corp",
		/* 0143 */ "Telit Wireless Solutions GmbH (formerly Stollmann E+V GmbH)",
		/* 0144 */ "Funai Electric Co., Ltd.",
		/* 0145 */ "Advanced PANMOBIL systems GmbH & Co. KG",
		/* 0146 */ "ThinkOptics, Inc.",
		/* 0147 */ "Universal Electronics, Inc.",
		/* 0148 */ "Airoha Technology Corp.",
		/* 0149 */ "NEC Lighting, Ltd.",
		/* 0150 */ "ODM Technology, Inc.",
		/* 0151 */ "ConnecteDevice Ltd.",
		/* 0152 */ "zero1.tv GmbH",
		/* 0153 */ "i.Tech Dynamic Global Distribution Ltd.",
		/* 0154 */ "Alpwise",
		/* 0155 */ "Jiangsu Toppower Automotive Electronics Co., Ltd.",
		/* 0156 */ "Colorfy, Inc.",
		/* 0157 */ "Geoforce Inc.",
		/* 0158 */ "Bose Corporation",
		/* 0159 */ "Suunto Oy",
		/* 0160 */ "Kensington Computer Products Group",
		/* 0161 */ "SR-Medizinelektronik",
		/* 0162 */ "Vertu Corporation Limited",
		/* 0163 */ "Meta Watch Ltd.",
		/* 0164 */ "LINAK A/S",
		/* 0165 */ "OTL Dynamics LLC",
		/* 0166 */ "Panda Ocean Inc.",
		/* 0167 */ "Visteon Corporation",
		/* 0168 */ "ARP Devices Limited",
		/* 0169 */ "MARELLI EUROPE S.P.A. (formerly Magneti Marelli S.p.A.)",
		/* 0170 */ "CAEN RFID srl",
		/* 0171 */ "Ingenieur-Systemgruppe Zahn GmbH",
		/* 0172 */ "Green Throttle Games",
		/* 0173 */ "Peter Systemtechnik GmbH",
		/* 0174 */ "Omegawave Oy",
		/* 0175 */ "Cinetix",
		/* 0176 */ "Passif Semiconductor Corp",
		/* 0177 */ "Saris Cycling Group, Inc",
		/* 0178 */ "Bekey A/S",
		/* 0179 */ "Clarinox Technologies Pty. Ltd.",
		/* 0180 */ "BDE Technology Co., Ltd.",
		/* 0181 */ "Swirl Networks",
		/* 0182 */ "Meso international",
		/* 0183 */ "TreLab Ltd",
		/* 0184 */ "Qualcomm Innovation Center, Inc. (QuIC)",
		/* 0185 */ "Johnson Controls, Inc.",
		/* 0186 */ "Starkey Laboratories Inc.",
		/* 0187 */ "S-Power Electronics Limited",
		/* 0188 */ "Ace Sensor Inc",
		/* 0189 */ "Aplix Corporation",
		/* 0190 */ "AAMP of America",
		/* 0191 */ "Stalmart Technology Limited",
		/* 0192 */ "AMICCOM Electronics Corporation",
		/* 0193 */ "Shenzhen Excelsecu Data Technology Co.,Ltd",
		/* 0194 */ "Geneq Inc.",
		/* 0195 */ "adidas AG",
		/* 0196 */ "LG Electronics",
		/* 0197 */ "Onset Computer Corporation",
		/* 0198 */ "Selfly BV",
		/* 0199 */ "Quuppa Oy.",
		/* 0200 */ "GeLo Inc",
		/* 0201 */ "Evluma",
		/* 0202 */ "MC10",
		/* 0203 */ "Binauric SE",
		/* 0204 */ "Beats Electronics",
		/* 0205 */ "Microchip Technology Inc.",
		/* 0206 */ "Elgato Systems GmbH",
		/* 0207 */ "ARCHOS SA",
		/* 0208 */ "Dexcom, Inc.",
		/* 0209 */ "Polar Electro Europe B.V.",
		/* 0210 */ "Dialog Semiconductor B.V.",
		/* 0211 */ "Taixingbang Technology (HK) Co,. LTD.",
		/* 0212 */ "Kawantech",
		/* 0213 */ "Austco Communication Systems",
		/* 0214 */ "Timex Group USA, Inc.",
		/* 0215 */ "Qualcomm Technologies, Inc.",
		/* 0216 */ "Qualcomm Connected Experiences, Inc.",
		/* 0217 */ "Voyetra Turtle Beach",
		/* 0218 */ "txtr GmbH",
		/* 0219 */ "Biosentronics",
		/* 0220 */ "Procter & Gamble",
		/* 0221 */ "Hosiden Corporation",
		/* 0222 */ "Muzik LLC",
		/* 0223 */ "Misfit Wearables Corp",
		/* 0224 */ "Google",
		/* 0225 */ "Danlers Ltd",
		/* 0226 */ "Semilink Inc",
		/* 0227 */ "inMusic Brands, Inc",
		/* 0228 */ "L.S. Research Inc.",
		/* 0229 */ "Eden Software Consultants Ltd.",
		/* 0230 */ "Freshtemp",
		/* 0231 */ "KS Technologies",
		/* 0232 */ "ACTS Technologies",
		/* 0233 */ "Vtrack Systems",
		/* 0234 */ "Nielsen-Kellerman Company",
		/* 0235 */ "Server Technology Inc.",
		/* 0236 */ "BioResearch Associates",
		/* 0237 */ "Jolly Logic, LLC",
		/* 0238 */ "Above Average Outcomes, Inc.",
		/* 0239 */ "Bitsplitters GmbH",
		/* 0240 */ "PayPal, Inc.",
		/* 0241 */ "Witron Technology Limited",
		/* 0242 */ "Morse Project Inc.",
		/* 0243 */ "Kent Displays Inc.",
		/* 0244 */ "Nautilus Inc.",
		/* 0245 */ "Smartifier Oy",
		/* 0246 */ "Elcometer Limited",
		/* 0247 */ "VSN Technologies, Inc.",
		/* 0248 */ "AceUni Corp., Ltd.",
		/* 0249 */ "StickNFind",
		/* 0250 */ "Crystal Code AB",
		/* 0251 */ "KOUKAAM a.s.",
		/* 0252 */ "Delphi Corporation",
		/* 0253 */ "ValenceTech Limited",
		/* 0254 */ "Stanley Black and Decker",
		/* 0255 */ "Typo Products, LLC",
		/* 0256 */ "TomTom International BV",
		/* 0257 */ "Fugoo, Inc.",
		/* 0258 */ "Keiser Corporation",
		/* 0259 */ "Bang & Olufsen A/S",
		/* 0260 */ "PLUS Location Systems Pty Ltd",
		/* 0261 */ "Ubiquitous Computing Technology Corporation",
		/* 0262 */ "Innovative Yachtter Solutions",
		/* 0263 */ "William Demant Holding A/S",
		/* 0264 */ "Chicony Electronics Co., Ltd.",
		/* 0265 */ "Atus BV",
		/* 0266 */ "Codegate Ltd",
		/* 0267 */ "ERi, Inc",
		/* 0268 */ "Transducers Direct, LLC",
		/* 0269 */ "DENSO TEN LIMITED (formerly Fujitsu Ten LImited)",
		/* 0270 */ "Audi AG",
		/* 0271 */ "HiSilicon Technologies CO., LIMITED",
		/* 0272 */ "Nippon Seiki Co., Ltd.",
		/* 0273 */ "Steelseries ApS",
		/* 0274 */ "Visybl Inc.",
		/* 0275 */ "Openbrain Technologies, Co., Ltd.",
		/* 0276 */ "Xensr",
		/* 0277 */ "e.solutions",
		/* 0278 */ "10AK Technologies",
		/* 0279 */ "Wimoto Technologies Inc",
		/* 0280 */ "Radius Networks, Inc.",
		/* 0281 */ "Wize Technology Co., Ltd.",
		/* 0282 */ "Qualcomm Labs, Inc.",
		/* 0283 */ "Hewlett Packard Enterprise",
		/* 0284 */ "Baidu",
		/* 0285 */ "Arendi AG",
		/* 0286 */ "Skoda Auto a.s.",
		/* 0287 */ "Volkswagen AG",
		/* 0288 */ "Porsche AG",
		/* 0289 */ "Sino Wealth Electronic Ltd.",
		/* 0290 */ "AirTurn, Inc.",
		/* 0291 */ "Kinsa, Inc",
		/* 0292 */ "HID Global",
		/* 0293 */ "SEAT es",
		/* 0294 */ "Promethean Ltd.",
		/* 0295 */ "Salutica Allied Solutions",
		/* 0296 */ "GPSI Group Pty Ltd",
		/* 0297 */ "Nimble Devices Oy",
		/* 0298 */ "Changzhou Yongse Infotech Co., Ltd.",
		/* 0299 */ "SportIQ",
		/* 0300 */ "TEMEC Instruments B.V.",
		/* 0301 */ "Sony Corporation",
		/* 0302 */ "ASSA ABLOY",
		/* 0303 */ "Clarion Co. Inc.",
		/* 0304 */ "Warehouse Innovations",
		/* 0305 */ "Cypress Semiconductor",
		/* 0306 */ "MADS Inc",
		/* 0307 */ "Blue Maestro Limited",
		/* 0308 */ "Resolution Products, Ltd.",
		/* 0309 */ "Aireware LLC",
		/* 0310 */ "Silvair, Inc.",
		/* 0311 */ "Prestigio Plaza Ltd.",
		/* 0312 */ "NTEO Inc.",
		/* 0313 */ "Focus Systems Corporation",
		/* 0314 */ "Tencent Holdings Ltd.",
		/* 0315 */ "Allegion",
		/* 0316 */ "Murata Manufacturing Co., Ltd.",
		/* 0317 */ "WirelessWERX",
		/* 0318 */ "Nod, Inc.",
		/* 0319 */ "B&B Manufacturing Company",
		/* 0320 */ "Alpine Electronics (China) Co., Ltd",
		/* 0321 */ "FedEx Services",
		/* 0322 */ "Grape Systems Inc.",
		/* 0323 */ "Bkon Connect",
		/* 0324 */ "Lintech GmbH",
		/* 0325 */ "Novatel Wireless",
		/* 0326 */ "Ciright",
		/* 0327 */ "Mighty Cast, Inc.",
		/* 0328 */ "Ambimat Electronics",
		/* 0329 */ "Perytons Ltd.",
		/* 0330 */ "Tivoli Audio, LLC",
		/* 0331 */ "Master Lock",
		/* 0332 */ "Mesh-Net Ltd",
		/* 0333 */ "HUIZHOU DESAY SV AUTOMOTIVE CO., LTD.",
		/* 0334 */ "Tangerine, Inc.",
		/* 0335 */ "B&W Group Ltd.",
		/* 0336 */ "Pioneer Corporation",
		/* 0337 */ "OnBeep",
		/* 0338 */ "Vernier Software & Technology",
		/* 0339 */ "ROL Ergo",
		/* 0340 */ "Pebble Technology",
		/* 0341 */ "NETATMO",
		/* 0342 */ "Accumulate AB",
		/* 0343 */ "Anhui Huami Information Technology Co., Ltd.",
		/* 0344 */ "Inmite s.r.o.",
		/* 0345 */ "ChefSteps, Inc.",
		/* 0346 */ "micas AG",
		/* 0347 */ "Biomedical Research Ltd.",
		/* 0348 */ "Pitius Tec S.L.",
		/* 0349 */ "Estimote, Inc.",
		/* 0350 */ "Unikey Technologies, Inc.",
		/* 0351 */ "Timer Cap Co.",
		/* 0352 */ "AwoX",
		/* 0353 */ "yikes",
		/* 0354 */ "MADSGlobalNZ Ltd.",
		/* 0355 */ "PCH International",
		/* 0356 */ "Qingdao Yeelink Information Technology Co., Ltd.",
		/* 0357 */ "Milwaukee Tool (Formally Milwaukee Electric Tools)",
		/* 0358 */ "MISHIK Pte Ltd",
		/* 0359 */ "Ascensia Diabetes Care US Inc.",
		/* 0360 */ "Spicebox LLC",
		/* 0361 */ "emberlight",
		/* 0362 */ "Cooper-Atkins Corporation",
		/* 0363 */ "Qblinks",
		/* 0364 */ "MYSPHERA",
		/* 0365 */ "LifeScan Inc",
		/* 0366 */ "Volantic AB",
		/* 0367 */ "Podo Labs, Inc",
		/* 0368 */ "Roche Diabetes Care AG",
		/* 0369 */ "Amazon.com Services, LLC (formerly Amazon Fulfillment Service)",
		/* 0370 */ "Connovate Technology Private Limited",
		/* 0371 */ "Kocomojo, LLC",
		/* 0372 */ "Everykey Inc.",
		/* 0373 */ "Dynamic Controls",
		/* 0374 */ "SentriLock",
		/* 0375 */ "I-SYST inc.",
		/* 0376 */ "CASIO COMPUTER CO., LTD.",
		/* 0377 */ "LAPIS Semiconductor Co., Ltd.",
		/* 0378 */ "Telemonitor, Inc.",
		/* 0379 */ "taskit GmbH",
		/* 0380 */ "Daimler AG",
		/* 0381 */ "BatAndCat",
		/* 0382 */ "BluDotz Ltd",
		/* 0383 */ "XTel Wireless ApS",
		/* 0384 */ "Gigaset Communications GmbH",
		/* 0385 */ "Gecko Health Innovations, Inc.",
		/* 0386 */ "HOP Ubiquitous",
		/* 0387 */ "Walt Disney",
		/* 0388 */ "Nectar",
		/* 0389 */ "bel'apps LLC",
		/* 0390 */ "CORE Lighting Ltd",
		/* 0391 */ "Seraphim Sense Ltd",
		/* 0392 */ "Unico RBC",
		/* 0393 */ "Physical Enterprises Inc.",
		/* 0394 */ "Able Trend Technology Limited",
		/* 0395 */ "Konica Minolta, Inc.",
		/* 0396 */ "Wilo SE",
		/* 0397 */ "Extron Design Services",
		/* 0398 */ "Fitbit, Inc.",
		/* 0399 */ "Fireflies Systems",
		/* 0400 */ "Intelletto Technologies Inc.",
		/* 0401 */ "FDK CORPORATION",
		/* 0402 */ "Cloudleaf, Inc",
		/* 0403 */ "Maveric Automation LLC",
		/* 0404 */ "Acoustic Stream Corporation",
		/* 0405 */ "Zuli",
		/* 0406 */ "Paxton Access Ltd",
		/* 0407 */ "WiSilica Inc.",
		/* 0408 */ "VENGIT Korlatolt Felelossegu Tarsasag",
		/* 0409 */ "SALTO SYSTEMS S.L.",
		/* 0410 */ "TRON Forum (formerly T-Engine Forum)",
		/* 0411 */ "CUBETECH s.r.o.",
		/* 0412 */ "Cokiya Incorporated",
		/* 0413 */ "CVS Health",
		/* 0414 */ "Ceruus",
		/* 0415 */ "Strainstall Ltd",
		/* 0416 */ "Channel Enterprises (HK) Ltd.",
		/* 0417 */ "FIAMM",
		/* 0418 */ "GIGALANE.CO.,LTD",
		/* 0419 */ "EROAD",
		/* 0420 */ "Mine Safety Appliances",
		/* 0421 */ "Icon Health and Fitness",
		/* 0422 */ "Wille Engineering (formely as Asandoo GmbH)",
		/* 0423 */ "ENERGOUS CORPORATION",
		/* 0424 */ "Taobao",
		/* 0425 */ "Canon Inc.",
		/* 0426 */ "Geophysical Technology Inc.",
		/* 0427 */ "Facebook, Inc.",
		/* 0428 */ "Trividia Health, Inc.",
		/* 0429 */ "FlightSafety International",
		/* 0430 */ "Earlens Corporation",
		/* 0431 */ "Sunrise Micro Devices, Inc.",
		/* 0432 */ "Star Micronics Co., Ltd.",
		/* 0433 */ "Netizens Sp. z o.o.",
		/* 0434 */ "Nymi Inc.",
		/* 0435 */ "Nytec, Inc.",
		/* 0436 */ "Trineo Sp. z o.o.",
		/* 0437 */ "Nest Labs Inc.",
		/* 0438 */ "LM Technologies Ltd",
		/* 0439 */ "General Electric Company",
		/* 0440 */ "i+D3 S.L.",
		/* 0441 */ "HANA Micron",
		/* 0442 */ "Stages Cycling LLC",
		/* 0443 */ "Cochlear Bone Anchored Solutions AB",
		/* 0444 */ "SenionLab AB",
		/* 0445 */ "Syszone Co., Ltd",
		/* 0446 */ "Pulsate Mobile Ltd.",
		/* 0447 */ "Hong Kong HunterSun Electronic Limited",
		/* 0448 */ "pironex GmbH",
		/* 0449 */ "BRADATECH Corp.",
		/* 0450 */ "Transenergooil AG",
		/* 0451 */ "Bunch",
		/* 0452 */ "DME Microelectronics",
		/* 0453 */ "Bitcraze AB",
		/* 0454 */ "HASWARE Inc.",
		/* 0455 */ "Abiogenix Inc.",
		/* 0456 */ "Poly-Control ApS",
		/* 0457 */ "Avi-on",
		/* 0458 */ "Laerdal Medical AS",
		/* 0459 */ "Fetch My Pet",
		/* 0460 */ "Sam Labs Ltd.",
		/* 0461 */ "Chengdu Synwing Technology Ltd",
		/* 0462 */ "HOUWA SYSTEM DESIGN, k.k.",
		/* 0463 */ "BSH",
		/* 0464 */ "Primus Inter Pares Ltd",
		/* 0465 */ "August Home, Inc",
		/* 0466 */ "Gill Electronics",
		/* 0467 */ "Sky Wave Design",
		/* 0468 */ "Newlab S.r.l.",
		/* 0469 */ "ELAD srl",
		/* 0470 */ "G-wearables inc.",
		/* 0471 */ "Squadrone Systems Inc.",
		/* 0472 */ "Code Corporation",
		/* 0473 */ "Savant Systems LLC",
		/* 0474 */ "Logitech International SA",
		/* 0475 */ "Innblue Consulting",
		/* 0476 */ "iParking Ltd.",
		/* 0477 */ "Koninklijke Philips Electronics N.V.",
		/* 0478 */ "Minelab Electronics Pty Limited",
		/* 0479 */ "Bison Group Ltd.",
		/* 0480 */ "Widex A/S",
		/* 0481 */ "Jolla Ltd",
		/* 0482 */ "Lectronix, Inc.",
		/* 0483 */ "Caterpillar Inc",
		/* 0484 */ "Freedom Innovations",
		/* 0485 */ "Dynamic Devices Ltd",
		/* 0486 */ "Technology Solutions (UK) Ltd",
		/* 0487 */ "IPS Group Inc.",
		/* 0488 */ "STIR",
		/* 0489 */ "Sano, Inc.",
		/* 0490 */ "Advanced Application Design, Inc.",
		/* 0491 */ "AutoMap LLC",
		/* 0492 */ "Spreadtrum Communications Shanghai Ltd",
		/* 0493 */ "CuteCircuit LTD",
		/* 0494 */ "Valeo Service",
		/* 0495 */ "Fullpower Technologies, Inc.",
		/* 0496 */ "KloudNation",
		/* 0497 */ "Zebra Technologies Corporation",
		/* 0498 */ "Itron, Inc.",
		/* 0499 */ "The University of Tokyo",
		/* 0500 */ "UTC Fire and Security",
		/* 0501 */ "Cool Webthings Limited",
		/* 0502 */ "DJO Global",
		/* 0503 */ "Gelliner Limited",
		/* 0504 */ "Anyka (Guangzhou) Microelectronics Technology Co, LTD",
		/* 0505 */ "Medtronic Inc.",
		/* 0506 */ "Gozio Inc.",
		/* 0507 */ "Form Lifting, LLC",
		/* 0508 */ "Wahoo Fitness, LLC",
		/* 0509 */ "Kontakt Micro-Location Sp. z o.o.",
		/* 0510 */ "Radio Systems Corporation",
		/* 0511 */ "Freescale Semiconductor, Inc.",
		/* 0512 */ "Verifone Systems Pte Ltd. Taiwan Branch",
		/* 0513 */ "AR Timing",
		/* 0514 */ "Rigado LLC",
		/* 0515 */ "Kemppi Oy",
		/* 0516 */ "Tapcentive Inc.",
		/* 0517 */ "Smartbotics Inc.",
		/* 0518 */ "Otter Products, LLC",
		/* 0519 */ "STEMP Inc.",
		/* 0520 */ "LumiGeek LLC",
		/* 0521 */ "InvisionHeart Inc.",
		/* 0522 */ "Macnica Inc.",
		/* 0523 */ "Jaguar Land Rover Limited",
		/* 0524 */ "CoroWare Technologies, Inc",
		/* 0525 */ "Simplo Technology Co., LTD",
		/* 0526 */ "Omron Healthcare Co., LTD",
		/* 0527 */ "Comodule GMBH",
		/* 0528 */ "ikeGPS",
		/* 0529 */ "Telink Semiconductor Co. Ltd",
		/* 0530 */ "Interplan Co., Ltd",
		/* 0531 */ "Wyler AG",
		/* 0532 */ "IK Multimedia Production srl",
		/* 0533 */ "Lukoton Experience Oy",
		/* 0534 */ "MTI Ltd",
		/* 0535 */ "Tech4home, Lda",
		/* 0536 */ "Hiotech AB",
		/* 0537 */ "DOTT Limited",
		/* 0538 */ "Blue Speck Labs, LLC",
		/* 0539 */ "Cisco Systems, Inc",
		/* 0540 */ "Mobicomm Inc",
		/* 0541 */ "Edamic",
		/* 0542 */ "Goodnet, Ltd",
		/* 0543 */ "Luster Leaf Products Inc",
		/* 0544 */ "Manus Machina BV",
		/* 0545 */ "Mobiquity Networks Inc",
		/* 0546 */ "Praxis Dynamics",
		/* 0547 */ "Philip Morris Products S.A.",
		/* 0548 */ "Comarch SA",
		/* 0549 */ "Nestlé Nespresso S.A.",
		/* 0550 */ "Merlinia A/S",
		/* 0551 */ "LifeBEAM Technologies",
		/* 0552 */ "Twocanoes Labs, LLC",
		/* 0553 */ "Muoverti Limited",
		/* 0554 */ "Stamer Musikanlagen GMBH",
		/* 0555 */ "Tesla Motors",
		/* 0556 */ "Pharynks Corporation",
		/* 0557 */ "Lupine",
		/* 0558 */ "Siemens AG",
		/* 0559 */ "Huami (Shanghai) Culture Communication CO., LTD",
		/* 0560 */ "Foster Electric Company, Ltd",
		/* 0561 */ "ETA SA",
		/* 0562 */ "x-Senso Solutions Kft",
		/* 0563 */ "Shenzhen SuLong Communication Ltd",
		/* 0564 */ "FengFan (BeiJing) Technology Co, Ltd",
		/* 0565 */ "Qrio Inc",
		/* 0566 */ "Pitpatpet Ltd",
		/* 0567 */ "MSHeli s.r.l.",
		/* 0568 */ "Trakm8 Ltd",
		/* 0569 */ "JIN CO, Ltd",
		/* 0570 */ "Alatech Tehnology",
		/* 0571 */ "Beijing CarePulse Electronic Technology Co, Ltd",
		/* 0572 */ "Awarepoint",
		/* 0573 */ "ViCentra B.V.",
		/* 0574 */ "Raven Industries",
		/* 0575 */ "WaveWare Technologies Inc.",
		/* 0576 */ "Argenox Technologies",
		/* 0577 */ "Bragi GmbH",
		/* 0578 */ "16Lab Inc",
		/* 0579 */ "Masimo Corp",
		/* 0580 */ "Iotera Inc",
		/* 0581 */ "Endress+Hauser",
		/* 0582 */ "ACKme Networks, Inc.",
		/* 0583 */ "FiftyThree Inc.",
		/* 0584 */ "Parker Hannifin Corp",
		/* 0585 */ "Transcranial Ltd",
		/* 0586 */ "Uwatec AG",
		/* 0587 */ "Orlan LLC",
		/* 0588 */ "Blue Clover Devices",
		/* 0589 */ "M-Way Solutions GmbH",
		/* 0590 */ "Microtronics Engineering GmbH",
		/* 0591 */ "Schneider Schreibgeräte GmbH",
		/* 0592 */ "Sapphire Circuits LLC",
		/* 0593 */ "Lumo Bodytech Inc.",
		/* 0594 */ "UKC Technosolution",
		/* 0595 */ "Xicato Inc.",
		/* 0596 */ "Playbrush",
		/* 0597 */ "Dai Nippon Printing Co., Ltd.",
		/* 0598 */ "G24 Power Limited",
		/* 0599 */ "AdBabble Local Commerce Inc.",
		/* 0600 */ "Devialet SA",
		/* 0601 */ "ALTYOR",
		/* 0602 */ "University of Applied Sciences Valais/Haute Ecole Valaisanne",
		/* 0603 */ "Five Interactive, LLC dba Zendo",
		/* 0604 */ "NetEase (Hangzhou) Network co.Ltd.",
		/* 0605 */ "Lexmark International Inc.",
		/* 0606 */ "Fluke Corporation",
		/* 0607 */ "Yardarm Technologies",
		/* 0608 */ "SensaRx",
		/* 0609 */ "SECVRE GmbH",
		/* 0610 */ "Glacial Ridge Technologies",
		/* 0611 */ "Identiv, Inc.",
		/* 0612 */ "DDS, Inc.",
		/* 0613 */ "SMK Corporation",
		/* 0614 */ "Schawbel Technologies LLC",
		/* 0615 */ "XMI Systems SA",
		/* 0616 */ "Cerevo",
		/* 0617 */ "Torrox GmbH & Co KG",
		/* 0618 */ "Gemalto",
		/* 0619 */ "DEKA Research & Development Corp.",
		/* 0620 */ "Domster Tadeusz Szydlowski",
		/* 0621 */ "Technogym SPA",
		/* 0622 */ "FLEURBAEY BVBA",
		/* 0623 */ "Aptcode Solutions",
		/* 0624 */ "LSI ADL Technology",
		/* 0625 */ "Animas Corp",
		/* 0626 */ "Alps Electric Co., Ltd.",
		/* 0627 */ "OCEASOFT",
		/* 0628 */ "Motsai Research",
		/* 0629 */ "Geotab",
		/* 0630 */ "E.G.O. Elektro-Geraetebau GmbH",
		/* 0631 */ "bewhere inc",
		/* 0632 */ "Johnson Outdoors Inc",
		/* 0633 */ "steute Schaltgerate GmbH & Co. KG",
		/* 0634 */ "Ekomini inc.",
		/* 0635 */ "DEFA AS",
		/* 0636 */ "Aseptika Ltd",
		/* 0637 */ "HUAWEI Technologies Co., Ltd.",
		/* 0638 */ "HabitAware, LLC",
		/* 0639 */ "ruwido austria gmbh",
		/* 0640 */ "ITEC corporation",
		/* 0641 */ "StoneL",
		/* 0642 */ "Sonova AG",
		/* 0643 */ "Maven Machines, Inc.",
		/* 0644 */ "Synapse Electronics",
		/* 0645 */ "Standard Innovation Inc.",
		/* 0646 */ "RF Code, Inc.",
		/* 0647 */ "Wally Ventures S.L.",
		/* 0648 */ "Willowbank Electronics Ltd",
		/* 0649 */ "SK Telecom",
		/* 0650 */ "Jetro AS",
		/* 0651 */ "Code Gears LTD",
		/* 0652 */ "NANOLINK APS",
		/* 0653 */ "IF, LLC",
		/* 0654 */ "RF Digital Corp",
		/* 0655 */ "Church & Dwight Co., Inc",
		/* 0656 */ "Multibit Oy",
		/* 0657 */ "CliniCloud Inc",
		/* 0658 */ "SwiftSensors",
		/* 0659 */ "Blue Bite",
		/* 0660 */ "ELIAS GmbH",
		/* 0661 */ "Sivantos GmbH",
		/* 0662 */ "Petzl",
		/* 0663 */ "storm power ltd",
		/* 0664 */ "EISST Ltd",
		/* 0665 */ "Inexess Technology Simma KG",
		/* 0666 */ "Currant, Inc.",
		/* 0667 */ "C2 Development, Inc.",
		/* 0668 */ "Blue Sky Scientific, LLC",
		/* 0669 */ "ALOTTAZS LABS, LLC",
		/* 0670 */ "Kupson spol. s r.o.",
		/* 0671 */ "Areus Engineering GmbH",
		/* 0672 */ "Impossible Camera GmbH",
		/* 0673 */ "InventureTrack Systems",
		/* 0674 */ "LockedUp",
		/* 0675 */ "Itude",
		/* 0676 */ "Pacific Lock Company",
		/* 0677 */ "Tendyron Corporation",
		/* 0678 */ "Robert Bosch GmbH",
		/* 0679 */ "Illuxtron international B.V.",
		/* 0680 */ "miSport Ltd.",
		/* 0681 */ "Chargelib",
		/* 0682 */ "Doppler Lab",
		/* 0683 */ "BBPOS Limited",
		/* 0684 */ "RTB Elektronik GmbH & Co. KG",
		/* 0685 */ "Rx Networks, Inc.",
		/* 0686 */ "WeatherFlow, Inc.",
		/* 0687 */ "Technicolor USA Inc.",
		/* 0688 */ "Bestechnic(Shanghai),Ltd",
		/* 0689 */ "Raden Inc",
		/* 0690 */ "JouZen Oy",
		/* 0691 */ "CLABER S.P.A.",
		/* 0692 */ "Hyginex, Inc.",
		/* 0693 */ "HANSHIN ELECTRIC RAILWAY CO.,LTD.",
		/* 0694 */ "Schneider Electric",
		/* 0695 */ "Oort Technologies LLC",
		/* 0696 */ "Chrono Therapeutics",
		/* 0697 */ "Rinnai Corporation",
		/* 0698 */ "Swissprime Technologies AG",
		/* 0699 */ "Koha.,Co.Ltd",
		/* 0700 */ "Genevac Ltd",
		/* 0701 */ "Chemtronics",
		/* 0702 */ "Seguro Technology Sp. z o.o.",
		/* 0703 */ "Redbird Flight Simulations",
		/* 0704 */ "Dash Robotics",
		/* 0705 */ "LINE Corporation",
		/* 0706 */ "Guillemot Corporation",
		/* 0707 */ "Techtronic Power Tools Technology Limited",
		/* 0708 */ "Wilson Sporting Goods",
		/* 0709 */ "Lenovo (Singapore) Pte Ltd.",
		/* 0710 */ "Ayatan Sensors",
		/* 0711 */ "Electronics Tomorrow Limited",
		/* 0712 */ "VASCO Data Security International, Inc.",
		/* 0713 */ "PayRange Inc.",
		/* 0714 */ "ABOV Semiconductor",
		/* 0715 */ "AINA-Wireless Inc.",
		/* 0716 */ "Eijkelkamp Soil & Water",
		/* 0717 */ "BMA ergonomics b.v.",
		/* 0718 */ "Teva Branded Pharmaceutical Products R&D, Inc.",
		/* 0719 */ "Anima",
		/* 0720 */ "3M",
		/* 0721 */ "Empatica Srl",
		/* 0722 */ "Afero, Inc.",
		/* 0723 */ "Powercast Corporation",
		/* 0724 */ "Secuyou ApS",
		/* 0725 */ "OMRON Corporation",
		/* 0726 */ "Send Solutions",
		/* 0727 */ "NIPPON SYSTEMWARE CO.,LTD.",
		/* 0728 */ "Neosfar",
		/* 0729 */ "Fliegl Agrartechnik GmbH",
		/* 0730 */ "Gilvader",
		/* 0731 */ "Digi International Inc (R)",
		/* 0732 */ "DeWalch Technologies, Inc.",
		/* 0733 */ "Flint Rehabilitation Devices, LLC",
		/* 0734 */ "Samsung SDS Co., Ltd.",
		/* 0735 */ "Blur Product Development",
		/* 0736 */ "University of Michigan",
		/* 0737 */ "Victron Energy BV",
		/* 0738 */ "NTT docomo",
		/* 0739 */ "Carmanah Technologies Corp.",
		/* 0740 */ "Bytestorm Ltd.",
		/* 0741 */ "Espressif Incorporated",
		/* 0742 */ "Unwire",
		/* 0743 */ "Connected Yard, Inc.",
		/* 0744 */ "American Music Environments",
		/* 0745 */ "Sensogram Technologies, Inc.",
		/* 0746 */ "Fujitsu Limited",
		/* 0747 */ "Ardic Technology",
		/* 0748 */ "Delta Systems, Inc",
		/* 0749 */ "HTC Corporation",
		/* 0750 */ "Citizen Holdings Co., Ltd.",
		/* 0751 */ "SMART-INNOVATION.inc",
		/* 0752 */ "Blackrat Software",
		/* 0753 */ "The Idea Cave, LLC",
		/* 0754 */ "GoPro, Inc.",
		/* 0755 */ "AuthAir, Inc",
		/* 0756 */ "Vensi, Inc.",
		/* 0757 */ "Indagem Tech LLC",
		/* 0758 */ "Intemo Technologies",
		/* 0759 */ "DreamVisions co., Ltd.",
		/* 0760 */ "Runteq Oy Ltd",
		/* 0761 */ "IMAGINATION TECHNOLOGIES LTD",
		/* 0762 */ "CoSTAR TEchnologies",
		/* 0763 */ "Clarius Mobile Health Corp.",
		/* 0764 */ "Shanghai Frequen Microelectronics Co., Ltd.",
		/* 0765 */ "Uwanna, Inc.",
		/* 0766 */ "Lierda Science & Technology Group Co., Ltd.",
		/* 0767 */ "Silicon Laboratories",
		/* 0768 */ "World Moto Inc.",
		/* 0769 */ "Giatec Scientific Inc.",
		/* 0770 */ "Loop Devices, Inc",
		/* 0771 */ "IACA electronique",
		/* 0772 */ "Proxy Technologies, Inc.",
		/* 0773 */ "Swipp ApS",
		/* 0774 */ "Life Laboratory Inc.",
		/* 0775 */ "FUJI INDUSTRIAL CO.,LTD.",
		/* 0776 */ "Surefire, LLC",
		/* 0777 */ "Dolby Labs",
		/* 0778 */ "Ellisys",
		/* 0779 */ "Magnitude Lighting Converters",
		/* 0780 */ "Hilti AG",
		/* 0781 */ "Devdata S.r.l.",
		/* 0782 */ "Deviceworx",
		/* 0783 */ "Shortcut Labs",
		/* 0784 */ "SGL Italia S.r.l.",
		/* 0785 */ "PEEQ DATA",
		/* 0786 */ "Ducere Technologies Pvt Ltd",
		/* 0787 */ "DiveNav, Inc.",
		/* 0788 */ "RIIG AI Sp. z o.o.",
		/* 0789 */ "Thermo Fisher Scientific",
		/* 0790 */ "AG Measurematics Pvt. Ltd.",
		/* 0791 */ "CHUO Electronics CO., LTD.",
		/* 0792 */ "Aspenta International",
		/* 0793 */ "Eugster Frismag AG",
		/* 0794 */ "Amber wireless GmbH",
		/* 0795 */ "HQ Inc",
		/* 0796 */ "Lab Sensor Solutions",
		/* 0797 */ "Enterlab ApS",
		/* 0798 */ "Eyefi, Inc.",
		/* 0799 */ "MetaSystem S.p.A.",
		/* 0800 */ "SONO ELECTRONICS. CO., LTD",
		/* 0801 */ "Jewelbots",
		/* 0802 */ "Compumedics Limited",
		/* 0803 */ "Rotor Bike Components",
		/* 0804 */ "Astro, Inc.",
		/* 0805 */ "Amotus Solutions",
		/* 0806 */ "Healthwear Technologies (Changzhou)Ltd",
		/* 0807 */ "Essex Electronics",
		/* 0808 */ "Grundfos A/S",
		/* 0809 */ "Eargo, Inc.",
		/* 0810 */ "Electronic Design Lab",
		/* 0811 */ "ESYLUX",
		/* 0812 */ "NIPPON SMT.CO.,Ltd",
		/* 0813 */ "BM innovations GmbH",
		/* 0814 */ "indoormap",
		/* 0815 */ "OttoQ Inc",
		/* 0816 */ "North Pole Engineering",
		/* 0817 */ "3flares Technologies Inc.",
		/* 0818 */ "Electrocompaniet A.S.",
		/* 0819 */ "Mul-T-Lock",
		/* 0820 */ "Corentium AS",
		/* 0821 */ "Enlighted Inc",
		/* 0822 */ "GISTIC",
		/* 0823 */ "AJP2 Holdings, LLC",
		/* 0824 */ "COBI GmbH",
		/* 0825 */ "Blue Sky Scientific, LLC",
		/* 0826 */ "Appception, Inc.",
		/* 0827 */ "Courtney Thorne Limited",
		/* 0828 */ "Virtuosys",
		/* 0829 */ "TPV Technology Limited",
		/* 0830 */ "Monitra SA",
		/* 0831 */ "Automation Components, Inc.",
		/* 0832 */ "Letsense s.r.l.",
		/* 0833 */ "Etesian Technologies LLC",
		/* 0834 */ "GERTEC BRASIL LTDA.",
		/* 0835 */ "Drekker Development Pty. Ltd.",
		/* 0836 */ "Whirl Inc",
		/* 0837 */ "Locus Positioning",
		/* 0838 */ "Acuity Brands Lighting, Inc",
		/* 0839 */ "Prevent Biometrics",
		/* 0840 */ "Arioneo",
		/* 0841 */ "VersaMe",
		/* 0842 */ "Vaddio",
		/* 0843 */ "Libratone A/S",
		/* 0844 */ "HM Electronics, Inc.",
		/* 0845 */ "TASER International, Inc.",
		/* 0846 */ "SafeTrust Inc.",
		/* 0847 */ "Heartland Payment Systems",
		/* 0848 */ "Bitstrata Systems Inc.",
		/* 0849 */ "Pieps GmbH",
		/* 0850 */ "iRiding(Xiamen)Technology Co.,Ltd.",
		/* 0851 */ "Alpha Audiotronics, Inc.",
		/* 0852 */ "TOPPAN FORMS CO.,LTD.",
		/* 0853 */ "Sigma Designs, Inc.",
		/* 0854 */ "Spectrum Brands, Inc.",
		/* 0855 */ "Polymap Wireless",
		/* 0856 */ "MagniWare Ltd.",
		/* 0857 */ "Novotec Medical GmbH",
		/* 0858 */ "Medicom Innovation Partner a/s",
		/* 0859 */ "Matrix Inc.",
		/* 0860 */ "Eaton Corporation",
		/* 0861 */ "KYS",
		/* 0862 */ "Naya Health, Inc.",
		/* 0863 */ "Acromag",
		/* 0864 */ "Insulet Corporation",
		/* 0865 */ "Wellinks Inc.",
		/* 0866 */ "ON Semiconductor",
		/* 0867 */ "FREELAP SA",
		/* 0868 */ "Favero Electronics Srl",
		/* 0869 */ "BioMech Sensor LLC",
		/* 0870 */ "BOLTT Sports technologies Private limited",
		/* 0871 */ "Saphe International",
		/* 0872 */ "Metormote AB",
		/* 0873 */ "littleBits",
		/* 0874 */ "SetPoint Medical",
		/* 0875 */ "BRControls Products BV",
		/* 0876 */ "Zipcar",
		/* 0877 */ "AirBolt Pty Ltd",
		/* 0878 */ "KeepTruckin Inc",
		/* 0879 */ "Motiv, Inc.",
		/* 0880 */ "Wazombi Labs OU",
		/* 0881 */ "ORBCOMM",
		/* 0882 */ "Nixie Labs, Inc.",
		/* 0883 */ "AppNearMe Ltd",
		/* 0884 */ "Holman Industries",
		/* 0885 */ "Expain AS",
		/* 0886 */ "Electronic Temperature Instruments Ltd",
		/* 0887 */ "Plejd AB",
		/* 0888 */ "Propeller Health",
		/* 0889 */ "Shenzhen iMCO Electronic Technology Co.,Ltd",
		/* 0890 */ "Algoria",
		/* 0891 */ "Apption Labs Inc.",
		/* 0892 */ "Cronologics Corporation",
		/* 0893 */ "MICRODIA Ltd.",
		/* 0894 */ "lulabytes S.L.",
		/* 0895 */ "Societe des Produits Nestle S.A. (formerly Nestec S.A.)",
		/* 0896 */ "LLC \"MEGA-F service\"",
		/* 0897 */ "Sharp Corporation",
		/* 0898 */ "Precision Outcomes Ltd",
		/* 0899 */ "Kronos Incorporated",
		/* 0900 */ "OCOSMOS Co., Ltd.",
		/* 0901 */ "Embedded Electronic Solutions Ltd. dba e2Solutions",
		/* 0902 */ "Aterica Inc.",
		/* 0903 */ "BluStor PMC, Inc.",
		/* 0904 */ "Kapsch TrafficCom AB",
		/* 0905 */ "ActiveBlu Corporation",
		/* 0906 */ "Kohler Mira Limited",
		/* 0907 */ "Noke",
		/* 0908 */ "Appion Inc.",
		/* 0909 */ "Resmed Ltd",
		/* 0910 */ "Crownstone B.V.",
		/* 0911 */ "Xiaomi Inc.",
		/* 0912 */ "INFOTECH s.r.o.",
		/* 0913 */ "Thingsquare AB",
		/* 0914 */ "T&D",
		/* 0915 */ "LAVAZZA S.p.A.",
		/* 0916 */ "Netclearance Systems, Inc.",
		/* 0917 */ "SDATAWAY",
		/* 0918 */ "BLOKS GmbH",
		/* 0919 */ "LEGO System A/S",
		/* 0920 */ "Thetatronics Ltd",
		/* 0921 */ "Nikon Corporation",
		/* 0922 */ "NeST",
		/* 0923 */ "South Silicon Valley Microelectronics",
		/* 0924 */ "ALE International",
		/* 0925 */ "CareView Communications, Inc.",
		/* 0926 */ "SchoolBoard Limited",
		/* 0927 */ "Molex Corporation",
		/* 0928 */ "IVT Wireless Limited",
		/* 0929 */ "Alpine Labs LLC",
		/* 0930 */ "Candura Instruments",
		/* 0931 */ "SmartMovt Technology Co., Ltd",
		/* 0932 */ "Token Zero Ltd",
		/* 0933 */ "ACE CAD Enterprise Co., Ltd. (ACECAD)",
		/* 0934 */ "Medela, Inc",
		/* 0935 */ "AeroScout",
		/* 0936 */ "Esrille Inc.",
		/* 0937 */ "THINKERLY SRL",
		/* 0938 */ "Exon Sp. z o.o.",
		/* 0939 */ "Meizu Technology Co., Ltd.",
		/* 0940 */ "Smablo LTD",
		/* 0941 */ "XiQ",
		/* 0942 */ "Allswell Inc.",
		/* 0943 */ "Comm-N-Sense Corp DBA Verigo",
		/* 0944 */ "VIBRADORM GmbH",
		/* 0945 */ "Otodata Wireless Network Inc.",
		/* 0946 */ "Propagation Systems Limited",
		/* 0947 */ "Midwest Instruments & Controls",
		/* 0948 */ "Alpha Nodus, inc.",
		/* 0949 */ "petPOMM, Inc",
		/* 0950 */ "Mattel",
		/* 0951 */ "Airbly Inc.",
		/* 0952 */ "A-Safe Limited",
		/* 0953 */ "FREDERIQUE CONSTANT SA",
		/* 0954 */ "Maxscend Microelectronics Company Limited",
		/* 0955 */ "Abbott",
		/* 0956 */ "ASB Bank Ltd",
		/* 0957 */ "amadas",
		/* 0958 */ "Applied Science, Inc.",
		/* 0959 */ "iLumi Solutions Inc.",
		/* 0960 */ "Arch Systems Inc.",
		/* 0961 */ "Ember Technologies, Inc.",
		/* 0962 */ "Snapchat Inc",
		/* 0963 */ "Casambi Technologies Oy",
		/* 0964 */ "Pico Technology Inc.",
		/* 0965 */ "St. Jude Medical, Inc.",
		/* 0966 */ "Intricon",
		/* 0967 */ "Structural Health Systems, Inc.",
		/* 0968 */ "Avvel International",
		/* 0969 */ "Gallagher Group",
		/* 0970 */ "In2things Automation Pvt. Ltd.",
		/* 0971 */ "SYSDEV Srl",
		/* 0972 */ "Vonkil Technologies Ltd",
		/* 0973 */ "Wynd Technologies, Inc.",
		/* 0974 */ "CONTRINEX S.A.",
		/* 0975 */ "MIRA, Inc.",
		/* 0976 */ "Watteam Ltd",
		/* 0977 */ "Density Inc.",
		/* 0978 */ "IOT Pot India Private Limited",
		/* 0979 */ "Sigma Connectivity AB",
		/* 0980 */ "PEG PEREGO SPA",
		/* 0981 */ "Wyzelink Systems Inc.",
		/* 0982 */ "Yota Devices LTD",
		/* 0983 */ "FINSECUR",
		/* 0984 */ "Zen-Me Labs Ltd",
		/* 0985 */ "3IWare Co., Ltd.",
		/* 0986 */ "EnOcean GmbH",
		/* 0987 */ "Instabeat, Inc",
		/* 0988 */ "Nima Labs",
		/* 0989 */ "Andreas Stihl AG & Co. KG",
		/* 0990 */ "Nathan Rhoades LLC",
		/* 0991 */ "Grob Technologies, LLC",
		/* 0992 */ "Actions (Zhuhai) Technology Co., Limited",
		/* 0993 */ "SPD Development Company Ltd",
		/* 0994 */ "Sensoan Oy",
		/* 0995 */ "Qualcomm Life Inc",
		/* 0996 */ "Chip-ing AG",
		/* 0997 */ "ffly4u",
		/* 0998 */ "IoT Instruments Oy",
		/* 0999 */ "TRUE Fitness Technology",
		/* 1000 */ "Reiner Kartengeraete GmbH & Co. KG.",
		/* 1001 */ "SHENZHEN LEMONJOY TECHNOLOGY CO., LTD.",
		/* 1002 */ "Hello Inc.",
		/* 1003 */ "Evollve Inc.",
		/* 1004 */ "Jigowatts Inc.",
		/* 1005 */ "BASIC MICRO.COM,INC.",
		/* 1006 */ "CUBE TECHNOLOGIES",
		/* 1007 */ "foolography GmbH",
		/* 1008 */ "CLINK",
		/* 1009 */ "Hestan Smart Cooking Inc.",
		/* 1010 */ "WindowMaster A/S",
		/* 1011 */ "Flowscape AB",
		/* 1012 */ "PAL Technologies Ltd",
		/* 1013 */ "WHERE, Inc.",
		/* 1014 */ "Iton Technology Corp.",
		/* 1015 */ "Owl Labs Inc.",
		/* 1016 */ "Rockford Corp.",
		/* 1017 */ "Becon Technologies Co.,Ltd.",
		/* 1018 */ "Vyassoft Technologies Inc",
		/* 1019 */ "Nox Medical",
		/* 1020 */ "Kimberly-Clark",
		/* 1021 */ "Trimble Navigation Ltd.",
		/* 1022 */ "Littelfuse",
		/* 1023 */ "Withings",
		/* 1024 */ "i-developer IT Beratung UG",
		/* 1025 */ "Relations Inc.",
		/* 1026 */ "Sears Holdings Corporation",
		/* 1027 */ "Gantner Electronic GmbH",
		/* 1028 */ "Authomate Inc",
		/* 1029 */ "Vertex International, Inc.",
		/* 1030 */ "Airtago",
		/* 1031 */ "Swiss Audio SA",
		/* 1032 */ "ToGetHome Inc.",
		/* 1033 */ "AXIS",
		/* 1034 */ "Openmatics",
		/* 1035 */ "Jana Care Inc.",
		/* 1036 */ "Senix Corporation",
		/* 1037 */ "NorthStar Battery Company, LLC",
		/* 1038 */ "SKF (U.K.) Limited",
		/* 1039 */ "CO-AX Technology, Inc.",
		/* 1040 */ "Fender Musical Instruments",
		/* 1041 */ "Luidia Inc",
		/* 1042 */ "SEFAM",
		/* 1043 */ "Wireless Cables Inc",
		/* 1044 */ "Lightning Protection International Pty Ltd",
		/* 1045 */ "Uber Technologies Inc",
		/* 1046 */ "SODA GmbH",
		/* 1047 */ "Fatigue Science",
		/* 1048 */ "Alpine Electronics Inc.",
		/* 1049 */ "Novalogy LTD",
		/* 1050 */ "Friday Labs Limited",
		/* 1051 */ "OrthoAccel Technologies",
		/* 1052 */ "WaterGuru, Inc.",
		/* 1053 */ "Benning Elektrotechnik und Elektronik GmbH & Co. KG",
		/* 1054 */ "Dell Computer Corporation",
		/* 1055 */ "Kopin Corporation",
		/* 1056 */ "TecBakery GmbH",
		/* 1057 */ "Backbone Labs, Inc.",
		/* 1058 */ "DELSEY SA",
		/* 1059 */ "Chargifi Limited",
		/* 1060 */ "Trainesense Ltd.",
		/* 1061 */ "Unify Software and Solutions GmbH & Co. KG",
		/* 1062 */ "Husqvarna AB",
		/* 1063 */ "Focus fleet and fuel management inc",
		/* 1064 */ "SmallLoop, LLC",
		/* 1065 */ "Prolon Inc.",
		/* 1066 */ "BD Medical",
		/* 1067 */ "iMicroMed Incorporated",
		/* 1068 */ "Ticto N.V.",
		/* 1069 */ "Meshtech AS",
		/* 1070 */ "MemCachier Inc.",
		/* 1071 */ "Danfoss A/S",
		/* 1072 */ "SnapStyk Inc.",
		/* 1073 */ "Amway Corporation",
		/* 1074 */ "Silk Labs, Inc.",
		/* 1075 */ "Pillsy Inc.",
		/* 1076 */ "Hatch Baby, Inc.",
		/* 1077 */ "Blocks Wearables Ltd.",
		/* 1078 */ "Drayson Technologies (Europe) Limited",
		/* 1079 */ "eBest IOT Inc.",
		/* 1080 */ "Helvar Ltd",
		/* 1081 */ "Radiance Technologies",
		/* 1082 */ "Nuheara Limited",
		/* 1083 */ "Appside co., ltd.",
		/* 1084 */ "DeLaval",
		/* 1085 */ "Coiler Corporation",
		/* 1086 */ "Thermomedics, Inc.",
		/* 1087 */ "Tentacle Sync GmbH",
		/* 1088 */ "Valencell, Inc.",
		/* 1089 */ "iProtoXi Oy",
		/* 1090 */ "SECOM CO., LTD.",
		/* 1091 */ "Tucker International LLC",
		/* 1092 */ "Metanate Limited",
		/* 1093 */ "Kobian Canada Inc.",
		/* 1094 */ "NETGEAR, Inc.",
		/* 1095 */ "Fabtronics Australia Pty Ltd",
		/* 1096 */ "Grand Centrix GmbH",
		/* 1097 */ "1UP USA.com llc",
		/* 1098 */ "SHIMANO INC.",
		/* 1099 */ "Nain Inc.",
		/* 1100 */ "LifeStyle Lock, LLC",
		/* 1101 */ "VEGA Grieshaber KG",
		/* 1102 */ "Xtrava Inc.",
		/* 1103 */ "TTS Tooltechnic Systems AG & Co. KG",
		/* 1104 */ "Teenage Engineering AB",
		/* 1105 */ "Tunstall Nordic AB",
		/* 1106 */ "Svep Design Center AB",
		/* 1107 */ "GreenPeak Technologies BV",
		/* 1108 */ "Sphinx Electronics GmbH & Co KG",
		/* 1109 */ "Atomation",
		/* 1110 */ "Nemik Consulting Inc",
		/* 1111 */ "RF INNOVATION",
		/* 1112 */ "Mini Solution Co., Ltd.",
		/* 1113 */ "Lumenetix, Inc",
		/* 1114 */ "2048450 Ontario Inc",
		/* 1115 */ "SPACEEK LTD",
		/* 1116 */ "Delta T Corporation",
		/* 1117 */ "Boston Scientific Corporation",
		/* 1118 */ "Nuviz, Inc.",
		/* 1119 */ "Real Time Automation, Inc.",
		/* 1120 */ "Kolibree",
		/* 1121 */ "vhf elektronik GmbH",
		/* 1122 */ "Bonsai Systems GmbH",
		/* 1123 */ "Fathom Systems Inc.",
		/* 1124 */ "Bellman & Symfon",
		/* 1125 */ "International Forte Group LLC",
		/* 1126 */ "CycleLabs Solutions inc.",
		/* 1127 */ "Codenex Oy",
		/* 1128 */ "Kynesim Ltd",
		/* 1129 */ "Palago AB",
		/* 1130 */ "INSIGMA INC.",
		/* 1131 */ "PMD Solutions",
		/* 1132 */ "Qingdao Realtime Technology Co., Ltd.",
		/* 1133 */ "BEGA Gantenbrink-Leuchten KG",
		/* 1134 */ "Pambor Ltd.",
		/* 1135 */ "Develco Products A/S",
		/* 1136 */ "iDesign s.r.l.",
		/* 1137 */ "TiVo Corp",
		/* 1138 */ "Control-J Pty Ltd",
		/* 1139 */ "Steelcase, Inc.",
		/* 1140 */ "iApartment co., ltd.",
		/* 1141 */ "Icom inc.",
		/* 1142 */ "Oxstren Wearable Technologies Private Limited",
		/* 1143 */ "Blue Spark Technologies",
		/* 1144 */ "FarSite Communications Limited",
		/* 1145 */ "mywerk system GmbH",
		/* 1146 */ "Sinosun Technology Co., Ltd.",
		/* 1147 */ "MIYOSHI ELECTRONICS CORPORATION",
		/* 1148 */ "POWERMAT LTD",
		/* 1149 */ "Occly LLC",
		/* 1150 */ "OurHub Dev IvS",
		/* 1151 */ "Pro-Mark, Inc.",
		/* 1152 */ "Dynometrics Inc.",
		/* 1153 */ "Quintrax Limited",
		/* 1154 */ "POS Tuning Udo Vosshenrich GmbH & Co. KG",
		/* 1155 */ "Multi Care Systems B.V.",
		/* 1156 */ "Revol Technologies Inc",
		/* 1157 */ "SKIDATA AG",
		/* 1158 */ "DEV TECNOLOGIA INDUSTRIA, COMERCIO E MANUTENCAO DE EQUIPAMENTOS LTDA. - ME",
		/* 1159 */ "Centrica Connected Home",
		/* 1160 */ "Automotive Data Solutions Inc",
		/* 1161 */ "Igarashi Engineering",
		/* 1162 */ "Taelek Oy",
		/* 1163 */ "CP Electronics Limited",
		/* 1164 */ "Vectronix AG",
		/* 1165 */ "S-Labs Sp. z o.o.",
		/* 1166 */ "Companion Medical, Inc.",
		/* 1167 */ "BlueKitchen GmbH",
		/* 1168 */ "Matting AB",
		/* 1169 */ "SOREX - Wireless Solutions GmbH",
		/* 1170 */ "ADC Technology, Inc.",
		/* 1171 */ "Lynxemi Pte Ltd",
		/* 1172 */ "SENNHEISER electronic GmbH & Co. KG",
		/* 1173 */ "LMT Mercer Group, Inc",
		/* 1174 */ "Polymorphic Labs LLC",
		/* 1175 */ "Cochlear Limited",
		/* 1176 */ "METER Group, Inc. USA",
		/* 1177 */ "Ruuvi Innovations Ltd.",
		/* 1178 */ "Situne AS",
		/* 1179 */ "nVisti, LLC",
		/* 1180 */ "DyOcean",
		/* 1181 */ "Uhlmann & Zacher GmbH",
		/* 1182 */ "AND!XOR LLC",
		/* 1183 */ "tictote AB",
		/* 1184 */ "Vypin, LLC",
		/* 1185 */ "PNI Sensor Corporation",
		/* 1186 */ "ovrEngineered, LLC",
		/* 1187 */ "GT-tronics HK Ltd",
		/* 1188 */ "Herbert Waldmann GmbH & Co. KG",
		/* 1189 */ "Guangzhou FiiO Electronics Technology Co.,Ltd",
		/* 1190 */ "Vinetech Co., Ltd",
		/* 1191 */ "Dallas Logic Corporation",
		/* 1192 */ "BioTex, Inc.",
		/* 1193 */ "DISCOVERY SOUND TECHNOLOGY, LLC",
		/* 1194 */ "LINKIO SAS",
		/* 1195 */ "Harbortronics, Inc.",
		/* 1196 */ "Undagrid B.V.",
		/* 1197 */ "Shure Inc",
		/* 1198 */ "ERM Electronic Systems LTD",
		/* 1199 */ "BIOROWER Handelsagentur GmbH",
		/* 1200 */ "Weba Sport und Med. Artikel GmbH",
		/* 1201 */ "Kartographers Technologies Pvt. Ltd.",
		/* 1202 */ "The Shadow on the Moon",
		/* 1203 */ "mobike (Hong Kong) Limited",
		/* 1204 */ "Inuheat Group AB",
		/* 1205 */ "Swiftronix AB",
		/* 1206 */ "Diagnoptics Technologies",
		/* 1207 */ "Analog Devices, Inc.",
		/* 1208 */ "Soraa Inc.",
		/* 1209 */ "CSR Building Products Limited",
		/* 1210 */ "Crestron Electronics, Inc.",
		/* 1211 */ "Neatebox Ltd",
		/* 1212 */ "Draegerwerk AG & Co. KGaA",
		/* 1213 */ "AlbynMedical",
		/* 1214 */ "Averos FZCO",
		/* 1215 */ "VIT Initiative, LLC",
		/* 1216 */ "Statsports International",
		/* 1217 */ "Sospitas, s.r.o.",
		/* 1218 */ "Dmet Products Corp.",
		/* 1219 */ "Mantracourt Electronics Limited",
		/* 1220 */ "TeAM Hutchins AB",
		/* 1221 */ "Seibert Williams Glass, LLC",
		/* 1222 */ "Insta GmbH",
		/* 1223 */ "Svantek Sp. z o.o.",
		/* 1224 */ "Shanghai Flyco Electrical Appliance Co., Ltd.",
		/* 1225 */ "Thornwave Labs Inc",
		/* 1226 */ "Steiner-Optik GmbH",
		/* 1227 */ "Novo Nordisk A/S",
		/* 1228 */ "Enflux Inc.",
		/* 1229 */ "Safetech Products LLC",
		/* 1230 */ "GOOOLED S.R.L.",
		/* 1231 */ "DOM Sicherheitstechnik GmbH & Co. KG",
		/* 1232 */ "Olympus Corporation",
		/* 1233 */ "KTS GmbH",
		/* 1234 */ "Anloq Technologies Inc.",
		/* 1235 */ "Queercon, Inc",
		/* 1236 */ "5th Element Ltd",
		/* 1237 */ "Gooee Limited",
		/* 1238 */ "LUGLOC LLC",
		/* 1239 */ "Blincam, Inc.",
		/* 1240 */ "FUJIFILM Corporation",
		/* 1241 */ "RandMcNally",
		/* 1242 */ "Franceschi Marina snc",
		/* 1243 */ "Engineered Audio, LLC.",
		/* 1244 */ "IOTTIVE (OPC) PRIVATE LIMITED",
		/* 1245 */ "4MOD Technology",
		/* 1246 */ "Lutron Electronics Co., Inc.",
		/* 1247 */ "Emerson",
		/* 1248 */ "Guardtec, Inc.",
		/* 1249 */ "REACTEC LIMITED",
		/* 1250 */ "EllieGrid",
		/* 1251 */ "Under Armour",
		/* 1252 */ "Woodenshark",
		/* 1253 */ "Avack Oy",
		/* 1254 */ "Smart Solution Technology, Inc.",
		/* 1255 */ "REHABTRONICS INC.",
		/* 1256 */ "STABILO International",
		/* 1257 */ "Busch Jaeger Elektro GmbH",
		/* 1258 */ "Pacific Bioscience Laboratories, Inc",
		/* 1259 */ "Bird Home Automation GmbH",
		/* 1260 */ "Motorola Solutions",
		/* 1261 */ "R9 Technology, Inc.",
		/* 1262 */ "Auxivia",
		/* 1263 */ "DaisyWorks, Inc",
		/* 1264 */ "Kosi Limited",
		/* 1265 */ "Theben AG",
		/* 1266 */ "InDreamer Techsol Private Limited",
		/* 1267 */ "Cerevast Medical",
		/* 1268 */ "ZanCompute Inc.",
		/* 1269 */ "Pirelli Tyre S.P.A.",
		/* 1270 */ "McLear Limited",
		/* 1271 */ "Shenzhen Huiding Technology Co.,Ltd.",
		/* 1272 */ "Convergence Systems Limited",
		/* 1273 */ "Interactio",
		/* 1274 */ "Androtec GmbH",
		/* 1275 */ "Benchmark Drives GmbH & Co. KG",
		/* 1276 */ "SwingLync L. L. C.",
		/* 1277 */ "Tapkey GmbH",
		/* 1278 */ "Woosim Systems Inc.",
		/* 1279 */ "Microsemi Corporation",
		/* 1280 */ "Wiliot LTD.",
		/* 1281 */ "Polaris IND",
		/* 1282 */ "Specifi-Kali LLC",
		/* 1283 */ "Locoroll, Inc",
		/* 1284 */ "PHYPLUS Inc",
		/* 1285 */ "Inplay Technologies LLC",
		/* 1286 */ "Hager",
		/* 1287 */ "Yellowcog",
		/* 1288 */ "Axes System sp. z o. o.",
		/* 1289 */ "myLIFTER Inc.",
		/* 1290 */ "Shake-on B.V.",
		/* 1291 */ "Vibrissa Inc.",
		/* 1292 */ "OSRAM GmbH",
		/* 1293 */ "TRSystems GmbH",
		/* 1294 */ "Yichip Microelectronics (Hangzhou) Co.,Ltd.",
		/* 1295 */ "Foundation Engineering LLC",
		/* 1296 */ "UNI-ELECTRONICS, INC.",
		/* 1297 */ "Brookfield Equinox LLC",
		/* 1298 */ "Soprod SA",
		/* 1299 */ "9974091 Canada Inc.",
		/* 1300 */ "FIBRO GmbH",
		/* 1301 */ "RB Controls Co., Ltd.",
		/* 1302 */ "Footmarks",
		/* 1303 */ "Amtronic Sverige AB (formerly Amcore AB)",
		/* 1304 */ "MAMORIO.inc",
		/* 1305 */ "Tyto Life LLC",
		/* 1306 */ "Leica Camera AG",
		/* 1307 */ "Angee Technologies Ltd.",
		/* 1308 */ "EDPS",
		/* 1309 */ "OFF Line Co., Ltd.",
		/* 1310 */ "Detect Blue Limited",
		/* 1311 */ "Setec Pty Ltd",
		/* 1312 */ "Target Corporation",
		/* 1313 */ "IAI Corporation",
		/* 1314 */ "NS Tech, Inc.",
		/* 1315 */ "MTG Co., Ltd.",
		/* 1316 */ "Hangzhou iMagic Technology Co., Ltd",
		/* 1317 */ "HONGKONG NANO IC TECHNOLOGIES CO., LIMITED",
		/* 1318 */ "Honeywell International Inc.",
		/* 1319 */ "Albrecht JUNG",
		/* 1320 */ "Lunera Lighting Inc.",
		/* 1321 */ "Lumen UAB",
		/* 1322 */ "Keynes Controls Ltd",
		/* 1323 */ "Novartis AG",
		/* 1324 */ "Geosatis SA",
		/* 1325 */ "EXFO, Inc.",
		/* 1326 */ "LEDVANCE GmbH",
		/* 1327 */ "Center ID Corp.",
		/* 1328 */ "Adolene, Inc.",
		/* 1329 */ "D&M Holdings Inc.",
		/* 1330 */ "CRESCO Wireless, Inc.",
		/* 1331 */ "Nura Operations Pty Ltd",
		/* 1332 */ "Frontiergadget, Inc.",
		/* 1333 */ "Smart Component Technologies Limited",
		/* 1334 */ "ZTR Control Systems LLC",
		/* 1335 */ "MetaLogics Corporation",
		/* 1336 */ "Medela AG",
		/* 1337 */ "OPPLE Lighting Co., Ltd",
		/* 1338 */ "Savitech Corp.,",
		/* 1339 */ "prodigy",
		/* 1340 */ "Screenovate Technologies Ltd",
		/* 1341 */ "TESA SA",
		/* 1342 */ "CLIM8 LIMITED",
		/* 1343 */ "Silergy Corp",
		/* 1344 */ "SilverPlus, Inc",
		/* 1345 */ "Sharknet srl",
		/* 1346 */ "Mist Systems, Inc.",
		/* 1347 */ "MIWA LOCK CO.,Ltd",
		/* 1348 */ "OrthoSensor, Inc.",
		/* 1349 */ "Candy Hoover Group s.r.l",
		/* 1350 */ "Apexar Technologies S.A.",
		/* 1351 */ "LOGICDATA d.o.o.",
		/* 1352 */ "Knick Elektronische Messgeraete GmbH & Co. KG",
		/* 1353 */ "Smart Technologies and Investment Limited",
		/* 1354 */ "Linough Inc.",
		/* 1355 */ "Advanced Electronic Designs, Inc.",
		/* 1356 */ "Carefree Scott Fetzer Co Inc",
		/* 1357 */ "Sensome",
		/* 1358 */ "FORTRONIK storitve d.o.o.",
		/* 1359 */ "Sinnoz",
		/* 1360 */ "Versa Networks, Inc.",
		/* 1361 */ "Sylero",
		/* 1362 */ "Avempace SARL",
		/* 1363 */ "Nintendo Co., Ltd.",
		/* 1364 */ "National Instruments",
		/* 1365 */ "KROHNE Messtechnik GmbH",
		/* 1366 */ "Otodynamics Ltd",
		/* 1367 */ "Arwin Technology Limited",
		/* 1368 */ "benegear, inc.",
		/* 1369 */ "Newcon Optik",
		/* 1370 */ "CANDY HOUSE, Inc.",
		/* 1371 */ "FRANKLIN TECHNOLOGY INC",
		/* 1372 */ "Lely",
		/* 1373 */ "Valve Corporation",
		/* 1374 */ "Hekatron Vertriebs GmbH",
		/* 1375 */ "PROTECH S.A.S. DI GIRARDI ANDREA & C.",
		/* 1376 */ "Sarita CareTech APS (formerly Sarita CareTech IVS)",
		/* 1377 */ "Finder S.p.A.",
		/* 1378 */ "Thalmic Labs Inc.",
		/* 1379 */ "Steinel Vertrieb GmbH",
		/* 1380 */ "Beghelli Spa",
		/* 1381 */ "Beijing Smartspace Technologies Inc.",
		/* 1382 */ "CORE TRANSPORT TECHNOLOGIES NZ LIMITED",
		/* 1383 */ "Xiamen Everesports Goods Co., Ltd",
		/* 1384 */ "Bodyport Inc.",
		/* 1385 */ "Audionics System, INC.",
		/* 1386 */ "Flipnavi Co.,Ltd.",
		/* 1387 */ "Rion Co., Ltd.",
		/* 1388 */ "Long Range Systems, LLC",
		/* 1389 */ "Redmond Industrial Group LLC",
		/* 1390 */ "VIZPIN INC.",
		/* 1391 */ "BikeFinder AS",
		/* 1392 */ "Consumer Sleep Solutions LLC",
		/* 1393 */ "PSIKICK, INC.",
		/* 1394 */ "AntTail.com",
		/* 1395 */ "Lighting Science Group Corp.",
		/* 1396 */ "AFFORDABLE ELECTRONICS INC",
		/* 1397 */ "Integral Memroy Plc",
		/* 1398 */ "Globalstar, Inc.",
		/* 1399 */ "True Wearables, Inc.",
		/* 1400 */ "Wellington Drive Technologies Ltd",
		/* 1401 */ "Ensemble Tech Private Limited",
		/* 1402 */ "OMNI Remotes",
		/* 1403 */ "Duracell U.S. Operations Inc.",
		/* 1404 */ "Toor Technologies LLC",
		/* 1405 */ "Instinct Performance",
		/* 1406 */ "Beco, Inc",
		/* 1407 */ "Scuf Gaming International, LLC",
		/* 1408 */ "ARANZ Medical Limited",
		/* 1409 */ "LYS TECHNOLOGIES LTD",
		/* 1410 */ "Breakwall Analytics, LLC",
		/* 1411 */ "Code Blue Communications",
		/* 1412 */ "Gira Giersiepen GmbH & Co. KG",
		/* 1413 */ "Hearing Lab Technology",
		/* 1414 */ "LEGRAND",
		/* 1415 */ "Derichs GmbH",
		/* 1416 */ "ALT-TEKNIK LLC",
		/* 1417 */ "Star Technologies",
		/* 1418 */ "START TODAY CO.,LTD.",
		/* 1419 */ "Maxim Integrated Products",
		/* 1420 */ "MERCK Kommanditgesellschaft auf Aktien",
		/* 1421 */ "Jungheinrich Aktiengesellschaft",
		/* 1422 */ "Oculus VR, LLC",
		/* 1423 */ "HENDON SEMICONDUCTORS PTY LTD",
		/* 1424 */ "Pur3 Ltd",
		/* 1425 */ "Viasat Group S.p.A.",
		/* 1426 */ "IZITHERM",
		/* 1427 */ "Spaulding Clinical Research",
		/* 1428 */ "Kohler Company",
		/* 1429 */ "Inor Process AB",
		/* 1430 */ "My Smart Blinds",
		/* 1431 */ "RadioPulse Inc",
		/* 1432 */ "rapitag GmbH",
		/* 1433 */ "Lazlo326, LLC.",
		/* 1434 */ "Teledyne Lecroy, Inc.",
		/* 1435 */ "Dataflow Systems Limited",
		/* 1436 */ "Macrogiga Electronics",
		/* 1437 */ "Tandem Diabetes Care",
		/* 1438 */ "Polycom, Inc.",
		/* 1439 */ "Fisher & Paykel Healthcare",
		/* 1440 */ "RCP Software Oy",
		/* 1441 */ "Shanghai Xiaoyi Technology Co.,Ltd.",
		/* 1442 */ "ADHERIUM(NZ) LIMITED",
		/* 1443 */ "Axiomware Systems Incorporated",
		/* 1444 */ "O. E. M. Controls, Inc.",
		/* 1445 */ "Kiiroo BV",
		/* 1446 */ "Telecon Mobile Limited",
		/* 1447 */ "Sonos Inc",
		/* 1448 */ "Tom Allebrandi Consulting",
		/* 1449 */ "Monidor",
		/* 1450 */ "Tramex Limited",
		/* 1451 */ "Nofence AS",
		/* 1452 */ "GoerTek Dynaudio Co., Ltd.",
		/* 1453 */ "INIA",
		/* 1454 */ "CARMATE MFG.CO.,LTD",
		/* 1455 */ "ONvocal",
		/* 1456 */ "NewTec GmbH",
		/* 1457 */ "Medallion Instrumentation Systems",
		/* 1458 */ "CAREL INDUSTRIES S.P.A.",
		/* 1459 */ "Parabit Systems, Inc.",
		/* 1460 */ "White Horse Scientific ltd",
		/* 1461 */ "verisilicon",
		/* 1462 */ "Elecs Industry Co.,Ltd.",
		/* 1463 */ "Beijing Pinecone Electronics Co.,Ltd.",
		/* 1464 */ "Ambystoma Labs Inc.",
		/* 1465 */ "Suzhou Pairlink Network Technology",
		/* 1466 */ "igloohome",
		/* 1467 */ "Oxford Metrics plc",
		/* 1468 */ "Leviton Mfg. Co., Inc.",
		/* 1469 */ "ULC Robotics Inc.",
		/* 1470 */ "RFID Global by Softwork SrL",
		/* 1471 */ "Real-World-Systems Corporation",
		/* 1472 */ "Nalu Medical, Inc.",
		/* 1473 */ "P.I.Engineering",
		/* 1474 */ "Grote Industries",
		/* 1475 */ "Runtime, Inc.",
		/* 1476 */ "Codecoup sp. z o.o. sp. k.",
		/* 1477 */ "SELVE GmbH & Co. KG",
		/* 1478 */ "Smart Animal Training Systems, LLC",
		/* 1479 */ "Lippert Components, INC",
		/* 1480 */ "SOMFY SAS",
		/* 1481 */ "TBS Electronics B.V.",
		/* 1482 */ "MHL Custom Inc",
		/* 1483 */ "LucentWear LLC",
		/* 1484 */ "WATTS ELECTRONICS",
		/* 1485 */ "RJ Brands LLC",
		/* 1486 */ "V-ZUG Ltd",
		/* 1487 */ "Biowatch SA",
		/* 1488 */ "Anova Applied Electronics",
		/* 1489 */ "Lindab AB",
		/* 1490 */ "frogblue TECHNOLOGY GmbH",
		/* 1491 */ "Acurable Limited",
		/* 1492 */ "LAMPLIGHT Co., Ltd.",
		/* 1493 */ "TEGAM, Inc.",
		/* 1494 */ "Zhuhai Jieli technology Co.,Ltd",
		/* 1495 */ "modum.io AG",
		/* 1496 */ "Farm Jenny LLC",
		/* 1497 */ "Toyo Electronics Corporation",
		/* 1498 */ "Applied Neural Research Corp",
		/* 1499 */ "Avid Identification Systems, Inc.",
		/* 1500 */ "Petronics Inc.",
		/* 1501 */ "essentim GmbH",
		/* 1502 */ "QT Medical INC.",
		/* 1503 */ "VIRTUALCLINIC.DIRECT LIMITED",
		/* 1504 */ "Viper Design LLC",
		/* 1505 */ "Human, Incorporated",
		/* 1506 */ "stAPPtronics GmbH",
		/* 1507 */ "Elemental Machines, Inc.",
		/* 1508 */ "Taiyo Yuden Co., Ltd",
		/* 1509 */ "INEO ENERGY& SYSTEMS",
		/* 1510 */ "Motion Instruments Inc.",
		/* 1511 */ "PressurePro",
		/* 1512 */ "COWBOY",
		/* 1513 */ "iconmobile GmbH",
		/* 1514 */ "ACS-Control-System GmbH",
		/* 1515 */ "Bayerische Motoren Werke AG",
		/* 1516 */ "Gycom Svenska AB",
		/* 1517 */ "Fuji Xerox Co., Ltd",
		/* 1518 */ "Glide Inc.",
		/* 1519 */ "SIKOM AS",
		/* 1520 */ "beken",
		/* 1521 */ "The Linux Foundation",
		/* 1522 */ "Try and E CO.,LTD.",
		/* 1523 */ "SeeScan",
		/* 1524 */ "Clearity, LLC",
		/* 1525 */ "GS TAG",
		/* 1526 */ "DPTechnics",
		/* 1527 */ "TRACMO, INC.",
		/* 1528 */ "Anki Inc.",
		/* 1529 */ "Hagleitner Hygiene International GmbH",
		/* 1530 */ "Konami Sports Life Co., Ltd.",
		/* 1531 */ "Arblet Inc.",
		/* 1532 */ "Masbando GmbH",
		/* 1533 */ "Innoseis",
		/* 1534 */ "Niko nv",
		/* 1535 */ "Wellnomics Ltd",
		/* 1536 */ "iRobot Corporation",
		/* 1537 */ "Schrader Electronics",
		/* 1538 */ "Geberit International AG",
		/* 1539 */ "Fourth Evolution Inc",
		/* 1540 */ "Cell2Jack LLC",
		/* 1541 */ "FMW electronic Futterer u. Maier-Wolf OHG",
		/* 1542 */ "John Deere",
		/* 1543 */ "Rookery Technology Ltd",
		/* 1544 */ "KeySafe-Cloud",
		/* 1545 */ "BUCHI Labortechnik AG",
		/* 1546 */ "IQAir AG",
		/* 1547 */ "Triax Technologies Inc",
		/* 1548 */ "Vuzix Corporation",
		/* 1549 */ "TDK Corporation",
		/* 1550 */ "Blueair AB",
		/* 1551 */ "Signify Netherlands",
		/* 1552 */ "ADH GUARDIAN USA LLC",
		/* 1553 */ "Beurer GmbH",
		/* 1554 */ "Playfinity AS",
		/* 1555 */ "Hans Dinslage GmbH",
		/* 1556 */ "OnAsset Intelligence, Inc.",
		/* 1557 */ "INTER ACTION Corporation",
		/* 1558 */ "OS42 UG (haftungsbeschraenkt)",
		/* 1559 */ "WIZCONNECTED COMPANY LIMITED",
		/* 1560 */ "Audio-Technica Corporation",
		/* 1561 */ "Six Guys Labs, s.r.o.",
		/* 1562 */ "R.W. Beckett Corporation",
		/* 1563 */ "silex technology, inc.",
		/* 1564 */ "Univations Limited",
		/* 1565 */ "SENS Innovation ApS",
		/* 1566 */ "Diamond Kinetics, Inc.",
		/* 1567 */ "Phrame Inc.",
		/* 1568 */ "Forciot Oy",
		/* 1569 */ "Noordung d.o.o.",
		/* 1570 */ "Beam Labs, LLC",
		/* 1571 */ "Philadelphia Scientific (U.K.) Limited",
		/* 1572 */ "Biovotion AG",
		/* 1573 */ "Square Panda, Inc.",
		/* 1574 */ "Amplifico",
		/* 1575 */ "WEG S.A.",
		/* 1576 */ "Ensto Oy",
		/* 1577 */ "PHONEPE PVT LTD",
		/* 1578 */ "Lunatico Astronomia SL",
		/* 1579 */ "MinebeaMitsumi Inc.",
		/* 1580 */ "ASPion GmbH",
		/* 1581 */ "Vossloh-Schwabe Deutschland GmbH",
		/* 1582 */ "Procept",
		/* 1583 */ "ONKYO Corporation",
		/* 1584 */ "Asthrea D.O.O.",
		/* 1585 */ "Fortiori Design LLC",
		/* 1586 */ "Hugo Muller GmbH & Co KG",
		/* 1587 */ "Wangi Lai PLT",
		/* 1588 */ "Fanstel Corp",
		/* 1589 */ "Crookwood",
		/* 1590 */ "ELECTRONICA INTEGRAL DE SONIDO S.A.",
		/* 1591 */ "GiP Innovation Tools GmbH",
		/* 1592 */ "LX SOLUTIONS PTY LIMITED",
		/* 1593 */ "Shenzhen Minew Technologies Co., Ltd.",
		/* 1594 */ "Prolojik Limited",
		/* 1595 */ "Kromek Group Plc",
		/* 1596 */ "Contec Medical Systems Co., Ltd.",
		/* 1597 */ "Xradio Technology Co.,Ltd.",
		/* 1598 */ "The Indoor Lab, LLC",
		/* 1599 */ "LDL TECHNOLOGY",
		/* 1600 */ "Parkifi",
		/* 1601 */ "Revenue Collection Systems FRANCE SAS",
		/* 1602 */ "Bluetrum Technology Co.,Ltd",
		/* 1603 */ "makita corporation",
		/* 1604 */ "Apogee Instruments",
		/* 1605 */ "BM3",
		/* 1606 */ "SGV Group Holding GmbH & Co. KG",
		/* 1607 */ "MED-EL",
		/* 1608 */ "Ultune Technologies",
		/* 1609 */ "Ryeex Technology Co.,Ltd.",
		/* 1610 */ "Open Research Institute, Inc.",
		/* 1611 */ "Scale-Tec, Ltd",
		/* 1612 */ "Zumtobel Group AG",
		/* 1613 */ "iLOQ Oy",
		/* 1614 */ "KRUXWorks Technologies Private Limited",
		/* 1615 */ "Digital Matter Pty Ltd",
		/* 1616 */ "Coravin, Inc.",
		/* 1617 */ "Stasis Labs, Inc.",
		/* 1618 */ "ITZ Innovations- und Technologiezentrum GmbH",
		/* 1619 */ "Meggitt SA",
		/* 1620 */ "Ledlenser GmbH & Co. KG",
		/* 1621 */ "Renishaw PLC",
		/* 1622 */ "ZhuHai AdvanPro Technology Company Limited",
		/* 1623 */ "Meshtronix Limited",
		/* 1624 */ "Payex Norge AS",
		/* 1625 */ "UnSeen Technologies Oy",
		/* 1626 */ "Zound Industries International AB",
		/* 1627 */ "Sesam Solutions BV",
		/* 1628 */ "PixArt Imaging Inc.",
		/* 1629 */ "Panduit Corp.",
		/* 1630 */ "Alo AB",
		/* 1631 */ "Ricoh Company Ltd",
		/* 1632 */ "RTC Industries, Inc.",
		/* 1633 */ "Mode Lighting Limited",
		/* 1634 */ "Particle Industries, Inc.",
		/* 1635 */ "Advanced Telemetry Systems, Inc.",
		/* 1636 */ "RHA TECHNOLOGIES LTD",
		/* 1637 */ "Pure International Limited",
		/* 1638 */ "WTO Werkzeug-Einrichtungen GmbH",
		/* 1639 */ "Spark Technology Labs Inc.",
		/* 1640 */ "Bleb Technology srl",
		/* 1641 */ "Livanova USA, Inc.",
		/* 1642 */ "Brady Worldwide Inc.",
		/* 1643 */ "DewertOkin GmbH",
		/* 1644 */ "Ztove ApS",
		/* 1645 */ "Venso EcoSolutions AB",
		/* 1646 */ "Eurotronik Kranj d.o.o.",
		/* 1647 */ "Hug Technology Ltd",
		/* 1648 */ "Gema Switzerland GmbH",
		/* 1649 */ "Buzz Products Ltd.",
		/* 1650 */ "Kopi",
		/* 1651 */ "Innova Ideas Limited",
		/* 1652 */ "BeSpoon",
		/* 1653 */ "Deco Enterprises, Inc.",
		/* 1654 */ "Expai Solutions Private Limited",
		/* 1655 */ "Innovation First, Inc.",
		/* 1656 */ "SABIK Offshore GmbH",
		/* 1657 */ "4iiii Innovations Inc.",
		/* 1658 */ "The Energy Conservatory, Inc.",
		/* 1659 */ "I.FARM, INC.",
		/* 1660 */ "Tile, Inc.",
		/* 1661 */ "Form Athletica Inc.",
		/* 1662 */ "MbientLab Inc",
		/* 1663 */ "NETGRID S.N.C. DI BISSOLI MATTEO, CAMPOREALE SIMONE, TOGNETTI FEDERICO",
		/* 1664 */ "Mannkind Corporation",
		/* 1665 */ "Trade FIDES a.s.",
		/* 1666 */ "Photron Limited",
		/* 1667 */ "Eltako GmbH",
		/* 1668 */ "Dermalapps, LLC",
		/* 1669 */ "Greenwald Industries",
		/* 1670 */ "inQs Co., Ltd.",
		/* 1671 */ "Cherry GmbH",
		/* 1672 */ "Amsted Digital Solutions Inc.",
		/* 1673 */ "Tacx b.v.",
		/* 1674 */ "Raytac Corporation",
		/* 1675 */ "Jiangsu Teranovo Tech Co., Ltd.",
		/* 1676 */ "Changzhou Sound Dragon Electronics and Acoustics Co., Ltd",
		/* 1677 */ "JetBeep Inc.",
		/* 1678 */ "Razer Inc.",
		/* 1679 */ "JRM Group Limited",
		/* 1680 */ "Eccrine Systems, Inc.",
		/* 1681 */ "Curie Point AB",
		/* 1682 */ "Georg Fischer AG",
		/* 1683 */ "Hach - Danaher",
		/* 1684 */ "T&A Laboratories LLC",
		/* 1685 */ "Koki Holdings Co., Ltd.",
		/* 1686 */ "Gunakar Private Limited",
		/* 1687 */ "Stemco Products Inc",
		/* 1688 */ "Wood IT Security, LLC",
		/* 1689 */ "RandomLab SAS",
		/* 1690 */ "Adero, Inc. (formerly as TrackR, Inc.)",
		/* 1691 */ "Dragonchip Limited",
		/* 1692 */ "Noomi AB",
		/* 1693 */ "Vakaros LLC",
		/* 1694 */ "Delta Electronics, Inc.",
		/* 1695 */ "FlowMotion Technologies AS",
		/* 1696 */ "OBIQ Location Technology Inc.",
		/* 1697 */ "Cardo Systems, Ltd",
		/* 1698 */ "Globalworx GmbH",
		/* 1699 */ "Nymbus, LLC",
		/* 1700 */ "Sanyo Techno Solutions Tottori Co., Ltd.",
		/* 1701 */ "TEKZITEL PTY LTD",
		/* 1702 */ "Roambee Corporation",
		/* 1703 */ "Chipsea Technologies (ShenZhen) Corp.",
		/* 1704 */ "GD Midea Air-Conditioning Equipment Co., Ltd.",
		/* 1705 */ "Soundmax Electronics Limited",
		/* 1706 */ "Produal Oy",
		/* 1707 */ "HMS Industrial Networks AB",
		/* 1708 */ "Ingchips Technology Co., Ltd.",
		/* 1709 */ "InnovaSea Systems Inc.",
		/* 1710 */ "SenseQ Inc.",
		/* 1711 */ "Shoof Technologies",
		/* 1712 */ "BRK Brands, Inc.",
		/* 1713 */ "SimpliSafe, Inc.",
		/* 1714 */ "Tussock Innovation 2013 Limited",
		/* 1715 */ "The Hablab ApS",
		/* 1716 */ "Sencilion Oy",
		/* 1717 */ "Wabilogic Ltd.",
		/* 1718 */ "Sociometric Solutions, Inc.",
		/* 1719 */ "iCOGNIZE GmbH",
		/* 1720 */ "ShadeCraft, Inc",
		/* 1721 */ "Beflex Inc.",
		/* 1722 */ "Beaconzone Ltd",
		/* 1723 */ "Leaftronix Analogic Solutions Private Limited",
		/* 1724 */ "TWS Srl",
		/* 1725 */ "ABB Oy",
		/* 1726 */ "HitSeed Oy",
		/* 1727 */ "Delcom Products Inc.",
		/* 1728 */ "CAME S.p.A.",
		/* 1729 */ "Alarm.com Holdings, Inc",
		/* 1730 */ "Measurlogic Inc.",
		/* 1731 */ "King I Electronics.Co.,Ltd",
		/* 1732 */ "Dream Labs GmbH",
		/* 1733 */ "Urban Compass, Inc",
		/* 1734 */ "Simm Tronic Limited",
		/* 1735 */ "Somatix Inc",
		/* 1736 */ "Storz & Bickel GmbH & Co. KG",
		/* 1737 */ "MYLAPS B.V.",
		/* 1738 */ "Shenzhen Zhongguang Infotech Technology Development Co., Ltd",
		/* 1739 */ "Dyeware, LLC",
		/* 1740 */ "Dongguan SmartAction Technology Co.,Ltd.",
		/* 1741 */ "DIG Corporation",
		/* 1742 */ "FIOR & GENTZ",
		/* 1743 */ "Belparts N.V.",
		/* 1744 */ "Etekcity Corporation",
		/* 1745 */ "Meyer Sound Laboratories, Incorporated",
		/* 1746 */ "CeoTronics AG",
		/* 1747 */ "TriTeq Lock and Security, LLC",
		/* 1748 */ "DYNAKODE TECHNOLOGY PRIVATE LIMITED",
		/* 1749 */ "Sensirion AG",
		/* 1750 */ "JCT Healthcare Pty Ltd",
		/* 1751 */ "FUBA Automotive Electronics GmbH",
		/* 1752 */ "AW Company",
		/* 1753 */ "Shanghai Mountain View Silicon Co.,Ltd.",
		/* 1754 */ "Zliide Technologies ApS",
		/* 1755 */ "Automatic Labs, Inc.",
		/* 1756 */ "Industrial Network Controls, LLC",
		/* 1757 */ "Intellithings Ltd.",
		/* 1758 */ "Navcast, Inc.",
		/* 1759 */ "Hubbell Lighting, Inc.",
		/* 1760 */ "Avaya",
		/* 1761 */ "Milestone AV Technologies LLC",
		/* 1762 */ "Alango Technologies Ltd",
		/* 1763 */ "Spinlock Ltd",
		/* 1764 */ "Aluna",
		/* 1765 */ "OPTEX CO.,LTD.",
		/* 1766 */ "NIHON DENGYO KOUSAKU",
		/* 1767 */ "VELUX A/S",
		/* 1768 */ "Almendo Technologies GmbH",
		/* 1769 */ "Zmartfun Electronics, Inc.",
		/* 1770 */ "SafeLine Sweden AB",
		/* 1771 */ "Houston Radar LLC",
		/* 1772 */ "Sigur",
		/* 1773 */ "J Neades Ltd",
		/* 1774 */ "Avantis Systems Limited",
		/* 1775 */ "ALCARE Co., Ltd.",
		/* 1776 */ "Chargy Technologies, SL",
		/* 1777 */ "Shibutani Co., Ltd.",
		/* 1778 */ "Trapper Data AB",
		/* 1779 */ "Alfred International Inc.",
		/* 1780 */ "Near Field Solutions Ltd",
		/* 1781 */ "Vigil Technologies Inc.",
		/* 1782 */ "Vitulo Plus BV",
		/* 1783 */ "WILKA Schliesstechnik GmbH",
		/* 1784 */ "BodyPlus Technology Co.,Ltd",
		/* 1785 */ "happybrush GmbH",
		/* 1786 */ "Enequi AB",
		/* 1787 */ "Sartorius AG",
		/* 1788 */ "Tom Communication Industrial Co.,Ltd.",
		/* 1789 */ "ESS Embedded System Solutions Inc.",
		/* 1790 */ "Mahr GmbH",
		/* 1791 */ "Redpine Signals Inc",
		/* 1792 */ "TraqFreq LLC",
		/* 1793 */ "PAFERS TECH",
		/* 1794 */ "Akciju sabiedriba \"SAF TEHNIKA\"",
		/* 1795 */ "Beijing Jingdong Century Trading Co., Ltd.",
		/* 1796 */ "JBX Designs Inc.",
		/* 1797 */ "AB Electrolux",
		/* 1798 */ "Wernher von Braun Center for ASdvanced Research",
		/* 1799 */ "Essity Hygiene and Health Aktiebolag",
		/* 1800 */ "Be Interactive Co., Ltd",
		/* 1801 */ "Carewear Corp.",
		/* 1802 */ "Huf Hülsbeck & Fürst GmbH & Co. KG",
		/* 1803 */ "Element Products, Inc.",
		/* 1804 */ "Beijing Winner Microelectronics Co.,Ltd",
		/* 1805 */ "SmartSnugg Pty Ltd",
		/* 1806 */ "FiveCo Sarl",
		/* 1807 */ "California Things Inc.",
		/* 1808 */ "Audiodo AB",
		/* 1809 */ "ABAX AS",
		/* 1810 */ "Bull Group Company Limited",
		/* 1811 */ "Respiri Limited",
		/* 1812 */ "MindPeace Safety LLC",
		/* 1813 */ "Vgyan Solutions",
		/* 1814 */ "Altonics",
		/* 1815 */ "iQsquare BV",
		/* 1816 */ "IDIBAIX enginneering",
		/* 1817 */ "ECSG",
		/* 1818 */ "REVSMART WEARABLE HK CO LTD",
		/* 1819 */ "Precor",
		/* 1820 */ "F5 Sports, Inc",
		/* 1821 */ "exoTIC Systems",
		/* 1822 */ "DONGGUAN HELE ELECTRONICS CO., LTD",
		/* 1823 */ "Dongguan Liesheng Electronic Co.Ltd",
		/* 1824 */ "Oculeve, Inc.",
		/* 1825 */ "Clover Network, Inc.",
		/* 1826 */ "Xiamen Eholder Electronics Co.Ltd",
		/* 1827 */ "Ford Motor Company",
		/* 1828 */ "Guangzhou SuperSound Information Technology Co.,Ltd",
		/* 1829 */ "Tedee Sp. z o.o.",
		/* 1830 */ "PHC Corporation",
		/* 1831 */ "STALKIT AS",
		/* 1832 */ "Eli Lilly and Company",
		/* 1833 */ "SwaraLink Technologies",
		/* 1834 */ "JMR embedded systems GmbH",
		/* 1835 */ "Bitkey Inc.",
		/* 1836 */ "GWA Hygiene GmbH",
		/* 1837 */ "Safera Oy",
		/* 1838 */ "Open Platform Systems LLC",
		/* 1839 */ "OnePlus Electronics (Shenzhen) Co., Ltd.",
		/* 1840 */ "Wildlife Acoustics, Inc.",
		/* 1841 */ "ABLIC Inc.",
		/* 1842 */ "Dairy Tech, Inc.",
		/* 1843 */ "Iguanavation, Inc.",
		/* 1844 */ "DiUS Computing Pty Ltd",
		/* 1845 */ "UpRight Technologies LTD",
		/* 1846 */ "FrancisFund, LLC",
		/* 1847 */ "LLC Navitek",
		/* 1848 */ "Glass Security Pte Ltd",
		/* 1849 */ "Jiangsu Qinheng Co., Ltd.",
		/* 1850 */ "Chandler Systems Inc.",
		/* 1851 */ "Fantini Cosmi s.p.a.",
		/* 1852 */ "Acubit ApS",
		/* 1853 */ "Beijing Hao Heng Tian Tech Co., Ltd.",
		/* 1854 */ "Bluepack S.R.L.",
		/* 1855 */ "Beijing Unisoc Technologies Co., Ltd.",
		/* 1856 */ "HITIQ LIMITED",
		/* 1857 */ "MAC SRL",
		/* 1858 */ "DML LLC",
		/* 1859 */ "Sanofi",
		/* 1860 */ "SOCOMEC",
		/* 1861 */ "WIZNOVA, Inc.",
		/* 1862 */ "Seitec Elektronik GmbH",
		/* 1863 */ "OR Technologies Pty Ltd",
		/* 1864 */ "GuangZhou KuGou Computer Technology Co.Ltd",
		/* 1865 */ "DIAODIAO (Beijing) Technology Co., Ltd.",
		/* 1866 */ "Illusory Studios LLC",
		/* 1867 */ "Sarvavid Software Solutions LLP",
		/* 1868 */ "iopool s.a.",
		/* 1869 */ "Amtech Systems, LLC",
		/* 1870 */ "EAGLE DETECTION SA",
		/* 1871 */ "MEDIATECH S.R.L.",
		/* 1872 */ "Hamilton Professional Services of Canada Incorporated",
		/* 1873 */ "Changsha JEMO IC Design Co.,Ltd",
		/* 1874 */ "Elatec GmbH",
		/* 1875 */ "JLG Industries, Inc.",
		/* 1876 */ "Michael Parkin",
		/* 1877 */ "Brother Industries, Ltd",
		/* 1878 */ "Lumens For Less, Inc",
		/* 1879 */ "ELA Innovation",
		/* 1880 */ "umanSense AB",
		/* 1881 */ "Shanghai InGeek Cyber Security Co., Ltd.",
		/* 1882 */ "HARMAN CO.,LTD.",
		/* 1883 */ "Smart Sensor Devices AB",
		/* 1884 */ "Antitronics Inc.",
		/* 1885 */ "RHOMBUS SYSTEMS, INC.",
		/* 1886 */ "Katerra Inc.",
		/* 1887 */ "Remote Solution Co., LTD.",
		/* 1888 */ "Vimar SpA",
		/* 1889 */ "Mantis Tech LLC",
		/* 1890 */ "TerOpta Ltd",
		/* 1891 */ "PIKOLIN S.L.",
		/* 1892 */ "WWZN Information Technology Company Limited",
		/* 1893 */ "Voxx International",
		/* 1894 */ "ART AND PROGRAM, INC.",
		/* 1895 */ "NITTO DENKO ASIA TECHNICAL CENTRE PTE. LTD.",
		/* 1896 */ "Peloton Interactive Inc.",
		/* 1897 */ "Force Impact Technologies",
		/* 1898 */ "Dmac Mobile Developments, LLC",
		/* 1899 */ "Engineered Medical Technologies",
		/* 1900 */ "Noodle Technology inc",
		/* 1901 */ "Graesslin GmbH",
		/* 1902 */ "WuQi technologies, Inc.",
		/* 1903 */ "Successful Endeavours Pty Ltd",
		/* 1904 */ "InnoCon Medical ApS",
		/* 1905 */ "Corvex Connected Safety",
		/* 1906 */ "Thirdwayv Inc.",
		/* 1907 */ "Echoflex Solutions Inc.",
		/* 1908 */ "C-MAX Asia Limited",
		/* 1909 */ "4eBusiness GmbH",
		/* 1910 */ "Cyber Transport Control GmbH",
		/* 1911 */ "Cue",
		/* 1912 */ "KOAMTAC INC.",
		/* 1913 */ "Loopshore Oy",
		/* 1914 */ "Niruha Systems Private Limited",
		/* 1915 */ "AmaterZ, Inc.",
		/* 1916 */ "radius co., ltd.",
		/* 1917 */ "Sensority, s.r.o.",
		/* 1918 */ "Sparkage Inc.",
		/* 1919 */ "Glenview Software Corporation",
		/* 1920 */ "Finch Technologies Ltd.",
		/* 1921 */ "Qingping Technology (Beijing) Co., Ltd.",
		/* 1922 */ "DeviceDrive AS",
		/* 1923 */ "ESEMBER LIMITED LIABILITY COMPANY",
		/* 1924 */ "audifon GmbH & Co. KG",
		/* 1925 */ "O2 Micro, Inc.",
		/* 1926 */ "HLP Controls Pty Limited",
		/* 1927 */ "Pangaea Solution",
		/* 1928 */ "BubblyNet, LLC",
		/* 1930 */ "The Wildflower Foundation",
		/* 1931 */ "Optikam Tech Inc.",
		/* 1932 */ "MINIBREW HOLDING B.V",
		/* 1933 */ "Cybex GmbH",
		/* 1934 */ "FUJIMIC NIIGATA, INC.",
		/* 1935 */ "Hanna Instruments, Inc.",
		/* 1936 */ "KOMPAN A/S",
		/* 1937 */ "Scosche Industries, Inc.",
		/* 1938 */ "Provo Craft",
		/* 1939 */ "AEV spol. s r.o.",
		/* 1940 */ "The Coca-Cola Company",
		/* 1941 */ "GASTEC CORPORATION",
		/* 1942 */ "StarLeaf Ltd",
		/* 1943 */ "Water-i.d. GmbH",
		/* 1944 */ "HoloKit, Inc.",
		/* 1945 */ "PlantChoir Inc.",
		/* 1946 */ "GuangDong Oppo Mobile Telecommunications Corp., Ltd.",
		/* 1947 */ "CST ELECTRONICS (PROPRIETARY) LIMITED",
		/* 1948 */ "Sky UK Limited",
		/* 1949 */ "Digibale Pty Ltd",
		/* 1950 */ "Smartloxx GmbH",
		/* 1951 */ "Pune Scientific LLP",
		/* 1952 */ "Regent Beleuchtungskorper AG",
		/* 1953 */ "Apollo Neuroscience, Inc.",
		/* 1954 */ "Roku, Inc.",
		/* 1955 */ "Comcast Cable",
		/* 1956 */ "Xiamen Mage Information Technology Co., Ltd.",
		/* 1957 */ "RAB Lighting, Inc.",
		/* 1958 */ "Musen Connect, Inc.",
		/* 1959 */ "Zume, Inc.",
		/* 1960 */ "conbee GmbH",
		/* 1961 */ "Bruel & Kjaer Sound & Vibration",
		/* 1962 */ "The Kroger Co.",
		/* 1963 */ "Granite River Solutions, Inc.",
		/* 1964 */ "LoupeDeck Oy",
		/* 1965 */ "New H3C Technologies Co.,Ltd",
		/* 1966 */ "Aurea Solucoes Tecnologicas Ltda.",
		/* 1967 */ "Hong Kong Bouffalo Lab Limited",
		/* 1968 */ "GV Concepts Inc.",
		/* 1969 */ "Thomas Dynamics, LLC",
		/* 1970 */ "Moeco IOT Inc.",
		/* 1971 */ "2N TELEKOMUNIKACE a.s.",
		/* 1972 */ "Hormann KG Antriebstechnik",
		/* 1973 */ "CRONO CHIP, S.L.",
		/* 1974 */ "Soundbrenner Limited",
		/* 1975 */ "ETABLISSEMENTS GEORGES RENAULT",
		/* 1976 */ "iSwip",
		/* 1977 */ "Epona Biotec Limited",
		/* 1978 */ "Battery-Biz Inc.",
		/* 1979 */ "EPIC S.R.L.",
		/* 1980 */ "KD CIRCUITS LLC",
		/* 1981 */ "Genedrive Diagnostics Ltd",
		/* 1982 */ "Axentia Technologies AB",
		/* 1983 */ "REGULA Ltd.",
		/* 1984 */ "Biral AG",
		/* 1985 */ "A.W. Chesterton Company",
		/* 1986 */ "Radinn AB",
		/* 1987 */ "CIMTechniques, Inc.",
		/* 1988 */ "Johnson Health Tech NA",
		/* 1989 */ "June Life, Inc.",
		/* 1990 */ "Bluenetics GmbH",
		/* 1991 */ "iaconicDesign Inc.",
		/* 1992 */ "WRLDS Creations AB",
		/* 1993 */ "Skullcandy, Inc.",
		/* 1994 */ "Modul-System HH AB",
		/* 1995 */ "West Pharmaceutical Services, Inc.",
		/* 1996 */ "Barnacle Systems Inc.",
		/* 1997 */ "Smart Wave Technologies Canada Inc",
		/* 1998 */ "Shanghai Top-Chip Microelectronics Tech. Co., LTD",
		/* 1999 */ "NeoSensory, Inc.",
		/* 2000 */ "Hangzhou Tuya Information Technology Co., Ltd",
		/* 2001 */ "Shanghai Panchip Microelectronics Co., Ltd",
		/* 2002 */ "React Accessibility Limited",
		/* 2003 */ "LIVNEX Co.,Ltd.",
		/* 2004 */ "Kano Computing Limited",
		/* 2005 */ "hoots classic GmbH",
		/* 2006 */ "ecobee Inc.",
		/* 2007 */ "Nanjing Qinheng Microelectronics Co., Ltd",
		/* 2008 */ "SOLUTIONS AMBRA INC.",
		/* 2009 */ "Micro-Design, Inc.",
		/* 2010 */ "STARLITE Co., Ltd.",
		/* 2011 */ "Remedee Labs",
		/* 2012 */ "ThingOS GmbH",
		/* 2013 */ "Linear Circuits",
		/* 2014 */ "Unlimited Engineering SL",
		/* 2015 */ "Snap-on Incorporated",
		/* 2016 */ "Edifier International Limited",
		/* 2017 */ "Lucie Labs",
		/* 2018 */ "Alfred Kaercher SE & Co. KG",
		/* 2019 */ "Audiowise Technology Inc.",
		/* 2020 */ "Geeksme S.L.",
		/* 2021 */ "Minut, Inc.",
		/* 2022 */ "Autogrow Systems Limited",
		/* 2023 */ "Komfort IQ, Inc.",
		/* 2024 */ "Packetcraft, Inc.",
		/* 2025 */ "Häfele GmbH & Co KG",
		/* 2026 */ "ShapeLog, Inc.",
		/* 2027 */ "NOVABASE S.R.L.",
		/* 2028 */ "Frecce LLC",
		/* 2029 */ "Joule IQ, INC.",
		/* 2030 */ "KidzTek LLC",
		/* 2031 */ "Aktiebolaget Sandvik Coromant",
		/* 2032 */ "e-moola.com Pty Ltd",
		/* 2033 */ "GSM Innovations Pty Ltd",
		/* 2034 */ "SERENE GROUP, INC",
		/* 2035 */ "DIGISINE ENERGYTECH CO. LTD.",
		/* 2036 */ "MEDIRLAB Orvosbiologiai Fejleszto Korlatolt Felelossegu Tarsasag",
		/* 2037 */ "Byton North America Corporation",
		/* 2038 */ "Shenzhen TonliScience and Technology Development Co.,Ltd",
		/* 2039 */ "Cesar Systems Ltd.",
		/* 2040 */ "quip NYC Inc.",
		/* 2041 */ "Direct Communication Solutions, Inc.",
		/* 2042 */ "Klipsch Group, Inc.",
		/* 2043 */ "Access Co., Ltd",
		/* 2044 */ "Renault SA",
		/* 2045 */ "JSK CO., LTD.",
		/* 2046 */ "BIROTA",
		/* 2047 */ "maxon motor ltd.",
		/* 2048 */ "Optek",
		/* 2049 */ "CRONUS ELECTRONICS LTD",
		/* 2050 */ "NantSound, Inc.",
		/* 2051 */ "Domintell s.a.",
		/* 2052 */ "Andon Health Co.,Ltd",
		/* 2053 */ "Urbanminded Ltd",
		/* 2054 */ "TYRI Sweden AB",
		/* 2055 */ "ECD Electronic Components GmbH Dresden",
		/* 2056 */ "SISTEMAS KERN, SOCIEDAD ANÓMINA",
		/* 2057 */ "Trulli Audio",
		/* 2058 */ "Altaneos",
		/* 2059 */ "Nanoleaf Canada Limited",
		/* 2060 */ "Ingy B.V.",
		/* 2061 */ "Azbil Co.",
		/* 2062 */ "TATTCOM LLC",
		/* 2063 */ "Paradox Engineering SA",
		/* 2064 */ "LECO Corporation",
		/* 2065 */ "Becker Antriebe GmbH",
		/* 2066 */ "Mstream Technologies., Inc.",
		/* 2067 */ "Flextronics International USA Inc.",
		/* 2068 */ "Ossur hf.",
		/* 2069 */ "SKC Inc",
		/* 2070 */ "SPICA SYSTEMS LLC",
		/* 2071 */ "Wangs Alliance Corporation",
		/* 2072 */ "tatwah SA",
		/* 2073 */ "Hunter Douglas Inc",
		/* 2074 */ "Shenzhen Conex",
		/* 2075 */ "DIM3",
		/* 2076 */ "Bobrick Washroom Equipment, Inc.",
		/* 2077 */ "Potrykus Holdings and Development LLC",
		/* 2078 */ "iNFORM Technology GmbH",
		/* 2079 */ "eSenseLab LTD",
		/* 2080 */ "Brilliant Home Technology, Inc.",
		/* 2081 */ "INOVA Geophysical, Inc.",
		/* 2082 */ "adafruit industries",
		/* 2083 */ "Nexite Ltd",
		/* 2084 */ "8Power Limited",
		/* 2085 */ "CME PTE. LTD.",
		/* 2086 */ "Hyundai Motor Company",
		/* 2087 */ "Kickmaker",
		/* 2088 */ "Shanghai Suisheng Information Technology Co., Ltd.",
		/* 2089 */ "HEXAGON",
		/* 2090 */ "Mitutoyo Corporation",
		/* 2091 */ "shenzhen fitcare electronics Co.,Ltd",
		/* 2092 */ "INGICS TECHNOLOGY CO., LTD.",
		/* 2093 */ "INCUS PERFORMANCE LTD.",
		/* 2094 */ "ABB S.p.A.",
		/* 2095 */ "Blippit AB",
		/* 2096 */ "Core Health and Fitness LLC",
		/* 2097 */ "Foxble, LLC",
		/* 2098 */ "Intermotive,Inc.",
		/* 2099 */ "Conneqtech B.V.",
		/* 2100 */ "RIKEN KEIKI CO., LTD.,",
		/* 2101 */ "Canopy Growth Corporation",
		/* 2102 */ "Bitwards Oy",
		/* 2103 */ "vivo Mobile Communication Co., Ltd.",
		/* 2104 */ "Etymotic Research, Inc.",
		/* 2105 */ "A puissance 3",
		/* 2106 */ "BPW Bergische Achsen Kommanditgesellschaft",
		/* 2107 */ "Piaggio Fast Forward",
		/* 2108 */ "BeerTech LTD",
		/* 2109 */ "Tokenize, Inc.",
		/* 2110 */ "Zorachka LTD",
		/* 2111 */ "D-Link Corp.",
		/* 2112 */ "Down Range Systems LLC",
		/* 2113 */ "General Luminaire (Shanghai) Co., Ltd.",
		/* 2114 */ "Tangshan HongJia electronic technology co., LTD.",
		/* 2115 */ "FRAGRANCE DELIVERY TECHNOLOGIES LTD",
		/* 2116 */ "Pepperl + Fuchs GmbH",
		/* 2117 */ "Dometic Corporation",
		/* 2118 */ "USound GmbH",
		/* 2119 */ "DNANUDGE LIMITED",
		/* 2120 */ "JUJU JOINTS CANADA CORP.",
		/* 2121 */ "Dopple Technologies B.V.",
		/* 2122 */ "ARCOM",
		/* 2123 */ "Biotechware SRL",
		/* 2124 */ "ORSO Inc.",
		/* 2125 */ "SafePort",
		/* 2126 */ "Carol Cole Company",
		/* 2127 */ "Embedded Fitness B.V.",
		/* 2128 */ "Yealink (Xiamen) Network Technology Co.,LTD",
		/* 2129 */ "Subeca, Inc.",
		/* 2130 */ "Cognosos, Inc.",
		/* 2131 */ "Pektron Group Limited",
		/* 2132 */ "Tap Sound System",
		/* 2133 */ "Helios Hockey, Inc.",
		/* 2134 */ "Canopy Growth Corporation",
		/* 2135 */ "Parsyl Inc",
		/* 2136 */ "SOUNDBOKS",
		/* 2137 */ "BlueUp",
		/* 2138 */ "DAKATECH",
		/* 2139 */ "RICOH ELECTRONIC DEVICES CO., LTD.",
		/* 2140 */ "ACOS CO.,LTD.",
		/* 2141 */ "Guilin Zhishen Information Technology Co.,Ltd.",
		/* 2142 */ "Krog Systems LLC",
		/* 2143 */ "COMPEGPS TEAM,SOCIEDAD LIMITADA",
		/* 2144 */ "Alflex Products B.V.",
		/* 2145 */ "SmartSensor Labs Ltd",
		/* 2146 */ "SmartDrive Inc.",
		/* 2147 */ "Yo-tronics Technology Co., Ltd.",
		/* 2148 */ "Rafaelmicro",
		/* 2149 */ "Emergency Lighting Products Limited",
		/* 2150 */ "LAONZ Co.,Ltd",
		/* 2151 */ "Western Digital Techologies, Inc.",
		/* 2152 */ "WIOsense GmbH & Co. KG",
		/* 2153 */ "EVVA Sicherheitstechnologie GmbH",
		/* 2154 */ "Odic Incorporated",
		/* 2155 */ "Pacific Track, LLC",
		/* 2156 */ "Revvo Technologies, Inc.",
		/* 2157 */ "Biometrika d.o.o.",
		/* 2158 */ "Vorwerk Elektrowerke GmbH & Co. KG",
		/* 2159 */ "Trackunit A/S",
		/* 2160 */ "Wyze Labs, Inc",
		/* 2161 */ "Dension Elektronikai Kft. (formerly: Dension Audio Systems Ltd.)",
		/* 2162 */ "11 Health & Technologies Limited",
		/* 2163 */ "Innophase Incorporated",
		/* 2164 */ "Treegreen Limited",
		/* 2165 */ "Berner International LLC",
		/* 2166 */ "SmartResQ ApS",
		/* 2167 */ "Tome, Inc.",
		/* 2168 */ "The Chamberlain Group, Inc.",
		/* 2169 */ "MIZUNO Corporation",
		/* 2170 */ "ZRF, LLC",
		/* 2171 */ "BYSTAMP",
		/* 2172 */ "Crosscan GmbH",
		/* 2173 */ "Konftel AB",
		/* 2174 */ "1bar.net Limited",
		/* 2175 */ "Phillips Connect Technologies LLC",
		/* 2176 */ "imagiLabs AB",
		/* 2177 */ "Optalert",
		/* 2178 */ "PSYONIC, Inc.",
		/* 2179 */ "Wintersteiger AG",
		/* 2180 */ "Controlid Industria, Comercio de Hardware e Servicos de Tecnologia Ltda",
		/* 2181 */ "LEVOLOR, INC.",
		/* 2182 */ "Xsens Technologies B.V.",
		/* 2183 */ "Hydro-Gear Limited Partnership",
		/* 2184 */ "EnPointe Fencing Pty Ltd",
		/* 2185 */ "XANTHIO",
		/* 2186 */ "sclak s.r.l.",
		/* 2187 */ "Tricorder Arraay Technologies LLC",
		/* 2188 */ "GB Solution co.,Ltd",
		/* 2189 */ "Soliton Systems K.K.",
		/* 2190 */ "GIGA-TMS INC",
		/* 2191 */ "Tait International Limited",
		/* 2192 */ "NICHIEI INTEC CO., LTD.",
		/* 2193 */ "SmartWireless GmbH & Co. KG",
		/* 2194 */ "Ingenieurbuero Birnfeld UG (haftungsbeschraenkt)",
		/* 2195 */ "Maytronics Ltd",
		/* 2196 */ "EPIFIT",
		/* 2197 */ "Gimer medical",
		/* 2198 */ "Nokian Renkaat Oyj",
		/* 2199 */ "Current Lighting Solutions LLC",
		/* 2200 */ "Sensibo, Inc.",
		/* 2201 */ "SFS unimarket AG",
		/* 2202 */ "Private limited company \"Teltonika\"",
		/* 2203 */ "Saucon Technologies",
		/* 2204 */ "Embedded Devices Co. Company",
		/* 2205 */ "J-J.A.D.E. Enterprise LLC",
		/* 2206 */ "i-SENS, inc.",
		/* 2207 */ "Witschi Electronic Ltd",
		/* 2208 */ "Aclara Technologies LLC",
		/* 2209 */ "EXEO TECH CORPORATION",
		/* 2210 */ "Epic Systems Co., Ltd.",
		/* 2211 */ "Hoffmann SE",
		/* 2212 */ "Realme Chongqing Mobile Telecommunications Corp., Ltd.",
		/* 2213 */ "UMEHEAL Ltd",
		/* 2214 */ "Intelligenceworks Inc.",
		/* 2215 */ "TGR 1.618 Limited",
		/* 2216 */ "Shanghai Kfcube Inc",
		/* 2217 */ "Fraunhofer IIS",
		/* 2218 */ "SZ DJI TECHNOLOGY CO.,LTD",
		/* 2219 */ "Coburn Technology, LLC",
		/* 2220 */ "Topre Corporation",
		/* 2221 */ "Kayamatics Limited",
		/* 2222 */ "Moticon ReGo AG",
		/* 2223 */ "Polidea Sp. z o.o.",
		/* 2224 */ "Trivedi Advanced Technologies LLC",
		/* 2225 */ "CORE|vision BV",
		/* 2226 */ "PF SCHWEISSTECHNOLOGIE GMBH",
		/* 2227 */ "IONIQ Skincare GmbH & Co. KG",
		/* 2228 */ "Sengled Co., Ltd.",
		/* 2229 */ "TransferFi",
		/* 2230 */ "Boehringer Ingelheim Vetmedica GmbH"
        };

	return (m >= SIZE(t)? "?" : t[m]);
} /* hci_manufacturer2str */

char const *
hci_commands2str(uint8_t *commands, char *buffer, int size)
{
	static char const * const	t[][8] = {
	{ /* byte 0 */
		/* 0 */ "<HCI_Inquiry> ",
		/* 1 */ "<HCI_Inquiry_Cancel> ",
		/* 2 */ "<HCI_Periodic_Inquiry_Mode> ",
		/* 3 */ "<HCI_Exit_Periodic_Inquiry_Mode> ",
		/* 4 */ "<HCI_Create_Connection> ",
		/* 5 */ "<HCI_Disconnect> ",
		/* 6 */ "<HCI_Add_SCO_Connection (deprecated)> ",
		/* 7 */ "<HCI_Create_Connection_Cancel> "
	},
	{ /* byte 1 */
		/* 0 */ "<HCI_Accept_Connection_Request> ",
		/* 1 */ "<HCI_Reject_Connection_Request> ",
		/* 2 */ "<HCI_Link_Key_Request_Reply> ",
		/* 3 */ "<HCI_Link_Key_Request_Negative_Reply> ",
		/* 4 */ "<HCI_PIN_Code_Request_Reply> ",
		/* 5 */ "<HCI_PIN_Code_Request_Negative_Reply> ",
		/* 6 */ "<HCI_Change_Connection_Packet_Type> ",
		/* 7 */ "<HCI_Authentication_Requested> "
	},
	{ /* byte 2 */
		/* 0 */ "<HCI_Set_Connection_Encryption> ",
		/* 1 */ "<HCI_Change_Connection_Link_Key> ",
		/* 2 */ "<HCI_Master_Link_Key> ",
		/* 3 */ "<HCI_Remote_Name_Request> ",
		/* 4 */ "<HCI_Remote_Name_Request_Cancel> ",
		/* 5 */ "<HCI_Read_Remote_Supported_Features> ",
		/* 6 */ "<HCI_Read_Remote_Extended_Features> ",
		/* 7 */ "<HCI_Read_Remote_Version_Information> "
	},
	{ /* byte 3 */
		/* 0 */ "<HCI_Read_Clock_Offset> ",
		/* 1 */ "<HCI_Read_LMP_Handle> ",
		/* 2 */ "<Unknown 3.2> ",
		/* 3 */ "<Unknown 3.3> ",
		/* 4 */ "<Unknown 3.4> ",
		/* 5 */ "<Unknown 3.5> ",
		/* 6 */ "<Unknown 3.6> ",
		/* 7 */ "<Unknown 3.7> "
	},
	{ /* byte 4 */
		/* 0 */ "<Unknown 4.0> ",
		/* 1 */ "<HCI_Hold_Mode> ",
		/* 2 */ "<HCI_Sniff_Mode> ",
		/* 3 */ "<HCI_Exit_Sniff_Mode> ",
		/* 4 */ "<Previously used 4.4> ",
		/* 5 */ "<Previously used 4.5> ",
		/* 6 */ "<HCI_QoS_Setup> ",
		/* 7 */ "<HCI_Role_Discovery> "
	},
	{ /* byte 5 */
		/* 0 */ "<HCI_Switch_Role> ",
		/* 1 */ "<HCI_Read_Link_Policy_Settings> ",
		/* 2 */ "<HCI_Write_Link_Policy_Settings> ",
		/* 3 */ "<HCI_Read_Default_Link_Policy_Settings> ",
		/* 4 */ "<HCI_Write_Default_Link_Policy_Settings> ",
		/* 5 */ "<HCI_Flow_Specification> ",
		/* 6 */ "<HCI_Set_Event_Mask> ",
		/* 7 */ "<HCI_Reset> "
	},
	{ /* byte 6 */
		/* 0 */ "<HCI_Set_Event_Filter> ",
		/* 1 */ "<HCI_Flush> ",
		/* 2 */ "<HCI_Read_PIN_Type> ",
		/* 3 */ "<HCI_Write_PIN_Type> ",
		/* 4 */ "<Previously used 6.4> ",
		/* 5 */ "<HCI_Read_Stored_Link_Key> ",
		/* 6 */ "<HCI_Write_Stored_Link_Key> ",
		/* 7 */ "<HCI_Delete_Stored_Link_Key> "
	},
	{ /* byte 7 */
		/* 0 */ "<HCI_Write_Local_Name> ",
		/* 1 */ "<HCI_Read_Local_Name> ",
		/* 2 */ "<HCI_Read_Connection_Accept_Timeout> ",
		/* 3 */ "<HCI_Write_Connection_Accept_Timeout> ",
		/* 4 */ "<HCI_Read_Page_Timeout> ",
		/* 5 */ "<HCI_Write_Page_Timeout> ",
		/* 6 */ "<HCI_Read_Scan_Enable> ",
		/* 7 */ "<HCI_Write_Scan_Enable> "
	},
	{ /* byte 8 */
		/* 0 */ "<HCI_Read_Page_Scan_Activity> ",
		/* 1 */ "<HCI_Write_Page_Scan_Activity> ",
		/* 2 */ "<HCI_Read_Inquiry_Scan_Activity> ",
		/* 3 */ "<HCI_Write_Inquiry_Scan_Activity> ",
		/* 4 */ "<HCI_Read_Authentication_Enable> ",
		/* 5 */ "<HCI_Write_Authentication_Enable> ",
		/* 6 */ "<HCI_Read_Encryption_Mode (deprecated)> ",
		/* 7 */ "<HCI_Write_Encryption_Mode (deprecated)> "
	},
	{ /* byte 9 */
		/* 0 */ "<HCI_Read_Class_Of_Device> ",
		/* 1 */ "<HCI_Write_Class_Of_Device> ",
		/* 2 */ "<HCI_Read_Voice_Setting> ",
		/* 3 */ "<HCI_Write_Voice_Setting> ",
		/* 4 */ "<HCI_Read_Automatic_Flush_Timeout> ",
		/* 5 */ "<HCI_Write_Automatic_Flush_Timeout> ",
		/* 6 */ "<HCI_Read_Num_Broadcast_Retransmissions> ",
		/* 7 */ "<HCI_Write_Num_Broadcast_Retransmissions> "
	},
	{ /* byte 10 */
		/* 0 */ "<HCI_Read_Hold_Mode_Activity> ",
		/* 1 */ "<HCI_Write_Hold_Mode_Activity> ",
		/* 2 */ "<HCI_Read_Transmit_Power_Level> ",
		/* 3 */ "<HCI_Read_Synchronous_Flow_Control_Enable> ",
		/* 4 */ "<HCI_Write_Synchronous_Flow_Control_Enable> ",
		/* 5 */ "<HCI_Set_Controller_To_Host_Flow_Control> ",
		/* 6 */ "<HCI_Host_Buffer_Size> ",
		/* 7 */ "<HCI_Host_Number_Of_Completed_Packets> "
	},
	{ /* byte 11 */
		/* 0 */ "<HCI_Read_Link_Supervision_Timeout> ",
		/* 1 */ "<HCI_Write_Link_Supervision_Timeout> ",
		/* 2 */ "<HCI_Read_Number_Of_Supported_IAC> ",
		/* 3 */ "<HCI_Read_Current_IAC_LAP> ",
		/* 4 */ "<HCI_Write_Current_IAC_LAP> ",
		/* 5 */ "<HCI_Read_Page_Scan_Mode_Period (deprecated)> ",
		/* 6 */ "<HCI_Write_Page_Scan_Mode_Period (deprecated)> ",
		/* 7 */ "<HCI_Read_Page_Scan_Mode (deprecated)> "
	},
	{ /* byte 12 */
		/* 0 */ "<HCI_Write_Page_Scan_Mode (deprecated)> ",
		/* 1 */ "<HCI_Set_AFH_Host_Channel_Classification> ",
		/* 2 */ "<Unknown 12.2> ",
		/* 3 */ "<Unknown 12.3> ",
		/* 4 */ "<HCI_Read_Inquiry_Scan_Type> ",
		/* 5 */ "<HCI_Write_Inquiry_Scan_Type> ",
		/* 6 */ "<HCI_Read_Inquiry_Mode> ",
		/* 7 */ "<HCI_Write_Inquiry_Mode> "
	},
	{ /* byte 13 */
		/* 0 */ "<HCI_Read_Page_Scan_Type> ",
		/* 1 */ "<HCI_Write_Page_Scan_Type> ",
		/* 2 */ "<HCI_Read_AFH_Channel_Assessment_Mode> ",
		/* 3 */ "<HCI_Write_AFH_Channel_Assessment_Mode> ",
		/* 4 */ "<Unknown 13.4> ",
		/* 5 */ "<Unknown 13.5> ",
		/* 6 */ "<Unknown 13.6> ",
		/* 7 */ "<Unknown 13.7> "
	},
	{ /* byte 14 */
		/* 0 */ "<Unknown 14.0> ",
		/* 1 */ "<Unknown 14.1>",
		/* 2 */ "<Unknown 14.2> ",
		/* 3 */ "<HCI_Read_Local_Version_Information> ",
		/* 4 */ "<Unknown 14.4> ",
		/* 5 */ "<HCI_Read_Local_Supported_Features> ",
		/* 6 */ "<HCI_Read_Local_Extended_Features> ",
		/* 7 */ "<HCI_Read_Buffer_Size> "
	},
	{ /* byte 15 */
		/* 0 */ "<HCI_Read_Country_Code (deprecated)> ",
		/* 1 */ "<HCI_Read_BD_ADDR> ",
		/* 2 */ "<HCI_Read_Failed_Contact_Counter> ",
		/* 3 */ "<HCI_Reset_Failed_Contact_Counter> ",
		/* 4 */ "<HCI_Read_Link_Quality> ",
		/* 5 */ "<HCI_Read_RSSI> ",
		/* 6 */ "<HCI_Read_AFH_Channel_Map> ",
		/* 7 */ "<HCI_Read_Clock> "
	},
	{ /* byte 16 */
		/* 0 */ "<HCI_Read_Loopback_Mode> ",
		/* 1 */ "<HCI_Write_Loopback_Mode> ",
		/* 2 */ "<HCI_Enable_Device_Under_Test_Mode> ",
		/* 3 */ "<HCI_Setup_Synchronous_Connection_Request> ",
		/* 4 */ "<HCI_Accept_Synchronous_Connection_Request> ",
		/* 5 */ "<HCI_Reject_Synchronous_Connection_Request> ",
		/* 6 */ "<Unknown 16.6> ",
		/* 7 */ "<Unknown 16,7> "
	},
	{ /* byte 17 */
		/* 0 */ "<HCI_Read_Extended_Inquiry_Response> ",
		/* 1 */ "<HCI_Write_Extended_Inquiry_Response> ",
		/* 2 */ "<HCI_Refresh_Encryption_Key> ",
		/* 3 */ "<Unknown 17.3> ",
		/* 4 */ "<HCI_Sniff_Subrating> ",
		/* 5 */ "<HCI_Read_Simple_Pairing_Mode> ",
		/* 6 */ "<HCI_Write_Simple_Pairing_Mode> ",
		/* 7 */ "<HCI_Read_Local_OOB_Data> "
	},
	{ /* byte 18 */
		/* 0 */ "<HCI_Read_Inquiry_Response_Transmit_Power_Level> ",
		/* 1 */ "<HCI_Write_Inquiry_Transmit_Power_Level> ",
		/* 2 */ "<HCI_Read_Default_Erroneous_Data_Reporting> ",
		/* 3 */ "<HCI_Write_Default_Erroneous_Data_Reporting> ",
		/* 4 */ "<Unknown 18.4> ",
		/* 5 */ "<Unknown 18.5> ",
		/* 6 */ "<Unknown 18.6> ",
		/* 7 */ "<HCI_IO_Capability_Request_Reply> "
	},
	{ /* byte 19 */
		/* 0 */ "<HCI_User_Confirmation_Request_Reply> ",
		/* 1 */ "<HCI_User_Confirmation_Request_Negative_Reply> ",
		/* 2 */ "<HCI_User_Passkey_Request_Reply> ",
		/* 3 */ "<HCI_User_Passkey_Request_Negative_Reply> ",
		/* 4 */ "<HCI_Remote_OOB_Data_Request_Reply> ",
		/* 5 */ "<HCI_Write_Simple_Pairing_Debug_Mode> ",
		/* 6 */ "<HCI_Enhanced_Flush> ",
		/* 7 */ "<HCI_Remote_OOB_Data_Request_Negative_Reply> "
	},
	{ /* byte 20 */
		/* 0 */ "<Unknown 20.0> ",
		/* 1 */ "<Unknown 20.1> ",
		/* 2 */ "<HCI_Send_Keypress_Notification> ",
		/* 3 */ "<HCI_IO_Capability_Request_Negative_Reply> ",
		/* 4 */ "<HCI_Read_Encryption_Key_Size> ",
		/* 5 */ "<Unknown 20.5> ",
		/* 6 */ "<Unknown 20.6> ",
		/* 7 */ "<Unknown 20.7> "
	},
	{ /* byte 21 */
		/* 0 */ "<HCI_Create_Physical_Link> ",
		/* 1 */ "<HCI_Accept_Physical_Link> ",
		/* 2 */ "<HCI_Disconnect_Physical_Link> ",
		/* 3 */ "<HCI_Create_Logical_Link> ",
		/* 4 */ "<HCI_Accept_Logical_Link> ",
		/* 5 */ "<HCI_Disconnect_Logical_Link> ",
		/* 6 */ "<HCI_Logical_Link_Cancel> ",
		/* 7 */ "<HCI_Flow_Spec_Modify> "
	},
	{ /* byte 22 */
		/* 0 */ "<HCI_Read_Logical_Link_Accept_Timeout> ",
		/* 1 */ "<HCI_Write_Logical_Link_Accept_Timeout> ",
		/* 2 */ "<HCI_Set_Event_Mask_Page_2> ",
		/* 3 */ "<HCI_Read_Location_Data> ",
		/* 4 */ "<HCI_Write_Location_Data> ",
		/* 5 */ "<HCI_Read_Local_AMP_Info> ",
		/* 6 */ "<HCI_Read_Local_AMP_ASSOC> ",
		/* 7 */ "<HCI_Write_Remote_AMP_ASSOC> "
	},
	{ /* byte 23 */
		/* 0 */ "<HCI_Read_Flow_Control_Mode> ",
		/* 1 */ "<HCI_Write_Flow_Control_Mode> ",
		/* 2 */ "<HCI_Read_Data_Block_Size> ",
		/* 3 */ "<Unknown 23.3> ",
		/* 4 */ "<Unknown 23.4> ",
		/* 5 */ "<HCI_Enable_AMP_Receiver_Reports> ",
		/* 6 */ "<HCI_AMP_Test_End> ",
		/* 7 */ "<HCI_AMP_Test> "
	},
	{ /* byte 24 */
		/* 0 */ "<HCI_Read_Enhanced_Transmit_Power_Level> ",
		/* 1 */ "<Unknown 24.1> ",
		/* 2 */ "<HCI_Read_Best_Effort_Flush_Timeout> ",
		/* 3 */ "<HCI_Write_Best_Effort_Flush_Timeout> ",
		/* 4 */ "<HCI_Short_Range_Mode> ",
		/* 5 */ "<HCI_Read_LE_Host_Support> ",
		/* 6 */ "<HCI_Write_LE_Host_Support> ",
		/* 7 */ "<Unknown 24.7> "
	},
	{ /* byte 25 */
		/* 0 */ "<HCI_LE_Set_Event_Mask> ",
		/* 1 */ "<HCI_LE_Read_Buffer_Size [v1]> ",
		/* 2 */ "<HCI_LE_Read_Local_Supported_Features> ",
		/* 3 */ "<Unknown 25.3> ",
		/* 4 */ "<HCI_LE_Set_Random_Address> ",
		/* 5 */ "<HCI_LE_Set_Advertising_Parameters> ",
		/* 6 */ "<HCI_LE_Read_Advertising_Physical_Channel_Tx_Power> ",
		/* 7 */ "<HCI_LE_Set_Advertising_Data> "
	},
	{ /* byte 26 */
		/* 0 */ "<HCI_LE_Set_Scan_Response_Data> ",
		/* 1 */ "<HCI_LE_Set_Advertising_Enable> ",
		/* 2 */ "<HCI_LE_Set_Scan_Parameters> ",
		/* 3 */ "<HCI_LE_Set_Scan_Enable> ",
		/* 4 */ "<HCI_LE_Create_Connection> ",
		/* 5 */ "<HCI_LE_Create_Connection_Cancel> ",
		/* 6 */ "<HCI_LE_Read_White_List_Size> ",
		/* 7 */ "<HCI_LE_Clear_White_List> "
	},
	{ /* byte 27 */
		/* 0 */ "<HCI_LE_Add_Device_To_White_List> ",
		/* 1 */ "<HCI_LE_Remove_Device_From_White_List> ",
		/* 2 */ "<HCI_LE_Connection_Update> ",
		/* 3 */ "<HCI_LE_Set_Host_Channel_Classification> ",
		/* 4 */ "<HCI_LE_Read_Channel_Map> ",
		/* 5 */ "<HCI_LE_Read_Remote_Features> ",
		/* 6 */ "<HCI_LE_Encrypt> ",
		/* 7 */ "<HCI_LE_Rand> "
	},
	{ /* byte 28 */
		/* 0 */ "<HCI_LE_Enable_Encryption> ",
		/* 1 */ "<HCI_LE_Long_Term_Key_Request_Reply> ",
		/* 2 */ "<HCI_LE_Long_Term_Key_Request_Negative_Reply> ",
		/* 3 */ "<HCI_LE_Read_Supported_States> ",
		/* 4 */ "<HCI_LE_Receiver_Test [v1]> ",
		/* 5 */ "<HCI_LE_Transmitter_Test [v1]> ",
		/* 6 */ "<HCI_LE_Test_End> ",
		/* 7 */ "<Unknown 28.7> "
	},
	{ /* byte 29 */
		/* 0 */ "<Unknown 29.0> ",
		/* 1 */ "<Unknown 29.1> ",
		/* 2 */ "<Unknown 29.2> ",
		/* 3 */ "<HCI_Enhanced_Setup_Synchronous_Connection> ",
		/* 4 */ "<HCI_Enhanced_Accept_Synchronous_Connection> ",
		/* 5 */ "<HCI_Read_Local_Supported_Codecs> ",
		/* 6 */ "<HCI_Set_MWS_Channel_Parameters> ",
		/* 7 */ "<HCI_Set_External_Frame_Configuration> "
	},
	{ /* byte 30 */
		/* 0 */ "<HCI_Set_MWS_Signaling> ",
		/* 1 */ "<HCI_Set_MWS_Transport_Layer> ",
		/* 2 */ "<HCI_Set_MWS_Scan_Frequency_Table> ",
		/* 3 */ "<HCI_Get_MWS_Transport_Layer_Configuration> ",
		/* 4 */ "<HCI_Set_MWS_PATTERN_Configuration> ",
		/* 5 */ "<HCI_Set_Triggered_Clock_Capture> ",
		/* 6 */ "<HCI_Truncated_Page> ",
		/* 7 */ "<HCI_Truncated_Page_Cancel> "
	},
	{ /* byte 31 */
		/* 0 */ "<HCI_Set_Connectionless_Slave_Broadcast> ",
		/* 1 */ "<HCI_Set_Connectionless_Slave_Broadcast_Receive> ",
		/* 2 */ "<HCI_Start_Synchronization_Train> ",
		/* 3 */ "<HCI_Receive_Synchronization_Train> ",
		/* 4 */ "<HCI_Set_Reserved_LT_ADDR> ",
		/* 5 */ "<HCI_Delete_Reserved_LT_ADDR> ",
		/* 6 */ "<HCI_Set_Connectionless_Slave_Broadcast_Data> ",
		/* 7 */ "<HCI_Read_Synchronization_Train_Parameters> "
	},
	{ /* byte 32 */
		/* 0 */ "<HCI_Write_Synchronization_Train_Parameters> ",
		/* 1 */ "<HCI_Remote_OOB_Extended_Data_Request_Reply> ",
		/* 2 */ "<HCI_Read_Secure_Connections_Host_Support> ",
		/* 3 */ "<HCI_Write_Secure_Connections_Host_Support> ",
		/* 4 */ "<HCI_Read_Authenticated_Payload_Timeout> ",
		/* 5 */ "<HCI_Write_Authenticated_Payload_Timeout> ",
		/* 6 */ "<HCI_Read_Local_OOB_Extended_Data> ",
		/* 7 */ "<HCI_Write_Secure_Connections_Test_Mode> "
	},
	{ /* byte 33 */
		/* 0 */ "<HCI_Read_Extended_Page_Timeout> ",
		/* 1 */ "<HCI_Write_Extended_Page_Timeout> ",
		/* 2 */ "<HCI_Read_Extended_Inquiry_Length> ",
		/* 3 */ "<HCI_Write_Extended_Inquiry_Length> ",
		/* 4 */ "<HCI_LE_Remote_Connection_Parameter_Request_Reply> ",
		/* 5 */ "<HCI_LE_Remote_Connection_Parameter_Request_Negative_Reply> ",
		/* 6 */ "<HCI_LE_Set_Data_Length> ",
		/* 7 */ "<HCI_LE_Read_Suggested_Default_Data_Length> "
	},
	{ /* byte 34 */
		/* 0 */ "<HCI_LE_Write_Suggested_Default_Data_Length> ",
		/* 1 */ "<HCI_LE_Read_Local_P-256_Public_Key> ",
		/* 2 */ "<HCI_LE_Generate_DHKey [v1]> ",
		/* 3 */ "<HCI_LE_Add_Device_To_Resolving_List> ",
		/* 4 */ "<HCI_LE_Remove_Device_From_Resolving_List> ",
		/* 5 */ "<HCI_LE_Clear_Resolving_List> ",
		/* 6 */ "<HCI_LE_Read_Resolving_List_Size> ",
		/* 7 */ "<HCI_LE_Read_Peer_Resolvable_Address> "
	},
	{ /* byte 35 */
		/* 0 */ "<HCI_LE_Read_Local_Resolvable_Address> ",
		/* 1 */ "<HCI_LE_Set_Address_Resolution_Enable> ",
		/* 2 */ "<HCI_LE_Set_Resolvable_Private_Address_Timeout> ",
		/* 3 */ "<HCI_LE_Read_Maximum_Data_Length> ",
		/* 4 */ "<HCI_LE_Read_PHY> ",
		/* 5 */ "<HCI_LE_Set_Default_PHY> ",
		/* 6 */ "<HCI_LE_Set_PHY> ",
		/* 7 */ "<HCI_LE_Receiver_Test [v2]> "
	},
	{ /* byte 36 */
		/* 0 */ "<HCI_LE_Transmitter_Test [v2]> ",
		/* 1 */ "<HCI_LE_Set_Advertising_Set_Random_Address> ",
		/* 2 */ "<HCI_LE_Set_Extended_Advertising_Parameters> ",
		/* 3 */ "<HCI_LE_Set_Extended_Advertising_Data> ",
		/* 4 */ "<HCI_LE_Set_Extended_Scan_Response_Data> ",
		/* 5 */ "<HCI_LE_Set_Extended_Advertising_Enable> ",
		/* 6 */ "<HCI_LE_Read_Maximum_Advertising_Data_Length> ",
		/* 7 */ "<HCI_LE_Read_Number_of_Supported_Advertising_Sets> "
	},
	{ /* byte 37 */
		/* 0 */ "<HCI_LE_Remove_Advertising_Set> ",
		/* 1 */ "<HCI_LE_Clear_Advertising_Sets> ",
		/* 2 */ "<HCI_LE_Set_Periodic_Advertising_Parameters> ",
		/* 3 */ "<HCI_LE_Set_Periodic_Advertising_Data> ",
		/* 4 */ "<HCI_LE_Set_Periodic_Advertising_Enable> ",
		/* 5 */ "<HCI_LE_Set_Extended_Scan_Parameters> ",
		/* 6 */ "<HCI_LE_Set_Extended_Scan_Enable> ",
		/* 7 */ "<HCI_LE_Extended_Create_Connection> "
	},
	{ /* byte 38 */
		/* 0 */ "<HCI_LE_Periodic_Advertising_Create_Sync> ",
		/* 1 */ "<HCI_LE_Periodic_Advertising_Create_Sync_Cancel> ",
		/* 2 */ "<HCI_LE_Periodic_Advertising_Terminate_Sync> ",
		/* 3 */ "<HCI_LE_Add_Device_To_Periodic_Advertiser_List> ",
		/* 4 */ "<HCI_LE_Remove_Device_From_Periodic_Advertiser_List> ",
		/* 5 */ "<HCI_LE_Clear_Periodic_Advertiser_List> ",
		/* 6 */ "<HCI_LE_Read_Periodic_Advertiser_List_Size> ",
		/* 7 */ "<HCI_LE_Read_Transmit_Power> "
	},
	{ /* byte 39 */
		/* 0 */ "<HCI_LE_Read_RF_Path_Compensation> ",
		/* 1 */ "<HCI_LE_Write_RF_Path_Compensation> ",
		/* 2 */ "<HCI_LE_Set_Privacy_Mode> ",
		/* 3 */ "<HCI_LE_Receiver_Test [v3]> ",
		/* 4 */ "<HCI_LE_Transmitter_Test [v3]> ",
		/* 5 */ "<HCI_LE_Set_Connectionless_CTE_Transmit_Parameters> ",
		/* 6 */ "<HCI_LE_Set_Connectionless_CTE_Transmit_Enable> ",
		/* 7 */ "<HCI_LE_Set_Connectionless_IQ_Sampling_Enable> "
	},
	{ /* byte 40 */
		/* 0 */ "<HCI_LE_Set_Connection_CTE_Receive_Parameters> ",
		/* 1 */ "<HCI_LE_Set_Connection_CTE_Transmit_Parameters> ",
		/* 2 */ "<HCI_LE_Connection_CTE_Request_Enable> ",
		/* 3 */ "<HCI_LE_Connection_CTE_Response_Enable> ",
		/* 4 */ "<HCI_LE_Read_Antenna_Information> ",
		/* 5 */ "<HCI_LE_Set_Periodic_Advertising_Receive_Enable> ",
		/* 6 */ "<HCI_LE_Periodic_Advertising_Sync_Transfer> ",
		/* 7 */ "<HCI_LE_Periodic_Advertising_Set_Info_Transfer> "
	},
	{ /* byte 41 */
		/* 0 */ "<HCI_LE_Set_Periodic_Advertising_Sync_Transfer_Parameters> ",
		/* 1 */ "<HCI_LE_Set_Default_Periodic_Advertising_Sync_Transfer_- Parameters> ",
		/* 2 */ "<HCI_LE_Generate_DHKey [v2]> ",
		/* 3 */ "<HCI_Read_Local_Simple_Pairing_Options> ",
		/* 4 */ "<HCI_LE_Modify_Sleep_Clock_Accuracy> ",
		/* 5 */ "<HCI_LE_Read_Buffer_Size [v2]> ",
		/* 6 */ "<HCI_LE_Read_ISO_TX_Sync> ",
		/* 7 */ "<HCI_LE_Set_CIG_Parameters> "
	},
	{ /* byte 42 */
		/* 0 */ "<HCI_LE_Set_CIG_Parameters_Test> ",
		/* 1 */ "<HCI_LE_Create_CIS> ",
		/* 2 */ "<HCI_LE_Remove_CIG> ",
		/* 3 */ "<HCI_LE_Accept_CIS_Request> ",
		/* 4 */ "<HCI_LE_Reject_CIS_Request> ",
		/* 5 */ "<HCI_LE_Create_BIG> ",
		/* 6 */ "<HCI_LE_Create_BIG_Test> ",
		/* 7 */ "<HCI_LE_Terminate_BIG> "
	},
	{ /* byte 43 */
		/* 0 */ "<HCI_LE_BIG_Create_Sync> ",
		/* 1 */ "<HCI_LE_BIG_Terminate_Sync> ",
		/* 2 */ "<HCI_LE_Request_Peer_SCA> ",
		/* 3 */ "<HCI_LE_Setup_ISO_Data_Path> ",
		/* 4 */ "<HCI_LE_Remove_ISO_Data_Path> ",
		/* 5 */ "<HCI_LE_ISO_Transmit_Test> ",
		/* 6 */ "<HCI_LE_ISO_Receive_Test> ",
		/* 7 */ "<HCI_LE_ISO_Read_Test_Counters> "
	},
	{ /* byte 44 */
		/* 0 */ "<HCI_LE_ISO_Test_End> ",
		/* 1 */ "<HCI_LE_Set_Host_Feature> ",
		/* 2 */ "<HCI_LE_Read_ISO_Link_Quality> ",
		/* 3 */ "<HCI_LE_Enhanced_Read_Transmit_Power_Level> ",
		/* 4 */ "<HCI_LE_Read_Remote_Transmit_Power_Level> ",
		/* 5 */ "<HCI_LE_Set_Path_Loss_Reporting_Parameters> ",
		/* 6 */ "<HCI_LE_Set_Path_Loss_Reporting_Enable> ",
		/* 7 */ "<HCI_LE_Set_Transmit_Power_Reporting_Enable> "
	},
	{ /* byte 45 */
		/* 0 */ "<HCI_LE_Transmitter_Test [v4]> ",
		/* 1 */ "<HCI_Set_Ecosystem_Base_Interval> ",
		/* 2 */ "<HCI_Read_Local_Supported_Codecs [v2]> ",
		/* 3 */ "<HCI_Read_Local_Supported_Codec_Capabilities> ",
		/* 4 */ "<HCI_Read_Local_Supported_Controller_Delay> ",
		/* 5 */ "<HCI_Configure_Data_Path> ",
		/* 6 */ "<Unknown 45.6> ",
		/* 7 */ "<Unknown 45.7> "
	}};

	if (buffer != NULL && size > 0) {
		int	n, i, len0, len1;

		memset(buffer, 0, size);
		size--;


		for (n = 0; n < SIZE(t); n++) {
			for (i = 0; i < SIZE(t[n]); i++) {
				len0 = strlen(buffer);
				if (len0 >= size)
					goto done;

				if (commands[n] & (1 << i)) {
					if (len1 + strlen(t[n][i]) > 60) {
						len1 = 0;
						buffer[len0 - 1] = '\n';
					}

					len1 += strlen(t[n][i]);
					strncat(buffer, t[n][i], size - len0);
				}

			}
		}
	}
done:
	return (buffer);
} /* hci_commands2str */

char const *
hci_features2str(uint8_t *features, char *buffer, int size)
{
	static char const * const	t[][8] = {
	{ /* byte 0 */
		/* 0 */ "<3-Slot> ",
		/* 1 */ "<5-Slot> ",
		/* 2 */ "<Encryption> ",
		/* 3 */ "<Slot offset> ",
		/* 4 */ "<Timing accuracy> ",
		/* 5 */ "<Switch> ",
		/* 6 */ "<Hold mode> ",
		/* 7 */ "<Sniff mode> "
	},
	{ /* byte 1 */
		/* 0 */ "<Park mode> ",
		/* 1 */ "<RSSI> ",
		/* 2 */ "<Channel quality> ",
		/* 3 */ "<SCO link> ",
		/* 4 */ "<HV2 packets> ",
		/* 5 */ "<HV3 packets> ",
		/* 6 */ "<u-law log> ",
		/* 7 */ "<A-law log> "
	},
	{ /* byte 2 */
		/* 0 */ "<CVSD> ",
		/* 1 */ "<Paging scheme> ",
		/* 2 */ "<Power control> ",
		/* 3 */ "<Transparent SCO data> ",
		/* 4 */ "<Flow control lag (bit0)> ",
		/* 5 */ "<Flow control lag (bit1)> ",
		/* 6 */ "<Flow control lag (bit2)> ",
		/* 7 */ "<Broadcast Encryption> "
	},
	{ /* byte 3 */
		/* 0 */ "<Unknown 3.0> ",
		/* 1 */ "<EDR ACL 2 Mb/s> ",
		/* 2 */ "<EDR ACL 3 Mb/s> ",
		/* 3 */ "<Enhanced inquiry scan> ",
		/* 4 */ "<Interlaced inquiry scan> ",
		/* 5 */ "<Interlaced page scan> ",
		/* 6 */ "<RSSI with inquiry results> ",
		/* 7 */ "<Extended SCO link (EV3 packets)> "
	},
	{ /* byte 4 */
		/* 0 */ "<EV4 packets> ",
		/* 1 */ "<EV5 packets> ",
		/* 2 */ "<Unknown 4.2> ",
		/* 3 */ "<AFH capable slave> ",
		/* 4 */ "<AFH classification slave> ",
		/* 5 */ "<BR/EDR Not Supported> ",
		/* 6 */ "<LE Supported (Controller)> ",
		/* 7 */ "<3-Slot EDR ACL packets> "
	},
	{ /* byte 5 */
		/* 0 */ "<5-Slot EDR ACL packets> ",
		/* 1 */ "<Sniff subrating> ",
		/* 2 */ "<Pause encryption> ",
		/* 3 */ "<AFH capable master> ",
		/* 4 */ "<AFH classification master> ",
		/* 5 */ "<EDR eSCO 2 Mb/s mode> ",
		/* 6 */ "<EDR eSCO 3 Mb/s mode> ",
		/* 7 */ "<3-Slot EDR eSCO packets> "
	},
	{ /* byte 6 */
		/* 0 */ "<Enhanced Inquiry Response> ",
		/* 1 */ "<Simultaneous LE and BR/EDR (Controller)> ",
		/* 2 */ "<Unknown 6.2> ",
		/* 3 */ "<Secure Simple Pairing (Controller Support)> ",
		/* 4 */ "<Encapsulated PDU> ",
		/* 5 */ "<Erroneous Data Reporting> ",
		/* 6 */ "<Non-flushable Packed Boundary Flag> ",
		/* 7 */ "<Unknown 6.7> "
	},
	{ /* byte 7 */
		/* 0 */ "<HCI_Link_Supervision_Timeout_Changed event> ",
		/* 1 */ "<Variable Inquiry TX Power Level> ",
		/* 2 */ "<Enhanced Power Control> ",
		/* 3 */ "<Unknown 7.3> ",
		/* 4 */ "<Unknown 7.4> ",
		/* 5 */ "<Unknown 7.5> ",
		/* 6 */ "<Unknown 7.6> ",
		/* 7 */ "<Extended features> "
	}};

	if (buffer != NULL && size > 0) {
		int	n, i, len0, len1;

		memset(buffer, 0, size);
		len1 = 0;
		size--;

		for (n = 0; n < SIZE(t); n++) {
			for (i = 0; i < SIZE(t[n]); i++) {
				len0 = strlen(buffer);
				if (len0 >= size)
					goto done;

				if (features[n] & (1 << i)) {
					if (len1 + strlen(t[n][i]) > 60) {
						len1 = 0;
						buffer[len0 - 1] = '\n';
					}

					len1 += strlen(t[n][i]);
					strncat(buffer, t[n][i], size - len0);
				}
			}
		}
	}
done:
	return (buffer);
} /* hci_features2str */

char const *
hci_le_features2str(uint8_t *features, char *buffer, int size)
{
	static char const * const t[][8] = {
	{ /* byte 0 */
		/* 0 */ "<LE Encryption> ",
		/* 1 */ "<Connection Parameters Request Procedure> ",
		/* 2 */ "<Extended Reject Indication> ",
		/* 3 */ "<Slave-initiated Features Exchange> ",
		/* 4 */ "<LE Ping> ",
		/* 5 */ "<LE Data Packet Length Extension> ",
		/* 6 */ "<LL Privacy> ",
		/* 7 */ "<Extended Scanner Filter Policies> "
	},
	{ /* byte 1 */
		/* 0 */ "<LE 2M PHY> ",
		/* 1 */ "<Stable Modulation Index - Transmitter> ",
		/* 2 */ "<Stable Modulation Index - Receiver> ",
		/* 3 */ "<LE Coded PHY> ",
		/* 4 */ "<LE Extended Advertising> ",
		/* 5 */ "<LE Periodic Advertising> ",
		/* 6 */ "<Channel Selection Algorithm #2> ",
		/* 7 */ "<LE Power Class 1> "
	},
	{ /* byte 2 */
		/* 0 */ "<Minimum Number of Used Channels Procedure> ",
		/* 1 */ "<Connection CTE Request> ",
		/* 2 */ "<Connection CTE Response> ",
		/* 3 */ "<Connectionless CTE Transmitter> ",
		/* 4 */ "<Connectionless CTE Receiver> ",
		/* 5 */ "<Antenna Switching During CTE Transmission (AoD)> ",
		/* 6 */ "<Antenna Switching During CTE Reception (AoA)> ",
		/* 7 */ "<Receiving Constant Tone Extensions> "
	},
	{ /* byte 3 */
		/* 0 */ "<Periodic Advertising Sync Transfer - Sender> ",
		/* 1 */ "<Periodic Advertising Sync Transfer - Recipient> ",
		/* 2 */ "<Sleep Clock Accuracy Updates> ",
		/* 3 */ "<Remote Public Key Validation> ",
		/* 4 */ "<Connected Isochronous Stream - Master> ",
		/* 5 */ "<Connected Isochronous Stream - Slave> ",
		/* 6 */ "<Isochronous Broadcaster> ",
		/* 7 */ "<Synchronized Receiver> "
	},
	{ /* byte 4 */
		/* 0 */ "<Isochronous Channels (Host Support)> ",
		/* 1 */ "<LE Power Control Request> ",
		/* 2 */ "<LE Power Change Indication> ",
		/* 3 */ "<LE Path Loss Monitoring> ",
		/* 4 */ "<Reserved for future use> ",
		/* 5 */ "<Unknown 4.5> ",
		/* 6 */ "<Unknown 4.6> ",
		/* 7 */ "<Unknown 4.7> "
	},
	{ /* byte 5 */
		/* 0 */ "<Unknown 5.0> ",
		/* 1 */ "<Unknown 5.1> ",
		/* 2 */ "<Unknown 5.2> ",
		/* 3 */ "<Unknown 5.3> ",
		/* 4 */ "<Unknown 5.4> ",
		/* 5 */ "<Unknown 5.5> ",
		/* 6 */ "<Unknown 5.6> ",
		/* 7 */ "<Unknown 5.7> "
	},
	{ /* byte 6 */
		/* 0 */ "<Unknown 6.0> ",
		/* 1 */ "<Unknown 6.1> ",
		/* 2 */ "<Unknown 6.2> ",
		/* 3 */ "<Unknown 6.3> ",
		/* 4 */ "<Unknown 6.4> ",
		/* 5 */ "<Unknown 6.5> ",
		/* 6 */ "<Unknown 6.6> ",
		/* 7 */ "<Unknown 6.7> "
	},
	{ /* byte 7 */
		/* 0 */ "<Unknown 7.0> ",
		/* 1 */ "<Unknown 7.1> ",
		/* 2 */ "<Unknown 7.2> ",
		/* 3 */ "<Unknown 7.3> ",
		/* 4 */ "<Unknown 7.4> ",
		/* 5 */ "<Unknown 7.5> ",
		/* 6 */ "<Unknown 7.6> ",
		/* 7 */ "<Unknown 7.7> "
	}};

	if (buffer != NULL && size > 0) {
		int n, i, len0, len1;

		memset(buffer, 0, size);
		len1 = 0;
		size--;

		for (n = 0; n < SIZE(t); n++) {
			for (i = 0; i < SIZE(t[n]); i++) {
				len0 = strlen(buffer);
				if (len0 >= size)
					goto done;

				if (features[n] & (1 << i)) {
					if (len1 + strlen(t[n][i]) > 60) {
						len1 = 0;
						buffer[len0 - 1] = '\n';
					}

					len1 += strlen(t[n][i]);
					strncat(buffer, t[n][i], size - len0);
				}
			}
		}
	}
done:
	return (buffer);
}

char const *
hci_cc2str(int cc)
{
	static char const * const	t[] = {
		/* 0x00 */ "North America, Europe, Japan",
		/* 0x01 */ "France"
	};

	return (cc >= SIZE(t)? "?" : t[cc]);
} /* hci_cc2str */

char const *
hci_con_state2str(int state)
{
	static char const * const	t[] = {
		/* NG_HCI_CON_CLOSED */           "CLOSED",
		/* NG_HCI_CON_W4_LP_CON_RSP */    "W4_LP_CON_RSP",
		/* NG_HCI_CON_W4_CONN_COMPLETE */ "W4_CONN_COMPLETE",
		/* NG_HCI_CON_OPEN */             "OPEN"
        };

	return (state >= SIZE(t)? "UNKNOWN" : t[state]);
} /* hci_con_state2str */

char const *
hci_status2str(int status)
{
	static char const * const       t[] = {
		/* 0x00 */ "No error",
		/* 0x01 */ "Unknown HCI command",
		/* 0x02 */ "No connection",
		/* 0x03 */ "Hardware failure",
		/* 0x04 */ "Page timeout",
		/* 0x05 */ "Authentication failure",
		/* 0x06 */ "Key missing",
		/* 0x07 */ "Memory full",
		/* 0x08 */ "Connection timeout",
		/* 0x09 */ "Max number of connections",
		/* 0x0a */ "Max number of SCO connections to a unit",
		/* 0x0b */ "ACL connection already exists",
		/* 0x0c */ "Command disallowed",
		/* 0x0d */ "Host rejected due to limited resources",
		/* 0x0e */ "Host rejected due to security reasons",
		/* 0x0f */ "Host rejected due to remote unit is a personal unit",
		/* 0x10 */ "Host timeout",
		/* 0x11 */ "Unsupported feature or parameter value",
		/* 0x12 */ "Invalid HCI command parameter",
		/* 0x13 */ "Other end terminated connection: User ended connection",
		/* 0x14 */ "Other end terminated connection: Low resources",
		/* 0x15 */ "Other end terminated connection: About to power off",
		/* 0x16 */ "Connection terminated by local host",
		/* 0x17 */ "Repeated attempts",
		/* 0x18 */ "Pairing not allowed",
		/* 0x19 */ "Unknown LMP PDU",
		/* 0x1a */ "Unsupported remote feature",
		/* 0x1b */ "SCO offset rejected",
		/* 0x1c */ "SCO interval rejected",
		/* 0x1d */ "SCO air mode rejected",
		/* 0x1e */ "Invalid LMP parameters",
		/* 0x1f */ "Unspecified error",
		/* 0x20 */ "Unsupported LMP parameter value",
		/* 0x21 */ "Role change not allowed",
		/* 0x22 */ "LMP response timeout",
		/* 0x23 */ "LMP error transaction collision",
		/* 0x24 */ "LMP PSU not allowed",
		/* 0x25 */ "Encryption mode not acceptable",
		/* 0x26 */ "Unit key used",
		/* 0x27 */ "QoS is not supported",
		/* 0x28 */ "Instant passed",
		/* 0x29 */ "Pairing with unit key not supported",
		/* 0x2a */ "Different Transaction Collision",
		/* 0x2b */ "Unknown error (Reserved for future use)",
		/* 0x2c */ "QoS Unacceptable Parameter",
		/* 0x2d */ "QoS Rejected",
		/* 0x2e */ "Channel Classification Not Supported",
		/* 0x2f */ "Insufficient Security",
		/* 0x30 */ "Parameter Out Of Mandatory Range",
		/* 0x31 */ "Unknown error (Reserved for future use)",
		/* 0x32 */ "Role Switch Pending",
		/* 0x33 */ "Unknown error (Reserved for future use)",
		/* 0x34 */ "Reserved Slot Violation",
		/* 0x35 */ "Role Switch Failed",
		/* 0x36 */ "Extended Inquiry Response Too Large",
		/* 0x37 */ "Secure Simple Pairing Not Supported By Host",
		/* 0x38 */ "Host Busy - Pairing",
		/* 0x39 */ "Connection Rejected due to No Suitable Channel Found",
		/* 0x3a */ "Controller Busy",
		/* 0x3b */ "Unacceptable Connection Parameters",
		/* 0x3c */ "Advertising Timeout",
		/* 0x3d */ "Connection Terminated due to MIC Failure",
		/* 0x3e */ "Connection Failed to be Established / Synchronization Timeout",
		/* 0x3f */ "MAC Connection Failed",
		/* 0x40 */ "Coarse Clock Adjustment Rejected but Will Try to Adjust Using Clock Dragging",
		/* 0x41 */ "Type0 Submap Not Defined",
		/* 0x42 */ "Unknown Advertising Identifier",
		/* 0x43 */ "Limit Reached",
		/* 0x44 */ "Operation Cancelled by Host",
		/* 0x45 */ "Packet Too Long"
	};

	return (status >= SIZE(t)? "Unknown error" : t[status]);
} /* hci_status2str */

char const *
hci_bdaddr2str(bdaddr_t const *ba)
{
	extern int	 numeric_bdaddr;
	static char	 buffer[MAXHOSTNAMELEN];
	struct hostent	*he = NULL;

	if (memcmp(ba, NG_HCI_BDADDR_ANY, sizeof(*ba)) == 0) {
		buffer[0] = '*';
		buffer[1] = 0;

		return (buffer);
	}

	if (!numeric_bdaddr &&
	    (he = bt_gethostbyaddr((char *)ba, sizeof(*ba), AF_BLUETOOTH)) != NULL) {
		strlcpy(buffer, he->h_name, sizeof(buffer));

		return (buffer);
	}

	bt_ntoa(ba, buffer);

	return (buffer);
} /* hci_bdaddr2str */


char const *
hci_addrtype2str(int type)
{
	static char const * const	t[] = {
		/* 0x00 */ "Public Device Address",
		/* 0x01 */ "Random Device Address",
		/* 0x02 */ "Public Identity Address",
		/* 0x03 */ "Random (static) Identity Address"
	};

	return (type >= SIZE(t)? "?" : t[type]);
} /* hci_addrtype2str */

char const *
hci_role2str(int role)
{
	static char const * const	roles[] = {
		/* 0x00 */ "Master",
		/* 0x01 */ "Slave",
	};

	return (role >= SIZE(roles)? "Unknown role" : roles[role]);
} /* hci_role2str */

char const *
hci_mc_accuracy2str(int accuracy)
{
	static char const * const	acc[] = {
		/* 0x00 */ "500 ppm",
		/* 0x01 */ "250 ppm",
		/* 0x02 */ "150 ppm",
		/* 0x03 */ "100 ppm",
		/* 0x04 */ "75 ppm",
		/* 0x05 */ "50 ppm",
		/* 0x06 */ "30 ppm",
		/* 0x07 */ "20 ppm",
	};

	return (accuracy >= SIZE(acc)? "Unknown accuracy" : acc[accuracy]);
} /* hci_mc_accuracy2str */

char const *
hci_le_chanmap2str(uint8_t *map, char *buffer, int size)
{
	char	chantxt[4];
	if (buffer != NULL && size > 0) {
		int n, i, len0, len1;

		memset(buffer, 0, size);
		len1 = 0;
		size--;

		for (n = 0; n < 5; n++) {
			fprintf(stdout, "%02x ", map[n]);
			for (i = 0; i < 8; i++) {
				len0 = strlen(buffer);
				if (len0 >= size)
					goto done;

				if (map[n] & (1 << i)) {
					if (len1 + 3 > 60) {
						len1 = 0;
						buffer[len0 - 1] = '\n';
					}

					len1 += 3;
					snprintf(
						chantxt,
						sizeof(chantxt),
						"%02d ",
						(n * 8 + i));
					strncat(
						buffer,
						chantxt,
						size - len0);
				}
			}
		}
		fprintf(stdout, "\n");
	}
done:
	return (buffer);
}
