%{

/*
   +----------------------------------------------------------------------+
   | Zend Engine                                                          |
   +----------------------------------------------------------------------+
   | Copyright (c) 1998-2002 Zend Technologies Ltd. (http://www.zend.com) |
   +----------------------------------------------------------------------+
   | This source file is subject to version 2.00 of the Zend license,     |
   | that is bundled with this package in the file LICENSE, and is        | 
   | available at through the world-wide-web at                           |
   | http://www.zend.com/license/2_00.txt.                                |
   | If you did not receive a copy of the Zend license and are unable to  |
   | obtain it through the world-wide-web, please send a note to          |
   | license@zend.com so we can mail you a copy immediately.              |
   +----------------------------------------------------------------------+
   | Authors: Andi Gutmans <andi@zend.com>                                |
   |          Zeev Suraski <zeev@zend.com>                                |
   +----------------------------------------------------------------------+
*/

#define yyleng SCNG(yy_leng)
#define yytext SCNG(yy_text)
#define yytext_ptr SCNG(yy_text)
#define yyin SCNG(yy_in)
#define yyout SCNG(yy_out)
#define yy_last_accepting_state SCNG(_yy_last_accepting_state)
#define yy_last_accepting_cpos SCNG(_yy_last_accepting_cpos)
#define yy_more_flag SCNG(_yy_more_flag)
#define yy_more_len SCNG(_yy_more_len)

%}

%x ST_IN_SCRIPTING
%x ST_DOUBLE_QUOTES
%x ST_SINGLE_QUOTE
%x ST_BACKQUOTE
%x ST_HEREDOC
%x ST_LOOKING_FOR_PROPERTY
%x ST_LOOKING_FOR_VARNAME
%x ST_COMMENT
%x ST_ONE_LINE_COMMENT
%option stack

%{

#include <errno.h>
#include "zend.h"
#include "zend_alloc.h"
#include "zend_language_parser.h"
#include "zend_compile.h"
#include "zend_language_scanner.h"
#include "zend_highlight.h"
#include "zend_constants.h"
#include "zend_variables.h"
#include "zend_operators.h"

#ifdef HAVE_STDARG_H
# include <stdarg.h>
#endif

#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif

#ifdef ZEND_WIN32
#include <io.h>
#endif

#define YY_DECL int lex_scan(zval *zendlval TSRMLS_DC)

#define ECHO { ZEND_WRITE( yytext, yyleng ); }

#ifdef ZTS
#  define MY_INPUT yyinput
#else
#  define MY_INPUT input
#endif


#ifndef STDIN_FILENO
# define STDIN_FILENO 0
#endif


/* Globals Macros */
#define SCNG	LANG_SCNG
#ifdef ZTS
ZEND_API ts_rsrc_id language_scanner_globals_id;
#else
ZEND_API zend_scanner_globals language_scanner_globals;
#endif


#define YY_FATAL_ERROR zend_fatal_scanner_error

#define HANDLE_NEWLINES(s, l)													\
do {																			\
	char *p = (s), *boundary = p+(l);											\
																				\
	while (p<boundary) {														\
		if (*p == '\n') {														\
			CG(zend_lineno)++;													\
		} else if ((*p == '\r') && (p+1 < boundary) && (*(p+1) != '\n')) {		\
			CG(zend_lineno)++;													\
		}																		\
		p++;																	\
	}																			\
} while (0)

#define HANDLE_NEWLINE(c) \
{ \
	if (c == '\n' || c == '\r') { \
		CG(zend_lineno)++; \
	} \
}
		

#define ZEND_IS_OCT(c)  ((c)>='0' && (c)<='7')
#define ZEND_IS_HEX(c)  (((c)>='0' && (c)<='9') || ((c)>='a' && (c)<='f') || ((c)>='A' && (c)<='F'))


void zend_fatal_scanner_error(char *message)
{
	zend_error(E_COMPILE_ERROR, "%s", message);
}

BEGIN_EXTERN_C()
void startup_scanner(TSRMLS_D)
{
	CG(heredoc) = NULL;
	CG(heredoc_len)=0;
	SCNG(yy_start_stack_ptr) = 0;
	SCNG(yy_start_stack_depth) = 0;
#ifdef ZEND_MULTIBYTE
	SCNG(code) = NULL;
	SCNG(code_size) = 0;
	SCNG(current_code) = NULL;
	SCNG(current_code_size) = 0;
#endif /* ZEND_MULTIBYTE */
}


void shutdown_scanner(TSRMLS_D)
{
	if (CG(heredoc)) {
		efree(CG(heredoc));
		CG(heredoc_len)=0;
	}
#ifdef ZEND_MULTIBYTE
	if (SCNG(code)) {
		efree(SCNG(code));
		SCNG(code) = NULL;
	}
	if (SCNG(current_code)) {
		efree(SCNG(current_code));
		SCNG(current_code) = NULL;
	}
#endif /* ZEND_MULTIBYTE */
}
END_EXTERN_C()


ZEND_API void zend_save_lexical_state(zend_lex_state *lex_state TSRMLS_DC)
{
	memcpy(&lex_state->buffer_state, &YY_CURRENT_BUFFER, sizeof(YY_BUFFER_STATE));
	lex_state->in = SCNG(yy_in);
	lex_state->state = YYSTATE;
	lex_state->filename = zend_get_compiled_filename(TSRMLS_C);
	lex_state->lineno = CG(zend_lineno);
#ifdef ZEND_MULTIBYTE
	lex_state->code = SCNG(code);
	lex_state->code_size = SCNG(code_size);
	lex_state->current_code = SCNG(current_code);
	lex_state->current_code_size = SCNG(current_code_size);
	lex_state->input_filter = SCNG(input_filter);
	lex_state->output_filter = SCNG(output_filter);
	lex_state->script_encoding = SCNG(script_encoding);
	lex_state->internal_encoding = SCNG(internal_encoding);
#endif /* ZEND_MULTIBYTE */
}


ZEND_API void zend_restore_lexical_state(zend_lex_state *lex_state TSRMLS_DC)
{
	YY_BUFFER_STATE original_buffer_state = YY_CURRENT_BUFFER;

	if (lex_state->buffer_state) {
		yy_switch_to_buffer(lex_state->buffer_state TSRMLS_CC);
	} else {
		YY_CURRENT_BUFFER = NULL;
	}
		
	yy_delete_buffer(original_buffer_state TSRMLS_CC);
	SCNG(yy_in) = lex_state->in;
	BEGIN(lex_state->state);
	CG(zend_lineno) = lex_state->lineno;
	zend_restore_compiled_filename(lex_state->filename TSRMLS_CC);
#ifdef ZEND_MULTIBYTE
	SCNG(code) = lex_state->code;
	SCNG(code_size) = lex_state->code_size;
	SCNG(current_code) = lex_state->current_code;
	SCNG(current_code_size) = lex_state->current_code_size;
	SCNG(input_filter) = lex_state->input_filter;
	SCNG(output_filter) = lex_state->output_filter;
	SCNG(script_encoding) = lex_state->script_encoding;
	SCNG(internal_encoding) = lex_state->internal_encoding;
#endif /* ZEND_MULTIBYTE */
}


BEGIN_EXTERN_C()


ZEND_API void zend_file_handle_dtor(zend_file_handle *fh)
{
	switch (fh->type) {
		case ZEND_HANDLE_SOCKET_FD:
#ifdef ZEND_WIN32
			closesocket(fh->handle.fd);
			break;
#endif
		/* fall-through */ 
		case ZEND_HANDLE_FD:
			close(fh->handle.fd);
			break;
		case ZEND_HANDLE_FP:
			fclose(fh->handle.fp);
			break;
		case ZEND_HANDLE_FILENAME:
			/* We're only supposed to get here when destructing the used_files hash,
			 * which doesn't really contain open files, but references to their names/paths
			 */
			break;
	}
	if (fh->opened_path) {
		efree(fh->opened_path);
		fh->opened_path = NULL;
	}
	if (fh->free_filename && fh->filename) {
		efree(fh->filename);
		fh->filename = NULL;
	}
}


int zend_compare_file_handles(zend_file_handle *fh1, zend_file_handle *fh2)
{
	if (fh1->type != fh2->type) {
		return 0;
	}
	switch (fh1->type) {
		case ZEND_HANDLE_FP:
			return fh1->handle.fp==fh2->handle.fp;
			break;
		case ZEND_HANDLE_SOCKET_FD:
		case ZEND_HANDLE_FD:
			return fh1->handle.fd==fh2->handle.fd;
			break;
	}
	return 0;
}


ZEND_API void zend_destroy_file_handle(zend_file_handle *file_handle TSRMLS_DC)
{
	zend_llist_del_element(&CG(open_files), file_handle, (int (*)(void *, void *)) zend_compare_file_handles);
}


ZEND_API int open_file_for_scanning(zend_file_handle *file_handle TSRMLS_DC)
{
	char *file_path=NULL;
	int do_add = 1;

	switch (file_handle->type) {
		case ZEND_HANDLE_FILENAME:
			if (zend_open(file_handle->filename, file_handle) != SUCCESS)
				return FAILURE;
			break;
	}

	if (!ZEND_IS_VALID_FILE_HANDLE(file_handle))
			return FAILURE;
	
	if (file_handle->type == ZEND_HANDLE_FP && file_handle->handle.fp == stdin)
		do_add = 0;
	else if (file_handle->type == ZEND_HANDLE_FD && file_handle->handle.fd == STDIN_FILENO)
		do_add = 0;
	
	if (do_add) {
		zend_llist_add_element(&CG(open_files), file_handle);
	}
	/* Reset the scanner for scanning the new file */
	SCNG(yy_in) = file_handle;
#ifdef ZEND_MULTIBYTE
	if (zend_read_file(TSRMLS_C) != 0) {
		return FAILURE;
	}
	
	/* force flex to use buffer only */
	SCNG(yy_in) = NULL;
	SCNG(init) = 0;
	SCNG(start) = 1;

	zend_multibyte_set_filter(NULL TSRMLS_CC);

	if (!SCNG(input_filter)) {
		SCNG(current_code) = (char*)emalloc(SCNG(code_size)+1);
		memcpy(SCNG(current_code), SCNG(code), SCNG(code_size)+1);
		SCNG(current_code_size) = SCNG(code_size);
	} else {
		SCNG(input_filter)(&SCNG(current_code), &SCNG(current_code_size), SCNG(code), SCNG(code_size) TSRMLS_CC);
		if (!SCNG(current_code) || !SCNG(current_code_size)) {
			return FAILURE;
		}
	}
	
	/* we need two nulls so that ralloc... */
	SCNG(current_code) = (char*)erealloc(SCNG(current_code), SCNG(current_code_size)+2);
	*(SCNG(current_code)+SCNG(current_code_size)) = (char)NULL;
	*(SCNG(current_code)+SCNG(current_code_size)+1) = (char)NULL;
	yy_scan_buffer(SCNG(current_code), SCNG(current_code_size)+2 TSRMLS_CC);
#else /* ZEND_MULTIBYTE */
	yy_switch_to_buffer(yy_create_buffer(SCNG(yy_in), YY_BUF_SIZE TSRMLS_CC) TSRMLS_CC);
#endif /* ZEND_MULTIBYTE */
	BEGIN(INITIAL);

	if (file_handle->opened_path) {
		file_path = file_handle->opened_path; 
	} else {
		file_path = file_handle->filename;
	}

	zend_set_compiled_filename(file_path TSRMLS_CC);
	
	if (CG(zend_lineno) < 0) { /* position is (n_lines * -1), position was changed by an external app */
		CG(zend_lineno) = CG(zend_lineno) * -1;
	} else {
		CG(zend_lineno) = 1;
	}
	
	CG(increment_lineno) = 0;
	return SUCCESS;
}
END_EXTERN_C()


ZEND_API zend_op_array *compile_file(zend_file_handle *file_handle, int type TSRMLS_DC)
{
	zend_lex_state original_lex_state;
	zend_op_array *op_array = (zend_op_array *) emalloc(sizeof(zend_op_array));
	zend_op_array *original_active_op_array = CG(active_op_array);
	zend_op_array *retval=NULL;
	int compiler_result;
	zend_bool compilation_successful=0;
	znode retval_znode;
	zend_bool original_in_compilation = CG(in_compilation);

	retval_znode.op_type = IS_CONST;
	retval_znode.u.constant.type = IS_LONG;
	retval_znode.u.constant.value.lval = 1;
	retval_znode.u.constant.is_ref = 0;
	retval_znode.u.constant.refcount = 1;

	zend_save_lexical_state(&original_lex_state TSRMLS_CC);

	retval = op_array; /* success oriented */

	if (open_file_for_scanning(file_handle TSRMLS_CC)==FAILURE) {
		if (type==ZEND_REQUIRE) {
			zend_message_dispatcher(ZMSG_FAILED_REQUIRE_FOPEN, file_handle->filename);
			zend_bailout();		
		} else {
			zend_message_dispatcher(ZMSG_FAILED_INCLUDE_FOPEN, file_handle->filename);
		}
		compilation_successful=0;
	} else {
		init_op_array(op_array, ZEND_USER_FUNCTION, INITIAL_OP_ARRAY_SIZE TSRMLS_CC);
		CG(in_compilation) = 1;
		CG(active_op_array) = op_array;
		compiler_result = zendparse(TSRMLS_C);
		zend_do_return(&retval_znode, 0 TSRMLS_CC);
		CG(in_compilation) = original_in_compilation;
		if (compiler_result==1) { /* parser error */
			zend_bailout();
		}
		compilation_successful=1;
#ifdef ZEND_MULTIBYTE
		if (SCNG(code)) {
			efree(SCNG(code));
			SCNG(code) = NULL;
		}
		if (SCNG(current_code)) {
			efree(SCNG(current_code));
			SCNG(current_code) = NULL;
		}
#endif /* ZEND_MULTIBYTE */
	}

	if (retval) {
		CG(active_op_array) = original_active_op_array;
		if (compilation_successful) {
			pass_two(op_array TSRMLS_CC);
		} else {
			efree(op_array);
			retval = NULL;
		}
	}
	if (compilation_successful) {
		zend_restore_lexical_state(&original_lex_state TSRMLS_CC);
	}
	return retval;
}


zend_op_array *compile_filename(int type, zval *filename TSRMLS_DC)
{
	zend_file_handle file_handle = {0};
	zval tmp;
	zend_op_array *retval;
	char *opened_path = NULL;

	if (filename->type != IS_STRING) {
		tmp = *filename;
		zval_copy_ctor(&tmp);
		convert_to_string(&tmp);
		filename = &tmp;
	}
	file_handle.filename = filename->value.str.val;
	file_handle.free_filename = 0;
	file_handle.type = ZEND_HANDLE_FILENAME;
	file_handle.opened_path = NULL;

	retval = zend_compile_file(&file_handle, type TSRMLS_CC);
	if (retval && ZEND_IS_VALID_FILE_HANDLE(&file_handle)) {
		int dummy = 1;
	
		if (!file_handle.opened_path) {
			file_handle.opened_path = opened_path = estrndup(filename->value.str.val, filename->value.str.len);
		}
		
		zend_hash_add(&EG(included_files), file_handle.opened_path, strlen(file_handle.opened_path)+1, (void *)&dummy, sizeof(int), NULL);
		
		if (opened_path) {
			efree(opened_path);
		}
	}
	zend_destroy_file_handle(&file_handle TSRMLS_CC);

	if (filename==&tmp) {
		zval_dtor(&tmp);
	}
	return retval;
}

ZEND_API int zend_prepare_string_for_scanning(zval *str, char *filename TSRMLS_DC)
{
	/* enforce two trailing NULLs for flex... */
	STR_REALLOC(str->value.str.val, str->value.str.len+2);

	str->value.str.val[str->value.str.len+1]=0;

	SCNG(yy_in)=NULL;
#ifdef ZEND_MULTIBYTE
	/* instead of zend_readfile() */
	SCNG(code) = estrdup(str->value.str.val);
	SCNG(code_size) = str->value.str.len;

	zend_multibyte_set_filter(CG(internal_encoding) TSRMLS_CC);

	if (!SCNG(input_filter)) {
		SCNG(current_code) = (char*)emalloc(SCNG(code_size)+1);
		memcpy(SCNG(current_code), SCNG(code), SCNG(code_size)+1);
		SCNG(current_code_size) = SCNG(code_size);
	} else {
		SCNG(input_filter)(&SCNG(current_code), &SCNG(current_code_size), SCNG(code), SCNG(code_size) TSRMLS_CC);
		if (!SCNG(current_code) || !SCNG(current_code_size)) {
			return FAILURE;
		}
	}

	SCNG(current_code) = (char*)erealloc(SCNG(current_code), SCNG(current_code_size)+2);
	*(SCNG(current_code)+SCNG(current_code_size)) = (char)NULL;
	*(SCNG(current_code)+SCNG(current_code_size)+1) = (char)NULL;
	yy_scan_buffer(SCNG(current_code), SCNG(current_code_size)+2 TSRMLS_CC);
#else /* ZEND_MULTIBYTE */
	yy_scan_buffer(str->value.str.val, str->value.str.len+2 TSRMLS_CC);
#endif /* ZEND_MULTIBYTE */

	zend_set_compiled_filename(filename TSRMLS_CC);
	CG(zend_lineno) = 1;
	CG(increment_lineno) = 0;
	return SUCCESS;
}


zend_op_array *compile_string(zval *source_string, char *filename TSRMLS_DC)
{
	zend_lex_state original_lex_state;
	zend_op_array *op_array = (zend_op_array *) emalloc(sizeof(zend_op_array));
	zend_op_array *original_active_op_array = CG(active_op_array);
	zend_op_array *retval;
	zval tmp;
	int compiler_result;
	zend_bool original_in_compilation = CG(in_compilation);

	if (source_string->value.str.len==0) {
		efree(op_array);
		return NULL;
	}

	CG(in_compilation) = 1;

	tmp = *source_string;
	zval_copy_ctor(&tmp);
	convert_to_string(&tmp);
	source_string = &tmp;

	zend_save_lexical_state(&original_lex_state TSRMLS_CC);
	if (zend_prepare_string_for_scanning(source_string, filename TSRMLS_CC)==FAILURE) {
		efree(op_array);
		retval = NULL;
	} else {
		init_op_array(op_array, ZEND_EVAL_CODE, INITIAL_OP_ARRAY_SIZE TSRMLS_CC);
		CG(active_op_array) = op_array;
		BEGIN(ST_IN_SCRIPTING);
		compiler_result = zendparse(TSRMLS_C);
#ifdef ZEND_MULTIBYTE
		if (SCNG(code)) {
			efree(SCNG(code));
			SCNG(code) = NULL;
		}
		if (SCNG(current_code)) {
			efree(SCNG(current_code));
			SCNG(current_code) = NULL;
		}
#endif /* ZEND_MULTIBYTE */
		
		if (compiler_result==1) {
			CG(active_op_array) = original_active_op_array;
			CG(unclean_shutdown)=1;
			retval = NULL;
		} else {
			zend_do_return(NULL, 0 TSRMLS_CC);
			CG(active_op_array) = original_active_op_array;
			pass_two(op_array TSRMLS_CC);
			retval = op_array;
		}
		zend_restore_lexical_state(&original_lex_state TSRMLS_CC);
	}
	zval_dtor(&tmp);
	CG(in_compilation) = original_in_compilation;
	return retval;
}


BEGIN_EXTERN_C()
int highlight_file(char *filename, zend_syntax_highlighter_ini *syntax_highlighter_ini TSRMLS_DC)
{
	zend_lex_state original_lex_state;
	zend_file_handle file_handle = {0};

	file_handle.type = ZEND_HANDLE_FILENAME;
	file_handle.filename = filename;
	file_handle.free_filename = 0;
	file_handle.opened_path = NULL;
	zend_save_lexical_state(&original_lex_state TSRMLS_CC);
	if (open_file_for_scanning(&file_handle TSRMLS_CC)==FAILURE) {
		zend_message_dispatcher(ZMSG_FAILED_HIGHLIGHT_FOPEN, filename);
		return FAILURE;
	}
	zend_highlight(syntax_highlighter_ini TSRMLS_CC);
#ifdef ZEND_MULTIBYTE
	if (SCNG(code)) {
		efree(SCNG(code));
		SCNG(code) = NULL;
	}
	if (SCNG(current_code)) {
		efree(SCNG(current_code));
		SCNG(current_code) = NULL;
	}
#endif /* ZEND_MULTIBYTE */
	zend_destroy_file_handle(&file_handle TSRMLS_CC);
	zend_restore_lexical_state(&original_lex_state TSRMLS_CC);
	return SUCCESS;
}

int highlight_string(zval *str, zend_syntax_highlighter_ini *syntax_highlighter_ini, char *str_name TSRMLS_DC)
{
	zend_lex_state original_lex_state;
	zval tmp = *str;

	str = &tmp;
	zval_copy_ctor(str);
	zend_save_lexical_state(&original_lex_state TSRMLS_CC);
	if (zend_prepare_string_for_scanning(str, str_name TSRMLS_CC)==FAILURE) {
		return FAILURE;
	}
	zend_highlight(syntax_highlighter_ini TSRMLS_CC);
#ifdef ZEND_MULTIBYTE
	if (SCNG(code)) {
		efree(SCNG(code));
		SCNG(code) = NULL;
	}
	if (SCNG(current_code)) {
		efree(SCNG(current_code));
		SCNG(current_code) = NULL;
	}
#endif /* ZEND_MULTIBYTE */
	zend_restore_lexical_state(&original_lex_state TSRMLS_CC);
	zval_dtor(str);
	return SUCCESS;
}
END_EXTERN_C()

#ifdef ZEND_MULTIBYTE
BEGIN_EXTERN_C()
ZEND_API void zend_yyinput_again(zend_multibyte_filter old_input_filter, zend_encoding *old_encoding TSRMLS_DC)
{
	YY_BUFFER_STATE b = YY_CURRENT_BUFFER;
	int offset, original_offset, length, free_flag;
	char *p;
	zend_encoding *new_encoding;

	/* calculate current position */
	offset = original_offset = yy_c_buf_p - b->yy_ch_buf;
	if (old_input_filter && original_offset > 0) {
		/* we need to adjust offset ? (i've gotta improve this...) */
		new_encoding = SCNG(script_encoding);
		SCNG(script_encoding) = old_encoding;
		do {
			(old_input_filter)(&p, &length, SCNG(code), offset TSRMLS_CC);
			if (!p) {
				SCNG(script_encoding) = new_encoding;
				return;
			}
			efree(p);
			if (length > original_offset) {
				offset--;
			} else if (length < original_offset) {
				offset++;
			}
		} while (original_offset != length);
		SCNG(script_encoding) = new_encoding;
	}

	/* convert and set */
	if (!SCNG(input_filter)) {
		length = SCNG(code_size)-offset-1;
		p = SCNG(code)+offset+1;
		free_flag = 0;
	} else {
		SCNG(input_filter)(&p, &length, SCNG(code)+offset+1, SCNG(code_size)-offset-1 TSRMLS_CC);
		free_flag = 1;
	}
	if (original_offset+length+1 > (int)b->yy_buf_size) {
		b->yy_buf_size = original_offset+length+1;
		b->yy_ch_buf = (char*)erealloc(b->yy_ch_buf, b->yy_buf_size+2);
		SCNG(current_code) = b->yy_ch_buf;
		SCNG(current_code_size) = b->yy_buf_size;
	}
	yy_c_buf_p = b->yy_ch_buf + original_offset;
	strncpy(yy_c_buf_p+1, p, length);
 	b->yy_n_chars = original_offset + length + 1;
	SCNG(yy_n_chars) = b->yy_n_chars;
	b->yy_ch_buf[SCNG(yy_n_chars)] = YY_END_OF_BUFFER_CHAR;
	b->yy_ch_buf[SCNG(yy_n_chars)+1] = YY_END_OF_BUFFER_CHAR;

	if (free_flag) {
		efree(p);
	}
}


ZEND_API int zend_read_file(TSRMLS_D)
{
	char buf[8192];
	int n;

	SCNG(code) = NULL;
	SCNG(code_size) = 0;

	for (;;) {
		switch (SCNG(yy_in)->type) {
			case ZEND_HANDLE_FD:
				n = read(SCNG(yy_in)->handle.fd, buf, sizeof(buf));
				break;
			case ZEND_HANDLE_FP:
				n = fread(buf, sizeof(buf[0]), sizeof(buf) / sizeof(buf[0]), SCNG(yy_in)->handle.fp);
				break;
			case ZEND_HANDLE_SOCKET_FD:
				n =	recv(SCNG(yy_in)->handle.fd, buf, sizeof(buf), 0);
		}

		if (n <= 0) {
			break;
		}

		SCNG(code_size) += n;
		if (SCNG(code)) {
			SCNG(code) = (char*)erealloc(SCNG(code), SCNG(code_size)+1);
		} else {
			SCNG(code) = (char*)emalloc(SCNG(code_size)+1);
		}
		memcpy(SCNG(code)+SCNG(code_size)-n, buf, n);
	}

	if (n < 0) {
		return -1;
	}

	if (!SCNG(code)) {
		SCNG(code) = emalloc(SCNG(code_size)+1);
	}
	*(SCNG(code)+SCNG(code_size)) = (char)NULL;

	return 0;
}


/*
 *	macros for yytext -> zval
 */
# define zend_copy_value(zendlval, yytext, yyleng) \
	if (SCNG(output_filter)) { \
		SCNG(output_filter)(&(zendlval->value.str.val), &(zendlval->value.str.len), yytext, yyleng TSRMLS_CC); \
	} else { \
		zendlval->value.str.val = (char *) estrndup(yytext, yyleng); \
		zendlval->value.str.len = yyleng; \
	}
#else /* ZEND_MULTIBYTE */
# define zend_copy_value(zendlval, yytext, yyleng) \
	zendlval->value.str.val = (char *)estrndup(yytext, yyleng); \
	zendlval->value.str.len = yyleng;
#endif /* ZEND_MULTIBYTE */
%}

LNUM	[0-9]+
DNUM	([0-9]*[\.][0-9]+)|([0-9]+[\.][0-9]*)
EXPONENT_DNUM	(({LNUM}|{DNUM})[eE][+-]?{LNUM})
HNUM	"0x"[0-9a-fA-F]+
LABEL	[a-zA-Z_\x7f-\xff][a-zA-Z0-9_\x7f-\xff]*
WHITESPACE [ \n\r\t]+
TABS_AND_SPACES [ \t]*
TOKENS [;:,.\[\]()|^&+-/*=%!~$<>?@]
ENCAPSED_TOKENS [\[\]{}$]
ESCAPED_AND_WHITESPACE [\n\t\r #'.:;,()|^&+-/*=%!~<>?@]+
ANY_CHAR (.|[\n])
NEWLINE ("\r"|"\n"|"\r\n")

%option noyylineno
%option noyywrap
%%

<ST_IN_SCRIPTING>"exit" {
	return T_EXIT;
}

<ST_IN_SCRIPTING>"die" {
	return T_EXIT;
}

<ST_IN_SCRIPTING>"old_function" {
	return T_OLD_FUNCTION;
}

<ST_IN_SCRIPTING>"function"|"cfunction" {
	return T_FUNCTION;
}

<ST_IN_SCRIPTING>"const" {
	return T_CONST;
}

<ST_IN_SCRIPTING>"return" {
	return T_RETURN;
}

<ST_IN_SCRIPTING>"if" {
	return T_IF;
}

<ST_IN_SCRIPTING>"elseif" {
	return T_ELSEIF;
}

<ST_IN_SCRIPTING>"endif" {
	return T_ENDIF;
}

<ST_IN_SCRIPTING>"else" {
	return T_ELSE;
}

<ST_IN_SCRIPTING>"while" {
	return T_WHILE;
}

<ST_IN_SCRIPTING>"endwhile" {
	return T_ENDWHILE;
}

<ST_IN_SCRIPTING>"do" {
	return T_DO;
}

<ST_IN_SCRIPTING>"for" {
	return T_FOR;
}

<ST_IN_SCRIPTING>"endfor" {
	return T_ENDFOR;
}

<ST_IN_SCRIPTING>"foreach" {
	return T_FOREACH;
}

<ST_IN_SCRIPTING>"endforeach" {
	return T_ENDFOREACH;
}

<ST_IN_SCRIPTING>"declare" {
	return T_DECLARE;
}

<ST_IN_SCRIPTING>"enddeclare" {
	return T_ENDDECLARE;
}

<ST_IN_SCRIPTING>"as" {
	return T_AS;
}

<ST_IN_SCRIPTING>"switch" {
	return T_SWITCH;
}

<ST_IN_SCRIPTING>"endswitch" {
	return T_ENDSWITCH;
}

<ST_IN_SCRIPTING>"case" {
	return T_CASE;
}

<ST_IN_SCRIPTING>"default" {
	return T_DEFAULT;
}

<ST_IN_SCRIPTING>"break" {
	return T_BREAK;
}

<ST_IN_SCRIPTING>"continue" {
	return T_CONTINUE;
}

<ST_IN_SCRIPTING>"echo" {
	return T_ECHO;
}

<ST_IN_SCRIPTING>"print" {
	return T_PRINT;
}

<ST_IN_SCRIPTING>"class" {
	return T_CLASS;
}

<ST_IN_SCRIPTING>"extends" {
	return T_EXTENDS;
}

<ST_IN_SCRIPTING,ST_DOUBLE_QUOTES,ST_BACKQUOTE,ST_HEREDOC>"->" {
	yy_push_state(ST_LOOKING_FOR_PROPERTY TSRMLS_CC);
	return T_OBJECT_OPERATOR;
}

<ST_LOOKING_FOR_PROPERTY>{LABEL} {
	yy_pop_state(TSRMLS_C);
 	zend_copy_value(zendlval, yytext, yyleng);
	zendlval->value.str.len = yyleng;
	zendlval->type = IS_STRING;
	return T_STRING;
}

<ST_LOOKING_FOR_PROPERTY>{ANY_CHAR} {
	yyless(0);
	yy_pop_state(TSRMLS_C);
}

<ST_IN_SCRIPTING>"::" {
	return T_PAAMAYIM_NEKUDOTAYIM;
}

<ST_IN_SCRIPTING>"new" {
	return T_NEW;
}

<ST_IN_SCRIPTING>"var" {
	return T_VAR;
}

<ST_IN_SCRIPTING>"("{TABS_AND_SPACES}("int"|"integer"){TABS_AND_SPACES}")" {
	return T_INT_CAST;
}

<ST_IN_SCRIPTING>"("{TABS_AND_SPACES}("real"|"double"|"float"){TABS_AND_SPACES}")" {
	return T_DOUBLE_CAST;
}

<ST_IN_SCRIPTING>"("{TABS_AND_SPACES}"string"{TABS_AND_SPACES}")" {
	return T_STRING_CAST;
}

<ST_IN_SCRIPTING>"("{TABS_AND_SPACES}"array"{TABS_AND_SPACES}")" {
	return T_ARRAY_CAST;
}

<ST_IN_SCRIPTING>"("{TABS_AND_SPACES}"object"{TABS_AND_SPACES}")" {
	return T_OBJECT_CAST;
}

<ST_IN_SCRIPTING>"("{TABS_AND_SPACES}("bool"|"boolean"){TABS_AND_SPACES}")" {
	return T_BOOL_CAST;
}

<ST_IN_SCRIPTING>"("{TABS_AND_SPACES}("unset"){TABS_AND_SPACES}")" {
	return T_UNSET_CAST;
}

<ST_IN_SCRIPTING>"eval" {
	return T_EVAL;
}

<ST_IN_SCRIPTING>"include" {
	return T_INCLUDE;
}

<ST_IN_SCRIPTING>"include_once" {
	return T_INCLUDE_ONCE;
}

<ST_IN_SCRIPTING>"require" {
	return T_REQUIRE;
}

<ST_IN_SCRIPTING>"require_once" {
	return T_REQUIRE_ONCE;
}

<ST_IN_SCRIPTING>"use" {
	return T_USE;
}

<ST_IN_SCRIPTING>"global" {
	return T_GLOBAL;
}

<ST_IN_SCRIPTING>"isset" {
	return T_ISSET;
}

<ST_IN_SCRIPTING>"empty" {
	return T_EMPTY;
}

<ST_IN_SCRIPTING>"static" {
	return T_STATIC;
}

<ST_IN_SCRIPTING>"unset" {
	return T_UNSET;
}

<ST_IN_SCRIPTING>"=>" {
	return T_DOUBLE_ARROW;
}

<ST_IN_SCRIPTING>"list" {
	return T_LIST;
}

<ST_IN_SCRIPTING>"array" {
	return T_ARRAY;
}

<ST_IN_SCRIPTING>"++" {
	return T_INC;
}

<ST_IN_SCRIPTING>"--" {
	return T_DEC;
}

<ST_IN_SCRIPTING>"===" {
	return T_IS_IDENTICAL;
}

<ST_IN_SCRIPTING>"!==" {
	return T_IS_NOT_IDENTICAL;
}

<ST_IN_SCRIPTING>"==" {
	return T_IS_EQUAL;
}

<ST_IN_SCRIPTING>"!="|"<>" {
	return T_IS_NOT_EQUAL;
}

<ST_IN_SCRIPTING>"<=" {
	return T_IS_SMALLER_OR_EQUAL;
}

<ST_IN_SCRIPTING>">=" {
	return T_IS_GREATER_OR_EQUAL;
}

<ST_IN_SCRIPTING>"+=" {
	return T_PLUS_EQUAL;
}

<ST_IN_SCRIPTING>"-=" {
	return T_MINUS_EQUAL;
}

<ST_IN_SCRIPTING>"*=" {
	return T_MUL_EQUAL;
}

<ST_IN_SCRIPTING>"/=" {
	return T_DIV_EQUAL;
}

<ST_IN_SCRIPTING>".=" {
	return T_CONCAT_EQUAL;
}

<ST_IN_SCRIPTING>"%=" {
	return T_MOD_EQUAL;
}

<ST_IN_SCRIPTING>"<<=" {
	return T_SL_EQUAL;
}

<ST_IN_SCRIPTING>">>=" {
	return T_SR_EQUAL;
}

<ST_IN_SCRIPTING>"&=" {
	return T_AND_EQUAL;
}

<ST_IN_SCRIPTING>"|=" {
	return T_OR_EQUAL;
}

<ST_IN_SCRIPTING>"^=" {
	return T_XOR_EQUAL;
}

<ST_IN_SCRIPTING>"||" {
	return T_BOOLEAN_OR;
}

<ST_IN_SCRIPTING>"&&" {
	return T_BOOLEAN_AND;
}

<ST_IN_SCRIPTING>"OR" {
	return T_LOGICAL_OR;
}

<ST_IN_SCRIPTING>"AND" {
	return T_LOGICAL_AND;
}

<ST_IN_SCRIPTING>"XOR" {
	return T_LOGICAL_XOR;
}

<ST_IN_SCRIPTING>"<<" {
	return T_SL;
}

<ST_IN_SCRIPTING>">>" {
	return T_SR;
}

<ST_IN_SCRIPTING>{TOKENS} {
	return yytext[0];
}


<ST_IN_SCRIPTING>"{" {
	yy_push_state(ST_IN_SCRIPTING TSRMLS_CC);
	return '{';
}


<ST_DOUBLE_QUOTES,ST_BACKQUOTE,ST_HEREDOC>"${" {
	yy_push_state(ST_LOOKING_FOR_VARNAME TSRMLS_CC);
	return T_DOLLAR_OPEN_CURLY_BRACES;
}


<ST_IN_SCRIPTING>"}" {
	/* This is a temporary fix which is dependant on flex and it's implementation */
	if (yy_start_stack_ptr) {
		yy_pop_state(TSRMLS_C);
	}
	return '}';
}


<ST_LOOKING_FOR_VARNAME>{LABEL} {
 	zend_copy_value(zendlval, yytext, yyleng);
	zendlval->type = IS_STRING;
	yy_pop_state(TSRMLS_C);
	yy_push_state(ST_IN_SCRIPTING TSRMLS_CC);
	return T_STRING_VARNAME;
}


<ST_LOOKING_FOR_VARNAME>{ANY_CHAR} {
	yyless(0);
	yy_pop_state(TSRMLS_C);
	yy_push_state(ST_IN_SCRIPTING TSRMLS_CC);
}


<ST_IN_SCRIPTING>{LNUM} {
	errno = 0;
	zendlval->value.lval = strtol(yytext, NULL, 0);
	if (errno == ERANGE) { /* overflow */
		zendlval->value.dval = strtod(yytext, NULL);
		zendlval->type = IS_DOUBLE;
		return T_DNUMBER;
	} else {
		zendlval->type = IS_LONG;
		return T_LNUMBER;
	}
}

<ST_IN_SCRIPTING>{HNUM} {
	errno = 0;
	zendlval->value.lval = strtoul(yytext, NULL, 16);
	if (errno == ERANGE) { /* overflow */
		/* not trying strtod - it returns trash on 0x-es */
		zendlval->value.lval = LONG_MAX; /* maximal long */
		zend_error(E_NOTICE,"Hex number is too big: %s", yytext);
	} else {
		if (zendlval->value.lval < 0) {
			/* maintain consistency with the old way */
			zendlval->value.dval = (unsigned long) zendlval->value.lval;
			zendlval->type = IS_DOUBLE;
			return T_DNUMBER;
		}
		zendlval->type = IS_LONG;
	}
	zendlval->type = IS_LONG;
	return T_LNUMBER;
}

<ST_DOUBLE_QUOTES,ST_BACKQUOTE,ST_HEREDOC>{LNUM}|{HNUM} { /* treat numbers (almost) as strings inside encapsulated strings */
	zendlval->value.str.val = (char *)estrndup(yytext, yyleng);
	zendlval->value.str.len = yyleng;
	zendlval->type = IS_STRING;
	return T_NUM_STRING;
}

<ST_IN_SCRIPTING>{DNUM}|{EXPONENT_DNUM} {
	zendlval->value.dval = strtod(yytext, NULL);
	zendlval->type = IS_DOUBLE;
	return T_DNUMBER;
}

<ST_IN_SCRIPTING>"__CLASS__" {
	char *class_name = NULL;

	if (CG(active_class_entry)) {
		class_name = CG(active_class_entry)->name;
	}
	
	if (!class_name) {
		class_name = "";
	}
	zendlval->value.str.len = strlen(class_name);
	zendlval->value.str.val = estrndup(class_name, zendlval->value.str.len);
	zendlval->type = IS_STRING;
	return T_CLASS_C;
}

<ST_IN_SCRIPTING>"__FUNCTION__" {
	char *func_name = NULL;
	
	if (CG(active_op_array)) {
		func_name = CG(active_op_array)->function_name;
	}

	if (!func_name) {
		func_name = "";
	}
	zendlval->value.str.len = strlen(func_name);
	zendlval->value.str.val = estrndup(func_name, zendlval->value.str.len);
	zendlval->type = IS_STRING;
	return T_FUNC_C;
}

<ST_IN_SCRIPTING>"__LINE__" {
	zendlval->value.lval = CG(zend_lineno);
	zendlval->type = IS_LONG;
	return T_LINE;
}

<ST_IN_SCRIPTING>"__FILE__" {
	char *filename = zend_get_compiled_filename(TSRMLS_C);
	
	if (!filename) {
		filename = "";
	}
	zendlval->value.str.len = strlen(filename);
	zendlval->value.str.val = estrndup(filename, zendlval->value.str.len);
	zendlval->type = IS_STRING;
	return T_FILE;
}


<INITIAL>(([^<]|"<"[^?%s<]){1,400})|"<s"|"<" {
#ifdef ZEND_MULTIBYTE
	if (SCNG(output_filter)) {
		int readsize;
		readsize = SCNG(output_filter)(&(zendlval->value.str.val), &(zendlval->value.str.len), yytext, yyleng TSRMLS_CC);
		if (readsize < yyleng) {
			yyless(readsize);
		}
	} else {
		zendlval->value.str.val = (char *) estrndup(yytext, yyleng);
		zendlval->value.str.len = yyleng;
	}
#else /* ZEND_MULTIBYTE */
	zendlval->value.str.val = (char *) estrndup(yytext, yyleng);
	zendlval->value.str.len = yyleng;
#endif /* ZEND_MULTIBYTE */
	zendlval->type = IS_STRING;
	HANDLE_NEWLINES(yytext, yyleng);
	return T_INLINE_HTML;
}

<INITIAL>"<?"|"<script"{WHITESPACE}+"language"{WHITESPACE}*"="{WHITESPACE}*("php"|"\"php\""|"\'php\'"){WHITESPACE}*">" {
	HANDLE_NEWLINES(yytext, yyleng);
	if (CG(short_tags) || yyleng>2) { /* yyleng>2 means it's not <? but <script> */
		zendlval->value.str.val = yytext; /* no copying - intentional */
		zendlval->value.str.len = yyleng;
		zendlval->type = IS_STRING;
		BEGIN(ST_IN_SCRIPTING);
		return T_OPEN_TAG;
	} else {
		zendlval->value.str.val = (char *) estrndup(yytext, yyleng);
		zendlval->value.str.len = yyleng;
		zendlval->type = IS_STRING;
		return T_INLINE_HTML;
	}
}


<INITIAL>"<%="|"<?=" {
	if ((yytext[1]=='%' && CG(asp_tags)) || (yytext[1]=='?' && CG(short_tags))) {
		zendlval->value.str.val = yytext; /* no copying - intentional */
		zendlval->value.str.len = yyleng;
		zendlval->type = IS_STRING;
		BEGIN(ST_IN_SCRIPTING);
		return T_OPEN_TAG_WITH_ECHO;
	} else {
		zendlval->value.str.val = (char *) estrndup(yytext, yyleng);
		zendlval->value.str.len = yyleng;
		zendlval->type = IS_STRING;
		return T_INLINE_HTML;
	}
}


<INITIAL>"<%" {
	if (CG(asp_tags)) {
		zendlval->value.str.val = yytext; /* no copying - intentional */
		zendlval->value.str.len = yyleng;
		zendlval->type = IS_STRING;
		BEGIN(ST_IN_SCRIPTING);
		return T_OPEN_TAG;
	} else {
		zendlval->value.str.val = (char *) estrndup(yytext, yyleng);
		zendlval->value.str.len = yyleng;
		zendlval->type = IS_STRING;
		return T_INLINE_HTML;
	}
}


<INITIAL>"<?php"([ \t]|{NEWLINE}) {
	zendlval->value.str.val = yytext; /* no copying - intentional */
	zendlval->value.str.len = yyleng;
	zendlval->type = IS_STRING;
	HANDLE_NEWLINE(yytext[yyleng-1]);
	BEGIN(ST_IN_SCRIPTING);
	return T_OPEN_TAG;
}


<INITIAL>"<?php_track_vars?>"{NEWLINE}? {
	zend_error(E_WARNING, "&lt;?php_track_vars?&gt; is no longer supported - please use the track_vars INI directive instead");
	HANDLE_NEWLINE(yytext[yyleng-1]);
	zendlval->value.str.val = (char *) estrndup(yytext, yyleng);
	zendlval->value.str.len = yyleng;
	zendlval->type = IS_STRING;
	return T_INLINE_HTML;
}

<ST_IN_SCRIPTING,ST_DOUBLE_QUOTES,ST_HEREDOC,ST_BACKQUOTE>"$"{LABEL} {
 	zend_copy_value(zendlval, (yytext+1), (yyleng-1));
	zendlval->type = IS_STRING;
	return T_VARIABLE;
}


<ST_IN_SCRIPTING>{LABEL} {
 	zend_copy_value(zendlval, yytext, yyleng);
	zendlval->type = IS_STRING;
	return T_STRING;
}


<ST_DOUBLE_QUOTES,ST_BACKQUOTE,ST_HEREDOC>{LABEL} {
 	zend_copy_value(zendlval, yytext, yyleng);
	zendlval->type = IS_STRING;
	return T_STRING;
}


<ST_IN_SCRIPTING>{WHITESPACE} {
	zendlval->value.str.val = yytext; /* no copying - intentional */
	zendlval->value.str.len = yyleng;
	zendlval->type = IS_STRING;
	HANDLE_NEWLINES(yytext, yyleng);
	return T_WHITESPACE;
}


<ST_IN_SCRIPTING>"#"|"//" {
	BEGIN(ST_ONE_LINE_COMMENT);
	yymore();
}

<ST_ONE_LINE_COMMENT>"?"|"%"|">" {
	yymore();
}

<ST_ONE_LINE_COMMENT>[^\n\r?%>]+ {
	yymore();
}

<ST_ONE_LINE_COMMENT>{NEWLINE} {
	zendlval->value.str.val = yytext; /* no copying - intentional */
	zendlval->value.str.len = yyleng;
	zendlval->type = IS_STRING;
	BEGIN(ST_IN_SCRIPTING);
	CG(zend_lineno)++;
	return T_COMMENT;
}

<ST_ONE_LINE_COMMENT>"?>"|"%>" {
    if (CG(asp_tags) || yytext[yyleng-2] != '%') { /* asp comment? */
		zendlval->value.str.val = yytext; /* no copying - intentional */
		zendlval->value.str.len = yyleng;
		zendlval->type = IS_STRING;
		yyless(yyleng-2);
		BEGIN(ST_IN_SCRIPTING);
		return T_COMMENT;
	} else {
		yymore();
	}
}

<ST_IN_SCRIPTING>"/*" {
	CG(comment_start_line) = CG(zend_lineno);
	BEGIN(ST_COMMENT);
	yymore();
}


<ST_COMMENT>[^*]+ {
	yymore();
}

<ST_COMMENT>"*/" {
	HANDLE_NEWLINES(yytext, yyleng);
	BEGIN(ST_IN_SCRIPTING);
	return T_COMMENT;
}

<ST_COMMENT>"*" {
	yymore();
}

<ST_IN_SCRIPTING>("?>"|"</script"{WHITESPACE}*">"){NEWLINE}? {
	zendlval->value.str.val = yytext; /* no copying - intentional */
	zendlval->value.str.len = yyleng;
	zendlval->type = IS_STRING;
	BEGIN(INITIAL);
	return T_CLOSE_TAG;  /* implicit ';' at php-end tag */
}


<ST_IN_SCRIPTING>"%>"{NEWLINE}? {
	if (CG(asp_tags)) {
		BEGIN(INITIAL);
		zendlval->value.str.len = yyleng;
		zendlval->type = IS_STRING;
		zendlval->value.str.val = yytext; /* no copying - intentional */
		return T_CLOSE_TAG;  /* implicit ';' at php-end tag */
	} else {
		yyless(1);
		return yytext[0];
	}
}


<ST_IN_SCRIPTING>(["]([^$"\\]|("\\".))*["]) {
	register char *s, *t;
	char *end;
	
	zendlval->value.str.val = estrndup(yytext+1, yyleng-2);
	zendlval->value.str.len = yyleng-2;
	zendlval->type = IS_STRING;
	HANDLE_NEWLINES(yytext, yyleng);
	
	/* convert escape sequences */
	s = t = zendlval->value.str.val;
	end = s+zendlval->value.str.len;
	while (s<end) {
		if (*s=='\\') {
			s++;
			if (s>=end) {
				continue;
			}
			switch(*s) {
				case 'n':
					*t++ = '\n';
					zendlval->value.str.len--;
					break;
				case 'r':
					*t++ = '\r';
					zendlval->value.str.len--;
					break;
				case 't':
					*t++ = '\t';
					zendlval->value.str.len--;
					break;
				case '\\':
				case '$':
				case '"':
					*t++ = *s;
					zendlval->value.str.len--;
					break;
				default:
					/* check for an octal */
					if (ZEND_IS_OCT(*s)) {
						char octal_buf[4] = { 0, 0, 0, 0 };

						octal_buf[0] = *s;
						zendlval->value.str.len--;
						if ((s+1)<end && ZEND_IS_OCT(*(s+1))) {
							octal_buf[1] = *(++s);
							zendlval->value.str.len--;
							if ((s+1)<end && ZEND_IS_OCT(*(s+1))) {
								octal_buf[2] = *(++s);
								zendlval->value.str.len--;
							}
						}
						*t++ = (char) strtol(octal_buf, NULL, 8);
					} else if (*s=='x' && (s+1)<end && ZEND_IS_HEX(*(s+1))) {
						char hex_buf[3] = { 0, 0, 0};

						zendlval->value.str.len--; /* for the 'x' */

						hex_buf[0] = *(++s);
						zendlval->value.str.len--;
						if ((s+1)<end && ZEND_IS_HEX(*(s+1))) {
							hex_buf[1] = *(++s);
							zendlval->value.str.len--;
						}
						*t++ = (char) strtol(hex_buf, NULL, 16);
					} else {
						*t++ = '\\';
						*t++ = *s;
					}
					break;
			}
			s++;
		} else {
			*t++ = *s++;
		}
	}
	*t = 0;

#ifdef ZEND_MULTIBYTE
	if (SCNG(output_filter)) {
		s = zendlval->value.str.val;
		SCNG(output_filter)(&(zendlval->value.str.val), &(zendlval->value.str.len), s, zendlval->value.str.len TSRMLS_CC);
		efree(s);
	}
#endif /* ZEND_MULTIBYTE */

	return T_CONSTANT_ENCAPSED_STRING;
}


<ST_IN_SCRIPTING>([']([^'\\]|("\\".))*[']) {
	register char *s, *t;
	char *end;
	
	zendlval->value.str.val = estrndup(yytext+1, yyleng-2);
	zendlval->value.str.len = yyleng-2;
	zendlval->type = IS_STRING;
	HANDLE_NEWLINES(yytext, yyleng);
	
	/* convert escape sequences */
	s = t = zendlval->value.str.val;
	end = s+zendlval->value.str.len;
	while (s<end) {
		if (*s=='\\') {
			s++;
			if (s>=end) {
				continue;
			}
			switch(*s) {
				case '\\':
				case '\'':
					*t++ = *s;
					zendlval->value.str.len--;
					break;
				default:
					*t++ = '\\';
					*t++ = *s;
					break;
			}
			s++;
		} else {
			*t++ = *s++;
		}
	}
	*t = 0;

#ifdef ZEND_MULTIBYTE
	if (SCNG(output_filter)) {
		s = zendlval->value.str.val;
		SCNG(output_filter)(&(zendlval->value.str.val), &(zendlval->value.str.len), s, zendlval->value.str.len TSRMLS_CC);
		efree(s);
	}
#endif /* ZEND_MULTIBYTE */

	return T_CONSTANT_ENCAPSED_STRING;
}


<ST_IN_SCRIPTING>["] {
	BEGIN(ST_DOUBLE_QUOTES);
	return '\"';
}


<ST_IN_SCRIPTING>"<<<"{TABS_AND_SPACES}{LABEL}{NEWLINE} {
	char *s;
	CG(zend_lineno)++;
	CG(heredoc_len) = yyleng-3-1-(yytext[yyleng-2]=='\r'?1:0);
	s = yytext+3;
	while ((*s == ' ') || (*s == '\t')) {
		s++;
		CG(heredoc_len)--;
	}
	CG(heredoc) = estrndup(s, CG(heredoc_len));
	BEGIN(ST_HEREDOC);
	return T_START_HEREDOC;
}
	

<ST_IN_SCRIPTING>[`] {
	BEGIN(ST_BACKQUOTE);
	return '`';
}


<ST_IN_SCRIPTING>['] {
	BEGIN(ST_SINGLE_QUOTE);
	return '\'';
}


<ST_HEREDOC>^{LABEL}(";")?{NEWLINE} {
	int label_len;
	unsigned char unput_semicolon;

	CG(zend_lineno)++;
	if (yytext[yyleng-2]=='\r') {
		label_len = yyleng-2;
	} else {
		label_len = yyleng-1;
	}
	
	if (yytext[label_len-1]==';') {
		label_len--;
		unput_semicolon=1;
	} else{
		unput_semicolon=0;
	}

	if (label_len==CG(heredoc_len) && !memcmp(yytext, CG(heredoc), label_len)) {
		zendlval->value.str.val = estrndup(yytext, label_len); /* unput destroys yytext */
		zendlval->value.str.len = label_len;
		if (unput_semicolon) {
			unput(';');
		}
		efree(CG(heredoc));
		CG(heredoc)=NULL;
		CG(heredoc_len)=0;
		BEGIN(ST_IN_SCRIPTING);
		return T_END_HEREDOC;
	} else {
 		zend_copy_value(zendlval, yytext, yyleng);
		zendlval->type = IS_STRING;
		return T_STRING;
	}
}


<ST_DOUBLE_QUOTES,ST_BACKQUOTE,ST_HEREDOC>{ESCAPED_AND_WHITESPACE} {
	HANDLE_NEWLINES(yytext, yyleng);
	zendlval->value.str.val = (char *) estrndup(yytext, yyleng);
	zendlval->value.str.len = yyleng;
	zendlval->type = IS_STRING;
	return T_ENCAPSED_AND_WHITESPACE;
}

<ST_SINGLE_QUOTE>([^'\\]|\\[^'\\])+ {
	HANDLE_NEWLINES(yytext, yyleng);
 	zend_copy_value(zendlval, yytext, yyleng);
	zendlval->type = IS_STRING;
	return T_ENCAPSED_AND_WHITESPACE;
}


<ST_DOUBLE_QUOTES>[`]+ {
 	zend_copy_value(zendlval, yytext, yyleng);
	zendlval->type = IS_STRING;
	return T_ENCAPSED_AND_WHITESPACE;
}


<ST_BACKQUOTE>["]+ {
 	zend_copy_value(zendlval, yytext, yyleng);
	zendlval->type = IS_STRING;
	return T_ENCAPSED_AND_WHITESPACE;
}


<ST_DOUBLE_QUOTES,ST_BACKQUOTE,ST_HEREDOC>"$"[^a-zA-Z_\x7f-\xff{] {
	zendlval->value.lval = (long) yytext[0];
	if (yyleng == 2) {
		yyless(1);
	}
	return T_CHARACTER;	
}


<ST_DOUBLE_QUOTES,ST_BACKQUOTE,ST_HEREDOC>{ENCAPSED_TOKENS} {
	zendlval->value.lval = (long) yytext[0];
	return yytext[0];
}

<ST_DOUBLE_QUOTES,ST_BACKQUOTE,ST_HEREDOC>"{$" {
	zendlval->value.lval = (long) yytext[0];
	yy_push_state(ST_IN_SCRIPTING TSRMLS_CC);
	yyless(1);
	return T_CURLY_OPEN;
}


<ST_SINGLE_QUOTE>"\\'" {
	zendlval->value.lval = (long) '\'';
	return T_CHARACTER;
}

<ST_SINGLE_QUOTE>"\\\\" {
	zendlval->value.lval = (long)'\\';
	return T_CHARACTER;
}

<ST_DOUBLE_QUOTES>"\\\"" {
	zendlval->value.lval = (long) '"';
	return T_CHARACTER;
}

<ST_BACKQUOTE>"\\`" {
	zendlval->value.lval = (long) '`';
	return T_CHARACTER;
}

<ST_DOUBLE_QUOTES,ST_BACKQUOTE,ST_HEREDOC>"\\"[0-7]{1,3} {
	zendlval->value.lval = strtol(yytext+1, NULL, 8);
	return T_CHARACTER;
}

<ST_DOUBLE_QUOTES,ST_BACKQUOTE,ST_HEREDOC>"\\x"[0-9A-Fa-f]{1,2} {
	zendlval->value.lval = strtol (yytext+2, NULL, 16);
	return T_CHARACTER;
}

<ST_DOUBLE_QUOTES,ST_BACKQUOTE,ST_HEREDOC>"\\"{ANY_CHAR} {
	switch (yytext[1]) {
		case 'n':
			zendlval->value.lval = (long) '\n';
			break;
		case 't':
			zendlval->value.lval = (long) '\t';
			break;
		case 'r':
			zendlval->value.lval = (long) '\r';
			break;
		case '\\':
			zendlval->value.lval = (long) '\\';
			break;
		case '$':
			zendlval->value.lval = (long) yytext[1];
			break;
		case '{':
			zendlval->value.lval = (long) yytext[1];
			break;
		default:
			zendlval->value.str.val = estrndup(yytext, yyleng);
			zendlval->value.str.len = yyleng;
			zendlval->type = IS_STRING;
			return T_BAD_CHARACTER;
			break;
	}
	return T_CHARACTER;
}


<ST_HEREDOC>["'`]+ {
	zendlval->value.str.val = (char *) estrndup(yytext, yyleng);
	zendlval->value.str.len = yyleng;
	zendlval->type = IS_STRING;
	return T_ENCAPSED_AND_WHITESPACE;
}


<ST_DOUBLE_QUOTES>["] {
	BEGIN(ST_IN_SCRIPTING);
	return '\"';
}


<ST_BACKQUOTE>[`] {
	BEGIN(ST_IN_SCRIPTING);
	return '`';
}


<ST_SINGLE_QUOTE>['] {
	BEGIN(ST_IN_SCRIPTING);
	return '\'';
}


<ST_DOUBLE_QUOTES,ST_BACKQUOTE,INITIAL,ST_IN_SCRIPTING,ST_LOOKING_FOR_PROPERTY><<EOF>> {
	return 0;
}

<ST_COMMENT><<EOF>> {
	zend_error(E_COMPILE_WARNING,"Unterminated comment starting line %d", CG(comment_start_line));
	return 0;
}
	


<ST_IN_SCRIPTING,INITIAL,ST_DOUBLE_QUOTES,ST_BACKQUOTE,ST_SINGLE_QUOTE,ST_HEREDOC>{ANY_CHAR} {
	zend_error(E_COMPILE_WARNING,"Unexpected character in input:  '%c' (ASCII=%d) state=%d", yytext[0], yytext[0], YYSTATE);
}
