#include "EXTERN.h"
#include "perl.h"
#include "XSUB.h"

#include "dsp.h"
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h> 

static struct dsp_patch_manager mgr;
static int errcode=0;
static int debug_level=0;
static int deferring=0;
static char str[256];

static int dspLoad (void)
{
  if( debug_level ) fprintf(stderr,"Emu10K1::dspLoad: ");
  errcode = dsp_load(&mgr);
  if( debug_level ) fprintf(stderr,errcode<0?"failed\n":"OK\n");
  return errcode>=0;
}

// open_dev borrowed from emu-dspmgr.c
static int open_dev()
{
	if ( ( mgr.mixer_fd = open("/dev/mixer", O_RDWR, 0)) < 0) {
		perror("/dev/mixer");
		return mgr.mixer_fd;
	}
	
	if ((mgr.dsp_fd = open("/dev/dsp", O_RDWR, 0)) < 0) {
		perror("/dev/dsp");
		return mgr.dsp_fd;
	}
	return 0;
}


MODULE = Emu10K1		PACKAGE = Emu10K1		

BOOT:
#
# init mgr struct and error code
#
mgr.init = 0;
mgr.debug = 0;
if( errcode=open_dev(&mgr) )
  fprintf(stderr,"Emu10K1: open_dev() failed\n");
if( (errcode=dsp_init(&mgr)) < 0 )
  fprintf(stderr,"Emu10K1: dsp_init() failed\n");
errcode = 0;

int
dspLoad ()
#
# does a dsp_load() which makes all deferred changes take effect
#
  CODE:
    deferring = 0;
    RETVAL = dspLoad();
  OUTPUT:
    RETVAL
    
int
dspStop ()
#
# stops the DSP
#
  CODE:
    dsp_stop_proc(&mgr);
    if( debug_level ) fprintf(stderr,"Emu10K1::dspStop\n");
    RETVAL = 1;
  OUTPUT:
    RETVAL
    
int
dspStart ()
#
# starts the DSP
#
  CODE:
    dsp_start_proc(&mgr);
    if( debug_level ) fprintf(stderr,"Emu10K1::dspStart\n");
    RETVAL = 1;
  OUTPUT:
    RETVAL

int
deferGprUpdates ()
#
# defers further GPR updates until a dspLoad() call
#
  CODE:
    if( debug_level ) fprintf(stderr,"Emu10K1::deferGprUpdates\n");
    RETVAL = deferring = 1;
  OUTPUT:
    RETVAL
  
int
debug (level)
#
# sets the debug level
#
  int level
  CODE:
    fprintf(stderr,"Emu10K1: debug level was %d, now %d\n",debug_level,level);
    RETVAL = debug_level = level;
  OUTPUT:
    RETVAL

int
getdebug ()
#
# returns the debug level
#
  CODE:
    RETVAL = debug_level;
  OUTPUT:
    RETVAL

int
errcode ()
#
# returns last error code
#
  CODE:
    RETVAL = errcode;
  OUTPUT:
    RETVAL
  

int 
printPatches()
#
# prints list of patches
#
  CODE:
    dsp_print_patch_list(&mgr);
    RETVAL = 1;
  OUTPUT:
    RETVAL
    

SV *
getGpr(patch,gpr)
#
#     Gets a GPR value. Returns scalar (int) value, or undef on error
#
  char * patch
  char * gpr
  CODE:
    __s32 value;
    ST(0) = sv_newmortal();
    if( debug_level ) fprintf(stderr,"Emu10K1::getGpr(%s,%s): ",patch,gpr);
    errcode = dsp_get_control_gpr_value(&mgr, patch, gpr, &value);
    if( debug_level ) fprintf(stderr,errcode?"bad patch or gpr name\n":"OK\n");
    if( !errcode )
      sv_setiv(ST(0),value);
      
    
int
setGprInt(patch,gpr,value)
#
#     Sets a GPR. 
#
  char * patch
  char * gpr
  int  value
  CODE:
    if( debug_level ) fprintf(stderr,"Emu10K1::setGpr(%s,%s,0x%x): ",patch,gpr,value);
    errcode = dsp_set_control_gpr_value(&mgr, patch, gpr, value);
    if( debug_level ) fprintf(stderr,errcode?"bad patch or gpr name\n":"OK\n");
    RETVAL = !errcode;
    if( !errcode && !deferring )
      RETVAL = dspLoad();
  OUTPUT:
    RETVAL

int 
getInputLine(name)
#
# returns index of input line associated with symbolic name
  char * name
  CODE:
    RETVAL = get_input_name(name);
  OUTPUT:
    RETVAL    

int 
getOutputLine(name)
#
# returns index of input line associated with symbolic name
  char * name
  CODE:
    RETVAL = get_output_name(name);
  OUTPUT:
    RETVAL    

int 
getStereoInputLine(name)
#
# returns index of input line associated with symbolic name
  char * name
  CODE:
    RETVAL = get_stereo_input_name(name);
  OUTPUT:
    RETVAL    

int 
getStereoOutputLine(name)
#
# returns index of input line associated with symbolic name
  char * name
  CODE:
    RETVAL = get_stereo_output_name(name);
  OUTPUT:
    RETVAL    
    

int
attach_patch(file,patchname,io,placement,line,...)
#
# Loads patch from file and attaches to specified lines using the name
# patchname. Placement=0 for head, =1 for tail of patch list.
# Called from Emu10K1::loadPatchAs sub.
#
  char * file
  char * patchname
  int io
  int placement
  int line
  PREINIT:
    int input[DSP_NUM_INPUTS],output[DSP_NUM_OUTPUTS],n=0,i;
    SV * svline;
  CODE:
    n = items-4;
    if( n<1 || n>=DSP_NUM_OUTPUTS )
    {
      fprintf(stderr,"illegal number of lines (%d)\n",file,patchname,n);
      RETVAL = 0;
      return;
    }
    for( i=0; i<n; i++ )
      input[i] = output[i] = SvIV(ST(i+4));
    strncpy(str,patchname,DSP_PATCH_NAME_SIZE);
    str[DSP_PATCH_NAME_SIZE] = 0;
		errcode = dsp_read_patch(&mgr,file,input,output,n,n,io,str,placement);
    if( debug_level || errcode ) fprintf(stderr,errcode?"failed\n":"'%s' OK\n",str);
    RETVAL = !errcode;
    if( !errcode )
      RETVAL = dspLoad();
  OUTPUT:
    RETVAL

int
unloadPatch(name)
#
# unloads a patch
#
  char * name
  CODE:
    if( debug_level ) fprintf(stderr,"Emu10K1::unloadPatch(%s): ",name);
  	errcode = dsp_unload_patch(&mgr,name);
    if( debug_level ) fprintf(stderr,errcode?"failed\n":"OK\n");
    RETVAL = !errcode;
    if( !errcode )
      RETVAL = dspLoad();
  OUTPUT:
    RETVAL
