/* This file is part of the KDE project
 * Copyright (C) 2007-2008 Thomas Zander <zander@kde.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "KWPageRemoveCommand.h"
#include "KWFrameDeleteCommand.h"
#include "frames/KWTextFrameSet.h"
#include "KWDocument.h"
#include "KWPage.h"
#include "frames/KWFrame.h"
#include "KWRootAreaProvider.h"
#include "KoTextLayoutRootArea.h"

#include <KoShapeMoveCommand.h>

#include <klocale.h>
#include <kdebug.h>

KWPageRemoveCommand::KWPageRemoveCommand(KWDocument *document, KWPage page, KUndo2Command *parent)
        : KUndo2Command(i18nc("(qtundo-format)", "Remove Page"), parent),
        m_document(document),
        m_pageSide(page.pageSide()),
        m_pageLayout(page.pageStyle().pageLayout()),
        m_orientation(page.orientationHint()),
        m_pageNumber(page.pageNumber()),
        m_masterPageName(page.pageStyle().name()),
        m_direction(page.directionHint())
{
    Q_ASSERT(page.isValid());
    Q_ASSERT(document);
    Q_ASSERT(page.pageStyle().isValid());
    Q_ASSERT(document->pageManager()->page(m_pageNumber) == page);

    kDebug(32001) << "pageNumber=" << page.pageNumber();
#if 0
    const qreal top = page.offsetInDocument();
    KoInsets padding = document->pageManager()->padding();
    const qreal bottom = top + page.height();
    const qreal height = page.height() + padding.top + padding.bottom; // size from previous page

    QList<KoShape*> shapesToMove;
    QList<QPointF> previousPositions;
    QList<QPointF> newPositions;
    foreach (KWFrameSet *fs, document->frameSets()) {
        KWTextFrameSet *tfs = dynamic_cast<KWTextFrameSet*>(fs);
        bool autoGeneratedFrameSet = tfs
                && (tfs->textFrameSetType() == Words::OddPagesHeaderTextFrameSet ||
                 tfs->textFrameSetType() == Words::EvenPagesHeaderTextFrameSet ||
                 tfs->textFrameSetType() == Words::OddPagesFooterTextFrameSet ||
                 tfs->textFrameSetType() == Words::EvenPagesFooterTextFrameSet ||
                 tfs->textFrameSetType() == Words::MainTextFrameSet);
        int count = -1;
        foreach (KWFrame *frame, fs->frames()) {
            ++count;
            KoShape *shape = frame->shape();
            QPointF pos = shape->position();
            if (pos.y() < top)
                continue;
            if (autoGeneratedFrameSet) {
                // this is a bit special; we can't refer to them using pointers as they are regenerated.
                AutoGenFrameSet afs;
                afs.frameSet = tfs;
                afs.deleteFromFrame = count;
                m_autoGenFrameSets << afs;
                break;
            }
            else if (pos.y() > bottom) { // move up
                newPositions << QPointF(pos.x(), pos.y() - height);
                previousPositions << pos;
                shapesToMove << shape;
            }
            else { // remove shapes that are *on* the page we are about to remove.
                new KWFrameDeleteCommand(m_document, frame, this);
            }
        }
    }
#if 0
    if (! shapesToMove.isEmpty())
        new KoShapeMoveCommand(shapesToMove, previousPositions, newPositions, this);
#endif
#endif
}

KWPageRemoveCommand::~KWPageRemoveCommand()
{
}

void KWPageRemoveCommand::redo()
{
    kDebug(32001) << "pageNumber=" << m_pageNumber;
    KUndo2Command::redo();

    KWPage page = m_document->pageManager()->page(m_pageNumber);
    Q_ASSERT(page.isValid());

    // remove the text displayed on the page from the accros all pages shared QTextDocument
    KWTextFrameSet *mainfs = m_document->frameLayout()->mainFrameSet();
    KWRootAreaProvider *provider = mainfs->rootAreaProvider();
    QList<KWRootAreaPage *> pages = provider->pages();
    Q_ASSERT(m_pageNumber <= pages.count());
    KWRootAreaPage *rootAreaPage = pages[m_pageNumber - 1];
    foreach(KoTextLayoutRootArea *area, rootAreaPage->rootAreas) {
        QTextFrame::iterator it = area->startTextFrameIterator();
        QTextFrame::iterator stop = area->endTextFrameIterator();
        if (it == stop)
            continue;
        --stop;
        QTextCursor cursor(it.currentBlock());
        cursor.setPosition(stop.currentBlock().position() + stop.currentBlock().length(), QTextCursor::KeepAnchor);
        cursor.removeSelectedText();
    }

    //TODO remove also to the page / a character anchored objects like e.g. images

    // remove the KWFrame's displayed on that page
    foreach(KWFrame *frame, m_document->frameLayout()->framesInPage(page.pageNumber())) {
        new KWFrameDeleteCommand(m_document, frame, this);
    }

    // finally remove the page from the KWPageManager
    m_document->pageManager()->removePage(page);

    // and ask for a relayout
    KoTextDocumentLayout *lay = dynamic_cast<KoTextDocumentLayout*>(m_document->frameLayout()->mainFrameSet()->document()->documentLayout());
    Q_ASSERT(lay);

    m_document->frameLayout()->mainFrameSet()->rootAreaProvider()->clearPages(1);
    lay->removeRootArea(0);

    lay->layout();
    
    m_document->firePageSetupChanged();
}

void KWPageRemoveCommand::undo()
{
    kDebug(32001) << "pageNumber=" << m_pageNumber;

    KUndo2Command::undo();
#if 0
    // insert the page
    KWPage page = m_document->pageManager()->insertPage(m_pageNumber);
    page.setPageSide(m_pageSide);
    m_pageLayout.orientation = m_orientation;
    page.pageStyle().setPageLayout(m_pageLayout);
    page.setDirectionHint(m_direction);
    KWPageStyle pageStyle = m_document->pageManager()->pageStyle(m_masterPageName);
    if (pageStyle.isValid())
        page.setPageStyle(pageStyle);

    m_document->firePageSetupChanged();
    //m_document->relayout(); //needed?
#endif
}

