/*
 *	linux/arch/i386/kernel/ioport.c
 *
 * This contains the io-permission bitmap code - written by obz, with changes
 * by Linus.
 */

#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/types.h>
#include <linux/ioport.h>
#include <linux/mm.h>
#include <linux/smp.h>
#include <linux/smp_lock.h>
#include <linux/stddef.h>
#include <linux/grsecurity.h>

/* RSBAC */
#ifdef CONFIG_RSBAC
#include <rsbac/adf.h>
#endif

/* Set EXTENT bits starting at BASE in BITMAP to value TURN_ON. */
static void set_bitmap(unsigned long *bitmap, short base, short extent, int new_value)
{
	int mask;
	unsigned long *bitmap_base = bitmap + (base >> 5);
	unsigned short low_index = base & 0x1f;
	int length = low_index + extent;

	if (low_index != 0) {
		mask = (~0 << low_index);
		if (length < 32)
				mask &= ~(~0 << length);
		if (new_value)
			*bitmap_base++ |= mask;
		else
			*bitmap_base++ &= ~mask;
		length -= 32;
	}

	mask = (new_value ? ~0 : 0);
	while (length >= 32) {
		*bitmap_base++ = mask;
		length -= 32;
	}

	if (length > 0) {
		mask = ~(~0 << length);
		if (new_value)
			*bitmap_base++ |= mask;
		else
			*bitmap_base++ &= ~mask;
	}
}

/*
 * this changes the io permissions bitmap in the current task.
 */
asmlinkage int sys_ioperm(unsigned long from, unsigned long num, int turn_on)
{
	struct thread_struct * t = &current->thread;
	struct tss_struct * tss;

        /* RSBAC */
        #ifdef CONFIG_RSBAC
        union rsbac_target_id_t       rsbac_target_id;
        union rsbac_attribute_value_t rsbac_attribute_value;
        #endif

	if ((from + num <= from) || (from + num > IO_BITMAP_SIZE*32))
		return -EINVAL;
#ifdef CONFIG_GRKERNSEC_IO
	if (grsec_enable_secure_io) {
	if (turn_on) {
		gr_handle_ioperm();
		return -EPERM;
	} else
	if (turn_on && !capable(CAP_SYS_RAWIO))
		return -EPERM;
	}
#else
	if (turn_on && !capable(CAP_SYS_RAWIO))
		return -EPERM;
#endif

        /* RSBAC */
        #ifdef CONFIG_RSBAC
#ifdef CONFIG_RSBAC_DEBUG
        if (rsbac_debug_aef) printk(KERN_DEBUG "sys_ioperm(): calling ADF\n");
#endif
        rsbac_target_id.scd = ST_ioports;
        rsbac_attribute_value.dummy = 0;
        if (!rsbac_adf_request(R_MODIFY_PERMISSIONS_DATA,
                               current->pid,
                               T_SCD,
                               rsbac_target_id,
                               A_none,
                               rsbac_attribute_value))
          {
            return -EPERM;
          }
        #endif

	/*
	 * If it's the first ioperm() call in this thread's lifetime, set the
	 * IO bitmap up. ioperm() is much less timing critical than clone(),
	 * this is why we delay this operation until now:
	 */
	preempt_disable();
	tss = init_tss + smp_processor_id();
	if (!t->ioperm) {
		/*
		 * just in case ...
		 */
		memset(t->io_bitmap,0xff,(IO_BITMAP_SIZE+1)*4);
		memset(tss->io_bitmap,0xff,(IO_BITMAP_SIZE+1)*4);
		t->ioperm = 1;
		/*
		 * this activates it in the TSS
		 */
		tss->bitmap = IO_BITMAP_OFFSET;
	}

	/*
	 * do it in the per-thread copy and in the TSS ...
	 */
	set_bitmap(t->io_bitmap, from, num, !turn_on);
	set_bitmap(tss->io_bitmap, from, num, !turn_on);

	preempt_enable();

	return 0;
}

/*
 * sys_iopl has to be used when you want to access the IO ports
 * beyond the 0x3ff range: to get the full 65536 ports bitmapped
 * you'd need 8kB of bitmaps/process, which is a bit excessive.
 *
 * Here we just change the eflags value on the stack: we allow
 * only the super-user to do it. This depends on the stack-layout
 * on system-call entry - see also fork() and the signal handling
 * code.
 */

asmlinkage int sys_iopl(unsigned long unused)
{
	struct pt_regs * regs = (struct pt_regs *) &unused;
	unsigned int level = regs->ebx;
	unsigned int old = (regs->eflags >> 12) & 3;

        /* RSBAC */
        #ifdef CONFIG_RSBAC
        union rsbac_target_id_t       rsbac_target_id;
        union rsbac_attribute_value_t rsbac_attribute_value;
        #endif

	if (level > 3)
		return -EINVAL;

	/* Trying to gain more privileges? */
#ifdef CONFIG_GRKERNSEC_IO
	if (level > old) {
	    if (grsec_enable_secure_io) {
		gr_handle_iopl();
		return -EPERM;
	    } else
		if (!capable(CAP_SYS_RAWIO))
			return -EPERM;
	}
#else
	if (level > old) {
		if (!capable(CAP_SYS_RAWIO))
			return -EPERM;
	}
#endif

        /* RSBAC */
        #ifdef CONFIG_RSBAC
#ifdef CONFIG_RSBAC_DEBUG
        if (rsbac_debug_aef) printk(KERN_DEBUG "sys_iopl(): calling ADF\n");
#endif
        rsbac_target_id.scd = ST_ioports;
        rsbac_attribute_value.dummy = 0;
        if (!rsbac_adf_request(R_MODIFY_PERMISSIONS_DATA,
                               current->pid,
                               T_SCD,
                               rsbac_target_id,
                               A_none,
                               rsbac_attribute_value))
          {
            return -EPERM;
          }
        #endif

	regs->eflags = (regs->eflags & 0xffffcfff) | (level << 12);
	return 0;
}
