/* suncons.c: Sun SparcStation console support.
 *
 * Copyright (C) 1995 Peter Zaitcev (zaitcev@lab.ipmce.su)
 * Copyright (C) 1995 David S. Miller (davem@caip.rutgers.edu)
 * Copyright (C) 1995 Miguel de Icaza (miguel@nuclecu.unam.mx)
 *
 * Added font loading Nov/21, Miguel de Icaza (miguel@nuclecu.unam.mx)
 * Added render_screen and faster scrolling Nov/27, miguel
 * Added console palette code for cg6 Dec/13/95, miguel
 * Added generic frame buffer support Dec/14/95, miguel
 * Added cgsix and bwtwo drivers Jan/96, miguel
 * Added 4m, and cg3 driver Feb/96, miguel
 * Fixed the cursor on color displays Feb/96, miguel.
 *
 * Cleaned up the detection code, generic 8bit depth display 
 * code, Mar/96 miguel
 * 
 * This file contains the frame buffer device drivers.
 * Each driver is kept together in case we would like to
 * split this file.
 *
 * Much of this driver is derived from the DEC TGA driver by
 * Jay Estabrook who has done a nice job with the console
 * driver abstraction btw.
 *
 * We try to make everything a power of two if possible to
 * speed up the bit blit.  Doing multiplies, divides, and
 * remainder routines end up calling software library routines
 * since not all Sparcs have the hardware to do it.
 *
 * TODO:
 * do not use minor to index into instances of the frame buffer,
 * since the numbers assigned to us are not consecutive.
 *
 * do not blank the screen when frame buffer is mapped.
 *
 * Change the detection loop to use more than one video card.
 */


/* Define this one if you are debugging something in X, it will not disable the console output */
/* #define DEBUGGING_X */
/* See also: sparc/keyboard.c: CODING_NEW_DRIVER */

#define GRAPHDEV_MAJOR 29

#define FRAME_BUFFERS 1

#include <linux/sched.h>
#include <linux/timer.h>
#include <linux/interrupt.h>
#include <linux/tty.h>
#include <linux/tty_flip.h>
#include <linux/kernel.h>
#include <linux/string.h>
#include <linux/errno.h>
#include <linux/kd.h>
#include <linux/malloc.h>
#include <linux/major.h>
#include <linux/mm.h>
#include <linux/types.h>

#include <asm/system.h>
#include <asm/segment.h>
#include <asm/page.h>
#include <asm/pgtable.h>
#include <asm/bitops.h>
#include <asm/oplib.h>
#include <asm/sbus.h>
#include <asm/fbio.h>
#include <asm/io.h>
#include <asm/pgtsun4c.h>	/* for the sun4c_nocache */

#include "kbd_kern.h"
#include "vt_kern.h"
#include "consolemap.h"
#include "selection.h"
#include "console_struct.h"

#define cmapsz 8192

extern void register_console(void (*proc)(const char *));
extern void console_print(const char *);
extern unsigned char vga_font[];
extern int graphics_on;
extern int serial_console;

/* Based upon what the PROM tells us, we can figure out where
 * the console is currently located.  The situation can be either
 * of the following two scenarios:
 *
 * 1) Console i/o is done over the serial line, ttya or ttyb
 * 2) Console output on frame buffer (video card) and input
 *    coming from the keyboard/mouse which each use a zilog8530
 *    serial channel a piece.
 */

/* The following variables describe a Sparc console. */

/* From the PROM */
static char con_name[40];

/* Screen dimensions and color depth. */
static int con_depth, con_width, con_height, con_type;

static int con_linebytes;

/* Base address of first line. */
static unsigned char *con_fb_base;

/* Screen parameters: we compute those at startup to make the code faster */
static int chars_per_line;	/* number of bytes per line */
static int ints_per_line;	/* number of ints per  line */
static int skip_bytes;		/* number of bytes we skip for the y margin */
static int x_margin, y_margin;	/* the x and y margins */
static int bytes_per_row;	/* bytes used by one screen line (of 16 scan lines)  */

/* Functions used by the SPARC dependent console code
 * to perform the restore_palette function.
 */
static void (*restore_palette)(void);
void set_palette (void);


 /* Our screen looks like at 1152 X 900:
 *
 *  0,0
 *      ------------------------------------------------------------------
 *      |                          ^^^^^^^^^^^                           |
 *      |                          18 y-pixels                           |
 *      |                          ^^^^^^^^^^^                           |
 *   13 | <-64 pixels->|  <-- 128 8x16 characters -->    | <-64 pixels-> |
 *    ....
 *                         54 chars from top to bottom
 *    ....
 *  888 | <-64 pixels->|  <-- 128 8x16 characters -->    | <-64 pixels-> |
 *      |                          ^^^^^^^^^^^                           |
 *      |                          18 y-pixels                           |
 *      |                          ^^^^^^^^^^^                           |
 *      ------------------------------------------------------------------
 */
/* First for MONO displays. */
#define SCREEN_WIDTH     1152     /* Screen width in pixels  */
#define SCREEN_HEIGHT    900      /* Screen height in pixels */
#define CHARS_PER_LINE   144      /* Make this empirical for speed */
#define NICE_Y_MARGIN    18       /* We skip 18 y-pixels at top/bottom */
#define NICE_X_MARGIN    8        /* We skip 64 x-pixels at left/right */
#define FBUF_TOP_SKIP    2592     /* Empirical, (CHARS_PER_LINE * NICE_Y_MARGIN) */
#define CHAR_HEIGHT      16
#define ONE_ROW          2304     /* CHARS_PER_LINE * CHAR_HEIGHT */

/* Now we have this, to compute the base frame buffer position
 * for a new character to be rendered. 1 and 8 bit depth.
 */
#define FBUF_OFFSET(cindex) \
        (((FBUF_TOP_SKIP) + (((cindex)>>7) * ONE_ROW)) + \
	 ((NICE_X_MARGIN) + (((cindex)&127))))


#define COLOR_FBUF_OFFSET(cindex) \
        (((skip_bytes) + (((cindex)>>7) * bytes_per_row)) + \
	 ((x_margin) + (((cindex)&127) << 3)))

void
__set_origin(unsigned short offset)
{
	/*
	 * should not be called, but if so, do nothing...
	 */
}

/* For the cursor, we just invert the 8x16 block at the cursor
 * location.  Easy enough...
 *
 * Hide the cursor from view, during blanking, usually...
 */
static int cursor_pos = -1;
void
hide_cursor(void)
{
	unsigned long flags;
	int j;

	save_flags(flags); cli();

	if(cursor_pos == -1) {
		restore_flags (flags);
		return;
	}
	/* We just zero out the area for now.  Certain graphics
	 * cards like the cg6 have a hardware cursor that we could
	 * use, but this is an optimization for some time later.
	 */
	switch (con_depth){
	case 1: {
		unsigned char *dst;
		dst = (unsigned char *)((unsigned long)con_fb_base +
					FBUF_OFFSET(cursor_pos));
		for(j = 0; j < CHAR_HEIGHT; j++, dst += CHARS_PER_LINE)
			*dst = ~(0);
		break;
	}
	case 8: {
		unsigned long *dst;
		const    int ipl = ints_per_line;
		
		dst = (unsigned long *)((unsigned long)con_fb_base + COLOR_FBUF_OFFSET(cursor_pos));
		for(j = 0; j < CHAR_HEIGHT; j++, dst += ipl) {
			*dst = ~(0UL);
			*(dst + 1) = ~(0UL);
		}
		break;
	}
	default:
		break;
	}
	restore_flags(flags);
}

/* The idea is the following:
 * we only use the colors in the range 0..15, and we only
 * setup the palette on that range, so we better keep the
 * pixel inversion using those colors, that's why we have
 * those constants below.
 */
inline static void
cursor_reverse (long *dst, int height, const int ints_on_line)
{
    int j;

    for (j = 0; j < height; j++){
	*dst     = ~(*dst)     & 0x0f0f0f0f;
	*(dst+1) = ~(*(dst+1)) & 0x0f0f0f0f;
	dst += ints_on_line;
    }
}

void
set_cursor(int currcons)
{
	int j, idx, oldpos;
	unsigned long flags;

	if (currcons != fg_console || console_blanked || vcmode == KD_GRAPHICS)
		return;

	if (__real_origin != __origin)
		__set_origin(__real_origin);

	save_flags(flags); cli();

	idx = (pos - video_mem_base) >> 1;
	oldpos = cursor_pos;
	cursor_pos = idx;
	if (!deccm) {
		hide_cursor ();
		restore_flags (flags);
		return;
	}
	switch (con_depth){
	case 1: {
		unsigned char *dst, *opos;
		
		dst = (unsigned char *)((unsigned long)con_fb_base + FBUF_OFFSET(idx));
		opos = (unsigned char *)((unsigned long)con_fb_base + FBUF_OFFSET(oldpos));
		if(oldpos != -1) {
			/* Restore what was at the old position */
			for(j=0; j < CHAR_HEIGHT; j++, opos += CHARS_PER_LINE) {
				*opos = ~*opos;
			}
		}
		for(j=0; j < 16; j++, dst+=CHARS_PER_LINE) {
			*dst = ~*dst;
		}
		break;
	}
	case 8: {
		unsigned long *dst, *opos;
		dst = (unsigned long *)((unsigned long)con_fb_base + COLOR_FBUF_OFFSET(idx));
		opos = (unsigned long *)((unsigned long)con_fb_base + COLOR_FBUF_OFFSET(oldpos));
			
		if(oldpos != -1) 
			cursor_reverse(opos, CHAR_HEIGHT, ints_per_line);
		cursor_reverse (dst, CHAR_HEIGHT, ints_per_line);
		break;
	}
	default:
	}
	restore_flags(flags);
}

/*
 * Render the current screen
 * Only used at startup to avoid the caching that is being done in selection.h
 */
static void
render_screen(void)
{
    int count;
    unsigned short *contents;

    count = video_num_columns * video_num_lines;
    contents = (unsigned short *) video_mem_base;
    
    for (;count--; contents++)
	sun_blitc (*contents, (unsigned long) contents);
}

unsigned long
con_type_init(unsigned long kmem_start, const char **display_desc)
{
        can_do_color = (con_type != FBTYPE_SUN2BW);

        video_type = VIDEO_TYPE_SUN;
        *display_desc = "SUN";

	if (!serial_console) {
		/* If we fall back to PROM than our output have to remain readable. */
		prom_putchar('\033');  prom_putchar('[');  prom_putchar('H');

		/*
		 * fake the screen memory with some CPU memory
		 */
		video_mem_base = kmem_start;
		kmem_start += video_screen_size;
		video_mem_term = kmem_start;

		render_screen();
	}
	return kmem_start;
}

/*
 * NOTE: get_scrmem() and set_scrmem() are here only because
 * the VGA version of set_scrmem() has some direct VGA references.
 */
void
get_scrmem(int currcons)
{
	memcpyw((unsigned short *)vc_scrbuf[currcons],
		(unsigned short *)origin, video_screen_size);
	origin = video_mem_start = (unsigned long)vc_scrbuf[currcons];
	scr_end = video_mem_end = video_mem_start + video_screen_size;
	pos = origin + y*video_size_row + (x<<1);
}

void
set_scrmem(int currcons, long offset)
{
	if (video_mem_term - video_mem_base < offset + video_screen_size)
		offset = 0;
	memcpyw((unsigned short *)(video_mem_base + offset),
		(unsigned short *) origin, video_screen_size);
	video_mem_start = video_mem_base;
	video_mem_end = video_mem_term;
	origin = video_mem_base + offset;
	scr_end = origin + video_screen_size;
	pos = origin + y*video_size_row + (x<<1);
}

/*
 * PIO_FONT support.
 */
int
set_get_font(char * arg, int set, int ch512)
{
	int error, i, line;

	if (!arg)
		return -EINVAL;
	error = verify_area (set ? VERIFY_READ : VERIFY_WRITE, (void *) arg,
			     ch512 ? 2* cmapsz : cmapsz);
	if (error)
		return error;

	/* download the current font */
	if (!set){
		memset (arg, 0, cmapsz);
		for (i = 0; i < 256; i++)
		    for (line = 0; line < CHAR_HEIGHT; line++)
			put_user (vga_font [i], arg+(i*32+line));
		return 0;
	}
	
        /* set the font */
	for (i = 0; i < 256; i++)
		for (line = 0; line < CHAR_HEIGHT; line++){
			vga_font [i*CHAR_HEIGHT + line] = (get_user (arg + (i * 32 + line)));
			if (con_depth == 1)
				vga_font [i*CHAR_HEIGHT + line] = vga_font [i*CHAR_HEIGHT + line];
		}
	return 0;
}

/*
 * Adjust the screen to fit a font of a certain height
 *
 * Returns < 0 for error, 0 if nothing changed, and the number
 * of lines on the adjusted console if changed.
 *
 * for now, we only support the built-in font...
 */
int
con_adjust_height(unsigned long fontheight)
{
	return -EINVAL;
}

int
set_get_cmap(unsigned char * arg, int set)
{
	int i;

	i = verify_area(set ? VERIFY_READ : VERIFY_WRITE, (void *)arg, 16*3);
	if (i)
		return i;

	for (i=0; i<16; i++) {
		if (set) {
			default_red[i] = get_user(arg++) ;
			default_grn[i] = get_user(arg++) ;
			default_blu[i] = get_user(arg++) ;
		} else {
			put_user (default_red[i], arg++) ;
			put_user (default_grn[i], arg++) ;
			put_user (default_blu[i], arg++) ;
		}
	}
	if (set) {
		for (i=0; i<MAX_NR_CONSOLES; i++)
			if (vc_cons_allocated(i)) {
				int j, k ;
				for (j=k=0; j<16; j++) {
					vc_cons[i].d->vc_palette[k++] = default_red[j];
					vc_cons[i].d->vc_palette[k++] = default_grn[j];
					vc_cons[i].d->vc_palette[k++] = default_blu[j];
				}
			}
		set_palette();
	}

	return 0;
}


void
sun_clear_screen(void)
{
	memset (con_fb_base, (con_depth == 1 ? ~(0) : (0)),
		(con_depth * con_height * con_width) / 8);
	/* also clear out the "shadow" screen memory */
	memset((char *)video_mem_base, 0, (video_mem_term - video_mem_base));
}

/*
 * dummy routines for the VESA blanking code, which is VGA only,
 * so we don't have to carry that stuff around for the Sparc...
 */
void vesa_blank(void)
{
}
void vesa_unblank(void)
{
}
void set_vesa_blanking(const unsigned long arg)
{
}

void vesa_powerdown(void)
{
}

#undef color
/* cg6 cursor status, kernel tracked copy */
struct cg6_cursor {
        short   enable;	        /* cursor is enabled */
        struct  fbcurpos cpos;	/* position */
        struct  fbcurpos chot;	/* hot-spot */
        struct  fbcurpos size;	/* size of mask & image fields */
        int     bits[2][32];	/* space for mask & image bits */
	char    color [6];	/* cursor colors */
};

struct cg6_info {
	struct bt_regs *bt;	/* color control */
	void *fbc;
	struct cg6_fhc *fhc;
	struct cg6_tec *tec;
	struct cg6_thc *thc;
	struct cg6_cursor cursor; /* cursor control */
	void *dhc;
};

struct bwtwo_info {
        struct bwtwo_regs *regs;
};

struct cg3_info {
	struct bt_regs *bt;	/* brooktree (color) registers */
};

/* Array holding the information for the frame buffers */
typedef struct {
	union {
		struct bwtwo_info bwtwo;
		struct cg3_info   cg3;
		struct cg6_info   cg6;
	} info;		        /* per frame information */
	int    space;           /* I/O space this card resides in */
	int    blanked;		/* true if video blanked */
	int    open;		/* is this fb open? */
	int    mmaped;		/* has this fb been mmapped? */
	int    vtconsole;	/* virtual console where it is opened */
	long   base;		/* frame buffer base    */
	struct fbtype type;	/* frame buffer type    */
	int    (*mmap)(struct inode *, struct file *, struct vm_area_struct *, long fb_base, void *);
	void   (*loadcmap)(void *this, int index, int count);
	void   (*blank)(void *this);
	void   (*unblank)(void *this);
	int    (*ioctl)(struct inode *, struct file *, unsigned int, unsigned long, void *);
} fbinfo_t;

static fbinfo_t fbinfo [FRAME_BUFFERS];

/* We need to keep a copy of the color map to answer ioctl requests */
static union {
	unsigned char   map[256][3];    /* reasonable way to access */
        unsigned int    raw[256*3/4];   /* hardware wants it like this */
} color_map;

#define FB_MMAP_VM_FLAGS (VM_SHM| VM_LOCKED)

static int
fb_open (struct inode * inode, struct file * file)
{
	int minor = MINOR (inode->i_rdev);

	if (minor >= FRAME_BUFFERS)
		return -EBADF;
	if (fbinfo [minor].open)
		return -EBUSY;
	fbinfo [minor].open = 1;
	fbinfo [minor].mmaped = 0;
	return 0;
}

static int
fb_ioctl (struct inode *inode, struct file *file, unsigned int cmd, unsigned long arg)
{
	int minor = MINOR (inode->i_rdev);
	fbinfo_t *fb;
	struct fbcmap *cmap;
	int i;
	
	if (minor >= FRAME_BUFFERS)
		return -EBADF;
	fb = &fbinfo [minor];
	
	switch (cmd){
	case FBIOGTYPE:		/* return frame buffer type */
		i = verify_area (VERIFY_WRITE, (void *) arg, sizeof (struct fbtype));
		if (i) return i;
		*(struct fbtype *)arg = (fb->type);
		break;
	case FBIOGATTR:{
		struct fbgattr *fba = (struct fbgattr *) arg;
		
		i = verify_area (VERIFY_WRITE, (void *) arg, sizeof (struct fbgattr));
		if (i) return i;
		fba->real_type = fb->type.fb_type;
		fba->owner = 0;
		fba->fbtype = fb->type;
		fba->sattr.flags = 0;
		fba->sattr.emu_type = fb->type.fb_type;
		fba->sattr.dev_specific [0] = -1;
		fba->emu_types [0] = fb->type.fb_type;
		fba->emu_types [1] = -1;
		break;
	}
	case FBIOSVIDEO:
		i = verify_area(VERIFY_READ, (void *)arg, sizeof(int));
		if (i) return i;
		
		if (*(int *)arg){
			if (!fb->blanked || !fb->unblank)
				break;
			(*fb->unblank)(fb);
			fb->blanked = 0;
		} else {
			if (fb->blanked || !fb->blank)
				break;
			(*fb->blank)(fb);
			fb->blanked = 1;
		}
		break;
	case FBIOGVIDEO:
		i = verify_area (VERIFY_WRITE, (void *) arg, sizeof (int));
		if (i) return i;
		*(int *) arg = fb->blanked;
		break;
	case FBIOPUTCMAP: {	/* load color map entries */
		char *rp, *gp, *bp;
		int end, count;;
		
		if (!fb->loadcmap)
			return -EINVAL;
		i = verify_area (VERIFY_READ, (void *) arg, sizeof (struct fbcmap));
		if (i) return i;
		cmap = (struct fbcmap *) arg;
		count = cmap->count;
		if ((cmap->index < 0) || (cmap->index > 255))
			return -EINVAL;
		if (cmap->index + count > 256)
			count = 256 - cmap->index;
		i = verify_area (VERIFY_READ, rp = cmap->red, cmap->count);
		if (i) return i;
		i = verify_area (VERIFY_READ, gp = cmap->green, cmap->count);
		if (i) return i;
		i = verify_area (VERIFY_READ, bp = cmap->blue, cmap->count);
		if (i) return i;

		end = cmap->index + count;
		for (i = cmap->index; i < end; i++){
			color_map.map [i][0] = *rp++;
			color_map.map [i][1] = *gp++;
			color_map.map [i][2] = *bp++;
		}
		(*fb->loadcmap)(fb, cmap->index, count);
                break;			
	}

	default:
		if (fb->ioctl){
			i = fb->ioctl (inode, file, cmd, arg, fb);
			if (i == -EINVAL)
				printk ("[[FBIO: %8.8x]]\n", cmd);
			return i;
		}
		printk ("[[FBIO: %8.8x]]\n", cmd);
		return -EINVAL;
	}
	return 0;
}

static void
fb_close (struct inode * inode, struct file *filp)
{
	int minor = MINOR(inode->i_rdev);
	struct fbcursor cursor;
	
	if (minor >= FRAME_BUFFERS)
		return;
	if (fbinfo [minor].open)
		fbinfo [minor].open = 0;
	vt_cons [fbinfo [minor].vtconsole]->vc_mode = KD_TEXT;

	/* Leaving graphics mode, turn off the cursor */
	graphics_on = 0;
	if (fbinfo [minor].mmaped)
		sun_clear_screen ();
	cursor.set    = FB_CUR_SETCUR;
	cursor.enable = 0;
	fb_ioctl (inode, filp, FBIOSCURPOS, (unsigned long) &cursor);
	set_palette ();
	render_screen ();
	return;
}

static int
fb_mmap (struct inode *inode, struct file *file, struct vm_area_struct *vma)
{
	int minor = MINOR (inode->i_rdev);
	fbinfo_t *fb;

	if (minor >= FRAME_BUFFERS)
		return -ENXIO;
	/* FIXME: the fg_console below should actually be the
	 * console on which the invoking process is running
	 */
	if (vt_cons [fg_console]->vc_mode == KD_GRAPHICS)
		return -ENXIO;
	fbinfo [minor].vtconsole = fg_console;
	fb = &fbinfo [minor];

	if (fb->mmap){
		int v;
		
		v = (*fb->mmap)(inode, file, vma, fb->base, fb);
		if (v) return v;
		fbinfo [minor].mmaped = 1;
		vt_cons [fg_console]->vc_mode = KD_GRAPHICS;
		graphics_on = 1;
		return 0;
	} else
		return -ENXIO;
}

static struct file_operations graphdev_fops =
{
	NULL,			/* lseek */
	NULL,			/* read */
	NULL,			/* write */
	NULL,			/* readdir */
	NULL,			/* select */
	fb_ioctl,
	fb_mmap,
	fb_open,		/* open */
	fb_close,		/* close */
};

/* Call the frame buffer routine for setting the palette */
void
set_palette (void)
{
	if (console_blanked || vt_cons [fg_console]->vc_mode == KD_GRAPHICS)
		return;

	if (fbinfo [0].loadcmap){
		int i, j;
	
		/* First keep color_map with the palette colors */
		for (i = 0; i < 16; i++){
			j = color_table [i];
			color_map.map [i][0] = default_red [j];
			color_map.map [i][1] = default_grn [j];
			color_map.map [i][2] = default_blu [j];
		}
		(*fbinfo [0].loadcmap)(&fbinfo [0], 0, 16);
	}
}

/* Called when returning to prom */
void
console_restore_palette (void)
{
        if (restore_palette)
	        (*restore_palette) ();
}

/* This routine should be moved to srmmu.c */
static __inline__ unsigned int
srmmu_get_pte (unsigned long addr)
{
	register unsigned long entry;

	__asm__ __volatile__("\n\tlda [%1] %2,%0\n\t" :
			     "=r" (entry):
			     "r" ((addr & 0xfffff000) | 0x400), "i" (ASI_M_FLUSH_PROBE));
	return entry;
}

unsigned int
get_phys (unsigned int addr)
{
	switch (sparc_cpu_model){
	case sun4c:
		return sun4c_get_pte (addr) << PAGE_SHIFT;
	case sun4m:
		return ((srmmu_get_pte (addr) & 0xffffff00) << 4);
	default:
		panic ("get_phys called for unsupported cpu model\n");
		return 0;
	}
}

/* CG6 support code */

/* Offset of interesting structures in the OBIO space */
/*
 * Brooktree is the video dac and is funny to program on the cg6.
 * (it's even funnier on the cg3)
 * The FBC could be the the frame buffer control
 * The FHC could be the frame buffer hardware control.
 */
#define CG6_ROM_OFFSET       0x0
#define CG6_BROOKTREE_OFFSET 0x200000
#define CG6_DHC_OFFSET       0x240000
#define CG6_ALT_OFFSET       0x280000
#define CG6_FHC_OFFSET       0x300000
#define CG6_THC_OFFSET       0x301000
#define CG6_FBC_OFFSET       0x700000
#define CG6_TEC_OFFSET       0x701000
#define CG6_RAM_OFFSET       0x800000

struct bt_regs {
	unsigned int  addr;	/* address register */
	unsigned int  color_map; /* color map */
	unsigned int  control;	/* control register */
	unsigned int  cursor;	/* cursor map register */
};

/* The contents are unknown */
struct cg6_tec {
	int tec_matrix;
	int tec_clip;
	int tec_vdc;
};

struct cg6_thc {
        unsigned int thc_xxx0[512];  /* ??? */
        unsigned int thc_hsync1;     /* hsync timing */
        unsigned int thc_hsync2;
        unsigned int thc_hsync3;
        unsigned int thc_vsync1;     /* vsync timing */
        unsigned int thc_vsync2;
        unsigned int thc_refresh;
        unsigned int thc_misc;
        unsigned int thc_xxx1[56];
        unsigned int thc_cursxy;             /* cursor x,y position (16 bits each) */
        unsigned int thc_cursmask[32];       /* cursor mask bits */
        unsigned int thc_cursbits[32];       /* what to show where mask enabled */
};

static void
cg6_restore_palette (void)
{
	volatile struct bt_regs *bt;

	bt = fbinfo [0].info.cg6.bt;
	bt->addr = 0;
	bt->color_map = 0xffffffff;
	bt->color_map = 0xffffffff;
	bt->color_map = 0xffffffff;
}

/* Ugh: X wants to mmap a bunch of cute stuff at the same time :-( */
/* So, we just mmap the things that are being asked for */
static int
cg6_mmap (struct inode *inode, struct file *file, struct vm_area_struct *vma, long base, void *xx)
{
	unsigned int size, page, r, map_size;
	unsigned int map_offset = 0;
	fbinfo_t *fb = (fbinfo_t *) xx;
	
	size = vma->vm_end - vma->vm_start;
        if (vma->vm_offset & ~PAGE_MASK)
                return -ENXIO;

	/* To stop the swapper from even considering these pages */
	vma->vm_flags |= FB_MMAP_VM_FLAGS;
	
	/* Each page, see which map applies */
	for (page = 0; page < size; ){
		switch (vma->vm_offset+page){
		case CG6_TEC:
			map_size = PAGE_SIZE;
			map_offset = get_phys ((uint)fb->info.cg6.tec);
			break;
		case CG6_FBC:
			map_size = PAGE_SIZE;
			map_offset = get_phys ((uint)fb->info.cg6.fbc);
			break;
		case CG6_FHC:
			map_size = PAGE_SIZE;
			map_offset = get_phys ((uint)fb->info.cg6.fhc);
			break;
		case CG6_THC:
			map_size = PAGE_SIZE;
			map_offset = get_phys ((uint)fb->info.cg6.thc);
			break;
		case CG6_BTREGS:
			map_size = PAGE_SIZE;
			map_offset = get_phys ((uint)fb->info.cg6.bt);
			break;
			
		case CG6_DHC:
			map_size = PAGE_SIZE * 40;
			map_offset = get_phys ((uint)fb->info.cg6.dhc);
			break;
			
		case CG6_ROM:
			map_size = 0;
			break;

		case CG6_RAM:
			map_size = size-page;
			map_offset = get_phys ((uint) con_fb_base);
			if (map_size < fb->type.fb_size)
				map_size = fb->type.fb_size;
			break;
		default:
			map_size = 0;
			break;
		}
		if (!map_size){
			page += PAGE_SIZE;
			continue;
		}
		r = io_remap_page_range (vma->vm_start+page,
					 map_offset,
					 map_size, vma->vm_page_prot,
					 fb->space);
		if (r) return -EAGAIN;
		page += map_size;
	}
        vma->vm_inode = inode;
        inode->i_count++;
        return 0;
}

#define BT_D4M3(x) ((((x) >> 2) << 1) + ((x) >> 2))     /* (x / 4) * 3 */
#define BT_D4M4(x) ((x) & ~3)                           /* (x / 4) * 4 */

static void
cg6_loadcmap (void *fbinfo, int index, int count)
{
	fbinfo_t *fb = (fbinfo_t *) fbinfo;
	struct bt_regs *bt = fb->info.cg6.bt;
	int i;
	
	bt->addr = index << 24;
	for (i = index; count--; i++){
		bt->color_map = color_map.map [i][0] << 24;
		bt->color_map = color_map.map [i][1] << 24;
		bt->color_map = color_map.map [i][2] << 24;
	}
}

/* Load cursor information */
static void
cg6_setcursor (struct cg6_info *info)
{
	unsigned int v;
	struct cg6_cursor *c = &info->cursor;
	
	if (c->enable){
		v = ((c->cpos.fbx - c->chot.fbx) << 16)
		   |((c->cpos.fby - c->chot.fby) & 0xffff);
	} else {
		/* Magic constant to turn off the cursor */
		v = ((65536-32) << 16) | (65536-32);
	}
	info->thc->thc_cursxy = v;
}

#undef pos
static int
cg6_scursor (struct fbcursor *cursor, fbinfo_t *fb)
{
	int op = cursor->set;
	volatile struct cg6_thc *thc = fb->info.cg6.thc;
	struct cg6_cursor *cursor_info = &fb->info.cg6.cursor;
	int i, bytes = 0;
	
	if (op & FB_CUR_SETSHAPE){
		if ((unsigned int) cursor->size.fbx > 32)
			return -EINVAL;
		if ((unsigned int) cursor->size.fby > 32)
			return -EINVAL;
		bytes = (cursor->size.fby * 32)/8;
		i = verify_area (VERIFY_READ, cursor->image, bytes);
		if (i) return i;
		i = verify_area (VERIFY_READ, cursor->mask, bytes);
		if (i) return i;
	}
	if (op & (FB_CUR_SETCUR | FB_CUR_SETPOS | FB_CUR_SETHOT)){
		if (op & FB_CUR_SETCUR)
			cursor_info->enable = cursor->enable;
		if (op & FB_CUR_SETPOS)
			cursor_info->cpos = cursor->pos;
		if (op & FB_CUR_SETHOT)
			cursor_info->chot = cursor->hot;
		cg6_setcursor (&fb->info.cg6);
	}
	if (op & FB_CUR_SETSHAPE){
		unsigned int u;
		
		cursor_info->size = cursor->size;
		memset ((void *)&cursor_info->bits, 0, sizeof (cursor_info->size));
		memcpy (cursor_info->bits [0], cursor->mask, bytes);
		memcpy (cursor_info->bits [1], cursor->image, bytes);
		u = ~0;
		if (cursor_info->size.fbx < 32)
			u = ~(u  >> cursor_info->size.fbx);
		for (i = 0; i < 32; i++){
			int m = cursor_info->bits [0][i] & u;
			thc->thc_cursmask [i] = m;
			thc->thc_cursbits [i] = m & cursor_info->bits [1][i];
		}
	}
	return 0;
}

/* Handle cg6-specific ioctls */
static int
cg6_ioctl (struct inode *inode, struct file *file, unsigned cmd, unsigned long arg, fbinfo_t *fb)
{
	int i;

	switch (cmd){
	case FBIOGCURMAX:
		i = verify_area (VERIFY_WRITE, (void *) arg, sizeof (struct fbcurpos));
		if (i) return i;
		((struct fbcurpos *) arg)->fbx = 32;
		((struct fbcurpos *) arg)->fby = 32;
		break;

	case FBIOSVIDEO:
		/* vesa_blank and vesa_unblank could do the job on fb [0] */
		break;

	case FBIOSCURSOR:
		return cg6_scursor ((struct fbcursor *) arg, fb);

	case FBIOSCURPOS:
		/*
		i= verify_area (VERIFY_READ, (void *) arg, sizeof (struct fbcurpos));
		if (i) return i;
		*/
		fb->info.cg6.cursor.cpos = *(struct fbcurpos *)arg;
		cg6_setcursor (&fb->info.cg6);
		break;
	default:
		return -EINVAL;
	}
	return 0;
}

static void
cg6_setup (int slot, unsigned int cg6, int cg6_io)
{
	struct cg6_info *cg6info;

	printk ("cgsix%d at 0x%8.8x\n", slot, (unsigned int) cg6);
	
	/* Fill in parameters we left out */
	fbinfo [slot].type.fb_cmsize = 256;
	fbinfo [slot].mmap = cg6_mmap;
	fbinfo [slot].loadcmap = cg6_loadcmap;
	fbinfo [slot].ioctl = (void *) cg6_ioctl;
	fbinfo [slot].blank = 0;
	fbinfo [slot].unblank = 0;
	
	cg6info = (struct cg6_info *) &fbinfo [slot].info.cg6;

	/* Map the hardware registers */
	cg6info->bt = sparc_alloc_io ((void *) cg6+CG6_BROOKTREE_OFFSET, 0,
		 sizeof (struct bt_regs),"cgsix_dac", cg6_io, 0);
	cg6info->fhc = sparc_alloc_io ((void *) cg6+CG6_FHC_OFFSET, 0,
		 sizeof (int), "cgsix_fhc", cg6_io, 0);
	cg6info->thc = sparc_alloc_io ((void *) cg6+CG6_THC_OFFSET, 0,
		 sizeof (struct cg6_thc), "cgsix_thc", cg6_io, 0);
	cg6info->tec = sparc_alloc_io ((void *) cg6+CG6_TEC_OFFSET, 0,
		 sizeof (struct cg6_tec), "cgsix_tec", cg6_io, 0);
	cg6info->dhc = sparc_alloc_io ((void *) cg6+CG6_DHC_OFFSET, 0,
		 0x40000, "cgsix_dhc", cg6_io, 0);
	cg6info->fbc = sparc_alloc_io ((void *) cg6+CG6_FBC_OFFSET, 0,
		 0x1000, "cgsix_fbc", cg6_io, 0);
	if (!con_fb_base){
		con_fb_base = sparc_alloc_io ((void *) cg6+CG6_RAM_OFFSET, 0,
                    fbinfo [slot].type.fb_size, "cgsix_ram", cg6_io, 0);
	}
	if (!slot)
		restore_palette = cg6_restore_palette;
}

/* The cg3 driver, obio space addresses for mapping the cg3 stuff */
#define CG3_REGS 0x400000
#define CG3_RAM  0x800000
#define D4M3(x) ((((x)>>2)<<1) + ((x)>>2))      /* (x/4)*3 */
#define D4M4(x) ((x)&~0x3)                      /* (x/4)*4 */

/* The cg3 palette is loaded with 4 color values at each time  */
/* so you end up with: (rgb)(r), (gb)(rg), (b)(rgb), and so on */
static void
cg3_loadcmap (void *fbinfo, int index, int count)
{
	fbinfo_t *fb = (fbinfo_t *) fbinfo;
	struct bt_regs *bt = fb->info.cg3.bt;
	int *i, steps;

	i = &color_map.raw [D4M3(index)];
	steps = D4M3(index+count-1) - D4M3(index)+3;
	bt->addr = D4M4(index);
	while (steps--)
		bt->color_map = *i++;
}

/* The cg3 is presumed to emulate a cg4, I guess older programs will want that */
/* addresses above 0x4000000 are for cg3, below that it's cg4 emulation          */
static int
cg3_mmap (struct inode *inode, struct file *file, struct vm_area_struct *vma, long base, void *xx)
{
	unsigned int size, page, r, map_size;
	unsigned int map_offset = 0;
	fbinfo_t *fb = (fbinfo_t *) xx;
	
	size = vma->vm_end - vma->vm_start;
        if (vma->vm_offset & ~PAGE_MASK)
                return -ENXIO;

	/* To stop the swapper from even considering these pages */
	vma->vm_flags |= FB_MMAP_VM_FLAGS; 
	
	/* Each page, see which map applies */
	for (page = 0; page < size; ){
		switch (vma->vm_offset+page){
		case CG3_MMAP_OFFSET:
			map_size = size-page;
			map_offset = get_phys ((uint) con_fb_base);
			if (map_size > fb->type.fb_size)
				map_size = fb->type.fb_size;
			break;
		default:
			map_size = 0;
			break;
		}
		if (!map_size){
			page += PAGE_SIZE;
			continue;
		}
		r = io_remap_page_range (vma->vm_start+page,
					 map_offset,
					 map_size, vma->vm_page_prot,
					 fb->space);
		if (r) return -EAGAIN;
		page += map_size;
	}
        vma->vm_inode = inode;
        inode->i_count++;
        return 0;
}

static void
cg3_setup (int slot, unsigned int cg3, int cg3_io)
{
	struct cg3_info *cg3info;

	printk ("cgthree%d at 0x%8.8x\n", slot, cg3);
	
	/* Fill in parameters we left out */
	fbinfo [slot].type.fb_cmsize = 256;
	fbinfo [slot].mmap = cg3_mmap;
	fbinfo [slot].loadcmap = cg3_loadcmap;
	fbinfo [slot].ioctl = 0; /* no special ioctls */

	cg3info = (struct cg3_info *) &fbinfo [slot].info.cg3;

	/* Map the card registers */
	cg3info->bt = sparc_alloc_io ((void *) cg3+CG3_REGS, 0,
		 sizeof (struct bt_regs),"cg3_bt", cg3_io, 0);
	
	if (!con_fb_base){
		con_fb_base=sparc_alloc_io ((void*) cg3+CG3_RAM, 0,
                    fbinfo [slot].type.fb_size, "cg3_ram", cg3_io, 0);
	}
}

/* OBio addresses for the bwtwo registers */
#define BWTWO_REGISTER_OFFSET 0x400000

struct bwtwo_regs {
	char          unknown [16];
#define BWTWO_ENABLE_VIDEO 0x40
	unsigned char control;
	char          unknown2 [15];
};

static int
bwtwo_mmap (struct inode *inode, struct file *file, struct vm_area_struct *vma, long base, void *xx)
{
	unsigned int size, map_offset, r;
	fbinfo_t *fb = (fbinfo_t *) xx;
	int map_size;
	
	map_size = size = vma->vm_end - vma->vm_start;
	
	if (vma->vm_offset & ~PAGE_MASK)
		return -ENXIO;

	/* To stop the swapper from even considering these pages */
	vma->vm_flags |= FB_MMAP_VM_FLAGS;
	printk ("base=%8.8xl start=%8.8xl size=%x offset=%8.8x\n",
		(unsigned int) base,
		(unsigned int) vma->vm_start, size,
		(unsigned int) vma->vm_offset);

	/* This routine should also map the register if asked for, but we don't do that yet */
	map_offset = get_phys ((uint) con_fb_base);
	r = io_remap_page_range (vma->vm_start, map_offset, map_size, vma->vm_page_prot,
				 fb->space);
	if (r) return -EAGAIN;
	vma->vm_inode = inode;
	inode->i_count++;
	return 0;
}

static void
bwtwo_blank (void *xx)
{
	fbinfo_t *fb = (fbinfo_t *) xx;

	fb->info.bwtwo.regs->control &= ~BWTWO_ENABLE_VIDEO;
}

static void
bwtwo_unblank (void *xx)
{
	fbinfo_t *fb = (fbinfo_t *) xx;
	fb->info.bwtwo.regs->control |= BWTWO_ENABLE_VIDEO;
}

static void
bwtwo_setup (int slot, unsigned int bwtwo, int bw2_io)
{
	printk ("bwtwo%d at 0x%8.8x\n", slot, bwtwo);
	fbinfo [slot].type.fb_cmsize = 2;
	fbinfo [slot].mmap = bwtwo_mmap;
	fbinfo [slot].loadcmap = 0;
	fbinfo [slot].ioctl = 0;
	fbinfo [slot].blank = bwtwo_blank;
	fbinfo [slot].unblank = bwtwo_unblank;
	fbinfo [slot].info.bwtwo.regs = sparc_alloc_io ((void *) bwtwo+BWTWO_REGISTER_OFFSET,
		0, sizeof (struct bwtwo_regs), "bwtwo_regs", bw2_io, 0);
}

static char *known_cards [] = {
	"cgsix", "cgthree", "bwtwo", "SUNW,tcx", 0
};

static int
known_card (char *name)
{
	int i;

	for (i = 0; known_cards [i]; i++)
		if (strcmp (name, known_cards [i]) == 0)
			return 1;
	return 0;
}

static struct {
	int depth;
	int resx, resy;
	int x_margin, y_margin;
} scr_def [] = {
	{ 1, 1152, 900,  8,  18 },
	{ 8, 1152, 900,  64, 18 },
	{ 8, 1280, 1024, 96, 80 },
	{ 8, 1024, 768,  0,  0 },
	{ 0 },
};

static int
sparc_console_probe(void)
{
	int propl, con_node, i;
	struct linux_sbus_device *sbdp;
	unsigned int fbbase = 0xb001b001;
	int fbiospace = 0;

	/* XXX The detection code needs to support multiple video cards in one system */
	con_node = 0;
	switch(prom_vers) {
	case PROM_V0:
		/* V0 proms are at sun4c only. Can skip many checks. */
		con_type = FBTYPE_NOTYPE;
		if(SBus_chain == 0) {
			prom_printf("SBUS chain is NULL, bailing out...\n");
			prom_halt();
		}
		for_each_sbusdev(sbdp, SBus_chain) {
			con_node = sbdp->prom_node;

			/* If no "address" than it is not the PROM console. */
			if(sbdp->num_vaddrs) {
				if(!strncmp(sbdp->prom_name, "cgsix", 5)) {
					con_type = FBTYPE_SUNFAST_COLOR;
					fbbase = (uint) sbdp->reg_addrs [0].phys_addr;
					fbiospace = sbdp->reg_addrs[0].which_io;
					break;
				} else if(!strncmp(sbdp->prom_name, "cgthree", 7)) {
					con_type = FBTYPE_SUN3COLOR;
					fbbase = (uint) sbdp->reg_addrs [0].phys_addr;
					fbiospace = sbdp->reg_addrs[0].which_io;
					break;
				} else if (!strncmp(sbdp->prom_name, "bwtwo", 5)) {
					con_type = FBTYPE_SUN2BW;
					fbbase = (uint) sbdp->reg_addrs [0].phys_addr;
					fbiospace = sbdp->reg_addrs[0].which_io;
					break;
				}
			}
		}
		if(con_type == FBTYPE_NOTYPE) return -1;
		con_fb_base = (unsigned char *) sbdp->sbus_vaddrs[0];
		strncpy(con_name, sbdp->prom_name, sizeof (con_name));
		break;
	case PROM_V2:
	case PROM_V3:
	case PROM_P1275:
		for_each_sbusdev(sbdp, SBus_chain) {
			if (known_card (sbdp->prom_name))
				break;
		}
		if (!sbdp){
			prom_printf ("Could not find a know video card on this machine\n");
			prom_halt ();
		}
		prom_apply_sbus_ranges (&sbdp->reg_addrs [0], sbdp->num_registers);
		fbbase = (long) sbdp->reg_addrs [0].phys_addr;
		fbiospace = sbdp->reg_addrs[0].which_io;
		con_node = (*romvec->pv_v2devops.v2_inst2pkg)
			(*romvec->pv_v2bootargs.fd_stdout);
		/*
		 * Determine the type of hardware accelerator.
		 */
		propl = prom_getproperty(con_node, "emulation", con_name, sizeof (con_name));
		if (propl < 0 || propl >= sizeof (con_name)) {
			/* Early cg3s had no "emulation". */
			propl = prom_getproperty(con_node, "name", con_name, sizeof (con_name));
			if (propl < 0) {
				prom_printf("console: no device name!!\n");
				return -1;
			}
		}
		if(!strncmp(con_name, "cgsix", sizeof (con_name))) {
			con_type = FBTYPE_SUNFAST_COLOR;
		} else if(!strncmp(con_name, "cgthree", sizeof (con_name))) {
			con_type = FBTYPE_SUN3COLOR;
		} else if(!strncmp(con_name, "cgfourteen", sizeof (con_name))) {
			con_type = FBTYPE_MDICOLOR;
		} else if(!strncmp(con_name, "bwtwo", sizeof (con_name))) {
			con_type = FBTYPE_SUN2BW;
		} else if(!strncmp(con_name,"SUNW,tcx", sizeof (con_name))){
			con_type = FBTYPE_SUN3COLOR;
		} else {
			prom_printf("console: \"%s\" is unsupported\n", con_name);
			return -1;
		}
		propl = prom_getproperty(con_node, "address", (char *) &con_fb_base, 4);
		if (propl != 4) {
		    con_fb_base = 0;
		}
		break;
	default:
		return -1;
	};

	/* Get the device geometry */
	con_linebytes = prom_getintdefault(con_node, "linebytes", 1152);
	con_width = prom_getintdefault(con_node, "width", 1152);
	con_height = prom_getintdefault(con_node, "height", 900);

	/* Currently we just support 1-bit and 8-bit depth displays */
	if (con_type == FBTYPE_SUN2BW) {
		con_depth = 1;
	} else {
		con_depth = 8;
	}
	for (i = 0; scr_def [i].depth; i++){
		if (scr_def [i].resx != con_width || scr_def [i].resy != con_height)
			continue;
		if (scr_def [i].depth != con_depth)
		        continue;
		x_margin = scr_def [i].x_margin;
		y_margin = scr_def [i].y_margin;
		chars_per_line = (con_width * con_depth) / 8;
		skip_bytes = chars_per_line * y_margin;
		ints_per_line = chars_per_line / 4;
		bytes_per_row = CHAR_HEIGHT * chars_per_line;
		break;
	}
	if (!scr_def [i].depth){
		x_margin = y_margin = 0;
		prom_printf ("PenguinCon: unknown video resolution %dx%d may be slow\n", con_width, con_height);
		prom_halt ();
	}
	/* P3: I fear this strips 15inch 1024/768 PC-like monitors out. */
	if ((con_linebytes*8) / con_depth != con_width) {
		prom_printf("console: UNUSUAL VIDEO, linebytes=%d, width=%d, depth=%d\n",
			con_linebytes, con_width, con_depth);
		return -1;
	}

	/* Negate the font table on 1 bit depth cards so we have white on black */
	if (con_depth == 1)
		for(i=0; i<(16 * 256); i++)
			vga_font[i] = ~vga_font[i];

	/* Fill in common fb information */
	fbinfo [0].type.fb_type   = con_type;
	fbinfo [0].type.fb_height = con_height;
	fbinfo [0].type.fb_width  = con_width;
	fbinfo [0].type.fb_depth  = con_depth;
	fbinfo [0].type.fb_size   = PAGE_ALIGN((con_linebytes) * (con_height));
	fbinfo [0].space = fbiospace;
	fbinfo [0].blanked = 0;

	/* Should be filled in for supported video cards */
	fbinfo [0].mmap = 0; 
	fbinfo [0].loadcmap = 0;
	fbinfo [0].ioctl = 0;
	fbinfo [0].blank = 0;
	fbinfo [0].unblank = 0;

	if (fbbase == 0xb001b001){
		printk ("Mail miguel@nuclecu.unam.mx video_card=%d (%s)\n", con_type, con_name);
	}

	/* Per card setup */
	switch (con_type){
	case FBTYPE_SUN3COLOR:
		cg3_setup (0, fbbase, fbiospace);
		break;
	case FBTYPE_SUNFAST_COLOR:
		cg6_setup (0, fbbase, fbiospace);
		break;
	case FBTYPE_SUN2BW:
		bwtwo_setup (0, fbbase, fbiospace);
		break;
	default:
		break;
	}
	if (!con_fb_base){
		prom_printf ("PROM does not have an 'address' property for this\n"
			     "frame buffer and the Linux drivers do not know how\n"
			     "to map the video of this device\n");
		prom_halt ();
	}
	fbinfo [0].base = (long) con_fb_base;
	
	/* Register the frame buffer device */
	if (register_chrdev (GRAPHDEV_MAJOR, "graphics", &graphdev_fops)){
		printk ("Could not register graphics device\n");
		return -EIO;
	}
	return 0; /* success */
}

/* video init code, called from within the SBUS bus scanner at
 * boot time.
 */
void
sun_console_init(void)
{
	if(serial_console)
		return;

	if(sparc_console_probe()) {
		prom_printf("Could not probe console, bailing out...\n");
		prom_halt();
	}
	sun_clear_screen();
}

/*
 * sun_blitc
 *
 * Displays an ASCII character at a specified character cell
 *  position.
 *
 * Called from scr_writew() when the destination is
 *  the "shadow" screen
 */
static unsigned int
fontmask_bits[16] = {
    0x00000000,
    0x000000ff,
    0x0000ff00,
    0x0000ffff,
    0x00ff0000,
    0x00ff00ff,
    0x00ffff00,
    0x00ffffff,
    0xff000000,
    0xff0000ff,
    0xff00ff00,
    0xff00ffff,
    0xffff0000,
    0xffff00ff,
    0xffffff00,
    0xffffffff
};

int
sun_blitc(unsigned int charattr, unsigned long addr)
{
	int j, idx;
	unsigned char *font_row;

#ifndef DEBUGGING_X
	if (graphics_on)
		return 0;
#endif
	idx = (addr - video_mem_base) >> 1;

	/* Invalidate the cursor position if necessary. */
	if(idx == cursor_pos)
		cursor_pos = -1;
	font_row = &vga_font[(charattr & 0xff) << 4];

	switch (con_depth){
	case 1: {
		register unsigned char *dst;
		
		dst = (unsigned char *)(((unsigned long)con_fb_base) + FBUF_OFFSET(idx));
		for(j = 0; j < CHAR_HEIGHT; j++, font_row++, dst+=CHARS_PER_LINE)
			*dst = *font_row;
		break;
	}
	case 8: {
		register unsigned long *dst;
		unsigned long fgmask, bgmask, data, rowbits, attrib;
		const int ipl = ints_per_line;
		
		dst = (unsigned long *)(((unsigned long)con_fb_base) + COLOR_FBUF_OFFSET(idx));
		attrib = (charattr >> 8) & 0x0ff;
		fgmask = attrib & 0x0f;
		bgmask = (attrib >> 4) & 0x0f;
		fgmask = fgmask << 8 | fgmask;
		fgmask |= fgmask << 16;
		bgmask = bgmask << 8 | bgmask;
		bgmask |= bgmask << 16;
		
		for(j = 0; j < CHAR_HEIGHT; j++, font_row++, dst += ipl) {
			rowbits = *font_row;
			data = fontmask_bits[(rowbits>>4)&0xf];
			data = (data & fgmask) | (~data & bgmask);
			*dst = data;
			data = fontmask_bits[rowbits&0xf];
			data = (data & fgmask) | (~data & bgmask);
			*(dst+1) = data;
		}
		break;
	} /* case */
	} /* switch */
	return (0);
}

unsigned char vga_font[cmapsz] = {
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7e, 0x81, 0xa5, 0x81, 0x81, 0xbd, 
0x99, 0x81, 0x81, 0x7e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7e, 0xff, 
0xdb, 0xff, 0xff, 0xc3, 0xe7, 0xff, 0xff, 0x7e, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x6c, 0xfe, 0xfe, 0xfe, 0xfe, 0x7c, 0x38, 0x10, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x38, 0x7c, 0xfe, 
0x7c, 0x38, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 
0x3c, 0x3c, 0xe7, 0xe7, 0xe7, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x18, 0x3c, 0x7e, 0xff, 0xff, 0x7e, 0x18, 0x18, 0x3c, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x3c, 
0x3c, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 
0xff, 0xff, 0xe7, 0xc3, 0xc3, 0xe7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x3c, 0x66, 0x42, 0x42, 0x66, 0x3c, 0x00, 
0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc3, 0x99, 0xbd, 
0xbd, 0x99, 0xc3, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x1e, 0x0e, 
0x1a, 0x32, 0x78, 0xcc, 0xcc, 0xcc, 0xcc, 0x78, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x3c, 0x66, 0x66, 0x66, 0x66, 0x3c, 0x18, 0x7e, 0x18, 0x18, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3f, 0x33, 0x3f, 0x30, 0x30, 0x30, 
0x30, 0x70, 0xf0, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7f, 0x63, 
0x7f, 0x63, 0x63, 0x63, 0x63, 0x67, 0xe7, 0xe6, 0xc0, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x18, 0x18, 0xdb, 0x3c, 0xe7, 0x3c, 0xdb, 0x18, 0x18, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0xc0, 0xe0, 0xf0, 0xf8, 0xfe, 0xf8, 
0xf0, 0xe0, 0xc0, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x06, 0x0e, 
0x1e, 0x3e, 0xfe, 0x3e, 0x1e, 0x0e, 0x06, 0x02, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x18, 0x3c, 0x7e, 0x18, 0x18, 0x18, 0x7e, 0x3c, 0x18, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 
0x66, 0x00, 0x66, 0x66, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7f, 0xdb, 
0xdb, 0xdb, 0x7b, 0x1b, 0x1b, 0x1b, 0x1b, 0x1b, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x7c, 0xc6, 0x60, 0x38, 0x6c, 0xc6, 0xc6, 0x6c, 0x38, 0x0c, 0xc6, 
0x7c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0xfe, 0xfe, 0xfe, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x3c, 
0x7e, 0x18, 0x18, 0x18, 0x7e, 0x3c, 0x18, 0x7e, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x18, 0x3c, 0x7e, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 
0x18, 0x7e, 0x3c, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x18, 0x0c, 0xfe, 0x0c, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x60, 0xfe, 0x60, 0x30, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0xc0, 
0xc0, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x24, 0x66, 0xff, 0x66, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x10, 0x38, 0x38, 0x7c, 0x7c, 0xfe, 0xfe, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0xfe, 0x7c, 0x7c, 
0x38, 0x38, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x18, 0x3c, 0x3c, 0x3c, 0x18, 0x18, 0x18, 0x00, 0x18, 0x18, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x66, 0x66, 0x66, 0x24, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x6c, 
0x6c, 0xfe, 0x6c, 0x6c, 0x6c, 0xfe, 0x6c, 0x6c, 0x00, 0x00, 0x00, 0x00, 
0x18, 0x18, 0x7c, 0xc6, 0xc2, 0xc0, 0x7c, 0x06, 0x06, 0x86, 0xc6, 0x7c, 
0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc2, 0xc6, 0x0c, 0x18, 
0x30, 0x60, 0xc6, 0x86, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x38, 0x6c, 
0x6c, 0x38, 0x76, 0xdc, 0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x30, 0x30, 0x30, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x18, 0x30, 0x30, 0x30, 0x30, 
0x30, 0x30, 0x18, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x18, 
0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x18, 0x30, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x66, 0x3c, 0xff, 0x3c, 0x66, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x7e, 
0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x18, 0x30, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7e, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x02, 0x06, 0x0c, 0x18, 0x30, 0x60, 0xc0, 0x80, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xce, 0xde, 0xf6, 0xe6, 0xc6, 0xc6, 0x7c, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x38, 0x78, 0x18, 0x18, 0x18, 
0x18, 0x18, 0x18, 0x7e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6, 
0x06, 0x0c, 0x18, 0x30, 0x60, 0xc0, 0xc6, 0xfe, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x7c, 0xc6, 0x06, 0x06, 0x3c, 0x06, 0x06, 0x06, 0xc6, 0x7c, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x1c, 0x3c, 0x6c, 0xcc, 0xfe, 
0x0c, 0x0c, 0x0c, 0x1e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0xc0, 
0xc0, 0xc0, 0xfc, 0x06, 0x06, 0x06, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x38, 0x60, 0xc0, 0xc0, 0xfc, 0xc6, 0xc6, 0xc6, 0xc6, 0x7c, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0xc6, 0x06, 0x06, 0x0c, 0x18, 
0x30, 0x30, 0x30, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6, 
0xc6, 0xc6, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0x7e, 0x06, 0x06, 0x06, 0x0c, 0x78, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x00, 
0x00, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x18, 0x18, 0x00, 0x00, 0x00, 0x18, 0x18, 0x30, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x06, 0x0c, 0x18, 0x30, 0x60, 0x30, 0x18, 0x0c, 0x06, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7e, 0x00, 0x00, 
0x7e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 
0x30, 0x18, 0x0c, 0x06, 0x0c, 0x18, 0x30, 0x60, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x7c, 0xc6, 0xc6, 0x0c, 0x18, 0x18, 0x18, 0x00, 0x18, 0x18, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xde, 0xde, 
0xde, 0xdc, 0xc0, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x38, 
0x6c, 0xc6, 0xc6, 0xfe, 0xc6, 0xc6, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0xfc, 0x66, 0x66, 0x66, 0x7c, 0x66, 0x66, 0x66, 0x66, 0xfc, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3c, 0x66, 0xc2, 0xc0, 0xc0, 0xc0, 
0xc0, 0xc2, 0x66, 0x3c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf8, 0x6c, 
0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x6c, 0xf8, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0xfe, 0x66, 0x62, 0x68, 0x78, 0x68, 0x60, 0x62, 0x66, 0xfe, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0x66, 0x62, 0x68, 0x78, 0x68, 
0x60, 0x60, 0x60, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3c, 0x66, 
0xc2, 0xc0, 0xc0, 0xde, 0xc6, 0xc6, 0x66, 0x3a, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0xc6, 0xc6, 0xc6, 0xc6, 0xfe, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3c, 0x18, 0x18, 0x18, 0x18, 0x18, 
0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1e, 0x0c, 
0x0c, 0x0c, 0x0c, 0x0c, 0xcc, 0xcc, 0xcc, 0x78, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0xe6, 0x66, 0x66, 0x6c, 0x78, 0x78, 0x6c, 0x66, 0x66, 0xe6, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf0, 0x60, 0x60, 0x60, 0x60, 0x60, 
0x60, 0x62, 0x66, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc3, 0xe7, 
0xff, 0xff, 0xdb, 0xc3, 0xc3, 0xc3, 0xc3, 0xc3, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0xc6, 0xe6, 0xf6, 0xfe, 0xde, 0xce, 0xc6, 0xc6, 0xc6, 0xc6, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 
0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfc, 0x66, 
0x66, 0x66, 0x7c, 0x60, 0x60, 0x60, 0x60, 0xf0, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xd6, 0xde, 0x7c, 
0x0c, 0x0e, 0x00, 0x00, 0x00, 0x00, 0xfc, 0x66, 0x66, 0x66, 0x7c, 0x6c, 
0x66, 0x66, 0x66, 0xe6, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6, 
0xc6, 0x60, 0x38, 0x0c, 0x06, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0xff, 0xdb, 0x99, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x3c, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 
0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc3, 0xc3, 
0xc3, 0xc3, 0xc3, 0xc3, 0xc3, 0x66, 0x3c, 0x18, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0xc3, 0xc3, 0xc3, 0xc3, 0xc3, 0xdb, 0xdb, 0xff, 0x66, 0x66, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc3, 0xc3, 0x66, 0x3c, 0x18, 0x18, 
0x3c, 0x66, 0xc3, 0xc3, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc3, 0xc3, 
0xc3, 0x66, 0x3c, 0x18, 0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0xff, 0xc3, 0x86, 0x0c, 0x18, 0x30, 0x60, 0xc1, 0xc3, 0xff, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3c, 0x30, 0x30, 0x30, 0x30, 0x30, 
0x30, 0x30, 0x30, 0x3c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 
0xc0, 0xe0, 0x70, 0x38, 0x1c, 0x0e, 0x06, 0x02, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x3c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x3c, 
0x00, 0x00, 0x00, 0x00, 0x10, 0x38, 0x6c, 0xc6, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 
0x30, 0x30, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x78, 0x0c, 0x7c, 
0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xe0, 0x60, 
0x60, 0x78, 0x6c, 0x66, 0x66, 0x66, 0x66, 0x7c, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6, 0xc0, 0xc0, 0xc0, 0xc6, 0x7c, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1c, 0x0c, 0x0c, 0x3c, 0x6c, 0xcc, 
0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x7c, 0xc6, 0xfe, 0xc0, 0xc0, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x38, 0x6c, 0x64, 0x60, 0xf0, 0x60, 0x60, 0x60, 0x60, 0xf0, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0xcc, 0xcc, 
0xcc, 0xcc, 0xcc, 0x7c, 0x0c, 0xcc, 0x78, 0x00, 0x00, 0x00, 0xe0, 0x60, 
0x60, 0x6c, 0x76, 0x66, 0x66, 0x66, 0x66, 0xe6, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x18, 0x18, 0x00, 0x38, 0x18, 0x18, 0x18, 0x18, 0x18, 0x3c, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x06, 0x00, 0x0e, 0x06, 0x06, 
0x06, 0x06, 0x06, 0x06, 0x66, 0x66, 0x3c, 0x00, 0x00, 0x00, 0xe0, 0x60, 
0x60, 0x66, 0x6c, 0x78, 0x78, 0x6c, 0x66, 0xe6, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x38, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x3c, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xe6, 0xff, 0xdb, 
0xdb, 0xdb, 0xdb, 0xdb, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0xdc, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0x7c, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xdc, 0x66, 0x66, 
0x66, 0x66, 0x66, 0x7c, 0x60, 0x60, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x76, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0x7c, 0x0c, 0x0c, 0x1e, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0xdc, 0x76, 0x66, 0x60, 0x60, 0x60, 0xf0, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6, 0x60, 
0x38, 0x0c, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x30, 
0x30, 0xfc, 0x30, 0x30, 0x30, 0x30, 0x36, 0x1c, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0x76, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc3, 0xc3, 0xc3, 
0xc3, 0x66, 0x3c, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0xc3, 0xc3, 0xc3, 0xdb, 0xdb, 0xff, 0x66, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0xc3, 0x66, 0x3c, 0x18, 0x3c, 0x66, 0xc3, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc6, 0xc6, 0xc6, 
0xc6, 0xc6, 0xc6, 0x7e, 0x06, 0x0c, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0xfe, 0xcc, 0x18, 0x30, 0x60, 0xc6, 0xfe, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x0e, 0x18, 0x18, 0x18, 0x70, 0x18, 0x18, 0x18, 0x18, 0x0e, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x18, 0x18, 0x00, 0x18, 
0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x70, 0x18, 
0x18, 0x18, 0x0e, 0x18, 0x18, 0x18, 0x18, 0x70, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x76, 0xdc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x38, 0x6c, 0xc6, 
0xc6, 0xc6, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3c, 0x66, 
0xc2, 0xc0, 0xc0, 0xc0, 0xc2, 0x66, 0x3c, 0x0c, 0x06, 0x7c, 0x00, 0x00, 
0x00, 0x00, 0xcc, 0x00, 0x00, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0x76, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x18, 0x30, 0x00, 0x7c, 0xc6, 0xfe, 
0xc0, 0xc0, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x38, 0x6c, 
0x00, 0x78, 0x0c, 0x7c, 0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0xcc, 0x00, 0x00, 0x78, 0x0c, 0x7c, 0xcc, 0xcc, 0xcc, 0x76, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0x30, 0x18, 0x00, 0x78, 0x0c, 0x7c, 
0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00, 0x00, 0x38, 0x6c, 0x38, 
0x00, 0x78, 0x0c, 0x7c, 0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x3c, 0x66, 0x60, 0x60, 0x66, 0x3c, 0x0c, 0x06, 
0x3c, 0x00, 0x00, 0x00, 0x00, 0x10, 0x38, 0x6c, 0x00, 0x7c, 0xc6, 0xfe, 
0xc0, 0xc0, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc6, 0x00, 
0x00, 0x7c, 0xc6, 0xfe, 0xc0, 0xc0, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x60, 0x30, 0x18, 0x00, 0x7c, 0xc6, 0xfe, 0xc0, 0xc0, 0xc6, 0x7c, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x66, 0x00, 0x00, 0x38, 0x18, 0x18, 
0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x3c, 0x66, 
0x00, 0x38, 0x18, 0x18, 0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x60, 0x30, 0x18, 0x00, 0x38, 0x18, 0x18, 0x18, 0x18, 0x18, 0x3c, 
0x00, 0x00, 0x00, 0x00, 0x00, 0xc6, 0x00, 0x10, 0x38, 0x6c, 0xc6, 0xc6, 
0xfe, 0xc6, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00, 0x38, 0x6c, 0x38, 0x00, 
0x38, 0x6c, 0xc6, 0xc6, 0xfe, 0xc6, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00, 
0x18, 0x30, 0x60, 0x00, 0xfe, 0x66, 0x60, 0x7c, 0x60, 0x60, 0x66, 0xfe, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x6e, 0x3b, 0x1b, 
0x7e, 0xd8, 0xdc, 0x77, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3e, 0x6c, 
0xcc, 0xcc, 0xfe, 0xcc, 0xcc, 0xcc, 0xcc, 0xce, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x10, 0x38, 0x6c, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0x7c, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc6, 0x00, 0x00, 0x7c, 0xc6, 0xc6, 
0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0x30, 0x18, 
0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x30, 0x78, 0xcc, 0x00, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0x76, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0x30, 0x18, 0x00, 0xcc, 0xcc, 0xcc, 
0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc6, 0x00, 
0x00, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0x7e, 0x06, 0x0c, 0x78, 0x00, 
0x00, 0xc6, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0x7c, 
0x00, 0x00, 0x00, 0x00, 0x00, 0xc6, 0x00, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 
0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x7e, 
0xc3, 0xc0, 0xc0, 0xc0, 0xc3, 0x7e, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x38, 0x6c, 0x64, 0x60, 0xf0, 0x60, 0x60, 0x60, 0x60, 0xe6, 0xfc, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc3, 0x66, 0x3c, 0x18, 0xff, 0x18, 
0xff, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfc, 0x66, 0x66, 
0x7c, 0x62, 0x66, 0x6f, 0x66, 0x66, 0x66, 0xf3, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x0e, 0x1b, 0x18, 0x18, 0x18, 0x7e, 0x18, 0x18, 0x18, 0x18, 0x18, 
0xd8, 0x70, 0x00, 0x00, 0x00, 0x18, 0x30, 0x60, 0x00, 0x78, 0x0c, 0x7c, 
0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x18, 0x30, 
0x00, 0x38, 0x18, 0x18, 0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x18, 0x30, 0x60, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0x7c, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x30, 0x60, 0x00, 0xcc, 0xcc, 0xcc, 
0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0xdc, 
0x00, 0xdc, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x00, 0x00, 0x00, 0x00, 
0x76, 0xdc, 0x00, 0xc6, 0xe6, 0xf6, 0xfe, 0xde, 0xce, 0xc6, 0xc6, 0xc6, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x3c, 0x6c, 0x6c, 0x3e, 0x00, 0x7e, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x38, 0x6c, 0x6c, 
0x38, 0x00, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x30, 0x30, 0x00, 0x30, 0x30, 0x60, 0xc0, 0xc6, 0xc6, 0x7c, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0xc0, 
0xc0, 0xc0, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0xfe, 0x06, 0x06, 0x06, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0xc0, 0xc0, 0xc2, 0xc6, 0xcc, 0x18, 0x30, 0x60, 0xce, 0x9b, 0x06, 
0x0c, 0x1f, 0x00, 0x00, 0x00, 0xc0, 0xc0, 0xc2, 0xc6, 0xcc, 0x18, 0x30, 
0x66, 0xce, 0x96, 0x3e, 0x06, 0x06, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 
0x00, 0x18, 0x18, 0x18, 0x3c, 0x3c, 0x3c, 0x18, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x36, 0x6c, 0xd8, 0x6c, 0x36, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xd8, 0x6c, 0x36, 
0x6c, 0xd8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x11, 0x44, 0x11, 0x44, 
0x11, 0x44, 0x11, 0x44, 0x11, 0x44, 0x11, 0x44, 0x11, 0x44, 0x11, 0x44, 
0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa, 
0x55, 0xaa, 0x55, 0xaa, 0xdd, 0x77, 0xdd, 0x77, 0xdd, 0x77, 0xdd, 0x77, 
0xdd, 0x77, 0xdd, 0x77, 0xdd, 0x77, 0xdd, 0x77, 0x18, 0x18, 0x18, 0x18, 
0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 
0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0xf8, 0x18, 0x18, 0x18, 0x18, 
0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0xf8, 0x18, 0xf8, 
0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x36, 0x36, 0x36, 0x36, 
0x36, 0x36, 0x36, 0xf6, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0x36, 0x36, 0x36, 0x36, 
0x36, 0x36, 0x36, 0x36, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf8, 0x18, 0xf8, 
0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x36, 0x36, 0x36, 0x36, 
0x36, 0xf6, 0x06, 0xf6, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 
0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 
0x36, 0x36, 0x36, 0x36, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0x06, 0xf6, 
0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 
0x36, 0xf6, 0x06, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0xfe, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x18, 0x18, 0x18, 0xf8, 0x18, 0xf8, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0xf8, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 
0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x1f, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0xff, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0xff, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 
0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x1f, 0x18, 0x18, 0x18, 0x18, 
0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x18, 0x18, 
0x18, 0x18, 0x18, 0xff, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 
0x18, 0x18, 0x18, 0x18, 0x18, 0x1f, 0x18, 0x1f, 0x18, 0x18, 0x18, 0x18, 
0x18, 0x18, 0x18, 0x18, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x37, 
0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 
0x36, 0x37, 0x30, 0x3f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x3f, 0x30, 0x37, 0x36, 0x36, 0x36, 0x36, 
0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0xf7, 0x00, 0xff, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0xff, 0x00, 0xf7, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 
0x36, 0x36, 0x36, 0x36, 0x36, 0x37, 0x30, 0x37, 0x36, 0x36, 0x36, 0x36, 
0x36, 0x36, 0x36, 0x36, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00, 0xff, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x36, 0x36, 0x36, 0x36, 
0x36, 0xf7, 0x00, 0xf7, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 
0x18, 0x18, 0x18, 0x18, 0x18, 0xff, 0x00, 0xff, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0xff, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0xff, 0x00, 0xff, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x36, 0x36, 0x36, 0x36, 
0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x3f, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x18, 0x18, 
0x18, 0x1f, 0x18, 0x1f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x1f, 0x18, 0x1f, 0x18, 0x18, 0x18, 0x18, 
0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3f, 
0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 
0x36, 0x36, 0x36, 0xff, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 
0x18, 0x18, 0x18, 0x18, 0x18, 0xff, 0x18, 0xff, 0x18, 0x18, 0x18, 0x18, 
0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0xf8, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x1f, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 
0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 
0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0xf0, 0xf0, 0xf0, 
0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 
0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 
0x0f, 0x0f, 0x0f, 0x0f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x76, 0xdc, 0xd8, 0xd8, 0xd8, 0xdc, 0x76, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x78, 0xcc, 0xcc, 0xcc, 0xd8, 0xcc, 0xc6, 0xc6, 0xc6, 0xcc, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0xc6, 0xc6, 0xc0, 0xc0, 0xc0, 
0xc0, 0xc0, 0xc0, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0xfe, 0x6c, 0x6c, 0x6c, 0x6c, 0x6c, 0x6c, 0x6c, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0xfe, 0xc6, 0x60, 0x30, 0x18, 0x30, 0x60, 0xc6, 0xfe, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7e, 0xd8, 0xd8, 
0xd8, 0xd8, 0xd8, 0x70, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x66, 0x66, 0x66, 0x66, 0x66, 0x7c, 0x60, 0x60, 0xc0, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x76, 0xdc, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7e, 0x18, 0x3c, 0x66, 0x66, 
0x66, 0x3c, 0x18, 0x7e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x38, 
0x6c, 0xc6, 0xc6, 0xfe, 0xc6, 0xc6, 0x6c, 0x38, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x38, 0x6c, 0xc6, 0xc6, 0xc6, 0x6c, 0x6c, 0x6c, 0x6c, 0xee, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1e, 0x30, 0x18, 0x0c, 0x3e, 0x66, 
0x66, 0x66, 0x66, 0x3c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x7e, 0xdb, 0xdb, 0xdb, 0x7e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x03, 0x06, 0x7e, 0xdb, 0xdb, 0xf3, 0x7e, 0x60, 0xc0, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1c, 0x30, 0x60, 0x60, 0x7c, 0x60, 
0x60, 0x60, 0x30, 0x1c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 
0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0xfe, 0x00, 0x00, 0xfe, 0x00, 0x00, 0xfe, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x7e, 0x18, 
0x18, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 
0x18, 0x0c, 0x06, 0x0c, 0x18, 0x30, 0x00, 0x7e, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x0c, 0x18, 0x30, 0x60, 0x30, 0x18, 0x0c, 0x00, 0x7e, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0e, 0x1b, 0x1b, 0x1b, 0x18, 0x18, 
0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 
0x18, 0x18, 0x18, 0x18, 0xd8, 0xd8, 0xd8, 0x70, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x7e, 0x00, 0x18, 0x18, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0xdc, 0x00, 
0x76, 0xdc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x38, 0x6c, 0x6c, 
0x38, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x0c, 0x0c, 
0x0c, 0x0c, 0x0c, 0xec, 0x6c, 0x6c, 0x3c, 0x1c, 0x00, 0x00, 0x00, 0x00, 
0x00, 0xd8, 0x6c, 0x6c, 0x6c, 0x6c, 0x6c, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x70, 0xd8, 0x30, 0x60, 0xc8, 0xf8, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x7c, 0x7c, 0x7c, 0x7c, 0x7c, 0x7c, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
};
