#!/usr/bin/env python

"""***********************************************************************
** $Id: progressbar.py 322023 2004-06-19 20:33:37Z sedwards $
**
** Copyright (C) 1992-2000 Trolltech AS.  All rights reserved.
**
** This file is part of an example program for Qt.  This example
** program may be used, distributed and modified without limitation.
**
**************************************************************************"""

#import pdb
import sys
from qt import *

FALSE = 0
TRUE  = 1

class ProgressBar( QButtonGroup ):
    # Constructor
    # Creates child widgets of the ProgressBar widget
    def __init__( self, parent=None, name=None ):
        QButtonGroup.__init__( self, 0, Qt.Horizontal, "Progress Bar", parent, name )
        
        self.timer = QTimer()
        
        self.setMargin( 10 )
        toplayout = QGridLayout( self.layout(), 2, 2, 5)
            
        self.setRadioButtonExclusive( TRUE )
        
        # insert three radiobuttons which the user can use
        # to set the speed of the progress and two pushbuttons
        # to start/pause/continue and reset the progress
        self.slow   = QRadioButton( "&Slow",   self )
        self.normal = QRadioButton( "&Normal", self )
        self.fast   = QRadioButton( "&Fast",   self )
        vb1 = QVBoxLayout( )
        toplayout.addLayout( vb1, 0, 0 )
        vb1.addWidget( self.slow )
        vb1.addWidget( self.normal )
        vb1.addWidget( self.fast )
        
        # two push buttons, one for start, for for reset.
        self.start = QPushButton( "&Start", self )
        self.reset = QPushButton( "&Reset", self )
        
        vb2 = QVBoxLayout()
        toplayout.addLayout( vb2, 0, 1 )
        vb2.addWidget( self.start )
        vb2.addWidget( self.reset )
        
        # Create the progressbar
        self.progress = QProgressBar( 100, self )
        #self.progress.setStyle( QMotifStyle() )
        toplayout.addMultiCellWidget( self.progress, 1, 1, 0, 1 )
        
        # connect the clicked() SIGNALs of the pushbuttons to SLOTs
        self.connect( self.start, SIGNAL("clicked()"), self.slotStart )
        self.connect( self.reset, SIGNAL("clicked()"), self.slotReset )
        
        # connect the timeout() SIGNAL of the progress-timer to a SLOT
        self.connect( self.timer, SIGNAL("timeout()"), self.slotTimeout )
        # Let's start with normal speed...
        self.normal.setChecked( TRUE )

        # some contraints
        self.start.setFixedWidth( 80 )
        self.setMinimumWidth( 300 )
        
    # SLOT slotStart
    # This SLOT is called if the user clicks start/pause/continue button
    def slotStart( self ):
        # If the progress bar is at the beginning...
        if self.progress.progress() == -1 :
            # ...set according to the checked speed-radiobutton the
            # the number of steps which are needed to complete the process
            if self.slow.isChecked():
                self.progress.setTotalSteps( 10000 )
            elif self.normal.isChecked() :
                self.progress.setTotalSteps( 1000 )
            else :
                self.progress.setTotalSteps( 50 )
    
            # disable the speed-radiobuttons
            self.slow.setEnabled( FALSE )
            self.normal.setEnabled( FALSE )
            self.fast.setEnabled( FALSE )
            
        # If the progress is not running...
        if not self.timer.isActive() :
            # ...start the timer (and so the progress) with a interval of 1 ms...
            self.timer.start( 1 )
            # ...and rename the start/pause/continue button to Pause
            self.start.setText( "&Pause" )
        else : # if the prgress is running...
            # ...stop the timer (and so the prgress)...
            self.timer.stop()
            # ...and rename the start/pause/continue button to Continue
            self.start.setText( "&Continue" )
    
    # SLOT slotReset
    # This SLOT is called when the user clicks the reset button
    def slotReset( self ):
        # stop the timer and progress
        self.timer.stop()
    
        # rename the start/pause/continue button to Start...
        self.start.setText( "&Start" )
        # ...and enable this button
        self.start.setEnabled( TRUE )
    
        # enable the speed-radiobuttons
        self.slow.setEnabled( TRUE )
        self.normal.setEnabled( TRUE )
        self.fast.setEnabled( TRUE )
    
        # reset the progressbar
        self.progress.reset()
    
    # SLOT slotTimeout
    # This SLOT is called each ms when the timer is
    # active (== progress is running)
    def slotTimeout( self ):
        p = self.progress.progress()

        # If the progress is complete...
        if p == self.progress.totalSteps() :
            # ...rename the start/pause/continue button to Start...
            self.start.setText( "&Start" )
            # ...and disable it...
            self.start.setEnabled( FALSE )
            # ...and return
            return

        # If the process is not complete increase it
        p += 1
        self.progress.setProgress( p )
    

    #QPushButton *start, *pause, *reset;

def main( args ):
    a = QApplication( args )
    
    progressbar = ProgressBar()
    progressbar.setCaption("Qt Example - ProgressBar")
    a.setMainWidget(progressbar)
    progressbar.show()
    
    a.exec_loop()
    
if __name__=="__main__":
    main(sys.argv)
