/*
  KPF - Public fileserver for KDE

  Copyright 2001 Rik Hemsley (rikkus) <rik@kde.org>

  Permission is hereby granted, free of charge, to any person obtaining a copy
  of this software and associated documentation files (the "Software"), to
  deal in the Software without restriction, including without limitation the
  rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
  sell copies of the Software, and to permit persons to whom the Software is
  furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included in
  all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
  AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
  ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
  WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#include <cmath>

#include <qapplication.h>
#include <qdir.h>
#include <qstring.h>
#include <qcache.h>
#include <qstylesheet.h>
#include <qpalette.h>
#include <qtextstream.h>
#include <kglobalsettings.h>
#include <klocale.h>
#include <kmimetype.h>
#include <kurl.h>

#include "Defines.h"
#include "DirectoryLister.h"

namespace KPF
{
  class DirectoryLister::Private
  {
    public:

      Private()
      {
      }

      static QCache<QString> * cache;
  };

  QCache<QString> * DirectoryLister::Private::cache = 0L;

  QString colorToCSS(const QColor &c) {
    return "rgb(" + QString::number(c.red()) + ", " + QString::number(c.green()) + ", " + QString::number(c.blue()) + ")";
  }

  QByteArray buildHTML(const QString & title, const QString & body)
  {
    QPalette pal = qApp->palette();
    QByteArray temp_string;
    QTextStream html_stream(temp_string, IO_WriteOnly);

    html_stream.setEncoding(QTextStream::UnicodeUTF8);

    html_stream << "<?xml version=\"1.0\" encoding=\"UTF-8\"?>" << endl;
    html_stream <<
      "<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Strict//EN\"\n"
      "\"http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd\">\n"
      "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
      "\t<head>\n"
      "\t\t<title>";

    html_stream << title;
    html_stream << "</title>\n";

    html_stream << "<style type=\"text/css\">" << endl;
    html_stream << "<!--" << endl;

    html_stream << "table.filelist { ";
    html_stream << "color: " << colorToCSS(pal.color(QPalette::Normal, QColorGroup::Foreground)) << "; ";
    html_stream << "background-color: " << colorToCSS(pal.color(QPalette::Normal, QColorGroup::Background)) << "; ";
    html_stream << "border: thin outset; ";
    html_stream << "width: 100%; ";
    html_stream << "}" << endl;

    html_stream << "td { "; 
    html_stream << "margin: 0px; ";
    html_stream << "white-space: nowrap; ";
    html_stream << "}" << endl;

    html_stream << "td.norm { "; 
    html_stream << "background-color: " << colorToCSS(pal.color(QPalette::Normal, QColorGroup::Base)) << "; ";
    html_stream << "}" << endl;

    html_stream << "td.alt { "; 
    html_stream << "background-color: " << 
                    colorToCSS(
                               KGlobalSettings::calculateAlternateBackgroundColor(pal.color(QPalette::Normal, QColorGroup::Base))
                    ) << "; ";
    html_stream << "}" << endl;

    html_stream << "a { ";
    html_stream << "color: " << colorToCSS(pal.color(QPalette::Normal, QColorGroup::Text)) << "; ";
    html_stream << "text-decoration: none; ";
    html_stream << "}" << endl;

    html_stream << "th.listheading { ";
    html_stream << "color: " << colorToCSS(pal.color(QPalette::Normal, QColorGroup::ButtonText)) << "; ";
    html_stream << "background-color: " << colorToCSS(pal.color(QPalette::Normal, QColorGroup::Button)) << "; ";
    html_stream << "text-align: left; ";
    html_stream << "white-space: nowrap; ";
    html_stream << "border: thin outset; ";
    html_stream << "}" << endl;

    html_stream << "a.direntry { ";
    html_stream << "font-weight: bold; ";
    html_stream << "}" << endl;

    html_stream << "div.sizeentry { ";
    html_stream << "color: " << colorToCSS(pal.color(QPalette::Normal, QColorGroup::Text)) << "; ";
    html_stream << "text-align: right; ";
    html_stream << "}" << endl;


    html_stream << "-->" << endl;
    html_stream << "</style>" << endl;

    html_stream <<
      "\t</head>\n"
      "\t<body>\n";

    html_stream << body;

    html_stream <<
      "\t</body>\n"
      "</html>\n";

    return temp_string;
  }

  QString prettySize(uint size)
  {
    QString suffix;
    QString temp;
    float floated_size;

    if (size > 1023)
    {
      if (size > 1048575)
      {
        floated_size = size / 1048576.0;
        suffix = i18n(" MB");
      }
      else
      {
        floated_size = size / 1024.0;
        suffix = i18n(" KB");
      }
    }
    else 
    {
      temp.setNum(size);
      temp += i18n(" bytes");
      return temp;
    }

    temp.setNum(floated_size, 'f', 1);
    temp += suffix;
    return temp;
  }

  DirectoryLister * DirectoryLister::instance_ = 0L;

    DirectoryLister *
  DirectoryLister::instance()
  {
    if (0 == instance_)
      instance_ = new DirectoryLister;

    return instance_;
  }

  DirectoryLister::DirectoryLister()
  {
    Private::cache = new QCache<QString>(32 * 1024 /* 32 kb */);

    d = new Private;
  }

  DirectoryLister::~DirectoryLister()
  {
    delete d;

    delete Private::cache;
    Private::cache = 0;
  }

    QByteArray
  DirectoryLister::html(const QString & root, const QString & _path)
  {
    QString path;

    if (_path.right(1) != "/")
      path = _path + "/";
    else 
      path = _path;

    QByteArray temp_string;
    QTextStream html_stream(temp_string, IO_WriteOnly);
    QDir d(root + path);

    html_stream.setEncoding(QTextStream::UnicodeUTF8);

    if (!d.exists())
    {
      return buildHTML
        ("Error", "Directory does not exist: " + root + path);
    }

    const QFileInfoList * infoList = d.entryInfoList(QDir::DefaultFilter, QDir::Name | QDir::DirsFirst);

    if (0 == infoList)
    {
      return buildHTML
        ("Error", "Directory unreadable: " + root + path);
    }

    html_stream << "<table";
    html_stream << " width=\"100%\"";
    html_stream << " class=\"filelist\">\n";

    html_stream << "<tr>\n";
    html_stream << "<th align=\"left\" class=\"listheading\" width=\"100%\">Name</th>\n";
    html_stream << "<th align=\"left\" class=\"listheading\">Size</th>\n";
    html_stream << "</tr>\n";

    for (QPtrListIterator<QFileInfo> it(*infoList); it.current(); ++it)
    {
      static int counter = 0;
      QFileInfo * fi(it.current());

      if ((fi->fileName()[0] == '.') && ((fi->fileName() != "..") || path == "/"))
      {
        // Don't show hidden files
        continue;
      }

      counter++;
      QString td_class = (counter % 2) ? "alt" : "norm";

      html_stream << "<tr>\n";

      html_stream << "<td class=\"" << td_class << "\">";

      html_stream << "<img src=\"/?icon=";
      if (fi->fileName() != "..")
          html_stream << KMimeType::iconForURL(KURL(root + path + fi->fileName()));
      else
          html_stream << "folder_open" << endl;
      html_stream << "\" alt=\"\">&nbsp;";

      QString item_class = QString((fi->isDir()) ? "direntry" : "fileentry");
      html_stream << "<a href=\"" + QStyleSheet::escape(path + fi->fileName()) + "\" class=\"" << item_class << "\">";

      if (fi->fileName() != "..")
          html_stream << QStyleSheet::escape(fi->fileName());
      else 
          html_stream << i18n("Parent Directory");

      html_stream << "</a>";

      if (fi->isDir())
        html_stream << "/";

      html_stream << "</td>\n";

      html_stream << "<td class=\"" << td_class << "\">";
      if (!fi->isDir())
          html_stream << "<div class=\"sizeentry\">" + prettySize(fi->size()) + "</div>";
      html_stream << "</td>\n";
      html_stream << "</tr>\n";
    }

    html_stream << "</table>\n";

    return buildHTML("Directory listing for " + QStyleSheet::escape(path), temp_string);
  }

} // End namespace KPF

// vim:ts=2:sw=2:tw=78:et
