/*  ksim - a system monitor for kde
 *
 *  Copyright (C) 2001  Robbie Ward <linuxphreak@gmx.co.uk>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <time.h>
#include <unistd.h>
#include <stdlib.h>
#include <sys/stat.h>

#ifdef __FreeBSD__
#include <sys/types.h>
#include <sys/sysctl.h>
#include <sys/socket.h>
#include <net/if.h>
#include <net/if_dl.h>
#include <net/route.h>

static int mib[] = { CTL_NET, PF_ROUTE, 0, 0, NET_RT_IFLIST, 0 };
#endif

#include <qpushbutton.h>
#include <qtextstream.h>
#include <qfile.h>
#include <qdatetime.h>
#include <qlayout.h>
#include <qtimer.h>
#include <qregexp.h>

#include <kdebug.h>
#include <klocale.h>
#include <kaboutapplication.h>
#include <kaboutdata.h>
#include <ksimpleconfig.h>
#include <kglobal.h>
#include <krun.h>
#include <kapplication.h>

#include "ksimnet.h"
#include "netconfig.h"
#include <themetypes.h>
#include <label.h>
#include <led.h>
#include <chart.h>

#define NET_UPDATE 1000
#define LED_UPDATE 125

KSIM_INIT_PLUGIN(NetPlugin)

NetPlugin::NetPlugin(const char *name)
   : KSim::PluginObject(name)
{
  setConfigFileName(instanceName());
}

NetPlugin::~NetPlugin()
{
}

KSim::PluginView *NetPlugin::createView(const char *className)
{
  return new NetView(this, className);
}

KSim::PluginPage *NetPlugin::createConfigPage(const char *className)
{
  return new NetConfig(this, className);
}

void NetPlugin::showAbout()
{
  QString version = kapp->aboutData()->version();

  KAboutData aboutData(instanceName(),
     I18N_NOOP("KSim Net Plugin"), version.latin1(),
     I18N_NOOP("A net plugin for KSim"),
     KAboutData::License_GPL, "(C) 2001 Robbie Ward");

  aboutData.addAuthor("Robbie Ward", I18N_NOOP("Author"),
     "linuxphreak@gmx.co.uk");
  aboutData.addAuthor("Heitham Omar", I18N_NOOP("FreeBSD ports"),
     "super_ice@ntlworld.com");

  KAboutApplication(&aboutData).exec();
}

NetView::NetView(KSim::PluginObject *parent, const char *name)
   : KSim::PluginView(parent, name)
{
#ifdef __linux__
  m_procStream = 0L;
  if ((m_procFile = fopen("/proc/net/dev", "r")))
    m_procStream = new QTextStream(m_procFile, IO_ReadOnly);
#endif
#ifdef __FreeBSD__
  m_buf = 0;
  m_allocSize = 0;
#endif

  m_data = 0;
  m_oldData = 0;
  m_max = 0L;
  m_firstTime = 0;
  m_netLayout = new QVBoxLayout(this);

  config()->setGroup("Net");
  int deviceAmount = config()->readNumEntry("deviceAmount", 0);
  m_deviceList = createDeviceList(deviceAmount);
  init(deviceAmount);

  m_netTimer = new QTimer(this);
  connect(m_netTimer, SIGNAL(timeout()), SLOT(updateGraph()));
  m_netTimer->start(NET_UPDATE);
  m_lightTimer = new QTimer(this);
  connect(m_lightTimer, SIGNAL(timeout()), SLOT(updateLights()));
  m_lightTimer->start(LED_UPDATE);
  updateGraph();
}

NetView::~NetView()
{
#ifdef __linux__
  if (m_procFile)
    fclose(m_procFile);

  delete m_procStream;
#endif

  cleanup();
}

void NetView::reparseConfig()
{
  config()->setGroup("Net");

  int deviceAmount = config()->readNumEntry("deviceAmount", 0);
  DeviceList deviceList = createDeviceList(deviceAmount);
  if (deviceList == m_deviceList)
    return;

  m_netTimer->stop();
  m_lightTimer->stop();
  m_firstTime = 0;

  DeviceList::Iterator it;
  for (it = m_widgetList.begin(); it != m_widgetList.end(); ++it) {
    delete (*it).netLabel;
    delete (*it).netLed;
    delete (*it).netChart;
    delete (*it).cMenu;
    (*it).netLabel = 0;
    (*it).netLed = 0;
    (*it).netChart = 0;
    (*it).cMenu = 0;
  }

  m_deviceList = deviceList;
  cleanup();
  init(deviceAmount);
  m_netTimer->start(NET_UPDATE);
  m_lightTimer->start(LED_UPDATE);
}

void NetView::cleanup()
{
  if (m_data) {
    delete[] m_data;
    m_data = 0;
  }

  if (m_oldData) {
    delete[] m_oldData;
    m_oldData = 0;
  }

  if (m_max) {
    delete[] m_max;
    m_max = 0;
  }
}

void NetView::init(int amount)
{
  m_widgetList.clear();

  config()->setGroup("Net");

  if (!m_data)
    m_data = new NetData[amount];

  if (!m_oldData)
    m_oldData = new NetData[amount];

  if (!m_max)
    m_max = new unsigned long[amount];

  int i = 0;
  DeviceList::ConstIterator it;
  for (it = m_deviceList.begin(); it != m_deviceList.end(); ++it) {
    if (menu()->indexOf(100 + i) != -1)
      menu()->removeItem(100 + i);

    m_max[i] = 0;
    newNetMonitor((*it), i);
    ++i;
  }

  qHeapSort(m_widgetList);
}

void NetView::newNetMonitor(const NetDevice &device, int value)
{
  KSim::Chart *chart = (device.showGraph ? addChart() : 0L);
  KSim::LedLabel *led = addLedLabel(device.deviceName);
  KSim::Label *label = (device.showTimer ? addLabel() : 0L);

  m_widgetList.append(NetDevice(
     QString::number(value), device.showTimer,
     device.timerFormat, device.showLabel,
     device.deviceName, chart, led, label,
     (device.commandsEnabled ?
     addPopupMenu(device.deviceName, value) : 0L)));
}

// Run the connect command
void NetView::runConnectCommand(int value)
{
  int i = 0;
  DeviceList::ConstIterator it;
  for (it = m_deviceList.begin(); it != m_deviceList.end(); ++it) {
    if (value == i) {
      // I use KRun here as it provides startup notification
      if (!(*it).dCommand.isNull())
        KRun::runCommand((*it).cCommand);
      break;
    }
    ++i;
  }
}

// Run the disconnect command
void NetView::runDisconnectCommand(int value)
{
  int i = 0;
  DeviceList::ConstIterator it;
  for (it = m_deviceList.begin(); it != m_deviceList.end(); ++it) {
    if (value == i) {
      // I use KRun here as it provides startup notification
      if (!(*it).dCommand.isNull())
        KRun::runCommand((*it).dCommand);
      break;
    }
    ++i;
  }
}

DeviceList NetView::createDeviceList(int amount) const
{
  DeviceList deviceList;
  for (int i = 0; i < amount; ++i) {
    if (config()->hasGroup("device-" + QString::number(i))) {
      config()->setGroup("device-" + QString::number(i));

      deviceList.append(NetDevice(
         config()->readBoolEntry("showTimer"),
         config()->readEntry("deviceFormat"),
         config()->readBoolEntry("commands"),
         config()->readEntry("cCommand"),
         config()->readEntry("dCommand"),
         config()->readBoolEntry("showGraph"),
         config()->readBoolEntry("showLabel"),
         config()->readEntry("deviceName")));
    }
  }

  return deviceList;
}

void NetView::updateLights()
{
  int i = 0;
  DeviceList::ConstIterator it;
  for (it = m_widgetList.begin(); it != m_widgetList.end(); ++it) {
    if (isOnline((*it).deviceName)) {
      unsigned long recieveDiff = m_data[i].in - m_oldData[i].in;
      unsigned long sendDiff = m_data[i].out - m_oldData[i].out;

      if (recieveDiff == 0 && sendDiff == 0) {
        (*it).netLed->setValue(0);
        (*it).netLed->setOff(KSim::Led::First);
        (*it).netLed->setOff(KSim::Led::Second);
        continue;
      }

      unsigned long halfMax = m_max[i] / 2;
      (*it).netLed->setMaxValue(m_max[i] / 1024);
      (*it).netLed->setValue(recieveDiff / 1024);
      if (recieveDiff == 0)
        (*it).netLed->setOff(KSim::Led::First);
      else
        if ((recieveDiff / 1024) >= halfMax)
          (*it).netLed->setOn(KSim::Led::First);
        else
          (*it).netLed->toggle(KSim::Led::First);

      if (sendDiff == 0)
        (*it).netLed->setOff(KSim::Led::Second);
      else
        if ((sendDiff / 1024) >= halfMax)
          (*it).netLed->setOn(KSim::Led::Second);
        else
          (*it).netLed->toggle(KSim::Led::Second);
    }
    else {
      (*it).netLed->setMaxValue(0);
      (*it).netLed->setValue(0);
      (*it).netLed->setOff(KSim::Led::First);
      (*it).netLed->setOff(KSim::Led::Second);
    }
    ++i;
  }
}

void NetView::updateGraph()
{
  int i = 0;
  QString hours, minutes, seconds;
  int timer = 0, h = 0, m = 0, s = 0;
  time_t start = 0;
  struct stat st;
  QTime netTime;
  QString timerFormat;
  QString pid("/var/run/%1.pid");
  QString newPid;

  DeviceList::ConstIterator it;
  for (it = m_widgetList.begin(); it != m_widgetList.end(); ++it) {
    if (isOnline((*it).deviceName)) {
      timerFormat = (*it).timerFormat;
      newPid = pid.arg((*it).deviceName);
      
      if (QFile::exists(newPid) && stat(QFile::encodeName(newPid).data(), &st) == 0) {
        start = st.st_mtime;

        timer = (int)(time(0) - start);
        h = timer / 3600;
        m = (timer - h * 3600) / 60;
        s = timer % 60;
        if (netTime.isValid(h, m, s))
          netTime.setHMS(h, m, s);
      }

      m_oldData[i] = m_data[i];

      if ((*it).showTimer) {
        hours.sprintf("%02i", netTime.hour());
        minutes.sprintf("%02i", netTime.minute());
        seconds.sprintf("%02i", netTime.second());

        timerFormat.replace(QRegExp("%h"), hours);
        timerFormat.replace(QRegExp("%m"), minutes);
        timerFormat.replace(QRegExp("%s"), seconds);
        (*it).netLabel->setText(timerFormat);
      }

      netStatistics((*it).deviceName, m_data[i]);

      unsigned long recieveDiff = m_data[i].in - m_oldData[i].in;
      unsigned long sendDiff = m_data[i].out - m_oldData[i].out;

      if (m_firstTime == 0) {
        recieveDiff = 0;
        sendDiff = 0;
      }

      if ((*it).netChart) {
        (*it).netChart->setValue(recieveDiff, sendDiff);
        m_max[i] = (*it).netChart->maxValue();

        if ((*it).showLabel)
          (*it).netChart->setText(i18n("in: %1k")
             .arg(KGlobal::locale()->formatNumber((float)recieveDiff / 1024.0, 1)),
             i18n("out: %1k").arg(KGlobal::locale()->formatNumber((float)sendDiff / 1024.0, 1)));
      }
    }
    else {
      m_data[i] = m_oldData[i] = 0;
      m_max[i] = 0;

      if ((*it).netChart) {
        (*it).netChart->setValue(0, 0);

        if ((*it).showLabel)
          (*it).netChart->setText(i18n("in: %1k")
             .arg(KGlobal::locale()->formatNumber(0.0, 1)),
             i18n("out: %1k").arg(KGlobal::locale()->formatNumber(0.0, 1)));
      }

      if ((*it).showTimer)
        (*it).netLabel->setText(i18n("offline"));
    }
    ++i;
  }

  if (m_firstTime != 1)
    m_firstTime = 1;
}

KSim::Chart *NetView::addChart()
{
  KSim::Chart *chart = new KSim::Chart(false, 0, this);
  m_netLayout->addWidget(chart, 0, AlignHCenter);
  chart->show();
  return chart;
}

KSim::LedLabel *NetView::addLedLabel(const QString &device)
{
  KSim::LedLabel *ledLabel = new KSim::LedLabel(0, KSim::Types::Net,
     device, this);

  ledLabel->show();
  m_netLayout->addWidget(ledLabel);
  return ledLabel;
}

KSim::Label *NetView::addLabel()
{
  KSim::Label *label = new KSim::Label(KSim::Types::None, this);
  label->show();
  m_netLayout->addWidget(label);
  return label;
}

QPopupMenu *NetView::addPopupMenu(const QString &device, int value)
{
  QPopupMenu *popup = new QPopupMenu(this);
  popup->insertItem(i18n("Connect"), this,
     SLOT(runConnectCommand(int)), 0, 1);
  popup->setItemParameter(1, value);
  popup->insertItem(i18n("Disconnect"), this,
     SLOT(runDisconnectCommand(int)), 0, 2);
  popup->setItemParameter(2, value);
  menu()->insertItem(device, popup, 100 + value);
  return popup;
}

void NetView::netStatistics(const QString &device, NetData &data)
{
#ifdef __linux__
  if (m_procFile == 0) {
    data.in = 0;
    data.out = 0;
    return;
  }

  QString output;
  QString parser;
  // Parse the proc file
  while (!m_procStream->atEnd()) {
    parser = m_procStream->readLine();
    // remove all the entrys apart from the line containing 'device'
    if (parser.find(device) != -1)
      output = parser;
  }

  if (output.isEmpty()) {
    data.in = 0;
    data.out = 0;
    return;
  }

  // make sure our output doesnt contain "eth0:11210107" so we dont
  // end up with netList[1] actually being netList[2]
  output.replace(QRegExp(":"), " ");
  QStringList netList = QStringList::split(' ', output);

  data.in = netList[1].toULong();
  data.out = netList[9].toULong();

  fseek(m_procFile, 0L, SEEK_SET);
#endif

#ifdef __FreeBSD__
  struct if_msghdr *ifm, *nextifm;
  struct sockaddr_dl *sdl;
  char *lim, *next;
  size_t needed;
  char s[32];

  if (sysctl(mib, 6, NULL, &needed, NULL, 0) < 0)
    return;

  if (m_allocSize < needed) {
    if (m_buf != NULL)
      delete[] m_buf;

    m_buf = new char[needed];

    if (m_buf == NULL)
      return;

    m_allocSize = needed;
  }

  if (sysctl(mib, 6, m_buf, &needed, NULL, 0) < 0)
    return;

  lim = m_buf + needed;

  next = m_buf;
  while (next < lim) {
    ifm = (struct if_msghdr *)next;
    if (ifm->ifm_type != RTM_IFINFO)
      return;

    next += ifm->ifm_msglen;

    // get an interface with a network address
    while (next < lim) {
      nextifm = (struct if_msghdr *)next;
      if (nextifm->ifm_type != RTM_NEWADDR)
        break;

      next += nextifm->ifm_msglen;
    }

    // if the interface is up
    if (ifm->ifm_flags & IFF_UP) {
      sdl = (struct sockaddr_dl *)(ifm + 1);
      if (sdl->sdl_family != AF_LINK)
        continue;

      strncpy(s, sdl->sdl_data, sdl->sdl_nlen);
      s[sdl->sdl_nlen] = '\0';

      if (strcmp(device.local8Bit().data(), s) == 0) {
        data.in = ifm->ifm_data.ifi_ibytes;
        data.out = ifm->ifm_data.ifi_obytes;
        return;
      }
    }
  }
#endif
}

bool NetView::isOnline(const QString &device)
{
#ifdef __linux__
  QFile file("/proc/net/route");
  if (!file.open(IO_ReadOnly))
    return -1;

  return (QTextStream(&file).read().find(device) != -1 ? true : false);
#endif

#ifdef __FreeBSD__
  struct if_msghdr *ifm, *nextifm;
  struct sockaddr_dl *sdl;
  char *lim, *next;
  size_t needed;
  char s[32];

  if (sysctl(mib, 6, NULL, &needed, NULL, 0) < 0)
    return false;

  if (m_allocSize < needed) {
    if (m_buf != NULL)
      delete[] m_buf;

    m_buf = new char[needed];

    if (m_buf == NULL)
      return false;

    m_allocSize = needed;
  }

  if (sysctl(mib, 6, m_buf, &needed, NULL, 0) < 0)
    return false;

  lim = m_buf + needed;

  next = m_buf;
  while (next < lim) {
    ifm = (struct if_msghdr *)next;
    if (ifm->ifm_type != RTM_IFINFO)
      return false;

    next += ifm->ifm_msglen;

    // get an interface with a network address
    while (next < lim) {
      nextifm = (struct if_msghdr *)next;
      if (nextifm->ifm_type != RTM_NEWADDR)
        break;

      next += nextifm->ifm_msglen;
    }

    // if the interface is up
    if (ifm->ifm_flags & IFF_UP) {
      sdl = (struct sockaddr_dl *)(ifm + 1);
      if (sdl->sdl_family != AF_LINK)
        continue;

      strncpy(s, sdl->sdl_data, sdl->sdl_nlen);
      s[sdl->sdl_nlen] = '\0';

    if (strcmp(s, device.local8Bit().data()) == 0)
      return true;
    }
  }

  return false;
#endif
}

#include "ksimnet.moc"
