// **********************************************************************
// * IBMASM HEADER FILE                                                 *
// *                                                                    *
// * Written by Matthew DeLoera (deloera@us.ibm.com)                    *
// * (C) Copyright IBM Corporation, 1998-2001                           *
// *                                                                    *
// *  This software may be used and distributed according to the terms  *
// *  of the GNU Public License, incorporated herein by reference.      *
// **********************************************************************

// -------------------------------------------------------------------------------------------------------
// SLIM.C
// This file contains the implementation of the slim protocol
//
// CHANGE HISTORY:
// Date      Author       Description
//
//  ??         ??         Creation
//
// 08/22/97  A. Parsons   Formatted, added comments.  Looks OK for Kiowa.
//
// 03/04/98  A. Parsons   Added comments.  No code changes.
//
// 04/07/98  A. Parsons   Changed an error condition in MsDataStateCksum2 to
//                        MsDataStateEnd in order to properly discard extra bytes
//                        from a bad receive packet.
//
// 04/23/99  M. DeLoera   Inserted _FAR directive where needed for building under OS/2.
//
// -------------------------------------------------------------------------------------------------------
                                                        
#include "slim.h"
// #include "../../slim/slim.h"
// #include "ntddk.h"    // for WINNT's DbgPrint() function

// **************************************************************************
//  NAME:
//     htons
//
//  DESCRIPTION:
//     byte swaps the upper and lower bytes in a 2-byte word
//
//  INPUTS:
//     unsigned short val
//
//  OUTPUTS:
//     none
//
//  RETURNS
//     byte swapped val
//
// **************************************************************************
// swap a short
unsigned short htons(unsigned short val)
{
   return ((unsigned short)(((val << 8)&0xff00)|((val >> 8)&0x00ff)));
}


// **************************************************************************
//  NAME:
//     rfc_cksum - computes a check sum according to RFC1071
//
//  DESCRIPTION:
//     Computes a check sum
//
//  INPUTS:
//     unsigned short type - message type
//     unsigned short len  - message length
//     PBYTE buf           - message buffer
//     ULONG size          - buffer size
//
//  OUTPUTS:
//     none
//
//  RETURN
//     check sum
//
// **************************************************************************

unsigned short rfc_cksum(unsigned short     type,
                         unsigned short     len,
                         unsigned char _FAR *buf,
                         unsigned short     size)
{
   unsigned long sum;

   sum = htons(type);
   sum += htons(len);
   while (size > 1)
   {
      sum += *(unsigned short _FAR *)buf;  // add the next word to "sum"
      buf += 2;                            // point to next word in slim packet
      size -= (unsigned short)2;           // dec "size" by 2 bytes
   }

   if (size)                           // if 1 byte remains to be added...
   {
      sum += *buf;                     // add buffer byte value assuming zero pad
   }

   // DbgPrint("SLIM.C Calculating checksum.....\n");
   // DbgPrint("       Packet sum is %08X\n", sum);
   while (sum >> 16)
      sum = (sum&0xffff)+(sum >> 16);

   // DbgPrint("       After adding carry %08X\n", sum);
   sum = (unsigned short)~sum;
   // DbgPrint("       After one's compliment %08X\n", sum);
   return (unsigned short)sum;
}


// **************************************************************************
//  NAME:
//     InitializeSlimReceive - initialize the recieve state machine members
//
//  DESCRIPTION:
//     Initializes recieve state machinedata members
//
//  INPUTS:
//     bufferSize
//       This routine is only called from ProcessIncoming() DDRI2C.C and RpInit()
//       DDRCMD.C.  From these 2 routines, the pass parm bufferSize is set to
//       MIN_EVENT_SIZE, which is defined as 2048 in DDR.H
//
//  OUTPUTS:
//     none
//
//  RETURN
//     none
//
// *************************************************************************

void InitializeSlimReceive(SLIM_STATUS        *slim,
                           unsigned char _FAR *buffer,
                           unsigned short     bufferSize)
{
   slim->state      = MsRcvStateIdle;
   slim->buffer     = buffer;
   slim->bufferSize = bufferSize;
   slim->escape     = FALSE;
   slim->charCount  = 0;
}


// **************************************************************************
//  NAME:
//     InitializeSlimSend - initialize the send state machine members
//
//  DESCRIPTION:
//     Initializes send state machinedata members
//
//  INPUTS:
//
//
//
//  OUTPUTS:
//     none
//
//  RETURN
//     none
//
// *************************************************************************

void InitializeSlimSend(SLIM_STATUS        *slim,
                        unsigned char _FAR *buffer,
                        unsigned short     bufferSize)
{
   slim->state               = SendHeader;
   slim->buffer              = buffer;
   slim->bufferSize          = bufferSize;
   slim->charCount           = 0;
   slim->escape              = FALSE;
   slim->u1.header.syn       = SYN;
   slim->u1.header.stx       = STX;
   slim->u1.header.type      = htons(TYPE_I2C);
   slim->u1.header.length    = htons((unsigned short)bufferSize);
   slim->u2.trailer.checksum = rfc_cksum(TYPE_I2C,
                                         (unsigned short)bufferSize,
                                         buffer,
                                         (unsigned short)bufferSize);
   slim->u2.trailer.end      = END;
}


// **************************************************************************
//  NAME:
//     ResetSlimSend - reset the send state machine members
//
//  DESCRIPTION:
//     Sets the state machine to complete so buffer will not be accessed
//
//  INPUTS:
//
//
//
//  OUTPUTS:
//     none
//
//  RETURN
//     none
//
// *************************************************************************

void ResetSlimSend(SLIM_STATUS *slim)
{
   slim->state = SendComplete;         // Reset slim state machine to complete state
   slim->escape = FALSE;               // GetNextSlimByte will not access slim->buffer
}

//This is the main handeling routine for incoming bytes
//to be converted to an acual command packet.


// **************************************************************************
//  NAME:
//     SetNextSlimByte
//
//  DESCRIPTION:
//     Process a raw received character
//
//  INPUTS:
//     SLIM_STATUS *slim,  - state variables
//     unsigned char *ch           - raw character to process
//
//  OUTPUTS:
//     none
//
//  RETURNS
//     0  - complete message not received
//     !0 - message received - returns message size
//
// *************************************************************************
// For reference, this is what a slim packet looks like........
//
//                      |<-------- Compute checksum across this data ------------------>|
//                      |        |<----------------- LENGTH -------------------->|      |
//                      |        |<----------- slim->charCount ----------------->|      |
//    16   02    00 03    00 0B   00   03    02 00    00    00   09 01 04   01 30    00    C0 EC    C0
//    SYN  STX  MSB LSB  MSB LSB  CMD  CMD  LSB MSB  STAT  RSVD  COMMAND   COMMAND   PAD EVEN ODD   END
//                TYPE   LENGTH   TYPE LEN  DATA-LEN             STRING      DATA        CHECKSUM
//                                                                                         BYTES
//                               |<------ spCmd struct ------->|
//    |<----- Packet Header ---->|<----------------- Packet Data ---------------------->|<- Pkt Ftr ->|
//                                 ^
//                                 |
//                           slim->buffer
//
// *************************************************************************

unsigned short SetNextSlimByte(SLIM_STATUS *slim, unsigned char _FAR *ch)
{
   unsigned int  calcChkSum;    // calculated checksum value

   // Check for esc sequence and some invalid character sequences
   if (CheckESC(slim, ch))
   {
      // process current character if appropriate
      switch (slim->state)
      {
      // ignore characters until a sync, stx received
         case  MsRcvStateIdle :
            if (*ch == SYN)
            {
               // ************************************************************
               // Got a SYN character
               // ************************************************************
               KernalConsolePrint("\n\rSLIM.C ProcessChar got SYN");
               slim->state = MsRcvStateNeedSTX;
               slim->charCount = 0;
            }                          /* endif                                                   */
            break;

         // reset if not STX else start data state on header
         case  MsRcvStateNeedSTX :
            // DbgPrint("SLIM.C Got a SYN character\n", *ch);
            if (*ch == STX)
            {
               // ************************************************************
               // Got a STX character
               // ************************************************************
               slim->state = MsRcvStatePktType1;
            }
            else
            {
               if (*ch == SYN)
               {
                  slim->state = MsRcvStateNeedSTX;
               }
               else
               {
                  KernalConsolePrint("\n\rSLIM.C ProcessChar no STX");
                  slim->state = MsRcvStateIdle;
               }
            }
            break;

         case  MsRcvStatePktType1 :
            // ************************************************************
            // Got MSB of packet type
            // ************************************************************
            slim->u1.header.type = *ch;
            slim->state = MsRcvStatePktType2;
            break;

         case  MsRcvStatePktType2 :
            // ************************************************************
            // Got LSB of packet type
            // ************************************************************
            slim->u1.header.type = (slim->u1.header.type << 8)+*ch;
            if (slim->u1.header.type == TYPE_I2C)
            {
               slim->state = MsDataStateLgth1;
            }
            else
            {
               if (*ch == SYN)
               {
                  slim->state = MsRcvStateNeedSTX;
               }
               else
               {
                  KernalConsolePrint("\n\rSLIM.C ProcessChar bad pkt type");
                  slim->state = MsRcvStateIdle;
               }
            }
            break;

         case  MsDataStateLgth1 :
            // ************************************************************
            // Got MSB of packet length
            // ************************************************************
            slim->u1.header.length = *ch;
            slim->state = MsDataStateLgth2;
            break;

         case  MsDataStateLgth2 :
            // ************************************************************
            // Got LSB of packet length
            // ************************************************************
            slim->u1.header.length = (slim->u1.header.length << 8)+*ch;
            slim->state = MsDataStateType;
            break;

/* Trying to remove this state.                                                                   */

         case  MsDataStateType :
            if (slim->buffer == NULL)
            {
            // ************************************************************
            // Got first byte of packet data.  This is also the first byte of
            // the spCmd structure and is the command type.
            // ************************************************************
            // Up to now, the pointer to where to put the guts of the message has
            // been a NULL pointer.  At this point, the message header has been
            // received, and the msg type (response, event, heartbeat) came in.
            // GetSLIMBuffer will setup the buffer pointer appropriately
            // so that the incoming data can be put in the correct place.  If the
            // message is a response, the buffer is the application-allocated
            // space whose pointer was passed in during the SystemDataIO() call
            // for example.  If it is an event, the message will be dumped into one
            // of the event buffers allocated by the driver when it was loaded and
            // first started (ie: DriverInfo.EventsRcvd[i] in Netware).
               if (!GetSLIMBuffer(slim, *ch))
               {
                  if (*ch == SYN)
                  {
                     slim->state = MsRcvStateNeedSTX;
                  }
                  else
                  {
                     KernalConsolePrint("\n\rSLIM.C ProcessChar bad cmd type");
                     slim->state = MsRcvStateIdle;
                  }
                  break;
               }
            }
            slim->buffer[slim->charCount++] = *ch;
            slim->state = MsDataStateData;
            break;
         case  MsDataStateData :
            // ************************************************************
            // Continue pulling packet data bytes until reaching the packet
            // length
            // ************************************************************
            slim->buffer[slim->charCount++] = *ch;
            if (slim->charCount == slim->u1.header.length)
            {
               // Just received the last data byte in the slim packet, so
               // predicit what the next slim state will be so this routine
               // knows where to go when the next byte arrives.
               if (slim->charCount&0x01)               // if odd number of characters...
                  slim->state = MsDataStatePad;        //  the next byte should be a pad
               else
                  slim->state = MsDataStateCksum1;     // else it is the 1st checksum byte
            }
            else
            {
               if (slim->charCount >= slim->bufferSize)
               {
                  if (*ch == SYN)
                  {
                     slim->state = MsRcvStateNeedSTX;
                  }
                  else
                  {
                     KernalConsolePrint("\n\rSLIM.C ProcessChar bfr overflow");
                     slim->state = MsRcvStateIdle;
                  }
               }
            }
            break;
         case  MsDataStatePad :
            // ************************************************************
            // Got the pad byte.
            // ************************************************************
            slim->buffer[slim->charCount++] = *ch;
            slim->state = MsDataStateCksum1;

            // KernalConsolePrint("\n\rSLIM.C Got the pad byte (%02X)", *ch);
            // DbgPrint("SLIM.C Got a pad byte (%02X)\n", *ch);

            break;
         case  MsDataStateCksum1 :
            // ************************************************************
            // Got the even checksum byte
            // ************************************************************
            slim->u2.trailer.checksum = *ch;
            slim->state = MsDataStateCksum2;

            // DbgPrint("SLIM.C Got the even checksum byte (%02X)\n", *ch);

            break;
         case  MsDataStateCksum2 :
            // ************************************************************
            // Got the odd checksum byte
            // ************************************************************
            slim->u2.trailer.checksum += (*ch << 8);

            // DbgPrint("SLIM.C Got the odd checksum byte (%02X)\n", *ch);
            // DbgPrint("SLIM.C Received checksum is %04X\n", slim->u2.trailer.checksum);

         // check checksum - includes type, length, data, pad
            calcChkSum = rfc_cksum(slim->u1.header.type,
                                   slim->u1.header.length,
                                   (unsigned char _FAR *)slim->buffer,
                                   (unsigned short)slim->charCount);

            // DbgPrint("SLIM.C Calculated checksum is %04X\n", calcChkSum);

            if (calcChkSum != slim->u2.trailer.checksum)
            {
               if (*ch == SYN)
               {
                  slim->state = MsRcvStateNeedSTX;
               }
               else
               {
                  // slim->state = MsRcvStateIdle;

                  // The proper state after receiving a bad checksum is MsDataStateEnd.
                  // In this state, any remaining bytes from the bad packet will
                  // be discarded because the code will only be looking for a SYN
                  // character that marks the begining of a new packet.
                  slim->state = MsDataStateEnd;
               }
               KernalConsolePrint("\n\rSLIM.C ProcessChar FAILED BAD XSUM");
               break;
            }
            slim->state = MsDataStateEnd;
            break;
         case  MsDataStateEnd :
            if (*ch == SYN)
            {
               slim->state = MsRcvStateNeedSTX;
            }
            else
            {
               // ************************************************************
               // Got the END byte.
               // ************************************************************
               slim->state = MsRcvStateIdle;
            }
            return (unsigned short)(slim->u1.header.length);
         default  :
            break;
      }
   }
   return (0);
}


// *************************************************************************
//  NAME:
//     GetNextSlimByte
//
//  DESCRIPTION:
//     Get next byte to send using SLIM protocol
//
//  INPUTS:
//     SLIM_STATUS *slim,  - state variables
//     unsigned char *ch   - raw character to send
//
//  OUTPUTS:
//     none
//
//  RETURNS
//     SendComplete = 0  - SLIM package sent
//                   !0  - More data to send
//
// *************************************************************************
unsigned short GetNextSlimByte(SLIM_STATUS *slim, unsigned char _FAR *ch)
{
   // Check if sending an escape character sequence
   if (slim->escape)
   {
      switch (slim->escape)
      {
         case  END :                   // send 0xC0 as 0xDB 0xDC
            *ch = ESC_END;             // 0xDC, second of two escape byte
            break;
         case  ESC :                   // send 0xDB as 0xDB 0xDE
            *ch = ESC_ESC;             // 0xDE, second of two escape bytes
            break;
      }
      slim->escape = FALSE;
   }
   else
   {
      // get next character to send
      switch (slim->state)
      {
         // Send the SYN, STX, Type and Length parts of the SLIM package
         case  SendHeader :
            *ch = slim->u1.header_buf[slim->charCount++];
            if (slim->charCount == 6)
            {
               slim->charCount = 0;
               slim->state = SendData;
            }
            break;
         // Send the data part of the SLIM package
         case  SendData :
            *ch = slim->buffer[slim->charCount++];
            if (slim->charCount == slim->bufferSize)
            {
               slim->charCount = 0;
               if (slim->bufferSize%2)
               {
                  slim->state = SendPad;
               }
               else
               {
                  slim->state = SendTrailer;
               }
            }
            break;
         // Send the data part of the SLIM package if required
         case  SendPad :
            *ch = 0;
            slim->state = SendTrailer;
            break;
         // Send the Checksum and END parts of the SLIM package
         case  SendTrailer :
            *ch = slim->u2.trailer_buf[slim->charCount++];
            if (slim->charCount == 3)
            {
               slim->state = SendComplete;
               return  *ch;            // avoid escaping last character (ugly?)
            }
            break;
      }
      // setup for escape characters

      switch (*ch)
      {
         case  END :                   // send 0xC0 as 0xDB 0xDC
         case  ESC :                   // send 0xDB as 0xDB 0xDE
            slim->escape = *ch;
            *ch = ESC;                 // 0xDB, first of two escape bytes
      }
   }
   return  slim->state;
}


// *************************************************************************
//  NAME:
//     CheckESC
//
//  DESCRIPTION:
//     Looks for double byte escape sequences and resets escaped character to
//     its original value.   0xDB 0xDC => 0xC0     (end of packet)
//                           0xDB 0xDE => 0xDB     (escape character)
//                           0xDB 0xDF => 0xFD     (I2C acknowledge)
//
//  INPUTS:
//     SLIM_STATUS *slim,  - state variables
//     unsigned char *ch   - raw character to send
//
//  OUTPUTS:
//     Escaped character, *ch, reset to its pre-escape sequence value
//
//  RETURNS
//     0            0  (FALSE) for non-escape flag characters
//     ValidChar = !0  (TRUE) for escape flag and for escaped character
//
// *************************************************************************
unsigned int CheckESC(SLIM_STATUS *slim, unsigned char _FAR *ch)
{
   unsigned int ValidChar = TRUE;      // = !0

   // slim->escape = FALSE after InitializeSlim()
   if (slim->escape)
   {
      // The "else" stanza has detected an escape flag, so this time, the *ch character
      // is the byte that was "escaped".  Figure out what it is and reset *ch
      // accordingly.
      switch (*ch)
      {
         case (ESC_ESC) :         // 0xDE
            *ch = ESC;            // *ch = DB
            // DbgPrint("SLIM.C CheckESC() - ESC_ESC detected\n", *ch);
            break;

         case (ESC_END) :         // 0xDC
            *ch = END;            // *ch = C0
            // DbgPrint("SLIM.C CheckESC() - ESC_END detected\n", *ch);
            break;

         case (ESC_ACK) :         // 0xDF
            *ch = ACK;            // *ch = FD
            // DbgPrint("SLIM.C CheckESC() - ESC_ACK detected\n", *ch);
            break;

         default  :
            // set state to idle, escape to FALSE then process char
            slim->state = MsRcvStateIdle;

            KernalConsolePrint("\n\rSLIM.C ProcessChar bad ESC seq.");
            // DbgPrint("SLIM.C CheckESC(), bad ESC sequence\n", *ch);
            break;
      }
      slim->escape = FALSE;   // = 0
   }
   else
   {
      // Looking for any of three escape flags that signal an ESC, END or ACK in
      // the next byte.  If any of these 3 flags are detected, set the slim->escape
      // flag so the next time this routine is called, the code runs through the
      // above stanza rather than this one.
      switch (*ch)
      {
         case  ESC :              // 0xDB
            ValidChar = FALSE;
            slim->escape = TRUE;
            // DbgPrint("SLIM.C CheckESC() - Valid ESC detected\n", *ch);
            break;

         case  ACK :              // 0xFD    (host byte received) invalid i2c data
            ValidChar = FALSE;
            slim->state = MsRcvStateIdle;
            // DbgPrint("SLIM.C CheckESC() - Invalid ACK detected\n", *ch);
            break;

         case  END :              // 0xC0    only valid when expected
            if (slim->state != MsDataStateEnd)
            {
               // DbgPrint("SLIM.C CheckESC() - Invalid END detected\n", *ch);
               KernalConsolePrint("\n\rSLIM.C ProcessChar unexpected END");
               slim->state = MsRcvStateIdle;
               ValidChar = FALSE;
            }
            else
            {
               // DbgPrint("SLIM.C CheckESC() - Valid END detected\n", *ch);
            }
            break;
      }
   }
   return (ValidChar);
}
