/******************************************************************************
 *
 * Name:	skgihw.c
 * Project:	SysKonnect SK-9Dxx Gigabit Ethernet
 * Version:	$Revision: 1.11 $
 * Date:	$Date: 2001/10/23 09:53:38 $
 * Purpose:	HardWare Access Module
 *
 ******************************************************************************/

/******************************************************************************
 *
 *	(C)Copyright 2001 SysKonnect GmbH.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	The information in this file is provided "AS IS" without warranty.
 *
 ******************************************************************************/

/******************************************************************************
 *
 * History:
 *
 *	$Log: skgihw.c,v $
 *	Revision 1.11  2001/10/23 09:53:38  rschmidt
 *	Changed define for START_TIMEOUT() to avoid RedHat's GCC-Linker problem
 *	
 *	Revision 1.10  2001/09/27 11:36:52  jschmalz
 *	Added some SK_U64 casts in conjunction with SkOsGetTime and
 *	change of some structure-names: ProdIdx to ConProdIdx, Rx to ProdRx and
 *	Tx to ConTx
 *	
 *	Revision 1.9  2001/09/26 13:53:25  rschmidt
 *	Fixed timeouts during initialization.
 *	Eliminated old routines for SEEPROM.
 *	Editorial changes.
 *	
 *	Revision 1.8  2001/09/21 11:13:48  mmoser
 *	make module runnable with W2K
 *	
 *	Revision 1.7  2001/09/13 15:10:28  jschmalz
 *	Changed to new C-Style for common-modules
 *	
 *	Revision 1.6  2001/08/23 08:13:16  mlindner
 *	Fixed CurrentTime_us-Error with Linux
 *	
 *	Revision 1.5  2001/08/14 14:50:21  rschmidt
 *	Fixed copper link bug.
 *	Added copper 1000Base-T master/slave negotiation.
 *	Eliminated define for SK_JUMBO.
 *	Editorial changes.
 *	
 *	Revision 1.4  2001/06/13 15:11:21  rschmidt
 *	Added auto-negotiation for Fiber.
 *	
 *	Revision 1.3  2001/06/05 15:11:21  rassmann
 *	Changed initialization of pAC->MiscLocal.
 *	
 *	Revision 1.2  2001/06/05 08:49:14  rassmann
 *	Added SysKonnectFileId.
 *	
 *	Revision 1.1  2001/06/05 08:28:24  rassmann
 *	First public version.
 *
 *
 ******************************************************************************/

/******************************************************************************
 *
 * Description:
 *
 *	This is the hardware access module of the
 *	SysKonnect SK-9Dxx Gigabit Ethernet Linux driver.
 *
 ******************************************************************************/

#ifndef	lint
static const char SysKonnectFileId[] =
	"@(#) $Id: skgihw.c,v 1.11 2001/10/23 09:53:38 rschmidt Exp $ (C) SysKonnect.";
#endif	/* !defined(lint) */

#define __SKGIHW_C

#ifdef __cplusplus
extern "C" {
#endif	/* cplusplus */

#include	"h/skdrv1st.h"
#include	"h/skdrv2nd.h"

/* defines ******************************************************************/
/* For debuging on x86 only */
/* #define BREAKPOINT() asm(" int $3"); */

#define INITSTEP(x, s)
//	SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
//		("init step %d: %s\n", x, s));

#define SHUTDOWNSTEP(x, s)
//	SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
//		("shutdown step %d: %s\n", x, s));


/* function prototypes ********************************************************/
static SK_STATUS SkReadI2cDword(
	SK_AC	*pAC,		/* Pointer to Adapter Context */
	SK_IOC	IoC,		/* I/O contex */	
	SK_U8	DeviceID,	/* I2C Device */
	SK_U32	Offset,		/* Offset in the I2C*/
	SK_U32	*Data32 );	/* Data */

static SK_STATUS SkWriteI2cDword(
	SK_AC	*pAC,		/* Pointer to Adapter Context */
	SK_IOC	IoC,		/* I/O contex */	
	SK_U8	DeviceID,	/* I2C Device */
	SK_U32	Offset,		/* Offset in the I2C*/
	SK_U32	Data32 );	/* Data */

#ifdef __cplusplus
}
#endif	/* __cplusplus */

/* global variables *********************************************************/
/* None */


/* local variables **********************************************************/
/* None */


/*****************************************************************************
 *
 *	SkCheckTimeout - handle HWAC events
 *
 * Description:
 *	This routine waits duration time and prints a debugmessage on failure
 *
 * Returns: SK_STATUS
 *
 */
SK_STATUS SkCheckTimeout(
SK_AC		*pAC,	/* Pointer to Adapter Context */
SK_U64		StartTime,
SK_U64		Duration,
SK_U32		Step)
{
	SK_U64 AktTime;
	SK_U64 DiffTime;
	
	AktTime = START_TIMEOUT();

	if (AktTime >= StartTime) {
		DiffTime = AktTime - StartTime;
	}
	else {
		DiffTime = SK_CONSTU64(0xFFFFFFFFFFFFFFFF) - StartTime + AktTime;
	}
	
	if (DiffTime > Duration) {			
		SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT, ("Failure in step %d\n", Step));			
		return (SK_STATUS_EXPIRED);										
	}
	
	return (SK_STATUS_OK);										
}

/******************************************************************************
 *
 *	AutoNegTxConfig - send configuration data
 *
 * Description:
 *	Sends configuration data to the link partner through the
 *	Gigabit	Transmit 1000 BaseX auto-negotiation register
 *
 * Returns:	N/A
 *
 */
static void AutoNegTxConfig(
SK_AC	*pAC,	/* Pointer to Adapter Context */
SK_IOC	IoC)	/* I/O Context */
{
    SK_OUT32(IoC, TX_AUTONEG, (SK_U32)pAC->Hw.AutoNeg.TxConfig.AsUSHORT);

	/* Start sending config commands */
	pAC->Hw.MacMode |= SEND_CONFIGS;

	SK_OUT32(IoC, ETH_MAC_MODE, pAC->Hw.MacMode);
}	/* AutoNegTxConfig */


/******************************************************************************
 *
 *	AutoNegRxConfig - receive configuration data
 *
 * Description:
 *	Receives configuration data from the link partner through the
 *	Gigabit Receive 1000 BaseX auto-negotiation register
 *
 * Return:
 *		SK_STATUS_OK		- if auto-negotiation data was received
 *		SK_STATUS_FAILED 	- if not
 */
static SK_STATUS AutoNegRxConfig(
SK_AC	*pAC,	/* Pointer to Adapter Context */
SK_IOC	IoC,	/* I/O Context */
SK_U32	*pRxConfig)
{
    SK_U32	Data32;

	SK_IN32(IoC, ETH_MAC_STATUS, &Data32);

    if ((Data32 & RECEIVING_CFG) != 0) {
		SK_IN32(IoC, RX_AUTONEG, pRxConfig);
        return (SK_STATUS_OK);
    }

    return (SK_STATUS_FAILED);
}	/* AutoNegRxConfig */


/******************************************************************************
 *
 *	AutoNeg8023z - 1000 BaseX auto-negotiation as described in IEEE 802.3z
 *
 * Description:
 *	Exchange configuration data with the link partner through the Gigabit
 *	Transmit & Receive 1000 BaseX auto-negotiation registers
 *
 * Returns:
 *		SK_STATUS_OK
 *		SK_STATUS_DONE
 *		SK_STATUS_TIMER_ENABLED
 *		SK_STATUS_FAILED
 */
static SK_STATUS AutoNeg8023z(
SK_AC	*pAC,	/* Pointer to Adapter Context */
SK_IOC	IoC)	/* I/O Context */
{
    SK_U32	CurrentTime_us;
    SK_U32	Delta_us;
    SK_U32	RxConfig;
    SK_BOOL	AbilityMatch;
    SK_BOOL	IdleMatch;
    SK_BOOL	AckMatch;

    SK_STATUS			RetValue;
    P_AUTO_NEG_STATE	pAutoNeg;

	pAutoNeg = &pAC->Hw.AutoNeg;

    /* Get the current time */
    SkOsGetTimeUsec(pAC, &CurrentTime_us);

    if (pAutoNeg->State == AN_STATE_UNKNOWN) {
        pAutoNeg->IdleTime_us = CurrentTime_us;
        pAutoNeg->RxConfigTime_us = CurrentTime_us;
        pAutoNeg->AckTime_us = CurrentTime_us;
        pAutoNeg->RxConfig.AsUSHORT = 0;
    }

    /* Init the match flags */
    AbilityMatch = SK_FALSE;
    IdleMatch = SK_FALSE;
    AckMatch = SK_FALSE;

    /* Set the AbilityMatch, IdleMatch, and AckMatch flags if their */
    /* corresponding conditions are satisfied */
    if (SK_STAT_CMP_SUCCEEDED(AutoNegRxConfig(pAC, IoC, &RxConfig))) {

        pAutoNeg->ReceivingConfigs = SK_TRUE;

        /* Reset RxConfigTime_us if this Config is different than the */
        /* previous one.  Also disable the AckTime_us */
        if (((SK_U32)(pAutoNeg->RxConfig.AsUSHORT & ~AN_CONFIG_ACK)) !=
            ((SK_U32)(RxConfig & ~AN_CONFIG_ACK))) {
            pAutoNeg->RxConfigTime_us = CurrentTime_us;

            /* Don't start the ACK timer yet */
            pAutoNeg->AckTime_us = 0;
        }
        else {
            /* Save the AckTime_us for the first time if the ACK bit is set */
            if (RxConfig & AN_CONFIG_ACK) {
                if (pAutoNeg->AckTime_us == 0) {
                    pAutoNeg->AckTime_us = CurrentTime_us;
                }
            }
            else {
                /* ACK bit not set, don't start the ACK timer yet */
                pAutoNeg->AckTime_us = 0;
            }

            /* Compute the AbilityMatch time period */
            if (CurrentTime_us >= pAutoNeg->RxConfigTime_us) {
                Delta_us = CurrentTime_us - pAutoNeg->RxConfigTime_us;
            }
            else {
                Delta_us = 0xffffffff - pAutoNeg->RxConfigTime_us +
                    CurrentTime_us + 1;
            }

            /* Set the AbilityMatch flag if we have received "three" */
            /* consecutive Configs that are the same */
            if (Delta_us >= AN_THREE_RX_CONFIG_TIME) {
                AbilityMatch = SK_TRUE;

                if (pAutoNeg->AckTime_us) {
                    /* Compute the AckMatch time period */
                    if (CurrentTime_us >= pAutoNeg->AckTime_us) {
                        Delta_us = CurrentTime_us - pAutoNeg->AckTime_us;
                    }
                    else {
                        Delta_us = 0xffffffff - pAutoNeg->AckTime_us +
                            CurrentTime_us + 1;
                    }

                    /* Set the AckMatch flag if we have received "three" */
                    /* consecutive Configs that are the same with the ACK */
                    /* bit set */
                    if (Delta_us >= AN_THREE_RX_CONFIG_TIME) {
                        AckMatch = SK_TRUE;
                    }
                }
            }
        }
    }
    else {
        /* Save the current time, if we are previously receiving configs */
        /* and now are receiving IDLEs.  We need this time to compute how */
        /* we have been receiving IDLEs */
        if (pAutoNeg->ReceivingConfigs) {
            pAutoNeg->IdleTime_us = CurrentTime_us;
            pAutoNeg->ReceivingConfigs = SK_FALSE;
        }

        /* Compute the idle time period */
        if (CurrentTime_us >= pAutoNeg->IdleTime_us) {
            Delta_us = CurrentTime_us - pAutoNeg->IdleTime_us;
        }
        else {
            Delta_us = 0xffffffff - pAutoNeg->IdleTime_us + CurrentTime_us + 1;
        }

        /* Set the idle_match flag if we have received "three" IDLES */
        if (Delta_us >= AN_THREE_RX_CONFIG_TIME) {
            IdleMatch = SK_TRUE;
        }
        else if (Delta_us > AN_NO_RX_CONFIG_TIME_OUT) {
            SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_ERR,
				("Not receive Configs\n"));
            return (SK_STATUS_FAILED);
        }
    }

    /* Save the last Config */
    pAutoNeg->RxConfig.AsUSHORT = (SK_U16)RxConfig;

    /* Default return code */
    RetValue = SK_STATUS_WORKING;

    /* Auto-negotiation state machine as defined in 802.3z section 37.3.1.5 */
    switch (pAutoNeg->State) {
        case AN_STATE_UNKNOWN:
            if (pAutoNeg->mr_an_enable || pAutoNeg->mr_restart_an) {
                pAutoNeg->State = AN_STATE_AN_ENABLE;
            }
            break;

        case AN_STATE_AN_ENABLE:
            pAutoNeg->mr_an_complete = SK_FALSE;
            pAutoNeg->mr_page_rx = SK_FALSE;

            if (pAutoNeg->mr_an_enable) {

                pAutoNeg->TxConfig.AsUSHORT = 0;
				AutoNegTxConfig(pAC, IoC);

                pAutoNeg->State = AN_STATE_AN_RESTART_INIT;
            }
            else {
                pAutoNeg->State = AN_STATE_DISABLE_LINK_OK;
            }
            break;

        case AN_STATE_AN_RESTART_INIT:
            pAutoNeg->LinkTime_us = CurrentTime_us;
            pAutoNeg->mr_np_loaded = SK_FALSE;

            pAutoNeg->TxConfig.AsUSHORT = 0;
            AutoNegTxConfig(pAC, IoC);

            RetValue = SK_STATUS_TIMER_ENABLED;

            pAutoNeg->State = AN_STATE_AN_RESTART;
            break;

        case AN_STATE_AN_RESTART:
            /* Get the current time and compute the delta */
            if (CurrentTime_us > pAutoNeg->LinkTime_us) {
                Delta_us = CurrentTime_us - pAutoNeg->LinkTime_us;
            }
            else {
                Delta_us = 0xffffffff - pAutoNeg->LinkTime_us + CurrentTime_us+1;
            }

            if (Delta_us > AN_LINK_TIMER_INTERVAL_US) {
#ifdef DEBUG
                if (Delta_us > 2*AN_LINK_TIMER_INTERVAL_US) {
                    SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_ERR,
						("Timer out of tolerance (%d, %d)\n",
                        pAutoNeg->LinkTime_us, CurrentTime_us));

                    RetValue = SK_STATUS_FAILED;
                    break;
                }
#endif
                pAutoNeg->State = AN_STATE_ABILITY_DETECT_INIT;
            }
            else {
                RetValue = SK_STATUS_TIMER_ENABLED;
            }
            break;

        case AN_STATE_DISABLE_LINK_OK:
            RetValue = SK_STATUS_OK;
            break;

        case AN_STATE_ABILITY_DETECT_INIT:
            /* Note: in the state diagram, this variable is set to */
            /* mr_adv_ability<12>.  Is this right? */
            pAutoNeg->mr_toggle_tx = SK_FALSE;

#ifdef DEBUG
            SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT, ("TxConfig: "));

            if (pAutoNeg->mr_adv_full_duplex) {
                SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT, ("FD "));
            }

            if (pAutoNeg->mr_adv_half_duplex) {
                SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT, ("HD "));
            }

            if (pAutoNeg->mr_adv_sym_pause) {
                SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT, ("PS1 "));
            }

            if (pAutoNeg->mr_adv_asym_pause) {
                SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT, ("PS2 "));
            }

            if (pAutoNeg->mr_adv_remote_fault1) {
                SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT, ("RF1 "));
            }

            if (pAutoNeg->mr_adv_remote_fault2) {
                SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT, ("RF2 "));
            }

            if (pAutoNeg->mr_adv_next_page) {
                SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT, ("NP "));
            }

            SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT, ("\n"));
#endif

            /* Send the config as advertised in the advertisement register */
            pAutoNeg->TxConfig.AsUSHORT = 0;
            pAutoNeg->TxConfig.D5_FD = pAutoNeg->mr_adv_full_duplex;
            pAutoNeg->TxConfig.D6_HD = pAutoNeg->mr_adv_half_duplex;
            pAutoNeg->TxConfig.D7_PS1 = pAutoNeg->mr_adv_sym_pause;
            pAutoNeg->TxConfig.D8_PS2 = pAutoNeg->mr_adv_asym_pause;
            pAutoNeg->TxConfig.D12_RF1 = pAutoNeg->mr_adv_remote_fault1;
            pAutoNeg->TxConfig.D13_RF2 = pAutoNeg->mr_adv_remote_fault2;
            pAutoNeg->TxConfig.D15_NP = pAutoNeg->mr_adv_next_page;

            AutoNegTxConfig(pAC, IoC);

            pAutoNeg->State = AN_STATE_ABILITY_DETECT;
            break;

        case AN_STATE_ABILITY_DETECT:
            if (AbilityMatch && pAutoNeg->RxConfig.AsUSHORT != 0) {

                pAutoNeg->AbilityMatchConfig =
                    pAutoNeg->RxConfig.AsUSHORT & ~AN_CONFIG_ACK;

                pAutoNeg->State = AN_STATE_ACK_DETECT_INIT;
            }
            break;

        case AN_STATE_ACK_DETECT_INIT:
            pAutoNeg->TxConfig.D14_ACK = 1;
            AutoNegTxConfig(pAC, IoC);

            pAutoNeg->State = AN_STATE_ACK_DETECT;
            break;

        case AN_STATE_ACK_DETECT:
            if (AckMatch) {
                if ((pAutoNeg->RxConfig.AsUSHORT & ~AN_CONFIG_ACK) ==
                    pAutoNeg->AbilityMatchConfig) {
                    pAutoNeg->State = AN_STATE_COMPLETE_ACK_INIT;
                }
                else {
                    SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_ERR,
						("1 ACK_DETECT ===> AN_ENABLE\n"));
                    pAutoNeg->State = AN_STATE_AN_ENABLE;
                }
            }
            else if (AbilityMatch && pAutoNeg->RxConfig.AsUSHORT == 0) {
                SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_ERR,
					("2 ACK_DETECT ===> AN_ENABLE\n"));
                pAutoNeg->State = AN_STATE_AN_ENABLE;
            }
            break;

        case AN_STATE_COMPLETE_ACK_INIT:
            /* Make sure invalid bits are not set */
            if (pAutoNeg->RxConfig.bits.D0 || pAutoNeg->RxConfig.bits.D1 ||
                pAutoNeg->RxConfig.bits.D2 || pAutoNeg->RxConfig.bits.D3 ||
                pAutoNeg->RxConfig.bits.D4 || pAutoNeg->RxConfig.bits.D9 ||
                pAutoNeg->RxConfig.bits.D10 || pAutoNeg->RxConfig.bits.D11) {
                SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_ERR,
					("Received an invalid Config\n"));
                RetValue = SK_STATUS_FAILED;
                break;
            }

            /* Set up the link partner advertisement register */
            pAutoNeg->mr_lp_adv_full_duplex = pAutoNeg->RxConfig.D5_FD;
            pAutoNeg->mr_lp_adv_half_duplex = pAutoNeg->RxConfig.D6_HD;
            pAutoNeg->mr_lp_adv_sym_pause = pAutoNeg->RxConfig.D7_PS1;
            pAutoNeg->mr_lp_adv_asym_pause = pAutoNeg->RxConfig.D8_PS2;
            pAutoNeg->mr_lp_adv_remote_fault1 = pAutoNeg->RxConfig.D12_RF1;
            pAutoNeg->mr_lp_adv_remote_fault2 = pAutoNeg->RxConfig.D13_RF2;
            pAutoNeg->mr_lp_adv_next_page = pAutoNeg->RxConfig.D15_NP;

#ifdef DEBUG
            SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT, ("RxConfig: "));

            if (pAutoNeg->mr_lp_adv_full_duplex) {
                SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT, ("FD "));
            }

            if (pAutoNeg->mr_lp_adv_half_duplex) {
                SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT, ("HD "));
            }

            if (pAutoNeg->mr_lp_adv_sym_pause) {
                SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT, ("PS1 "));
            }

            if (pAutoNeg->mr_lp_adv_asym_pause) {
                SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT, ("PS2 "));
            }

            if (pAutoNeg->mr_lp_adv_remote_fault1) {
                SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT, ("RF1 "));
            }

            if (pAutoNeg->mr_lp_adv_remote_fault2) {
                SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT, ("RF2 "));
            }

            if (pAutoNeg->mr_lp_adv_next_page) {
                SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT, ("NP "));
            }

            SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT, ("\n"));
#endif
            pAutoNeg->LinkTime_us = CurrentTime_us;

            pAutoNeg->mr_toggle_tx = !pAutoNeg->mr_toggle_tx;
            pAutoNeg->mr_toggle_rx = pAutoNeg->RxConfig.bits.D11;
            pAutoNeg->mr_np_rx = pAutoNeg->RxConfig.D15_NP;
            pAutoNeg->mr_page_rx = SK_TRUE;

            pAutoNeg->State = AN_STATE_COMPLETE_ACK;
            RetValue = SK_STATUS_TIMER_ENABLED;

            break;

        case AN_STATE_COMPLETE_ACK:
            if (AbilityMatch && pAutoNeg->RxConfig.AsUSHORT == 0) {
                SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_ERR,
					("COMPLETE_ACK ===> AN_ENABLE\n"));
                pAutoNeg->State = AN_STATE_AN_ENABLE;
                break;
            }

            if (CurrentTime_us > pAutoNeg->LinkTime_us) {
                Delta_us = CurrentTime_us - pAutoNeg->LinkTime_us;
            }
            else {
                Delta_us = 0xffffffff - pAutoNeg->LinkTime_us +
					CurrentTime_us + 1;
            }

            if (Delta_us > AN_LINK_TIMER_INTERVAL_US) {
#ifdef DEBUG
                if (Delta_us > 2*AN_LINK_TIMER_INTERVAL_US) {
                    SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_ERR,
						("Timer out of tolerance (%d, %d)\n",
                        pAutoNeg->LinkTime_us, CurrentTime_us));

                    RetValue = SK_STATUS_FAILED;
                    break;
                }
#endif

                if (pAutoNeg->mr_adv_next_page == 0 ||
                    pAutoNeg->mr_lp_adv_next_page == 0) {
                    pAutoNeg->State = AN_STATE_IDLE_DETECT_INIT;
                }
                else {
                    if (pAutoNeg->TxConfig.bits.D15 == 0 &&
                        pAutoNeg->mr_np_rx == 0) {
                        pAutoNeg->State = AN_STATE_IDLE_DETECT_INIT;
                    }
                    else {
                        SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_ERR,
							("Next page not implemented\n"));
                        RetValue = SK_STATUS_FAILED;
                    }
                }
            }
            break;

        case AN_STATE_IDLE_DETECT_INIT:
            pAutoNeg->LinkTime_us = CurrentTime_us;

			/* Stop sending config commands */
			pAC->Hw.MacMode &= ~SEND_CONFIGS;

			SK_OUT32(IoC, ETH_MAC_MODE, pAC->Hw.MacMode);

			pAutoNeg->State = AN_STATE_IDLE_DETECT;

            RetValue = SK_STATUS_TIMER_ENABLED;
            break;

        case AN_STATE_IDLE_DETECT:
            if (AbilityMatch && pAutoNeg->RxConfig.AsUSHORT == 0) {
                SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_ERR,
					("IDLE_DETECT ===> AN_ENABLE\n"));
                pAutoNeg->State = AN_STATE_AN_ENABLE;
                break;
            }

            if (CurrentTime_us > pAutoNeg->LinkTime_us) {
                Delta_us = CurrentTime_us - pAutoNeg->LinkTime_us;
            }
            else {
                Delta_us = 0xffffffff - pAutoNeg->LinkTime_us + CurrentTime_us+1;
            }

            if (Delta_us > AN_LINK_TIMER_INTERVAL_US) {
#ifdef DEBUG
                if (Delta_us > 2*AN_LINK_TIMER_INTERVAL_US) {
                    SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_ERR,
						("Timer out of tolerance (%d, %d)\n",
                        pAutoNeg->LinkTime_us, CurrentTime_us));

                    RetValue = SK_STATUS_FAILED;
                    break;
                }
#endif

                if (IdleMatch) {
                    pAutoNeg->State = AN_STATE_LINK_OK;
                }
                else {
                    SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_ERR,
						("Auto-negotiation failed in IDLE_DETECT\n"));
                    RetValue = SK_STATUS_FAILED;
                    break;
                }
            }
            break;

        case AN_STATE_LINK_OK:
            pAutoNeg->mr_an_complete = SK_TRUE;
            pAutoNeg->mr_link_ok = SK_TRUE;
            RetValue = SK_STATUS_OK;
            break;

        case AN_STATE_NEXT_PAGE_WAIT_INIT:
            SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_ERR,
				("Not implemented\n"));
            break;

        case AN_STATE_NEXT_PAGE_WAIT:
            SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_ERR,
				("Not implemented\n"));
            break;

        default:
            SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_ERR,
				("Invalid AN state\n"));
            RetValue = SK_STATUS_FAILED;
            break;
    }

    return (RetValue);
}	/* AutoNeg8023z */

/******************************************************************************
 *
 *  SkInitAutoI2cAccess - Enables Auto Access and Reset I2C Statemachine
 *
 * Description:
 *  This routine sets the Bit for Auto access in the MISC-Local Register
 *	and resets the FSM for the I2C access
 *
 * Notes:
 *
 * Context:
 *
 * Returns: SK_STATUS
 *
 */
SK_STATUS SkInitAutoI2cAccess(
SK_AC	*pAC,	/* Pointer to Adapter Context */
SK_IOC	IoC)	/* I/O contex */	
{
	SK_U32	Value32;
	SK_U32	AddrReg;
	SK_U64	StartTime;
						
	SK_IN32(IoC, MISC_LOCAL ,&Value32);
	SK_OUT32(IoC, MISC_LOCAL,  Value32 | AUTO_SEEPROM_ACC);

	AddrReg = EEPROM_A_READ | EEPROM_A_FSM_RESET | EEPROM_A_CLK;
	SK_OUT32(IoC, EEPROM_ADDR, AddrReg);
	
    StartTime = START_TIMEOUT();
	do {
		SK_IN32(IoC, EEPROM_ADDR, &AddrReg);
		if (TIMEOUT_REACHED(StartTime, 500, 0)) { /* 5 seconds */	
			return (SK_STATUS_FAILED);
		}
	} while ((AddrReg & EEPROM_A_FSM_RESET) != 0);
	
	return (SK_STATUS_OK);
}	/* SkInitAutoI2cAccess */

/******************************************************************************
 *
 *  SkReadI2cDword - Reads a Dword from a I2C EEPROM Device
 *
 * Description:
 *	reads a 32-bit dword (4 bytes) from an aligned address
 *
 * Notes:
 *
 * Context:
 *
 * Returns: SK_STATUS
 *
 */
SK_STATUS SkReadI2cDword(
SK_AC	*pAC,	/* Pointer to Adapter Context */
SK_IOC	IoC,		/* I/O contex */	
SK_U8	DeviceID,	/* I2C Device */
SK_U32	Offset,		/* Offset in the I2C*/
SK_U32	*Data32 )	/* Data */
{
	SK_U32	DeviceMask;
	SK_U32	AddrReg;
	SK_U64	StartTime;

	DeviceMask = EEPROM_A_DEV_ID((SK_U32) DeviceID);
	DeviceMask &= EEPROM_A_DEV_MASK;
	
	Offset &= 0xfffffffc;		/* 4 Byte alignment */

	AddrReg =	DeviceMask |
				EEPROM_A_READ |
				EEPROM_A_CLK |
				EEPROM_A_ADDR(Offset) |
				EEPROM_A_START |
				EEPROM_A_COMPL;
	SK_OUT32(IoC, EEPROM_ADDR, AddrReg);

    StartTime = START_TIMEOUT();
	do {
		SK_IN32(IoC, EEPROM_ADDR, &AddrReg);
		if (TIMEOUT_REACHED(StartTime, 500, 0)) { /* 5 seconds */	
			return (SK_STATUS_FAILED);
		}
	} while ((AddrReg & EEPROM_A_COMPL) == 0);

	SK_IN32(IoC, EEPROM_DATA, Data32);
	
	return (SK_STATUS_OK);
}	/* SkReadI2cDword */

/******************************************************************************
 *
 *  SkWriteI2cDword - Writes a Dword into a I2C EEPROM Device
 *
 * Description:
 *	writes a 32-bit dword (4 bytes) to an aligned address
 *
 * Notes:
 *
 * Context:
 *
 * Returns: SK_STATUS
 *
 */
SK_STATUS SkWriteI2cDword(
SK_AC	*pAC,	/* Pointer to Adapter Context */
SK_IOC	IoC,		/* I/O contex */	
SK_U8	DeviceID,	/* I2C Device */
SK_U32	Offset,		/* Offset in the I2C*/
SK_U32	Data32 )	/* Data */
{
	SK_U32	DeviceMask;
	SK_U32	AddrReg;
	SK_U64	StartTime;

	DeviceMask = EEPROM_A_DEV_ID((SK_U32) DeviceID);
	DeviceMask &= EEPROM_A_DEV_MASK;
	
	Offset &= 0xfffffffc;		/* 4 Byte alignment */

	SK_OUT32(IoC, EEPROM_DATA, Data32);

	AddrReg =	DeviceMask |
				EEPROM_A_WRITE |
				EEPROM_A_CLK |
				EEPROM_A_ADDR(Offset) |
				EEPROM_A_START |
				EEPROM_A_COMPL;
	SK_OUT32(IoC, EEPROM_ADDR, AddrReg);
    StartTime = START_TIMEOUT();
	do {
		SK_IN32(IoC, EEPROM_ADDR, &AddrReg);
		if (TIMEOUT_REACHED(StartTime, 500, 0)) { /* 5 seconds */	
			return (SK_STATUS_FAILED);
		}
	}while ((AddrReg & EEPROM_A_COMPL) == 0);

	return (SK_STATUS_OK);
}	/* SkWriteI2cDword */

/******************************************************************************
 *
 *  SkReadI2cBlock - Reads a block from a I2C Device
 *
 * Description:
 *  This routine reads a block of memory at offset from the I2C-Bus device
 *
 * Notes:
 *
 * Context:
 *
 * Returns: SK_STATUS
 *
 */
SK_STATUS SkReadI2cBlock(
SK_AC		*pAC,		/* Adapter contex */	
SK_IOC	IoC,		/* I/O contex */	
SK_U8		DeviceId,	/* I2C Device */
SK_U32	Offset,		/* Offset in the I2C*/
SK_U32	Amount,		/* Number of bytes */
SK_U8		*Data )		/* Data array */
{
	SK_U32	Value32;
	SK_U32	StartOffset;
	SK_U32	StartRest;
	SK_U32	Rest;
	SK_U8	*pData;

	if(Amount == 0)
		return (SK_STATUS_OK);

#if 0
	if(Amount > pAC->HwDiag.I2CDeviceSize[DeviceId] ||
	   Offset + Amount > pAC->HwDiag.I2CDeviceSize[DeviceId])
		return (SK_STATUS_FAILED);
#endif
	
	pData = Data;

	/* ----------------------------------------------------------------------------
	 * Copy the odd bytes of a 4 byte alignment
	 * ----------------------------------------------------------------------------
	 */
	StartOffset = Offset & 0xfffffffc;		/* 4 Byte alignment */

	if (SK_STAT_CMP_FAILED(SkReadI2cDword( pAC, IoC, DeviceId, StartOffset, &Value32))) {
		return (SK_STATUS_FAILED);
	}

	Rest = Amount;
	StartRest = 4 - (Offset % 4);

	if (StartRest > 3) {
		*(pData++) = (SK_U8)((Value32 >>  0) & 0xff);
		Rest--;
	}
	if (StartRest > 2 && Amount > 1) {
		*(pData++) = (SK_U8)((Value32 >>  8) & 0xff);
		Rest--;
	}
	if (StartRest > 1 && Amount > 2) {
		*(pData++) = (SK_U8)((Value32 >> 16) & 0xff);
		Rest--;
	}
	if (StartRest > 0 && Amount > 3) {
		*(pData++) = (SK_U8)((Value32 >> 24) & 0xff);
		Rest--;
	}

	StartOffset += 4; /* next 4 Bytes */

	/* ----------------------------------------------------------------------------
	 * Copy the rest bytes in a loop
	 * ----------------------------------------------------------------------------
	 */
	while (Rest > 0) {
		
		if (SK_STAT_CMP_FAILED(SkReadI2cDword(pAC, IoC, DeviceId, StartOffset, &Value32))) {
			return (SK_STATUS_FAILED);
		}
		
		*(pData++) = (SK_U8)((Value32 >>  0) & 0xff);
		if (--Rest == 0)
			break;
		*(pData++) = (SK_U8)((Value32 >>  8) & 0xff);
		if (--Rest == 0)
			break;
		*(pData++) = (SK_U8)((Value32 >> 16) & 0xff);
		if (--Rest == 0)
			break;
		*(pData++) = (SK_U8)((Value32 >> 24) & 0xff);
		if (--Rest == 0)
			break;
		StartOffset += 4; /* next 4 bytes */
	}

	return (SK_STATUS_OK);
}	/* SkReadI2cBlock */

/******************************************************************************
 *
 *  SkWriteI2cBlock - Writes a block into a I2C Device
 *
 * Description:
 *  This routine writes a block of memory at offset into the I2C device
 *
 * Notes:
 *
 * Context:
 *
 * Returns: SK_STATUS
 *
 */
SK_STATUS SkWriteI2cBlock(
SK_AC		*pAC,		/* Adapter contex */
SK_IOC	IoC,		/* I/O contex */	
SK_U8		DeviceId,	/* I2C Device */
SK_U32	Offset,		/* Offset in the I2C*/
SK_U32	Amount,		/* Number of bytes */
SK_U8		*Data )		/* Data array */
{
	SK_U32	Value32;
	SK_U32	Rest;
	SK_U32	StartOffset;
	SK_U32	StartRest;
	SK_U8	*pData;

	if (Amount == 0)
    	return (SK_STATUS_OK);

#if 0
	if (Amount > pAC->HwDiag.I2CDeviceSize[DeviceId] ||
	    Offset + Amount > pAC->HwDiag.I2CDeviceSize[DeviceId])
    	return (SK_STATUS_FAILED);
#endif
	
	pData = Data;

	/* ----------------------------------------------------------------------------
	 * Copy the odd bytes of a 4 byte alignment
	 * ----------------------------------------------------------------------------
	 */
	StartOffset = Offset & 0xfffffffc;		/* 4 Byte alignment */

	if (SK_STAT_CMP_FAILED(SkReadI2cDword(pAC, IoC, DeviceId, StartOffset, &Value32))) {
		return (SK_STATUS_FAILED);
	}
	
	Rest = Amount;

	StartRest = 4 - (Offset % 4);

	if (StartRest > 3) {
		Value32 &= 0xffffff00;
		Value32 |= (SK_U32)(*(pData++) <<  0);
		Rest--;
	}
	if (StartRest > 2 && Amount > 1) {
		Value32 &= 0xffff00ff;
		Value32 |= (SK_U32)(*(pData++) <<  8);
		Rest--;
	}
	if (StartRest > 1 && Amount > 2) {
		Value32 &= 0xff00ffff;
		Value32 |= (SK_U32)(*(pData++) << 16);
		Rest--;
	}
	if (StartRest > 0 && Amount > 3) {
		Value32 &= 0x00ffffff;
		Value32 |= (SK_U32)(*(pData++) << 24);
		Rest--;
	}

	if (SK_STAT_CMP_FAILED(SkWriteI2cDword(pAC, IoC, DeviceId, StartOffset, Value32))) {
		return (SK_STATUS_FAILED);
	}
	
	StartOffset += 4; /* next 4 Bytes */

	/* ----------------------------------------------------------------------------
	 * Copy rest of full 4 bytes
	 * ----------------------------------------------------------------------------
	 */
	while ((Rest / 4) > 0) {
        Value32  = (SK_U32)(*(pData++) <<  0);
		Value32 |= (SK_U32)(*(pData++) <<  8);
		Value32 |= (SK_U32)(*(pData++) << 16);
		Value32 |= (SK_U32)(*(pData++) << 24);

		if (SK_STAT_CMP_FAILED(SkWriteI2cDword(pAC, IoC, DeviceId, StartOffset, Value32))) {
			return (SK_STATUS_FAILED);
		}
		
		StartOffset += 4; /* next 4 bytes */
		Rest -= 4;
	}

	/* ----------------------------------------------------------------------------
	 * Copy rest
	 * ----------------------------------------------------------------------------
	 */
	if (Rest > 0) {
		
		if (SK_STAT_CMP_FAILED(SkReadI2cDword(pAC, IoC, DeviceId, StartOffset, &Value32))) {
			return (SK_STATUS_FAILED);
		}
		if (Rest > 0) {
			Value32 &= 0xffffff00;
			Value32 |= (SK_U32)(*(pData++) <<  0);
			Rest--;
		}
		if (Rest > 0) {
			Value32 &= 0xffff00ff;
			Value32 |= (SK_U32)(*(pData++) <<  8);
			Rest--;
		}
		if (Rest > 0) {
			Value32 &= 0xff00ffff;
			Value32 |= (SK_U32)(*(pData++) << 16);
			Rest--;
		}
		if (SK_STAT_CMP_FAILED(SkWriteI2cDword(pAC, IoC, DeviceId, StartOffset, Value32))) {
			return (SK_STATUS_FAILED);
		}
	}
	return (SK_STATUS_OK);
}	/* SkWriteI2cBlock */


/*****************************************************************************
 *
 * 	SkGiReadPhy - read copper PHY
 *
 * Description:
 *	reads a 16-bit word from copper PHY through MI Communication register
 *
 * Returns:	SK_STATUS
 *
 */
SK_STATUS SkGiReadPhy(
SK_AC	*pAC,		/* Pointer to Adapter Context */
SK_IOC	IoC,		/* I/O Context */
SK_U16	Addr,		/* Register Address (Offset) */
SK_U16	*pData16)	/* Data */
{
	SK_U32	Data32;
	int		i;
	
	SK_OUT32(IoC, MI_COMMUNICATION, MI_START | MI_CMD_READ |
		MI_PHY_ADDR(MDI_PHY_ADDR) | MI_REG_ADDR(Addr));
	
	for (i = 0; i < 5000; i++) {
		SkOsWaitUsec(pAC, 10);
		SK_IN32(IoC, MI_COMMUNICATION, &Data32);
		if ((Data32 & MI_START) == 0) {
			*pData16 = (SK_U16)(Data32 & 0xffff);
			return (SK_STATUS_OK);
		}
	}
	
	SK_ERR_LOG(pAC, SK_ERRCL_HW, SKERR_HWI_E013, SKERR_HWI_E013MSG);
	return (SK_STATUS_FAILED);
}	/* SkGiReadPhy */

/*****************************************************************************
 *
 * 	SkGiWritePhy - write copper PHY
 *
 * Description:
 *	writes a 16-bit word to copper PHY through MI Communication register
 *
 * Returns:	SK_STATUS
 *
 */
SK_STATUS SkGiWritePhy(
SK_AC	*pAC,	/* Pointer to Adapter Context */
SK_IOC	IoC,	/* I/O Context */
SK_U16	Addr,	/* Register Address (Offset) */
SK_U16	Data16)	/* Data */
{
	SK_U32	Data32;
	int		i;
	
	SK_OUT32(IoC, MI_COMMUNICATION, MI_START | MI_CMD_WRITE |
		MI_PHY_ADDR(MDI_PHY_ADDR) | MI_REG_ADDR(Addr) | Data16);
	
	for (i = 0; i < 5000; i++) {
		SkOsWaitUsec(pAC, 10);
		SK_IN32(IoC, MI_COMMUNICATION, &Data32);
		if ((Data32 & MI_START) == 0) {
			return (SK_STATUS_OK);
		}
	}
	
	SK_ERR_LOG(pAC, SK_ERRCL_HW, SKERR_HWI_E014, SKERR_HWI_E014MSG);
	return (SK_STATUS_FAILED);
}	/* SkGiWritePhy */

/******************************************************************************
 *
 *	SetFlowControl
 *
 * Description:
 *	Set flow control
 *
 * Returns:	N/A
 *
 */
static void SetFlowControl(
SK_AC	*pAC,				/* Pointer to Adapter Context */
SK_IOC	IoC,				/* I/O Context */
SK_U16	LocalPhyAd,
SK_U16	RemotePhyAd)
{
    SK_U16	FlowCap;

	SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
		("SetFlowControl, Loc=0x%04X, Rem=0x%04X\n",
		 LocalPhyAd, RemotePhyAd));

#ifdef xDEBUG
	if ((LocalPhyAd & AD_PAUSE_CAPABLE) != 0) {
		SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
			("   PAUSE capable\n"));
	}

	if ((LocalPhyAd & AD_ASYM_PAUSE) != 0) {
		SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
			("   ASYM_PAUSE capable\n"));
	}

	SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
		("Remote flow control settings\n"));

	if ((RemotePhyAd & PHY_LINK_PARTNER_PAUSE_CAPABLE) != 0) {
		SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
			("   PAUSE capable\n"));
	}

	if ((RemotePhyAd & PHY_LINK_PARTNER_ASYM_PAUSE) != 0) {
		SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
			("   ASYM_PAUSE capable\n"));
	}
#endif /* DEBUG */

    /* Resolve flow control */
    FlowCap = FLOW_CONTROL_NONE;

	/* See Table 28B-3 of 802.3ab-1999 spec. */
	if ((pAC->Hw.FlowControl & FLOW_CONTROL_AUTO_PAUSE) != 0) {
		if ((LocalPhyAd & AD_PAUSE_CAPABLE) != 0) {
			if ((LocalPhyAd & AD_ASYM_PAUSE) != 0) {
				if ((RemotePhyAd & PAUSE_CAPABLE) != 0) {
					SK_DBG_MSG(pAC, SK_DBGMOD_HWAC,
						SK_DBGCAT_INIT, ("FlowCap: tx/rx\n"));
					FlowCap = FLOW_CONTROL_TRANSMIT_PAUSE |
						FLOW_CONTROL_RECEIVE_PAUSE;
				}
				else if ((RemotePhyAd & ASYM_PAUSE) != 0) {
					SK_DBG_MSG(pAC, SK_DBGMOD_HWAC,
						SK_DBGCAT_INIT,
						("FlowCap: rx PAUSE\n"));
					FlowCap = FLOW_CONTROL_RECEIVE_PAUSE;
				}
			}
			else {
				if ((RemotePhyAd & PAUSE_CAPABLE) != 0) {
					SK_DBG_MSG(pAC, SK_DBGMOD_HWAC,
						SK_DBGCAT_INIT, ("FlowCap: tx/rx\n"));
					FlowCap = FLOW_CONTROL_TRANSMIT_PAUSE |
						FLOW_CONTROL_RECEIVE_PAUSE;
				}
			}
		}
		else if ((LocalPhyAd & AD_ASYM_PAUSE) != 0) {
			if ((RemotePhyAd & PAUSE_CAPABLE) != 0 &&
				(RemotePhyAd & ASYM_PAUSE) != 0) {

				SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
					("FlowCap: tx PAUSE\n"));
				FlowCap = FLOW_CONTROL_TRANSMIT_PAUSE;
			}
		}
	}
	else {
		FlowCap = pAC->Hw.FlowControl;
	}
	
	/* Enable/disable rx PAUSE */
	pAC->Addr.Port.RxMode &= ~RX_ENAB_FLOW_CTRL;
	if ((FlowCap & FLOW_CONTROL_RECEIVE_PAUSE) != 0 &&
		(pAC->Hw.FlowControl == FLOW_CONTROL_AUTO_PAUSE ||
		(pAC->Hw.FlowControl & FLOW_CONTROL_RECEIVE_PAUSE) != 0)) {

		pAC->Addr.Port.RxMode |= RX_ENAB_FLOW_CTRL;

		SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
			("Enable rx PAUSE\n"));
	}
	SK_OUT32(IoC, RECEIVE_MODE, pAC->Addr.Port.RxMode);

	/* Enable/disable tx PAUSE */
	pAC->Addr.Port.TxMode &= ~TX_ENAB_FLOW_CTRL;

	if ((FlowCap & FLOW_CONTROL_TRANSMIT_PAUSE) != 0 &&
		(pAC->Hw.FlowControl == FLOW_CONTROL_AUTO_PAUSE ||
		(pAC->Hw.FlowControl & FLOW_CONTROL_TRANSMIT_PAUSE) != 0)) {

		pAC->Addr.Port.TxMode |= TX_ENAB_FLOW_CTRL;

		SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
			("Enable tx PAUSE\n"));
	}
	SK_OUT32(IoC, TRANSMIT_MODE, pAC->Addr.Port.TxMode);

}	/* SetFlowControl */

/******************************************************************************
 * Description:
 *
 * Return:
 *
 */
static void GetPhyAdFlowCtrlSets(
SK_AC	*pAC,				/* Pointer to Adapter Context */
SK_U16	*pData16)
{
	/* Auto-negotiation flow control only when auto-negotiation is enabled */
	if (pAC->Config.AutoNeg != SK_AUTONEG_OFF ||
		pAC->Hw.ReqMediaType == REQ_MEDIA_TYPE_AUTO ||
		pAC->Hw.ReqMediaType == REQ_MEDIA_TYPE_UTP_AUTO) {

		/* Please refer to Table 28B-3 of the 802.3ab-1999 spec. */
		if ((pAC->Hw.FlowControl == FLOW_CONTROL_AUTO_PAUSE) ||
			((pAC->Hw.FlowControl & FLOW_CONTROL_RECEIVE_PAUSE) != 0 &&
			 (pAC->Hw.FlowControl & FLOW_CONTROL_TRANSMIT_PAUSE) != 0)) {

			*pData16 = AD_PAUSE_CAPABLE;
		}
		else if ((pAC->Hw.FlowControl & FLOW_CONTROL_TRANSMIT_PAUSE) != 0) {
			*pData16 = AD_ASYM_PAUSE;
		}
		else if ((pAC->Hw.FlowControl & FLOW_CONTROL_RECEIVE_PAUSE) != 0) {
			*pData16 = AD_PAUSE_CAPABLE | AD_ASYM_PAUSE;
		}
	}

	SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
		("GetPhyAdFlowCtrlSets, Data16=0x%04X\n", *pData16));

}	/* GetPhyAdFlowCtrlSets */


/******************************************************************************
 *
 *  TranslReqMediaType
 *
 * Description:
 *	Sets up the default line speed and duplex modes based on the requested
 *	media type.
 *
 * Returns:	N/A
 *
 */
static void TranslReqMediaType(
SK_U16 ReqMediaType,
SK_U16 *pMediaType,
SK_U16 *pLineSpeed,
SK_U16 *pDuplexMode)
{
	/* determine media type */
	switch (ReqMediaType) {
		case REQ_MEDIA_TYPE_UTP_AUTO:
			*pMediaType = MEDIA_TYPE_UTP;
			*pLineSpeed = LINK_SPEED_UNKNOWN;
			*pDuplexMode = DUPLEX_MODE_UNKNOWN;
			break;

		case REQ_MEDIA_TYPE_UTP_10MBPS_HALF:
			*pMediaType = MEDIA_TYPE_UTP;
			*pLineSpeed = LINK_SPEED_10MBPS;
			*pDuplexMode = DUPLEX_MODE_HALF;
			break;

		case REQ_MEDIA_TYPE_UTP_10MBPS_FULL:
			*pMediaType = MEDIA_TYPE_UTP;
			*pLineSpeed = LINK_SPEED_10MBPS;
			*pDuplexMode = DUPLEX_MODE_FULL;
			break;

		case REQ_MEDIA_TYPE_UTP_100MBPS_HALF:
			*pMediaType = MEDIA_TYPE_UTP;
			*pLineSpeed = LINK_SPEED_100MBPS;
			*pDuplexMode = DUPLEX_MODE_HALF;
			break;

		case REQ_MEDIA_TYPE_UTP_100MBPS_FULL:
			*pMediaType = MEDIA_TYPE_UTP;
			*pLineSpeed = LINK_SPEED_100MBPS;
			*pDuplexMode = DUPLEX_MODE_FULL;
			break;

		case REQ_MEDIA_TYPE_UTP_1000MBPS_HALF:
			*pMediaType = MEDIA_TYPE_UTP;
			*pLineSpeed = LINK_SPEED_1000MBPS;
			*pDuplexMode = DUPLEX_MODE_HALF;
			break;

		case REQ_MEDIA_TYPE_UTP_1000MBPS_FULL:
			*pMediaType = MEDIA_TYPE_UTP;
			*pLineSpeed = LINK_SPEED_1000MBPS;
			*pDuplexMode = DUPLEX_MODE_FULL;
			break;

		case REQ_MEDIA_TYPE_FIBER_1000MBPS_HALF:
			*pMediaType = MEDIA_TYPE_FIBER;
			*pLineSpeed = LINK_SPEED_1000MBPS;
			*pDuplexMode = DUPLEX_MODE_HALF;
			break;

		case REQ_MEDIA_TYPE_FIBER_1000MBPS_FULL:
			*pMediaType = MEDIA_TYPE_FIBER;
			*pLineSpeed = LINK_SPEED_1000MBPS;
			*pDuplexMode = DUPLEX_MODE_FULL;
			break;

		case REQ_MEDIA_TYPE_AUTO:
		default:
			*pMediaType = MEDIA_TYPE_AUTO;
			*pLineSpeed = LINK_SPEED_UNKNOWN;
			*pDuplexMode = DUPLEX_MODE_UNKNOWN;
			break;
	} /* switch */

} /* TranslReqMediaType */


/******************************************************************************
 *
 *  ForceAutoNeg
 *
 * Description:
 *
 * Returns:	N/A
 *
 */
static void ForceAutoNeg(
SK_AC	*pAC,				/* Pointer to Adapter Context */
SK_IOC	IoC,				/* I/O Context */
SK_U16	ReqMediaType)
{
	SK_U16	MediaType;
	SK_U16	LineSpeed;
	SK_U16	DuplexMode;
	SK_U16	NewPhyCtrl;
	SK_U16	Data16;
	SK_U16	AutoNegAdvert;
	SK_U16	Ctrl1000BaseT;
	int		Cnt;

	SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
		("ForceAutoNeg, ReqMediaType=%u\n", ReqMediaType));

	/* Get the interface type, line speed, and duplex mode */
	TranslReqMediaType(ReqMediaType,
		&MediaType,
		&LineSpeed,
		&DuplexMode);

	/* Setup the 10/100 Mbps auto-negotiation advertisement register */
	GetPhyAdFlowCtrlSets(pAC, &AutoNegAdvert);
    AutoNegAdvert |= AD_PROTOCOL_802_3_CSMA_CD;
	
	/* Setup the 1000Base-T control & advertise register */
	Ctrl1000BaseT = 0;
	
	switch (LineSpeed) {
		case LINK_SPEED_UNKNOWN:
			AutoNegAdvert |= (AD_10BASET_HALF | AD_10BASET_FULL |
				AD_100BASETX_FULL | AD_100BASETX_HALF);
	
			/* Advertise 1000 Mbps */
			Ctrl1000BaseT |= (AD_1000BASET_HALF | AD_1000BASET_FULL);
			SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
				("Advertise 1000 Mbps\n"));
			break;
	
		case LINK_SPEED_1000MBPS:
			if (DuplexMode == DUPLEX_MODE_FULL) {
				SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
					("Advertise 1000Base-T FullDup\n"));
				Ctrl1000BaseT |= AD_1000BASET_FULL;
			}
			else {
				SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
					("Advertise 1000Base-T HalfDup\n"));
				Ctrl1000BaseT |= AD_1000BASET_HALF;
			}

			if (pAC->Config.AutoNeg == SK_AUTONEG_OFF &&
				pAC->Config.MsMode != SK_MS_MODE_AUTO) {
				
				Ctrl1000BaseT |= MASTER_SLAVE_ENABLE;
				
				if (pAC->Config.MsMode == SK_MS_MODE_MASTER) {
					SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
						("Set 1000Base-T: Enable Master\n"));
					Ctrl1000BaseT |= MASTER_SLAVE_VALUE;
				}
				else {
					SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
						("Set 1000Base-T: Enable Slave\n"));
				}
			}
			break;
		
		case LINK_SPEED_100MBPS:
			if (DuplexMode == DUPLEX_MODE_FULL) {
				SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
					("Advertise 100Mbps FD\n"));
				AutoNegAdvert |= AD_100BASETX_FULL;
			}
			else {
				SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
					("Advertise 100Mbps HD\n"));
				AutoNegAdvert |= AD_100BASETX_HALF;
			}
			break;
		
		case LINK_SPEED_10MBPS:
			if (DuplexMode == DUPLEX_MODE_FULL) {
				SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
					("Advertise 10Mbps FD\n"));
				AutoNegAdvert |= AD_10BASET_FULL;
			}
			else {
				SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
					("Advertise 10Mbps HD\n"));
				AutoNegAdvert |= AD_10BASET_HALF;
			}
			break;
	}

	SkGiWritePhy(pAC, IoC, PHY_AN_AD_REG, AutoNegAdvert);

	SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
		("Set PHY_AN_AD_REG=0x%04X\n", AutoNegAdvert));

	SkGiWritePhy(pAC, IoC, BCM540X_1000BASET_CTRL_REG, Ctrl1000BaseT);
	
	SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
		("Set 1000BASET_CTRL_REG=0x%04X\n", Ctrl1000BaseT));

	/* Force line speed if auto-negotiation is disabled */
	if (pAC->Config.AutoNeg == SK_AUTONEG_OFF &&
		LineSpeed != LINK_SPEED_UNKNOWN) {
		SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
			("Forcing line speed...\n"));

		/* This code path is executed only when there is link */
		pAC->Hw.MediaType = MediaType;
		pAC->Hw.LineSpeed = LineSpeed;
		pAC->Hw.DuplexMode = DuplexMode;

		/* Force line speed */
		NewPhyCtrl = 0;
		switch (LineSpeed) {
			case LINK_SPEED_10MBPS:
				NewPhyCtrl |= SPEED_SELECT_10MBPS;
				SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
					("to 10Mbps and ...\n"));
				break;

			case LINK_SPEED_100MBPS:
				NewPhyCtrl |= SPEED_SELECT_100MBPS;
				SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
					("to 100Mbps and ...\n"));
				break;

			case LINK_SPEED_1000MBPS:
				NewPhyCtrl |= SPEED_SELECT_1000MBPS;
				SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
					("to 1000Mbps and ...\n"));
				break;

			default:
				NewPhyCtrl |= SPEED_SELECT_1000MBPS;
				SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
					("default to 1000Mbps...\n"));
				break;
		}

		if (DuplexMode == DUPLEX_MODE_FULL) {
			NewPhyCtrl |= FULL_DUPLEX_MODE;
			SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
				("Full duplex\n"));
		}
		else {
			SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
				("Half duplex\n"));
		}

		/* Don't do anything if the PHY_CTRL is already setup */
		SkGiReadPhy(pAC, IoC, PHY_CTRL_REG, &Data16);
		SkGiReadPhy(pAC, IoC, PHY_CTRL_REG, &Data16);

		if (Data16 != NewPhyCtrl) {
			/* Bring the link down temporary before forcing line speed */
			SkGiWritePhy(pAC, IoC, PHY_CTRL_REG, LOOPBACK_MODE);

			/* Wait for link to go down */
			for (Cnt = 0; Cnt < 15000; Cnt++) {
				SkOsWaitUsec(pAC, 10);

				SkGiReadPhy(pAC, IoC, PHY_STATUS_REG, &Data16);
				SkGiReadPhy(pAC, IoC, PHY_STATUS_REG, &Data16);

				if ((Data16 & LINK_PASS) == 0) {
					SkOsWaitUsec(pAC, 40);
					break;
				}
			}

			SkGiWritePhy(pAC, IoC, PHY_CTRL_REG, NewPhyCtrl);
			SkOsWaitUsec(pAC, 40);
		}
	}
	else {
		SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
			("Restarting auto-negotiation\n"));
		SkGiWritePhy(pAC, IoC, PHY_CTRL_REG, AUTO_NEG_ENABLE | RESTART_AUTO_NEG);
	}

} /* ForceAutoNeg */


/*****************************************************************************
 *
 * 	InitCopperPhy - initialize copper PHY and execute auto-negotiation
 *
 * Description:
 *
 * Returns: SK_STATUS
 *
 */
static SK_STATUS InitCopperPhy(
SK_AC	*pAC,	/* Pointer to Adapter Context */
SK_IOC	IoC)	/* I/O Context */
{
	int		i;
	SK_U16	Data16;
	SK_U16	CurrLineSpeed;
	SK_U16	CurrDuplexMode;
	SK_U16	CurrLinkStatus;
	SK_U16	FlowCap;	 			/* Flow control capability */
	SK_U16	ExpectedPhyAd;
	SK_U16	LocalPhyAd;
	SK_U16	RemotePhyAd;
	SK_BOOL	LinkUp;
	SK_BOOL	LinkUpAux;

#ifdef xDEBUG
	SK_U32	Data32;
#endif /* DEBUG */

#ifdef DEBUG
	SK_U32	CurrentTime_us;
	SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT, ("InitCopperPhy\n"));

	SkOsGetTimeUsec(pAC, &CurrentTime_us);
	SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
		("CurrentTime_us=%12u\n", CurrentTime_us));
#endif /* DEBUG */
	
	/* Assume there is not link */
	CurrLinkStatus = STATUS_LINK_DOWN;
	LinkUp = SK_FALSE;
	LinkUpAux = SK_FALSE;

	/* Disable link change attention */
	SK_OUT32(IoC, ETH_MAC_EVENT_ENABLE, 0);

	/* Clear link change attention */
	SK_OUT32(IoC, ETH_MAC_STATUS, SYNC_CHANGED | CFG_CHANGED);

	/* Disable auto-polling for the moment */
	SK_OUT32(IoC, MI_MODE, MI_MODE_RESERVED);

	SkOsWaitUsec(pAC, 40);

#ifdef xDEBUG
	SkGiReadPhy(pAC, IoC, PHY_CTRL_REG, &Data16);
	SkGiReadPhy(pAC, IoC, PHY_CTRL_REG, &Data16);
	
	if ((Data16 & LOOPBACK_MODE) != 0) {
		SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
			("PHY is in loopback mode\n"));
		return (SK_STATUS_FAILED);
	}
#endif /* DEBUG */

	/* Exit lower power mode */
	SkGiWritePhy(pAC, IoC, BCM540X_AUX_CTRL, SEL_POWER_CONTROL |
		POWER_1_5_VOLT | POWER_RESERVED);

#ifdef PHY_WA_100BASE_TX
	/* BCM5411 errata : 100Base-TX operation issue */
	SkGiWritePhy(pAC, IoC, BCM540X_AUX_CTRL, 0x8107);
#endif

	/* Acknowledge outstanding interrupts */
	SkGiReadPhy(pAC, IoC, BCM540X_INT_STATUS, &Data16);
	SkGiReadPhy(pAC, IoC, BCM540X_INT_STATUS, &Data16);

	/* Configure the PHY interrupt mask */
	SkGiWritePhy(pAC, IoC, BCM540X_INT_MASK, ~LINK_CHANGE);

	/* Configure the "Tree LINK LED Mode" bit */
	SkGiWritePhy(pAC, IoC, BCM540X_EXT_CTRL_REG, LINK3_LED_MODE);

#ifdef xDEBUG
	for (i = 0; i < 5; i++) {
		SkGiReadPhy(pAC, IoC, PHY_STATUS_REG, &Data16);
		SkGiReadPhy(pAC, IoC, PHY_STATUS_REG, &Data16);
		SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
			("PhyStatus=0x%04X, Link:%c, Auto:%c\n", Data16,
			 ((Data16 & LINK_PASS) != 0) ? 'U': 'D',
			 ((Data16 & AUTO_NEG_COMPLETE) != 0) ? 'C': '-'));
		SK_IN32(IoC, US_TIMER_REG, &Data32);
		SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
			("Myst. Timer Reg=%12u\n", Data32));
		SkOsGetTimeUsec(pAC, &CurrentTime_us);
		SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
			("CurrentTime_us=%12Lu\n", CurrentTime_us));
		if (Data16 != 0xFFFF && Data32 > 300000 &&
			(Data16 & (LINK_PASS | AUTO_NEG_COMPLETE)) ==
			  (LINK_PASS | AUTO_NEG_COMPLETE)) {
			break;
		}
		SkOsWaitMsec(pAC, 100);
	}
#endif /* DEBUG */
	
	/* Get current link and duplex mode */
	SkGiReadPhy(pAC, IoC, PHY_STATUS_REG, &Data16);
	SkGiReadPhy(pAC, IoC, PHY_STATUS_REG, &Data16);
	SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
		("PhyStatus=0x%04X, Link:%c, Auto:%c\n", Data16,
		 ((Data16 & LINK_PASS) != 0) ? 'U': 'D',
		 ((Data16 & AUTO_NEG_COMPLETE) != 0) ? 'C': '-'));
	
	if (Data16 != 0xFFFF && (Data16 & LINK_PASS) != 0) {
		SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT, ("Phy Link is UP\n"));
		LinkUp = SK_TRUE;
	}

	SkGiReadPhy(pAC, IoC, BCM540X_AUX_STATUS, &Data16);
	for (i = 0; i < 100; i++) {
		SkOsWaitUsec(pAC, 100);

		SkGiReadPhy(pAC, IoC, BCM540X_AUX_STATUS, &Data16);
		if (Data16 != 0 && Data16 != 0xFFFF) {
			break;
		}
	}
	SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
		("PhyAuxStat=0x%04X (%u)\n", Data16, i));

	if (Data16 != 0xFFFF && (Data16 & X_LINK_PASS) != 0) {
		SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT, ("Aux Link is UP\n"));
		LinkUpAux = SK_TRUE;
	}

	if (LinkUp && LinkUpAux) {
		/* Determine the requested line speed and duplex */
		TranslReqMediaType(pAC->Hw.ReqMediaType,
			&pAC->Hw.MediaType,
			&pAC->Hw.LineSpeed,
			&pAC->Hw.DuplexMode);
/*		
		SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
			("TransReqMediaType, LineSpeed=%u, DuplexMode=%u\n",
			 pAC->Hw.LineSpeed, pAC->Hw.DuplexMode));
*/
		/* Determine the current line and duplex settings */
		switch (Data16 & X_SPEED_MASK) {
			case X_10BASET_HD:
				CurrLineSpeed = LINK_SPEED_10MBPS;
				CurrDuplexMode = DUPLEX_MODE_HALF;
				break;

			case X_10BASET_FD:
				CurrLineSpeed = LINK_SPEED_10MBPS;
				CurrDuplexMode = DUPLEX_MODE_FULL;
				break;

			case X_100BASETX_HD:
				CurrLineSpeed = LINK_SPEED_100MBPS;
				CurrDuplexMode = DUPLEX_MODE_HALF;
				break;

			case X_100BASETX_FD:
				CurrLineSpeed = LINK_SPEED_100MBPS;
				CurrDuplexMode = DUPLEX_MODE_FULL;
				break;

			case X_100BASET_HD:
				SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
					("Line setting: 1000BASET_HD\n"));
				CurrLineSpeed = LINK_SPEED_1000MBPS;
				CurrDuplexMode = DUPLEX_MODE_HALF;
				break;

			case X_100BASET_FD:
				SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
					("Line setting: 1000BASET_FD\n"));
				CurrLineSpeed = LINK_SPEED_1000MBPS;
				CurrDuplexMode = DUPLEX_MODE_FULL;
				break;

			default:
				SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
					("Unknown setting: AUX_STATUS_REG=0x%04X\n", Data16));
				CurrLineSpeed = LINK_SPEED_UNKNOWN;
				CurrDuplexMode = DUPLEX_MODE_UNKNOWN;
				break;
		}

		/* Make sure we are in auto-negotiation mode */
		SkGiReadPhy(pAC, IoC, PHY_CTRL_REG, &Data16);
		/* Bug? First read may not return the right value */
		SkGiReadPhy(pAC, IoC, PHY_CTRL_REG, &Data16);

		/* Use the current line settings for "auto" mode */
		if (pAC->Hw.ReqMediaType == REQ_MEDIA_TYPE_AUTO ||
			pAC->Hw.ReqMediaType == REQ_MEDIA_TYPE_UTP_AUTO) {

			if ((Data16 & AUTO_NEG_ENABLE) != 0) {
				CurrLinkStatus = STATUS_LINK_UP;
				/*
				 * We may be exiting low power mode and the link is in 10mb.
				 * In this case, we need to restart auto-negotiation.
				 */
				SkGiReadPhy(pAC, IoC, BCM540X_1000BASET_CTRL_REG, &Data16);

				if (pAC->Config.AutoNeg == SK_AUTONEG_OFF &&
					(Data16 & (AD_1000BASET_HALF | AD_1000BASET_FULL)) == 0) {

					SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
						("Gigabit speed not advertised\n"));
					CurrLinkStatus = STATUS_LINK_WRONG_SET;
				}
			}
			else {
				SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
					("No auto-negotiation link\n"));
				CurrLinkStatus = STATUS_LINK_WRONG_SET;
			}
		}
		else {
			/* Force line settings */
#ifdef DEBUG
			if (pAC->Hw.LineSpeed == LINK_SPEED_UNKNOWN ||
				pAC->Hw.DuplexMode == DUPLEX_MODE_UNKNOWN) {

				SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
					("Invalid line setting\n"));
			} /* if */
#endif /* DEBUG */

			/* Use the current setting if it matches the user's requested */
			if (pAC->Hw.LineSpeed == CurrLineSpeed &&
				pAC->Hw.DuplexMode == CurrDuplexMode) {

				CurrLinkStatus = STATUS_LINK_UP;
			}
			else {
				CurrLinkStatus = STATUS_LINK_WRONG_SET;
			}
		}

		/* Save line settings */
		pAC->Hw.LineSpeed = CurrLineSpeed;
		pAC->Hw.DuplexMode = CurrDuplexMode;
		pAC->Hw.MediaType = MEDIA_TYPE_UTP;
	}
	
	if (CurrLinkStatus == STATUS_LINK_WRONG_SET) {
		SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
			("Undesirable link speed\n"));
		CurrLinkStatus = STATUS_LINK_DOWN;
	}

	/* Setup flow control */
	if (CurrLinkStatus == STATUS_LINK_UP) {
		FlowCap = FLOW_CONTROL_NONE;

		if (pAC->Hw.DuplexMode == DUPLEX_MODE_FULL) {
			if (pAC->Config.AutoNeg != SK_AUTONEG_OFF ||
				pAC->Hw.ReqMediaType == REQ_MEDIA_TYPE_AUTO ||
				pAC->Hw.ReqMediaType == REQ_MEDIA_TYPE_UTP_AUTO) {

				SkGiReadPhy(pAC, IoC, PHY_AN_AD_REG, &LocalPhyAd);
				LocalPhyAd &= (AD_ASYM_PAUSE | AD_PAUSE_CAPABLE);

				SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
					("Get PHY_AN_AD_REG=0x%04X\n", LocalPhyAd));

				GetPhyAdFlowCtrlSets(pAC, &ExpectedPhyAd);

				SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
					("ExpectedPhyAd=0x%04X\n", ExpectedPhyAd));

				if (ExpectedPhyAd != 0 && LocalPhyAd != ExpectedPhyAd) {
					SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
						("Flowcontrol not advertised\n"));
					CurrLinkStatus = STATUS_LINK_DOWN;
				}
				else {
					SkGiReadPhy(pAC, IoC, PHY_LINK_PARTNER_ABIL_REG, &RemotePhyAd);

                    SetFlowControl(pAC, IoC, LocalPhyAd, RemotePhyAd);
				}
			}
		}
	}

	if (CurrLinkStatus == STATUS_LINK_DOWN) {
		ForceAutoNeg(pAC, IoC, pAC->Hw.ReqMediaType);

		/* If we force line speed, we make get link right away */
		SkGiReadPhy(pAC, IoC, PHY_STATUS_REG, &Data16);
		SkGiReadPhy(pAC, IoC, PHY_STATUS_REG, &Data16);
		SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
			("PhyStatus=0x%04X, Link:%c, Auto:%c\n", Data16,
			 ((Data16 & LINK_PASS) != 0) ? 'U': 'D',
			 ((Data16 & AUTO_NEG_COMPLETE) != 0) ? 'C': '-'));

		if ((Data16 & LINK_PASS) != 0) {
			CurrLinkStatus = STATUS_LINK_UP;
		}
	}

	/* GMII interface */
	pAC->Hw.MacMode &= ~PORT_MODE_MASK;

	if (CurrLinkStatus == STATUS_LINK_UP &&
		(pAC->Hw.LineSpeed == LINK_SPEED_100MBPS ||
		 pAC->Hw.LineSpeed == LINK_SPEED_10MBPS)) {

		pAC->Hw.MacMode |= PORT_MODE(PM_MII);
	}
	else {
		pAC->Hw.MacMode |= PORT_MODE(PM_GMII);
	}

	/* Set the MAC to operate in the appropriate duplex mode */
	if (pAC->Hw.DuplexMode == DUPLEX_MODE_HALF) {
		pAC->Hw.MacMode |= HALF_DUPLEX;
	}

	/* Set the link polarity bit */
	if ((pAC->Hw.RevisionId <= SK_REVISION_1_0) &&
		(pAC->Hw.PhyIdentifier == PHY_BCM5411_PHY_ID) &&
		!((CurrLinkStatus == STATUS_LINK_UP) &&
		  (pAC->Hw.LineSpeed == LINK_SPEED_10MBPS))) {

		pAC->Hw.MacMode |= LINK_POLARITY;
	}

	SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
		("SET MacMode=0x%04X\n", pAC->Hw.MacMode));

	SK_OUT32(IoC, ETH_MAC_MODE, pAC->Hw.MacMode);

	/* Enable PHY link change attention */
	SK_OUT32(IoC, ETH_MAC_EVENT_ENABLE, MI_INTERRUPT_ATN |
		LINK_STATE_CHANGED_ATN);

	/* Indicate link status */
	pAC->Hw.LinkStatus = CurrLinkStatus;

	return (SK_STATUS_OK);
}	/* InitCopperPhy */


/*****************************************************************************
 *
 * 	InitFiberPhy - activate TBI and execute auto-negotiation
 *
 * Description:
 *
 * Returns: SK_STATUS
 *
 */
static SK_STATUS InitFiberPhy(
SK_AC	*pAC,	/* Pointer to Adapter Context */
SK_IOC	IoC)	/* I/O Context */
{
	int		i;
	int		AnStatus;
	SK_U16	Data16;
	SK_U32	MacStat;
	SK_U16	LocalPhyAd;
	SK_U16	RemotePhyAd;
	SK_U16	CurrLinkStatus;
#ifdef DEBUG
	SK_U32	Data32;
    SK_U32	StartTime;
    SK_U32	StopTime;
#endif /* DEBUG */

	AnStatus = SK_STATUS_FAILED;

	SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT, ("InitFiberPhy\n"));

#ifdef xDEBUG
	/* Get the MAC status */
	SK_IN32(IoC, ETH_MAC_STATUS, &MacStat);

	SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
		("MAC Mode=0x%06X, MAC Status=0x%07X def\n", pAC->Hw.MacMode, MacStat));
#endif	/*DEBUG*/
	
	/* Disable link change attention */
	SK_OUT32(IoC, ETH_MAC_EVENT_ENABLE, 0);

	/* Clear link change attention */
	SK_OUT32(IoC, ETH_MAC_STATUS, SYNC_CHANGED | CFG_CHANGED |
		PORT_DECODE_ERROR);

	/* Enable TBI and full duplex mode */
	pAC->Hw.MacMode |= PORT_MODE(PM_TBI);
	
	SK_OUT32(IoC, ETH_MAC_MODE, pAC->Hw.MacMode);

	SkOsWaitUsec(pAC, 400);

	/* Get the link status */
	SK_IN32(IoC, ETH_MAC_STATUS, &MacStat);
	SK_IN32(IoC, ETH_MAC_STATUS, &MacStat);
	SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
		("MAC Status=0x%07X\n", MacStat));

	CurrLinkStatus = STATUS_LINK_DOWN;
	
	if ((MacStat & (PCS_SYNCED | SIGNAL_DETECTED)) ==
		(PCS_SYNCED | SIGNAL_DETECTED)) {
		SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT, ("PCS synched\n"));

#ifdef DEBUG
		/* determine link status */
		SK_IN32(IoC, TRANSMIT_STATUS, &Data32);
		SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
			("Tx Status=0x%04X\n", Data32));
		if ((Data32 & LINK_UP) != 0) {
			SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT, ("Tx Link is UP\n"));
		}

		SK_IN32(IoC, MI_STATUS, &Data32);
		SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
			("MI Status=0x%04X\n", Data32));
		if ((Data32 & LINK_STATUS) != 0) {
			SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT, ("MI Link is UP\n"));
		}
#endif /* DEBUG */

        if (pAC->Hw.ReqMediaType == REQ_MEDIA_TYPE_AUTO) {
            /* auto-negotiation mode */
#ifdef DEBUG
			SkOsGetTimeUsec(pAC, &StartTime);
#endif /* DEBUG */
            SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
				("Auto-negotiation started\n"));

            /* Initialize the auto-negotiation default capabilities */
			SK_MEMSET(&pAC->Hw.AutoNeg, 0, sizeof(AUTO_NEG_STATE));

			/* Initialize the default advertisement register */
			pAC->Hw.AutoNeg.mr_adv_full_duplex = 1;
			pAC->Hw.AutoNeg.mr_an_enable = 1;

            /* Setup flow control advertisement register */
            GetPhyAdFlowCtrlSets(pAC, &Data16);

            pAC->Hw.AutoNeg.mr_adv_sym_pause =
					((Data16 & AD_PAUSE_CAPABLE) != 0) ? 1 : 0;

            pAC->Hw.AutoNeg.mr_adv_asym_pause =
				((Data16 & AD_ASYM_PAUSE) != 0) ? 1 : 0;

            /* Run the auto-negotiation state machine */
            for (i = 0; i < 75000; i++) {
                AnStatus = AutoNeg8023z(pAC, IoC);

                if (AnStatus == SK_STATUS_OK ||
                    AnStatus == SK_STATUS_FAILED) {
                    break;
                }
				SkOsWaitUsec(pAC, 10);
            }
#ifdef DEBUG
			SkOsGetTimeUsec(pAC, &StopTime);
#endif /* DEBUG */

			SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
				("Auto-negotiation %s (c=%u, t=%u)\n",
				 (AnStatus == SK_STATUS_OK) ? "done" : "failed",
				 i, (StopTime-StartTime)));
			
			/* Stop sending config commands */
			pAC->Hw.MacMode &= ~SEND_CONFIGS;

			SK_OUT32(IoC, ETH_MAC_MODE, pAC->Hw.MacMode);

            /* Solve control settings */
            if (AnStatus == SK_STATUS_OK &&
                pAC->Hw.AutoNeg.mr_an_complete &&
				pAC->Hw.AutoNeg.mr_link_ok &&
                pAC->Hw.AutoNeg.mr_lp_adv_full_duplex) {

                SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
					("Auto-negotiation successful\n"));

                LocalPhyAd = 0;
                if (pAC->Hw.AutoNeg.mr_adv_sym_pause) {
                    LocalPhyAd |= AD_PAUSE_CAPABLE;
                }

                if (pAC->Hw.AutoNeg.mr_adv_asym_pause) {
                    LocalPhyAd |= AD_ASYM_PAUSE;
                }

                RemotePhyAd = 0;
                if (pAC->Hw.AutoNeg.mr_lp_adv_sym_pause) {
                    RemotePhyAd |= PAUSE_CAPABLE;
                }

                if (pAC->Hw.AutoNeg.mr_lp_adv_asym_pause) {
                    RemotePhyAd |= ASYM_PAUSE;
                }

                SetFlowControl(pAC, IoC, LocalPhyAd, RemotePhyAd);

                CurrLinkStatus = STATUS_LINK_UP;
            }
        }
        else {
            /* Force line speed */
            SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
				("Forcing fiber 1000FD\n"));

            CurrLinkStatus = STATUS_LINK_UP;
        }
	}

	/* Clear the status bits again (W2C) */
	SK_OUT32(IoC, ETH_MAC_STATUS, SYNC_CHANGED | CFG_CHANGED);

	/* Enable link change attention */
	SK_OUT32(IoC, ETH_MAC_EVENT_ENABLE, LINK_STATE_CHANGED_ATN);

    /* Initialize the current link status */
    if (CurrLinkStatus == STATUS_LINK_UP) {
        pAC->Hw.LineSpeed = LINK_SPEED_1000MBPS;
        pAC->Hw.DuplexMode = DUPLEX_MODE_FULL;
    }
    else {
        pAC->Hw.LineSpeed = LINK_SPEED_UNKNOWN;
        pAC->Hw.DuplexMode = DUPLEX_MODE_UNKNOWN;
    }

    /* Update link status */
    pAC->Hw.LinkStatus = CurrLinkStatus;
	
	return (SK_STATUS_OK);
}	/* InitFiberPhy */


/*****************************************************************************
 *
 * 	SkGiInitPhy - initialize the PHY
 *
 * Description:
 *
 * Returns:	SK_STATUS
 *
 */
SK_STATUS SkGiInitPhy(
SK_AC	*pAC,	/* Pointer to Adapter Context */
SK_IOC	IoC)	/* I/O Context */
{
	SK_U32		Data32;
	SK_STATUS	RetValue;
	SK_EVPARA	DummyEvPara;

	DummyEvPara.pParaPtr = NULL;

//	SK_IN32(IoC, ETH_MAC_MODE, &pAC->Hw.MacMode);
	SK_IN32(IoC, ETH_MAC_MODE, &Data32);

	SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
		("SkGiInitPhy, MacMode=0x%04X\n", Data32));

	SK_MEM_RD32(pAC, FW_RUNMODE, &Data32);
	SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
		("FwRunMode=0x%08X\n", Data32));

	if (pAC->Hw.IsCopper) {
		RetValue = InitCopperPhy(pAC, IoC);
	}
	else {	/* Fiber adapter */
		RetValue = InitFiberPhy(pAC, IoC);
	}
	
	SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
		("LinkStatus=%u\n", pAC->Hw.LinkStatus));
	
	if (pAC->Hw.LinkStatus == STATUS_LINK_UP) {
		SkEventQueue(pAC, SKGI_DRV, SK_DRV_NET_UP, DummyEvPara);
	}
	else if (pAC->Hw.LinkStatus == STATUS_LINK_DOWN){
		SkEventQueue(pAC, SKGI_DRV, SK_DRV_NET_DOWN, DummyEvPara);
	}
	SkEventDispatcher(pAC, IoC);

	return (RetValue);
}	/* SkGiInitPhy */


/*****************************************************************************
 *
 * 	SetPowerState - setup power management
 *
 * Description:
 *
 * Returns: SK_STATUS
 *
 */
static SK_STATUS SetPowerState(
SK_AC	*pAC,		/* Pointer to Adapter Context */
SK_IOC	IoC,		/* I/O Context */
int		Level)
{
	SK_U16	PmCtrl;
#ifdef DEBUG
	SK_U32	Data32;
#endif

	SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
		("SetPowerState, Level=%u\n", Level));

	SK_PCI_IN16(pAC, PCI_PM_STATUS_CTRL_REG, &PmCtrl);
	SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
		("PmCtrl=0x%04X\n", PmCtrl));

	/* Clear the PME_ASSERT bit and the power state bits */
	
	PmCtrl |= PM_PME_ASSERTED;
	PmCtrl &= ~PM_POWER_STATE_MASK;

	/* Set the appropriate power state */
	if (Level == PM_POWER_STATE_D0) {
		/* Bring the card out of low power mode */
		PmCtrl |= PM_POWER_STATE_D0;
	}
    else if (Level == PM_POWER_STATE_D3) {
		/* Set the phy to low power mode. */
		/* Put the the hardware in low power mode. */
		PmCtrl |= PM_POWER_STATE_D3;
	}
	else {
		return (SK_STATUS_FAILED);
	}

	SK_PCI_OUT16(pAC, PCI_PM_STATUS_CTRL_REG, PmCtrl);

#ifdef DEBUG
	SK_PCI_IN16(pAC, PCI_PM_STATUS_CTRL_REG, &PmCtrl);
	SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
		("PmCtrl=0x%04X after setup\n", PmCtrl));

	SK_IN32(IoC, MISC_LOCAL, &Data32);
	SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
		("MiscLocCtrl=0x%08X\n", Data32));
#endif	/* DEBUG */
	return (SK_STATUS_OK);
} /* SetPowerState */


/*****************************************************************************
 *
 * 	SkGiHwacDeinit - stop MAC & PCI chip
 *
 * Description:
 *	This function stops the hardware. It can be restarted by executing
 *	SkGiHwacInit(SK_INIT_RUN).
 *
 * Returns: SK_STATUS
 *
 */
SK_STATUS SkGiHwacDeinit(
SK_AC		*pAC,		/* Pointer to Adapter Context */
SK_IOC		IoC)		/* I/O Context */
{
	int		i;
	SK_U64	Time;
	SK_U16	Data16;
	SK_U32	Data32;
#ifdef PCI_CONF_16BIT
	SK_U16	PciCmd;
#else  /* !PCI_CONF_16BIT */
	SK_U32	PciCmd;
#endif /* !PCI_CONF_16BIT */
	
#ifdef DEBUG
	SK_IN32(IoC, ETH_MAC_MODE, &Data32);
	SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
		("GET MacMode=0x%04X\n", Data32));
#endif
	
	/*
	 * Note: init sequence according to BMC5700 Programmer's Reference Guide
	 * rev. 01 from 04/09/2001
	 */

	SK_PCI_OUT32(pAC, PCI_MISC_H_CTRL, pAC->Hw.MiscHost | CLR_INT);

	/*
	 * Save the PCI command register.
	 * Resetting the core clocks resets PCI command reg !!!
	 */
#ifdef PCI_CONF_16BIT
	SK_PCI_IN16(pAC, PCI_COMMAND, &PciCmd);
#else  /* PCI_CONF_16BIT */
	SK_PCI_IN32(pAC, PCI_COMMAND, &PciCmd);
#endif /* PCI_CONF_16BIT */

	PciCmd |= BUS_MASTER | MEM_SPACE;

	SK_MEM_WR32(pAC, FW_MAILBOX, FW_MAILBOX_VAL);

// init_again:
	INITSTEP(1, "Reset core clocks");
	SK_IN32(IoC, MISC_CONF, &Data32);
	SK_OUT32(IoC, MISC_CONF, Data32 | CORE_CLK_RESET);

	INITSTEP(2, "Wait for reset complete");
//	SkOsWaitUsec(pAC, 100);
	SkOsWaitMsec(pAC, 100);

	for (i = 0; i < 1000; i++) {
		/* Restore the PCI command register.*/
#ifdef PCI_CONF_16BIT
		SK_PCI_OUT16(pAC, PCI_COMMAND, PciCmd);
#else  /* PCI_CONF_16BIT */
		SK_PCI_OUT32(pAC, PCI_COMMAND, PciCmd);
#endif /* PCI_CONF_16BIT */

		SK_IN32(IoC, MISC_CONF, &Data32);
		if ((Data32 & CORE_CLK_RESET) == 0) {
			SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
				("Misc. Config=0x%04X (%u)\n", Data32, i));
			break;
		}
		SkOsWaitUsec(pAC, 1000);
	}

	if (i == 1000) {
		SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
			("Core clock reset not completed !!!\n"));
		
#ifdef DEBUG
		SK_PCI_IN32(pAC, PCI_STATE_REG, &Data32);
		SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
			("PCI State=0x%04X\n", (SK_U16)Data32));
#endif	/* DEBUG */
		
		return (SK_STATUS_FAILED);
	}

	INITSTEP(3, "Disable interrupts");
	SK_PCI_OUT32(pAC, PCI_MISC_H_CTRL, pAC->Hw.MiscHost);

	INITSTEP(4, "Enable mem space and bus master again");
#ifdef PCI_CONF_16BIT
	SK_PCI_IN16(pAC, PCI_COMMAND, &PciCmd);
	if ((PciCmd & (BUS_MASTER | MEM_SPACE)) !=
		(SK_U16)(BUS_MASTER | MEM_SPACE)) {

		SK_PCI_OUT16(pAC, PCI_COMMAND, PciCmd | BUS_MASTER | MEM_SPACE);
	}
#else  /* PCI_CONF_16BIT */
	SK_PCI_IN32(pAC, PCI_COMMAND, &PciCmd);
	if ((PciCmd & (BUS_MASTER | MEM_SPACE)) != (BUS_MASTER | MEM_SPACE)) {
		SK_PCI_OUT32(pAC, PCI_COMMAND, PciCmd | BUS_MASTER | MEM_SPACE);
	}
#endif /* PCI_CONF_16BIT */

	INITSTEP(5, "Disable PCI-X relaxed ordering");
#ifdef PCI_CONF_16BIT
	SK_PCI_IN16(pAC, PCIX_COMMAND, &Data16);
	Data16 &= ~ENAB_RELAX_ORD;
	SK_PCI_OUT16(pAC, PCIX_COMMAND, Data16);
#else  /* PCI_CONF_16BIT */
	SK_PCI_IN32(pAC, PCIX_CAP_REG, &Data32);
	Data32 &= ~(ENAB_RELAX_ORD << 16);
	SK_PCI_OUT32(pAC, PCIX_CAP_REG, Data32);
#endif /* PCI_CONF_16BIT */

	INITSTEP(6, "Enable MAC mem arbiter");
	SK_OUT32(IoC, MEM_ARB_MODE, MEM_ARB_ENAB);

	INITSTEP(7, "Set MiscHostCtrl register");
	SK_PCI_OUT32(pAC, PCI_MISC_H_CTRL, pAC->Hw.MiscHost);

	INITSTEP(8, "Disable PXE restart");
//	SK_MEM_WR32(pAC, FW_MAILBOX, FW_MAILBOX_VAL);

	INITSTEP(9, "Poll firmware started");
	Time = START_TIMEOUT();
	do {
		SK_MEM_RD32(pAC, FW_MAILBOX, &Data32);
		if (TIMEOUT_REACHED(Time, 100, 9)) {	/* 750..760 ms >= 750 ms */
			return (SK_STATUS_FAILED);
		}
		SkOsWaitUsec(pAC, 10);
	} while (Data32 != ~((SK_U32)FW_MAILBOX_VAL));

	SK_MEM_RD32(pAC, FW_RUNMODE, &Data32);
	SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
		("FwRunMode=0x%08X\n", Data32));

	/* info from boot-code */
	if ((Data32 & (BIT1 | BIT0)) != BIT1) {
		SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
			("Perhaps we should re-init ...\n"));
//		goto init_again;
	}

	INITSTEP(10, "Clear Ethernet MAC mode");
#ifdef DEBUG
	SK_IN32(IoC, ETH_MAC_MODE, &Data32);
	SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
		("SkGiHwacDeinit, MacMode=0x%04X\n", Data32));
#endif
//	SK_OUT32(IoC, ETH_MAC_MODE, /* 0 */ GLOBAL_RESET);			/* RS */
	
	INITSTEP(11, "Restore PCI CacheLineSize");
	SK_PCI_IN32(pAC, PCI_CACHE_LINE_SIZE, &Data32);
	if (pAC->Hw.SavedCacheLineReg != Data32) {
		SK_PCI_OUT32(pAC, PCI_CACHE_LINE_SIZE, pAC->Hw.SavedCacheLineReg);
	}
	
	SK_PCI_IN32(pAC, PCI_SUBSYSTEM_VENDOR_ID, &Data32);
	if (pAC->Hw.SubsystemVendorId != Data32) {
		SK_PCI_OUT32(pAC, PCI_SUBSYSTEM_VENDOR_ID, pAC->Hw.SubsystemVendorId);
	}

//	(void)SetPowerState(pAC, IoC, PM_POWER_STATE_D3);
	
	if (pAC->Hw.IsCopper) {					/* RS */
		/* Bring the link down temporary */
		SkGiReadPhy(pAC, IoC, PHY_CTRL_REG, &Data16);
		SkGiReadPhy(pAC, IoC, PHY_CTRL_REG, &Data16);
		
		SkGiWritePhy(pAC, IoC, PHY_CTRL_REG, Data16 | LOOPBACK_MODE);
	}

	return (SK_STATUS_OK);
}	/* SkGiHwacDeinit */


/*****************************************************************************
 *
 * 	SkGiInitAdapter - initialize adapter
 *
 * Description:
 *
 * Returns: SK_STATUS
 *
 */
static SK_STATUS SkGiInitAdapter(
SK_AC		*pAC,		/* Pointer to Adapter Context */
SK_IOC		IoC)		/* I/O Context */
{
	SK_U64	Time;
	SK_U32	Data32;
	int		i;

	/*
	 * Note: init sequence according to BMC5700 Programmer's Reference Guide
	 * rev. 01 from 04/09/2001
	 */

	INITSTEP(12, "Clear MAC statistics block");
	for (i = 0x300; i < 0xb00; i += 4) {
		SK_MEM_WR32(pAC, i, 0);
	}

	INITSTEP(13, "Clear driver status memory area");
	SK_MEMSET(pAC->pStatusBlock, 0, sizeof(STATUS_BLOCK));

	INITSTEP(14, "Set PCI read/write command");
	SK_PCI_OUT32(pAC, PCI_DMA_RW_CTRL,
		DMA_WR_COMMAND(0x7) | DMA_RD_COMMAND(0x6) | MINIMUM_DMA(0x0f));
	
	INITSTEP(15, "Set ModeCtrl register");
	Data32 =
#ifdef SK_LITTLE_ENDIAN
		W_SWAP_DATA | B_SWAP_DATA | W_SWAP_NONFRAME |
#endif	/* SK_LITTLE_ENDIAN */
		HOST_STACK_UP | HOST_SEND_BD | TX_NO_PH_CHKSUM | INT_MAC_ATTENT;
	SK_OUT32(IoC, MODE_CTRL, Data32);

	INITSTEP(16, "Configure MAC's timer");
	SK_OUT32(IoC, MISC_CONF, TIMER_PRESCALE(65));

	INITSTEP(17, "Configure MAC memory pool");
	SK_OUT32(IoC, MBUF_POOL_BASE, 0x8000);
	SK_OUT32(IoC, MBUF_POOL_LENGTH, 0x18000);

	INITSTEP(18, "Configure MAC DMA resource pool");
	SK_OUT32(IoC, DMA_DESCR_POOL_BASE, 0x2000);
	SK_OUT32(IoC, DMA_DESCR_POOL_LEN, 0x2000);

	INITSTEP(19, "Configure MAC memory pool watermarks");
	SK_OUT32(IoC, MBUF_POOL_RD_LOW_WM, pAC->Config.MbufDescLow);
	SK_OUT32(IoC, MBUF_POOL_RX_LOW_WM, pAC->Config.MbufRxLow);
	SK_OUT32(IoC, MBUF_POOL_HIGH_WM, pAC->Config.MbufHigh);

	INITSTEP(20, "Configure DMA resource watermarks");
	SK_OUT32(IoC, DMA_DESCR_P_LOW_WM, 5);
	SK_OUT32(IoC, DMA_DESCR_P_HIGH_WM, 10);

	INITSTEP(21, "Enable buffer manager");
	SK_OUT32(IoC, BM_MODE, MBUF_LOW_ATTN_ENAB | BM_ENAB);

	INITSTEP(22, "Poll for start of buffer manager");
	Time = START_TIMEOUT();
	do {
		SK_IN32(IoC, BM_MODE, &Data32);
		if (TIMEOUT_REACHED(Time, 100, 22)) {
			return (SK_STATUS_FAILED);
		}
	} while ((Data32 & BM_ENAB) == 0);

	INITSTEP(23, "Enable hardware queues");
	SK_OUT32(IoC, FTQ_RESET, 0xffffffff);
	SK_OUT32(IoC, FTQ_RESET, 0x0);

	/* Wait until FTQ is ready */
	for (i = 0; i < 2000; i++) {
		SK_IN32(IoC, FTQ_RESET, &Data32);

		if (Data32 == 0) {
			break;
		}
		SkOsWaitUsec(pAC, 10);
	}

	if (i >= 2000) {
		SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
			("FTQ initialization failed\n"));
		return (SK_STATUS_FAILED);
	}

	INITSTEP(24, "Init standard RX buffer ring");
	SK_OUT32(IoC, STD_RX_HOST_HIGH, HIGH32(pAC->RecStdPrdRing.Phys));
	SK_OUT32(IoC, STD_RX_HOST_LOW, LOW32(pAC->RecStdPrdRing.Phys));
	SK_OUT32(IoC, STD_RX_LEN_FLAGS,
		RING_MAX_LEN(pAC->RecStdPrdRing.BufferSize));
	SK_OUT32(IoC, STD_RX_NIC_ADDR, RX_STD_BD_ADDR);

	INITSTEP(25, "Init jumbo RX buffer ring");
	
	if (pAC->Config.UseJumbo) {
		SK_OUT32(IoC, JUM_RX_HOST_HIGH, HIGH32(pAC->RecJumPrdRing.Phys));
		SK_OUT32(IoC, JUM_RX_HOST_LOW, LOW32(pAC->RecJumPrdRing.Phys));
		SK_OUT32(IoC, JUM_RX_LEN_FLAGS,
			RING_MAX_LEN(pAC->RecJumPrdRing.BufferSize));
		
		SK_OUT32(IoC, JUM_RX_NIC_ADDR, RX_JUM_BD_ADDR);
	}
	else {
		SK_OUT32(IoC, JUM_RX_LEN_FLAGS, DISABLE_RING);
	}

	INITSTEP(26, "Init mini RX buffer ring");
	/* Usable with external adapter RAM only */
#ifdef SK_EXT_MEM
	SK_OUT32(IoC, MIN_RX_HOST_HIGH, HIGH32(pAC->RecMinPrdRing.Phys));
	SK_OUT32(IoC, MIN_RX_HOST_LOW, LOW32(pAC->RecMinPrdRing.Phys));
	SK_OUT32(IoC, MIN_RX_LEN_FLAGS,
		RING_MAX_LEN(pAC->RecMinPrdRing.BufferSize));
	SK_OUT32(IoC, MIN_RX_NIC_ADDR, RX_MIN_BD_ADDR);
#else	/* !SK_EXT_MEM */
	SK_OUT32(IoC, MIN_RX_LEN_FLAGS, DISABLE_RING);
#endif	/* !SK_EXT_MEM */

	INITSTEP(27, "Set RX rings replenish threshold");
	
	SK_OUT32(IoC, JUM_RX_REPLEN_TRHESH, (pAC->Config.UseJumbo) ?
		(pAC->RecJumPrdRing.NumDescs / DIV_JUM_RX_REPLEN_TRESH) : 0);
	
	SK_OUT32(IoC, STD_RX_REPLEN_TRHESH,
		pAC->RecStdPrdRing.NumDescs / DIV_STD_RX_REPLEN_TRESH);
#ifdef SK_EXT_MEM
	SK_OUT32(IoC, MIN_RX_REPLEN_TRHESH,
		pAC->RecMinPrdRing.NumDescs / DIV_MIN_RX_REPLEN_TRESH);
#endif	/* SK_EXT_MEM */

	INITSTEP(28, "Disable unused send producer rings");
	for (i = pAC->Config.TxRingNum; i < 16; i++) {
		SK_MEM_WR32(pAC, TX_RCB_BASE + i * 0x10 + TX_RCB_LEN_FLAGS_OFFS,
			DISABLE_RING);
	}

	INITSTEP(29, "Init send producer index regs");
	for (i = 0; i < 16; i++) {
		SK_OUT32(IoC, TX_HOST_PROD_IDX1 + i * 8, 0);
		SK_OUT32(IoC, TX_HOST_PROD_IDX1 + i * 8 + 4, 0);
	}
	for (i = 0; i < 16; i++) {
		SK_OUT32(IoC, TX_NIC_PROD_IDX1 + i * 8, 0);
		SK_OUT32(IoC, TX_NIC_PROD_IDX1 + i * 8 + 4, 0);
	}

	INITSTEP(30, "Init send ring(s)");
	for (i = 0; i < (int)pAC->Config.TxRingNum; i++) {
		SK_MEM_WR32(pAC, TX_RCB_BASE + i * 0x10 + TX_RCB_HIGH_OFFS,
			HIGH32(pAC->SndPrdRing[i].Phys));
		SK_MEM_WR32(pAC, TX_RCB_BASE + i * 0x10 + TX_RCB_LOW_OFFS,
			LOW32(pAC->SndPrdRing[i].Phys));
		SK_MEM_WR32(pAC, TX_RCB_BASE + i * 0x10 + TX_RCB_LEN_FLAGS_OFFS,
			RING_MAX_LEN(TX_RING_ENTRIES_MAX));
		SK_MEM_WR32(pAC, TX_RCB_BASE + i * 0x10 + TX_RCB_NIC_ADDR_OFFS,
			TX_BD_ADDR + i * 0x10 * TX_RING_ENTRIES_MAX / 4);
//		SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_TX,
//			("Send Ring %d: NIC addr offset is 0x%x\n",
//			i, TX_BD_ADDR + i * 0x10 * TX_RING_ENTRIES_MAX / 4));
	}

	INITSTEP(31, "Disable unused receive rings");
	for (i = pAC->Config.RxRingNum; i < 16; i++) {
		SK_MEM_WR32(pAC, RX_RCB_BASE + i * 0x10 + RX_RCB_LEN_FLAGS_OFFS,
			DISABLE_RING);
	}

	INITSTEP(32, "Init receive ring(s)");
	/* This is not indicated in the init sequence list */
	for (i = 0; i < (int)pAC->Config.RxRingNum; i++) {
		SK_MEM_WR32(pAC, RX_RCB_BASE + i * 0x10 + RX_RCB_HIGH_OFFS,
			HIGH32(pAC->RecRetRing[i].Phys));
		SK_MEM_WR32(pAC, RX_RCB_BASE + i * 0x10 + RX_RCB_LOW_OFFS,
			LOW32(pAC->RecRetRing[i].Phys));
		SK_MEM_WR32(pAC, RX_RCB_BASE + i * 0x10 + RX_RCB_LEN_FLAGS_OFFS,
			RING_MAX_LEN(RX_RETURN_RING_ENTRIES_MAX));
		/* original drivers set this to 0 */
		SK_MEM_WR32(pAC, RX_RCB_BASE + i * 0x10 + RX_RCB_NIC_ADDR_OFFS, 0);
	}

	INITSTEP(33, "Init RX producer index registers");
	SK_OUT32(IoC, STD_RX_RING_PROD_IDX, 0);
	SK_OUT32(IoC, STD_RX_RING_PROD_IDX + 4, 0);
	SK_OUT32(IoC, JUM_RX_RING_PROD_IDX, 0);
	SK_OUT32(IoC, JUM_RX_RING_PROD_IDX + 4, 0);
	SK_OUT32(IoC, MIN_RX_RING_PROD_IDX, 0);
	SK_OUT32(IoC, MIN_RX_RING_PROD_IDX + 4, 0);

	INITSTEP(34, "Configure MAC unicast addr");
	SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
		("Mac Addr: %02x:%02x:%02x:%02x:%02x:%02x\n",
		pAC->Addr.Net.CurrentMacAddress.a[0],
		pAC->Addr.Net.CurrentMacAddress.a[1],
		pAC->Addr.Net.CurrentMacAddress.a[2],
		pAC->Addr.Net.CurrentMacAddress.a[3],
		pAC->Addr.Net.CurrentMacAddress.a[4],
		pAC->Addr.Net.CurrentMacAddress.a[5]));
	SkAddrOverride(pAC, IoC, &pAC->Addr.Net.CurrentMacAddress,
		SK_ADDR_LOGICAL_ADDRESS);

	INITSTEP(35, "Configure random backoff seed");
	for (i = 0, Data32 = 0; i < 6; i++) {
		Data32 += pAC->Addr.Net.PermanentMacAddress.a[i];
	}
	Data32 &= 0x3f;
	SK_OUT32(IoC, TX_RANDOM_BACKOFF, Data32);

	INITSTEP(36, "Configure the MTU");
	
	SK_OUT32(IoC, RX_MTU_SIZE, (pAC->Config.UseJumbo) ?
		 pAC->RecJumPrdRing.BufferSize : pAC->RecStdPrdRing.BufferSize);
	
	INITSTEP(37, "Configure IPG for transmit");
	SK_OUT32(IoC, TRANSMIT_LENGTHS,
		IPG_CRS_LEN(2) | IPG_LEN(6) | SLOT_TIME_LEN(0x20));

	INITSTEP(38, "Configure default RX ring");
	/* use only one ring */
	SK_OUT32(IoC, RX_RULES_CONFIG, RX_DEFAULT_CLASS(1));

	INITSTEP(39, "Configure number of RX lists");
	SK_OUT32(IoC, RX_LIST_PLACE_CONFIG,
		DEFAULT_INT_QUEUE(0) | BAD_FRAMES_CLASS(1) |
		NUMBER_OF_LISTS(pAC->Config.RxRingNum) | NUMBER_LISTS_GROUP(1));

	INITSTEP(40, "Configure RX list placement statistics");
	SK_OUT32(IoC, RX_LIST_PLACE_STATS_EM, 0xffffff);
	SK_OUT32(IoC, RX_LIST_PLACE_STATS_C, 1);

	INITSTEP(41, "Configure TX list placement statistics");
	SK_OUT32(IoC, TX_DAT_INIT_STATS_MASK, 0xffffff);
	SK_OUT32(IoC, TX_DAT_INIT_STATS_CTRL, 1);

	INITSTEP(42, "Disable host coalescing engine");
	SK_OUT32(IoC, HOST_COAL_MODE, 0x0);

	INITSTEP(43, "Poll for host coalescing engine to stop");
	Time = START_TIMEOUT();
	do {
		SK_IN32(IoC, HOST_COAL_MODE, &Data32);
		if (TIMEOUT_REACHED(Time, 100, 43)) {
			return (SK_STATUS_FAILED);
		}
	} while ((Data32 & 0xf) != 0);

	INITSTEP(44, "Configure host coal. tick count");
	SK_OUT32(IoC, RX_COAL_TICKS, pAC->Config.RxCoalTicks);
	SK_OUT32(IoC, TX_COAL_TICKS, pAC->Config.TxCoalTicks);

	INITSTEP(45, "Configure host coal. frame count");
	SK_OUT32(IoC, RX_MAX_COAL_BD_CNT, pAC->Config.RxCoalFrames);
	SK_OUT32(IoC, TX_MAX_COAL_BD_CNT, pAC->Config.TxCoalFrames);

	INITSTEP(46, "Configure during int tick count");
	SK_OUT32(IoC, RX_COAL_TICKS_INT, pAC->Config.RxCoalTicksInt);
	SK_OUT32(IoC, TX_COAL_TICKS_INT, pAC->Config.TxCoalTicksInt);

	INITSTEP(47, "Configure during int frame count");
	SK_OUT32(IoC, RX_MAX_COAL_BD_INT, pAC->Config.RxCoalFramesInt);
	SK_OUT32(IoC, TX_MAX_COAL_BD_INT, pAC->Config.TxCoalFramesInt);

	INITSTEP(48, "Init host status block addr");
	SK_OUT32(IoC, STATUS_BLOCK_HOST_HIGH, HIGH32(pAC->StatusBlockPhys));
	SK_OUT32(IoC, STATUS_BLOCK_HOST_LOW, LOW32(pAC->StatusBlockPhys));

	INITSTEP(49, "Init host statistics addr");
	SK_OUT32(IoC, STATISTICS_HOST_HIGH, HIGH32(pAC->StatisticsPhys));
	SK_OUT32(IoC, STATISTICS_HOST_LOW, LOW32(pAC->StatisticsPhys));

	INITSTEP(50, "Set statistics coalescing tick count");
	SK_OUT32(IoC, STATISTICS_TICKS, pAC->Config.StatsCoalTicks);
	
	INITSTEP(51, "Configure statistics block addr in NIC mem");
	SK_OUT32(IoC, STATISTICS_BASE_ADDR, 0x300);

	INITSTEP(52, "Configure status block addr in NIC mem");
	SK_OUT32(IoC, STATUS_BLOCK_BASE_ADDR, 0xb00);

	INITSTEP(53, "Enable host coalescing engine");
	SK_OUT32(IoC, HOST_COAL_MODE, COAL_ENG_ENABLE |
		COAL_CLR_TICKS_ON_RX_BD_EVENT    |
		COAL_CLR_TICKS_ON_TX_BD_EVENT    |
		COAL_NO_INT_ON_COAL_NOW_MODE  );

	INITSTEP(54, "Enable RX BD completion block");
	SK_OUT32(IoC, RX_BD_COMPL_MODE, RX_BD_COMPL_ATTN_ENAB | RX_BD_COMPL_ENAB);

	INITSTEP(55, "Enable RX list placement block");
	SK_OUT32(IoC, RX_LIST_PLACE_MODE, RX_LIST_PL_ENAB);

	INITSTEP(56, "Enable RX list selector block");
	SK_OUT32(IoC, RX_LIST_SEL_MODE, RX_LIST_SEL_ATTN_ENAB | RX_LIST_SEL_ENAB);

	INITSTEP(57, "Enable DMA engines and statistics");
	pAC->Hw.MacMode = ENAB_FHDE | ENAB_RDE | ENAB_TDE | /* MAC_LOOPBACK | */
		CLEAR_TX_STATS | ENAB_TX_STATS | CLEAR_RX_STATS | ENAB_RX_STATS;
	SK_OUT32(IoC, ETH_MAC_MODE, pAC->Hw.MacMode);

#ifdef DEBUG
	SK_IN32(IoC, ETH_MAC_MODE, &Data32);
	SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
		("GET MacMode=0x%04X (0x%04X)\n", Data32, pAC->Hw.MacMode));
#endif
	pAC->Hw.MacMode &= ~(CLEAR_TX_STATS | CLEAR_RX_STATS);
	
	INITSTEP(58, "Configure misc local ctrl reg");
	pAC->Hw.MiscLocal = INT_ON_ATTENTION | AUTO_SEEPROM_ACC |
		MISC_PINS_OUT_ENAB(BIT2 | BIT1 | BIT0) | MISC_PINS_OUT(0);
	if (pAC->Hw.IsCopper) {
		/*
		 * set GPIO[2] to high to enable PCI voltage (VBUS_EN)
		 * set GPIO[1] to high to enable Phy BCM5411 (/RESET)
		 */
		pAC->Hw.MiscLocal |= MISC_PINS_OUT(BIT2 | BIT1);
	}
	SK_OUT32(IoC, MISC_LOCAL, pAC->Hw.MiscLocal);

	/* NEW! */
	/* Enable controller interrupts. */
	SK_OUT32(IoC, INTERRUPT_MAILBOX0, 0);
	SK_OUT32(IoC, INTERRUPT_MAILBOX0 + 4, 0);

	INITSTEP(59, "Enable DMA completion block");
	SK_OUT32(IoC, DMA_COMPL_MODE, DMA_COMPL_ENAB);

	INITSTEP(60, "Configure Write DMA regs");
	SK_OUT32(IoC, WRITE_DMA_MODE,
		WRITE_DMA_ENAB | PCI_ABORT_TARGET | PCI_MASTER_ABORT |
		PCI_PARITY | PCI_HOST_ADDR_OV | PCI_FIFO_OVERRUN |
		PCI_FIFO_UNDERRUN | PCI_FIFO_OVERREAD | READ_LONGER_DMA);

	INITSTEP(61, "Configure Read DMA regs");
	SK_OUT32(IoC, READ_DMA_MODE,
		READ_DMA_ENAB | PCI_ABORT_TARGET | PCI_MASTER_ABORT |
		PCI_PARITY | PCI_HOST_ADDR_OV | PCI_FIFO_OVERRUN |
		PCI_FIFO_UNDERRUN | PCI_FIFO_OVERREAD | WRITE_LONGER_DMA);

	INITSTEP(62, "Enable RX data completion block");
	SK_OUT32(IoC, RX_DAT_COMPL_MODE, RX_DAT_COMPL_ENAB | RX_DAT_ATTN_ENAB);

	INITSTEP(63, "Enable mbuf cluster free block");
	SK_OUT32(IoC, MBUF_CLUST_FREE_MODE, MBUF_CLUST_FREE_ENAB);

	INITSTEP(64, "Enable TX data completion block");
	SK_OUT32(IoC, TX_DAT_COMPL_MODE, TX_DAT_COMPL_ENAB);

	INITSTEP(65, "Enable TX BD completion block");
	SK_OUT32(IoC, TX_BD_COMPL_MODE, TX_BD_COMPL_ENAB | TX_BD_COMPL_ATTN_ENAB);

	INITSTEP(66, "Enable RX BD initiator block");
	SK_OUT32(IoC, RX_BD_INIT_MODE, RX_BD_AVAIL | RX_BD_INIT_ENAB);

	INITSTEP(67, "Enable RX data and BD initiator block");
	SK_OUT32(IoC, RX_DAT_RX_BD_MODE, RX_DAT_RX_BD_ENAB | ENAB_ILL_SIZE_ATTN);

	INITSTEP(68, "Enable TX data initiator block");
	SK_OUT32(IoC, TX_DAT_INIT_MODE, TX_DAT_INIT_ENAB);

	INITSTEP(69, "Enable TX BD initiator block");
	SK_OUT32(IoC, TX_BD_INIT_MODE, TX_DAT_INIT_ENAB | ENAB_TX_BD_INIT_ATTN);

	INITSTEP(70, "Enable TX BD selector block");
	SK_OUT32(IoC, TX_BD_SEL_MODE, TX_BD_SEL_ENAB | ENAB_TX_BD_SEL_ATTN);

	INITSTEP(71, "Enable TX MAC and flow control");
	pAC->Addr.Port.TxMode = TX_MAC_ENAB;
	SK_OUT32(IoC, TRANSMIT_MODE, pAC->Addr.Port.TxMode);

	INITSTEP(72, "Enable RX MAC and flow control");
	SK_OUT32(IoC, RECEIVE_MODE, pAC->Addr.Port.RxMode);

	/* Prevent device from violating PCI spec */
	if (pAC->Config.EnableNwi) {
		/*
		 * Set DMA write address boundary.
		 * only necessary for Rev. B?
		 */

		SK_PCI_IN32(pAC, PCI_DMA_RW_CTRL, &Data32);

		Data32 &= ~DMA_WR_BND(7);

		switch (pAC->Hw.CacheLineSize) {
		case 16:
			Data32 |= DMA_WR_BND(1);
			break;

		case 32:
			Data32 |= DMA_WR_BND(2);
			break;

		case 64:
			Data32 |= DMA_WR_BND(3);
			break;

		case 128:
			Data32 |= DMA_WR_BND(4);
			break;

		case 256:
			Data32 |= DMA_WR_BND(5);
			break;

		case 512:
			Data32 |= DMA_WR_BND(6);
			break;

		case 1024:
			Data32 |= DMA_WR_BND(7);
			break;

		default:
			break;
		}

		SK_PCI_OUT32(pAC, PCI_DMA_RW_CTRL, Data32);
	}

	INITSTEP(73, "Configure auto-polling the (G)MII? mgt? interface");
	SK_OUT32(IoC, MI_MODE, MI_MODE_RESERVED);

	INITSTEP(74, "Configure D0 power state");
	(void)SetPowerState(pAC, IoC, PM_POWER_STATE_D0);

	INITSTEP(75, "Program HW controlled LEDs");
	SK_OUT32(IoC, LED_CONTROL, 0);

	INITSTEP(76, "Activate Link and enable MAC block");
	SK_OUT32(IoC, MI_STATUS, LINK_STATUS);

	INITSTEP(77, "Setup Phy and restart Auto-negotiation");
	return (SkGiInitPhy(pAC, IoC));
} /* SkGiInitAdapter */


/*****************************************************************************
 *
 * 	SkGiHwacInit - initialize hardware, check link configuration
 *
 * Description:
 *	This function initializes the hardware and checks the link configuration.
 *
 * Returns:	SK_STATUS
 *
 */
SK_STATUS SkGiHwacInit(
SK_AC	*pAC,	/* Pointer to the adapter context structure */
SK_IOC	IoC,	/* I/O Context */
SK_U16	Level)	/* Initialization level */
{
	/*
	 *	The two parameters AutoNeg and DupCap map to one configuration
	 *	parameter. The mapping is described by this table:
	 *	DupCap ->	|	Both		|	Full		|	Half		|
	 *	AutoNeg		|				|				|				|
	 *	-------------------------------------------------------------
	 *	Off			|   -illegal-	|	Full		|	Half		|
	 *	-------------------------------------------------------------
	 *	On			|   AutoBoth	|   AutoFull	|   AutoHalf	|
	 *	-------------------------------------------------------------
	 *	Sense		|   AutoSense	|   -illegal-	|   -illegal-	|
	 */
	SK_U32	Capabilities[3][3] =
		{
			{(SK_U32)-1,			SK_LMODE_FULL,		SK_LMODE_HALF},
			{SK_LMODE_AUTOBOTH,		SK_LMODE_AUTOFULL,	SK_LMODE_AUTOHALF},
			{SK_LMODE_AUTOSENSE,	(SK_U32)-1,			(SK_U32)-1}
		};
	SK_U32	Data32;
	SK_U16	Data16;
	SK_STATUS	RetValue;
	int		i;
#ifdef DEBUG
	SK_U8	PartType[8];
#endif

	RetValue = SK_STATUS_OK;

	switch (Level) {
	case SK_INIT_DATA:
		/* Do some initialization ... */
		pAC->Hw.MiscHost = ENAB_INDIR_ADDR | ENAB_PCI_STATE |
#ifdef SK_LITTLE_ENDIAN
			ENAB_EW_SWAP |
#endif	/* SK_LITTLE_ENDIAN */
			MASK_PCI_INT;
		break;	/* INIT_DATA */

	case SK_INIT_IO:
		SK_PCI_IN32(pAC, PCI_CACHE_LINE_SIZE, &Data32);
		pAC->Hw.CacheLineSize = (SK_U8)Data32;
		pAC->Hw.SavedCacheLineReg = Data32;

		/* Get PCI subvendor ID and subsytem ID */
		SK_PCI_IN32(pAC, PCI_SUBSYSTEM_VENDOR_ID, &Data32);
		pAC->Hw.SubsystemVendorId = Data32;

		SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
			("SubsystemVendorId=0x%08X\n", Data32));
		
		/* Get PCI revision ID */
		SK_PCI_IN32(pAC, PCI_CLASS_REVISION, &Data32);
		pAC->Hw.RevisionId = (SK_U8)Data32;
		pAC->Hw.ClassCode = Data32 >> 8;

		SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
			("RevisionId=0x%02X\n", pAC->Hw.RevisionId));
		
		/* Get chip revision ID */
		SK_PCI_IN32(pAC, PCI_MISC_H_CTRL, &Data32);
		pAC->Hw.ChipRevId = (SK_U16)(Data32 >> 16);

		SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
			("ChipRevId=0x%04X\n", pAC->Hw.ChipRevId));
		
		SK_PCI_IN32(pAC, PCI_STATE_REG, &Data32);
		pAC->Hw.PciState = (SK_U16)Data32;
		
		SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
			("PCI State=0x%04X\n", pAC->Hw.PciState));

		if ((Data32 & PCI_BUS_32_BIT) == 0) {
			SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
				("ATTENTION: 64-bit PCI bus\n"));
		}
		if ((Data32 & CONVENTIONAL_PCI_MODE) == 0) {
			SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
				("ATTENTION: PCI-X bus\n"));
		}
		
		if (pAC->Config.AutoNeg != SK_AUTONEG_OFF) {
			pAC->Hw.ReqMediaType = REQ_MEDIA_TYPE_AUTO;
		}
		
		pAC->Hw.FlowControl = FLOW_CONTROL_AUTO_PAUSE
			| FLOW_CONTROL_RECEIVE_PAUSE | FLOW_CONTROL_TRANSMIT_PAUSE;
		
		if (SK_STAT_CMP_FAILED(SkInitAutoI2cAccess(pAC, IoC))) {
			return (SK_STATUS_FAILED);
		}
#ifdef DEBUG
		SkReadI2cDword(pAC, IoC, 0, EEPROM_BOOT_CODE_OFS, &Data32);
#ifdef SK_LITTLE_ENDIAN
		Data32 = SK_BYTE_SWAP32(Data32);
#endif	/* SK_LITTLE_ENDIAN */
		SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
			("BootCodeOffset=0x%08X\n", Data32));
		
		if (Data32 == 0x200) {
			SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
				("Contents of serial EEPROM too old!\n"));
		}
#endif	/* DEBUG */

		/*
		 * Determine if copper or fiber adapter from PCI SubsystemId
		 * Later check if TBI is enabled in HWAC module.
		 */
		
		for (i = 0; i < 2; i++) {
			SkReadI2cDword(pAC, IoC, 0, EEPROM_PART_NUMBER + 4*i, &Data32);

			SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
				("PartType_%u=0x%08X\n", i, Data32));
#ifdef DEBUG
			*((SK_U32 *)PartType + i) = Data32;
#endif
		}
		SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
			("PartType=%s\n", PartType));
		
		if (SK_STAT_CMP_FAILED(SkReadI2cDword(pAC, IoC, 0, EEPROM_BOARD_INFO, &Data32))) {
			return (SK_STATUS_FAILED);
		}

		SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
			("Revision=%c.%c, (0x%08X)\n",
			 (SK_U8)Data32, (SK_U8)(Data32 >> 8), Data32));

		pAC->Hw.PmdType = (SK_U8)(Data32 >> 16);

		SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
			("PMD Type='%c', Con. Type='%c'\n",
			 pAC->Hw.PmdType, (SK_U8)(Data32 >> 24)));
		
		pAC->Hw.IsCopper = (pAC->Hw.PmdType == 'T');
		
#if 0
		pAC->Hw.IsCopper = (SK_U16)(pAC->Hw.SubsystemVendorId >> 16) ==
			PCI_SUB_DEVICE_ID_SYSKONNECT_SK9D21;
#endif /* 0 */
		
		if (pAC->Hw.IsCopper) {

			if (pAC->Hw.ReqMediaType == REQ_MEDIA_TYPE_AUTO) {
				pAC->Hw.ReqMediaType = REQ_MEDIA_TYPE_UTP_AUTO;
			}
			
#ifdef DEBUG
			SK_IN32(IoC, MISC_LOCAL, &Data32);
			SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
				("MiscLocCtrl=0x%08X\n", Data32));
			
			SK_IN32(IoC, MDI_CTRL, &Data32);
			SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
				("MDI Control=0x%08X\n", Data32));
#endif	
	
			/* Get the PHY Id */
			SkGiReadPhy(pAC, IoC, PHY_ID1_REG, &Data16);
/*
			SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
				("PhyId1=0x%04X\n", Data16));
*/	
			pAC->Hw.PhyIdentifier = ((SK_U32)Data16 & ID1_OUI_MASK) << 10;
	
			SkGiReadPhy(pAC, IoC, PHY_ID2_REG, &Data16);
/*
			SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
				("PhyId2=0x%04X\n", Data16));
*/	
			pAC->Hw.PhyIdentifier |= (((SK_U32)Data16 & ID2_OUI_MASK) << 16) |
				(Data16 & ID2_MODEL_MASK) | (Data16 & ID2_REV_MASK);
	
			SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
				("PhyIdentifier=0x%08X\n", pAC->Hw.PhyIdentifier));
	
			if ((pAC->Hw.PhyIdentifier & PHY_ID_MASK) == PHY_BCM5411_PHY_ID) {
				 pAC->Hw.PhyIdentifier = PHY_BCM5411_PHY_ID;
			}
			
			SkGiReadPhy(pAC, IoC, PHY_CTRL_REG, &Data16);
			SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
				("PhyControl=0x%04X\n", Data16));
	
			SkGiReadPhy(pAC, IoC, BCM540X_AUX_STATUS, &Data16);
			SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
				("PhyAuxStat=0x%04X\n", Data16));
	
			SkGiReadPhy(pAC, IoC, PHY_STATUS_REG, &Data16);
			SkGiReadPhy(pAC, IoC, PHY_STATUS_REG, &Data16);
			SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
				("PhyStatus=0x%04X, Link:%c, Auto:%c\n", Data16,
				 ((Data16 & LINK_PASS) != 0) ? 'U': 'D',
				 ((Data16 & AUTO_NEG_COMPLETE) != 0) ? 'C': '-'));
	
#if 0
			/* Reset the PHY if we don't have link */
			if ((Data16 & LINK_PASS) == 0) {
				SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
					("Reset CopperPhy\n"));
				SkGiWritePhy(pAC, IoC, PHY_CTRL_REG, PHY_RESET);
	
				for (i = 0; i < 5000; i++) {
					SkOsWaitUsec(pAC, 10);
	
					SkGiReadPhy(pAC, IoC, PHY_CTRL_REG, &Data16);
					if ((Data16 != 0) && ((Data16 & PHY_RESET) == 0)) {
						SkOsWaitUsec(pAC, 40);
						break;
					}
				}
#ifdef DEBUG
				SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
					("PhyControl=0x%04X (%u)\n", Data16, i));
				
				if ((Data16 == 0) || ((Data16 & PHY_RESET) != 0)) {
					SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
						("CopperPhy Reset not done\n"));
					return (SK_STATUS_FAILED);
				}
#endif
	
//				SkOsWaitMsec(pAC, 400);
	
				SkGiReadPhy(pAC, IoC, PHY_STATUS_REG, &Data16);
				SkGiReadPhy(pAC, IoC, PHY_STATUS_REG, &Data16);
				SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
				("PhyStatus=0x%04X, Link:%c, Auto:%c\n", Data16,
				 ((Data16 & LINK_PASS) != 0) ? 'U': 'D',
				 ((Data16 & AUTO_NEG_COMPLETE) != 0) ? 'C': '-'));
			}
#endif /* 0	*/
	
			SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
				("Auto-Negotiation %s\n",
				((Data16 & AUTO_NEG_COMPLETE) == 0) ?
				 "in progres" : "completed"));
		}

		/* Check for illegal combinations of AutoNeg and DupCap */
		if (pAC->Config.AutoNegSet) {
			
			if (pAC->Config.AutoNeg == SK_AUTONEG_SENS &&
				pAC->Config.DupCapSet) {

				SK_ERR_LOG(pAC, SK_ERRCL_CONFIG, SKERR_HWI_E001,
					("AutoNeg: DupCap ignored, using Sense mode\n"));
				pAC->Config.DupCap = SK_DUPCAP_BOTH;
			}
			else if (pAC->Config.AutoNeg == SK_AUTONEG_OFF) {

				if (pAC->Config.DupCapSet) {
					if (pAC->Config.DupCap == SK_DUPCAP_BOTH) {
						SK_ERR_LOG(pAC, SK_ERRCL_CONFIG, SKERR_HWI_E001,
							("Illegal combination of AutoNeg and DupCap\n"
							"    Using Full Duplex\n"));
						pAC->Config.DupCap = SK_DUPCAP_FULL;
					}
				}
				else {
					pAC->Config.DupCap = SK_DUPCAP_FULL;
				}
			}
		}
        else if (pAC->Config.DupCapSet) {
//   		printk("%s: Duplex setting not possible in\n"
//				"    default Auto-negotiation mode (Sense)\n"
//				"    Using Auto-negotiation On\n", pAC->dev->name);
			SK_ERR_LOG(pAC, SK_ERRCL_CONFIG, SKERR_HWI_E001, SKERR_HWI_E001MSG);
			SK_ERR_LOG(pAC, SK_ERRCL_CONFIG, SKERR_HWI_E002, SKERR_HWI_E002MSG);
			SK_ERR_LOG(pAC, SK_ERRCL_CONFIG, SKERR_HWI_E003, SKERR_HWI_E003MSG);
			pAC->Config.AutoNeg = SK_AUTONEG_ON;
		}

		/*
		 * Check for illegal combinations of adapter type,
		 * AutoNeg, and LinkSpeed
		 */
		if (!pAC->Hw.IsCopper && pAC->Config.LinkSpeedSet &&
			pAC->Config.LinkSpeed != SK_LSPEED_AUTO &&
			pAC->Config.LinkSpeed != SK_LSPEED_1000MBPS) {

//			printk("%s: Speed not available for fiber adapters\n",
//				pAC->dev->name);
			SK_ERR_LOG(pAC, SK_ERRCL_CONFIG, SKERR_HWI_E006, SKERR_HWI_E006MSG);
			pAC->Config.LinkSpeed = SK_LSPEED_AUTO;
		}

		if (pAC->Config.AutoNegSet) {
			
			if (pAC->Config.AutoNeg == SK_AUTONEG_SENS &&
				pAC->Config.LinkSpeedSet &&
				pAC->Config.LinkSpeed != SK_LSPEED_AUTO) {
	
//				printk("%s: AutoNeg=Sense -> Speed ignored\n",
//					pAC->dev->name);
				SK_ERR_LOG(pAC, SK_ERRCL_CONFIG, SKERR_HWI_E007, SKERR_HWI_E007MSG);
				pAC->Config.LinkSpeed = SK_LSPEED_AUTO;
			}
			
			if (pAC->Config.AutoNeg == SK_AUTONEG_OFF) {
				
				if (pAC->Config.LinkSpeedSet) {
					if (pAC->Config.LinkSpeed == SK_LSPEED_AUTO) {
//						printk("%s: Illegal combination of values AutoNeg and Speed\n"
//							"    Using 1000 Mbps\n", pAC->dev->name);
						SK_ERR_LOG(pAC, SK_ERRCL_CONFIG, SKERR_HWI_E011,
							SKERR_HWI_E011MSG);
						pAC->Config.LinkSpeed = SK_LSPEED_1000MBPS;
					}
				}
				else {
					pAC->Config.LinkSpeed = SK_LSPEED_1000MBPS;
				}
			}
		}
		else if (pAC->Config.LinkSpeedSet) {
//			printk("%s: LinkSpeed setting not possible in\n"
//				"    default Auto-negotiation mode (Sense)\n"
//				"    Using Auto-negotiation On\n", pAC->dev->name);
			SK_ERR_LOG(pAC, SK_ERRCL_CONFIG, SKERR_HWI_E004, SKERR_HWI_E004MSG);
			SK_ERR_LOG(pAC, SK_ERRCL_CONFIG, SKERR_HWI_E002, SKERR_HWI_E002MSG);
			SK_ERR_LOG(pAC, SK_ERRCL_CONFIG, SKERR_HWI_E003, SKERR_HWI_E003MSG);
			pAC->Config.AutoNeg = SK_AUTONEG_ON;
		}

		/* Set the desired link mode */
		pAC->Hw.LinkMode =
			Capabilities[pAC->Config.AutoNeg][pAC->Config.DupCap];
		if (pAC->Hw.LinkMode == (SK_U32)-1) {
//			printk("%s: Illegal link mode\n", pAC->dev->name);
			SK_ERR_LOG(pAC, SK_ERRCL_CONFIG, SKERR_HWI_E005, SKERR_HWI_E005MSG);
		}

		/* Check for illegal combinations of AutoNeg and FlowCtrl */
		if (pAC->Config.AutoNeg == SK_AUTONEG_OFF && !pAC->Config.FlowCtrlSet) {
			pAC->Config.FlowCtrl = SK_FLOW_MODE_NONE;
		}

		if (pAC->Config.AutoNeg == SK_AUTONEG_OFF &&
			pAC->Config.FlowCtrl != SK_FLOW_MODE_NONE) {

//			printk("%s: Flow Control impossible without Auto-negotiation, "
//				"disabled\n", pAC->dev->name);
			SK_ERR_LOG(pAC, SK_ERRCL_CONFIG, SKERR_HWI_E008, SKERR_HWI_E008MSG);
			pAC->Config.FlowCtrl = SK_FLOW_MODE_NONE;
		}

		/* Check for illegal combinations of adapter type, Speed, and Role */
		if (pAC->Config.MsModeSet && !pAC->Hw.IsCopper) {
//			printk("%s: Role not available for fiber adapters\n",
//				pAC->dev->name);
			SK_ERR_LOG(pAC, SK_ERRCL_CONFIG, SKERR_HWI_E009, SKERR_HWI_E009MSG);
			pAC->Config.MsMode = SK_MS_MODE_AUTO;
		}

		if (pAC->Config.MsMode != SK_MS_MODE_AUTO &&
			(pAC->Config.LinkSpeed == SK_LSPEED_100MBPS ||
			 pAC->Config.LinkSpeed == SK_LSPEED_10MBPS)) {

			SK_ERR_LOG(pAC, SK_ERRCL_CONFIG, SKERR_HWI_E010, SKERR_HWI_E010MSG);
//			printk("%s: Role available only in 1000 Mbps mode\n",
//				pAC->dev->name, pAC->Config.LinkSpeed);
			pAC->Config.MsMode = SK_MS_MODE_AUTO;
		}
		break;	/* INIT_IO */

	case SK_INIT_RUN:
		RetValue = SkGiInitAdapter(pAC, IoC);
		break;	/* INIT_RUN */

	default:
		break;
	}

	return (RetValue);
} /* SkGiHwacInit */


/*****************************************************************************
 *
 *	SkGiHwEvent - handle HWAC events
 *
 * Description:
 *	This routine handles events for HWAC.
 *
 * Returns: SK_STATUS
 *
 */
SK_STATUS SkGiHwEvent(
SK_AC		*pAC,	/* Pointer to Adapter Context */
SK_IOC		IoC,
SK_U32		Event,
SK_EVPARA	Para)
{
	switch (Event) {
	case SK_HWEV_PORT_START:
		SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_QUEUE,
			("SkGiHwEvent: SK_HWEV_PORT_START\n"));
#if 0
		if (pPrt->PHWLinkUp) {
			/*
			 * Signal directly to RLMT to ensure correct
			 * sequence of SWITCH and RESET event.
			 */
			SkRlmtEvent(pAC, IoC, SK_RLMT_LINK_DOWN, Para);
		}

		SkHwLinkDown(pAC, IoC, Port);
#endif	/* 0 */

		/* Schedule Port RESET */
		SkEventQueue(pAC, SKGI_DRV, SK_DRV_PORT_RESET, Para);
		break;

	case SK_HWEV_PORT_STOP:
		SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_QUEUE,
			("SkGiHwEvent: SK_HWEV_PORT_STOP\n"));
#if 0
		if (pAC->GIni.GP[Port].PHWLinkUp) {
			/*
			 * Signal directly to RLMT to ensure correct
			 * sequence of SWITCH and RESET event.
			 */
			SkRlmtEvent(pAC, IoC, SK_RLMT_LINK_DOWN, Para);
		}

		SkHwLinkDown(pAC, IoC, Port);
#endif	/* 0 */

		SkRlmtEvent(pAC, IoC, SK_RLMT_LINK_DOWN, Para);
		
		if (SkGiHwacDeinit(pAC,	IoC) != SK_STATUS_OK) {
			SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
				("SkGiHwacDeinit failed!\n"));
		}
		break;

	default:
		SK_DBG_MSG(pAC, SK_DBGMOD_HWAC, SK_DBGCAT_INIT,
			("SkGiHwEvent %d\n", Event));
		break;
	}

	return (SK_STATUS_OK);
} /* SkGiHwEvent */


/*
 * Local variables:
 * compile-command: "make"
 * End:
 */


