/*
 * 
 * proc file system support for the BadMEM patch
 * 
 * Written by Nico Schmoigl, nico@writemail.com, August 2000
 * 
 */

#include <linux/kernel.h>
#include <linux/types.h>
#include <linux/proc_fs.h>
#include <linux/init.h>
#include <linux/mm.h>
#include <linux/bootmem.h>
#include <asm/byteorder.h>
#include <linux/badmem.h>

#ifdef CONFIG_BADMEM_MODSYSTEM
#include <badmemlib.h>
#endif /* CONFIG_BADMEM_MODSYSTEM */

#define BADMEM_MODDIR "badmem/modules"

static int badmem_proc_printmap (char *b) {
   int totallen, thislen, badpages = 0;
   int i, nopageprint = 0;

   totallen = thislen = sprintf (b, "Marked as bad are:\n");
   b += thislen;
   
   for (i=0;i<max_low_pfn;i++) {
	 char n[100];

      if (!PageBad(mem_map+i))
	continue;
      
      // this is a bad page its offset is in i

      if (++badpages >= 32) {
	      nopageprint = 1; // otherwise this would kill proc fs! (Overflow issue)
	      continue;
      }

      sprintf (n, "%s%s%s%s%s", page_is_ram(i) ? "isram " : "", PageReserved(mem_map+i) ? "reserved " : "",
	       Page_Uptodate(mem_map+i) ? "uptodate " : "", PageLocked(mem_map+i) ? "locked " : "",
	       PageSlab(mem_map+i) ? "slab" : "");
	 
      thislen = sprintf (b, "page #%i, mem 0x%08x-0x%08x, counter %u, %s\n", i, i << PAGE_SHIFT, ((i+1) << PAGE_SHIFT) - 1, atomic_read(&mem_map[i].count), n); 
      totallen += thislen;
      b += thislen;

   }
   
   if (nopageprint) {
      thislen = sprintf (b, "[...more...]\n");
      totallen += thislen;
      b+= thislen;
   }
   
   thislen = sprintf (b, "\nSummary: %i pages (=%ik) are marked as bad\n", badpages, badpages << (PAGE_SHIFT - 10)); 
   totallen += thislen;
   b += thislen;
   
   return totallen;
}


static int badmem_read_proc (char *buf, char **start, off_t off, int count, int *eof, void *data)
{
   char *dat = buf;
   
   int len, totallen = 0;
   register int i;
   
   len = sprintf (dat, "BADMEM driver version " BADMEM_VERSION "\nBADMEM features: " BADMEM_FEATURES "\n");
   totallen += len;
   dat += len;

#ifndef CONFIG_BADMEM_MODSYSTEM
   len = sprintf (dat, "BADMEM boot options: ");
   totallen += len;
   dat += len;
   
   /*
    * Get all parameters from the array and put them 
    */
   for (i=1;i<=badmem_paramcount;i++) {
      len = sprintf (dat, "0x%08lx ", badmem_params[i]);
      totallen += len;
      dat += len;
   }
   
   len = sprintf (dat, "\n\n");
   totallen += len;
   dat += len;
#endif /* CONFIG_BADMEM_MODSYSTEM */      

#ifdef CONFIG_BADMEM_MODSYSTEM
	// tell which modules are known to the kernel
	len = sprintf (dat, "BADMEM known modules: ");
	totallen += len; dat += len;
	
	i = 0;
	while (badmem_modtab[i].mod_name != NULL) {
		len = sprintf (dat, "%s,", badmem_modtab[i].mod_name);
		totallen += len; dat += len;
		i++;
	}
	// now dat has one comma at the end or at least a whitespace (on the case that the modtab was empty)...
	*(dat-1) = '\n';
#endif /* CONFIG_BADMEM_MODSYSTEM */
	
	
   // Now print the map
   totallen += badmem_proc_printmap (dat);
   
   // now we have to finish to proc read request... the buffer is already ok, but len, eof and start need
   // another update...
   *eof = 1;
   *start = buf+off;
   totallen -= off;
   
   if (totallen > count)
     totallen = count;
   
   if (totallen < 0)
     totallen = 0;
   
   return totallen;
   
}

#ifdef CONFIG_BADMEM_MODSYSTEM
/*
 * The read routine for the mdf files in the proc/badmem/modules/mdf directory
 */
static int lock = 0;

static int badmem_mdf_read_proc (char *buf, char **start, off_t off, int count, int *eof, void *data) {
	struct badmem_modules *dat = (struct badmem_modules *) data;
	register int i = 0, dummy;
	char *tmpbuffer;
	int internalcount;
	
	while (lock);
	
	lock = 1; // locking on;
	
	if (!data) 
		return 0;


#ifdef CONFIG_BADMEM_DEBUG
	printk (KERN_DEBUG "BadMEM: %p belongs to '%s'\n", data, dat->mod_name);
#endif	
	
        tmpbuffer = dat->mdfdata;
        internalcount = dat->mdflen;
        i = internalcount;  
   
#define min(a,b) ((a<b)?a:b)	

	*start = buf;

	dummy = min(i-off, min(count, internalcount-off));
	if (dummy < 0)
		dummy = 0;

	if (dummy < count)
		*eof = 1;
	else
		*eof = 0;

	if (i-off == dummy)
		*eof = 1;
	
#ifdef CONFIG_BADMEM_DEBUG
        printk (KERN_DEBUG "BadMEM: requested offset is %li\n", off);
	printk (KERN_DEBUG "BadMEM: able to pass %li bytes, sending %u bytes, eof=%u\n", i-off, dummy, *eof);
#endif	
        if (dummy > 0)
		memcpy (buf, tmpbuffer+off, dummy);
	
	lock = 0;
	return dummy;
#undef min
}

/*
 * Registers all modules in /proc/badmem/modules
 */
static struct proc_dir_entry *moddir_proc_entry;

static void __init register_modules_in_proc(void) {
	register int i = 0;
	struct proc_dir_entry *dentry;
	
	dentry = proc_mkdir("mdf", moddir_proc_entry);

#ifdef CONFIG_BADMEM_DEBUG
	printk ("BadMEM: register 'mdf' procfs-reporting: %p\n", dentry);
#endif /* CONFIG_BADMEM_DEBUG */	
	
	if (!dentry)  // something went wrong...
		return;

	while (badmem_modtab[i].mod_name != NULL) {
		create_proc_read_entry (badmem_modtab[i].mod_name, 0, dentry, badmem_mdf_read_proc, (void *) &badmem_modtab[i]);
#ifdef CONFIG_BADMEM_DEBUG
		printk ("BadMEM: created mdf entry %s in procfs\n", badmem_modtab[i].mod_name);
#endif /* CONFIG_BADMEM_DEBUG */
		i++;
	}

	printk ("BadMEM: summary - registered %i modules with mdf in procfs\n", i);
	
	return;
}

#endif /* CONFIG_BADMEM_MODSYSTEM */

static int __init badmem_proc_init(void) {
#ifdef CONFIG_BADMEM_MODSYSTEM
   moddir_proc_entry = proc_mkdir ("badmem", 0);
   
   if (moddir_proc_entry) {
	   register_modules_in_proc();
	   create_proc_read_entry ("badmem/summary", 0, NULL, badmem_read_proc, NULL);
   } else {
	   create_proc_read_entry ("badmem", 0, NULL, badmem_read_proc, NULL);
   }
#else
	create_proc_read_entry("badmem", 0, NULL, badmem_read_proc, NULL);
#endif	/* CONFIG_BADMEM_MODSYSTEM */
	   
   return 0;
}


__initcall(badmem_proc_init);
