/*
 This file is part of GNU Taler
 (C) 2021-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
import { AccessToken, TranslatedString } from "@gnu-taler/taler-util";
import {
  ButtonBetterBulma,
  Notification,
  useLocalNotificationBetter,
  useTranslationContext,
} from "@gnu-taler/web-util/browser";
import { Fragment, h, VNode } from "preact";
import { useState } from "preact/hooks";
import { useSessionContext } from "../../context/session.js";

function NotificationFieldFoot({
  notification,
}: {
  notification?: Notification;
}): VNode | null {
  if (!notification) return null;
  if (notification.message.type === "error")
    return (
      <p class="help is-danger" style={{ fontSize: 16 }}>
        {notification.message.description}
      </p>
    );
  return null;
}

export function JumpToElementById({
  onSelect,
  placeholder,
  description,
}: {
  placeholder: TranslatedString;
  description: TranslatedString;
  onSelect: (id: string) => void;
}): VNode {
  const { i18n } = useTranslationContext();
  const [id, setId] = useState<string>();

  const { state: session, lib } = useSessionContext();
  const [notification, safeFunctionHandler] = useLocalNotificationBetter();
  const checkExist = safeFunctionHandler(
    (token: AccessToken, id: string) =>
      lib.instance.getProductDetails(token, id),
    !session.token || !id ? undefined : [session.token, id],
  );
  checkExist.onSuccess = (success, t, id) => {
    onSelect(id);
  };
  checkExist.onFail = () => i18n.str`Not found`;

  return (
    <Fragment>
      <div class="level">
        <div class="level-left">
          <div class="level-item">
            <form>
              <div class="field has-addons">
                <div class="control">
                  <input
                    class={"input"}
                    type="text"
                    value={id ?? ""}
                    onChange={(e) => {
                      setId(e.currentTarget.value)
                      if (notification) notification.acknowledge()
                    }}
                    placeholder={placeholder}
                  />
                  <NotificationFieldFoot notification={notification} />
                </div>
                <span class="has-tooltip-bottom" data-tooltip={description}>
                  <ButtonBetterBulma
                    class="button"
                    type="submit"
                    onClick={checkExist}
                  >
                    <span class="icon">
                      <i class="mdi mdi-arrow-right" />
                    </span>
                  </ButtonBetterBulma>
                </span>
              </div>
            </form>
          </div>
        </div>
      </div>
    </Fragment>
  );
}
