/*
 This file is part of GNU Taler
 (C) 2021-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

import {
  Codec,
  buildCodecForObject,
  canonicalizeBaseUrl,
  codecForList,
  codecForString,
  codecOptional
} from "@gnu-taler/taler-util";

export interface MerchantUiSettings {
  // Where merchant backend is localted
  // default: window.origin without "webui/"
  backendBaseURL?: string;

}

/**
 * Global settings for the bank UI.
 */
const defaultSettings: MerchantUiSettings = {
  backendBaseURL: buildDefaultBackendBaseURL(),
};

const codecForBankUISettings = (): Codec<MerchantUiSettings> =>
  buildCodecForObject<MerchantUiSettings>()
    .property("backendBaseURL", codecOptional(codecForString()))
    .build("MerchantUiSettings");

function removeUndefineField<T extends object>(obj: T): T {
  const keys = Object.keys(obj) as Array<keyof T>;
  return keys.reduce((prev, cur) => {
    if (typeof prev[cur] === "undefined") {
      delete prev[cur];
    }
    return prev;
  }, obj);
}

export function fetchSettings(listener: (s: MerchantUiSettings) => void): void {
  fetch("./settings.json")
    .then((resp) => resp.json())
    .then((json) => codecForBankUISettings().decode(json))
    .then((result) =>
      listener({
        ...defaultSettings,
        ...removeUndefineField(result),
      }),
    )
    .catch((e) => {
      console.log("failed to fetch settings", e);
      listener(defaultSettings);
    });
}

export function buildDefaultBackendBaseURL(): string {
  if (typeof window !== "undefined") {
    const currentLocation = new URL(
      window.location.pathname,
      window.location.origin,
    ).href;
    /**
     * By default, merchant backend serves the html content
     * from the /webui root. This should cover most of the 
     * cases and the rootPath will be the merchant backend
     * URL where the instances are
     */
    return canonicalizeBaseUrl(currentLocation.replace("/webui", ""));
  }
  throw Error("No default URL")
}
