/*
 *  Ball And Paddle
 *
 *  Copyright (C) 2007 by Eric Hutchins
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 **/

#include "optionsmenu.h"
#include "skinmenu.h"
#include "mainmenu.h"

OptionsMenu
  OptionsMenu::m_instance;

void
OptionsMenu::init ()
{
  menuItems.push_back ("Fullscreen [ ]");
  menuItems.push_back ("Play Sounds [ ]");
  menuItems.push_back ("Change Skin");
  menuItems.push_back ("Back");
  spacing = 40;
  string location = IMAGEPATH + "/menuback.png";
  SDL_Surface *temp = IMG_Load (location.c_str ());
  if (temp != NULL)
    imgBackground = SDL_DisplayFormat (temp);
  SDL_FreeSurface (temp);
  cursor = 0;
}

void
OptionsMenu::cleanup ()
{
  while (!menuItems.empty ())
    menuItems.pop_back ();
  SDL_FreeSurface (imgBackground);
}

void
OptionsMenu::pause ()
{
}

void
OptionsMenu::resume ()
{
}

void
OptionsMenu::draw (GameStateManager * gameStateManager)
{
  SDL_BlitSurface (imgBackground, NULL, gameStateManager->screen, NULL);
  SDL_Rect r = { 160, 230, 320, 230 };
  SDL_FillRect (gameStateManager->screen, &r,
		SDL_MapRGB (gameStateManager->screen->format, 64, 64, 64));
  for (int i = 0; i < menuItems.size (); i++)
    {
      if (cursor == i)
	gameStateManager->drawStringCentered (FONT_ACTIVE, menuItems[i], 320,
					      250 + i * spacing);
      else
	gameStateManager->drawStringCentered (FONT_INACTIVE, menuItems[i],
					      320, 250 + i * spacing);
    }
}

void
OptionsMenu::handleEvents (GameStateManager * gameStateManager)
{

  int mouseX;
  int mouseY;

  SDL_Event event;

  while (SDL_PollEvent (&event))
    {
      switch (event.type)
	{
	case SDL_QUIT:
	  gameStateManager->quit ();
	  break;
	case SDL_KEYDOWN:
	  switch (event.key.keysym.sym)
	    {
	    case SDLK_RETURN:
	      handleItem (gameStateManager);
	      break;
	    case SDLK_UP:
	      cursor--;
	      if (cursor < 0)
		cursor = menuItems.size () - 1;
	      break;
	    case SDLK_DOWN:
	      cursor++;
	      if (cursor >= menuItems.size ())
		cursor = 0;
	      break;
	    }
	  break;
	case SDL_MOUSEMOTION:
	  mouseX = event.motion.x;
	  mouseY = event.motion.y;

	  int x1;
	  int x2;
	  for (int i = 0; i < menuItems.size (); i++)
	    {
	      x1 =
		320 -
		menuItems[i].length () *
		gameStateManager->getCharWidth (FONT_INACTIVE) / 2;
	      x2 =
		x1 +
		menuItems[i].length () *
		gameStateManager->getCharWidth (FONT_INACTIVE);
	      if (mouseX >= x1 && mouseX <= x2 && mouseY >= 250 + i * spacing
		  && mouseY <=
		  250 + gameStateManager->getCharHeight (FONT_INACTIVE) +
		  i * spacing)
		cursor = i;
	    }
	  break;
	case SDL_MOUSEBUTTONDOWN:
	  handleItem (gameStateManager);
	  break;
	}
    }

}

void
OptionsMenu::update (GameStateManager * gameStateManager)
{
  if (gameStateManager->isFullscreen ())
    {
      if (menuItems[0].compare ("Fullscreen [X]") != 0)
	menuItems[0] = "Fullscreen [X]";
    }
  else
    {
      if (menuItems[0].compare ("Fullscreen [ ]") != 0)
	menuItems[0] = "Fullscreen [ ]";
    }
  if (gameStateManager->playsSounds ())
    {
      if (menuItems[1].compare ("Play Sounds [X]") != 0)
	menuItems[1] = "Play Sounds [X]";
    }
  else
    {
      if (menuItems[1].compare ("Play Sounds [ ]") != 0)
	menuItems[1] = "Play Sounds [ ]";
    }
}

void
OptionsMenu::handleItem (GameStateManager * gameStateManager)
{
  switch (cursor)
    {
    case OPTIONS_FULLSCREEN:
      gameStateManager->toggleFullscreen ();
      gameStateManager->saveSettings ();
      break;
    case OPTIONS_SOUNDS:
      gameStateManager->toggleSounds ();
      gameStateManager->saveSettings ();
      break;
    case OPTIONS_SKIN:
      gameStateManager->changeState (SkinMenu::instance ());
      break;
    case OPTIONS_BACK:
      gameStateManager->changeState (MainMenu::instance ());
      break;
    }
}
