/*
 * Electric(tm) VLSI Design System
 *
 * File: graphmac.c
 * Interface for Apple Macintosh computers
 * Written by: Steven M. Rubin, Static Free Software
 *
 * Copyright (c) 2000 Static Free Software.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Static Free Software
 * 4119 Alpine Road
 * Portola Valley, California 94028
 * info@staticfreesoft.com
 */

/* #define INVERTHIGHLIGHT 1 /* */

#  ifdef __MWERKS__
#    if __MWERKS__ >= 0x2100
#      define NEWCODEWARRIOR    1
#    endif
#  endif

#include "global.h"
#include "database.h"
#include "egraphics.h"
#include "usr.h"
#include "eio.h"
#include "efunction.h"
#include "edialogs.h"
#include "usrdiacom.h"
#include "usrtrack.h"
#include "dblang.h"

#include <Controls.h>
#include <Devices.h>
#include <Dialogs.h>
#include <Fonts.h>
#include <Gestalt.h>
#include <LowMem.h>
#include <Menus.h>
#include <Retrace.h>
#include <Scrap.h>
#include <Script.h>
#include <signal.h>
#include <Sound.h>
#include <ToolUtils.h>
#include <unix.h>

#if	LANGTCL
#  include "tclMac.h"
#  ifdef USETK
#    include "tk.h"
#    include "tkInt.h"
#    include "tkMacInt.h"
#  endif
#endif

/****** the messages window ******/
static WindowPtr     gra_messageswindow;		/* the scrolled messages window */
static INTSML        gra_messagesinfront;		/* nonzero if messages is frontmost */
static INTSML        gra_linesInFolder;			/* lines in text folder */
static INTSML        gra_messagesfont;			/* font in messages window */
static INTSML        gra_messagesfontsize;		/* size of font in messages window */
static INTSML        gra_messagesleft;			/* left bound of messages window screen */
static INTSML        gra_messagesright;			/* right bound of messages window screen */
static INTSML        gra_messagestop;			/* top bound of messages window screen */
static INTSML        gra_messagesbottom;		/* bottom bound of messages window screen */
static ControlHandle gra_vScroll;				/* vertical scroll control in messages window */
static ControlHandle gra_hScroll;				/* horizontal scroll control in messages window */
static TEHandle      gra_TEH;					/* text editing handle in messages window */
static ControlActionUPP gra_scrollvprocUPP;		/* UPP for "gra_scrollvproc" */
static ControlActionUPP gra_scrollhprocUPP;		/* UPP for "gra_scrollhproc" */
static GWorldPtr     gra_textbuf = 0;			/* temp buffer for displaying text */
static INTSML        gra_textbufwid = 0;		/* width of text temp buffer */
static INTSML        gra_textbufhei = 0;		/* height of text temp buffer */
static char        **gra_textbufrowstart;		/* row start array for text temp buffer */

/****** events ******/
#define	CHARREAD          0177					/* character that was read */
#define	ISBUTTON          0200					/* set if button pushed (or released) */
#define	BUTTONUP          0400					/* set if button was released */
#define	SHIFTDOWN        01000					/* set if shift key was held down */
#define	COMMANDDOWN      02000					/* set if command key was held down */
#define	OPTIONDOWN       04000					/* set if option key was held down */
#define	CONTROLDOWN     010000					/* set if control key was held down */
#define	DOUBLECLICK     020000					/* set if this is second click */
#define	MOTION          040000					/* set if mouse motion detected */
#define	FILEREPLY      0100000					/* set if standard file reply detected */
#define	WINDOWCHANGE   0100400					/* set if window moved and/or grown */
#define	NOEVENT             -1					/* set if nothing happened */
#define	MENUEVENT     BUTTONUP					/* set if menu entry selected (values in cursor) */

#ifdef NEWCODEWARRIOR
#  define SIGNALCAST __signal_func_ptr
#else
#  define SIGNALCAST _Sigfun*
#endif

struct
{
	INTSML x;
	INTSML y;
	INTSML kind;
} gra_action;

static INTBIG        gra_inputstate;			/* current state of device input */
static INTBIG        gra_lastclick;				/* time of last click */
static INTBIG        gra_handlingdialog = 0;	/* nonzero if handling a dialog event */
static INTSML        gra_lstcurx, gra_lstcury;	/* current position of mouse */
static INTSML        gra_cursorx, gra_cursory;	/* current position of mouse */
static WindowPtr     gra_lastclickedwindow;		/* last window where mouse-down happened */
static WINDOWFRAME  *gra_lastclickedwindowframe;	/* last window frame where mouse-down happened */

#define	EVENTQUEUESIZE	100

typedef struct
{
	INTSML           cursorx, cursory;			/* current position of mouse */
	INTBIG           inputstate;				/* current state of device input */
} MYEVENTQUEUE;

static MYEVENTQUEUE  gra_eventqueue[EVENTQUEUESIZE];
static MYEVENTQUEUE *gra_eventqueuehead;		/* points to next event in queue */
static MYEVENTQUEUE *gra_eventqueuetail;		/* points to first free event in queue */

void gra_onint(void);

/****** timing ******/
#define FLUSHTICKS   60							/* ticks between display flushes */
#define MOTIONCHECK  2							/* ticks between mouse motion checks */
#define INTCHECK     30							/* MOTIONCHECKs between interrupt checks */

typedef struct VBLRect
{
	VBLTask myVBLTask;
	long    vblA5;
} VBLRec;

static VBLRec        gra_vblrec;
static INTSML        gra_checkcountdown;		/* countdown to interrupt checks */
static INTSML        gra_motioncheck;			/* nonzero if mouse motion can be checked */
	   INTSML        gra_cancheck;				/* nonzero if interrupt can be checked */
unsigned long        gra_timestart;

/****** stuff for TCL/TK ******/
#if	LANGTCL
  Tcl_Interp *myTCLInterp;
#endif
#ifdef	USETK
#  define MYEVENTS (ExposureMask | ButtonPressMask | ButtonReleaseMask | KeyPressMask |\
				   PointerMotionMask | StructureNotifyMask | FocusChangeMask)
  typedef int (*TclMacConvertEventPtr)(EventRecord *eventPtr);

  Tk_Window          gra_tktopwindow;			/* top TK window */
  static Tk_Window   gra_tkmessageswindow;		/* TK handle to messages window */

  void gra_initializetk(void);
  void tcl_init(void);
  void TkMacHandleMenuSelect(long mResult, int optionKeyPressed);
  void gra_messageEventProc(ClientData clientData, XEvent *eventPtr);
  Tk_Window gra_maketkwindow(short type, short left, short right, short top, short bottom,
	void (*eventproc)(ClientData, XEvent*));
  void gra_raisewindow(Tk_Window win);
  void gra_namewindow(Tk_Window win, char *name);
  void gra_clearwindowframe(WINDOWFRAME *wf);
#else
  static void       *gra_tkmessageswindow;		/* dummy TK messages window handle */
#endif

/****** pulldown menus ******/
#define	MENUSIZE              19				/* size of menu bar */
#define	appleMENU            128				/* resource ID for Apple menu */
#define	USERMENUBASE         130				/* base resource ID for other menus */
#define	aboutMeCommand         1				/* menu entry for "About Electric..." item */

static INTSML        gra_lowmenu;				/* low word of selected pulldown menu */
static INTSML        gra_highmenu;				/* high word of selected pulldown menu */
static INTSML        gra_pulldownmenucount;		/* number of pulldown menus */
static INTSML        gra_tkmenusloaded = 0;		/* nonzero if TK menus are built */
static MenuHandle   *gra_pulldownmenus;			/* the current pulldown menu handles */
static POPUPMENU   **gra_pulldowns;				/* the current pulldown menus */
static MenuHandle    gra_appleMenu;				/* the Apple menu */

/****** mouse buttons ******/
#define	BUTTONS      17							/* cannot exceed NUMBUTS in "usr.h" */

typedef struct
{
	char *name;									/* button name */
	INTSML unique;								/* number of letters that make it unique */
} BUTTONNAMES;

static BUTTONNAMES   gra_buttonname[BUTTONS] =
{						/* Shift Command Option Control */
	"Button",      1,   /*                              */
	"SButton",     2,	/* Shift                        */
	"MButton",     2,	/*       Command                */
	"SMButton",    3,	/* Shift Command                */
	"OButton",     2,	/*               Option         */
	"SOButton",    3,	/* Shift         Option         */
	"MOButton",    3,	/*       Command Option         */
	"SMOButton",   4,	/* Shift Command Option         */
	"CButton",     2,	/*                      Control */
	"SCButton",    3,	/* Shift                Control */
	"CMButton",    3,	/*       Command        Control */
	"SCMButton",   4,	/* Shift Command        Control */
	"COButton",    3,	/*               Option Control */
	"SCOButton",   4,	/* Shift         Option Control */
	"CMOButton",   4,	/*       Command Option Control */
	"SCMOButton",  5,	/* Shift Command Option Control */
	"DButton",     2
};

static INTSML        gra_doubleclick;					/* interval between double clicks */

/****** cursors ******/
enum {wantttyCURSOR = 129, penCURSOR, nullCURSOR, menulCURSOR, handCURSOR, techCURSOR,
	lrCURSOR, udCURSOR};

static CursHandle    gra_wantttyCurs;			/* a "use the TTY" cursor */
static CursHandle    gra_penCurs;				/* a "draw with pen" cursor */
static CursHandle    gra_nullCurs;				/* a null cursor */
static CursHandle    gra_menuCurs;				/* a menu selection cursor */
static CursHandle    gra_handCurs;				/* a hand dragging cursor */
static CursHandle    gra_techCurs;				/* a technology edit cursor */
static CursHandle    gra_ibeamCurs;				/* a text edit cursor */
static CursHandle    gra_lrCurs;				/* a left/right cursor */
static CursHandle    gra_udCurs;				/* an up/down cursor */

/****** rectangle saving ******/
#define	NOSAVEDBOX ((SAVEDBOX *)-1)

typedef struct Isavedbox
{
	char             *pix;
	WINDOWPART       *win;
	INTSML            lx, hx, ly, hy;
	INTSML            truelx, truehx;
	struct Isavedbox *nextsavedbox;
} SAVEDBOX;

SAVEDBOX *gra_firstsavedbox = NOSAVEDBOX;

/****** polygon decomposition ******/
#define NOPOLYSEG ((POLYSEG *)-1)

typedef struct Isegment
{
	INTBIG fx,fy, tx,ty, direction, increment;
	struct Isegment *nextedge;
	struct Isegment *nextactive;
} POLYSEG;

static POLYSEG      *gra_polysegs;
static INTSML        gra_polysegcount = 0;

/****** curve drawing ******/
#define MODM(x) ((x<1) ? x+8 : x)
#define MODP(x) ((x>8) ? x-8 : x)
#define ODD(x)  (x&1)

static INTSML        gra_arcocttable[] = {0,0,0,0,0,0,0,0,0};
static INTBIG        gra_arccenterx, gra_arccentery, gra_arcradius, gra_curvemaxx, gra_curvemaxy;
static INTBIG        gra_arclx, gra_archx, gra_arcly, gra_archy;
static INTSML        gra_curvecol, gra_curvemask, gra_curvestyle, gra_arcfirst;
static WINDOWFRAME  *gra_arcframe;				/* window frame for arc drawing */
static void        (*gra_arcpixeldraw)(INTBIG x, INTBIG y);
static void        (*gra_discrowdraw)(WINDOWFRAME *wf, INTBIG thisy, INTBIG startx, INTBIG endx,
						GRAPHICS *desc);

/****** dialogs ******/
#define	aboutMeDLOG          128				/* resource ID for "About Electric" alert dialog */
#define noscreenALERT        129				/* resource ID for "no valid screen" alert dialog */
#define errorALERT           130				/* resource ID for "error" alert dialog */
#define	FILEINDIALOG         136				/* resource ID for "file input" dialog */
#define	NAMERSRC            2000				/* resource ID for user name */
#define	COMPANYRSRC         2001				/* resource ID for company name */
#define	SPECIALRSRC         2002				/* resource ID for special instructions */
#define	CHECKRSRC           2003				/* resource ID for checksum */

static DIALOGITEM gra_fontdialogitems[] =
{
 /*  1 */ {0, {136,192,160,256}, BUTTON, N_("OK")},
 /*  2 */ {0, {88,192,112,256}, BUTTON, N_("Cancel")},
 /*  3 */ {0, {8,24,24,56}, MESSAGE, N_("Font")},
 /*  4 */ {0, {24,200,40,232}, MESSAGE, N_("Size")},
 /*  5 */ {0, {32,24,160,184}, SCROLL, ""},
 /*  6 */ {0, {48,192,64,248}, EDITTEXT, ""}
};
DIALOG gra_fontdialog = {{50,75,219,340}, N_("Messages Window Font"), 0, 6, gra_fontdialogitems};

static FileFilterYDUPP gra_fileFilterProcUPP;

pascal Boolean gra_fileFilterProc(CInfoPBPtr pb, Ptr mydata);
void   DTextCopy(void);
void   DTextCut(void);
char   DTextPaste(void);

/****** miscellaneous ******/
#define _Gestalt            0xA1AD
#define	EFONT	            kFontIDHelvetica	/* font in the editing window */
#define	TFONT               kFontIDCourier		/* font in text editing */
#define	SFONT	            kFontIDGeneva		/* font for status at the bottom of editing windows */
#define	MFONT	            0					/* font in menus */
#define	MSFONT	            kFontIDCourier		/* font in the messages windows */
#define	DFONT	            0					/* font in dialogs */
#define	DSFONT	            kFontIDCourier		/* small font in dialog scroll areas */
#define	SBARWIDTH           15					/* width of scroll bar at right of edit window */
#define	SBARHEIGHT          15					/* height of scroll bar at bottom of edit window */
#ifdef USETK
#  define	PALETTEWIDTH     0					/* width of palette */
#else
#  define	PALETTEWIDTH    80					/* width of palette */
#endif
#define	FLOATINGHEADERSIZE  10					/* size of tool palette drag bar */
#define MAXSTATUSLINES       1					/* lines in status bar */

static INTSML        gra_curfontsize;			/* size of current font */
static FontInfo      gra_curfontinfo;			/* current font information */
static INTSML        gra_screenleft;			/* left bound of any editor window screen */
static INTSML        gra_screenright;			/* right bound of any editor window screen */
static INTSML        gra_screentop;				/* top bound of any editor window screen */
static INTSML        gra_screenbottom;			/* bottom bound of any editor window screen */
static GDHandle      gra_origgdevh;				/* the original graphics device */
static char          gra_localstring[256];		/* local string */
static INTSML        gra_logrecordindex = 0;	/* count for session flushing */
static INTSML        gra_playbackmultiple = 0;	/* count for multi-key playback */
static INTSML        gra_multiwindow;			/* nonzero if using multiple windows */
static void         *gra_fileliststringarray = 0;
static INTSML        gra_windowframeindex = 0;

/****** prototypes for local routines ******/
void         gra_addeventtoqueue(INTBIG state, INTSML x, INTSML y);
INTSML       gra_addfiletolist(char *file);
void         gra_adjusthtext(INTSML);
void         gra_adjustvtext(void);
void         gra_applemenu(INTSML);
void         gra_arcbresccw(INTBIG x, INTBIG y, INTBIG x1, INTBIG y1);
void         gra_arcbrescw(INTBIG x, INTBIG y, INTBIG x1, INTBIG y1);
void         gra_arcbresmidccw(INTBIG x, INTBIG y);
void         gra_arcbresmidcw(INTBIG x, INTBIG y);
void         gra_arcdopixel(INTBIG x, INTBIG y);
INTSML       gra_arcfindoctant(INTBIG x, INTBIG y);
void         gra_arcoutxform(INTBIG x, INTBIG y);
void         gra_arcxformoctant(INTBIG x, INTBIG y, INTBIG oct, INTBIG *ox, INTBIG *oy);
void         gra_backup(void);
INTSML       gra_buildwindow(WINDOWFRAME*, INTSML);
void         gra_centermessage(char*, INTSML, INTSML, INTSML);
void         gra_drawdiscrow(WINDOWFRAME *wf, INTBIG thisy, INTBIG startx, INTBIG endx, GRAPHICS *desc);
void         gra_drawline(WINDOWFRAME*, INTSML, INTSML, INTSML, INTSML, INTSML, INTSML);
void         gra_drawpatline(WINDOWFRAME*, INTSML, INTSML, INTSML, INTSML, INTSML, INTSML, INTSML);
INTSML       gra_enternetwork(char*, char*);
void         gra_fakemouseup(void);
OSErr        gra_findprocess(OSType typeToFind, OSType creatorToFind, ProcessSerialNumberPtr processSN,
				ProcessInfoRecPtr infoRecToFill);
void         gra_getdevices(void);
Rect        *gra_geteditorwindowlocation(void);
CGrafPtr     gra_getoffscreen(WINDOWPART *win);
INTSML       gra_gettraptype(INTSML);
CGrafPtr     gra_getwindow(WINDOWPART *win);
pascal OSErr gra_handleodoc(AppleEvent*, AppleEvent*, long);
pascal OSErr gra_handlequit(AppleEvent*, AppleEvent*, long);
void         gra_hidemessageswindow(void);
void         gra_initfilelist(void);
void         gra_initializemenus(void);
int          gra_initializetcl(void);
void         gra_installvbl(void);
char        *gra_macintoshinitialization(void);
INTSML       gra_makebutton(INTBIG);
INTSML       gra_makeeditwindow(WINDOWFRAME*);
char        *gra_makefullname(char*, INTSML);
char        *gra_makefullnamefromFSS(FSSpec theFSS);
MenuHandle   gra_makepdmenu(POPUPMENU*, INTSML);
char        *gra_makepstring(char *string);
void         gra_mygrowwindow(WindowPtr, INTSML, INTSML, Rect*);
void         gra_nativemenudoone(INTSML low, INTSML high);
void         gra_nextevent(INTSML, EventRecord*);
INTSML       gra_numtoolboxtraps(void);
INTSML       gra_pulldownindex(POPUPMENU*);
void         gra_rebuildrowstart(WINDOWFRAME*);
void         gra_recachepixmap(PixPatHandle);
void         gra_redrawdisplay(WINDOWFRAME*);
void         gra_reloadmap(void);
INTSML       gra_remakeeditwindow(WINDOWFRAME*);
void         gra_removewindowextent(RECTAREA *r, RECTAREA *er);
void         gra_savewindowsettings(void);
pascal void  gra_scrollhproc(ControlHandle, INTSML);
pascal void  gra_scrollvproc(ControlHandle, INTSML);
void         gra_setcurrentwindowframe(void);
void         gra_setrect(WINDOWFRAME*, INTSML, INTSML, INTSML, INTSML);
void         gra_setuptextbuffer(INTSML width, INTSML height);
void         gra_setview(WindowPtr);
void         gra_setvscroll(void);
INTSML       gra_showmessageswindow(void);
void         gra_showselect(void);
INTSML       gra_trapavailable(INTSML);
void         gra_waitforaction(INTSML, EventRecord*);
void         Dredrawdialogwindow(void);
char        *gra_systemfoldername(void);
void         mac_settypecreator(char *name, INTBIG type, INTBIG creator);

#ifndef USETK
  void       gra_activatewindow(WindowPtr, INTSML);
  void       gra_disposeoswindow(WindowPtr, void*);
  void       gra_dragfloatingwindow(WindowPtr, Point);
  void       gra_drawosgraphics(WINDOWFRAME*);
  void       gra_drawpalettedragbar(WINDOWFRAME *wf);
  void       gra_frontnonfloatingwindow(WindowPtr *);
  WindowPtr  gra_lastfloatingwindow(void);
  void       gra_selectoswindow(WindowPtr);
  void       gra_showallfloaters(INTSML);
#endif
#ifdef	INVERTHIGHLIGHT
  void       gra_arcinvpixel(INTBIG x, INTBIG y);
  void       gra_invdiscrow(WINDOWFRAME *wf, INTBIG thisy, INTBIG startx, INTBIG endx, GRAPHICS *desc);
  void       gra_invertcircle(WINDOWPART *win, INTBIG atx, INTBIG aty, INTBIG radius);
#endif

/******************** INITIALIZATION ********************/

void main(void)
{
	short argc;
	char *argv[4], *progname;

	/* initialize for the operating system */
	progname = gra_macintoshinitialization();
	argc = 1;
	argv[0] = progname;

	/* primary initialization of Electric */
	osprimaryosinit();

#if	LANGTCL
#  ifdef USETK
	/* need TCL and TK, so must initialize early */
	if (gra_initializetcl() == TCL_ERROR)
		error(_("gra_initializetcl failed: %s\n"), myTCLInterp->result);
	gra_initializetk();
#  endif
#endif

	/* secondary initialization of Electric */
	ossecondaryinit(argc, argv);

#if	LANGTCL
#  ifndef USETK
	/* only need TCL, initialize later to get proper path */
	if (gra_initializetcl() == TCL_ERROR)
		error(_("gra_initializetcl failed: %s\n"), myTCLInterp->result);
#  endif
#endif

	/* now run Electric */
	for(;;)
	{
		tooltimeslice();
	}
}

/*
 * routine to establish the default display
 */
void graphicsoptions(char *name, INTBIG *argc, char **argv)
{
	/* set multiple window factor */
	if (*argc > 0 && argv[0][strlen(argv[0])-1] == '1') gra_multiwindow = 0; else
		gra_multiwindow = 1;
	us_erasech = DELETEKEY;
	us_killch = 025;
}

/*
 * routine to initialize the display device.  Creates status window if "messages" is nonzero.
 */
INTSML initgraphics(INTSML messages)
{
	WINDOWFRAME *wf;
	char fontname[100];
	Rect rstat, r;
	INTSML err;
	long response;
	Handle f;
	CTabHandle iclut;
	REGISTER INTSML i;
	RgnHandle grayrgn;

	/* get double-click interval */
	gra_doubleclick = GetDblTime();
	gra_lastclick = 0;
	gra_lstcurx = -1;

	/* setup UPPs */
	gra_scrollvprocUPP = NewControlActionProc(gra_scrollvproc);
	gra_scrollhprocUPP = NewControlActionProc(gra_scrollhproc);
	gra_fileFilterProcUPP = NewFileFilterYDProc(gra_fileFilterProc);

	/* initialize menus */
	gra_initializemenus();

	/* get cursors */
	gra_wantttyCurs = GetCursor(wantttyCURSOR);
	gra_penCurs = GetCursor(penCURSOR);
	gra_nullCurs = GetCursor(nullCURSOR);
	gra_menuCurs = GetCursor(menulCURSOR);
	gra_handCurs = GetCursor(handCURSOR);
	gra_techCurs = GetCursor(techCURSOR);
	gra_ibeamCurs = GetCursor(iBeamCursor);
	gra_lrCurs = GetCursor(lrCURSOR);
	gra_udCurs = GetCursor(udCURSOR);
	us_normalcursor = NORMALCURSOR;

	/* setup globals that describe location of windows */
	gra_getdevices();

	/* determine font and size of text in messages window */
	f = GetResource('MPSR', 1004);
	if (f == 0)
	{
		/* no window resource: use defaults */
		gra_messagesfont = MSFONT;
		gra_messagesfontsize = 12;
	} else
	{
		HLock(f);

		/* get messages window extent */
		rstat.left = ((INTSML *)(*f))[0];
		rstat.right = ((INTSML *)(*f))[1];
		rstat.top = ((INTSML *)(*f))[2];
		rstat.bottom = ((INTSML *)(*f))[3];

		/* is the top of the messages window visible on this display? */
		r.left = rstat.left;   r.right = rstat.right;
		r.top = rstat.top;     r.bottom = rstat.top + MENUSIZE;
		grayrgn = GetGrayRgn();
		if (RectInRgn(&r, grayrgn))
		{
			gra_messagesleft = rstat.left;   gra_messagesright = rstat.right;
			gra_messagestop = rstat.top;     gra_messagesbottom = rstat.bottom;
		}

		/* get font and size */
		gra_messagesfontsize = ((INTSML *)(*f))[8];
		(void)strcpy(&fontname[1], &((char *)(*f))[18]);
		fontname[0] = strlen(&fontname[1]);
		GetFNum((unsigned char *)fontname, &gra_messagesfont);
		HUnlock(f);
	}

	/* create the scrolling messages window */
	gra_messageswindow = 0;
	gra_TEH = 0;
	if (messages != 0)
	{
		if (gra_showmessageswindow() != 0) error(_("Cannot create messages window"));
	}

	/* initialize the mouse */
	if (gra_nullCurs != 0L) SetCursor(&(**gra_nullCurs)); else
		SetCursor(&qd.arrow);
	us_cursorstate = NULLCURSOR;
	gra_inputstate = NOEVENT;
	gra_eventqueuehead = gra_eventqueuetail = gra_eventqueue;

	/* create the CLUT for identity mapping on color displays */
	iclut = (CTabHandle)NewHandle(2056);
	if (iclut == 0) error(_("Cannot allocate identity color lookup table"));
	(*iclut)->ctSeed = GetCTSeed();
	(*iclut)->ctFlags = 0;
	(*iclut)->ctSize = 255;
	for(i=0; i<256; i++)
	{
		(*iclut)->ctTable[i].value = i;
		(*iclut)->ctTable[i].rgb.red = i<<8;
		(*iclut)->ctTable[i].rgb.green = i<<8;
		(*iclut)->ctTable[i].rgb.blue = i<<8;
	}

	/* setup to handle input files via apple events */
	err = Gestalt(gestaltAppleEventsAttr, &response);
	if (err == 0 && ((response >> gestaltAppleEventsPresent) & 1) != 0)
	{
		/* apple events method */
		err = AEInstallEventHandler(kCoreEventClass, kAEOpenDocuments,
			NewAEEventHandlerProc(gra_handleodoc), 0, 0);
		err = AEInstallEventHandler(kCoreEventClass, kAEQuitApplication,
			NewAEEventHandlerProc(gra_handlequit), 0, 0);
	}

	/* initialize vertical retrace manager to check for interrupts */
	gra_installvbl();

	/* create the first window frame */
	if (gra_multiwindow == 0)
	{
		wf = (WINDOWFRAME *)emalloc((sizeof (WINDOWFRAME)), us_aid->cluster);
		if (wf == 0) error(_("Cannot create Macintosh window info structure"));
		wf->numvar = 0;
		wf->windindex = gra_windowframeindex++;
		el_firstwindowframe = el_curwindowframe = wf;
		el_firstwindowframe->nextwindowframe = NOWINDOWFRAME;

		/* load an editor window into this frame */
		if (gra_buildwindow(el_curwindowframe, 0) != 0)
			error(_("Cannot create Macintosh window info structure"));
	}

	/* catch interrupts */
	(void)signal(SIGINT, (SIGNALCAST)gra_onint);

	return(0);
}

/*
 * Routine to establish the library directories from the environment.
 */
void setupenvironment(void)
{
	/* initialize system directories for library files */
	(void)initinfstr();
	(void)addstringtoinfstr(gra_systemfoldername());
	(void)addstringtoinfstr(LIBDIR);
	(void)allocstring(&el_libdir, returninfstr(), db_cluster);

	/* set machine name */
	nextvarchangequiet();
	(void)setvalkey((INTBIG)us_aid, VAID, makekey("USER_machine"),
		(INTBIG)"Macintosh", VSTRING|VDONTSAVE);
}

void setlibdir(char *libdir)
{
	char *pp;

	/* make sure UNIX '/' isn't in use */
	for(pp = libdir; *pp != 0; pp++)
		if (*pp == '/') *pp = DIRSEP;

	(void)initinfstr();
	if (libdir[0] == ':')
	{
		pp = gra_systemfoldername();
	} else
	{
		pp = currentdirectory();
	}
	(void)addstringtoinfstr(pp);
	(void)addstringtoinfstr(libdir);
	if (libdir[strlen(libdir)-1] != DIRSEP) (void)addtoinfstr(DIRSEP);
	(void)reallocstring(&el_libdir, returninfstr(), db_cluster);
}

/*
 * Routine to examine the display devices available and to setup globals that describe
 * the editing windows and messages window extents.  On exit, the globals "gra_screenleft",
 * "gra_screenright", "gra_screentop", and "gra_screenbottom" will describe the area
 * for the editing windows and the variables "gra_messagesleft", "gra_messagesright",
 * "gra_messagestop", and "gra_messagesbottom" will describe the messages window.
 */
void gra_getdevices(void)
{
	GDHandle gdevh, bestcgdevh, bestbwgdevh, curgdevh;
	INTSML totaldevices, iscolor, anycolor;
	INTBIG bestcpixels, bestbwpixels, pixels, ys;

	/* obtain the current device */
	gra_origgdevh = GetGDevice();

	/* find the devices */
	bestcpixels = bestbwpixels = 0L;
	totaldevices = anycolor = 0;
	for(gdevh = GetDeviceList(); gdevh != 0L; gdevh = GetNextDevice(gdevh))
	{
		totaldevices++;

		/* determine size of this display */
		pixels = (*gdevh)->gdRect.right - (*gdevh)->gdRect.left;
		ys = (*gdevh)->gdRect.bottom - (*gdevh)->gdRect.top;
		pixels *= ys;

		/* The low bit is set if the display is color */
		iscolor = 1;
		if (((*gdevh)->gdFlags&1) == 0) iscolor = 0;

		/* color displays must be at least 8 bits deep */
		if ((*(*gdevh)->gdPMap)->pixelSize < 8) iscolor = 0;

		/* accumulate the best of each display type */
		if (iscolor != 0)
		{
			if (pixels > bestcpixels || (pixels == bestcpixels && gdevh == gra_origgdevh))
			{
				bestcgdevh = gdevh;
				bestcpixels = pixels;
			}
		} else
		{
			if (pixels > bestbwpixels || (pixels == bestbwpixels && gdevh == gra_origgdevh))
			{
				bestbwgdevh = gdevh;
				bestbwpixels = pixels;
			}
		}
	}

	/* if there is a color device, choose it */
	if (bestcpixels != 0) curgdevh = bestcgdevh; else
		if (bestbwpixels != 0) curgdevh = bestbwgdevh; else
	{
		StopAlert(noscreenALERT, 0L);
		exitprogram();
	}

	/* set the extent of the editing windows */
	gra_screenleft   = (*curgdevh)->gdRect.left;
	gra_screenright  = (*curgdevh)->gdRect.right;
	gra_screentop    = (*curgdevh)->gdRect.top;
	gra_screenbottom = (*curgdevh)->gdRect.bottom;

	/* set the extent of the messages window */
	gra_messagesleft   = gra_screenleft + PALETTEWIDTH + 1;
	gra_messagesright  = gra_screenright - PALETTEWIDTH - 1;
	gra_messagestop    = gra_screentop + MENUSIZE*3 + 2 +
		(gra_screenbottom-(gra_screentop+MENUSIZE*2)) * 3 / 5;
	gra_messagesbottom = gra_screenbottom;

	/* if multiple displays exist, choose another for the messages window */
	if (totaldevices > 1)
	{
		/* look for another screen to be used for the messages display */
		for(gdevh = GetDeviceList(); gdevh != 0L; gdevh = GetNextDevice(gdevh))
			if (gdevh != curgdevh)
		{
			INTSML i, siz;
			gra_messagesleft = (*gdevh)->gdRect.left;
			gra_messagesright = (*gdevh)->gdRect.right;
			gra_messagestop = (*gdevh)->gdRect.top;
			gra_messagesbottom = (*gdevh)->gdRect.bottom;

			i = (gra_messagesleft + gra_messagesright) / 2;
			siz = (gra_messagesright - gra_messagesleft) * 2 / 5;
			gra_messagesleft = i - siz;     gra_messagesright = i + siz;
			i = (gra_messagestop + gra_messagesbottom) / 2;
			siz = (gra_messagesbottom - gra_messagestop) / 4;
			gra_messagestop = i - siz;      gra_messagesbottom = i + siz;
			return;
		}
	}
}

char *gra_macintoshinitialization(void)
{
	long ppcresponse;
	ProcessSerialNumber psn;
	ProcessInfoRec pinfo;
	unsigned char processname[50];
	static char appname[50];
#if defined(__MWERKS__) && __MC68020__
	long response;
#endif

	MaxApplZone();
	InitGraf(&qd.thePort);
	InitFonts();
	FlushEvents(everyEvent, 0);
	InitWindows();
	InitMenus();
	TEInit();
	InitDialogs(0L);
	InitCursor();

	/* must have system 7 */
	if (!gra_trapavailable(_Gestalt))
		error(_("The Macintosh must be running system 7"));

	/* see if this is a powerPC */
	Gestalt(gestaltPPCToolboxAttr, &ppcresponse);

	/* check for 020 if the code was compiled for it */
#if defined(__MWERKS__) && __MC68020__
	Gestalt(gestaltProcessorType, &response);
	if (ppcresponse == 0 &&
		(response == gestalt68000 || response == gestalt68010))
			error(_("The Macintosh must have a 68020 or better processor"));
#endif

	/* check for floating point unit if the code was compiled for it */
#if defined(__MWERKS__) && __MC68881__
	Gestalt(gestaltFPUType, &response);
	if (ppcresponse == 0 && response == gestaltNoFPU)
		error(_("The Macintosh must have a Floating Point Processor"));
#endif

	/* determine "arguments" to the program */
	pinfo.processInfoLength = sizeof (ProcessInfoRec);
	pinfo.processName = processname;
	pinfo.processAppSpec = 0;
	GetCurrentProcess(&psn);
	GetProcessInformation(&psn, &pinfo);
	strncpy(appname, (char *)&pinfo.processName[1], pinfo.processName[0]);
	appname[pinfo.processName[0]] = 0;
	return(appname);
}

/*
 * routine to determine whether a trap exists in the Mac
 */
INTSML gra_trapavailable(INTSML theTrap)
{
	INTSML ttype;

	ttype = gra_gettraptype(theTrap);
	if (ttype == ToolTrap)
	{
		theTrap &= 0x7FF;
		if (theTrap >= gra_numtoolboxtraps()) return(0);
	}
	return(NGetTrapAddress(theTrap, ttype) != NGetTrapAddress(0, ToolTrap));
}

/*
 * support routine to determine what kind of trap this is
 */
INTSML gra_gettraptype(INTSML theTrap)
{
	if ((theTrap&0x800) != 0) return(ToolTrap);
	return(OSTrap);
}

/*
 * support routine to determine how many Mac Toolbox traps are available
 */
INTSML gra_numtoolboxtraps(void)
{
	/* 0xA86E is "InitGraf" */
	if (NGetTrapAddress(0xA86E, 1) == NGetTrapAddress(0xAA6E, 1)) return(0x200);
	return(0x400);
}

#if	LANGTCL

int gra_initializetcl(void)
{
	int err;
	char *newArgv[2];

#ifdef	USETK
	tcl_macQdPtr = &qd;
	Tcl_MacSetEventProc(TkMacConvertEvent);
#endif

	/* set the program name/path */
	newArgv[0] = "Electric";
	newArgv[1] = NULL;
	(void)Tcl_FindExecutable(newArgv[0]);

	myTCLInterp = Tcl_CreateInterp();
	if (myTCLInterp == 0) error(_("from Tcl_CreateInterp"));

	/* tell Electric the TCL interpreter handle */
	el_tclinterpreter(myTCLInterp);

	/* Make command-line arguments available in the Tcl variables "argc" and "argv" */
	Tcl_SetVar(myTCLInterp, "argv", "", TCL_GLOBAL_ONLY);
	Tcl_SetVar(myTCLInterp, "argc", "0", TCL_GLOBAL_ONLY);
	Tcl_SetVar(myTCLInterp, "argv0", "electric", TCL_GLOBAL_ONLY);

	/* Set the "tcl_interactive" variable */
	Tcl_SetVar(myTCLInterp, "tcl_interactive", "1", TCL_GLOBAL_ONLY);

	/* initialize the interpreter */
	err = Tcl_Init(myTCLInterp);
	if (err != TCL_OK) error(_("(from Tcl_Init) %s"), myTCLInterp->result);

	return(err);
}

#endif

#ifdef	USETK

void gra_initializetk(void)
{
	int code;

	/* make the top tk window */
	gra_tktopwindow = TkCreateMainWindow(myTCLInterp, "", "Electric");
	if (gra_tktopwindow == NULL) return;
	(void)Tk_SetClass(gra_tktopwindow, "Electric");
	(void)Tcl_ResetResult(myTCLInterp);

	/* initialize tk package */
	if (Tcl_PkgRequire(myTCLInterp, "Tcl", TCL_VERSION, 1) == NULL)
		error(_("(from Tcl_PkgRequire) %s)", myTCLInterp->result);
	code = Tcl_PkgProvide(myTCLInterp, "Tk", TK_VERSION);
	if (code != TCL_OK) error(_("(from Tcl_PkgProvide) %s"), myTCLInterp->result);
	code = TkpInit(myTCLInterp);
	if (code != TCL_OK) return;
}

/*
 * Routine to create a TK window of type "type":
 *  1  Scrolling text window
 *  2  Graphic edit window
 *  3  Dialog window with title banner
 *  4  Dialog windiw with no title banner
 * The window runs from "left" to "right" and from "top" to "bottom".
 * The event proc to use for this window is in "eventproc"
 */
Tk_Window gra_maketkwindow(short type, short left, short right, short top, short bottom,
	void (*eventproc)(ClientData, XEvent*))
{
	Tk_Window win;
	char windowname[100], unsupportedcmd[200], *argv[3];
	static INTSML windowcount = 1;

	sprintf(windowname, ".electric%d", windowcount);
	windowcount++;
	sprintf(unsupportedcmd, "toplevel %s -container 1", windowname);
	if (Tcl_VarEval(myTCLInterp, unsupportedcmd, 0) == TCL_ERROR)
		ttyputerr("Tcl_VarEval returned %s", myTCLInterp->result);
	win = Tk_NameToWindow(myTCLInterp, windowname, gra_tktopwindow);
	switch (type)
	{
		case 1:
		case 2:
			sprintf(unsupportedcmd, "unsupported1 style %s zoomDocProc", windowname);
			break;
		case 3:
			sprintf(unsupportedcmd, "unsupported1 style %s noGrowNoCloseDocProc", windowname);
			break;
		case 4:
			sprintf(unsupportedcmd, "unsupported1 style %s dBoxProc", windowname);
			break;
	}
	if (Tcl_Eval(tcl_interp, unsupportedcmd) != TCL_OK)
	{
		ttyputmsg(_("ERROR from Tcl_Eval: %s"), tcl_interp->result);
		Tcl_ResetResult(tcl_interp);
	}

	(void)Tk_GeometryRequest(win, right-left, bottom-top);
	(void)Tk_MoveToplevelWindow(win, left, top);
	(void)Tk_MapWindow(win);
	(void)Tk_CreateEventHandler(win, MYEVENTS, eventproc, (ClientData)NULL);

	if (type == 3 || type == 4)
	{
		argv[0] = "grab";
		argv[1] = "-global";
		argv[2] = windowname;
		if (Tk_GrabCmd((ClientData)gra_tktopwindow, myTCLInterp, 3, argv) != TCL_OK)
		{
			ttyputmsg(_("ERROR from Tk_GrabCmd: %s"), myTCLInterp->result);
			Tcl_ResetResult(myTCLInterp);
		}
	}

	return(win);
}

#endif

/******************** TERMINATION ********************/

void termgraphics(void)
{
#ifdef USETK
	while (el_firstwindowframe != NOWINDOWFRAME)
		(void)Tk_DestroyWindow(el_firstwindowframe->tkwin);
	if (gra_messageswindow != 0) (void)Tk_DestroyWindow(gra_tkmessageswindow);
#else
	while (el_firstwindowframe != NOWINDOWFRAME)
		gra_disposeoswindow((WindowPtr)el_firstwindowframe->realwindow, el_firstwindowframe->tkwin);
	if (gra_messageswindow != 0)
		gra_disposeoswindow(gra_messageswindow, gra_tkmessageswindow);
#endif
}

/*
 * routine to handle a "quit" Apple Event
 */
pascal OSErr gra_handlequit(AppleEvent *theAppleEvent, AppleEvent *reply, long handlerRefCon)
{
	if (us_preventloss(NOLIBRARY, "quit", 1)) return(noErr);
	bringdown();
	return(noErr);
}

/*
 * routine to quit the program.  It unregisters the name first and then exits.
 */
void exitprogram(void)
{
	ExitToShell();
}

/******************** WINDOW CONTROL ********************/

/*
 * Routine to create a new window frame (floating if "floating" nonzero).
 */
WINDOWFRAME *newwindowframe(INTSML floating, RECTAREA *r)
{
	WINDOWFRAME *wf, *oldlisthead;

	if (gra_multiwindow == 0) return(NOWINDOWFRAME);

#ifdef USETK
	if (floating != 0) return(NOWINDOWFRAME);
#endif

	/* allocate one */
	wf = (WINDOWFRAME *)emalloc((sizeof (WINDOWFRAME)), us_aid->cluster);
	if (wf == 0) return(NOWINDOWFRAME);
	wf->numvar = 0;
	wf->windindex = gra_windowframeindex++;

	/* insert window-frame in linked list */
	oldlisthead = el_firstwindowframe;
	wf->nextwindowframe = el_firstwindowframe;
	el_firstwindowframe = wf;

	/* load an editor window into this frame */
	if (gra_buildwindow(wf, floating) != 0)
	{
		efree((char *)wf);
		el_firstwindowframe = oldlisthead;
		return(NOWINDOWFRAME);
	}

	/* remember that this is the current window frame */
	if (floating == 0) el_curwindowframe = wf;

	return(wf);
}

void killwindowframe(WINDOWFRAME *wf)
{
	if (gra_multiwindow != 0)
	{
		/* kill the actual window, that will remove the frame, too */
#ifdef USETK
		(void)Tk_DestroyWindow(wf->tkwin);
#else
		gra_disposeoswindow((WindowPtr)wf->realwindow, wf->tkwin);
#endif
	}
}

/*
 * Routine to return the current window frame.
 */
WINDOWFRAME *getwindowframe(INTSML canfloat)
{
	return(el_curwindowframe);
}

/*
 * routine to return size of window "win" in "wid" and "hei"
 */
void getwindowframesize(WINDOWFRAME *wf, INTSML *wid, INTSML *hei)
{
	*wid = wf->swid;
	*hei = wf->shei;
}

/*
 * Routine to return the current Macintosh window associated with Electric window "win".
 * This is only called from "ioplotmac.c"
 */
CGrafPtr gra_getwindow(WINDOWPART *win)
{
	return(win->frame->realwindow);
}

/*
 * Routine to return the offscreen Macintosh buffer associated with Electric window "win".
 * This is only called from "ioplotmac.c"
 */
CGrafPtr gra_getoffscreen(WINDOWPART *win)
{
	return(win->frame->window);
}

void sizewindowframe(WINDOWFRAME *wf, INTSML wid, INTSML hei)
{
	Rect         fr;

	fr = (*((WindowPeek)wf->realwindow)->strucRgn)->rgnBBox;

	/* determine new window size */
	wid++;
	hei += 2;
	if (wf != NOWINDOWFRAME && wf->floating != 0) hei += FLOATINGHEADERSIZE;
	if (wid+2 == fr.right-fr.left && hei+2 == fr.bottom-fr.top) return;

	/* resize the window */
#ifdef	USETK
	(void)Tk_ResizeWindow(wf->tkwin, wid, hei);
#else
	SizeWindow((WindowRef)wf->realwindow, wid, hei, 1);

	/* rebuild the offscreen windows */
	if (gra_remakeeditwindow(wf) > 0)
	{
		fr.left += 1;         fr.right -= 2;
		fr.top += MENUSIZE;   fr.bottom -= 2;
		SizeWindow((WindowRef)wf->realwindow, fr.right-fr.left, fr.bottom-fr.top, 1);
		return;
	}
	gra_reloadmap();

	SetPort((WindowPtr)wf->realwindow);
	EraseRect(&wf->realwindow->portRect);
	gra_drawosgraphics(wf);
#endif
}

void movewindowframe(WINDOWFRAME *wf, INTSML left, INTSML top)
{
	INTSML        oleft, otop;
	Rect         fr;

	fr = (*((WindowPeek)wf->realwindow)->strucRgn)->rgnBBox;

	/* determine new window location */
	top += MENUSIZE+1;
	if (left == fr.left && top == fr.top) return;
	oleft = fr.left;   otop = fr.top;

#ifdef	USETK
	(void)Tk_MoveWindow(wf->tkwin, left, top);
#else
	MoveWindow((WindowRef)wf->realwindow, left, top, 0);
#endif
}

/*
 * routine to grow window "w" to the new size of "wid" by "hei".  If the grow fails,
 * reset the window to its former size in "formerrect".
 */
void gra_mygrowwindow(WindowPtr w, INTSML wid, INTSML hei, Rect *formerrect)
{
	REGISTER WINDOWFRAME *wf;
	Rect r;
	REGISTER INTSML ret, top;

	if (w == gra_messageswindow && gra_messageswindow != 0)
	{
		SetPort(w);
		InvalRect(&w->portRect);

		gra_setview(w);
		HidePen();
		top = w->portRect.top;
		MoveControl(gra_vScroll, w->portRect.right-SBARWIDTH, top);
		SizeControl(gra_vScroll, SBARWIDTH+1, w->portRect.bottom-top-(SBARHEIGHT-2));
		MoveControl(gra_hScroll, w->portRect.left-1, w->portRect.bottom-SBARHEIGHT);
		SizeControl(gra_hScroll, w->portRect.right-w->portRect.left-(SBARWIDTH-2), SBARHEIGHT+1);
		ShowPen();

		gra_setvscroll();
		gra_adjustvtext();

		r = (*((WindowPeek)w)->strucRgn)->rgnBBox;
		gra_messagesleft = r.left;  gra_messagesright = r.right;
		gra_messagestop = r.top;    gra_messagesbottom = r.bottom;

		/* remember window settings */
		gra_savewindowsettings();
		return;
	}

	/* find the window */
	for(wf = el_firstwindowframe; wf != NOWINDOWFRAME; wf = wf->nextwindowframe)
		if (w == (WindowPtr)wf->realwindow) break;
	if (wf == NOWINDOWFRAME) return;

	/* rebuild the window pointers */
	ret = gra_remakeeditwindow(wf);

	/* no change necessary */
	if (ret < 0) return;

	/* change failed: move it back */
	if (ret > 0)
	{
#ifndef USETK
		formerrect->left += 1;         formerrect->right -= 2;
		formerrect->top += MENUSIZE;   formerrect->bottom -= 2;
		MoveWindow((WindowPtr)wf->realwindow, formerrect->left, formerrect->top, 0);
		SizeWindow((WindowPtr)wf->realwindow, formerrect->right-formerrect->left,
			formerrect->bottom-formerrect->top, 1);
#endif
		return;
	}

	/* change succeeded, redraw the window */
	gra_reloadmap();
	SetPort((WindowPtr)wf->realwindow);
	EraseRect(&wf->realwindow->portRect);
#ifdef	USETK
	DrawGrowIcon((WindowPtr)wf->realwindow);
#else
	RectRgn(wf->realwindow->clipRgn, &wf->realwindow->portRect);
	gra_drawosgraphics(wf);
#endif
	gra_redrawdisplay(wf);
}

/*
 * Routine to close the messages window if it is in front.  Returns nonzero if the
 * window was closed.
 */
INTSML closefrontmostmessages(void)
{
	if (gra_messagesinfront != 0)
	{
		gra_hidemessageswindow();
		return(1);
	}
	return(0);
}

/*
 * Routine to bring window "win" to the front.
 */
void bringwindowtofront(WINDOWFRAME *wf)
{
	SelectWindow((WindowPtr)wf->realwindow);
}

#define WMLEFTBORDER  2
#define WMTOPBORDER  MENUSIZE

/*
 * Routine to organize the windows according to "how" (0: tile horizontally,
 * 1: tile vertically, 2: cascade).
 */
void adjustwindowframe(INTSML how)
{
	RECTAREA r, wr;
	Rect fr;
	REGISTER INTBIG children, child, sizex, sizey;
	REGISTER WINDOWFRAME *wf, *compmenu;

	/* figure out how many windows need to be rearranged */
	children = 0;
	compmenu = 0;
	for(wf = el_firstwindowframe; wf != NOWINDOWFRAME; wf = wf->nextwindowframe)
		if (wf->floating == 0) children++; else
			compmenu = wf;
	if (children <= 0) return;

	/* determine area for windows */
	r.left = gra_screenleft;          r.right = gra_screenright;
	r.top = gra_screentop+MENUSIZE;   r.bottom = gra_screenbottom;

	if (compmenu != NULL)
	{
		/* remove component menu from area of tiling */
		fr = (*((WindowPeek)compmenu->realwindow)->strucRgn)->rgnBBox;
		wr.left = fr.left;   wr.right = fr.right;
		wr.top = fr.top;     wr.bottom = fr.bottom;
		gra_removewindowextent(&r, &wr);
	}
	if (gra_messageswindow != 0)
	{
		/* remove messages menu from area of tiling */
		fr = (*((WindowPeek)gra_messageswindow)->strucRgn)->rgnBBox;
		wr.left = fr.left;   wr.right = fr.right;
		wr.top = fr.top;     wr.bottom = fr.bottom;
		gra_removewindowextent(&r, &wr);
	}

	/* rearrange the windows */
	child = 0;
	for(wf = el_firstwindowframe; wf != NOWINDOWFRAME; wf = wf->nextwindowframe)
	{
		if (wf->floating != 0) continue;
		switch (how)
		{
			case 0:		/* tile horizontally */
				wr.left = r.left;
				wr.right = r.right;
				sizey = (r.bottom - r.top) / children;
				wr.top = r.top + child*sizey;
				wr.bottom = wr.top + sizey;
				break;
			case 1:		/* tile vertically */
				wr.top = r.top;
				wr.bottom = r.bottom;
				sizex = (r.right - r.left) / children;
				wr.left = r.left + child*sizex;
				wr.right = wr.left + sizex;
				break;
			case 2:		/* cascade */
				sizex = (r.right - r.left) / children / 2;
				sizey = (r.bottom - r.top) / children / 2;
				wr.left = r.left + child*sizex;
				wr.right = wr.left + (r.right-r.left)/2;
				wr.top = r.top + child*sizey;
				wr.bottom = wr.top + (r.bottom-r.top)/2;
				break;
		}
		wr.top += WMTOPBORDER;
		wr.left += WMLEFTBORDER;
		wr.right -= WMLEFTBORDER;
		wr.bottom -= WMLEFTBORDER;
		MoveWindow((WindowPtr)wf->realwindow, wr.left, wr.top, 0);
		SizeWindow((WindowPtr)wf->realwindow, wr.right-wr.left,
			wr.bottom-wr.top, 1);
		fr = (*((WindowPeek)wf->realwindow)->strucRgn)->rgnBBox;
		gra_mygrowwindow((WindowPtr)wf->realwindow, wr.right-wr.left,
			wr.bottom-wr.top, &fr);
		child++;
	}
}

/*
 * Helper routine to remove the location of window "wnd" from the rectangle "r".
 */
void gra_removewindowextent(RECTAREA *r, RECTAREA *er)
{
	if (er->right-er->left > er->bottom-er->top)
	{
		/* horizontal occluding window */
		if (er->left >= r->right || er->right <= r->left) return;
		if (er->bottom - r->top < r->bottom - er->top)
		{
			/* occluding window on top */
			r->top = er->bottom;
		} else
		{
			/* occluding window on bottom */
			r->bottom = er->top;
		}
	} else
	{
		/* vertical occluding window */
		if (er->top >= r->bottom || er->bottom <= r->top) return;
		if (er->right - r->left < r->right - er->left)
		{
			/* occluding window on left */
			r->left = er->right;
		} else
		{
			/* occluding window on right */
			r->right = er->left;
		}
	}
}

/*
 * routine to return the important pieces of information in placing the floating
 * palette with the fixed menu.  The maximum screen size is placed in "wid" and
 * "hei", and the amount of space that is being left for this palette on the
 * side of the screen is placed in "palettewidth".
 */
void getpaletteparameters(INTSML *wid, INTSML *hei, INTSML *palettewidth)
{
	*hei = gra_screenbottom - (gra_screentop+MENUSIZE) - FLOATINGHEADERSIZE;
	*wid = gra_screenright - gra_screenleft;
	*palettewidth = PALETTEWIDTH;
}

Rect *gra_geteditorwindowlocation(void)
{
	static Rect redit;
	static INTSML offset = 0;

	if (gra_multiwindow != 0)
	{
		redit.top = offset + gra_screentop + MENUSIZE;
		redit.bottom = redit.top + (gra_screenbottom-(gra_screentop+MENUSIZE)) * 3 / 4;
		redit.left = offset + gra_screenleft + PALETTEWIDTH + 1;
		redit.right = redit.left + (gra_screenright-gra_screenleft-PALETTEWIDTH-1) * 4 / 5;

		/* is the editing window visible on this display? */
		if (redit.bottom > gra_screenbottom || redit.right > gra_screenright)
		{
			offset = 0;
			redit.top = gra_screentop + MENUSIZE;
			redit.bottom = redit.top + (gra_screenbottom-(gra_screentop+MENUSIZE)) * 3 / 4;
			redit.left = gra_screenleft + PALETTEWIDTH + 1;
			redit.right = redit.left + (gra_screenright-gra_screenleft-PALETTEWIDTH-1) * 4 / 5;
		}
		offset += 30;
	} else
	{
		redit.top = gra_screentop + MENUSIZE;
		redit.bottom = gra_screenbottom;
		redit.left = gra_screenleft;
		redit.right = gra_screenright;
	}
#ifndef USETK
	redit.top += MENUSIZE;
#endif
	return(&redit);
}

/*
 * routine to build a new window frame
 */
INTSML gra_buildwindow(WINDOWFRAME *wf, INTSML floating)
{
	Rect          redit;
#ifndef	USETK
	WindowPtr     frontmost, wBehind;
#endif
	static INTSML  first = 1;
	WStateData   *wst;
	INTSML         cy, ty, left, right, top, bottom;
	char          line[200];
	Handle        name, company, special;

	wf->floating = floating;
	if (floating == 0)
	{
		/* get editor window location */
		redit = *gra_geteditorwindowlocation();

#ifdef	USETK
		/* create the editing window */
		wf->tkwin = gra_maketkwindow(2, redit.left, redit.right, redit.top, redit.bottom,
			gra_messageEventProc);
		if (wf->tkwin == NULL) return(1);
		wf->realwindow = (CGrafPtr)TkMacGetDrawablePort(Tk_WindowId(wf->tkwin));
		if (gra_tkmenusloaded != 0)
		{
			if (Tcl_VarEval(tcl_interp, Tk_PathName(wf->tkwin), " configure -menu .menus",
				0) == TCL_ERROR) ttyputerr("Tcl_VarEval returned %s", tcl_interp->result);
		}
#else
		/* deactivate any other nonfloating window that is current */
		gra_frontnonfloatingwindow(&frontmost);
		if (frontmost != 0) gra_activatewindow(frontmost, 0);

		/* figure out which window to put this behind */
		wBehind = gra_lastfloatingwindow();
		if (wBehind == 0) wBehind = (WindowPtr)-1;

		/* create the editing window */
		wf->realwindow = (CGrafPtr)NewCWindow(0L, &redit, "\p", 0, zoomDocProc, wBehind, 1, 0L);
#endif
	} else
	{
		/* default palette window location */
		redit.top = gra_screentop + MENUSIZE + 2;
		redit.bottom = (gra_screentop+gra_screenbottom)/2;
		redit.left = gra_screenleft + 1;
		redit.right = redit.left + PALETTEWIDTH/2;

		/* create the editing window */
#ifdef	USETK
		wf->tkwin = gra_maketkwindow(2, redit.left, redit.right, redit.top, redit.bottom,
			gra_messageEventProc);
		if (wf->tkwin == NULL) return(1);
		wf->realwindow = (CGrafPtr)TkMacGetDrawablePort(Tk_WindowId(wf->tkwin));
		if (gra_tkmenusloaded != 0)
		{
			if (Tcl_VarEval(tcl_interp, Tk_PathName(wf->tkwin), " configure -menu .menus",
				0) == TCL_ERROR) ttyputerr(_("Tcl_VarEval returned %s"), tcl_interp->result);
		}
#else
		wf->realwindow = (CGrafPtr)NewCWindow(0L, &redit, "\p", 0, plainDBox, (WindowRef)(-1L),
			1, 0L);
#endif
	}
	if (wf->realwindow == 0) return(1);
#ifdef	USETK
	SetPort((WindowPtr)wf->realwindow);
	EraseRect(&wf->realwindow->portRect);
	DrawGrowIcon((WindowPtr)wf->realwindow);
#else
	ShowHide((WindowRef)wf->realwindow, 1);
	gra_frontnonfloatingwindow(&frontmost);
	gra_activatewindow((WindowRef)wf->realwindow, 0);
	gra_activatewindow(frontmost, 1);
	SetPort((WindowPtr)wf->realwindow);
	EraseRect(&wf->realwindow->portRect);
	gra_drawosgraphics(wf);
#endif
	if (floating == 0)
	{
		if (first != 0)
		{
			/* get user name/company/special message */
			name = GetResource('STR ', NAMERSRC);
			company = GetResource('STR ', COMPANYRSRC);
			special = GetResource('STR ', SPECIALRSRC);

			(void)strcpy(line, _(" Version "));
			(void)strcat(line, el_version);
			line[0] = strlen(&line[1]);

			/* display the welcome message */
			TextFont(0);
			TextSize(12);
			left = wf->realwindow->portRect.left;
			right = wf->realwindow->portRect.right;
			top = wf->realwindow->portRect.top;
			bottom = wf->realwindow->portRect.bottom;
			cy = (top+bottom) / 2;
			ty = (bottom-top) / 5;
			gra_centermessage(gra_makepstring(_("Electric Design System")), left, right, ty);
			gra_centermessage(line, left, right, ty+15);
			gra_centermessage(gra_makepstring(_("Licensed from Static Free Software")),
				left, right, ty+40);
			if (name == 0)
				gra_centermessage(gra_makepstring(_("***** UNKNOWN USER *****")), left, right, cy); else
			{
				HLock(name);
				gra_centermessage(*name, left, right, cy);
				HUnlock(name);
			}
			if (company == 0)
				gra_centermessage(gra_makepstring(_("***** UNKNOWN LOCATION *****")), left, right, cy+20); else
			{
				HLock(company);
				gra_centermessage(*company, left, right, cy+20);
				HUnlock(company);
			}
			if (special != 0)
			{
				HLock(special);
				gra_centermessage(*special, left, right, cy+40);
				HUnlock(special);
			}

			gra_centermessage(gra_makepstring(_("Please wait for loading...")), left, right, bottom-30);
		}
		wst = (WStateData *) *(((WindowPeek)wf->realwindow)->dataHandle);
		wst->stdState.top = gra_screentop + MENUSIZE*2;
		wst->stdState.bottom = gra_screenbottom;
		wst->stdState.left = gra_screenleft + PALETTEWIDTH;
		wst->stdState.right = gra_screenright;
	}

	/* build the offscreen buffer for the window */
	if (gra_makeeditwindow(wf) != 0)
	{
#ifdef	USETK
		(void)Tk_DestroyWindow(wf->tkwin);
#else
		DisposeWindow((WindowPtr)wf->realwindow);
#endif
		return(1);
	}

	/* load any map the first time to establish the map segment length */
	el_maplength = 256;
	SetGWorld(wf->realwindow, gra_origgdevh);
	if (floating == 0 && first != 0) us_getcolormap(el_curtech, COLORSDEFAULT, 0); else
		gra_reloadmap();
	first = 0;

	/* want the editing window to be the primary one */
	SetPort((WindowPtr)wf->realwindow);

	return(0);
}

void gra_centermessage(char *msg, INTSML left, INTSML right, INTSML y)
{
	INTSML wid;

	wid = StringWidth((unsigned char *)msg);
	MoveTo((left+right-wid)/2, y);
	DrawString((unsigned char *)msg);
}

/*
 * Routine to redraw editing window "wf" due to a drag or grow
 */
void gra_redrawdisplay(WINDOWFRAME *wf)
{
#ifndef	USETK
	us_beginchanges();
#endif
	if (wf == NOWINDOWFRAME || wf->floating != 0) us_drawmenu(0, wf); else
		us_drawmenu(-1, wf);
	us_redostatus(wf);
#ifndef	USETK
	us_endchanges(NOWINDOWPART);
#endif
	us_state |= HIGHLIGHTSET;
	us_showallhighlight();
}

/*
 * Routine to allocate the offscreen buffer that corresponds with the window
 * "wf->realwindow".  The buffer is 8-bits deep.
 * Returns nonzero if memory cannot be allocated.
 */
INTSML gra_makeeditwindow(WINDOWFRAME *wf)
{
	INTSML height, bytes, i;
	char *addr;
	Rect r;

	r = wf->realwindow->portRect;
	if (wf->floating == 0) r.bottom -= SBARHEIGHT; else
		r.bottom -= FLOATINGHEADERSIZE;
	if (NewGWorld(&wf->window, 8, &r, 0L, 0L, 0) != noErr)
	{
		ParamText((unsigned char *)gra_makepstring(_("Cannot create the offscreen window")), "\p", "\p", "\p");
		Alert(errorALERT, 0L);
		return(1);
	}

	SetGWorld(wf->window, 0L);
	(void)LockPixels(wf->window->portPixMap);
	PenMode(patCopy);
	BackColor(0);
	EraseRect(&wf->window->portRect);
	UnlockPixels(wf->window->portPixMap);

	height = r.bottom - r.top;
	wf->rowstart = (char **)emalloc(height * SIZEOFINTBIG, us_aid->cluster);
	if (wf->rowstart == 0)
	{
		DisposeGWorld(wf->window);
		SetGWorld(wf->realwindow, gra_origgdevh);
		ParamText((unsigned char *)gra_makepstring(_("Cannot create the offscreen pointers")), "\p", "\p", "\p");
		Alert(errorALERT, 0L);
		return(1);
	}

	bytes = (*wf->window->portPixMap)->rowBytes & 0x7FFF;
	addr = GetPixBaseAddr(wf->window->portPixMap);
	for(i=0; i<height; i++)
	{
		wf->rowstart[i] = addr;
		addr += bytes;
	}
	SetGWorld(wf->realwindow, gra_origgdevh);

	wf->swid = wf->window->portRect.right - wf->window->portRect.left;
	wf->shei = wf->window->portRect.bottom - wf->window->portRect.top;
	wf->revy = wf->shei - 1;
	wf->offscreendirty = 0;
	return(0);
}

/*
 * Routine to rebuild the offscreen buffer when its size or depth has changed.
 * Returns -1 if no change is necessary, 0 if the change was successful, and
 * 1 if the change failed (memory error).
 */
INTSML gra_remakeeditwindow(WINDOWFRAME *wf)
{
	INTSML res, bytes, i, height;
	char *addr, **newrowstart;
	Rect r;

	/* quit now if there was no change */
	r = wf->realwindow->portRect;
	if (wf->floating == 0) r.bottom -= SBARHEIGHT; else
		r.bottom -= FLOATINGHEADERSIZE;
	height = r.bottom - r.top;
	if (wf->swid == r.right - r.left && wf->shei == height) return(-1);

	/* reallocate rowstart array */
	newrowstart = (char **)emalloc(height * SIZEOFINTBIG, us_aid->cluster);
	if (newrowstart == 0)
	{
		ParamText((unsigned char *)gra_makepstring(_("Not enough memory to modify the window")), "\p", "\p", "\p");
		Alert(errorALERT, 0L);
		return(1);
	}

	res = UpdateGWorld(&wf->window, 8, &r, 0L, 0L, 0);
	if (res != noErr)
	{
		efree((char *)newrowstart);
		ParamText((unsigned char *)gra_makepstring(_("Not enough memory to modify the window")), "\p", "\p", "\p");
		Alert(errorALERT, 0L);
		return(1);
	}

	/* this next line should not be needed (fixed in 7.0) */
	RectRgn(wf->window->clipRgn, &wf->window->portRect);

	/* clear the new buffer */
	SetGWorld(wf->window, 0L);
	(void)LockPixels(wf->window->portPixMap);
	PenMode(patCopy);
	BackColor(0);
	EraseRect(&wf->window->portRect);
	UnlockPixels(wf->window->portPixMap);
	SetGWorld(wf->realwindow, gra_origgdevh);

	/* load new row start array */
	bytes = (*wf->window->portPixMap)->rowBytes & 0x7FFF;
	addr = GetPixBaseAddr(wf->window->portPixMap);
	for(i=0; i<height; i++)
	{
		newrowstart[i] = addr;
		addr += bytes;
	}
	efree((char *)wf->rowstart);
	wf->rowstart = newrowstart;

	wf->swid = wf->window->portRect.right - wf->window->portRect.left;
	wf->shei = wf->window->portRect.bottom - wf->window->portRect.top;
	wf->revy = wf->shei - 1;
	wf->offscreendirty = 0;
	return(0);
}

/*
 * routine to rebuild the array "wf->rowstart".  Since it points to the
 * image buffer, and since that buffer is a handle, the buffer can be moved by
 * the system for no good reason.  Before using the array, each
 * routine should see if it is valid and call this to repair the array.
 */
void gra_rebuildrowstart(WINDOWFRAME *wf)
{
	INTSML height, bytes, i;
	char *addr;
	Rect r;

	/* the array has moved: recompute it */
	addr = GetPixBaseAddr(wf->window->portPixMap);
	r = wf->realwindow->portRect;
	if (wf->floating == 0) r.bottom -= SBARHEIGHT; else
		r.bottom -= FLOATINGHEADERSIZE;
	height = r.bottom - r.top;
	bytes = (*wf->window->portPixMap)->rowBytes & 0x7FFF;
	for(i=0; i<height; i++)
	{
		wf->rowstart[i] = addr;
		addr += bytes;
	}
}

#ifndef USETK
void gra_drawosgraphics(WINDOWFRAME *wf)
{
	WindowPtr win;

	/* figure out which window is being drawn */
	win = (WindowPtr)wf->realwindow;
	if (wf == NOWINDOWFRAME)
	{
		if (gra_messageswindow == 0) return;
		win = gra_messageswindow;
	}

	if (wf == NOWINDOWFRAME || wf->floating == 0)
	{
		/* for all but menu, draw the grow icon */
		DrawGrowIcon(win);
	}

	if (wf != NOWINDOWFRAME && wf->floating != 0)
	{
		/* for floating windows, draw the small drag bar */
		gra_drawpalettedragbar(wf);
	}
}

void gra_drawpalettedragbar(WINDOWFRAME *wf)
{
	Rect r;
	Pattern thePat;
	WindowPtr win, frontWindow;

	win = (WindowPtr)wf->realwindow;
	if (wf == NOWINDOWFRAME)
	{
		if (gra_messageswindow == 0) return;
		win = gra_messageswindow;
	}

	SetPort(win);
	r = win->portRect;
	r.bottom = FLOATINGHEADERSIZE;
	EraseRect(&r);
	MoveTo(0, FLOATINGHEADERSIZE);
	LineTo(r.right, FLOATINGHEADERSIZE);
	gra_frontnonfloatingwindow(&frontWindow);
	if (win != gra_messageswindow || win == frontWindow)
	{
		/* draw the small drag bar */
		r = win->portRect;
		r.bottom = FLOATINGHEADERSIZE;
		GetIndPattern(&thePat, 0, 24);
		PenPat(&thePat);
		PaintRect(&r);
		PenPat(&qd.black);

		/* now draw the close box */
		r.top = 2;
		r.bottom = FLOATINGHEADERSIZE-1;
		r.left = 4;
		r.right = FLOATINGHEADERSIZE+1;
		PenMode(patBic);
		PaintRect(&r);
		PenMode(patCopy);
		FrameRect(&r);
	}
}
#endif

/******************** FLOATING WINDOW ROUTINES ********************/

#ifndef USETK
/*
 * Routine to return the frontmost window that is NOT floating (zero if none).
 */
void gra_frontnonfloatingwindow(WindowPtr *result)
{
	WindowPeek   theWindow, firstWindow;
	WINDOWFRAME *wf;

	firstWindow = (WindowPeek)LMGetWindowList();
	for(theWindow = firstWindow; theWindow != 0; theWindow = theWindow->nextWindow)
	{
		/* see if this window is in the edit-window list */
		for(wf = el_firstwindowframe; wf != NOWINDOWFRAME; wf = wf->nextwindowframe)
			if (wf->realwindow == (CGrafPtr)theWindow) break;

		/* if not in the list, it must be messages or a dialog, and therefore valid */
		if (wf == NOWINDOWFRAME) break;

		/* accept editor window only if it is nonfloating */
		if (wf->floating == 0) break;
	}

	/* return what was found */
	*result = (WindowPtr)theWindow;
}

/*
 * Routine to make all floating windows visible/invisible.
 */
void gra_showallfloaters(INTSML vis)
{
	WINDOWFRAME *wf;

	for(wf = el_firstwindowframe; wf != NOWINDOWFRAME; wf = wf->nextwindowframe)
	{
		if (wf->floating == 0) continue;
		if (vis) ShowHide((WindowPtr)wf->realwindow, 1); else
			ShowHide((WindowPtr)wf->realwindow, 0);
	}
}

/*
 * Routine to return the address of the last floating window.  All normal
 * windows must go behind this.
 */
WindowPtr gra_lastfloatingwindow(void)
{
	WindowPeek   theWindow, firstWindow;
	WindowPtr    last;
	WINDOWFRAME *wf;

	last = 0;
	firstWindow = (WindowPeek)LMGetWindowList();
	for(theWindow = firstWindow; theWindow != 0; theWindow = theWindow->nextWindow)
	{
		/* see if this window is in the list */
		for(wf = el_firstwindowframe; wf != NOWINDOWFRAME; wf = wf->nextwindowframe)
			if (wf->realwindow == (CGrafPtr)theWindow) break;
		if (wf == NOWINDOWFRAME) continue;

		/* make sure it is floating */
		if (wf->floating == 0) continue;

		/* floating window found: save its address */
		last = (WindowPtr)wf->realwindow;
	}
	return(last);
}

void gra_dragfloatingwindow(WindowPtr windowToDrag, Point startPoint)
{
	Rect        dragRect, r;
	GrafPtr     savePort, windowManagerPort;
	RgnHandle   dragRegion;
	INTBIG       dragResult;
	INTSML       topLimit, newHorizontalWindowPosition, newVerticalWindowPosition,
				horizontalOffset, verticalOffset;
	WINDOWFRAME *wf;

	/* see if this is an editor window */
	for(wf = el_firstwindowframe; wf != NOWINDOWFRAME; wf = wf->nextwindowframe)
		if ((WindowPtr)wf->realwindow == windowToDrag) break;

	/* adjust the top of the dragging rectangle so that its below the menu bar */
	dragRect = qd.screenBits.bounds;
	topLimit = GetMBarHeight();
	if (dragRect.top < topLimit) dragRect.top = topLimit;
	InsetRect(&dragRect, 4, 4);

	/* Set up the Window Manager port */
	GetPort(&savePort);
	GetWMgrPort(&windowManagerPort);
	SetPort(windowManagerPort);
	SetClip(GetGrayRgn());

	/* Create a region to drag */
	dragRegion = NewRgn();

	r = (*((WindowPeek)windowToDrag)->strucRgn)->rgnBBox;
	RectRgn(dragRegion, &r);

	/* Drag the window around */
	dragResult = DragGrayRgn(dragRegion, startPoint, &dragRect, &dragRect, noConstraint, nil);

	/* Restore the port for coordinate conversion */
	SetPort(savePort);

	if (dragResult != 0)
	{
		horizontalOffset = dragResult & 0xFFFF;
		verticalOffset = dragResult >> 16;

		/* Only move it if it stayed inside the dragging box */
		if (verticalOffset != -32768)
		{
			r = (*((WindowPeek)windowToDrag)->strucRgn)->rgnBBox;
			newHorizontalWindowPosition = r.left + horizontalOffset + 1;
			newVerticalWindowPosition = r.top + verticalOffset;
			if (wf == NOWINDOWFRAME || wf->floating == 0)
				newVerticalWindowPosition += MENUSIZE;
			MoveWindow(windowToDrag, newHorizontalWindowPosition, newVerticalWindowPosition, false);
		}
	}

	/* Get rid of the dragging region */
	DisposeRgn(dragRegion);

	/* if this is the messages window, adjust the bits */
	if (wf == NOWINDOWFRAME)
	{
		r = (*((WindowPeek)windowToDrag)->strucRgn)->rgnBBox;
		gra_messagesleft = r.left;   gra_messagesright = r.right;
		gra_messagestop = r.top;     gra_messagesbottom = r.bottom;
	}
}

void gra_selectoswindow(WindowPtr windowToSelect)
{
	WindowPtr    currentFrontWindow;
	WINDOWFRAME *wf;
	WindowPtr    lastFloatingWindow;

	/* handle floating windows specially */
	for(wf = el_firstwindowframe; wf != NOWINDOWFRAME; wf = wf->nextwindowframe)
		if (wf->realwindow == (CGrafPtr)windowToSelect)
	{
		if (wf->floating != 0)
		{
			BringToFront(windowToSelect);
			return;
		}
		break;
	}

	/* quit now if this is already selected */
	gra_frontnonfloatingwindow(&currentFrontWindow);
	if (currentFrontWindow == windowToSelect) return;

	/* deactivate current frontmost window */
	if (currentFrontWindow != 0) gra_activatewindow(currentFrontWindow, 0);

	/* bring this window to the front */
	lastFloatingWindow = gra_lastfloatingwindow();
	if (lastFloatingWindow == 0) BringToFront(windowToSelect); else
		SendBehind(windowToSelect, lastFloatingWindow);

	/* activate this one */
	gra_activatewindow(windowToSelect, 1);
}

/*
 * Routine to highlight and activate a window.
 */
void gra_activatewindow(WindowPtr win, INTSML active)
{
	Rect r;
	WINDOWFRAME *wf;

	HiliteWindow(win, active);
	if (win == gra_messageswindow && gra_messageswindow != 0)
	{
		if (active != 0)
		{
			ShowControl(gra_vScroll);
			ShowControl(gra_hScroll);
			gra_messagesinfront = 1;
		} else
		{
			HideControl(gra_vScroll);
			HideControl(gra_hScroll);
			gra_messagesinfront = 0;
		}
		DrawGrowIcon(win);
		return;
	}

	/* see if it is a standard edit window */
	for(wf = el_firstwindowframe; wf != NOWINDOWFRAME; wf = wf->nextwindowframe)
		if ((WindowPtr)wf->realwindow == win) break;
	if (wf == NOWINDOWFRAME) return;
	if (wf->floating != 0) return;

	/* update the grow box in the edit window */
	r.right = win->portRect.right;
	r.left = r.right - 15;;
	r.bottom = win->portRect.bottom;
	r.top = r.bottom - 15;
	RectRgn(win->clipRgn, &r);
	DrawGrowIcon(win);
	RectRgn(win->clipRgn, &win->portRect);
}

/*
 * Routine to close a window.
 */
void gra_disposeoswindow(WindowPtr win, void *tkwin)
{
	WINDOWFRAME *wf, *lastwf;
	WindowPtr front;

	/* deactivate, make invisible, and dispose */
	gra_activatewindow(win, 0);
	ShowHide(win, 0);
	DisposeWindow(win);

	/* activate the next one (if there is) */
	gra_frontnonfloatingwindow(&front);
	if (front != 0) gra_activatewindow(front, 1);

	/* if this is an editor window, remove the structure */
	lastwf = NOWINDOWFRAME;
	for(wf = el_firstwindowframe; wf != NOWINDOWFRAME; wf = wf->nextwindowframe)
	{
		if (wf->realwindow == (CGrafPtr)win)
		{
			if (lastwf == NOWINDOWFRAME) el_firstwindowframe = wf->nextwindowframe; else
				lastwf->nextwindowframe = wf->nextwindowframe;
			DisposeGWorld(wf->window);
			efree((char *)wf);
			break;
		}
		lastwf = wf;
	}

	/* determine current window frame */
	gra_setcurrentwindowframe();
}
#endif

/*
 * routine to determine the current window frame and load the global "el_curwindowframe".
 */
void gra_setcurrentwindowframe(void)
{
	WindowPeek theWindow, firstWindow;
	WINDOWFRAME *wf;
	WINDOWPART *w;
	extern INTSML db_broadcasting;			/* nonzero if broadcasting */

	el_curwindowframe = NOWINDOWFRAME;
	firstWindow = (WindowPeek)LMGetWindowList();
	for(theWindow = firstWindow; theWindow != 0; theWindow = theWindow->nextWindow)
	{
		/* see if this window is in the edit-window list */
		for(wf = el_firstwindowframe; wf != NOWINDOWFRAME; wf = wf->nextwindowframe)
			if (wf->realwindow == (CGrafPtr)theWindow) break;

#ifdef USETK
		/* if not in the list, it must be a dialog */
		if (wf == NOWINDOWFRAME) continue;
#else
		/* if not in the list, it must be a dialog */
		if (wf == NOWINDOWFRAME)
		{
			el_curwindowframe = wf;
			break;
		}
#endif

		/* ignore floating windows */
		if (wf->floating != 0) continue;

		el_curwindowframe = wf;

		/* see if the change of window frame invalidates the current window */
		if (el_curwindowpart == NOWINDOWPART || el_curwindowpart->frame != wf)
		{
			/* must choose new window (if not broadcasting) */
			if (db_broadcasting == 0)
			{
				for(w = el_topwindowpart; w != NOWINDOWPART; w = w->nextwindowpart)
				{
					if (w->frame == wf)
					{
						(void)setvalkey((INTBIG)us_aid, VAID, us_current_window, (INTBIG)w,
							VWINDOWPART|VDONTSAVE);
						(void)setval((INTBIG)el_curlib, VLIBRARY, "curnodeproto",
							(INTBIG)w->curnodeproto, VNODEPROTO);
						break;
					}
				}
			}
		}
		break;
	}
}

void gra_savewindowsettings(void)
{
	Handle f;
	INTSML i;
	char line[256];

	/* get the current font name and messages window location */
	GetFontName(gra_messagesfont, (unsigned char *)line);

	/* remove any previous resource */
	f = GetResource('MPSR', 1004);
	if (f != 0) RemoveResource(f);

	/* make a handle for this information */
	f = NewHandle(line[0]+19);

	/* load the window location information */
	((INTSML *)(*f))[0] = gra_messagesleft;
	((INTSML *)(*f))[1] = gra_messagesright;
	((INTSML *)(*f))[2] = gra_messagestop;
	((INTSML *)(*f))[3] = gra_messagesbottom;
	((INTSML *)(*f))[4] = 0;
	((INTSML *)(*f))[5] = 0;
	((INTSML *)(*f))[6] = 0;
	((INTSML *)(*f))[7] = 0;

	/* load the font information */
	((INTSML *)(*f))[8] = gra_messagesfontsize;
	for(i=0; i<line[0]; i++) (*f)[i+18] = line[i+1];
	(*f)[line[0]+18] = 0;

	/* write the resource */
	AddResource(f, 'MPSR', 1004, "\pWindow");
	f = GetResource('MPSR', 1004);
	if (f != 0) WriteResource(f);
}

/******************** MISCELLANEOUS EXTERNAL ROUTINES ********************/

/*
 * return nonzero if the capabilities in "want" are present
 */
INTSML graphicshas(INTSML want)
{
	/* cannot run subprocesses */
	if ((want&CANRUNPROCESS) != 0) return(0);

	if (gra_multiwindow == 0)
	{
		if ((want&CANUSEFRAMES) != 0) return(0);
	}

	return(1);
}

/*
 * internal routine to make a beep
 */
void ttybeep(void)
{
	SysBeep(20);
}

void error(char *s, ...)
{
	va_list ap;
	char line[256];

	var_start(ap, s);
	(void)vsprintf(&line[1], s, ap);
	va_end(ap);
	line[0] = strlen(&line[1]);
	ParamText((unsigned char *)gra_makepstring(_("Error: ")), (unsigned char *)line, "\p", "\p");
	StopAlert(130, 0L);
	exitprogram();
}

/*
 * Routine to get the environment variable "name" and return its value.
 */
char *egetenv(char *name)
{
	return(0);
}

/*
 * Routine to fork a new process.  Returns the child process number if this is the
 * parent thread.  Returns 0 if this is the child thread.
 * Returns 1 if forking is not possible (process 1 is INIT on UNIX and can't possibly
 * be assigned to a normal process).
 */
INTBIG efork(void)
{
	return(1);
}

/*
 * Routine to run the string "command" in a shell.
 * Returns nonzero if the command cannot be run.
 */
INTBIG esystem(char *command)
{
	return(1);
}

/*
 * Routine to execute the program "program" with the arguments "args"
 */
void eexec(char *program, char *args[])
{
}

/*
 * routine to send signal "signal" to process "process".
 */
INTBIG ekill(INTBIG process)
{
	return(1);
}

/*
 * routine to wait for the completion of child process "process"
 */
void ewait(INTBIG process)
{
}

/*
 * Routine to determine the list of printers and return it.
 * The list terminates with a zero.
 */
char **eprinterlist(void)
{
	static char *lplist[1];

	lplist[0] = 0;
	return(lplist);
}

/******************** TIMING ROUTINES ********************/

#if defined(powerc) || defined(__powerc)
static void gra_dovbl(VBLRec *recPtr)
{
	long curA5;

	curA5 = SetA5(recPtr->vblA5);

	/* do the task */
	gra_motioncheck = 1;
	if (--gra_checkcountdown <= 0)
	{
		gra_checkcountdown = INTCHECK;
		gra_cancheck = 1;
	}

	/* reset counter and global pointers */
	recPtr->myVBLTask.vblCount = MOTIONCHECK;
	curA5 = SetA5(curA5);
}
#else
static pascal long GetVBLRec(void) = 0x2E88;

static void gra_dovbl(void)
{
	long curA5;
	VBLRec *recPtr;

	recPtr = (VBLRec *)GetVBLRec();
	curA5 = SetA5(recPtr->vblA5);

	/* do the task */
	gra_motioncheck = 1;
	if (--gra_checkcountdown <= 0)
	{
		gra_checkcountdown = INTCHECK;
		gra_cancheck = 1;
	}

	/* reset counter and global pointers */
	recPtr->myVBLTask.vblCount = MOTIONCHECK;
	curA5 = SetA5(curA5);
}
#endif

void gra_installvbl(void)
{
	OSErr err;

	gra_vblrec.myVBLTask.qType = vType;
	gra_vblrec.myVBLTask.vblAddr = NewVBLProc(gra_dovbl);
	gra_vblrec.myVBLTask.vblCount = MOTIONCHECK;
	gra_vblrec.myVBLTask.vblPhase = 0;
	gra_vblrec.vblA5 = SetCurrentA5();
	err = VInstall((QElemPtr)&gra_vblrec.myVBLTask);
	if (err == noErr) gra_cancheck = 1;
	gra_cancheck = 0;
	gra_motioncheck = 0;
	gra_checkcountdown = INTCHECK;
}

/******************** MESSAGES WINDOW ROUTINES ********************/

/*
 * Routine to delete the messages window.
 */
void gra_hidemessageswindow(void)
{
	if (gra_messageswindow == 0) return;
#ifdef USETK
	(void)Tk_DestroyWindow(gra_tkmessageswindow);
#else
	gra_disposeoswindow(gra_messageswindow, gra_tkmessageswindow);
#endif
	gra_messagesinfront = 0;
	gra_messageswindow = 0;
}

/*
 * Routine to create the messages window.
 */
INTSML gra_showmessageswindow(void)
{
	Rect r;

	if (gra_messageswindow == 0)
	{
#ifdef	USETK
		gra_tkmessageswindow = gra_maketkwindow(1, gra_messagesleft, gra_messagesright-3,
			gra_messagestop, gra_messagesbottom-21, gra_messageEventProc);
		if (gra_tkmessageswindow == NULL) return(1);
		gra_messageswindow = (WindowPtr)TkMacGetDrawablePort(Tk_WindowId(gra_tkmessageswindow));
		gra_namewindow(gra_tkmessageswindow, _("Electric Messages"));
		if (gra_tkmenusloaded != 0)
		{
			if (Tcl_VarEval(tcl_interp, Tk_PathName(gra_tkmessageswindow),
				" configure -menu .menus", 0) == TCL_ERROR)
					ttyputerr("Tcl_VarEval returned %s", tcl_interp->result);
		}
#else
		r.left = gra_messagesleft;   r.right = gra_messagesright;
		r.top = gra_messagestop;     r.bottom = gra_messagesbottom;
		gra_messageswindow = (WindowPtr)NewWindow(0L, &r, (unsigned char *)gra_makepstring(_("Electric Messages")),
			1, documentProc, (WindowRef)(-1L), 1, 0L);
		if (gra_messageswindow == 0) return(1);
#endif
		SetPort(gra_messageswindow);
		EraseRect(&gra_messageswindow->portRect);
		TextFont(gra_messagesfont);
		TextSize(gra_messagesfontsize);
		r = gra_messageswindow->portRect;
		r.left = r.right - SBARWIDTH;
		r.right++;
		r.bottom -= 14;
		r.top += 1;
		gra_vScroll = NewControl((WindowRef)gra_messageswindow, &r, "\p", 1, 0, 0, 0,
			scrollBarProc, 0L);
		if (gra_vScroll == 0) return(1);
		r = gra_messageswindow->portRect;
		r.top = r.bottom - SBARHEIGHT;
		r.bottom++;
		r.right -= 14;
		r.left--;
		gra_hScroll = NewControl((WindowRef)gra_messageswindow, &r, "\p", 1, 0, 0, 0,
			scrollBarProc, 0L);
		if (gra_hScroll == 0) return(1);
		HideControl(gra_vScroll);
		HideControl(gra_hScroll);
		DrawGrowIcon((WindowRef)gra_messageswindow);
		SetControlMinimum(gra_hScroll, 0);
		SetControlMaximum(gra_hScroll, 90);
	}
	if (gra_TEH == 0)
	{
		r = gra_messageswindow->portRect;
		gra_TEH = TENew(&r, &r);
		if (gra_TEH == 0) return(1);
	}
	gra_setview(gra_messageswindow);
	TEActivate(gra_TEH);
	return(0);
}

void gra_setview(WindowPtr w)
{
	INTSML width;

	(*gra_TEH)->viewRect = w->portRect;
	(*gra_TEH)->viewRect.right -= SBARWIDTH;
	(*gra_TEH)->viewRect.bottom -= SBARHEIGHT;
	width = (*gra_TEH)->viewRect.right - (*gra_TEH)->viewRect.left;
	InsetRect(&(*gra_TEH)->viewRect, 4, 4);
	(*gra_TEH)->destRect = (*gra_TEH)->viewRect;
	gra_linesInFolder = ((*gra_TEH)->destRect.bottom - (*gra_TEH)->destRect.top) /
		(*gra_TEH)->lineHeight;
	(*gra_TEH)->destRect.bottom = (*gra_TEH)->destRect.top + (*gra_TEH)->lineHeight *
		gra_linesInFolder;
	(*gra_TEH)->destRect.left += 4;
	(*gra_TEH)->destRect.right = (*gra_TEH)->destRect.left + width*10;
	TECalText(gra_TEH);
}

pascal void gra_scrollvproc(ControlHandle theControl, INTSML theCode)
{
	INTSML pageSize, scrollAmt;

	if (theCode == 0) return;
	pageSize = ((*gra_TEH)->viewRect.bottom-(*gra_TEH)->viewRect.top) / (*gra_TEH)->lineHeight - 1;
	switch (theCode)
	{
		case kControlUpButtonPart:   scrollAmt = -1;          break;
		case kControlDownButtonPart: scrollAmt = 1;           break;
		case kControlPageUpPart:     scrollAmt = -pageSize;   break;
		case kControlPageDownPart:   scrollAmt = pageSize;    break;
	}
	SetControlValue(theControl, GetControlValue(theControl)+scrollAmt);
	gra_adjustvtext();
}

pascal void gra_scrollhproc(ControlHandle theControl, INTSML theCode)
{
	INTSML scrollAmt, pos, oldpos;

	if (theCode == 0) return;
	switch (theCode)
	{
		case kControlUpButtonPart:   scrollAmt = -1;    break;
		case kControlDownButtonPart: scrollAmt = 1;     break;
		case kControlPageUpPart:     scrollAmt = -10;   break;
		case kControlPageDownPart:   scrollAmt = 10;    break;
	}
	oldpos = GetControlValue(theControl);
	pos = oldpos + scrollAmt;
	if (pos < 0) pos = 0;
	if (pos > 90) pos = 90;
	SetControlValue(theControl, pos);
	gra_adjusthtext(oldpos);
}

void gra_adjustvtext(void)
{
	INTSML oldScroll, newScroll, delta;

	oldScroll = (*gra_TEH)->viewRect.top - (*gra_TEH)->destRect.top;
	newScroll = GetControlValue(gra_vScroll) * (*gra_TEH)->lineHeight;
	delta = oldScroll - newScroll;
	if (delta != 0) TEScroll(0, delta, gra_TEH);
}

void gra_adjusthtext(INTSML oldpos)
{
	INTSML pos, wid, delta;

	pos = GetControlValue(gra_hScroll);
	wid = ((*gra_TEH)->viewRect.right - (*gra_TEH)->viewRect.left) / 10;
	delta = wid * (pos - oldpos);
	if (delta != 0) TEScroll(-delta, 0, gra_TEH);
}

void gra_setvscroll(void)
{
	INTSML n;

	n = (*gra_TEH)->nLines - gra_linesInFolder + 1;
	if ((*gra_TEH)->teLength > 0 && (*((*gra_TEH)->hText))[(*gra_TEH)->teLength-1] != '\r') n++;
	SetControlMaximum(gra_vScroll, n > 0 ? n : 0);
}

void gra_showselect(void)
{
	INTSML topLine, bottomLine, theLine;

	gra_setvscroll();
	gra_adjustvtext();

	topLine = GetControlValue(gra_vScroll);
	bottomLine = topLine + gra_linesInFolder;

	if ((*gra_TEH)->selStart < (*gra_TEH)->lineStarts[topLine] ||
		(*gra_TEH)->selStart >= (*gra_TEH)->lineStarts[bottomLine])
	{
		for (theLine = 0; (*gra_TEH)->selStart >= (*gra_TEH)->lineStarts[theLine]; theLine++) ;
		SetControlValue(gra_vScroll, theLine - gra_linesInFolder / 2);
		gra_adjustvtext();
	}
}

/*
 * routine to cut text from the messages window if it is current.  Returns nonzero
 * if sucessful.
 */
INTSML cutfrommessages(void)
{
	void io_maccopyhighlighted(void);

	/* if a desk accessory wants the "cut" command, stop now */
	if (SystemEdit(2) != 0) return(1);

	if (gra_messagesinfront != 0)
	{
		TECut(gra_TEH);
		ZeroScrap();
		TEToScrap();
		return(1);
	}
	if (el_curwindowpart == NOWINDOWPART) return(0);
	if ((el_curwindowpart->state&WINDOWTYPE) == DISPWINDOW)
		io_maccopyhighlighted();
	return(0);
}

/*
 * routine to copy text from the messages window if it is current.  Returns nonzero
 * if sucessful.
 */
INTSML copyfrommessages(void)
{
	void io_maccopyhighlighted(void);

	/* if a desk accessory wants the "copy" command, stop now */
	if (SystemEdit(3) != 0) return(1);

	if (gra_messagesinfront != 0)
	{
		TECopy(gra_TEH);
		ZeroScrap();
		TEToScrap();
		return(1);
	}
	if (el_curwindowpart == NOWINDOWPART) return(0);
	if ((el_curwindowpart->state&WINDOWTYPE) == DISPWINDOW)
		io_maccopyhighlighted();
	return(0);
}

/*
 * routine to paste text to the messages window if it is current.  Returns nonzero
 * if sucessful.
 */
INTSML pastetomessages(void)
{
	/* if a desk accessory wants the "paste" command, stop now */
	if (SystemEdit(4) != 0) return(1);

	if (gra_messagesinfront != 0)
	{
		TEFromScrap();
		TEPaste(gra_TEH);
		return(1);
	}
	return(0);
}

/*
 * routine to get the contents of the system cut buffer
 */
char *getcutbuffer(void)
{
	Handle sc;
	INTBIG scoffset, len, i;

	/* get cut buffer */
	sc = NewHandle(0);
	len = GetScrap(sc, 'TEXT', (long *)&scoffset);
	(void)initinfstr();
	for(i=0; i<len; i++) (void)addtoinfstr((*sc)[i]);
	return(returninfstr());
}

/*
 * routine to set the contents of the system cut buffer to "msg"
 */
void setcutbuffer(char *msg)
{
	ZeroScrap();
	PutScrap(strlen(msg), 'TEXT', msg);
}

/*
 * routine to select fonts in the messages window
 * Font list = 5 (user item)
 * Size      = 6 (edit text)
 */
void setmessagesfont(void)
{
	INTSML itemHit, i, tot, id, which, fontnum;
	INTBIG typ, fonttype;
	Handle f;
	char line[256];
	FontInfo finfo;

	/* display the font dialog box */
	DiaInitDialog(&gra_fontdialog);
	DiaInitTextDialog(5, DiaNullDlogList, DiaNullDlogItem, DiaNullDlogDone, -1,
		SCSELMOUSE|SCDOUBLEQUIT);

	/* if there are 'FONT's, then no names will be obtained by "GetResInfo" below!!! */
	fonttype = 'FOND';
	tot = CountResources(fonttype);
	if (tot == 0)
	{
		fonttype = 'FONT';
		tot = CountResources(fonttype);
	}
	which = fontnum = 0;
	for(i=1; i<=tot; i++)
	{
		SetResLoad(0);
		f = GetIndResource(fonttype, i);
		GetResInfo(f, &id, (ResType *)&typ, (unsigned char *)line);
		SetResLoad(1);
		if (line[0] == 0) continue;
		GetFNum((unsigned char *)line, &id);
		if (id == gra_messagesfont) which = fontnum;
		line[line[0]+1] = 0;
		DiaStuffLine(5, &line[1]);
		fontnum++;
	}
	DiaSelectLine(5, which);
	(void)sprintf(line, "%d", gra_messagesfontsize);
	DiaSetText(-6, line);

	/* loop until done */
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == OK || itemHit == CANCEL) break;
	}

	if (itemHit != CANCEL)
	{
		(void)strcpy(&line[1], DiaGetScrollLine(5, DiaGetCurLine(5)));
		line[0] = strlen(&line[1]);
		GetFNum((unsigned char *)line, &id);
		i = myatoi(DiaGetText(6));
		if (i != gra_messagesfontsize || id != gra_messagesfont)
		{
			if (gra_messageswindow == 0)
			{
				if (gra_showmessageswindow() != 0) return;
			}

			gra_messagesfontsize = i;
			gra_messagesfont = id;
			SetPort(gra_messageswindow);
			TextSize(i);
			TextFont(id);
			GetFontInfo(&finfo);
			(*gra_TEH)->txSize = i;
			(*gra_TEH)->txFont = id;
			(*gra_TEH)->fontAscent = finfo.ascent;
			(*gra_TEH)->lineHeight = finfo.ascent + finfo.descent + finfo.leading;
			gra_setview(gra_messageswindow);
			EraseRect(&gra_messageswindow->portRect);
			DrawControls(gra_messageswindow);
			DrawGrowIcon(gra_messageswindow);
			TEUpdate(&gra_messageswindow->portRect, gra_TEH);

			/* save new messages window settings */
			gra_savewindowsettings();
		}
	}
	DiaDoneDialog();
}

/*
 * Routine to put the string "s" into the messages window.
 * Pops up the messages window if "important" is nonzero.
 */
void putmessagesstring(char *s, INTSML important)
{
	INTBIG len, insert;
	Rect r;
	GrafPtr savePort;
	WINDOWFRAME *wf;

	len = strlen(s);
	GetPort(&savePort);
	if (gra_messageswindow == 0)
	{
		if (important == 0) return;
		if (gra_showmessageswindow() != 0) return;
	}

	/* see if the messages window occludes any edit window */
	if (important != 0)
	{
		for(wf = el_firstwindowframe; wf != NOWINDOWFRAME; wf = wf->nextwindowframe)
		{
			if (SectRect(&(*((WindowPeek)gra_messageswindow)->strucRgn)->rgnBBox,
				&(*((WindowPeek)wf->realwindow)->strucRgn)->rgnBBox, &r) != 0)
			{
#ifdef	USETK
				gra_raisewindow(gra_tkmessageswindow);
#else
				gra_selectoswindow(gra_messageswindow);
#endif
				break;
			}
		}
	}

	if ((*gra_TEH)->teLength > 30000)
	{
		/* cut out the top half of the buffer before it overflows */
		TESetSelect(0, 15000, gra_TEH);
		TEKey(DELETEKEY, gra_TEH);
		insert = 32767;
		TESetSelect(insert, insert, gra_TEH);
	}
	TEInsert(s, len, gra_TEH);
	TEKey('\r', gra_TEH);
	insert = 32767;
	TESetSelect(insert, insert, gra_TEH);
	gra_showselect();
	SetPort(savePort);
}

/*
 * Routine to return the name of the key that ends a session from the messages window.
 */
char *getmessageseofkey(void)
{
	return("^D");
}

/*
 * Routine to get a string from the scrolling messages window.  Returns zero if end-of-file
 * (^D) is typed.
 */
char *getmessagesstring(char *prompt)
{
	EventRecord theEvent;
	INTSML start, end, i, j, ch;
	static char outline[256];

	if (gra_messageswindow == 0)
	{
		if (gra_showmessageswindow() != 0) return("");
	}
#ifdef	USETK
	gra_raisewindow(gra_tkmessageswindow);
#else
	gra_selectoswindow(gra_messageswindow);
#endif

	/* show the prompt */
	TESetSelect(32767, 32767, gra_TEH);
	TEInsert(prompt, strlen(prompt), gra_TEH);
	TESetSelect(32767, 32767, gra_TEH);
	start = (*gra_TEH)->selStart;
	for(;;)
	{
		gra_waitforaction(0, &theEvent);
		end = (*gra_TEH)->selStart-1;
		if (end < start) continue;
		ch = (*(*gra_TEH)->hText)[end];
		if (ch == '\r' || ch == 4) break;
	}
	TEKey(DELETEKEY, gra_TEH);
	TEKey('\r', gra_TEH);
	if (ch == 4) return(0);
	j = 0;
	for(i=start; i<end; i++) outline[j++] = (*(*gra_TEH)->hText)[i];
	outline[j] = 0;
	return(outline);
}

/*
 * Routine to remove the last character from the scrolling messages window.
 * Called from "usrterminal.c"
 */
void gra_backup(void)
{
	GrafPtr savePort;

	GetPort(&savePort);
	if (gra_messageswindow == 0)
	{
		if (gra_showmessageswindow() != 0) return;
	}
	TEKey(DELETEKEY, gra_TEH);
	gra_showselect();
	SetPort(savePort);
}

/******************** STATUS BAR ROUTINES ********************/

/*
 * Routine to return the number of status lines on the display.
 */
INTSML ttynumstatuslines(void)
{
	return(MAXSTATUSLINES);
}

/*
 * Routine to display "message" in the status "field" of window "frame" (uses all windows
 * if "frame" is zero).  If "cancrop" is zero, field cannot be cropped and should be
 * replaced with "*" if there isn't room.
 */
void ttysetstatusfield(WINDOWFRAME *mwwant, STATUSFIELD *sf, char *message, INTSML cancrop)
{
	INTSML len, i, width, winwid, startx, endx;
	Rect clear;
	REGISTER WINDOWFRAME *wf;

	if (sf == 0) return;

	for(wf = el_firstwindowframe; wf != NOWINDOWFRAME; wf = wf->nextwindowframe)
	{
		if (wf->floating != 0) continue;
		if (mwwant != NOWINDOWFRAME && mwwant != wf) continue;

		/* construct the status line */
		(void)strcpy(&gra_localstring[1], sf->label);
		(void)strcat(&gra_localstring[1], message);
		len = strlen(&gra_localstring[1]);
		while (len > 0 && gra_localstring[len] == ' ') gra_localstring[len--] = 0;

		/* special case for window title */
		if (sf->line == 0)
		{
#ifdef	USETK
			gra_namewindow(wf->tkwin, &gra_localstring[1]);
#else
			gra_localstring[0] = strlen(&gra_localstring[1]);
			SetWTitle((WindowPtr)wf->realwindow, (unsigned char *)gra_localstring);
#endif
			return;
		}

		/* determine how much room there is for the status field */
		SetPort((WindowPtr)wf->realwindow);
		winwid = wf->realwindow->portRect.right - wf->realwindow->portRect.left - SBARWIDTH;
		startx = winwid * sf->startper / 100 + wf->realwindow->portRect.left;
		endx = winwid * sf->endper / 100 + wf->realwindow->portRect.left;

		/* make sure the message fits */
		width = TextWidth(&gra_localstring[1], 0, len);
		if (width > endx-startx && cancrop == 0)
		{
			for(i=strlen(sf->label); gra_localstring[i+1] != 0; i++)
				gra_localstring[i+1] = '*';
		}
		while (len > 0 && width > endx-startx)
		{
			len--;
			width = TextWidth(&gra_localstring[1], 0, len);
		}

		/* display the field */
		TextFont(SFONT);
		TextSize(9);
		TextMode(srcOr);
		clear.left = startx;
		clear.right = endx;
		clear.bottom = wf->realwindow->portRect.bottom;
		clear.top = clear.bottom - SBARHEIGHT+1;
		EraseRect(&clear);
		if (len > 0)
		{
			MoveTo(startx, wf->realwindow->portRect.bottom-4);
			DrawText(&gra_localstring[1], 0, len);
		}
	}
}

/*
 * Routine to free status field object "sf".
 */
void ttyfreestatusfield(STATUSFIELD *sf)
{
	efree(sf->label);
	efree((char *)sf);
}

/******************** GRAPHICS CONTROL ROUTINES ********************/

void flushscreen(void)
{
	WINDOWFRAME *wf;
	Rect dr, copyrect;

	for(wf = el_firstwindowframe; wf != NOWINDOWFRAME; wf = wf->nextwindowframe)
	{
		/* if screen has not changed, stop now */
		if (wf->offscreendirty == 0) continue;
		wf->offscreendirty = 0;

		/* make sure region falls inside screen */
		SetPort((WindowPtr)wf->realwindow);
		copyrect.left = wf->copyrect.left;   copyrect.right = wf->copyrect.right;
		copyrect.top = wf->copyrect.top;     copyrect.bottom = wf->copyrect.bottom;
		if (copyrect.left < 0) copyrect.left = 0;
		if (copyrect.right > (*wf->window->portPixMap)->bounds.right)
			copyrect.right = (*wf->window->portPixMap)->bounds.right;
		if (copyrect.top < 0) copyrect.top = 0;
		if (copyrect.bottom > (*wf->window->portPixMap)->bounds.bottom)
			copyrect.bottom = (*wf->window->portPixMap)->bounds.bottom;

		(void)LockPixels(wf->window->portPixMap);
		if (wf->floating == 0)
		{
			CopyBits((BitMap *)*(wf->window->portPixMap), (BitMap *)*wf->realwindow->portPixMap,
				&copyrect, &copyrect, srcCopy, 0L);
		} else
		{
			dr = copyrect;
			dr.top += FLOATINGHEADERSIZE;
			dr.bottom += FLOATINGHEADERSIZE;
			CopyBits((BitMap *)*(wf->window->portPixMap), (BitMap *)*wf->realwindow->portPixMap,
				&copyrect, &dr, srcCopy, 0L);
		}
		UnlockPixels(wf->window->portPixMap);
	}
}

/*
 * Routine to accumulate the rectangle of change to the offscreen PixMap
 */
void gra_setrect(WINDOWFRAME *wf, INTSML lx, INTSML hx, INTSML ly, INTSML hy)
{
	UINTBIG thistime;

	thistime = TickCount();
	if (wf->offscreendirty == 0)
	{
		wf->copyrect.left = lx;   wf->copyrect.right = hx;
		wf->copyrect.top = ly;    wf->copyrect.bottom = hy;
		wf->offscreendirty = 1;
		wf->starttime = thistime;
	} else
	{
		if (lx < wf->copyrect.left) wf->copyrect.left = lx;
		if (hx > wf->copyrect.right) wf->copyrect.right = hx;
		if (ly < wf->copyrect.top) wf->copyrect.top = ly;
		if (hy > wf->copyrect.bottom) wf->copyrect.bottom = hy;
	}

	/* flush the screen every two seconds */
	if (thistime - wf->starttime > FLUSHTICKS) flushscreen();
}

void gra_reloadmap(void)
{
	REGISTER VARIABLE *varred, *vargreen, *varblue;

	varred = getvalkey((INTBIG)us_aid, VAID, VINTEGER|VISARRAY, us_colormap_red);
	vargreen = getvalkey((INTBIG)us_aid, VAID, VINTEGER|VISARRAY, us_colormap_green);
	varblue = getvalkey((INTBIG)us_aid, VAID, VINTEGER|VISARRAY, us_colormap_blue);
	if (varred == NOVARIABLE || vargreen == NOVARIABLE || varblue == NOVARIABLE) return;
	colormapload((INTBIG *)varred->addr, (INTBIG *)vargreen->addr, (INTBIG *)varblue->addr, 0, 255);
}

void colormapload(INTBIG *red, INTBIG *green, INTBIG *blue, INTSML low, INTSML high)
{
	INTSML i;
	CTabHandle clut;
	REGISTER WINDOWFRAME *wf;

	for(wf = el_firstwindowframe; wf != NOWINDOWFRAME; wf = wf->nextwindowframe)
	{
		clut = (*wf->window->portPixMap)->pmTable;
		for(i=low; i<=high; i++)
		{
			(*clut)->ctTable[i].rgb.red = red[i-low] << 8;
			(*clut)->ctTable[i].rgb.green = green[i-low] << 8;
			(*clut)->ctTable[i].rgb.blue = blue[i-low] << 8;
			if (i == 255)
			{
				(*clut)->ctTable[i].rgb.red = (255-red[i-low]) << 8;
				(*clut)->ctTable[i].rgb.green = (255-green[i-low]) << 8;
				(*clut)->ctTable[i].rgb.blue = (255-blue[i-low]) << 8;
			}
		}
		(*clut)->ctSeed++;

		/* mark the entire screen for redrawing */
		if (low == 0 && high == 255)
			gra_setrect(wf, 0, (*wf->window->portPixMap)->bounds.right, 0,
				(*wf->window->portPixMap)->bounds.bottom);
	}

	/* recache table and all PixMaps */
	CTabChanged(clut);
}

/*
 * helper routine to set the cursor shape to "state"
 */
void setdefaultcursortype(INTSML state)
{
	if (us_cursorstate == state) return;

	switch (state)
	{
		case NORMALCURSOR:
			SetCursor(&qd.arrow);
			break;
		case WANTTTYCURSOR:
			if (gra_wantttyCurs != 0L) SetCursor(&(**gra_wantttyCurs)); else
			SetCursor(&qd.arrow);
			break;
		case PENCURSOR:
			if (gra_penCurs != 0L) SetCursor(&(**gra_penCurs)); else
				SetCursor(&qd.arrow);
			break;
		case NULLCURSOR:
			if (gra_nullCurs != 0L) SetCursor(&(**gra_nullCurs)); else
				SetCursor(&qd.arrow);
			break;
		case MENUCURSOR:
			if (gra_menuCurs != 0L) SetCursor(&(**gra_menuCurs)); else
				SetCursor(&qd.arrow);
			break;
		case HANDCURSOR:
			if (gra_handCurs != 0L) SetCursor(&(**gra_handCurs)); else
				SetCursor(&qd.arrow);
			break;
		case TECHCURSOR:
			if (gra_techCurs != 0L) SetCursor(&(**gra_techCurs)); else
				SetCursor(&qd.arrow);
			break;
		case IBEAMCURSOR:
			if (gra_ibeamCurs != 0L) SetCursor(&(**gra_ibeamCurs)); else
				SetCursor(&qd.arrow);
			break;
		case LRCURSOR:
			if (gra_lrCurs != 0L) SetCursor(&(**gra_lrCurs)); else
				SetCursor(&qd.arrow);
			break;
		case UDCURSOR:
			if (gra_udCurs != 0L) SetCursor(&(**gra_udCurs)); else
				SetCursor(&qd.arrow);
			break;
	}
	us_cursorstate = state;
}

/*
 * Routine to change the default cursor (to indicate modes).
 */
void setnormalcursor(INTSML curs)
{
	us_normalcursor = curs;
}

/******************** LINE DRAWING ********************/

void screeninvertline(WINDOWPART *win, INTSML x1, INTSML y1, INTSML x2, INTSML y2)
{
	REGISTER INTSML lx, hx, ly, hy, dx, dy, d, incr1, incr2, x, y, xend, yend, yincr, xincr;
	REGISTER WINDOWFRAME *wf;

	/* make sure the image buffer has not moved */
	wf = win->frame;
	if (GetPixBaseAddr(wf->window->portPixMap) != wf->rowstart[0])
		gra_rebuildrowstart(wf);

	/* get line type parameters */
	y1 = wf->revy - y1;   y2 = wf->revy - y2;

	/* initialize the Bresenham algorithm */
	dx = abs(x2-x1);
	dy = abs(y2-y1);
	if (dx > dy)
	{
		/* initialize for lines that increment along X */
		incr1 = 2 * dy;
		d = incr2 = 2 * (dy - dx);
		if (x1 > x2)
		{
			x = x2;   y = y2;   xend = x1;   yend = y1;
		} else
		{
			x = x1;   y = y1;   xend = x2;   yend = y2;
		}
		if (yend < y) yincr = -1; else yincr = 1;
		wf->rowstart[y][x] = ~wf->rowstart[y][x];

		/* draw line that increments along X */
		while (x < xend)
		{
			x++;
			if (d < 0) d += incr1; else
			{
				y += yincr;   d += incr2;
			}
			wf->rowstart[y][x] = ~wf->rowstart[y][x];
		}
	} else
	{
		/* initialize for lines that increment along Y */
		incr1 = 2 * dx;
		d = incr2 = 2 * (dx - dy);
		if (y1 > y2)
		{
			x = x2;   y = y2;   xend = x1;   yend = y1;
		} else
		{
			x = x1;   y = y1;   xend = x2;   yend = y2;
		}
		if (xend < x) xincr = -1; else xincr = 1;
		wf->rowstart[y][x] = ~wf->rowstart[y][x];

		/* draw line that increments along X */
		while (y < yend)
		{
			y++;
			if (d < 0) d += incr1; else
			{
				x += xincr;   d += incr2;
			}
			wf->rowstart[y][x] = ~wf->rowstart[y][x];
		}
	}
	if (x1 < x2) { lx = x1;   hx = x2; } else { lx = x2;   hx = x1; }
	if (y1 < y2) { ly = y1;   hy = y2; } else { ly = y2;   hy = y1; }
	gra_setrect(wf, lx, hx+1, ly, hy+1);
}

void screendrawline(WINDOWPART *win, INTSML x1, INTSML y1, INTSML x2, INTSML y2, GRAPHICS *desc, INTSML texture)
{
	REGISTER INTSML col, mask, lx, hx, ly, hy;
	static INTSML pat[] = {0xFF, 0x88, 0xE7, 0x80};
	REGISTER WINDOWFRAME *wf;

#ifdef	INVERTHIGHLIGHT
	/* if layer is HIGHLIGHT, invert */
	if (desc->bits == LAYERH)
	{
		screeninvertline(win, x1, y1, x2, y2);
		return;
	}
#endif

	/* make sure the image buffer has not moved */
	wf = win->frame;
	if (GetPixBaseAddr(wf->window->portPixMap) != wf->rowstart[0])
		gra_rebuildrowstart(wf);

	/* get line type parameters */
	col = desc->col;         mask = ~desc->bits;
	y1 = wf->revy - y1;   y2 = wf->revy - y2;

	if (texture != 0) gra_drawpatline(wf, x1, y1, x2, y2, col, mask, pat[texture]); else
		gra_drawline(wf, x1, y1, x2, y2, col, mask);
	if (x1 < x2) { lx = x1;   hx = x2; } else { lx = x2;   hx = x1; }
	if (y1 < y2) { ly = y1;   hy = y2; } else { ly = y2;   hy = y1; }
	gra_setrect(wf, lx, hx+1, ly, hy+1);
}

void gra_drawpatline(WINDOWFRAME *wf, INTSML x1, INTSML y1, INTSML x2, INTSML y2, INTSML col,
	INTSML mask, INTSML pattern)
{
	INTSML dx, dy, d, incr1, incr2, x, y, xend, yend, yincr, xincr, i;

	/* initialize counter for line style */
	i = 0;

	/* initialize the Bresenham algorithm */
	dx = abs(x2-x1);
	dy = abs(y2-y1);
	if (dx > dy)
	{
		/* initialize for lines that increment along X */
		incr1 = 2 * dy;
		d = incr2 = 2 * (dy - dx);
		if (x1 > x2)
		{
			x = x2;   y = y2;   xend = x1;   yend = y1;
		} else
		{
			x = x1;   y = y1;   xend = x2;   yend = y2;
		}
		if (yend < y) yincr = -1; else yincr = 1;
		wf->rowstart[y][x] = (wf->rowstart[y][x]&mask) | col;

		/* draw line that increments along X */
		while (x < xend)
		{
			x++;
			if (d < 0) d += incr1; else
			{
				y += yincr;   d += incr2;
			}
			if (i == 7) i = 0; else i++;
			if ((pattern & (1 << i)) == 0) continue;
			wf->rowstart[y][x] = (wf->rowstart[y][x]&mask) | col;
		}
	} else
	{
		/* initialize for lines that increment along Y */
		incr1 = 2 * dx;
		d = incr2 = 2 * (dx - dy);
		if (y1 > y2)
		{
			x = x2;   y = y2;   xend = x1;   yend = y1;
		} else
		{
			x = x1;   y = y1;   xend = x2;   yend = y2;
		}
		if (xend < x) xincr = -1; else xincr = 1;
		wf->rowstart[y][x] = (wf->rowstart[y][x]&mask) | col;

		/* draw line that increments along X */
		while (y < yend)
		{
			y++;
			if (d < 0) d += incr1; else
			{
				x += xincr;   d += incr2;
			}
			if (i == 7) i = 0; else i++;
			if ((pattern & (1 << i)) == 0) continue;
			wf->rowstart[y][x] = (wf->rowstart[y][x]&mask) | col;
		}
	}
}

void gra_drawline(WINDOWFRAME *wf, INTSML x1, INTSML y1, INTSML x2, INTSML y2, INTSML col, INTSML mask)
{
	INTSML dx, dy, d, incr1, incr2, x, y, xend, yend, yincr, xincr;

	/* initialize the Bresenham algorithm */
	dx = abs(x2-x1);
	dy = abs(y2-y1);
	if (dx > dy)
	{
		/* initialize for lines that increment along X */
		incr1 = 2 * dy;
		d = incr2 = 2 * (dy - dx);
		if (x1 > x2)
		{
			x = x2;   y = y2;   xend = x1;   yend = y1;
		} else
		{
			x = x1;   y = y1;   xend = x2;   yend = y2;
		}
		if (yend < y) yincr = -1; else yincr = 1;
		wf->rowstart[y][x] = (wf->rowstart[y][x]&mask) | col;

		/* draw line that increments along X */
		while (x < xend)
		{
			x++;
			if (d < 0) d += incr1; else
			{
				y += yincr;   d += incr2;
			}
			wf->rowstart[y][x] = (wf->rowstart[y][x]&mask) | col;
		}
	} else
	{
		/* initialize for lines that increment along Y */
		incr1 = 2 * dx;
		d = incr2 = 2 * (dx - dy);
		if (y1 > y2)
		{
			x = x2;   y = y2;   xend = x1;   yend = y1;
		} else
		{
			x = x1;   y = y1;   xend = x2;   yend = y2;
		}
		if (xend < x) xincr = -1; else xincr = 1;
		wf->rowstart[y][x] = (wf->rowstart[y][x]&mask) | col;

		/* draw line that increments along X */
		while (y < yend)
		{
			y++;
			if (d < 0) d += incr1; else
			{
				x += xincr;   d += incr2;
			}
			wf->rowstart[y][x] = (wf->rowstart[y][x]&mask) | col;
		}
	}
}

/******************** POLYGON DRAWING ********************/

/*
 * routine to draw a polygon in the arrays (x, y) with "count" points.
 */
void screendrawpolygon(WINDOWPART *win, INTBIG *x, INTBIG *y, INTSML count, GRAPHICS *desc)
{
	REGISTER INTBIG i, j, k, l, ycur, yrev, wrap, lx, hx, ly, hy;
	REGISTER char *row;
	REGISTER INTSML col, mask, style, pat;
	REGISTER POLYSEG *a, *active, *edge, *lastedge, *left, *edgelist;
	REGISTER WINDOWFRAME *wf;

	/* make sure the image buffer has not moved */
	wf = win->frame;
	if (GetPixBaseAddr(wf->window->portPixMap) != wf->rowstart[0])
		gra_rebuildrowstart(wf);

	/* get parameters */
	col = desc->col;   mask = ~desc->bits;
	style = desc->colstyle & NATURE;

	/* set redraw area */
	for(i=0; i<count; i++)
	{
		if (i == 0)
		{
			lx = hx = x[i];
			ly = hy = y[i];
		} else
		{
			lx = mini(lx, x[i]);
			hx = maxi(hx, x[i]);
			ly = mini(ly, y[i]);
			hy = maxi(hy, y[i]);
		}
	}

	/* color: make sure there is room in internal structures */
	if (count > gra_polysegcount)
	{
		if (gra_polysegcount > 0) efree((char *)gra_polysegs);
		gra_polysegcount = 0;
		gra_polysegs = (POLYSEG *)emalloc(count * (sizeof (POLYSEG)), us_aid->cluster);
		if (gra_polysegs == 0) return;
		gra_polysegcount = count;
	}

	/* fill in internal structures */
	edgelist = NOPOLYSEG;
	for(i=0; i<count; i++)
	{
		if (i == 0)
		{
			gra_polysegs[i].fx = x[count-1];
			gra_polysegs[i].fy = y[count-1];
		} else
		{
			gra_polysegs[i].fx = x[i-1];
			gra_polysegs[i].fy = y[i-1];
		}
		gra_polysegs[i].tx = x[i];   gra_polysegs[i].ty = y[i];

		/* draw the edge lines to make the polygon clean */
		if ((desc->colstyle&(NATURE|OUTLINEPAT)) != PATTERNED)
			gra_drawline(wf, (INTSML)gra_polysegs[i].fx, (INTSML)(wf->revy - gra_polysegs[i].fy),
				(INTSML)gra_polysegs[i].tx, (INTSML)(wf->revy - gra_polysegs[i].ty), col, mask);

		/* compute the direction of this edge */
		j = gra_polysegs[i].ty - gra_polysegs[i].fy;
		if (j > 0) gra_polysegs[i].direction = 1; else
			if (j < 0) gra_polysegs[i].direction = -1; else
				gra_polysegs[i].direction = 0;

		/* compute the X increment of this edge */
		if (j == 0) gra_polysegs[i].increment = 0; else
		{
			gra_polysegs[i].increment = gra_polysegs[i].tx - gra_polysegs[i].fx;
			if (gra_polysegs[i].increment != 0) gra_polysegs[i].increment =
				(gra_polysegs[i].increment * 65536 - j + 1) / j;
		}
		gra_polysegs[i].tx <<= 16;   gra_polysegs[i].fx <<= 16;

		/* make sure "from" is above "to" */
		if (gra_polysegs[i].fy > gra_polysegs[i].ty)
		{
			j = gra_polysegs[i].tx;
			gra_polysegs[i].tx = gra_polysegs[i].fx;
			gra_polysegs[i].fx = j;
			j = gra_polysegs[i].ty;
			gra_polysegs[i].ty = gra_polysegs[i].fy;
			gra_polysegs[i].fy = j;
		}

		/* insert this edge into the edgelist, sorted by ascending "fy" */
		if (edgelist == NOPOLYSEG)
		{
			edgelist = &gra_polysegs[i];
			gra_polysegs[i].nextedge = NOPOLYSEG;
		} else
		{
			/* insert by ascending "fy" */
			if (edgelist->fy > gra_polysegs[i].fy)
			{
				gra_polysegs[i].nextedge = edgelist;
				edgelist = &gra_polysegs[i];
			} else for(a = edgelist; a != NOPOLYSEG; a = a->nextedge)
			{
				if (a->nextedge == NOPOLYSEG || a->nextedge->fy > gra_polysegs[i].fy)
				{
					/* insert after this */
					gra_polysegs[i].nextedge = a->nextedge;
					a->nextedge = &gra_polysegs[i];
					break;
				}
			}
		}
	}

	/* scan polygon and render */
	active = NOPOLYSEG;
	while (active != NOPOLYSEG || edgelist != NOPOLYSEG)
	{
		if (active == NOPOLYSEG)
		{
			active = edgelist;
			active->nextactive = NOPOLYSEG;
			edgelist = edgelist->nextedge;
			ycur = active->fy;
		}

		/* introduce edges from edge list into active list */
		while (edgelist != NOPOLYSEG && edgelist->fy <= ycur)
		{
			/* insert "edgelist" into active list, sorted by "fx" coordinate */
			if (active->fx > edgelist->fx ||
				(active->fx == edgelist->fx && active->increment > edgelist->increment))
			{
				edgelist->nextactive = active;
				active = edgelist;
				edgelist = edgelist->nextedge;
			} else for(a = active; a != NOPOLYSEG; a = a->nextactive)
			{
				if (a->nextactive == NOPOLYSEG || a->nextactive->fx > edgelist->fx ||
					(a->nextactive->fx == edgelist->fx && a->nextactive->increment > edgelist->increment))
				{
					/* insert after this */
					edgelist->nextactive = a->nextactive;
					a->nextactive = edgelist;
					edgelist = edgelist->nextedge;
					break;
				}
			}
		}

		/* generate regions to be filled in on current scan line */
		wrap = 0;
		left = active;
		for(edge = active; edge != NOPOLYSEG; edge = edge->nextactive)
		{
			wrap = wrap + edge->direction;
			if (wrap == 0)
			{
				j = (left->fx + 32768) >> 16;
				k = (edge->fx + 32768) >> 16;
				yrev = wf->revy - ycur;
				row = wf->rowstart[yrev];
				if (style == PATTERNED)
				{
					/* patterned fill */
					pat = desc->raster[yrev&7];
					if (pat != 0)
					{
#ifdef	INVERTHIGHLIGHT
						/* if layer is HIGHLIGHT, invert */
						if (desc->bits == LAYERH)
						{
							for(l=j; l<=k; l++)
							{
								if ((pat & (1 << (15-(l&15)))) != 0) row[l] = ~row[l];
							}
						} else
#endif
						{
							for(l=j; l<=k; l++)
							{
								if ((pat & (1 << (15-(l&15)))) != 0) row[l] = (row[l] & mask) | col;
							}
						}
					}
				} else
				{
					/* solid fill */
#ifdef	INVERTHIGHLIGHT
					/* if layer is HIGHLIGHT, invert */
					if (desc->bits == LAYERH)
					{
						for(l=j; l<=k; l++) row[l] = ~row[l];
					} else
#endif
					{
						for(l=j; l<=k; l++) row[l] = (row[l] & mask) | col;
					}
				}
				left = edge->nextactive;
			}
		}
		ycur++;

		/* update edges in active list */
		lastedge = NOPOLYSEG;
		for(edge = active; edge != NOPOLYSEG; edge = edge->nextactive)
		{
			if (ycur >= edge->ty)
			{
				if (lastedge == NOPOLYSEG) active = edge->nextactive; else
					lastedge->nextactive = edge->nextactive;
			} else
			{
				edge->fx += edge->increment;
				lastedge = edge;
			}
		}
	}
	gra_setrect(wf, lx, hx + 1, wf->revy-hy, wf->revy-ly + 1);
}

/******************** BOX DRAWING ********************/

void screendrawbox(WINDOWPART *win, INTSML lowx, INTSML highx, INTSML lowy, INTSML highy, GRAPHICS *desc)
{
	Rect r;
	INTSML col, mask, style, x, y, pat;
	REGISTER char *thisrow;
	REGISTER WINDOWFRAME *wf;

#ifdef	INVERTHIGHLIGHT
	/* if layer is HIGHLIGHT, invert */
	if (desc->bits == LAYERH)
	{
		screeninvertbox(win, lowx, highx, lowy, highy);
		return;
	}
#endif

	/* make sure the image buffer has not moved */
	wf = win->frame;
	if (GetPixBaseAddr(wf->window->portPixMap) != wf->rowstart[0])
		gra_rebuildrowstart(wf);

	/* get graphics parameters */
	col = desc->col;   mask = ~desc->bits;
	style = desc->colstyle & NATURE;
	r.left = lowx;                     r.right = highx + 1;
	r.bottom = wf->revy-lowy + 1;   r.top = wf->revy-highy;

	/* handle color drawing */
	if (style == PATTERNED)
	{
		/* special case the patterned fill */
		for(y=r.top; y<r.bottom; y++)
		{
			pat = desc->raster[y&7];
			if (pat == 0) continue;
			thisrow = wf->rowstart[y];
			for(x=r.left; x<r.right; x++)
			{
				if ((pat & (1 << (15-(x&15)))) != 0)
					thisrow[x] = (thisrow[x] & mask) | col;
			}
		}
	} else
	{
		for(y=r.top; y<r.bottom; y++)
		{
			thisrow = wf->rowstart[y];
			for(x=r.left; x<r.right; x++)
				thisrow[x] = (thisrow[x] & mask) | col;
		}
	}
	gra_setrect(wf, r.left, r.right, r.top, r.bottom);
}

/*
 * routine to invert the bits in the box from (lowx, lowy) to (highx, highy)
 */
void screeninvertbox(WINDOWPART *win, INTSML lowx, INTSML highx, INTSML lowy, INTSML highy)
{
	Rect r;
	REGISTER WINDOWFRAME *wf;

	wf = win->frame;

	r.left = lowx;                  r.right = highx + 1;
	r.bottom = wf->revy-lowy + 1;   r.top = wf->revy-highy;

	/* prepare for graphics */
	SetGWorld(wf->window, 0L);
	(void)LockPixels(wf->window->portPixMap);

	PenMode(patXor);
	PenPat((ConstPatternParam)&qd.black);
	PaintRect(&r);

	UnlockPixels(wf->window->portPixMap);
	SetGWorld(wf->realwindow, gra_origgdevh);

	gra_setrect(wf, lowx, highx + 1, wf->revy-highy, wf->revy-lowy + 1);
}

/*
 * routine to move bits on the display starting with the area at
 * (sx,sy) and ending at (dx,dy).  The size of the area to be
 * moved is "wid" by "hei".
 */
void screenmovebox(WINDOWPART *win, INTSML sx, INTSML sy, INTSML wid, INTSML hei, INTSML dx, INTSML dy)
{
	Rect from, to;
	INTSML xsize, ysize, x, y, dir, fromstart, frominc, tostart, toinc;
	REGISTER char *frombase, *tobase;
	REGISTER WINDOWFRAME *wf;

	wf = win->frame;

	/* make sure the image buffer has not moved */
	if (GetPixBaseAddr(wf->window->portPixMap) != wf->rowstart[0])
		gra_rebuildrowstart(wf);

	/* setup source rectangle */
	from.left = sx;
	from.right = from.left + wid;
	from.top = wf->revy + 1 - sy - hei;
	from.bottom = from.top + hei;

	/* setup destination rectangle */
	to.left = dx;
	to.right = to.left + wid;
	to.top = wf->revy + 1 - dy - hei;
	to.bottom = to.top + hei;

	/* determine size of bits to move */
	xsize = wid;   ysize = hei;

	/* determine direction of bit copy */
	if (from.left < to.left) dir = 1; else dir = 0;
	if (from.top < to.top)
	{
		fromstart = from.bottom-1;   frominc = -1;
		tostart = to.bottom-1;       toinc = -1;
	} else
	{
		fromstart = from.top;   frominc = 1;
		tostart = to.top;       toinc = 1;
	}

	/* move the bits */
	if (dir == 0)
	{
		/* normal forward copy in X */
		for(y = 0; y < ysize; y++)
		{
			frombase = wf->rowstart[fromstart] + from.left;
			fromstart += frominc;
			tobase = wf->rowstart[tostart] + to.left;
			tostart += toinc;
			for(x = 0; x < xsize; x++) *tobase++ = *frombase++;
		}
	} else
	{
		/* reverse copy in X */
		for(y = 0; y < ysize; y++)
		{
			frombase = wf->rowstart[fromstart] + from.right;
			fromstart += frominc;
			tobase = wf->rowstart[tostart] + to.right;
			tostart += toinc;
			for(x = 0; x < xsize; x++) *tobase-- = *frombase--;
		}
	}
	gra_setrect(wf, to.left, to.right, to.top, to.bottom);
}

/*
 * routine to save the contents of the box from "lx" to "hx" in X and from
 * "ly" to "hy" in Y.  A code is returned that identifies this box for
 * overwriting and restoring.  The routine returns -1 if there is a error.
 */
INTBIG screensavebox(WINDOWPART *win, INTSML lx, INTSML hx, INTSML ly, INTSML hy)
{
	SAVEDBOX *box;
	REGISTER INTBIG toindex, xsize, ysize;
	REGISTER INTSML i, x, y, truelx, truehx;
	REGISTER WINDOWFRAME *wf;

	wf = win->frame;

	/* make sure the image buffer has not moved */
	if (GetPixBaseAddr(wf->window->portPixMap) != wf->rowstart[0])
		gra_rebuildrowstart(wf);

	truelx = lx;   truehx = hx;
	i = ly;   ly = wf->revy-hy;   hy = wf->revy-i;
	xsize = hx-lx+1;
	ysize = hy-ly+1;

	box = (SAVEDBOX *)emalloc((sizeof (SAVEDBOX)), us_aid->cluster);
	if (box == 0) return(-1);
	box->pix = (char *)emalloc(xsize * ysize, us_aid->cluster);
	if (box->pix == 0) return(-1);
	box->win = win;
	box->nextsavedbox = gra_firstsavedbox;
	gra_firstsavedbox = box;
	box->lx = lx;           box->hx = hx;
	box->ly = ly;           box->hy = hy;
	box->truelx = truelx;   box->truehx = truehx;

	/* move the bits */
	toindex = 0;
	for(y = ly; y <= hy; y++)
	{
		for(x = lx; x <= hx; x++)
			box->pix[toindex++] = wf->rowstart[y][x];
	}

	return((INTBIG)box);
}

/*
 * routine to shift the saved box "code" so that it is restored in a different location,
 * offset by (dx,dy)
 */
void screenmovesavedbox(INTBIG code, INTSML dx, INTSML dy)
{
	REGISTER SAVEDBOX *box;
	REGISTER WINDOWFRAME *wf;

	if (code == -1) return;
	box = (SAVEDBOX *)code;
	wf = box->win->frame;
	box->truelx += dx;   box->truehx += dx;
	box->lx += dx;       box->hx += dx;
	box->ly -= dy;       box->hy -= dy;
}

/*
 * routine to restore saved box "code" to the screen.  "destroy" is:
 *  0   restore box, do not free memory
 *  1   restore box, free memory
 * -1   free memory
 * Returns nonzero if there is an error.
 */
INTSML screenrestorebox(INTBIG code, INTSML destroy)
{
	REGISTER SAVEDBOX *box, *lbox, *tbox;
	REGISTER INTBIG fromindex;
	REGISTER INTSML x, y;
	REGISTER WINDOWFRAME *wf;

	/* get the box */
	if (code == -1) return(1);
	box = (SAVEDBOX *)code;
	wf = box->win->frame;

	/* move the bits */
	if (destroy >= 0)
	{
		/* make sure the image buffer has not moved */
		if (GetPixBaseAddr(wf->window->portPixMap) != wf->rowstart[0])
			gra_rebuildrowstart(wf);
		fromindex = 0;
		for(y = box->ly; y <= box->hy; y++)
			for(x = box->lx; x <= box->hx; x++)
				wf->rowstart[y][x] = box->pix[fromindex++];
		gra_setrect(wf, box->truelx, box->truehx+1, box->ly, box->hy+1);
	}

	/* destroy this box's memory if requested */
	if (destroy != 0)
	{
		lbox = NOSAVEDBOX;
		for(tbox = gra_firstsavedbox; tbox != NOSAVEDBOX; tbox = tbox->nextsavedbox)
		{
			if (tbox == box) break;
			lbox = tbox;
		}
		if (lbox == NOSAVEDBOX) gra_firstsavedbox = box->nextsavedbox; else
			lbox->nextsavedbox = box->nextsavedbox;
		efree((char *)box->pix);
		efree((char *)box);
	}
	return(0);
}

/******************** TEXT DRAWING ********************/

void screensettextsize(WINDOWPART *win, INTSML fnt)
{
	/* geneva has no 6, 7, 12 */
	/* helvetica has no 6, 7, 20 */
	static convert[9] = {6, 7, 9, 10, 12, 14, 18, 20, 24};
	REGISTER WINDOWFRAME *wf;

	/* make sure the temp buffer exists */
	if (gra_textbuf == 0) gra_setuptextbuffer(100, 25);

	/* set sizes in this buffer */
	SetGWorld(gra_textbuf, 0L);
	if (fnt == TXTEDITOR)
	{
		TextFont(TFONT);
		gra_curfontsize = 10;
	} else if (fnt == TXTMENU)
	{
		TextFont(MFONT);
		gra_curfontsize = 12;
	} else
	{
		TextFont(EFONT);
		gra_curfontsize = convert[fnt];
	}
	TextSize(gra_curfontsize);
	GetFontInfo(&gra_curfontinfo);

	/* restore the world */
	wf = win->frame;
	SetGWorld(wf->realwindow, gra_origgdevh);
}

void screengettextsize(WINDOWPART *win, char *str, INTSML *x, INTSML *y)
{
	REGISTER INTSML len;
	REGISTER WINDOWFRAME *wf;

	/* make sure the temp buffer exists */
	if (gra_textbuf == 0) gra_setuptextbuffer(100, 25);

	/* determine text size in this buffer */
	SetGWorld(gra_textbuf, 0L);
	len = strlen(str);
	*x = TextWidth(str, 0, len);
	*y = gra_curfontinfo.ascent + gra_curfontinfo.descent;

	/* restore the world */
	wf = win->frame;
	SetGWorld(wf->realwindow, gra_origgdevh);
}

void screendrawtext(WINDOWPART *win, INTSML atx, INTSML aty, char *s, GRAPHICS *desc)
{
	REGISTER INTSML col, bits, mask, xp, yp;
	INTSML width, height;
	Rect dr, sr;
	REGISTER WINDOWFRAME *wf;
	REGISTER char *sourcerow, *destrow;
	char **rowstart;

	wf = win->frame;
	gettextbits(win, s, &width, &height, &rowstart);
	col = desc->col;   bits = desc->bits;
	sr.left = 0;   sr.right = width;
	sr.top = 0;    sr.bottom = height;

	/* set destination area */
	dr.left = atx;
	dr.right = dr.left + width;
	dr.bottom = wf->revy - aty;
	dr.top = dr.bottom - gra_curfontinfo.ascent - gra_curfontinfo.descent;

	/* clip to the window */
	if (dr.left < win->uselx)
	{
		sr.left += win->uselx - dr.left;
		dr.left = win->uselx;
	}
	if (dr.right > win->usehx)
	{
		sr.right -= dr.right - win->usehx;
		dr.right = win->usehx;
	}
	if (dr.top < wf->revy - win->usehy)
	{
		sr.top += (wf->revy-win->usehy) - dr.top;
		dr.top = wf->revy-win->usehy;
	}
	if (dr.bottom > wf->revy - win->usely)
	{
		sr.bottom -= dr.bottom - (wf->revy-win->usely);
		dr.bottom = wf->revy - win->usely;
	}

	/* make sure the image buffer has not moved */
	if (GetPixBaseAddr(wf->window->portPixMap) != wf->rowstart[0])
		gra_rebuildrowstart(wf);

	/* handle offscreen display */
	mask = ~bits;
	for(yp=sr.top; yp<sr.bottom; yp++)
	{
		sourcerow = &gra_textbufrowstart[yp][sr.left];
		destrow = &wf->rowstart[dr.top+yp][dr.left];
#ifdef	INVERTHIGHLIGHT
		/* if layer is HIGHLIGHT, invert */
		if (bits == LAYERH)
		{
			for(xp=sr.left; xp<sr.right; xp++)
			{
				if (*sourcerow++ != 0) *destrow = ~(*destrow);
				destrow++;
			}
		} else
#endif
		{
			for(xp=sr.left; xp<sr.right; xp++)
			{
				if (*sourcerow++ != 0) *destrow = (*destrow & mask) | col;
				destrow++;
			}
		}
	}
	gra_setrect(wf, dr.left, dr.right, dr.top, dr.bottom);
}

INTSML gettextbits(WINDOWPART *win, char *msg, INTSML *wid, INTSML *hei, char ***rowstart)
{
	REGISTER INTSML i, len;
	Rect sr;
	REGISTER WINDOWFRAME *wf;

	wf = win->frame;

	/* copy the string and correct it */
	len = strlen(msg);
	for(i=0; i<len; i++)
	{
		gra_localstring[i] = msg[i] & 0177;
		if (gra_localstring[i] < ' ') gra_localstring[i] = ' ';
	}

	/* make sure the temp buffer exists */
	if (gra_textbuf == 0) gra_setuptextbuffer(100, 25);

	/* determine string size */
	SetGWorld(gra_textbuf, 0L);
	*wid = TextWidth(gra_localstring, 0, len);
	*hei = gra_curfontinfo.ascent + gra_curfontinfo.descent;

	/* make sure the temp buffer is big enough */
	gra_setuptextbuffer(*wid, *hei);

	/* write to the temp buffer */
	sr.left = 0;   sr.right = *wid;
	sr.top = 0;    sr.bottom = *hei;
	EraseRect(&sr);
	TextMode(srcCopy);
	MoveTo(0, gra_curfontinfo.ascent);
	DrawText(gra_localstring, 0, len);
	SetGWorld(wf->realwindow, gra_origgdevh);
	*rowstart = gra_textbufrowstart;
	SetFontLock(1);
	return(0);
}

void gra_setuptextbuffer(INTSML width, INTSML height)
{
	Rect r;
	REGISTER INTSML i, bytes;
	REGISTER char *addr;

	if (width <= gra_textbufwid && height <= gra_textbufhei) return;

	r.left = 0;
	r.right = width;
	r.top = 0;
	r.bottom = height;
	if (gra_textbuf == 0)
	{
		NewGWorld(&gra_textbuf, 8, &r, 0L, 0L, 0);
		(void)LockPixels(gra_textbuf->portPixMap);
	} else
	{
		UpdateGWorld(&gra_textbuf, 8, &r, 0L, 0L, 0);
		(void)LockPixels(gra_textbuf->portPixMap);
		DisposePtr((Ptr)gra_textbufrowstart);
	}
	gra_textbufrowstart = (char **)NewPtr(height * (sizeof (char *)));
	bytes = (*gra_textbuf->portPixMap)->rowBytes & 0x7FFF;
	addr = GetPixBaseAddr(gra_textbuf->portPixMap);
	for(i=0; i<height; i++)
	{
		gra_textbufrowstart[i] = addr;
		addr += bytes;
	}
	gra_textbufwid = width;   gra_textbufhei = height;
}

/******************** CIRCLE DRAWING ********************/

void screendrawcircle(WINDOWPART *win, INTBIG atx, INTBIG aty, INTBIG radius, GRAPHICS *desc)
{
	Rect r;
	REGISTER INTSML col, mask;
	REGISTER INTBIG x, y, d, maxx, maxy, thisx, thisy;
	REGISTER WINDOWFRAME *wf;
	REGISTER char *thisrow;

#ifdef	INVERTHIGHLIGHT
	/* if layer is HIGHLIGHT, invert */
	if (desc->bits == LAYERH)
	{
		gra_invertcircle(win, atx, aty, radius);
		return;
	}
#endif

	/* make sure the image buffer has not moved */
	wf = win->frame;
	if (GetPixBaseAddr(wf->window->portPixMap) != wf->rowstart[0])
		gra_rebuildrowstart(wf);

	/* get parameters */
	col = desc->col;   mask = ~desc->bits;
	aty = wf->revy - aty;

	/* set redraw area */
	r.left = atx - radius;
	r.right = atx + radius + 1;
	r.top = aty - radius;
	r.bottom = aty + radius + 1;
	gra_setrect(wf, r.left, r.right, r.top, r.bottom);

	maxx = (*wf->window->portPixMap)->bounds.right;
	maxy = (*wf->window->portPixMap)->bounds.bottom;
	x = 0;   y = radius;
	d = 3 - 2 * radius;
	if (r.left >= 0 && r.right < maxx && r.top >= 0 && r.bottom < maxy)
	{
		/* no clip version is faster */
		while (x <= y)
		{
			thisrow = wf->rowstart[aty + y];
			thisrow[atx + x] = (thisrow[atx + x] & mask) | col;
			thisrow[atx - x] = (thisrow[atx - x] & mask) | col;

			thisrow = wf->rowstart[aty - y];
			thisrow[atx + x] = (thisrow[atx + x] & mask) | col;
			thisrow[atx - x] = (thisrow[atx - x] & mask) | col;

			thisrow = wf->rowstart[aty + x];
			thisrow[atx + y] = (thisrow[atx + y] & mask) | col;
			thisrow[atx - y] = (thisrow[atx - y] & mask) | col;

			thisrow = wf->rowstart[aty - x];
			thisrow[atx + y] = (thisrow[atx + y] & mask) | col;
			thisrow[atx - y] = (thisrow[atx - y] & mask) | col;

			if (d < 0) d += 4*x + 6; else
			{
				d += 4 * (x-y) + 10;
				y--;
			}
			x++;
		}
	} else
	{
		/* clip version */
		while (x <= y)
		{
			thisy = aty + y;
			if (thisy >= 0 && thisy < maxy)
			{
				thisrow = wf->rowstart[thisy];
				thisx = atx + x;
				if (thisx >= 0 && thisx < maxx) thisrow[thisx] = (thisrow[thisx] & mask) | col;
				thisx = atx - x;
				if (thisx >= 0 && thisx < maxx) thisrow[thisx] = (thisrow[thisx] & mask) | col;
			}

			thisy = aty - y;
			if (thisy >= 0 && thisy < maxy)
			{
				thisrow = wf->rowstart[thisy];
				thisx = atx + x;
				if (thisx >= 0 && thisx < maxx) thisrow[thisx] = (thisrow[thisx] & mask) | col;
				thisx = atx - x;
				if (thisx >= 0 && thisx < maxx) thisrow[thisx] = (thisrow[thisx] & mask) | col;
			}

			thisy = aty + x;
			if (thisy >= 0 && thisy < maxy)
			{
				thisrow = wf->rowstart[thisy];
				thisx = atx + y;
				if (thisx >= 0 && thisx < maxx) thisrow[thisx] = (thisrow[thisx] & mask) | col;
				thisx = atx - y;
				if (thisx >= 0 && thisx < maxx) thisrow[thisx] = (thisrow[thisx] & mask) | col;
			}

			thisy = aty - x;
			if (thisy >= 0 && thisy < maxy)
			{
				thisrow = wf->rowstart[thisy];
				thisx = atx + y;
				if (thisx >= 0 && thisx < maxx) thisrow[thisx] = (thisrow[thisx] & mask) | col;
				thisx = atx - y;
				if (thisx >= 0 && thisx < maxx) thisrow[thisx] = (thisrow[thisx] & mask) | col;
			}

			if (d < 0) d += 4*x + 6; else
			{
				d += 4 * (x-y) + 10;
				y--;
			}
			x++;
		}
	}
}

#ifdef	INVERTHIGHLIGHT
void gra_invertcircle(WINDOWPART *win, INTBIG atx, INTBIG aty, INTBIG radius)
{
	Rect r;
	REGISTER INTBIG x, y, d, maxx, maxy, thisx, thisy;
	REGISTER WINDOWFRAME *wf;
	REGISTER char *thisrow;

	/* make sure the image buffer has not moved */
	wf = win->frame;
	if (GetPixBaseAddr(wf->window->portPixMap) != wf->rowstart[0])
		gra_rebuildrowstart(wf);

	/* get parameters */
	aty = wf->revy - aty;

	maxx = (*wf->window->portPixMap)->bounds.right;
	maxy = (*wf->window->portPixMap)->bounds.bottom;
	x = 0;   y = radius;
	d = 3 - 2 * radius;
	if (r.left >= 0 && r.right < maxx && r.top >= 0 && r.bottom < maxy)
	{
		/* no clip version is faster */
		while (x <= y)
		{
			thisrow = wf->rowstart[aty + y];
			thisrow[atx + x] = ~thisrow[atx + x];
			thisrow[atx - x] = ~thisrow[atx - x];

			thisrow = wf->rowstart[aty - y];
			thisrow[atx + x] = ~thisrow[atx + x];
			thisrow[atx - x] = ~thisrow[atx - x];

			thisrow = wf->rowstart[aty + x];
			thisrow[atx + y] = ~thisrow[atx + y];
			thisrow[atx - y] = ~thisrow[atx - y];

			thisrow = wf->rowstart[aty - x];
			thisrow[atx + y] = ~thisrow[atx + y];
			thisrow[atx - y] = ~thisrow[atx - y];

			if (d < 0) d += 4*x + 6; else
			{
				d += 4 * (x-y) + 10;
				y--;
			}
			x++;
		}
	} else
	{
		/* clip version */
		while (x <= y)
		{
			thisy = aty + y;
			if (thisy >= 0 && thisy < maxy)
			{
				thisrow = wf->rowstart[thisy];
				thisx = atx + x;
				if (thisx >= 0 && thisx < maxx) thisrow[thisx] = ~thisrow[thisx];
				thisx = atx - x;
				if (thisx >= 0 && thisx < maxx) thisrow[thisx] = ~thisrow[thisx];
			}

			thisy = aty - y;
			if (thisy >= 0 && thisy < maxy)
			{
				thisrow = wf->rowstart[thisy];
				thisx = atx + x;
				if (thisx >= 0 && thisx < maxx) thisrow[thisx] = ~thisrow[thisx];
				thisx = atx - x;
				if (thisx >= 0 && thisx < maxx) thisrow[thisx] = ~thisrow[thisx];
			}

			thisy = aty + x;
			if (thisy >= 0 && thisy < maxy)
			{
				thisrow = wf->rowstart[thisy];
				thisx = atx + y;
				if (thisx >= 0 && thisx < maxx) thisrow[thisx] = ~thisrow[thisx];
				thisx = atx - y;
				if (thisx >= 0 && thisx < maxx) thisrow[thisx] = ~thisrow[thisx];
			}

			thisy = aty - x;
			if (thisy >= 0 && thisy < maxy)
			{
				thisrow = wf->rowstart[thisy];
				thisx = atx + y;
				if (thisx >= 0 && thisx < maxx) thisrow[thisx] = ~thisrow[thisx];
				thisx = atx - y;
				if (thisx >= 0 && thisx < maxx) thisrow[thisx] = ~thisrow[thisx];
			}

			if (d < 0) d += 4*x + 6; else
			{
				d += 4 * (x-y) + 10;
				y--;
			}
			x++;
		}
	}

	/* set redraw area */
	r.left = atx - radius;
	r.right = atx + radius + 1;
	r.top = aty - radius;
	r.bottom = aty + radius + 1;
	gra_setrect(wf, r.left, r.right, r.top, r.bottom);
}
#endif

/******************** DISC DRAWING ********************/

/*
 * routine to draw row "thisy" of a disc from "startx" <= X <= "endx"
 */
void gra_drawdiscrow(WINDOWFRAME *wf, INTBIG thisy, INTBIG startx, INTBIG endx, GRAPHICS *desc)
{
	REGISTER char *thisrow;
	REGISTER INTBIG x;
	REGISTER INTSML pat;

	if (thisy < 0 || thisy >= gra_curvemaxy) return;
	thisrow = wf->rowstart[thisy];
	if (startx < 0) startx = 0;
	if (endx >= gra_curvemaxx) endx = gra_curvemaxy - 1;
	if (gra_curvestyle == PATTERNED)
	{
		pat = desc->raster[thisy&7];
		if (pat != 0)
		{
			for(x=startx; x<=endx; x++)
				if ((pat & (1 << (15-(x&15)))) != 0)
					thisrow[x] = (thisrow[x] & gra_curvemask) | gra_curvecol;
		}
	} else
	{
		for(x=startx; x<=endx; x++)
			thisrow[x] = (thisrow[x] & gra_curvemask) | gra_curvecol;
	}
}

#ifdef	INVERTHIGHLIGHT
/*
 * routine to invert row "thisy" of a disc from "startx" <= X <= "endx"
 */
void gra_invdiscrow(WINDOWFRAME *wf, INTBIG thisy, INTBIG startx, INTBIG endx, GRAPHICS *desc)
{
	REGISTER char *thisrow;
	REGISTER INTBIG x;
	REGISTER INTSML pat;

	if (thisy < 0 || thisy >= gra_curvemaxy) return;
	thisrow = wf->rowstart[thisy];
	if (startx < 0) startx = 0;
	if (endx >= gra_curvemaxx) endx = gra_curvemaxy - 1;
	if (gra_curvestyle == PATTERNED)
	{
		pat = desc->raster[thisy&7];
		if (pat != 0)
		{
			for(x=startx; x<=endx; x++)
				if ((pat & (1 << (15-(x&15)))) != 0) thisrow[x] = ~thisrow[x];
		}
	} else
	{
		for(x=startx; x<=endx; x++) thisrow[x] = ~thisrow[x];
	}
}
#endif

/*
 * routine to draw a filled-in circle at (atx,aty) with radius "radius"
 */
void screendrawdisc(WINDOWPART *win, INTBIG atx, INTBIG aty, INTBIG radius, GRAPHICS *desc)
{
	Rect r;
	REGISTER WINDOWFRAME *wf;
	REGISTER INTBIG x, y, d;

	/* make sure the image buffer has not moved */
	wf = win->frame;
	if (GetPixBaseAddr(wf->window->portPixMap) != wf->rowstart[0])
		gra_rebuildrowstart(wf);

	/* get parameters */
	gra_curvestyle = desc->colstyle & NATURE;
	gra_curvecol = desc->col;   gra_curvemask = ~desc->bits;
	aty = wf->revy - aty;

#ifdef	INVERTHIGHLIGHT
	if (desc->bits == LAYERH) gra_discrowdraw = gra_invdiscrow; else
#endif
		gra_discrowdraw = gra_drawdiscrow;

	gra_curvemaxx = (*wf->window->portPixMap)->bounds.right;
	gra_curvemaxy = (*wf->window->portPixMap)->bounds.bottom;
	x = 0;   y = radius;
	d = 3 - 2 * radius;
	while (x <= y)
	{
		(*gra_discrowdraw)(wf, aty+y, atx-x, atx+x, desc);
		(*gra_discrowdraw)(wf, aty-y, atx-x, atx+x, desc);
		(*gra_discrowdraw)(wf, aty+x, atx-y, atx+y, desc);
		(*gra_discrowdraw)(wf, aty-x, atx-y, atx+y, desc);

		if (d < 0) d += 4*x + 6; else
		{
			d += 4 * (x-y) + 10;
			y--;
		}
		x++;
	}

	/* set redraw area */
	r.left = atx - radius;
	r.right = atx + radius + 1;
	r.top = aty - radius;
	r.bottom = aty + radius + 1;
	gra_setrect(wf, r.left, r.right, r.top, r.bottom);
}

/******************** ARC DRAWING ********************/

INTSML gra_arcfindoctant(INTBIG x, INTBIG y)
{
	if (x > 0)
		if (y >= 0)
			if (y >= x)	 return 7;
			else         return 8;
		else
			if (x >= -y) return 1;
			else         return 2;
	else
		if (y > 0)
			if (y > -x)  return 6;
			else         return 5;
		else
			if (y > x)   return 4;
			else         return 3;
}

void gra_arcxformoctant(INTBIG x, INTBIG y, INTBIG oct, INTBIG *ox, INTBIG *oy)
{
	switch (oct)
	{
		case 1 : *ox = -y;   *oy = x;   break;
		case 2 : *ox = x;    *oy = -y;  break;
		case 3 : *ox = -x;   *oy = -y;  break;
		case 4 : *ox = -y;   *oy = -x;  break;
		case 5 : *ox = y;    *oy = -x;  break;
		case 6 : *ox = -x;   *oy = y;   break;
		case 7 : *ox = x;    *oy = y;   break;
		case 8 : *ox = y;    *oy = x;   break;
	}
}

#ifdef	INVERTHIGHLIGHT
void gra_arcinvpixel(INTBIG x, INTBIG y)
{
	if (x < 0 || x >= gra_curvemaxx || y < 0 || y >= gra_curvemaxy) return;
	if (gra_arcfirst != 0)
	{
		gra_arcfirst = 0;
		gra_arclx = gra_archx = x;
		gra_arcly = gra_archy = y;
	} else
	{
		if (x < gra_arclx) gra_arclx = x;
		if (x > gra_archx) gra_archx = x;
		if (y < gra_arcly) gra_arcly = y;
		if (y > gra_archy) gra_archy = y;
	}
	gra_arcframe->rowstart[y][x] = ~gra_arcframe->rowstart[y][x];
}
#endif

void gra_arcdopixel(INTBIG x, INTBIG y)
{
	if (x < 0 || x >= gra_curvemaxx || y < 0 || y >= gra_curvemaxy) return;
	if (gra_arcfirst != 0)
	{
		gra_arcfirst = 0;
		gra_arclx = gra_archx = x;
		gra_arcly = gra_archy = y;
	} else
	{
		if (x < gra_arclx) gra_arclx = x;
		if (x > gra_archx) gra_archx = x;
		if (y < gra_arcly) gra_arcly = y;
		if (y > gra_archy) gra_archy = y;
	}
	gra_arcframe->rowstart[y][x] = (gra_arcframe->rowstart[y][x] & gra_curvemask) | gra_curvecol;
}

void gra_arcoutxform(INTBIG x, INTBIG y)
{
	if (gra_arcocttable[1]) (*gra_arcpixeldraw)( y + gra_arccenterx, -x + gra_arccentery);
	if (gra_arcocttable[2]) (*gra_arcpixeldraw)( x + gra_arccenterx, -y + gra_arccentery);
	if (gra_arcocttable[3]) (*gra_arcpixeldraw)(-x + gra_arccenterx, -y + gra_arccentery);
	if (gra_arcocttable[4]) (*gra_arcpixeldraw)(-y + gra_arccenterx, -x + gra_arccentery);
	if (gra_arcocttable[5]) (*gra_arcpixeldraw)(-y + gra_arccenterx,  x + gra_arccentery);
	if (gra_arcocttable[6]) (*gra_arcpixeldraw)(-x + gra_arccenterx,  y + gra_arccentery);
	if (gra_arcocttable[7]) (*gra_arcpixeldraw)( x + gra_arccenterx,  y + gra_arccentery);
	if (gra_arcocttable[8]) (*gra_arcpixeldraw)( y + gra_arccenterx,  x + gra_arccentery);
}

void gra_arcbrescw(INTBIG x, INTBIG y, INTBIG x1, INTBIG y1)
{
	REGISTER INTBIG d;

	d = 3 - 2 * y + 4 * x;
	while (x < x1 && y > y1)
	{
		gra_arcoutxform(x, y);
		if (d < 0) d += 4*x+6; else
		{
			d += 4*(x-y)+10;
			y--;
		}
		x++;
	}

	/* get to the end */
	for ( ; x < x1; x++) gra_arcoutxform(x, y);
	for ( ; y > y1; y--) gra_arcoutxform(x, y);
   gra_arcoutxform(x1, y1);
}

void gra_arcbresmidcw(INTBIG x, INTBIG y)
{
	REGISTER INTBIG d;

	d = 3 - 2 * y + 4 * x;
	while (x < y)
	{
		gra_arcoutxform(x, y);
		if (d < 0) d += 4*x+6; else
		{
			d += 4*(x-y)+10;
			y--;
		}
		x++;
   }
   if (x == y) gra_arcoutxform(x, y);
}

void gra_arcbresmidccw(INTBIG x, INTBIG y)
{
	REGISTER INTBIG d;

	d = 3 + 2 * y - 4 * x;
	while (x > 0)
	{
		gra_arcoutxform(x, y);
		if (d > 0) d += 6-4*x; else
		{
			d += 4*(y-x)+10;
			y++;
		}
		x--;
   }
   gra_arcoutxform(0, gra_arcradius);
}

void gra_arcbresccw(INTBIG x, INTBIG y, INTBIG x1, INTBIG y1)
{
	REGISTER INTBIG d;

	d = 3 + 2 * y + 4 * x;
	while(x > x1 && y < y1)
	{
		/* not always correct */
		gra_arcoutxform(x, y);
		if (d > 0) d += 6 - 4*x; else
		{
			d += 4*(y-x)+10;
			y++;
		}
		x--;
	}

	/* get to the end */
	for ( ; x > x1; x--) gra_arcoutxform(x, y);
	for ( ; y < y1; y++) gra_arcoutxform(x, y);
	gra_arcoutxform(x1, y1);
}

/*
 * draws an arc centered at (centerx, centery), clockwise,
 * passing by (x1,y1) and (x2,y2)
 */
void screendrawcirclearc(WINDOWPART *win, INTBIG centerx, INTBIG centery, INTBIG p1_x, INTBIG p1_y,
	INTBIG p2_x, INTBIG p2_y, GRAPHICS *desc)
{
	REGISTER INTBIG alternate, pa_x, pa_y, pb_x, pb_y, i, diff;
	INTBIG x, y;
	REGISTER INTSML start_oct, end_oct;

	/* ignore tiny arcs */
	if (p1_x == p2_x && p1_y == p2_y) return;

	/* make sure the image buffer has not moved */
	gra_arcframe = win->frame;
	if (GetPixBaseAddr(gra_arcframe->window->portPixMap) != gra_arcframe->rowstart[0])
		gra_rebuildrowstart(gra_arcframe);

	/* get parameters */
	gra_curvecol = desc->col;   gra_curvemask = ~desc->bits;
	gra_curvemaxx = (*gra_arcframe->window->portPixMap)->bounds.right;
	gra_curvemaxy = (*gra_arcframe->window->portPixMap)->bounds.bottom;
	p1_y = gra_arcframe->revy - p1_y;
	p2_y = gra_arcframe->revy - p2_y;
	gra_arccentery = gra_arcframe->revy - centery;
	gra_arccenterx = centerx;
	pa_x = p2_x - gra_arccenterx;
	pa_y = p2_y - gra_arccentery;
	pb_x = p1_x - gra_arccenterx;
	pb_y = p1_y - gra_arccentery;
	gra_arcradius = computedistance(gra_arccenterx, gra_arccentery, p2_x, p2_y);
	alternate = computedistance(gra_arccenterx, gra_arccentery, p1_x, p1_y);
	start_oct = gra_arcfindoctant(pa_x, pa_y);
	end_oct   = gra_arcfindoctant(pb_x, pb_y);
	gra_arcfirst = 1;
#ifdef	INVERTHIGHLIGHT
	if (desc->bits == LAYERH) gra_arcpixeldraw = gra_arcinvpixel; else
#endif
		gra_arcpixeldraw = gra_arcdopixel;

	/* move the point */
	if (gra_arcradius != alternate)
	{
		diff = gra_arcradius-alternate;
		switch (end_oct)
		{
			case 6:
			case 7: /*  y >  x */ pb_y += diff;  break;
			case 8: /*  x >  y */
			case 1: /*  x > -y */ pb_x += diff;  break;
			case 2: /* -y >  x */
			case 3: /* -y > -x */ pb_y -= diff;  break;
			case 4: /* -y < -x */
			case 5: /*  y < -x */ pb_x -= diff;  break;
		}
	}

	for(i=1; i<9; i++) gra_arcocttable[i] = 0;

	if (start_oct == end_oct)
	{
		INTBIG x1, y1, x2, y2;

		gra_arcocttable[start_oct] = 1;
		gra_arcxformoctant(pa_x, pa_y, start_oct, &x1, &y1);
		gra_arcxformoctant(pb_x, pb_y, start_oct, &x2 ,&y2);

		if (ODD(start_oct)) gra_arcbrescw(x1, y1, x2, y2);
		else				gra_arcbresccw(x1 ,y1, x2, y2);
		gra_arcocttable[start_oct] = 0;
	} else
	{
		gra_arcocttable[start_oct] = 1;
		gra_arcxformoctant(pa_x, pa_y, start_oct, &x, &y);
		if (ODD(start_oct)) gra_arcbresmidcw(x, y);
		else				gra_arcbresmidccw(x, y);
		gra_arcocttable[start_oct] = 0;

		gra_arcocttable[end_oct] = 1;
		gra_arcxformoctant(pb_x, pb_y, end_oct, &x, &y);
		if (ODD(end_oct)) gra_arcbresmidccw(x, y);
		else			  gra_arcbresmidcw(x, y);
		gra_arcocttable[end_oct] = 0;

		if (MODP(start_oct+1) != end_oct)
		{
			if (MODP(start_oct+1) == MODM(end_oct-1)) gra_arcocttable[MODP(start_oct+1)] = 1; else
				for(i = MODP(start_oct+1); i != end_oct; i = MODP(i+1)) gra_arcocttable[i] = 1;
			gra_arcbresmidcw(0, gra_arcradius);
		}
	}

	/* set redraw area */
	if (gra_arcfirst == 0)
		gra_setrect(gra_arcframe, gra_arclx, gra_archx+1, gra_arcly, gra_archy+1);
}

/******************** GRID DRAWING ********************/

/*
 * grid drawing routine
 */
void screendrawgrid(WINDOWPART *win, POLYGON *obj)
{
	REGISTER INTBIG i, j, xnum, xden, ynum, yden, x0,y0, x1,y1, x2,y2, x3,y3,
		x4,y4, x5,y5, x10, y10, y10mod, xspacing, yspacing, y1base, x1base;
	REGISTER INTSML x, y, fatdots;
	REGISTER WINDOWFRAME *wf;
	REGISTER VARIABLE *var;

	wf = win->frame;

	/* make sure the image buffer has not moved */
	if (GetPixBaseAddr(wf->window->portPixMap) != wf->rowstart[0])
		gra_rebuildrowstart(wf);

	x0 = obj->xv[0];   y0 = obj->yv[0];		/* screen space grid spacing */
	x1 = obj->xv[1];   y1 = obj->yv[1];		/* screen space grid start */
	x2 = obj->xv[2];   y2 = obj->yv[2];		/* display space low */
	x3 = obj->xv[3];   y3 = obj->yv[3];		/* display space high */
	x4 = obj->xv[4];   y4 = obj->yv[4];		/* screen space low */
	x5 = obj->xv[5];   y5 = obj->yv[5];		/* screen space high */

	var = getvalkey((INTBIG)us_aid, VAID, -1, us_gridboldspacing);
	if (var == NOVARIABLE) xspacing = yspacing = 10; else
	{
		if ((var->type&VISARRAY) == 0)
			xspacing = yspacing = var->addr; else
		{
			xspacing = ((INTBIG *)var->addr)[0];
			yspacing = ((INTBIG *)var->addr)[1];
		}
	}

	xnum = x3 - x2;
	xden = x5 - x4;
	ynum = y3 - y2;
	yden = y5 - y4;
	x10 = x0*xspacing;       y10 = y0*yspacing;
	y1base = y1 - (y1 / y0 * y0);
	x1base = x1 - (x1 / x0 * x0);

	/* adjust grid placement according to scale */
	fatdots = 0;
	if (muldiv(x0, xnum, xden) < 5 || muldiv(y0, ynum, yden) < 5)
	{
		x1 = x1base - (x1base - x1) / x10 * x10;   x0 = x10;
		y1 = y1base - (y1base - y1) / y10 * y10;   y0 = y10;
	} else if (muldiv(x0, xnum, xden) > 75 && muldiv(y0, ynum, yden) > 75)
	{
		fatdots = 1;
	}

	/* draw the grid to the offscreen PixMap */
	for(i = y1; i < y5; i += y0)
	{
		y = muldiv(i-y4, ynum, yden) + y2;
		if (y < y2 || y > y3) continue;
		y = wf->revy - y;
		y10mod = (i-y1base) % y10;

		for(j = x1; j < x5; j += x0)
		{
			x = muldiv(j-x4, xnum, xden) + x2;
			if (x >= x2 && x <= x3) wf->rowstart[y][x] |= GRID;

			/* special case every 10 grid points in each direction */
			if (fatdots != 0 || ((j-x1base)%x10) == 0 && y10mod == 0)
			{
				if (x > x2) wf->rowstart[y][x-1] |= GRID;
				if (x < x3) wf->rowstart[y][x+1] |= GRID;
				if (y > y2) wf->rowstart[y-1][x] |= GRID;
				if (y < y3) wf->rowstart[y+1][x] |= GRID;
				if (fatdots != 0 && ((j-x1base)%x10) == 0 && y10mod == 0)
				{
					if (x-1 > x2) wf->rowstart[y][x-2] |= GRID;
					if (x+1 < x3) wf->rowstart[y][x+2] |= GRID;
					if (y-1 > y2) wf->rowstart[y-2][x] |= GRID;
					if (y+1 < y3) wf->rowstart[y+2][x] |= GRID;
					if (x > x2 && y > y2) wf->rowstart[y-1][x-1] |= GRID;
					if (x > x2 && y < y3) wf->rowstart[y+1][x-1] |= GRID;
					if (x < x3 && y > y2) wf->rowstart[y-1][x+1] |= GRID;
					if (x < x3 && y < y3) wf->rowstart[y+1][x+1] |= GRID;
				}
			}
		}
	}

	/* copy it back to the screen */
	gra_setrect(wf, x2, x3, wf->revy-y3, wf->revy-y2);
}

/******************** MOUSE CONTROL ********************/

/*
 * routine to return the number of buttons on the mouse
 */
INTSML buttoncount(void)
{
	return(mini(BUTTONS, NUMBUTS));
}

/*
 * routine to tell whether button "but" is a double-click
 */
INTSML doublebutton(INTSML b)
{
	if (b == 16) return(1);
	return(0);
}

/*
 * routine to tell whether button "but" has the "shift" key held
 */
INTSML shiftbutton(INTSML b)
{
	if ((b%2) == 1) return(1);
	return(0);
}

/*
 * routine to return the name of button "b" (from 0 to "buttoncount()").
 * The number of letters unique to the button is placed in "important".
 */
char *buttonname(INTSML b, INTSML *important)
{
	*important = gra_buttonname[b].unique;
	return(gra_buttonname[b].name);
}

/*
 * routine to convert from "gra_inputstate" (the typical input parameter)
 * to button numbers (the table "gra_buttonname")
 */
INTSML gra_makebutton(INTBIG state)
{
	INTSML base;

	if ((state&DOUBLECLICK) != 0) return(16);
	base = 0;
	if ((state&CONTROLDOWN) != 0) base += 8;
	if ((state&OPTIONDOWN) != 0) base += 4;
	if ((state&COMMANDDOWN) != 0) base += 2;
	if ((state&SHIFTDOWN) != 0) base++;
	return(base);
}

/*
 * routine to wait for a button push and return its index (0 based) in "*but".
 * The coordinates of the cursor are placed in "*x" and "*y".  If there is no
 * button push, the value of "*but" is negative.
 */
void waitforbutton(INTSML *x, INTSML *y, INTSML *but)
{
	EventRecord theEvent;

	if (gra_inputstate != NOEVENT && (gra_inputstate&(ISBUTTON|BUTTONUP)) == ISBUTTON)
	{
		*but = gra_makebutton(gra_inputstate);
		*x = gra_cursorx;
		*y = gra_cursory;
		gra_inputstate = NOEVENT;
		if (us_cursorstate != IBEAMCURSOR) setdefaultcursortype(NULLCURSOR);
		return;
	}

	gra_waitforaction(1, &theEvent);

	if (gra_inputstate != NOEVENT && (gra_inputstate&(ISBUTTON|BUTTONUP)) == ISBUTTON)
	{
		*but = gra_makebutton(gra_inputstate);
		*x = gra_cursorx;
		*y = gra_cursory;
		gra_inputstate = NOEVENT;
		if (us_cursorstate != IBEAMCURSOR) setdefaultcursortype(NULLCURSOR);
		return;
	}
	*but = -1;
}

/*
 * routine to track the cursor until a button is released, calling "whileup" for
 * each co-ordinate when the mouse moves before the first button push, calling
 * "whendown" once when the button goes down, calling "eachdown" for each
 * co-ordinate when the mouse moves after the button is pushed, calling
 * "eachchar" for each key that is typed at any time, and calling "done" once
 * when done.  The "whendown" and "done" routines are called with no parameters;
 * "whileup" and "eachdown" are called with the X and Y coordinates of the
 * cursor; and "eachchar" is called with the X, Y, and character value that was
 * typed.  The "whileup", "eachdown", and "eachchar" routines return nonzero to
 * abort tracking.
 * If "waitforpush" is nonzero then the routine will wait for a button to
 * actually be pushed before tracking (otherwise it will begin tracking
 * immediately).  The value of "purpose" determines what the cursor will look
 * like during dragging: 0 for normal (the standard cursor), 1 for drawing (a pen),
 * 2 for dragging (a hand), 3 for popup menu selection (a horizontal arrow), 4 for
 * hierarchical popup menu selection (arrow, stays at end).
 */
void trackcursor(INTSML waitforpush, INTSML (*whileup)(INTBIG, INTBIG), void (*whendown)(void),
	INTSML (*eachdown)(INTBIG, INTBIG), INTSML (*eachchar)(INTBIG, INTBIG, INTSML), void (*done)(void),
		INTSML purpose)
{
	REGISTER INTSML keepon;
	EventRecord theEvent;

	/* change the cursor to an appropriate icon */
	switch (purpose)
	{
		case TRACKDRAWING:    setdefaultcursortype(PENCURSOR);    break;
		case TRACKDRAGGING:   setdefaultcursortype(HANDCURSOR);   break;
		case TRACKSELECTING:
		case TRACKHSELECTING: setdefaultcursortype(MENUCURSOR);   break;
	}

	/* now wait for a button to go down, if requested */
	keepon = 0;
	if (waitforpush != 0)
	{
		while (keepon == 0)
		{
			gra_waitforaction(2, &theEvent);
			if (gra_inputstate == NOEVENT) continue;

			/* if button just went down, stop this loop */
			if ((gra_inputstate&ISBUTTON) != 0 && (gra_inputstate&BUTTONUP) == 0) break;

			if ((gra_inputstate&MOTION) != 0)
			{
				keepon = (*whileup)(gra_cursorx, gra_cursory);
			} else if ((gra_inputstate&CHARREAD) != 0)
			{
				keepon = (*eachchar)(gra_cursorx, gra_cursory, gra_inputstate&CHARREAD);
			}
			if (el_pleasestop != 0) keepon = 1;
		}
	}

	/* button is now down, real tracking begins */
	if (keepon == 0)
	{
		(*whendown)();
		keepon = (*eachdown)(gra_cursorx, gra_cursory);
	}

	/* now track while the button is down */
	while (keepon == 0)
	{
		gra_waitforaction(2, &theEvent);

		/* for each motion, report the coordinates */
		if (gra_inputstate == NOEVENT) continue;

		if ((gra_inputstate&ISBUTTON) != 0 && (gra_inputstate&BUTTONUP) != 0) break;
		if ((gra_inputstate&MOTION) != 0)
		{
			keepon = (*eachdown)(gra_cursorx, gra_cursory);
		} else if ((gra_inputstate&CHARREAD) != 0)
		{
			keepon = (*eachchar)(gra_cursorx, gra_cursory, gra_inputstate&CHARREAD);
		}
		if (el_pleasestop != 0) keepon = 1;
	}
	gra_inputstate = NOEVENT;

	/* inform the user that all is done */
	(*done)();

	/* restore the state of the world */
	if (purpose != TRACKHSELECTING) setdefaultcursortype(NULLCURSOR);
}

/*
 * routine to read the current co-ordinates of the tablet and return them
 * in "*x" and "*y".
 */
void readtablet(INTSML *x, INTSML *y)
{
	*x = gra_cursorx;   *y = gra_cursory;
}

/*
 * routine to turn off the cursor tracking if it is on
 */
void stoptablet(void)
{
	if (us_cursorstate != IBEAMCURSOR) setdefaultcursortype(NULLCURSOR);
}

/******************** KEYBOARD CONTROL ********************/

/*
 * routine to get the next character from the keyboard
 */
INTSML getnxtchar(void)
{
	REGISTER INTSML i;
	EventRecord theEvent;

	if (gra_inputstate != NOEVENT && (gra_inputstate&CHARREAD) != 0)
	{
		i = gra_inputstate & CHARREAD;
		if ((gra_inputstate&COMMANDDOWN) != 0) i |= 0200;
		gra_inputstate = NOEVENT;
		return(i);
	}
	if (us_cursorstate != IBEAMCURSOR) setdefaultcursortype(WANTTTYCURSOR);
	for(;;)
	{
		gra_waitforaction(0, &theEvent);
		if (gra_inputstate != NOEVENT && (gra_inputstate&CHARREAD) != 0) break;
	}
	i = gra_inputstate & CHARREAD;
	if ((gra_inputstate&COMMANDDOWN) != 0) i |= 0200;
	gra_inputstate = NOEVENT;
	if (us_cursorstate != IBEAMCURSOR) setdefaultcursortype(NULLCURSOR);
	return(i);
}

void checkforinterrupt(void)
{
	EventRecord theEvent;
	short oak;
	extern INTSML gra_cancheck;

	if (el_pleasestop == 0 && gra_cancheck != 0)
	{
		gra_cancheck = 0;
		oak = EventAvail(keyDownMask, &theEvent);
		if (oak != 0)
		{
			(void)GetNextEvent(keyDownMask, &theEvent);
			if (theEvent.what == keyDown)
				if ((theEvent.modifiers & cmdKey) != 0)
					if ((theEvent.message & charCodeMask) == '.')
			{
				ttyputmsg(_("Interrupted..."));
				el_pleasestop = 1;
			}
		}
	}
}

/*
 * routine to tell whether data is waiting at the terminal.  Returns nonzero
 * if data is ready.
 */
INTSML ttydataready(void)
{
	EventRecord theEvent;

	/* see if something is already pending */
	if (gra_inputstate != NOEVENT)
	{
		if ((gra_inputstate&CHARREAD) != 0) return(1);
		return(0);
	}

	/* wait for something and analyze it */
	gra_waitforaction(1, &theEvent);
	if (gra_inputstate != NOEVENT && (gra_inputstate&CHARREAD) != 0) return(1);
	return(0);
}

/****************************** FILES ******************************/

/*
 * Routine to set the type and creator of file "name" to "type" and "creator"
 */
void mac_settypecreator(char *name, INTBIG type, INTBIG creator)
{
	FInfo finfo;
	char pname[256];

	(void)strcpy(&pname[1], name);
	pname[0] = strlen(name);
	(void)GetFInfo((unsigned char *)pname, 0, &finfo);
	finfo.fdType = type;
	finfo.fdCreator = creator;
	(void)SetFInfo((unsigned char *)pname, 0, &finfo);
}

/*
 * Macintosh routine to return the name of the system folder
 */
char *gra_systemfoldername(void)
{
	static char foldername[256];
	char thisname[256];
	INTSML i, j, len;
	WDPBRec wpb;
	CInfoPBRec cpb;
	SysEnvRec sysenv;

	/* get system folder reference number */
	SysEnvirons(2, &sysenv);

	/* determine directory ID of system folder */
	wpb.ioNamePtr = (unsigned char *)foldername;
	wpb.ioVRefNum = sysenv.sysVRefNum;
	wpb.ioWDIndex = 0;
	wpb.ioWDProcID = 0L;
	wpb.ioWDVRefNum = wpb.ioVRefNum;
	i = PBGetWDInfo(&wpb, 0);
	if (i != noErr) return("");

	/* find the name of the system folder */
	foldername[0] = 0;
	cpb.dirInfo.ioDrParID = wpb.ioWDDirID;
	for(j=0; ; j++)
	{
		cpb.dirInfo.ioNamePtr = (unsigned char *)thisname;
		cpb.dirInfo.ioVRefNum = wpb.ioWDVRefNum;
		cpb.dirInfo.ioFDirIndex = -1;
		cpb.dirInfo.ioDrDirID = cpb.dirInfo.ioDrParID;
		i = PBGetCatInfo(&cpb, 0);
		if (i != noErr) return("");
		len = thisname[0] + 1;
		if (j != 0) thisname[len++] = ':';
		for(i=0; i<foldername[0]; i++) thisname[len++] = foldername[i+1];
		thisname[len] = 0;
		for(i=0; i<len; i++) foldername[i+1] = thisname[i+1];
		foldername[0] = len-1;
		if (cpb.dirInfo.ioDrDirID == fsRtDirID) break;
	}

	/* see if there is a folder with the name ":Preferences:Electric Files" added on */
	for(i=0; i<foldername[0]; i++) thisname[i+1] = foldername[i+1];
	thisname[foldername[0]+1] = 0;
	(void)strcat(&thisname[1], ":Preferences:Electric Files");
	thisname[0] = strlen(&thisname[1]);
	cpb.dirInfo.ioNamePtr = (unsigned char *)thisname;
	cpb.dirInfo.ioVRefNum = wpb.ioWDVRefNum;
	cpb.dirInfo.ioFDirIndex = 0;
	i = PBGetCatInfo(&cpb, 0);
	if (i != noErr) return(&foldername[1]);
	if ((cpb.dirInfo.ioFlAttrib&16) == 0) return(&foldername[1]);
	(void)strcpy(foldername, &thisname[1]);
	return(foldername);
}

/*
 * Routine to convert a Pascal string file name in "thisname" and its volume
 * reference number in "refnum" into a full path name (and a C string).
 */
char *gra_makefullname(char *thisname, INTSML refnum)
{
	INTSML err, len, i;
	CInfoPBRec cpb;
	char line[256];
	static char sofar[256];

	len = thisname[0];
	for(i=0; i<len; i++) sofar[i] = thisname[i+1];
	sofar[len] = 0;
	cpb.hFileInfo.ioVRefNum = refnum;
	cpb.hFileInfo.ioDirID = 0;
	cpb.hFileInfo.ioCompletion = 0L;
	cpb.hFileInfo.ioNamePtr = (StringPtr)line;
	cpb.hFileInfo.ioFDirIndex = -1;
	for(;;)
	{
		err = PBGetCatInfo(&cpb, 0);
		if (err != noErr) break;
		line[line[0]+1] = 0;
		strcat(line, ":");
		strcat(line, sofar);
		strcpy(sofar, &line[1]);
		if (cpb.hFileInfo.ioFlParID == 0) break;
		cpb.hFileInfo.ioDirID = cpb.hFileInfo.ioFlParID;
	}
	return(sofar);
}

/*
 * Routine to convert a FSS specification in "theFSS" into a full path name
 * (and a C string).
 */
char *gra_makefullnamefromFSS(FSSpec theFSS)
{
	DirInfo block;
	short len;
	static char fileName[256];
	char dirName[256];
	OSErr err;

	if (theFSS.parID != 0)
	{
		theFSS.name[theFSS.name[0]+1] = 0;
		strcpy(fileName, (char *)&theFSS.name[1]);
		block.ioDrParID = theFSS.parID;
		block.ioNamePtr = (StringPtr)dirName;
		do {
			block.ioVRefNum = theFSS.vRefNum;
			block.ioFDirIndex = -1;
			block.ioDrDirID = block.ioDrParID;
			err = PBGetCatInfo((CInfoPBPtr)&block, 0);
			dirName[dirName[0]+1] = 0;
			len = strlen(&dirName[1]);
			BlockMove(fileName, fileName + len+1, strlen(fileName)+1);
			strcpy(fileName, &dirName[1]);
			fileName[len] = ':';
		} while (block.ioDrDirID != 2);
		return(fileName);
	}

	/* no directory ID specified in FSS, use old name/vrefnum method */
	return(gra_makefullname((char *)theFSS.name, theFSS.vRefNum));
}

/*
 * File filter for binary library files.  Accepts "Elec" type files or those
 * that end with ".elib".
 */
pascal Boolean gra_fileFilterProc(CInfoPBPtr pb, Ptr mydata)
{
	INTSML pos;
	unsigned char *str;

	if ((pb->hFileInfo.ioFlAttrib&ioDirMask) != 0) return(0);
	str = pb->hFileInfo.ioNamePtr;
	pos = str[0];
	if (str[pos-4] == '.' && str[pos-3] == 'e' && str[pos-2] == 'l' && str[pos-1] == 'i' &&
		str[pos] == 'b') return(0);
	if (pb->hFileInfo.ioFlFndrInfo.fdType == 'Elec') return(0);
	return(1);
}

/*
 * Routine to prompt for multiple files of type "filetype", giving the
 * message "msg".  Returns a string that contains all of the file names,
 * separated by the NONFILECH (a character that cannot be in a file name).
 */
char *multifileselectin(char *msg, INTBIG filetype)
{
	return(fileselect(msg, filetype, ""));
}

/*
 * Routine to display a standard file prompt dialog and return the selected file.
 * The prompt message is in "msg" and the kind of file is in "filetype".  The default
 * output file name is in "defofile" (only used if "filetype" is negative).
 */
char *fileselect(char *msg, INTBIG filetype, char *defofile)
{
	SFTypeList myTypes;
	static Point SFwhere = {90, 82};
	INTBIG len;
	REGISTER INTSML i, save;
	StandardFileReply reply;
	WindowPtr savewin;
	WDPBRec wdpb;
	char leng, ofile[256], prompt[256];

	if (us_logplay != NULL)
	{
		(void)xfread((char *)&gra_action, 1, sizeof (gra_action), us_logplay);
		if ((gra_action.kind&0xFFFF) != FILEREPLY) gra_localstring[0] = 0; else
		{
			(void)xfread(&leng, 1, 1, us_logplay);
			len = leng;
			(void)xfread(gra_localstring, 1, len, us_logplay);
			gra_localstring[len] = 0;
		}
	} else
	{
		strcpy(&prompt[1], msg);
		prompt[0] = strlen(msg);
		GetPort(&savewin);
		if ((filetype&FILETYPEWRITE) == 0)
		{
			/* input file selection */
			ParamText((unsigned char *)prompt, "\p", "\p", "\p");
			if ((filetype&FILETYPE) == io_filetypeblib)
			{
				/* special case for Electric binary libraries */
				CustomGetFile(gra_fileFilterProcUPP, -1, myTypes, &reply, FILEINDIALOG, SFwhere,
					0, 0, 0, 0, 0);
			} else
			{
				/* standard file input */
				CustomGetFile(0, -1, myTypes, &reply, FILEINDIALOG, SFwhere, 0, 0, 0, 0, 0);
			}
		} else
		{
			/* output file selection */
			for(i = strlen(defofile)-1; i > 0; i--)
				if (defofile[i] == ':') break;
			if (i > 0)
			{
				/* there is a ":" in the path, set the default directory */
				i++;
				save = defofile[i];
				defofile[i] = 0;
				(void)strcpy(&ofile[1], defofile);
				ofile[0] = strlen(defofile);
				wdpb.ioNamePtr = (StringPtr)ofile;
				PBHSetVol(&wdpb, 0);
				defofile[i] = save;
				defofile = &defofile[i];
			}
			(void)strcpy(&ofile[1], defofile);
			ofile[0] = strlen(defofile);
			StandardPutFile((unsigned char *)prompt, (unsigned char *)ofile, &reply);
		}
		SetPort(savewin);
		if (reply.sfGood == 0) gra_localstring[0] = 0; else
			(void)strcpy(gra_localstring, gra_makefullnamefromFSS(reply.sfFile));
	}

	/* log this result if logging */
	if (us_logrecord != NULL)
	{
		gra_action.kind = FILEREPLY;
		(void)xfwrite((char *)&gra_action, 1, sizeof (gra_action), us_logrecord);
		leng = strlen(gra_localstring);
		(void)xfwrite(&leng, 1, 1, us_logrecord);
		len = leng;
		(void)xfwrite(gra_localstring, 1, len, us_logrecord);
		gra_logrecordindex++;
		if (gra_logrecordindex >= us_logflushfreq)
		{
			/* flush the session log file */
			gra_logrecordindex = 0;
			xflushbuf(us_logrecord);
		}
	}
	return(gra_localstring);
}

/*
 * routine to handle an "open document" Apple Event and read a library
 */
pascal OSErr gra_handleodoc(AppleEvent *theAppleEvent, AppleEvent *reply, long handlerRefCon)
{
	FSSpec myFSS;
	AEDescList docList;
	OSErr err;
	char *argv[3];
	long sindex, itemsInList;
	Size actualSize;
	AEKeyword keywd;
	DescType returnedType;

	/* get the direct parameter, a descriptor list, and put it into a doclist */
	err = AEGetParamDesc(theAppleEvent, keyDirectObject, typeAEList, &docList);
	if (err) return(err);

	/* check for missing parameters */
	err = AEGetAttributePtr(theAppleEvent, keyMissedKeywordAttr, typeWildCard,
		&returnedType, nil, 0, &actualSize);
	if (err != errAEDescNotFound)
	{
		if (!err) return(errAEEventNotHandled);
		return(err);
	}

	/* count the number of descriptor records in the list */
	err = AECountItems(&docList, &itemsInList);

	/*
	 * now get each descriptor record from the list, coerce the returned data to
	 * an FSSpec record, and open the associated file
	 */
	for(sindex = 1; sindex <= itemsInList; sindex++)
	{
		err = AEGetNthPtr(&docList, sindex, typeFSS, &keywd, &returnedType,
			(Ptr)&myFSS, sizeof(myFSS), &actualSize);
		if (err) return(err);
		us_beginchanges();
		argv[0] = "read";
		argv[1] = gra_makefullnamefromFSS(myFSS);
		argv[2] = "make-current";
		us_library(3, argv);
		if (el_curwindowpart != NOWINDOWPART)
			us_ensurepropertechnology(el_curwindowpart->curnodeproto, 0, 1);
		us_endchanges(NOWINDOWPART);		
	}
	err = AEDisposeDesc(&docList);

	return(noErr);
}

/*
 * Helper routine to initialize the list of files in a directory.
 */
void gra_initfilelist(void)
{
	if (gra_fileliststringarray == 0)
	{
		gra_fileliststringarray = newstringarray(db_cluster);
		if (gra_fileliststringarray == 0) return;
	}
	clearstrings(gra_fileliststringarray);
}

/*
 * Helper routine to add "file" to the list of files in a directory.
 * Returns nonzero on error.
 */
INTSML gra_addfiletolist(char *file)
{
	addtostringarray(gra_fileliststringarray, file);
	return(0);
}

/*
 * Routine to search for all of the files/directories in directory "directory" and
 * return them in the array of strings "filelist".  Returns the number of files found.
 */
INTSML filesindirectory(char *directory, char ***filelist)
{
	INTSML err, i;
	INTBIG dirid, len;
	char file[256];
	CInfoPBRec cinfo;

	if (*directory == 0) dirid = 0; else
	{
		(void)strcpy(&file[1], directory);
		file[0] = strlen(directory);
		SetVol((unsigned char *)file, 0);
		cinfo.hFileInfo.ioCompletion = 0L;
		cinfo.hFileInfo.ioNamePtr = (StringPtr)file;
		cinfo.hFileInfo.ioVRefNum = 0;
		cinfo.hFileInfo.ioFDirIndex = 0;
		cinfo.hFileInfo.ioDirID = 0L;
		err = PBGetCatInfo(&cinfo, 0);
		if (err != noErr) return(0);
		dirid = cinfo.hFileInfo.ioDirID;
	}

	gra_initfilelist();
	for(i=1; ; i++)
	{
		cinfo.hFileInfo.ioCompletion = 0L;
		cinfo.hFileInfo.ioNamePtr = (StringPtr)file;
		cinfo.hFileInfo.ioVRefNum = 0;
		cinfo.hFileInfo.ioFDirIndex = i;
		cinfo.hFileInfo.ioDirID = dirid;
		err = PBGetCatInfo(&cinfo, 0);
		if (err != noErr) break;
		file[file[0]+1] = 0;
		if (gra_addfiletolist(&file[1]) != 0) return(0);
	}

	*filelist = getstringarray(gra_fileliststringarray, &len);
	return(len);
}

/* routine to convert a path name with "~" to a real path */
char *truepath(char *line)
{
	/* only have tilde parsing on UNIX */
	return(line);
}

/*
 * Routine to turn the file name "file" into a unique one by changing the
 * "XXX" part to a number that makes the file unique.
 */
void emaketemp(char *file)
{
}

/*
 * routine to rename file "file" to "newfile"
 * returns nonzero on error
 */
INTBIG erename(char *file, char *newfile)
{
	return(1);
}

/*
 * routine to delete file "file"
 */
INTBIG eunlink(char *file)
{
	return(-1);
}

/*
 * Routine to return information about the file or directory "name":
 *  0: does not exist
 *  1: is a file
 *  2: is a directory
 */
INTSML fileexistence(char *name)
{
	struct stat buf;

	if (stat(name, &buf) < 0) return(0);
	if ((buf.st_mode & S_IFMT) == S_IFDIR) return(2);
	return(1);
}

/*
 * Routine to create a directory.
 * Returns nonzero on error.
 */
INTSML createdirectory(char *dirname)
{
	FSSpec fsp;
	long dir;
	INTSML err;
	char pname[256];

	strcpy(&pname[1], dirname);
	pname[0] = strlen(dirname);
	err = FSMakeFSSpec(0, 0, (unsigned char *)pname, &fsp);
	if (err != noErr && err != fnfErr) return(1);
	err = FSpDirCreate(&fsp, smSystemScript, &dir);
	if (err != noErr && err != dupFNErr) return(1);
	return(0);
}

/*
 * Routine to return the current directory name
 */
char *currentdirectory(void)
{
	return(gra_makefullname((char *)"\p", 0));
}

/*
 * Routine to return the home directory (returns 0 if it doesn't exist)
 */
char *hashomedir(void)
{
	return(0);
}

/*
 * Routine to return the path to the "options" library.
 */
char *optionsfilepath(void)
{
	(void)initinfstr();
	(void)addstringtoinfstr(el_libdir);
	(void)addstringtoinfstr("electricoptions.elib");
	return(returninfstr());
}

/*
 * routine to return the date of the last modification to file "filename"
 */
UINTBIG filedate(char *filename)
{
	struct stat buf;

	stat(filename, &buf);
	buf.st_mtime -= machinetimeoffset();
	return(buf.st_mtime);
}

/*
 * Routine to lock a resource called "lockfilename" by creating such a file
 * if it doesn't exist.  Returns nonzero if successful, zero if unable to
 * lock the file.
 */
INTSML lockfile(char *lockfilename)
{
	INTSML err;
	FSSpec fsp;
	char pname[256];

	strcpy(&pname[1], lockfilename);
	pname[0] = strlen(lockfilename);
	err = FSMakeFSSpec(0, 0, (unsigned char *)pname, &fsp);
	err = FSpCreate(&fsp, 'Elec', 'Lock', smSystemScript);
	if (err == noErr) return(1);
	return(0);
}

/*
 * Routine to unlock a resource called "lockfilename" by deleting such a file.
 */
void unlockfile(char *lockfilename)
{
	FSSpec fsp;
	char pname[256];

	strcpy(&pname[1], lockfilename);
	pname[0] = strlen(lockfilename);
	FSMakeFSSpec(0, 0, (unsigned char *)pname, &fsp);
	FSpDelete(&fsp);
}

/*
 * Routine to show file "document" in a browser window.
 * Returns nonzero if the operation cannot be done.
 *
 * This routine is taken from "FinderOpenSel" by:
 * C.K. Haun <TR>
 * Apple Developer Tech Support
 * April 1992, Cupertino, CA USA
 * Of course, Copyright 1991-1992, Apple Computer Inc.
 */
INTSML browsefile(char *document)
{
	int launch;
	AppleEvent aeEvent, aeReply;
	AEDesc aeDirDesc, listElem;
	FSSpec dirSpec, procSpec;
	AEDesc fileList;
	OSErr myErr;
	ProcessSerialNumber process;
	AliasHandle DirAlias, FileAlias;
	FSSpec theFileToOpen;
	ProcessInfoRec infoRec;
	Str31 processName;
	Str255 pname;
	static AEDesc myAddressDesc;
	static int gotFinderAddress = 0;

	/* get the FSSpec of the file */
	launch = 1;
	strcpy((char *)&pname[1], document);
	pname[0] = strlen(document);
	FSMakeFSSpec(0, 0, pname, &theFileToOpen);

	/* we are working locally, find the finder on this machine */
	if (gotFinderAddress == 0)
	{
		infoRec.processInfoLength = sizeof(ProcessInfoRec);
		infoRec.processName = (StringPtr)&processName;
		infoRec.processAppSpec = &procSpec;
		myErr = gra_findprocess('FNDR', 'MACS', &process, &infoRec);
		if (myErr != noErr) return(1);

		myErr = AECreateDesc(typeProcessSerialNumber, (Ptr)&process, sizeof(process), &myAddressDesc);
		if (myErr != noErr) return(1);
		gotFinderAddress++;
	}

	/* Create the FinderEvent */
	if (launch != 0)
	{
		myErr = AECreateAppleEvent('FNDR', 'sope', &myAddressDesc,
			kAutoGenerateReturnID, kAnyTransactionID, &aeEvent);
	} else
	{
		myErr = AECreateAppleEvent('FNDR', 'srev', &myAddressDesc,
			kAutoGenerateReturnID, kAnyTransactionID, &aeEvent);
	}
	if (myErr != noErr) return(1);

	/*
	 * Now we build all the bits of an OpenSelection event.  Basically, we need to create
	 * an alias for the item to open, and an alias to the parent folder (directory) of that
	 * item.  We can also pass a list of files if we want.
	 */

	/* make a spec for the parent folder */
	FSMakeFSSpec(theFileToOpen.vRefNum, theFileToOpen.parID, nil, &dirSpec);
	NewAlias(nil, &dirSpec, &DirAlias);

	/* Create alias for file */
	NewAlias(nil, &theFileToOpen, &FileAlias);

	/* Create the file list */
	myErr = AECreateList(nil, 0, false, &fileList);

	/* create the folder descriptor */
	HLock((Handle)DirAlias);
	AECreateDesc(typeAlias, (Ptr)*DirAlias, GetHandleSize((Handle)DirAlias), &aeDirDesc);
	HUnlock((Handle)DirAlias);
	if ((myErr = AEPutParamDesc(&aeEvent, keyDirectObject, &aeDirDesc)) == noErr)
	{
		/* done with the desc, kill it */
		AEDisposeDesc(&aeDirDesc);

		/* create the file descriptor and add to aliasList */
		HLock((Handle)FileAlias);
		AECreateDesc(typeAlias, (Ptr)*FileAlias, GetHandleSize((Handle)FileAlias), &listElem);
		HLock((Handle)FileAlias);
		myErr = AEPutDesc(&fileList, 0, &listElem);
	}
	if (myErr == noErr)
	{
		AEDisposeDesc(&listElem);

		/* Add the file alias list to the event */
		myErr = AEPutParamDesc(&aeEvent, 'fsel', &fileList);
		AEDisposeDesc(&fileList);

		if (myErr == noErr)
			myErr = AESend(&aeEvent, &aeReply, kAENoReply + kAEAlwaysInteract + kAECanSwitchLayer,
				kAENormalPriority, kAEDefaultTimeout, nil, nil);
		if (launch == 0)
			SetFrontProcess(&process);
	}
	AEDisposeDesc(&aeEvent);

	if ((Handle)DirAlias)
		DisposeHandle((Handle)DirAlias);
	if ((Handle)FileAlias)
		DisposeHandle((Handle)FileAlias);
	return(0);
}

/* This runs through the process list looking for the indicated application */
OSErr gra_findprocess(OSType typeToFind, OSType creatorToFind, ProcessSerialNumberPtr processSN,
	ProcessInfoRecPtr infoRecToFill)
{
	ProcessSerialNumber tempPSN;
	OSErr myErr = noErr;
	tempPSN.lowLongOfPSN = kNoProcess;
	processSN->lowLongOfPSN = kNoProcess;
	processSN->highLongOfPSN = kNoProcess;
	do
	{
		myErr = GetNextProcess(processSN);
		if (myErr != noErr) break;
		GetProcessInformation(processSN, infoRecToFill);
	}
	while (infoRecToFill->processSignature != creatorToFind ||
		infoRecToFill->processType != typeToFind);
	return(myErr);
}

/****************************** CHANNELS ******************************/

/*
 * routine to create a pipe connection between the channels in "channels"
 */
INTBIG epipe(int channels[2])
{
	return(0);
}

/*
 * Routine to set channel "channel" into an appropriate mode for single-character
 * interaction (i.e. break mode).
 */
void setinteractivemode(int channel)
{
}

/*
 * Routine to replace channel "channel" with a pointer to file "file".
 * Returns a pointer to the original channel.
 */
INTBIG channelreplacewithfile(int channel, char *file)
{
	return(0);
}

/*
 * Routine to replace channel "channel" with new channel "newchannel".
 * Returns a pointer to the original channel.
 */
INTBIG channelreplacewithchannel(int channel, int newchannel)
{
	return(0);
}

/*
 * Routine to restore channel "channel" to the pointer that was returned
 * by "channelreplacewithfile" or "channelreplacewithchannel" (and is in "saved").
 */
void channelrestore(int channel, INTBIG saved)
{
}

/*
 * Routine to read "count" bytes from channel "channel" into "addr".
 * Returns the number of bytes read.
 */
INTBIG eread(int channel, char *addr, INTBIG count)
{
	return(0);
}

/*
 * Routine to write "count" bytes to channel "channel" from "addr".
 * Returns the number of bytes written.
 */
INTBIG ewrite(int channel, char *addr, INTBIG count)
{
	return(0);
}

/*
 * routine to close a channel in "channel"
 */
INTBIG eclose(int channel)
{
	return(0);
}

/*************************** TIME ROUTINES ***************************/

/*
 * The Macintosh uses an "epoch" of January 1, 1904.
 * All other machines use an "epoch" of January 1, 1970.
 * This means that time on the Macintosh is off by 66 years.
 * Specifically, there were 17 leap years between 1904 and 1970, so
 * there were  66*365+17 = 24107 days
 * Since each day has 24*60*60 = 86400 seconds in it, there were
 * 24107 * 86400 = 2,082,844,800 seconds difference between epochs.
 *
 * However, since Macintosh systems deal with local time and other
 * systems deal with GMT time, the Mac must also offset by the time zone.
 */
#define	MACEPOCHOFFSET 2082844800

/*
 * This routine returns the amount to add to the operating-system time
 * to adjust for a common time format.
 */
UINTBIG machinetimeoffset(void)
{
	static UINTBIG offset;
	static INTSML offsetcomputed = 0;
	UINTBIG timezoneoffset;
	MachineLocation ml;

	if (offsetcomputed == 0)
	{
		offsetcomputed = 1;
		offset = MACEPOCHOFFSET;
		ReadLocation(&ml);
		timezoneoffset = ml.u.gmtDelta & 0xFFFFFF;
		if ((timezoneoffset & 0x800000) != 0)
			timezoneoffset = -(((~timezoneoffset) & 0xFFFFFF) + 1);
		offset += timezoneoffset;
	}
	return(offset);
}

/* returns the current time in 60ths of a second */
UINTBIG ticktime(void)
{
	return(TickCount());
}

/* returns the double-click interval in 60ths of a second */
INTBIG doubleclicktime(void)
{
	return(gra_doubleclick);
}

/*
 * Routine to wait "ticks" sixtieths of a second and then return.
 */
void gotosleep(INTBIG ticks)
{
#ifdef NEWCODEWARRIOR
	unsigned long l;

	Delay(ticks, &l);
#else
	long l;

	Delay(ticks, &l);
#endif
}

/*
 * Routine to start counting time.
 */
void starttimer(void)
{
	gra_timestart = clock();
}

/*
 * Routine to stop counting time and return the number of elapsed seconds
 * since the last call to "starttimer()".
 */
float endtimer(void)
{
	float seconds;
	UINTBIG thistime;

	thistime = clock();
	seconds = ((float)(thistime - gra_timestart)) / 60.0;
	return(seconds);
}

/*************************** EVENT ROUTINES ***************************/

/*
 * Hack routine that writes a fake "mouse up" event to the session logging file.  This
 * is necessary because certain toolbox routines (i.e. TEClick) track the
 * mouse while it is down in order to handle selection of text.  Since this tracking
 * cannot be obtained and logged, a fake "SHIFT mouse" is logged after the routine finishes
 * so that the text selection is effected in the same way
 */
void gra_fakemouseup(void)
{
	Point p;

	if (us_logrecord == NULL) return;
	GetMouse(&p);
	gra_action.kind = ISBUTTON | SHIFTDOWN;
	gra_action.x = p.h;
	gra_action.y = p.v;
	if (xfwrite((char *)&gra_action, 1, sizeof (gra_action), us_logrecord) == 0)
	{
		ttyputerr(_("Error writing session log file: recording disabled"));
		logfinishrecord();
	}
	gra_logrecordindex++;
	if (gra_logrecordindex >= us_logflushfreq)
	{
		/* flush the session log file */
		gra_logrecordindex = 0;
		xflushbuf(us_logrecord);
	}
}

/*
 * helper routine to wait for some keyboard or mouse input.  The value of "nature" is:
 *   0  allow mouse, keyboard
 *   1  allow mouse, keyboard, pulldown menus
 *   2  allow mouse, keyboard, pulldown menus, motion
 */
void gra_waitforaction(INTSML nature, EventRecord *theEvent)
{
	REGISTER INTBIG err;
	static INTBIG last_cursorx, last_cursory;
	REGISTER INTSML saveevent;
	static INTBIG fakewhen = 0;
	Rect r, fr;

	if (us_logplay != NULL)
	{
		if (EventAvail(updateMask | activMask, theEvent) != 0)
		{
			gra_nextevent(4, theEvent);
			return;
		}
		flushscreen();
		err = xfread((char *)&gra_action, 1, sizeof (gra_action), us_logplay);
		if (stopping(STOPREASONPLAYBACK)) err = 0;
		if (err != 0)
		{
			if (gra_playbackmultiple <= 0)
			{
				gra_playbackmultiple = 0;
				for(;;)
				{
					gra_inputstate = NOEVENT;
					gra_nextevent(0, theEvent);
					if (gra_inputstate == NOEVENT) continue;
					if ((gra_inputstate&CHARREAD) == 0) continue;
					if ((gra_inputstate&CHARREAD) < '0' || (gra_inputstate&CHARREAD) > '9') break;
					gra_playbackmultiple = gra_playbackmultiple * 10 + (gra_inputstate&CHARREAD) - '0';
				}
				if (gra_inputstate != NOEVENT && (gra_inputstate&CHARREAD) == 'q')
					err = 0;
			}
			gra_playbackmultiple--;

			/* allow Command-. to interrupt long playbacks */
			if ((gra_playbackmultiple%10) == 9)
			{
				if (EventAvail(keyDownMask, theEvent) != 0)
				{
					(void)WaitNextEvent(keyDownMask, theEvent, 0, 0);
					if ((theEvent->modifiers & cmdKey) != 0 &&
						(theEvent->message & charCodeMask) == '.')
							gra_playbackmultiple = 0;
				}
			}

			gra_inputstate = gra_action.kind & 0xFFFF;
			gra_cursorx = gra_action.x;
			gra_cursory = gra_action.y;
			us_state &= ~GOTXY;
			if (gra_inputstate == MENUEVENT)
			{
				(void)xfread((char *)&gra_action, 1, sizeof (gra_action), us_logplay);
				gra_lowmenu = gra_action.x;
				gra_highmenu = gra_action.y;
			} else if (gra_inputstate == WINDOWCHANGE)
			{
				(void)xfread((char *)&r, 1, sizeof (r), us_logplay);
				fr = (*((WindowPeek)el_curwindowframe->realwindow)->strucRgn)->rgnBBox;
				r.left += 1;         r.right -= 2;
				r.top += MENUSIZE;   r.bottom -= 2;
				MoveWindow((WindowPtr)el_curwindowframe->realwindow, r.left, r.top, 0);
				SizeWindow((WindowPtr)el_curwindowframe->realwindow, r.right-r.left,
					r.bottom-r.top, 1);
				gra_mygrowwindow((WindowPtr)el_curwindowframe->realwindow, r.right-r.left,
					r.bottom-r.top, &fr);
			}
		}

		/* convert to an event */
		fakewhen += gra_doubleclick + 1;
		theEvent->what = nullEvent;
		theEvent->when = fakewhen;
		theEvent->modifiers = 0;
		if ((gra_inputstate&SHIFTDOWN) != 0) theEvent->modifiers |= shiftKey;
		if ((gra_inputstate&COMMANDDOWN) != 0) theEvent->modifiers |= cmdKey;
		if ((gra_inputstate&OPTIONDOWN) != 0) theEvent->modifiers |= optionKey;
		if ((gra_inputstate&CONTROLDOWN) != 0) theEvent->modifiers |= controlKey;
		if ((gra_inputstate&BUTTONUP) != 0) theEvent->modifiers |= btnState;
		theEvent->where.h = gra_cursorx;
		theEvent->where.v = el_curwindowframe->revy - gra_cursory;
		if ((gra_inputstate&CHARREAD) != 0)
		{
			theEvent->what = keyDown;
			theEvent->message = gra_inputstate & CHARREAD;
		} else if ((gra_inputstate&ISBUTTON) != 0)
		{
			if ((gra_inputstate&BUTTONUP) == 0) theEvent->what = mouseDown; else
				theEvent->what = mouseUp;
			if ((gra_inputstate&DOUBLECLICK) != 0)
			{
				theEvent->when--;
				fakewhen--;
			}
		}

		/* stop now if end of playback file */
		if (err == 0)
		{
			ttyputmsg(_("End of session playback file"));
			xclose(us_logplay);
			us_logplay = NULL;
			return;
		}
	} else
	{
		flushscreen();
		gra_nextevent(nature, theEvent);
	}

	if (us_logrecord != NULL && gra_inputstate != NOEVENT)
	{
		saveevent = 1;
		if ((gra_inputstate&MOTION) != 0)
		{
			if (gra_cursorx == last_cursorx && gra_cursory == last_cursory) saveevent = 0;
		}
		if (saveevent != 0)
		{
			gra_action.kind = gra_inputstate;
			gra_action.x = last_cursorx = gra_cursorx;
			gra_action.y = last_cursory = gra_cursory;
			if (xfwrite((char *)&gra_action, 1, sizeof (gra_action), us_logrecord) == 0)
			{
				ttyputerr(_("Error writing session log file: recording disabled"));
				logfinishrecord();
			}
			if (gra_inputstate == MENUEVENT)
			{
				gra_action.x = gra_lowmenu;
				gra_action.y = gra_highmenu;
				(void)xfwrite((char *)&gra_action, 1, sizeof (gra_action), us_logrecord);
			} else if (gra_inputstate == WINDOWCHANGE)
			{
				if (el_curwindowframe != NOWINDOWFRAME)
				{
					r = (*((WindowPeek)el_curwindowframe->realwindow)->strucRgn)->rgnBBox;
					(void)xfwrite((char *)&r, 1, sizeof (r), us_logrecord);
				}
			}
			gra_logrecordindex++;
			if (gra_logrecordindex >= us_logflushfreq)
			{
				/* flush the session log file */
				gra_logrecordindex = 0;
				xflushbuf(us_logrecord);
			}
		}
	}

	/* deal with special event types */
	if (gra_inputstate == MENUEVENT)
	{
		if (gra_highmenu == appleMENU && gra_lowmenu == 1)
		{
			gra_applemenu(gra_lowmenu);
		} else gra_nativemenudoone(gra_lowmenu, gra_highmenu);
		gra_inputstate = NOEVENT;
	} else if (gra_inputstate == WINDOWCHANGE) gra_inputstate = NOEVENT;
}

/*
 * Routine to get the next Electric input action and set the global "gra_inputstate"
 * accordingly.  The value of "nature" is:
 *   0  allow mouse, keyboard (no window switching)
 *   1  allow mouse, keyboard, pulldown menus
 *   2  allow mouse, keyboard, motion
 *   4  allow update and activate events only
 */
void gra_nextevent(INTSML nature, EventRecord *theEvent)
{
#ifdef	USETK
	if (gra_messageswindow != 0) TEIdle(gra_TEH);
	(void)Tcl_DoOneEvent(TCL_DONT_WAIT);
	if (gra_eventqueuehead != gra_eventqueuetail)
	{
		gra_inputstate = gra_eventqueuehead->inputstate;
		gra_cursorx = gra_eventqueuehead->cursorx;
		gra_cursory = gra_eventqueuehead->cursory;
		us_state &= ~GOTXY;
		gra_eventqueuehead++;
		if (gra_eventqueuehead >= &gra_eventqueue[EVENTQUEUESIZE])
			gra_eventqueuehead = gra_eventqueue;
	}
#else
	INTSML oak, cntlCode, x, y, stroke, oldpos, findres, inmenu;
	INTBIG key, theResult, xv, yv, lx, hx, ly, hy;
	WStateData *wst;
	REGISTER WINDOWPART *w;
	REGISTER EDITOR *e;
	REGISTER VARIABLE *var;
	char *par[1], *str, lastchar;
	static INTSML firstwupdate = 1;
	WindowPtr theWindow, win, frontWindow;
	Rect r, fr;
	ControlHandle theControl;
	WINDOWFRAME *wf;
	static INTSML overrodestatus = 0;
	COMMANDBINDING commandbinding;

	gra_inputstate = NOEVENT;
	HiliteMenu(0);
	if (gra_messageswindow != 0) TEIdle(gra_TEH);
	if (nature == 4) oak = WaitNextEvent(updateMask | activMask, theEvent, 0, 0); else
		oak = WaitNextEvent(everyEvent, theEvent, 0, 0);
	if (oak == 0 && nature != 4 && gra_motioncheck != 0)
	{
		gra_motioncheck = 0;
		oak = theEvent->what = app3Evt;
		if (Button())
		{
			theEvent->modifiers |= btnState;
		} else
		{
			theEvent->modifiers &= ~btnState;
		}
	}
	if (oak == 0)
	{
		if (nature != 2)
		{
			if (FindWindow(theEvent->where, &theWindow) != inContent)
				setdefaultcursortype(NORMALCURSOR);
		}
		return;
	}
	if (oak != 0) switch (theEvent->what)
	{
		case mouseUp:
			if (el_curwindowframe != NOWINDOWFRAME)
			{
				SetPort((WindowPtr)el_curwindowframe->realwindow);
				GlobalToLocal(&theEvent->where);
				gra_cursorx = theEvent->where.h;
				gra_cursory = el_curwindowframe->revy - theEvent->where.v;
				us_state &= ~GOTXY;
			}
			gra_inputstate = ISBUTTON | BUTTONUP;
			us_state |= DIDINPUT;
			break;
		case mouseDown:
			switch (FindWindow(theEvent->where, &theWindow))
			{
				case inMenuBar:
					if (nature != 1) break;
					key = MenuSelect(theEvent->where);
					if (HiWord(key) == appleMENU && LoWord(key) != 1)
					{
						gra_applemenu(LoWord(key));
						return;
					}
					gra_highmenu = HiWord(key);
					gra_lowmenu = LoWord(key);
					gra_inputstate = MENUEVENT;
					return;
				case inSysWindow:
					SystemClick(theEvent, theWindow);
					break;
				case inContent:
					SetPort(theWindow);
					GlobalToLocal(&theEvent->where);

					/* determine which editor window was hit */
					for(wf = el_firstwindowframe; wf != NOWINDOWFRAME; wf = wf->nextwindowframe)
						if ((WindowPtr)wf->realwindow == theWindow) break;
					gra_lastclickedwindow = theWindow;
					gra_lastclickedwindowframe = wf;

					/* handle clicks in title bar of floating windows */
					if (wf != NOWINDOWFRAME && wf->floating != 0 && theEvent->where.v < FLOATINGHEADERSIZE)
					{
						if (theEvent->where.v >= 2 && theEvent->where.v <= 8 &&
							theEvent->where.h >= 4 && theEvent->where.h <= 10)
						{
							/* click in close box: turn off floating window */
							par[0] = "off";
							us_menu(1, par);
						} else
						{
							/* click in drag bar: move it */
							LocalToGlobal(&theEvent->where);
							gra_dragfloatingwindow(theWindow, theEvent->where);
						}
						break;
					}

					gra_frontnonfloatingwindow(&frontWindow);
					if (theWindow != frontWindow)
					{
						/* click in new window: see if it is allowed */
						if (nature == 0)
						{
							ttybeep();
							break;
						}

						/* select this one as the frontmost window */
						gra_selectoswindow(theWindow);
						gra_setcurrentwindowframe();

						/* when reentering edit window, force proper cursor */
						if (wf == NOWINDOWFRAME) break;
						oak = us_cursorstate;
						us_cursorstate++;
						setdefaultcursortype(oak);
					}

					/* special case to ensure that "el_curwindowframe" gets floating window */
					if (el_curwindowframe != NOWINDOWFRAME && el_curwindowframe->floating != 0)
						gra_setcurrentwindowframe();
					if (wf != NOWINDOWFRAME && wf->floating != 0) el_curwindowframe = wf;

					if (theWindow == gra_messageswindow && gra_messageswindow != 0)
					{
						cntlCode = FindControl(theEvent->where, theWindow, &theControl);
						if (cntlCode == kControlIndicatorPart)
						{
							oldpos = GetControlValue(theControl);
							TrackControl(theControl, theEvent->where, 0L);
							if (theControl == gra_vScroll) gra_adjustvtext(); else
								gra_adjusthtext(oldpos);
							break;
						}
						if (cntlCode == kControlUpButtonPart || cntlCode == kControlDownButtonPart ||
							cntlCode == kControlPageUpPart || cntlCode == kControlPageDownPart)
						{
							if (theControl == gra_vScroll)
								TrackControl(theControl, theEvent->where, gra_scrollvprocUPP); else
									TrackControl(theControl, theEvent->where, gra_scrollhprocUPP);
							break;
						}
						TEClick(theEvent->where, (theEvent->modifiers&shiftKey) != 0, gra_TEH);
						gra_fakemouseup();
						break;
					}

					/* ignore clicks in the messages area at the bottom */
					if (wf != NOWINDOWFRAME && wf->floating == 0 &&
						theEvent->where.v >= theWindow->portRect.bottom - SBARHEIGHT) break;

					/* create the "click" event */
					gra_inputstate = ISBUTTON;
					if ((theEvent->modifiers&shiftKey) != 0) gra_inputstate |= SHIFTDOWN;
					if ((theEvent->modifiers&cmdKey) != 0) gra_inputstate |= COMMANDDOWN;
					if ((theEvent->modifiers&optionKey) != 0) gra_inputstate |= OPTIONDOWN;
					if ((theEvent->modifiers&controlKey) != 0) gra_inputstate |= CONTROLDOWN;
					gra_cursorx = theEvent->where.h;
					if (wf == NOWINDOWFRAME) gra_cursory = theEvent->where.v; else
					{
						gra_cursory = wf->revy - theEvent->where.v;
						if (wf->floating != 0) gra_cursory += FLOATINGHEADERSIZE;
					}
					us_state &= ~GOTXY;
					if (theEvent->when - gra_doubleclick < gra_lastclick &&
						(gra_inputstate & (SHIFTDOWN|COMMANDDOWN|OPTIONDOWN|CONTROLDOWN)) == 0 &&
						abs(gra_cursorx-gra_lstcurx) < 5 && abs(gra_cursory-gra_lstcury) < 5)
					{
						gra_inputstate |= DOUBLECLICK;
						gra_lastclick = theEvent->when - gra_doubleclick - 1;
					} else gra_lastclick = theEvent->when;
					gra_lstcurx = gra_cursorx;   gra_lstcury = gra_cursory;
					us_state |= DIDINPUT;
					break;
				case inDrag:
					if (theWindow != FrontWindow() && nature == 0)
					{
						ttybeep();
						break;
					}
					gra_selectoswindow(theWindow);
					gra_setcurrentwindowframe();
					gra_dragfloatingwindow(theWindow, theEvent->where);
					break;
				case inGrow:
					SetPort(theWindow);
					SetRect(&r, 80, 80, qd.screenBits.bounds.right+500,
						qd.screenBits.bounds.bottom+500);
					fr = (*((WindowPeek)theWindow)->strucRgn)->rgnBBox;
					theResult = GrowWindow(theWindow, theEvent->where, &r);
					if (theResult != 0)
					{
						SizeWindow(theWindow, LoWord(theResult), HiWord(theResult), 1);
						gra_mygrowwindow(theWindow, LoWord(theResult), HiWord(theResult), &fr);
					}
					gra_inputstate = WINDOWCHANGE;
					break;
				case inZoomIn:
					if (TrackBox(theWindow, theEvent->where, inZoomIn) == 0) break;
					SetPort(theWindow);
					EraseRect(&theWindow->portRect);
					wst = (WStateData *) *(((WindowPeek)theWindow)->dataHandle);
					fr = (*((WindowPeek)theWindow)->strucRgn)->rgnBBox;
					MoveWindow(theWindow, wst->userState.left, wst->userState.top, 0);
					SizeWindow(theWindow, wst->userState.right-wst->userState.left,
						wst->userState.bottom-wst->userState.top, 1);
					gra_mygrowwindow(theWindow, wst->userState.right-wst->userState.left,
						wst->userState.bottom-wst->userState.top, &fr);
					gra_inputstate = WINDOWCHANGE;
					break;
				case inZoomOut:
					if (TrackBox(theWindow, theEvent->where, inZoomOut) == 0) break;
					SetPort(theWindow);
					EraseRect(&theWindow->portRect);
					wst = (WStateData *) *(((WindowPeek)theWindow)->dataHandle);
					fr = (*((WindowPeek)theWindow)->strucRgn)->rgnBBox;
					MoveWindow(theWindow, wst->stdState.left, wst->stdState.top, 0);
					SizeWindow(theWindow, wst->stdState.right-wst->stdState.left,
						wst->stdState.bottom-wst->stdState.top, 1);
					gra_mygrowwindow(theWindow, wst->stdState.right-wst->stdState.left,
						wst->stdState.bottom-wst->stdState.top, &fr);
					gra_inputstate = WINDOWCHANGE;
					break;
				case inGoAway:
					SetPort(theWindow);
					if (TrackGoAway(theWindow, theEvent->where) == 0) break;
					if (theWindow == gra_messageswindow && gra_messageswindow != 0)
					{
						gra_hidemessageswindow();
					} else if (el_curwindowframe != NOWINDOWFRAME &&
						theWindow == (WindowPtr)el_curwindowframe->realwindow)
					{
						par[0] = "delete";
						us_window(1, par);
					} else
					{
						/* determine which editor window was hit */
						for(wf = el_firstwindowframe; wf != NOWINDOWFRAME; wf = wf->nextwindowframe)
							if ((WindowPtr)wf->realwindow == theWindow) break;
						if (wf != NOWINDOWFRAME) gra_disposeoswindow(theWindow, wf->tkwin); else
							gra_disposeoswindow(theWindow, 0);
					}
					break;
			}
			break;

		case keyDown:
		case autoKey:
			switch ((theEvent->message&keyCodeMask) >> 8)
			{
				case 120: gra_inputstate = FUNCTIONF1;    break;
				case 122: gra_inputstate = FUNCTIONF2;    break;
				case  99: gra_inputstate = FUNCTIONF3;    break;
				case 118: gra_inputstate = FUNCTIONF4;    break;
				case  96: gra_inputstate = FUNCTIONF5;    break;
				case  97: gra_inputstate = FUNCTIONF6;    break;
				case  98: gra_inputstate = FUNCTIONF7;    break;
				case 100: gra_inputstate = FUNCTIONF8;    break;
				case 101: gra_inputstate = FUNCTIONF9;    break;
				case 109: gra_inputstate = FUNCTIONF10;   break;
				case 103: gra_inputstate = FUNCTIONF11;   break;
				case 111: gra_inputstate = FUNCTIONF12;   break;
				default:  gra_inputstate = NOEVENT;       break;
			}
			if (gra_inputstate != NOEVENT)
			{
				gra_inputstate = (gra_inputstate & ~0200) | COMMANDDOWN;
				SetPort((WindowPtr)el_curwindowframe->realwindow);
				GlobalToLocal(&theEvent->where);
				gra_cursorx = theEvent->where.h;
				gra_cursory = el_curwindowframe->revy - theEvent->where.v;
				us_state &= ~GOTXY;
				us_state |= DIDINPUT;
				break;
			}
			stroke = theEvent->message & charCodeMask;
			if (stroke == 0) break;
			if (el_curwindowframe != NOWINDOWFRAME)
			{
				SetPort((WindowPtr)el_curwindowframe->realwindow);
				GlobalToLocal(&theEvent->where);
				gra_cursorx = theEvent->where.h;
				gra_cursory = el_curwindowframe->revy - theEvent->where.v;
				us_state &= ~GOTXY;
			}
			if ((theEvent->modifiers & cmdKey) != 0 && stroke == '.')
			{
				el_pleasestop = 2;
				return;
			}
			if ((theEvent->modifiers & cmdKey) != 0 && nature == 1)
			{
				/* handle cut/copy/paste specially in dialogs */
				if (gra_handlingdialog != 0)
				{
					if (stroke == 'c')
					{
						DTextCopy();
						break;
					}
					if (stroke == 'x')
					{
						DTextCut();
						gra_inputstate = DELETEKEY & CHARREAD;
						us_state |= DIDINPUT;
						break;
					}
					if (stroke == 'v')
					{
						lastchar = DTextPaste();
						if (lastchar != 0)
						{
							gra_inputstate = lastchar & CHARREAD;
							us_state |= DIDINPUT;
						}
						break;
					}
					break;
				}
				
				key = MenuKey((char)stroke);
				gra_highmenu = HiWord(key);
				gra_lowmenu = -LoWord(key);
				gra_inputstate = MENUEVENT;
				if (gra_highmenu != 0) return;
				gra_inputstate = (stroke & CHARREAD) | COMMANDDOWN;
				us_state |= DIDINPUT;
				break;
			}
			gra_frontnonfloatingwindow(&frontWindow);
			if (gra_messageswindow == frontWindow && gra_messageswindow != 0)
			{
				TESetSelect(32767, 32767, gra_TEH);
				TEKey(stroke, gra_TEH);
				gra_showselect();
				break;
			}
			gra_inputstate = stroke & CHARREAD;
			us_state |= DIDINPUT;
			break;

		case activateEvt:
			/* only interested in activation, not deactivation */
			if ((theEvent->modifiers&activeFlag) == 0) break;

			/* ignore messages window */
			win = (WindowPtr)theEvent->message;
			if (win == gra_messageswindow && gra_messageswindow != 0) break;

			/* ignore editor windows */
			for(wf = el_firstwindowframe; wf != NOWINDOWFRAME; wf = wf->nextwindowframe)
				if ((WindowPtr)wf->realwindow == win) break;
			if (wf != NOWINDOWFRAME) break;

			/* probably a dialog, just process it */
			SetPort(win);
			InvalRect(&win->portRect);
			break;

		case updateEvt:
			win = (WindowPtr)theEvent->message;
			SetPort(win);
			BeginUpdate(win);
			if (win == gra_messageswindow && gra_messageswindow != 0)
			{
				EraseRect(&win->portRect);
				DrawControls(win);
				DrawGrowIcon(win);
				TEUpdate(&win->portRect, gra_TEH);
				EndUpdate(win);
				break;
			}

			for(wf = el_firstwindowframe; wf != NOWINDOWFRAME; wf = wf->nextwindowframe)
				if ((WindowPtr)wf->realwindow == win) break;
			if (wf != NOWINDOWFRAME)
			{
				if (firstwupdate == 0)
				{
					gra_drawosgraphics(wf);
					if (wf->floating == 0)
						gra_setrect(wf, (*wf->window->portPixMap)->bounds.right-SBARWIDTH-1,
							(*wf->window->portPixMap)->bounds.right-SBARWIDTH+1,
								0, (*wf->window->portPixMap)->bounds.bottom);
					r = (*wf->realwindow->visRgn)->rgnBBox;
					if (wf->floating != 0)
					{
						r.top -= FLOATINGHEADERSIZE;
						r.bottom -= FLOATINGHEADERSIZE;
					}
					gra_setrect(wf, r.left, r.right, r.top, r.bottom);

					if (wf->floating == 0) us_redostatus(wf);
				}
				firstwupdate = 0;
			}

			/* probably a dialog, just process it */
			SetPort(win);
			Dredrawdialogwindow();
			EndUpdate(win);
			break;

		case app3Evt:
			/* see if this happened in an editor window */
			findres = FindWindow(theEvent->where, &theWindow);
			if (theWindow == 0)
			{
				if ((theEvent->modifiers&btnState) == 0) break;
				wf = el_curwindowframe;
			} else
			{
				for(wf = el_firstwindowframe; wf != NOWINDOWFRAME; wf = wf->nextwindowframe)
					if ((WindowPtr)wf->realwindow == theWindow) break;
				SetPort(theWindow);
			}
			GlobalToLocal(&theEvent->where);

			/* report the menu if over one */
			inmenu = 0;
			if (wf != NOWINDOWFRAME && (theEvent->modifiers&btnState) == 0 && wf->floating != 0)
			{
				gra_cursorx = theEvent->where.h;
				gra_cursory = wf->revy - theEvent->where.v;
				us_state &= ~GOTXY;
				x = (gra_cursorx-us_menulx) / us_menuxsz;
				y = (gra_cursory-us_menuly) / us_menuysz;
				if (x >= 0 && y >= 0 && x < us_menux && y < us_menuy)
				{
					inmenu = 1;
					var = getvalkey((INTBIG)us_aid, VAID, VSTRING|VISARRAY, us_binding_menu);
					if (var != NOVARIABLE)
					{
						if (us_menupos <= 1) str = ((char **)var->addr)[y * us_menux + x]; else
							str = ((char **)var->addr)[x * us_menuy + y];
						us_parsebinding(str, &commandbinding);
						if (*commandbinding.command != 0)
						{
							if (commandbinding.nodeglyph != NONODEPROTO)
							{
								ttysetstatusfield(NOWINDOWFRAME, us_statusarc, describearcproto(us_curarcproto), 1);
								ttysetstatusfield(NOWINDOWFRAME, us_statusnode, us_describemenunode(&commandbinding), 1);
								overrodestatus = 1;
							}
							if (commandbinding.arcglyph != NOARCPROTO)
							{
								ttysetstatusfield(NOWINDOWFRAME, us_statusarc, describearcproto(commandbinding.arcglyph), 1);
								if (us_curnodeproto == NONODEPROTO) str = ""; else
									str = describenodeproto(us_curnodeproto);
								ttysetstatusfield(NOWINDOWFRAME, us_statusnode, str, 1);
								overrodestatus = 1;
							}
						}
						us_freebindingparse(&commandbinding);
					}
				}
			}
			if (inmenu == 0 && overrodestatus != 0)
			{
				ttysetstatusfield(NOWINDOWFRAME, us_statusarc, describearcproto(us_curarcproto), 1);
				if (us_curnodeproto == NONODEPROTO) str = ""; else
					str = describenodeproto(us_curnodeproto);
				ttysetstatusfield(NOWINDOWFRAME, us_statusnode, str, 1);
				overrodestatus = 0;
			}

			if (nature == 2)
			{
				/* handle cursor motion */
				gra_cursorx = theEvent->where.h;
				if (wf == NOWINDOWFRAME) gra_cursory = theEvent->where.v; else
					gra_cursory = wf->revy - theEvent->where.v;
				us_state &= ~GOTXY;
				gra_inputstate = MOTION;
				if ((theEvent->modifiers&btnState) != 0) gra_inputstate |= BUTTONUP;
				return;
			}

			/* checkout the cursor position */
			if (findres == inContent)
			{
				if (theWindow == gra_messageswindow && gra_messageswindow != 0)
				{
					if (theEvent->where.h > theWindow->portRect.right - SBARWIDTH ||
						theEvent->where.v > theWindow->portRect.bottom - SBARHEIGHT)
							setdefaultcursortype(NORMALCURSOR); else
								setdefaultcursortype(IBEAMCURSOR);
					return;
				}
				if (wf == NOWINDOWFRAME || theWindow != (WindowPtr)wf->realwindow)
				{
					setdefaultcursortype(NORMALCURSOR);
					return;
				}
				x = theEvent->where.h;
				if (wf == NOWINDOWFRAME) y = theEvent->where.v; else
					y = wf->revy - theEvent->where.v;
				for(w = el_topwindowpart; w != NOWINDOWPART; w = w->nextwindowpart)
				{
					if (w->frame != wf) continue;

					/* see if the cursor is over a window partition separator */
					us_gettruewindowbounds(w, &lx, &hx, &ly, &hy);
					if (x >= lx-1 && x <= lx+1 && y > ly+1 && y < hy-1 &&
						us_hasotherwindowpart(lx-10, y, w) != 0)
					{
						setdefaultcursortype(LRCURSOR);
						return;
					} else if (x >= hx-1 && x <= hx+1 && y > ly+1 && y < hy-1 &&
						us_hasotherwindowpart(hx+10, y, w) != 0)
					{
						setdefaultcursortype(LRCURSOR);
						return;
					} else if (y >= ly-1 && y <= ly+1 && x > lx+1 && x < hx-1 &&
						us_hasotherwindowpart(x, ly-10, w) != 0)
					{
						setdefaultcursortype(UDCURSOR);
						return;
					} else if (y >= hy-1 && y <= hy+1 && x > lx+1 && x < hx-1 &&
						us_hasotherwindowpart(x, hy+10, w) != 0)
					{
						setdefaultcursortype(UDCURSOR);
						return;
					}

					if (x < w->uselx || x > w->usehx || y < w->usely || y > w->usehy) continue;
					if ((w->state&WINDOWTYPE) == POPTEXTWINDOW ||
						(w->state&WINDOWTYPE) == TEXTWINDOW)
					{
						e = w->editor;
						if ((e->state&EDITORTYPE) == PACEDITOR)
						{
							if (x > w->usehx - SBARWIDTH || y < w->usely + SBARHEIGHT || y >= e->revy)
								setdefaultcursortype(NORMALCURSOR); else
									setdefaultcursortype(IBEAMCURSOR);
							return;
						}
					} else if ((us_aid->aidstate&SHOWXY) != 0)
					{
						xv = x;   yv = y;
						xv = muldiv(xv - w->uselx, w->screenhx - w->screenlx,
							w->usehx - w->uselx) + w->screenlx;
						yv = muldiv(yv - w->usely, w->screenhy - w->screenly,
							w->usehy - w->usely) + w->screenly;
						gridalign(&xv, &yv, us_alignment);
						us_setcursorpos(wf, xv, yv);
					}
				}
				setdefaultcursortype(us_normalcursor);
				return;
			}
			setdefaultcursortype(NORMALCURSOR);
			return;

		case kHighLevelEvent:
			(void)AEProcessAppleEvent(theEvent);
			break;

		case osEvt:
			switch ((theEvent->message >> 24) & 0xFF)
			{
				case suspendResumeMessage:
					if ((theEvent->message&resumeFlag) == 0)
					{
						/* suspend the application */
						gra_frontnonfloatingwindow(&theWindow);
						if (theWindow != 0) gra_activatewindow(theWindow, 0);
						gra_showallfloaters(0);
					} else
					{
						/* resume the application */
						gra_frontnonfloatingwindow(&theWindow);
						if (theWindow != 0) gra_activatewindow(theWindow, 1);
						gra_showallfloaters(1);
					}
					break;
			}
	}
#endif
}

/* handle interrupts */
void gra_onint(void)
{
	(void)signal(SIGINT, (SIGNALCAST)gra_onint);
	el_pleasestop = 1;
	ttyputerr(_("Interrupted..."));
}

/*************************** SESSION LOGGING ROUTINES ***************************/

/*
 * routine to begin playback of session logging file "file".  The routine
 * returns nonzero if there is an error.
 */
INTSML logplayback(char *file)
{
	REGISTER INTBIG comcount;
	char *filename;

	us_logplay = xopen(file, us_filetypelog, "", &filename);
	if (us_logplay == NULL) return(1);
	ttyputmsg(_("Type any key to playback the next step in the log file"));
	ttyputmsg(_("Type a number followed by 'x' to playback that many steps"));
	ttyputmsg(_("Type 'q' to terminate playback"));

	comcount = filesize(us_logplay) / (sizeof (gra_action));
	ttyputmsg(_("There are no more than %ld steps to playback"), comcount);
	gra_playbackmultiple = comcount;
	return(0);
}

/*
 * routine to create a session logging file
 */
void logstartrecord(void)
{
#if 0		/* no session logging on Macintosh yet */
	us_logrecord = xcreate(ELECTRICLOG, us_filetypelog, 0, 0);
#else
	us_logrecord = NULL;
#endif
}

/*
 * routine to terminate session logging
 */
void logfinishrecord(void)
{
	if (us_logrecord != NULL) xclose(us_logrecord);
	us_logrecord = NULL;
}

/****************************** MENUS ******************************/

void gra_initializemenus(void)
{
#ifdef USETK
	tcl_nativemenuinitialize();
#else
	gra_pulldownmenucount = 0;
	gra_appleMenu = GetMenu(appleMENU);
	if (gra_appleMenu == 0) error(_("Cannot find menu!"));
	InsertMenu(gra_appleMenu, 0);
	DrawMenuBar();
	AppendResMenu(gra_appleMenu, 'DRVR');
#endif
}

/*
 * routine to handle the Apple menu, including the "About Electric..." dialog
 */
void gra_applemenu(INTSML sindex)
{
	Str255 name;
	GrafPtr savePort;

	GetPort(&savePort);
	if (sindex == aboutMeCommand)
	{
		(void)us_aboutdlog();
	} else
	{
		GetMenuItemText(gra_appleMenu, sindex, name);
		(void)OpenDeskAcc(name);
	}
	SetPort(savePort);
}

void getacceleratorstrings(char **acceleratorstring, char **acceleratorprefix)
{
	*acceleratorstring = "Cmd";
	*acceleratorprefix = "";
}

char *getinterruptkey(void)
{
	return(_("Command-."));
}

INTSML nativepopupmenu(POPUPMENU *menu, INTSML header, INTSML left, INTSML top)
{
	INTBIG ret, len, i, j;
	char myline[256], attrib[256], *pt;
	MenuHandle thismenu;
	Point p;
	REGISTER USERCOM *uc;
	REGISTER POPUPMENUITEM *mi;

	SetPort(gra_lastclickedwindow);
	if (left < 0 && top < 0)
	{
		p.h = gra_cursorx;
		if (gra_lastclickedwindowframe == NOWINDOWFRAME) p.v = gra_cursory; else
			p.v = gra_lastclickedwindowframe->revy - gra_cursory;
	} else
	{
		p.h = left;   p.v = top;
		LocalToGlobal(&p);
	}

	strcpy(&myline[1], us_stripampersand(menu->header));
	myline[0] = strlen(&myline[1]);
	thismenu = NewMenu(2048, (unsigned char *)myline);
	if (thismenu == 0) return(-1);

	/* build the actual menu */
	for(i=0; i<menu->total; i++)
	{
		mi = &menu->list[i];

		/* quote illegal characters */
		pt = attrib;
		for(j=0; mi->attribute[j] != 0; j++)
		{
			if (mi->attribute[j] == '&') continue;
			if (mi->attribute[j] == '(') *pt++ = '{'; else
				if (mi->attribute[j] == ')') *pt++ = '}'; else
					*pt++ = mi->attribute[j];
		}
		*pt = 0;

		uc = mi->response;
		if (uc->active < 0)
		{
			(void)strcpy(myline, " (");
			if (*attrib == 0) (void)strcat(myline, "-"); else
				(void)strcat(myline, attrib);
			myline[0] = strlen(&myline[1]);
			AppendMenu(thismenu, (unsigned char *)myline);
			continue;
		}

		/* insert command title */
		pt = attrib;
		if (pt[0] == '>' && pt[1] == ' ')
		{
			myline[1] = '!';
			myline[2] = 022;
			(void)strcpy(&myline[3], &pt[2]);
			len = strlen(myline);
			if (myline[len-2] == ' ' && myline[len-1] == '<') myline[len-2] = 0;
		} else (void)strcpy(&myline[1], pt);
		myline[0] = strlen(&myline[1]);
		AppendMenu(thismenu, (unsigned char *)myline);
	}

	InsertMenu(thismenu, -1);
	ret = PopUpMenuSelect(thismenu, p.v, p.h, 1);
	DeleteMenu(2048);
	DisposeMenu(thismenu);
	if (HiWord(ret) == 0) return(-1);
	return(LoWord(ret)-1);
}

void gra_nativemenudoone(INTSML low, INTSML high)
{
	INTSML i, j;
	POPUPMENU *pm;

	i = high - USERMENUBASE;
	if (i >= 0 && i < gra_pulldownmenucount)
	{
		pm = gra_pulldowns[i];
		j = abs(low) - 1;
		if (j >= 0 && j < pm->total)
		{
			us_state |= DIDINPUT;
			us_state &= ~GOTXY;
			setdefaultcursortype(NULLCURSOR);
			us_forceeditchanges();
			us_execute(pm->list[j].response, us_aid->aidstate&ECHOBIND, 1, 1);
			db_setcurrenttool(us_aid);
			setactivity(pm->list[j].attribute);
		}
	}
	HiliteMenu(0);
}

/* routine to redraw entry "sindex" of popupmenu "pm" because it changed */
void nativemenurename(POPUPMENU *pm, INTSML sindex)
{
#ifdef USETK
	void tcl_nativemenurename(POPUPMENU*, INTSML);

	tcl_nativemenurename(pm, sindex);
#else
	INTSML i, submenuindex, cmdchar;
	char line[100], *pt;
	USERCOM *uc;

	for(i=0; i<gra_pulldownmenucount; i++)
		if (gra_pulldowns[i] == pm)
	{
		uc = pm->list[sindex].response;
		if (uc->active < 0)
		{
			if (gra_pulldownmenus[i] != 0)
				DisableItem(gra_pulldownmenus[i], sindex+1);
			if (*pm->list[i].attribute == 0) (void)strcpy(line, " -");
		} else
		{
			if (gra_pulldownmenus[i] != 0)
				EnableItem(gra_pulldownmenus[i], sindex+1);
		}

		(void)strcpy(&line[1], us_stripampersand(pm->list[sindex].attribute));
		line[0] = strlen(&line[1]);
		pt = line;
		if (pt[1] != '>' && pt[2] != ' ')
		{
			if (gra_pulldownmenus[i] != 0)
				CheckItem(gra_pulldownmenus[i], sindex+1, 0);
		} else
		{
			pt[2] = pt[0] - 4;
			pt += 2;
			if (gra_pulldownmenus[i] != 0)
				CheckItem(gra_pulldownmenus[i], sindex+1, 1);
		}
		cmdchar = 0;
		if (pt[pt[0]-1] == '/')
		{
			cmdchar = pt[pt[0]];
			pt[0] -= 2;
		}
		if (gra_pulldownmenus[i] != 0)
		{
			SetMenuItemText(gra_pulldownmenus[i], sindex+1, (unsigned char *)pt);
			SetItemCmd(gra_pulldownmenus[i], sindex+1, cmdchar);
		}

		/* see if this command is another menu */
		if (uc->menu != NOPOPUPMENU)
		{
			for(submenuindex=0; submenuindex<gra_pulldownmenucount; submenuindex++)
				if (gra_pulldowns[submenuindex] == uc->menu) break;
			if (submenuindex >= gra_pulldownmenucount) continue;
			if (gra_pulldownmenus[i] != 0)
			{
				SetItemCmd(gra_pulldownmenus[i], sindex+1, 0x1B);
				SetItemMark(gra_pulldownmenus[i], sindex+1, USERMENUBASE+submenuindex);
			}
			continue;
		}
		break;
	}
#endif
}

/*
 * Routine to establish the "count" pulldown menu names in "par" as the pulldown menu bar.
 * Returns nonzero on error.
 */
INTSML nativemenuload(INTSML count, char *par[])
{
#ifdef USETK
	WINDOWFRAME *wf;

	/* create the TK menubar */
	if (tcl_nativemenuload(count, par) != 0) return(1);
	gra_tkmenusloaded = 1;

	/* attach it to all windows */
	if (Tcl_VarEval(tcl_interp, Tk_PathName(gra_tkmessageswindow), " configure -menu .menus",
		0) == TCL_ERROR) ttyputerr(_("Tcl_VarEval returned %s"), tcl_interp->result);
	for(wf = el_firstwindowframe; wf != NOWINDOWFRAME; wf = wf->nextwindowframe)
		if (Tcl_VarEval(tcl_interp, Tk_PathName(wf->tkwin), " configure -menu .menus",
			0) == TCL_ERROR) ttyputerr(_("Tcl_VarEval returned %s"), tcl_interp->result);
	return(0);
#else
	REGISTER INTSML i, menuindex;
	REGISTER POPUPMENU *pm;
	POPUPMENU *pulls[25];

	/* build the pulldown menu bar */
	for(i=0; i<count; i++)
	{
		pm = us_getpopupmenu(par[i]);
		if (pm == NOPOPUPMENU) continue;
		pulls[i] = pm;
		menuindex = gra_pulldownindex(pm);
		if (menuindex < 0) continue;
		InsertMenu(gra_pulldownmenus[menuindex], 0);
	}
	DrawMenuBar();
	return(0);
#endif
}

/*
 * Routine to create a pulldown menu from popup menu "pm".
 * Returns an index to the table of pulldown menus (-1 on error).
 */
INTSML gra_pulldownindex(POPUPMENU *pm)
{
	REGISTER INTSML i, sindex;
	MenuHandle *newpulldownmenus;
	POPUPMENU **newpulldowns;

	/* see if it is in the list already */
	for(i=0; i<gra_pulldownmenucount; i++)
		if (gra_pulldowns[i] == pm) return(i);

	/* allocate new space with one more */
	newpulldownmenus = (MenuHandle *)emalloc((gra_pulldownmenucount+1) *
		(sizeof (MenuHandle)), us_aid->cluster);
	if (newpulldownmenus == 0) return(-1);
	newpulldowns = (POPUPMENU **)emalloc((gra_pulldownmenucount+1) *
		(sizeof (POPUPMENU *)), us_aid->cluster);
	if (newpulldowns == 0) return(-1);

	/* copy former arrays then delete them */
	for(i=0; i<gra_pulldownmenucount; i++)
	{
		newpulldownmenus[i] = gra_pulldownmenus[i];
		newpulldowns[i] = gra_pulldowns[i];
	}
	if (gra_pulldownmenucount != 0)
	{
		efree((char *)gra_pulldownmenus);
		efree((char *)gra_pulldowns);
	}
	gra_pulldownmenus = newpulldownmenus;
	gra_pulldowns = newpulldowns;

	sindex = gra_pulldownmenucount++;
	gra_pulldownmenus[sindex] = gra_makepdmenu(pm, USERMENUBASE+sindex);
	if (gra_pulldownmenus[sindex] == 0) return(-1);
	gra_pulldowns[sindex] = pm;
	return(sindex);
}

/*
 * Routine to create pulldown menu number "value" from the popup menu in "pm" and return
 * the menu handle.
 */
MenuHandle gra_makepdmenu(POPUPMENU *pm, INTSML value)
{
	REGISTER INTSML i, j, submenuindex;
	char myline[256], attrib[256], *pt;
	REGISTER USERCOM *uc;
	REGISTER POPUPMENUITEM *mi;
	MenuHandle thismenu;
	REGISTER INTSML len;
	char submenu[4];

	strcpy(&myline[1], us_stripampersand(pm->header));
	myline[0] = strlen(&myline[1]);
	thismenu = NewMenu(value, (unsigned char *)myline);
	if (thismenu == 0) return(0);

	/* build the actual menu */
	for(i=0; i<pm->total; i++)
	{
		mi = &pm->list[i];

		/* quote illegal characters */
		pt = attrib;
		for(j=0; mi->attribute[j] != 0; j++)
		{
			if (mi->attribute[j] == '&') continue;
			if (mi->attribute[j] == '(') *pt++ = '{'; else
				if (mi->attribute[j] == ')') *pt++ = '}'; else
					*pt++ = mi->attribute[j];
		}
		*pt = 0;

		uc = mi->response;
		if (uc->active < 0)
		{
			(void)strcpy(myline, " (");
			if (*attrib == 0) (void)strcat(myline, "-"); else
				(void)strcat(myline, attrib);
			myline[0] = strlen(&myline[1]);
			AppendMenu(thismenu, (unsigned char *)myline);
			continue;
		}

		/* see if this command is another menu */
		if (uc->menu != NOPOPUPMENU)
		{
			submenuindex = gra_pulldownindex(uc->menu);
			if (submenuindex < 0) continue;
			InsertMenu(gra_pulldownmenus[submenuindex], -1);
			myline[1] = '!';   myline[2] = USERMENUBASE+submenuindex;
			(void)strcpy(&myline[3], attrib);
			submenu[0] = '/';   submenu[1] = 0x1B;   submenu[2] = 0;
			(void)strcat(&myline[1], submenu);
			myline[0] = strlen(&myline[1]);
			AppendMenu(thismenu, (unsigned char *)myline);
			continue;
		}

		/* insert command title */
		pt = attrib;
		if (pt[0] == '>' && pt[1] == ' ')
		{
			myline[1] = '!';
			myline[2] = 022;
			(void)strcpy(&myline[3], &pt[2]);
			len = strlen(myline);
			if (myline[len-2] == ' ' && myline[len-1] == '<') myline[len-2] = 0;
		} else (void)strcpy(&myline[1], pt);
		myline[0] = strlen(&myline[1]);
		AppendMenu(thismenu, (unsigned char *)myline);
	}
	return(thismenu);
}

/****************************** DIALOGS ******************************/

/* the four scroller arrows */
#define	UPARROW      0
#define	DOWNARROW    1
#define	LEFTARROW    2
#define	RIGHTARROW   3

#define	THUMBSIZE      16		/* width of the thumb area in scroll slider */
#define	MAXSCROLLS      4		/* maximum scroll items in a dialog */
#define	MAXDIALOGS      2		/* maximum subdialogs on screen */
#define MAXMATCH       50
#define MINSCROLLTICKS  2		/* minimum ticks between scrollbar slider arrows */
#define MINPAGETICKS   20		/* minimum ticks between scrollbar slider page shifts */

/* the scroll arrow definition */
#define	ARROWLEN     7
static INTSML dia_arrowx[] = {4, 10, 10, 13, 7, 1, 4};
static INTSML dia_arrowy[] = {12, 12, 8, 8, 2, 8, 8};

typedef struct
{
	INTBIG  count;
	INTBIG  current;
	char **namelist;
} POPUPDATA;

typedef struct
{
	INTBIG   scrollitem;		/* item number of SCROLL area (-1 if none) */
	RECTAREA userrect;			/* position of SCROLL area */
	INTBIG   flags;				/* state SCROLL area */
	INTBIG   vthumbpos;			/* position of vertical thumb slider  */
	INTBIG   hthumbpos;			/* position of horizontal thumb slider */
	INTBIG   horizfactor;		/* shift of horizontal text (0 to 100) */
	INTBIG   firstline;			/* line number of top line */
	INTBIG   linesinfolder;		/* number of lines displayable */
	INTBIG   which;				/* currently highlighted line */
	INTBIG   lineheight;		/* height of line of text */
	INTBIG   lineoffset;		/* offset to baseline for text */
	char   **scrolllist;		/* list of text lines */
	INTBIG   scrolllistsize;	/* size of line list */
	INTBIG   scrolllistlen;		/* number of valid lines/list */
} DSCROLL;

typedef struct
{
	DIALOG   *dlgresaddr;		/* address of this dialog */
	short     onscreen;			/* nonzero if displayed */
	short     defaultbutton;	/* default button */
	WindowPtr theDialog;
#ifdef USETK
	Tk_Window tkwindow;
#else
	void     *tkwindow;
#endif

	/* for the scroll item */
	INTBIG     scrollcount;			/* number of scroll items */
	INTBIG     curscroll;			/* current scroll item */
	DSCROLL    scroll[MAXSCROLLS];	/* data structures for the scroll item(s) */

	/* for the current edit text item */
	INTBIG     curitem;				/* current edit item */
	INTBIG     editstart, editend;	/* start/end selected text in edit item */
	INTBIG     firstch;				/* first displayed character in edit item */
	INTBIG     opaqueitem;			/* item number of opaque edit text */
} DIALOCAL;

DIALOCAL   dia_it, dia_save[MAXDIALOGS];
INTBIG     dia_active = 0;
INTBIG     dia_savepos = 0;
INTBIG     dia_slineheight;			/* height of a line of scroll text */
INTBIG     dia_slineoffset;			/* scroll text: distance up to baseline */
INTBIG     dia_lineheight;			/* height of a line of other text */
INTBIG     dia_lineoffset;			/* other text: distance up to baseline */
INTBIG     dia_curlineoffset;		/* current distance up to baseline */
INTBIG     dia_knowndialogcount = 0;/* number of resource-read dialogs */
DIALOG   **dia_knowndialogs;		/* array of resource-read dialogs */
INTBIG    *dia_knowdialognumbers;	/* array of resource-read dialog numbers */
INTBIG     dia_firstupdate;
INTBIG     dia_wasin, dia_lbase, dia_hbase, dia_offset, dia_ddata;
RECTAREA   dia_rect;				/* the current rectangle being tracked */
char      *dia_msg;
POPUPDATA *dia_pd;					/* the current popup menu */
INTBIG     dia_cnt;					/* the current entry count in the popup menu */
INTBIG     dia_sta;					/* the current start point in the popup menu */
UINTBIG    dia_lastdiatime = 0;		/* time of last scrollbar slider button action */

/* prototypes for local routines */
INTSML Dbuttondown(INTBIG x, INTBIG y);
INTSML Dcheckdown(INTBIG x, INTBIG y);
void Dclear(void);
void Ddonedialogwindow(void);
void Ddoneedit(void);
INTSML Ddownarrow(INTBIG x, INTBIG y);
INTSML Ddownpage(INTBIG x, INTBIG y);
void Ddragwindow(INTBIG x, INTBIG y);
void Ddrawarrow(INTBIG sc, INTBIG which, INTBIG filled);
void Ddrawbox(INTBIG lx, INTBIG hx, INTBIG ly, INTBIG hy, GRAPHICS *which);
void Ddrawcircle(RECTAREA *r, INTBIG dim);
void Ddrawdisc(RECTAREA *r);
void Ddrawhorizslider(INTBIG sc);
void Ddrawitem(INTBIG type, RECTAREA *r, char *msg, INTBIG dim);
void Ddrawline(INTBIG xf, INTBIG yf, INTBIG xt, INTBIG yt);
void Ddrawmsg(INTBIG sc, char *msg, INTBIG which);
void Ddrawpolygon(INTBIG *xv, INTBIG *yv, INTBIG count, INTBIG filled);
void Ddrawpopupentries(void);
void Ddrawrectframe(RECTAREA *r, INTBIG on, INTBIG dim);
void Ddrawroundrectframe(RECTAREA *r, INTBIG arc, INTBIG dim);
void Ddrawtext(char *msg, INTBIG len, INTBIG x, INTBIG y, INTBIG dim);
void Ddrawvertslider(INTBIG sc);
void Deditbox(RECTAREA *r, INTBIG draw, INTBIG dim);
INTSML Deditdown(INTBIG x, INTBIG y);
void Dforcedialog(void);
INTBIG Dgeteditpos(RECTAREA *r, INTBIG x, INTBIG y, char *msg);
INTBIG Dgettextsize(char *msg, INTBIG len);
void Dgrayrect(RECTAREA *r);
INTBIG Dhandlepopup(RECTAREA *r, POPUPDATA *pd);
void Dhighlight(INTBIG on);
void Dhighlightrect(RECTAREA *r);
INTSML Dhscroll(INTBIG x, INTBIG y);
void Dinsertstr(char *insmsg);
void Dinsetrect(RECTAREA *r, INTBIG amt);
void Dintdrawrect(RECTAREA *rect, INTBIG r, INTBIG g, INTBIG b);
void Dinvertbox(INTBIG lx, INTBIG hx, INTBIG ly, INTBIG hy);
void Dinvertentry(INTBIG sc, INTBIG on);
void Dinvertrect(RECTAREA *r);
void Dinvertrectframe(RECTAREA *r);
void Dinvertroundrect(RECTAREA *r);
INTSML Dleftarrow(INTBIG x, INTBIG y);
INTSML Dleftpage(INTBIG x, INTBIG y);
void Dnewdialogwindow(RECTAREA *r, char *movable);
INTBIG Dneweditbase(void);
void Dputicon(INTBIG x, INTBIG y, char *data);
void Dredrawscroll(INTBIG sc);
INTSML Drightarrow(INTBIG x, INTBIG y);
INTSML Drightpage(INTBIG x, INTBIG y);
void Dsethscroll(INTBIG sc);
void Dsettextsmall(INTBIG sc);
void Dsetvscroll(INTBIG sc);
void Dshiftbits(RECTAREA *sr, RECTAREA *dr);
void Dstuffmessage(char *msg, RECTAREA *r, INTBIG dim);
void Dstufftext(char *msg, RECTAREA *r);
void Dtextlocation(char *msg, INTBIG len, RECTAREA *r, INTBIG *wid, INTBIG *line);
void Dtrackcursor(INTBIG, INTBIG, INTSML (*eachdown)(INTBIG, INTBIG));
INTSML Dtrackpopup(INTBIG x, INTBIG y);
INTSML Duparrow(INTBIG x, INTBIG y);
INTSML Duppage(INTBIG x, INTBIG y);
INTSML Dvscroll(INTBIG x, INTBIG y);
INTBIG Dwaitforaction(INTBIG *x, INTBIG *y, INTBIG *chr, UINTBIG *time);
INTSML Dwhichitem(INTBIG x, INTBIG y);

/*
 * Routine to initialize a dialog described by "dialog".
 * Returns nonzero if dialog cannot be initialized.
 */
INTSML DiaInitDialog(DIALOG *dialog)
{
	INTBIG itemtype, i;
	RECTAREA r;
	char *save, *line;
	POPUPDATA *pd;

	if (dialog->translated == 0)
		ttyputerr("Dialog not translated!");

	/* save the current dialog if this is a subdialog */
	if (dia_savepos > 0)
	{
		if (dia_savepos > MAXDIALOGS) return(1);
		dia_save[dia_savepos-1] = dia_it;
	}
	dia_savepos++;

	/* initialize dialog data structures */
	dia_it.defaultbutton = OK;
	dia_it.dlgresaddr = dialog;
	dia_it.curitem = -1;
	dia_it.opaqueitem = -1;
	for(i=0; i<MAXSCROLLS; i++)
	{
		dia_it.scroll[i].scrollitem = -1;
		dia_it.scroll[i].horizfactor = 0;
		dia_it.scroll[i].scrolllistlen = 0;
		dia_it.scroll[i].scrolllistsize = 0;
	}
	dia_it.curscroll = 0;
	dia_it.scrollcount = 0;
	dia_it.onscreen = 1;
	dia_it.firstch = 0;

	/* make the window */
	Dnewdialogwindow(&dialog->windowRect, dialog->movable);

	/* find the default button */
	for(i=0; i<dialog->items; i++)
	{
		itemtype = dia_it.dlgresaddr->list[i].type;
		if ((itemtype&ITEMTYPE) == DEFBUTTON)
			dia_it.defaultbutton = i+1;
	}

	/* loop through all of the dialog entries, drawing them */
	for(i=0; i<dialog->items; i++)
	{
		/* draw the item */
		itemtype = dia_it.dlgresaddr->list[i].type;
		line = dia_it.dlgresaddr->list[i].msg;
		r = dia_it.dlgresaddr->list[i].r;

		if ((itemtype&ITEMTYPE) == EDITTEXT)
		{
			if (dia_it.curitem == -1) dia_it.curitem = i;
		}
		if ((itemtype&ITEMTYPE) == SCROLL)
		{
			if (dia_it.scrollcount < MAXSCROLLS)
				dia_it.scroll[dia_it.scrollcount++].scrollitem = i;
		}
		if ((itemtype&ITEMTYPE) == MESSAGE || (itemtype&ITEMTYPE) == EDITTEXT)
		{
			save = (char *)emalloc(strlen(line)+1, el_tempcluster);
			if (save == 0) return(1);
			(void)strcpy(save, line);
			dia_it.dlgresaddr->list[i].data = (INTBIG)save;
		} else if ((itemtype&ITEMTYPE) == POPUP)
		{
			pd = (POPUPDATA *)emalloc(sizeof (POPUPDATA), el_tempcluster);
			if (pd == 0) return(1);
			pd->count = 0;
			dia_it.dlgresaddr->list[i].data = (INTBIG)pd;
			line = (char *)pd;
		} else dia_it.dlgresaddr->list[i].data = 0;

		Ddrawitem(itemtype, &r, line, 0);

		/* highlight the default button */
		if (i == dia_it.defaultbutton-1 &&
			(itemtype == BUTTON || itemtype == DEFBUTTON)) Dhighlightrect(&r);
	}
	if (dia_it.curitem >= 0)
	{
		dia_it.editstart = 0;
		dia_it.editend = strlen(dia_it.dlgresaddr->list[dia_it.curitem].msg);
		Dhighlight(1);
	}
	dia_active++;
	dia_firstupdate = 1;
	return(0);
}

/*
 * Routine to handle actions and return the next item hit.
 */
INTBIG DiaNextHit(void)
{
	INTBIG chr, itemHit;
	char ch[2];

	for(;;)
	{
		/* if interrupted, stop dialog */
		if (el_pleasestop != 0) return(CANCEL);

		/* get the next event, ignore fluff */
		chr = DiaGetNextCharacter(&itemHit);
		if (chr == -1) continue;

		/* non-character events return immediately */
		if (chr == -2) break;

		/* handle special character events */
		if (chr == ESCKEY) return(CANCEL);
		if (chr == '\n' || chr == '\r' || chr == CTRLCKEY) return(dia_it.defaultbutton);

		/* handle delete/backspace key */
		if ((chr == DELETEKEY || chr == BACKSPACEKEY) && dia_it.curitem >= 0)
		{
			if (dia_it.editstart == dia_it.editend && dia_it.editstart > 0)
				dia_it.editstart--;
			chr = 0;
		}

		ch[0] = chr;   ch[1] = 0;
		Dinsertstr(ch);
		break;
	}
	return(itemHit);
}

/*
 * Routine to parse the next input event and return the next character typed into
 * the current edit item.  If the routine returns -1, nothing has happened.  If the
 * routine returns -2, an item has been hit (and is in "itemHit").
 */
INTSML DiaGetNextCharacter(INTBIG *itemHit)
{
	RECTAREA r;
	char *msg;
	INTBIG thumbval;
	UINTBIG time;
	static char match[MAXMATCH];
	static INTBIG matchpos = 0;
	static UINTBIG lasttime = 0;
	INTBIG i, chr, type, which, v, t, n, x, y, sc, oak, newfirst;
	DSCROLL *scr;
	POPUPDATA *pd;

	oak = Dwaitforaction(&x, &y, &chr, &time);
	if (oak == 1 || oak == 5)
	{
		/* hit in the window: find the item */
		*itemHit = Dwhichitem(x, y);
		if (*itemHit == 0) return(-1);
		type = dia_it.dlgresaddr->list[*itemHit-1].type;
		r = dia_it.dlgresaddr->list[*itemHit-1].r;

		if ((type&ITEMTYPE) == MESSAGE || (type&INACTIVE) != 0) return(-1);

		/* if the item is a popup menu, display it and track */
		if ((type&ITEMTYPE) == POPUP)
		{
			pd = (POPUPDATA *)dia_it.dlgresaddr->list[*itemHit-1].data;
			i = Dhandlepopup(&r, pd);
			if (i == pd->current) return(-1);
			pd->current = i;
			Ddrawitem(POPUP, &dia_it.dlgresaddr->list[*itemHit-1].r, (char *)pd, 0);
			return(-2);
		}

		/* items under the user's control are given to the user */
		if (type == USERDRAWN) return(-2);

		/* if the item is edit text, make it the current one */
		if (type == EDITTEXT)
		{
			if (dia_it.curitem != *itemHit - 1) Ddoneedit(); else Dhighlight(0);
			dia_it.curitem = *itemHit - 1;
			msg = (char *)dia_it.dlgresaddr->list[*itemHit-1].data;
			i = Dgeteditpos(&r, x, y, msg);
			if (oak == 5)
			{
				/* look for a full word about position "base" */
				for(dia_it.editstart=i-1; dia_it.editstart>=0; dia_it.editstart--)
					if (!isalnum(msg[dia_it.editstart])) break;
				dia_it.editstart++;
				for(dia_it.editend = dia_it.editstart; msg[dia_it.editend] != 0; dia_it.editend++)
					if (!isalnum(msg[dia_it.editend])) break;
				Dhighlight(1);
			} else
			{
				dia_it.editstart = dia_it.editend = i;
				Dhighlight(1);
			}
			dia_lbase = dia_it.editstart;   dia_hbase = dia_it.editend;
			dia_rect = r;
			dia_msg = msg;
			Dtrackcursor(x, y, Deditdown);
			return(-2);
		}

		/* if the item is a button, reverse it and track */
		if (type == BUTTON || type == DEFBUTTON)
		{
			r.left++;   r.right--;
			r.top++;    r.bottom--;
			Dinvertroundrect(&r);
			dia_rect = r;
			dia_wasin = 1;
			Dtrackcursor(x, y, Dbuttondown);
			if (dia_wasin == 0) return(-1);
			Dinvertroundrect(&r);
			return(-2);
		}

		/* if the item is a check, outline it and track */
		if (type == CHECK)
		{
			dia_rect = r;
			r.right = r.left + 11;   r.left++;
			r.top = (r.top + r.bottom) / 2 - 5;
			r.bottom = r.top + 10;
			Ddrawrectframe(&r, 1, 0);
			dia_wasin = 1;
			dia_ddata = dia_it.dlgresaddr->list[*itemHit-1].data;
			Dtrackcursor(x, y, Dcheckdown);
			if (dia_wasin == 0) return(-1);
			Ddrawrectframe(&r, 0, 0);
			if (dia_ddata != 0)
			{
				Ddrawline(r.left, r.top, r.right-1, r.bottom-1);
				Ddrawline(r.left, r.bottom-1, r.right-1, r.top);
			}
			return(-2);
		}

		/* if the item is a scroll area, select a line */
		if (type == SCROLL)
		{
			for(sc=0; sc<dia_it.scrollcount; sc++)
				if (dia_it.scroll[sc].scrollitem == *itemHit - 1) break;
			if (sc >= dia_it.scrollcount) return(-1);
			scr = &dia_it.scroll[dia_it.curscroll=sc];

			if (x > scr->userrect.right)
			{
				/* cursor in vertical slider */
				if (scr->scrolllistlen <= scr->linesinfolder) return(-1);
				if (y > scr->userrect.bottom-16)
				{
					/* the down arrow */
					Ddrawarrow(sc, DOWNARROW, 1);
					Dtrackcursor(x, y, Ddownarrow);
					Ddrawarrow(sc, DOWNARROW, 0);
					return(-1);
				}
				if (y < scr->userrect.top+16)
				{
					/* the up arrow */
					Ddrawarrow(sc, UPARROW, 1);
					Dtrackcursor(x, y, Duparrow);
					Ddrawarrow(sc, UPARROW, 0);
					return(-1);
				}
				if (y > scr->vthumbpos+THUMBSIZE/2 && y <= scr->userrect.bottom-16)
				{
					/* scroll down one page */
					Dtrackcursor(x, y, Ddownpage);
					return(-1);
				}
				if (y < scr->vthumbpos-THUMBSIZE/2 && y >= scr->userrect.top+16)
				{
					/* scroll up one page */
					Dtrackcursor(x, y, Duppage);
					return(-1);
				}
				if (y >= scr->vthumbpos-THUMBSIZE/2 && y <= scr->vthumbpos+THUMBSIZE/2)
				{
					/* drag slider appropriately */
					v = y;   t = scr->vthumbpos;
					dia_rect = dia_it.dlgresaddr->list[*itemHit-1].r;
					dia_rect.left = dia_rect.right - 14;
					dia_rect.right--;
					dia_rect.top = scr->vthumbpos - THUMBSIZE/2;
					dia_rect.bottom = scr->vthumbpos + THUMBSIZE/2;
					Dinvertrectframe(&dia_rect);
					dia_offset = t-v;
					Dtrackcursor(x, y, Dvscroll);
					dia_rect.top = scr->vthumbpos - THUMBSIZE/2;
					dia_rect.bottom = scr->vthumbpos + THUMBSIZE/2;
					Dinvertrectframe(&dia_rect);
					r = scr->userrect;
					r.top += 16;   r.bottom -= 16;
					thumbval = scr->vthumbpos - r.top - THUMBSIZE/2;
					thumbval *= scr->scrolllistlen - scr->linesinfolder;
					thumbval /= r.bottom - r.top - THUMBSIZE;
					i = thumbval;
					if (i < 0) i = 0;
					if (i == scr->firstline) return(-1);
					scr->firstline = i;
					Dredrawscroll(sc);
					Dinvertentry(sc, 1);
					Dsetvscroll(sc);
					return(-1);
				}
			} else if (y > scr->userrect.bottom)
			{
				/* cursor in horizontal slider */
				if (x > scr->userrect.right-16)
				{
					/* the right arrow */
					Ddrawarrow(sc, RIGHTARROW, 1);
					Dtrackcursor(x, y, Drightarrow);
					Ddrawarrow(sc, RIGHTARROW, 0);
					return(-1);
				}
				if (x < scr->userrect.left+16)
				{
					/* the left arrow */
					Ddrawarrow(sc, LEFTARROW, 1);
					Dtrackcursor(x, y, Dleftarrow);
					Ddrawarrow(sc, LEFTARROW, 0);
					return(-1);
				}
				if (x > scr->hthumbpos+THUMBSIZE/2 && x <= scr->userrect.right-16)
				{
					/* scroll right one page */
					Dtrackcursor(x, y, Drightpage);
					return(-1);
				}
				if (x < scr->hthumbpos-THUMBSIZE/2 && x >= scr->userrect.left+16)
				{
					/* scroll left one page */
					Dtrackcursor(x, y, Dleftpage);
					return(-1);
				}
				if (x >= scr->hthumbpos-THUMBSIZE/2 && x <= scr->hthumbpos+THUMBSIZE/2)
				{
					/* drag slider appropriately */
					v = x;   t = scr->hthumbpos;
					dia_rect = dia_it.dlgresaddr->list[*itemHit-1].r;
					dia_rect.top = dia_rect.bottom - 14;
					dia_rect.bottom--;
					dia_rect.left = scr->hthumbpos - THUMBSIZE/2;
					dia_rect.right = scr->hthumbpos + THUMBSIZE/2;
					Dinvertrectframe(&dia_rect);
					dia_offset = t - v;
					Dtrackcursor(x, y, Dhscroll);
					dia_rect.left = scr->hthumbpos - THUMBSIZE/2;
					dia_rect.right = scr->hthumbpos + THUMBSIZE/2;
					Dinvertrectframe(&dia_rect);
					r = scr->userrect;
					r.left += 16;   r.right -= 16;
					thumbval = scr->hthumbpos - r.left - THUMBSIZE/2;
					thumbval *= 100;
					thumbval /= r.right - r.left - THUMBSIZE;
					i = thumbval;
					if (i < 0) i = 0;
					if (i == scr->horizfactor) return(-1);
					scr->horizfactor = i;
					Dredrawscroll(sc);
					Dinvertentry(sc, 1);
					Dsethscroll(sc);
					return(-1);
				}
			} else
			{
				/* double click in scroll selects and returns */
				if (oak == 5 && scr->scrolllistlen > 0 && (scr->flags&SCDOUBLEQUIT) != 0)
				{
					*itemHit = dia_it.defaultbutton;
					return(-2);
				}

				if ((scr->flags&SCSELMOUSE) != 0)
				{
					/* cursor in list: select an entry */
					which = y - r.top;
					which /= scr->lineheight;
					which += scr->firstline;
					if (which >= scr->scrolllistlen) which = scr->scrolllistlen - 1;
					DiaSelectLine(*itemHit, which);
					if ((scr->flags&SCREPORT) == 0) return(-1);
				}
			}
		}
		return(-2);
	}

	/* get the character, return immediately if special */
	if (oak != 0) return(-1);
	if (chr == ESCKEY || chr == '\n' || chr == '\r' || chr == CTRLCKEY) return(chr);

	/* handle arrow positioning */
	if (chr == LEFTARROWKEY)
	{
		if (dia_it.curitem < 0) return(-1);
		Dhighlight(0);
		dia_it.editstart--;
		if (dia_it.editstart < 0) dia_it.editstart = 0;
		dia_it.editend = dia_it.editstart;
		newfirst = Dneweditbase();
		if (dia_it.firstch != newfirst)
		{
			dia_it.firstch = newfirst;
			msg = (char *)dia_it.dlgresaddr->list[dia_it.curitem].data;
			Dstufftext(msg, &dia_it.dlgresaddr->list[dia_it.curitem].r);
		}
		Dhighlight(1);
		return(-1);
	}
	if (chr == RIGHTARROWKEY)
	{
		if (dia_it.curitem < 0) return(-1);
		Dhighlight(0);
		msg = (char *)dia_it.dlgresaddr->list[dia_it.curitem].data;
		if (msg[dia_it.editend] != 0) dia_it.editend++;
		dia_it.editstart = dia_it.editend;
		newfirst = Dneweditbase();
		if (dia_it.firstch != newfirst)
		{
			dia_it.firstch = newfirst;
			Dstufftext(msg, &dia_it.dlgresaddr->list[dia_it.curitem].r);
		}
		Dhighlight(1);
		return(-1);
	}

	/* tab to next edit text item */
	if (chr == TABKEY)
	{
		type = 0;
		for(i=dia_it.curitem+1; i<dia_it.dlgresaddr->items; i++)
		{
			type = dia_it.dlgresaddr->list[i].type;
			if (type == EDITTEXT) break;
		}
		if (type != EDITTEXT)
		{
			for(i=0; i<dia_it.curitem; i++)
			{
				type = dia_it.dlgresaddr->list[i].type;
				if (type == EDITTEXT) break;
			}
		}
		if (type != EDITTEXT) return(-1);
		Ddoneedit();
		dia_it.curitem = i;
		dia_it.editstart = 0;
		msg = (char *)dia_it.dlgresaddr->list[i].data;
		dia_it.editend = strlen(msg);
		dia_it.firstch = 0;
		Dhighlight(1);
		return(-1);
	}

	if (chr == DOWNARROWKEY)
	{
		if (dia_it.scrollcount <= 0) return(-1);
		scr = &dia_it.scroll[sc = dia_it.curscroll];
		if (scr->scrolllistlen <= 0) return(-1);
		if (scr->which >= scr->scrolllistlen-1) return(-1);
		*itemHit = scr->scrollitem + 1;
		DiaSelectLine(*itemHit, scr->which+1);
		if ((scr->flags&SCREPORT) == 0) return(-1);
		return(-2);
	}
	if (chr == UPARROWKEY)
	{
		if (dia_it.scrollcount <= 0) return(-1);
		scr = &dia_it.scroll[sc = dia_it.curscroll];
		if (scr->scrolllistlen <= 0) return(-1);
		if (scr->which <= 0) return(-1);
		*itemHit = scr->scrollitem + 1;
		DiaSelectLine(*itemHit, scr->which-1);
		if ((scr->flags&SCREPORT) == 0) return(-1);
		return(-2);
	}

	if (dia_it.curitem < 0 && dia_it.scrollcount > 0 &&
		(dia_it.scroll[dia_it.curscroll].flags&SCSELKEY) != 0)
	{
		/* use key to select line in scroll area */
		scr = &dia_it.scroll[sc=dia_it.curscroll];
		*itemHit = scr->scrollitem + 1;
		if (chr >= 'A' && chr <= 'Z') chr += 040;

		/* if it has been more than a second, reset the match string */
		if (time - lasttime > 60) matchpos = 0;
		lasttime = time;

		/* add this character to the match string */
		if (matchpos < MAXMATCH)
		{
			match[matchpos] = chr;
			matchpos++;
		}

		/* find that string */
		for(which = 0; which < scr->scrolllistlen; which++)
		{
			for(i=0; i<matchpos; i++)
			{
				n = scr->scrolllist[which][i];
				if (n >= 'A' && n <= 'Z') n += 040;
				if (match[i] != n) break;
			}
			if (i >= matchpos)
			{
				DiaSelectLine(*itemHit, which);
				break;
			}
		}
		if ((scr->flags&SCREPORT) == 0) return(-1);
		return(-2);
	}

	/* insert character into edit text */
	*itemHit = dia_it.curitem + 1;
	return(chr);
}

void DiaDoneDialog(void)
{
	INTBIG i, j, type;
	POPUPDATA *oldpd;

	/* free all the edit text and message buffers */
	for(i=0; i<dia_it.dlgresaddr->items; i++)
	{
		type = dia_it.dlgresaddr->list[i].type;
		if ((type&ITEMTYPE) == POPUP)
		{
			oldpd = (POPUPDATA *)dia_it.dlgresaddr->list[i].data;
			for(j=0; j<oldpd->count; j++) efree((char *)oldpd->namelist[j]);
			if (oldpd->count > 0) efree((char *)oldpd->namelist);
		}
		if ((type&ITEMTYPE) == MESSAGE || (type&ITEMTYPE) == EDITTEXT ||
			(type&ITEMTYPE) == POPUP)
				efree((char *)dia_it.dlgresaddr->list[i].data);
	}

	/* free all items in the scroll area */
	for(i=0; i<dia_it.scrollcount; i++)
	{
		for(j=0; j<dia_it.scroll[i].scrolllistlen; j++)
			efree((char *)dia_it.scroll[i].scrolllist[j]);
		if (dia_it.scroll[i].scrolllistsize > 0) efree((char *)dia_it.scroll[i].scrolllist);
	}
	if (dia_it.onscreen != 0)
	{
		dia_it.onscreen = 0;
		Ddonedialogwindow();
		dia_active--;
	}

	dia_savepos--;
	if (dia_savepos <= 0) return;
	dia_it = dia_save[dia_savepos-1];
	Dforcedialog();
	Dredrawdialogwindow();
}

/*
 * Routine to set the text in item "item" to "msg"
 */
void DiaSetText(INTBIG item, char *msg)
{
	INTBIG highlight, type, oldcur, dim;
	INTBIG amt;
	char *save, *pt;
	RECTAREA r;

	/* determine whether item is highlighted */
	highlight = 0;
	if (item < 0)
	{
		item = -item;
		highlight = 1;
	}
	item--;
	if (item < 0 || item >= dia_it.dlgresaddr->items) return;
	type = dia_it.dlgresaddr->list[item].type;

	/* special case when renaming buttons */
	if ((type&ITEMTYPE) == BUTTON || (type&ITEMTYPE) == DEFBUTTON ||
		(type&ITEMTYPE) == CHECK || (type&ITEMTYPE) == RADIO)
	{
		r = dia_it.dlgresaddr->list[item].r;
		if ((type&INACTIVE) != 0) dim = 1; else dim = 0;
		Dintdrawrect(&r, 255, 255, 255);
		Ddrawitem(type, &r, msg, dim);
		if ((type&ITEMTYPE) == RADIO && dia_it.dlgresaddr->list[item].data != 0)
		{
			/* draw the circle in a selected radio button */
			r.right = r.left + 12;
			r.top = (r.top + r.bottom) / 2 - 6;
			r.bottom = r.top + 12;
			Dinsetrect(&r, 3);
			Ddrawdisc(&r);
		}
		if ((type&ITEMTYPE) == CHECK && dia_it.dlgresaddr->list[item].data != 0)
		{
			/* draw the "X" in a selected check box */
			r.right = r.left + 12;
			r.top = (r.top + r.bottom) / 2 - 6;
			r.bottom = r.top + 12;
			Ddrawline(r.left, r.top, r.right-1, r.bottom-1);
			Ddrawline(r.left, r.bottom-1, r.right-1, r.top);
		}
		return;
	}

	/* convert copyright character sequence */
	for(pt = msg; *pt != 0; pt++) if (strncmp(pt, "(c)", 3) == 0)
	{
		(void)strcpy(pt, "\251");		/* "copyright" character */
		(void)strcpy(&pt[1], &pt[3]);
		break;
	}

	/* handle messages and edit text */
	oldcur = dia_it.curitem;   Ddoneedit();
	if ((type&ITEMTYPE) == MESSAGE || (type&ITEMTYPE) == EDITTEXT)
	{
		/* save the new string */
		amt = strlen(msg)+1;
		save = (char *)emalloc(amt, el_tempcluster);
		if (save == 0) return;
		(void)strcpy(save, msg);
		efree((char *)dia_it.dlgresaddr->list[item].data);
		dia_it.dlgresaddr->list[item].data = (INTBIG)save;

		/* redisplay the item */
		if ((type&ITEMTYPE) == MESSAGE)
			Dstuffmessage(msg, &dia_it.dlgresaddr->list[item].r, 0); else
				Dstufftext(msg, &dia_it.dlgresaddr->list[item].r);
		if ((type&ITEMTYPE) == EDITTEXT)
		{
			if (highlight != 0)
			{
				Ddoneedit();
				oldcur = item;
				dia_it.editstart = 0;
				dia_it.editend = strlen(msg);
				dia_it.firstch = 0;
			} else if (oldcur == item)
			{
				dia_it.editstart = dia_it.editend = strlen(msg);
			}
		}
	}
	dia_it.curitem = oldcur;
	Dhighlight(1);
}

/*
 * Routine to return the text in item "item"
 */
char *DiaGetText(INTBIG item)
{
	INTBIG type;
	POPUPDATA *pd;

	item--;
	if (item < 0 || item >= dia_it.dlgresaddr->items) return("");
	type = dia_it.dlgresaddr->list[item].type;
	if ((type&ITEMTYPE) == POPUP)
	{
		pd = (POPUPDATA *)dia_it.dlgresaddr->list[item].data;
		return(pd->namelist[pd->current]);
	}
	if ((type&ITEMTYPE) == MESSAGE || (type&ITEMTYPE) == EDITTEXT ||
		(type&ITEMTYPE) == BUTTON || (type&ITEMTYPE) == DEFBUTTON ||
		(type&ITEMTYPE) == CHECK || (type&ITEMTYPE) == RADIO)
			return((char *)dia_it.dlgresaddr->list[item].data);
	return(0);
}

/*
 * Routine to set the value in item "item" to "value"
 */
void DiaSetControl(INTBIG item, INTBIG value)
{
	INTBIG type;
	RECTAREA r;

	item--;
	if (item < 0 || item >= dia_it.dlgresaddr->items) return;
	type = dia_it.dlgresaddr->list[item].type;
	r = dia_it.dlgresaddr->list[item].r;
	if ((type&ITEMTYPE) == CHECK)
	{
		/* check box */
		r.right = r.left + 12;
		r.top = (r.top + r.bottom) / 2 - 6;
		r.bottom = r.top + 12;
		Dintdrawrect(&r, 255, 255, 255);
		Ddrawrectframe(&r, 1, 0);
		if (value != 0)
		{
			Ddrawline(r.left, r.top, r.right-1, r.bottom-1);
			Ddrawline(r.left, r.bottom-1, r.right-1, r.top);
		}
		dia_it.dlgresaddr->list[item].data = (INTBIG)value;
	} else if ((type&ITEMTYPE) == RADIO)
	{
		/* radio button */
		r.right = r.left + 12;
		r.top = (r.top + r.bottom) / 2 - 6;
		r.bottom = r.top + 12;
		Dintdrawrect(&r, 255, 255, 255);
		Ddrawcircle(&r, 0);
		if (value != 0)
		{
			Dinsetrect(&r, 3);
			Ddrawdisc(&r);
		}
		dia_it.dlgresaddr->list[item].data = (INTBIG)value;
	}
}

/*
 * Routine to return the value in item "item"
 */
INTBIG DiaGetControl(INTBIG item)
{
	INTBIG type;

	item--;
	if (item < 0 || item >= dia_it.dlgresaddr->items) return(0);
	type = dia_it.dlgresaddr->list[item].type;
	if ((type&ITEMTYPE) == CHECK || (type&ITEMTYPE) == RADIO)
		return(dia_it.dlgresaddr->list[item].data);
	return(0);
}

/*
 * Routine to check item "item" to make sure that there is
 * text in it.  If so, it returns nonzero.  Otherwise it beeps and returns zero.
 */
INTSML DiaValidEntry(INTBIG item)
{
	char *msg;

	msg = DiaGetText(item);
	while (*msg == ' ') msg++;
	if (*msg != 0) return(1);
	ttybeep();
	return(0);
}

/*
 * Routine to dim item "item"
 */
void DiaDimItem(INTBIG item)
{
	char *msg;
	INTBIG type;
	RECTAREA r;

	item--;
	if (item < 0 || item >= dia_it.dlgresaddr->items) return;
	if (item == dia_it.curitem) Ddoneedit();
	dia_it.dlgresaddr->list[item].type |= INACTIVE;
	type = dia_it.dlgresaddr->list[item].type;
	r = dia_it.dlgresaddr->list[item].r;
	msg = (char *)dia_it.dlgresaddr->list[item].data;
	if ((type&ITEMTYPE) == BUTTON || (type&ITEMTYPE) == DEFBUTTON ||
		(type&ITEMTYPE) == CHECK || (type&ITEMTYPE) == RADIO)
			msg = dia_it.dlgresaddr->list[item].msg;
	Ddrawitem(type, &r, msg, 1);
}

/*
 * Routine to un-dim item "item"
 */
void DiaUnDimItem(INTBIG item)
{
	char *msg;
	INTBIG type;
	RECTAREA r;

	item--;
	if (item < 0 || item >= dia_it.dlgresaddr->items) return;
	dia_it.dlgresaddr->list[item].type &= ~INACTIVE;
	type = dia_it.dlgresaddr->list[item].type;
	r = dia_it.dlgresaddr->list[item].r;
	msg = (char *)dia_it.dlgresaddr->list[item].data;
	if ((type&ITEMTYPE) == BUTTON || (type&ITEMTYPE) == DEFBUTTON ||
		(type&ITEMTYPE) == CHECK || (type&ITEMTYPE) == RADIO)
			msg = dia_it.dlgresaddr->list[item].msg;
	Ddrawitem(type, &r, msg, 0);

	/* if undimming selected radio button, redraw disc */
	if ((type&ITEMTYPE) == RADIO && dia_it.dlgresaddr->list[item].data != 0)
	{
		r.right = r.left + 9;
		r.left += 3;
		r.top = (r.top + r.bottom) / 2 - 3;
		r.bottom = r.top + 6;
		Ddrawdisc(&r);
	}
}

/*
 * Routine to change item "item" to be a message rather
 * than editable text
 */
void DiaNoEditControl(INTBIG item)
{
	item--;
	if (item < 0 || item >= dia_it.dlgresaddr->items) return;
	if (item == dia_it.curitem) Ddoneedit();
	dia_it.dlgresaddr->list[item].type = MESSAGE;
	Deditbox(&dia_it.dlgresaddr->list[item].r, 0, 0);
}

/*
 * Routine to change item "item" to be editable text rather
 * than a message
 */
void DiaEditControl(INTBIG item)
{
	item--;
	if (item < 0 || item >= dia_it.dlgresaddr->items) return;
	dia_it.dlgresaddr->list[item].type = EDITTEXT;
	Deditbox(&dia_it.dlgresaddr->list[item].r, 1, 0);
}

void DiaOpaqueEdit(INTBIG item)
{
	item--;
	if (item < 0 || item >= dia_it.dlgresaddr->items) return;
	if (dia_it.dlgresaddr->list[item].type != EDITTEXT) return;
	dia_it.opaqueitem = item;
}

/*
 * Routine to cause item "item" to report character hits
 */
void DiaCharacterEdit(INTBIG item)
{
}

/*
 * Routine to cause item "item" to be the default button
 */
void DiaDefaultButton(INTBIG item)
{
	INTSML olddefault;
	INTBIG itemtype, dim;
	RECTAREA r;
	char *line;

	if (item == dia_it.defaultbutton) return;
	olddefault = dia_it.defaultbutton - 1;
	dia_it.defaultbutton = item;

	/* redraw the old item without highlighting */
	itemtype = dia_it.dlgresaddr->list[olddefault].type;
	r = dia_it.dlgresaddr->list[olddefault].r;
	Dinsetrect(&r, -4);  r.right++;
	Dintdrawrect(&r, 255, 255, 255);
	Dinsetrect(&r, 4);   r.right--;
	line = dia_it.dlgresaddr->list[olddefault].msg;
	if ((itemtype&INACTIVE) != 0) dim = 1; else dim = 0;
	Ddrawitem(itemtype, &r, line, dim);

	/* highlight the new default button */
	r = dia_it.dlgresaddr->list[item-1].r;
	Dhighlightrect(&r);
}

/*
 * Routine to change item "item" into a popup with "count" entries
 * in "names".
 */
void DiaSetPopup(INTBIG item, INTBIG count, char **names)
{
	POPUPDATA *pd;
	INTBIG i, type;

	item--;
	if (item < 0 || item >= dia_it.dlgresaddr->items) return;
	type = dia_it.dlgresaddr->list[item].type;
	if ((type&ITEMTYPE) != POPUP) return;

	/* copy into a POPUPDATA structure */
	pd = (POPUPDATA *)dia_it.dlgresaddr->list[item].data;
	for(i=0; i<pd->count; i++) efree((char *)pd->namelist[i]);
	if (pd->count > 0) efree((char *)pd->namelist);
	pd->count = count;
	pd->current = 0;
	pd->namelist = (char **)emalloc(count * (sizeof (char *)), el_tempcluster);
	if (pd->namelist == 0) return;
	for(i=0; i<count; i++)
	{
		pd->namelist[i] = (char *)emalloc((strlen(names[i])+1) * (sizeof (char)),
			el_tempcluster);
		if (pd->namelist[i] == 0) return;
		(void)strcpy(pd->namelist[i], names[i]);
	}

	/* display the popup */
	Ddrawitem(POPUP, &dia_it.dlgresaddr->list[item].r, (char *)pd, 0);
}

/*
 * Routine to change popup item "item" so that the current entry is "entry".
 */
void DiaSetPopupEntry(INTBIG item, INTBIG entry)
{
	POPUPDATA *pd;
	INTBIG type;

	item--;
	if (item < 0 || item >= dia_it.dlgresaddr->items) return;
	type = dia_it.dlgresaddr->list[item].type;
	if ((type&ITEMTYPE) != POPUP) return;
	pd = (POPUPDATA *)dia_it.dlgresaddr->list[item].data;
	if (entry < 0 || entry >= pd->count) return;
	pd->current = entry;

	Ddrawitem(POPUP, &dia_it.dlgresaddr->list[item].r, (char *)pd,
		type&INACTIVE);
}

/*
 * Routine to return the current item in popup menu item "item".
 */
INTBIG DiaGetPopupEntry(INTBIG item)
{
	POPUPDATA *pd;

	item--;
	if (item < 0 || item >= dia_it.dlgresaddr->items) return(0);
	if ((dia_it.dlgresaddr->list[item].type&ITEMTYPE) != POPUP) return(0);
	pd = (POPUPDATA *)dia_it.dlgresaddr->list[item].data;
	return(pd->current);
}

void DiaInitTextDialog(INTBIG item, INTSML (*toplist)(char **), char *(*nextinlist)(void),
	void (*donelist)(void), INTBIG sortpos, INTBIG flags)
{
	DSCROLL *scr;
	INTBIG sc;

	item--;
	if (item < 0 || item >= dia_it.dlgresaddr->items) return;
	for(sc=0; sc<dia_it.scrollcount; sc++) if (dia_it.scroll[sc].scrollitem == item) break;
	if (sc >= dia_it.scrollcount) return;
	scr = &dia_it.scroll[sc];

	/* save information about this scroll area */
	scr->flags = flags;
	if ((scr->flags&SCSMALLFONT) != 0)
	{
		scr->lineheight = dia_slineheight;
		scr->lineoffset = dia_slineoffset;
	} else
	{
		scr->lineheight = dia_lineheight;
		scr->lineoffset = dia_lineoffset;
	}

	/* compute size of actual area (not including scroll bars) */
	scr->userrect = dia_it.dlgresaddr->list[item].r;
	scr->userrect.right -= 14;
	if ((scr->flags&SCHORIZBAR) != 0) scr->userrect.bottom -= 14;
	scr->linesinfolder = (scr->userrect.bottom - scr->userrect.top) / scr->lineheight;

	/* draw sliders */
	Ddrawvertslider(sc);
	if ((scr->flags&SCHORIZBAR) != 0) Ddrawhorizslider(sc);

	/* load the text */
	scr->scrolllistlen = 0;
	DiaLoadTextDialog(item+1, toplist, nextinlist, donelist, sortpos);
}

void DiaLoadTextDialog(INTBIG item, INTSML (*toplist)(char **), char *(*nextinlist)(void),
	void (*donelist)(void), INTBIG sortpos)
{
	INTBIG i, j, sorted, items, *order, sc;
	char *next, **list, line[256];
	DSCROLL *scr;

	item--;
	if (item < 0 || item >= dia_it.dlgresaddr->items) return;
	for(sc=0; sc<dia_it.scrollcount; sc++) if (dia_it.scroll[sc].scrollitem == item) break;
	if (sc >= dia_it.scrollcount) return;
	scr = &dia_it.scroll[sc];

	/* deallocate all former items in the list */
	for(i=0; i<scr->scrolllistlen; i++) efree((char *)scr->scrolllist[i]);
	scr->scrolllistlen = 0;
	scr->firstline = 0;

	/* count the number of items to be put in the text editor */
	line[0] = 0;
	next = line;
	(void)(*toplist)(&next);
	for(items=0; ; items++) if ((*nextinlist)() == 0) break;
	(*donelist)();

	/* allocate space for the strings */
	if (items > 0)
	{
		list = (char **)emalloc(items * (sizeof (char *)), el_tempcluster);
		if (list == 0) return;
		order = (INTBIG *)emalloc(items * SIZEOFINTBIG, el_tempcluster);
		if (order == 0) return;
	}

	/* get the list */
	line[0] = 0;
	next = line;
	(void)(*toplist)(&next);
	for(i=0; i<items; i++)
	{
		next = (*nextinlist)();
		if (next == 0) next = "???";
		list[i] = (char *)emalloc(strlen(next)+1, el_tempcluster);
		if (list[i] == 0) return;
		strcpy(list[i], next);
		order[i] = i;
	}
	(*donelist)();

	/* sort the list */
	if (sortpos >= 0)
	{
		sorted = 0;
		while (sorted == 0)
		{
			sorted = 1;
			for(i=1; i<items; i++)
			{
				if (namesame(&list[order[i-1]][sortpos], &list[order[i]][sortpos]) <= 0)
					continue;
				j = order[i];   order[i] = order[i-1];   order[i-1] = j;
				sorted = 0;
			}
		}
	}

	/* stuff the list into the text editor */
	scr->which = -1;
	Dinsetrect(&scr->userrect, 1);
	Dintdrawrect(&scr->userrect, 255, 255, 255);
	Dinsetrect(&scr->userrect, -1);
	for(i=0; i<items; i++) DiaStuffLine(item+1, list[order[i]]);
	Dsetvscroll(sc);
	if (scr->scrolllistlen > 0) DiaSelectLine(item+1, 0);

	/* deallocate the list */
	if (items > 0)
	{
		for(i=0; i<items; i++) efree((char *)list[i]);
		efree((char *)(char *)list);
		efree((char *)(char *)order);
	}
}

/*
 * Routine to stuff line "line" at the end of the edit buffer.
 */
void DiaStuffLine(INTBIG item, char *line)
{
	char **newlist, *pt;
	INTBIG i, sc;
	DSCROLL *scr;

	item--;
	if (item < 0 || item >= dia_it.dlgresaddr->items) return;
	for(sc=0; sc<dia_it.scrollcount; sc++) if (dia_it.scroll[sc].scrollitem == item) break;
	if (sc >= dia_it.scrollcount) return;
	scr = &dia_it.scroll[sc];

	if (scr->scrolllistlen >= scr->scrolllistsize)
	{
		newlist = (char **)emalloc((scr->scrolllistsize+10) * (sizeof (char *)),
			el_tempcluster);
		if (newlist == 0) return;
		for(i=0; i<scr->scrolllistlen; i++) newlist[i] = scr->scrolllist[i];
		if (scr->scrolllistsize != 0) efree((char *)scr->scrolllist);
		scr->scrolllist = newlist;
		scr->scrolllistsize += 10;
	}
	pt = (char *)emalloc(strlen(line)+1, el_tempcluster);
	if (pt == 0) return;
	(void)strcpy(pt, line);
	if (scr->scrolllistlen < scr->firstline+scr->linesinfolder)
		Ddrawmsg(sc, line, scr->scrolllistlen);
	scr->scrolllist[scr->scrolllistlen++] = pt;
}

void DiaSelectLine(INTBIG item, INTBIG l)
{
	INTBIG n, sc;
	DSCROLL *scr;

	item--;
	if (item < 0 || item >= dia_it.dlgresaddr->items) return;
	for(sc=0; sc<dia_it.scrollcount; sc++) if (dia_it.scroll[sc].scrollitem == item) break;
	if (sc >= dia_it.scrollcount) return;
	scr = &dia_it.scroll[sc];

	/* if the new line is visible, simply shift the highlighting */
	if (l-scr->firstline >= 0 && l-scr->firstline < scr->linesinfolder)
	{
		Dinvertentry(sc, 0);
	} else
	{
		/* must shift the buffer */
		n = l - scr->linesinfolder/2;
		if (n > scr->scrolllistlen-scr->linesinfolder)
			n = scr->scrolllistlen-scr->linesinfolder;
		if (n < 0) n = 0;
		scr->firstline = n;
		Dredrawscroll(sc);
	}
	scr->which = l;
	Dinvertentry(sc, 1);
	Dsetvscroll(sc);
}

INTBIG DiaGetCurLine(INTBIG item)
{
	INTBIG sc;

	item--;
	if (item < 0 || item >= dia_it.dlgresaddr->items) return(-1);
	for(sc=0; sc<dia_it.scrollcount; sc++)
		if (dia_it.scroll[sc].scrollitem == item) return(dia_it.scroll[sc].which);
	return(-1);
}

char *DiaGetScrollLine(INTBIG item, INTBIG l)
{
	INTBIG sc;

	item--;
	if (item < 0 || item >= dia_it.dlgresaddr->items) return("");
	for(sc=0; sc<dia_it.scrollcount; sc++)
		if (dia_it.scroll[sc].scrollitem == item)
	{
		if (l < 0 || l >= dia_it.scroll[sc].scrolllistlen) break;
		return(dia_it.scroll[sc].scrolllist[l]);
	}
	return("");
}

void DiaSetScrollLine(INTBIG item, INTBIG l, char *msg)
{
	char *ins;
	INTBIG sc;
	DSCROLL *scr;

	item--;
	if (item < 0 || item >= dia_it.dlgresaddr->items) return;
	for(sc=0; sc<dia_it.scrollcount; sc++) if (dia_it.scroll[sc].scrollitem == item) break;
	if (sc >= dia_it.scrollcount) return;
	scr = &dia_it.scroll[sc];

	if (l >= scr->scrolllistlen) DiaStuffLine(item+1, msg); else
	{
		ins = (char *)emalloc(strlen(msg)+1, el_tempcluster);
		if (ins == 0) return;
		(void)strcpy(ins, msg);
		efree((char *)scr->scrolllist[l]);
		scr->scrolllist[l] = ins;
	}
	Dredrawscroll(sc);
	Dinvertentry(sc, 1);
	Dsetvscroll(sc);
}

void DiaItemRect(INTBIG item, RECTAREA *rect)
{
	item--;
	if (item < 0 || item >= dia_it.dlgresaddr->items) return;
	*rect = dia_it.dlgresaddr->list[item].r;
	Dforcedialog();
}

void DiaPercent(INTBIG item, INTBIG p)
{
	RECTAREA r;
	INTBIG type;

	item--;
	if (item < 0 || item >= dia_it.dlgresaddr->items) return;
	type = dia_it.dlgresaddr->list[item].type;
	if ((type&ITEMTYPE) != PROGRESS) return;
	r = dia_it.dlgresaddr->list[item].r;
	if (p == 0)
	{
		Dintdrawrect(&r, 255, 255, 255);
		Ddrawrectframe(&r, 1, 0);
		return;
	}
	r.left++;
	r.right = r.left + (r.right-1-r.left)*p/100;
	r.top++;   r.bottom--;
	Dgrayrect(&r);
	flushscreen();
}

void DiaRedispRoutine(INTBIG item, void (*routine)(RECTAREA*))
{
	INTBIG type;

	item--;
	if (item < 0 || item >= dia_it.dlgresaddr->items) return;
	type = dia_it.dlgresaddr->list[item].type;
	if ((type&ITEMTYPE) != USERDRAWN) return;
	dia_it.dlgresaddr->list[item].data = (INTBIG)routine;
}

void DiaFrameRect(INTBIG item, RECTAREA *r)
{
	Dintdrawrect(r, 255, 255, 255);
	Ddrawrectframe(r, 1, 0);
}

void DiaDrawLine(INTBIG item, INTBIG fx, INTBIG fy, INTBIG tx, INTBIG ty, INTBIG mode)
{
	SetPort(dia_it.theDialog);
	switch (mode)
	{
		case DLMODEON:     PenMode(patCopy);  break;
		case DLMODEOFF:    PenMode(patBic);   break;
		case DLMODEINVERT: PenMode(patXor);   break;

	}
	MoveTo(fx, fy);
	LineTo(tx, ty);
	PenMode(patCopy);
}

void DiaFillPoly(INTBIG item, INTBIG *x, INTBIG *y, INTBIG count, INTBIG r, INTBIG g, INTBIG b)
{
	RGBColor color, oldcolor;

	color.red = r << 8;
	color.green = g << 8;
	color.blue = b << 8;
	SetPort(dia_it.theDialog);
	GetForeColor(&oldcolor);
	RGBForeColor(&color);
	Ddrawpolygon(x, y, count, 1);
	RGBForeColor(&oldcolor);
}

void DiaDrawRect(INTBIG item, RECTAREA *rect, INTBIG r, INTBIG g, INTBIG b)
{
	Dintdrawrect(rect, r, g, b);
}

void DiaPutText(INTBIG item, char *msg, INTBIG x, INTBIG y)
{
	FontInfo fontinfo;

	SetPort(dia_it.theDialog);
	TextFont(DSFONT);
	TextSize(10);
	GetFontInfo(&fontinfo);
	MoveTo(x, y + fontinfo.ascent);
	DrawText(msg, 0, strlen(msg));
}

void DiaGetTextInfo(char *msg, INTBIG *wid, INTBIG *hei)
{
	FontInfo fontinfo;

	SetPort(dia_it.theDialog);
	TextFont(DSFONT);
	TextSize(10);
	GetFontInfo(&fontinfo);
	*wid = TextWidth(msg, 0, strlen(msg));
	*hei = fontinfo.ascent + fontinfo.descent;
}

void DiaGetMouse(INTSML *x, INTSML *y)
{
	readtablet(x, y);
}

INTSML DiaNullDlogList(char **c) { return(0); }

char *DiaNullDlogItem(void) { return(0); }

void DiaNullDlogDone(void) {}

/****************************** DIALOG SUPPORT ******************************/

INTSML Dbuttondown(INTBIG x, INTBIG y)
{
	INTBIG in;

	if (x < dia_rect.left || x > dia_rect.right || y < dia_rect.top || y > dia_rect.bottom)
		in = 0; else
			in = 1;
	if (in != dia_wasin)
		Dinvertroundrect(&dia_rect);
	dia_wasin = in;
	return(0);
}

INTSML Dtrackpopup(INTBIG x, INTBIG y)
{
	INTBIG which;
	RECTAREA r;

	if (x < dia_rect.left || x > dia_rect.right || y < dia_rect.top || y > dia_rect.bottom)
		which = -1; else
	{
		which = (y - dia_rect.top - 1) / dia_lineheight;
		if (which >= dia_cnt-dia_sta) which = dia_cnt-dia_sta-1;

	}
	if (which != dia_wasin)
	{
		r.left = dia_rect.left + 1;
		r.right = dia_rect.right - 1;
		if (dia_wasin >= 0)
		{
			r.top = dia_rect.top + (dia_wasin * dia_lineheight) + 1;
			r.bottom = r.top + dia_lineheight;
			Dinvertrect(&r);
		}

		/* special case if in the "up" arrow */
		if (which == 0 && dia_sta != 0)
		{
			dia_sta--;   dia_cnt--;
			Ddrawpopupentries();
			dia_wasin = -1;
			return(0);
		}

		/* special case if in the "down" arrow */
		if (which == dia_cnt-dia_sta-1 && dia_cnt != dia_pd->count)
		{
			dia_sta++;   dia_cnt++;
			Ddrawpopupentries();
			dia_wasin = -1;
			return(0);
		}

		if (which >= 0)
		{
			r.top = dia_rect.top + (which * dia_lineheight) + 1;
			r.bottom = r.top + dia_lineheight;
			Dinvertrect(&r);
		}

		dia_wasin = which;
	}
	return(0);
}

INTSML Dcheckdown(INTBIG x, INTBIG y)
{
	INTBIG in;
	RECTAREA r;

	if (x < dia_rect.left || x > dia_rect.right || y < dia_rect.top || y > dia_rect.bottom)
		in = 0; else
			in = 1;
	if (in != dia_wasin)
	{
		r = dia_rect;
		r.right = r.left + 11;   r.left++;
		r.top = (r.top + r.bottom) / 2 - 5;
		r.bottom = r.top + 10;
		if (in != 0) Ddrawrectframe(&r, 1, 0); else
		{
			Ddrawrectframe(&r, 0, 0);
			if (dia_ddata != 0)
			{
				Ddrawline(r.left, r.top, r.right-1, r.bottom-1);
				Ddrawline(r.left, r.bottom-1, r.right-1, r.top);
			}
		}
	}
	dia_wasin = in;
	return(0);
}

INTSML Deditdown(INTBIG x, INTBIG y)
{
	INTBIG l, h, pos, wid, len, prevpos, basepos;
	char *msg;

	/* shift the text if the cursor moves outside the field */
	if (dia_rect.bottom-dia_rect.top < dia_lineheight*2)
	{
		if (y < dia_rect.top || y > dia_rect.bottom) return(0);

		/* scroll horizontally if there is only 1 line in the edit field */
		if (x < dia_rect.left && dia_it.firstch > 0)
		{
			Dhighlight(0);
			dia_it.firstch--;
			Dstufftext(dia_msg, &dia_rect);
			Dhighlight(1);
			gotosleep(10);
			return(0);
		}
		if (x > dia_rect.right && dia_it.firstch < dia_it.editend-1)
		{
			Dhighlight(0);
			dia_it.firstch++;
			Dstufftext(dia_msg, &dia_rect);
			Dhighlight(1);
			gotosleep(10);
			return(0);
		}
	} else
	{
		if (x < dia_rect.left || x > dia_rect.right) return(0);

		/* scroll vertically if there are multiple lines in the field */
		if (y < dia_rect.top && dia_it.firstch > 0)
		{
			msg = dia_msg;
			prevpos = 0;
			basepos = 0;
			while (*msg != 0)
			{
				for(len = strlen(msg); len>0; len--)
				{
					wid = Dgettextsize(msg, len);
					if (wid < dia_rect.right-dia_rect.left-2) break;
				}
				if (len == 0) break;
				basepos += len;
				if (basepos == dia_it.firstch) break;
				prevpos += len;
				msg += len;
			}
			Dhighlight(0);
			dia_it.firstch = prevpos;
			Dstufftext(dia_msg, &dia_rect);
			Dhighlight(1);
			gotosleep(30);
			return(0);
		}
		if (y > dia_rect.bottom && dia_it.firstch < dia_it.editend-1)
		{
			msg = &dia_msg[dia_it.firstch];
			for(len = strlen(msg); len>0; len--)
			{
				wid = Dgettextsize(msg, len);
				if (wid < dia_rect.right-dia_rect.left-2) break;
			}
			if (len == strlen(msg)) return(0);
			Dhighlight(0);
			dia_it.firstch += len;
			Dstufftext(dia_msg, &dia_rect);
			Dhighlight(1);
			gotosleep(30);
			return(0);
		}
	}

	pos = Dgeteditpos(&dia_rect, x, y, dia_msg);
	l = dia_lbase;   h = dia_hbase;
	if (pos > h) h = pos;
	if (pos < l) l = pos;
	if (l != dia_it.editstart || h != dia_it.editend)
	{
		Dhighlight(0);
		dia_it.editstart = l;   dia_it.editend = h;
		Dhighlight(1);
	}
	return(0);
}

INTSML Ddownarrow(INTBIG x, INTBIG y)
{
	short sc;
	DSCROLL *scr;
	short i;
	RECTAREA r, fr, tr;
	UINTBIG thisdiatime;

	/* limit the speed of the button */
	thisdiatime = ticktime();
	if (thisdiatime - dia_lastdiatime < MINSCROLLTICKS) return(0);
	dia_lastdiatime = thisdiatime;

	scr = &dia_it.scroll[sc=dia_it.curscroll];
	if (scr->scrolllistlen-scr->firstline <= scr->linesinfolder) return(1);
	scr->firstline++;

	/* shift the window contents up by 1 line (scr->lineheight) */
	r = scr->userrect;
	fr.left = r.left+1;                 fr.right = r.right-1;
	fr.top = r.top+1+scr->lineheight;   fr.bottom = r.bottom-1;
	tr.left = r.left+1;   tr.right = r.right-1;
	tr.top = r.top+1;     tr.bottom = r.bottom-1-scr->lineheight;
	Dshiftbits(&fr, &tr);

	/* fill in a new last line */
	fr.top = r.bottom-1-scr->lineheight;   fr.bottom = r.bottom-1;
	Dintdrawrect(&fr, 255, 255, 255);
	i = scr->firstline + scr->linesinfolder - 1;
	if (i < scr->scrolllistlen)
		Ddrawmsg(sc, scr->scrolllist[i], i-scr->firstline);

	/* invert it if selected */
	if (i == scr->which) Dinvertentry(sc, 1);

	/* redo thumb position */
	Dsetvscroll(sc);
	return(0);
}

INTSML Duparrow(INTBIG x, INTBIG y)
{
	INTBIG sc;
	DSCROLL *scr;
	RECTAREA r, fr, tr;
	UINTBIG thisdiatime;

	/* limit the speed of the button */
	thisdiatime = ticktime();
	if (thisdiatime - dia_lastdiatime < MINSCROLLTICKS) return(0);
	dia_lastdiatime = thisdiatime;

	scr = &dia_it.scroll[sc=dia_it.curscroll];
	if (scr->firstline <= 0) return(1);
	scr->firstline--;

	/* shift the window contents down by 1 line (scr->lineheight) */
	r = scr->userrect;
	fr.left = r.left+1;   fr.right = r.right-1;
	fr.top = r.top+1;     fr.bottom = r.bottom-1-scr->lineheight;

	tr.left = r.left+1;                 tr.right = r.right-1;
	tr.top = r.top+1+scr->lineheight;   tr.bottom = r.bottom-1;
	Dshiftbits(&fr, &tr);

	/* fill in a new top line */
	fr.top = r.top+1;   fr.bottom = r.top+1+scr->lineheight;
	Dintdrawrect(&fr, 255, 255, 255);
	Ddrawmsg(sc, scr->scrolllist[scr->firstline], 0);

	/* invert it if selected */
	if (scr->firstline == scr->which) Dinvertentry(sc, 1);

	/* redo thumb position */
	Dsetvscroll(sc);
	return(0);
}

INTSML Ddownpage(INTBIG x, INTBIG y)
{
	INTBIG sc;
	DSCROLL *scr;
	UINTBIG thisdiatime;

	/* limit the speed of the button */
	thisdiatime = ticktime();
	if (thisdiatime - dia_lastdiatime < MINPAGETICKS) return(0);
	dia_lastdiatime = thisdiatime;

	scr = &dia_it.scroll[sc=dia_it.curscroll];
	if (y <= scr->vthumbpos+THUMBSIZE/2 || y > scr->userrect.bottom-16) return(1);
	if (scr->scrolllistlen-scr->firstline <= scr->linesinfolder) return(1);
	scr->firstline += scr->linesinfolder-1;
	if (scr->scrolllistlen-scr->firstline <= scr->linesinfolder)
		scr->firstline = scr->scrolllistlen-scr->linesinfolder;
	Dredrawscroll(sc);
	Dinvertentry(sc, 1);
	Dsetvscroll(sc);
	return(0);
}

INTSML Duppage(INTBIG x, INTBIG y)
{
	INTBIG sc;
	DSCROLL *scr;
	UINTBIG thisdiatime;

	/* limit the speed of the button */
	thisdiatime = ticktime();
	if (thisdiatime - dia_lastdiatime < MINPAGETICKS) return(0);
	dia_lastdiatime = thisdiatime;

	scr = &dia_it.scroll[sc=dia_it.curscroll];
	if (y >= scr->vthumbpos-THUMBSIZE/2 || y < scr->userrect.top+16) return(1);
	if (scr->firstline <= 0) return(1);
	scr->firstline -= scr->linesinfolder-1;
	if (scr->firstline < 0) scr->firstline = 0;
	Dredrawscroll(sc);
	Dinvertentry(sc, 1);
	Dsetvscroll(sc);
	return(0);
}

INTSML Dvscroll(INTBIG x, INTBIG y)
{
	INTBIG l, sc;
	DSCROLL *scr;

	scr = &dia_it.scroll[sc=dia_it.curscroll];
	l = scr->vthumbpos;
	scr->vthumbpos = y + dia_offset;
	if (scr->vthumbpos < scr->userrect.top+16+THUMBSIZE/2)
		scr->vthumbpos = scr->userrect.top+16+THUMBSIZE/2;
	if (scr->vthumbpos > scr->userrect.bottom-16-THUMBSIZE/2)
		scr->vthumbpos = scr->userrect.bottom-16-THUMBSIZE/2;
	if (scr->vthumbpos == l) return(0);
	dia_rect.top = l - THUMBSIZE/2;
	dia_rect.bottom = l + THUMBSIZE/2;
	Dinvertrectframe(&dia_rect);
	dia_rect.top = scr->vthumbpos - THUMBSIZE/2;
	dia_rect.bottom = scr->vthumbpos + THUMBSIZE/2;
	Dinvertrectframe(&dia_rect);
	return(0);
}

INTSML Drightarrow(INTBIG x, INTBIG y)
{
	INTBIG sc;
	DSCROLL *scr;
	UINTBIG thisdiatime;

	/* limit the speed of the button */
	thisdiatime = ticktime();
	if (thisdiatime - dia_lastdiatime < MINSCROLLTICKS) return(0);
	dia_lastdiatime = thisdiatime;

	scr = &dia_it.scroll[sc=dia_it.curscroll];
	if (scr->horizfactor >= 100) return(1);
	scr->horizfactor++;
	Dredrawscroll(sc);
	Dinvertentry(sc, 1);
	Dsethscroll(sc);
	return(0);
}

INTSML Dleftarrow(INTBIG x, INTBIG y)
{
	INTBIG sc;
	DSCROLL *scr;
	UINTBIG thisdiatime;

	/* limit the speed of the button */
	thisdiatime = ticktime();
	if (thisdiatime - dia_lastdiatime < MINSCROLLTICKS) return(0);
	dia_lastdiatime = thisdiatime;

	scr = &dia_it.scroll[sc=dia_it.curscroll];
	if (scr->horizfactor <= 0) return(1);
	scr->horizfactor--;
	Dredrawscroll(sc);
	Dinvertentry(sc, 1);
	Dsethscroll(sc);
	return(0);
}

INTSML Drightpage(INTBIG x, INTBIG y)
{
	INTBIG sc;
	DSCROLL *scr;
	UINTBIG thisdiatime;

	/* limit the speed of the button */
	thisdiatime = ticktime();
	if (thisdiatime - dia_lastdiatime < MINPAGETICKS) return(0);
	dia_lastdiatime = thisdiatime;

	scr = &dia_it.scroll[sc=dia_it.curscroll];
	if (x <= scr->hthumbpos+THUMBSIZE/2 || x > scr->userrect.right-16) return(1);
	if (scr->horizfactor >= 100) return(1);
	scr->horizfactor += 10;
	if (scr->horizfactor >= 100) scr->horizfactor = 100;
	Dredrawscroll(sc);
	Dinvertentry(sc, 1);
	Dsethscroll(sc);
	return(0);
}

INTSML Dleftpage(INTBIG x, INTBIG y)
{
	INTBIG sc;
	DSCROLL *scr;
	UINTBIG thisdiatime;

	/* limit the speed of the button */
	thisdiatime = ticktime();
	if (thisdiatime - dia_lastdiatime < MINPAGETICKS) return(0);
	dia_lastdiatime = thisdiatime;

	scr = &dia_it.scroll[sc=dia_it.curscroll];
	if (x >= scr->hthumbpos-THUMBSIZE/2 || x < scr->userrect.left+16) return(1);
	if (scr->horizfactor <= 0) return(1);
	scr->horizfactor -= 10;
	if (scr->horizfactor <= 0) scr->horizfactor = 0;
	Dredrawscroll(sc);
	Dinvertentry(sc, 1);
	Dsethscroll(sc);
	return(0);
}

INTSML Dhscroll(INTBIG x, INTBIG y)
{
	INTBIG l, sc;
	DSCROLL *scr;

	scr = &dia_it.scroll[sc=dia_it.curscroll];
	l = scr->hthumbpos;
	scr->hthumbpos = x + dia_offset;
	if (scr->hthumbpos < scr->userrect.left+16+THUMBSIZE/2)
		scr->hthumbpos = scr->userrect.left+16+THUMBSIZE/2;
	if (scr->hthumbpos > scr->userrect.right-16-THUMBSIZE/2)
		scr->hthumbpos = scr->userrect.right-16-THUMBSIZE/2;
	if (scr->hthumbpos == l) return(0);
	dia_rect.left = l - THUMBSIZE/2;
	dia_rect.right = l + THUMBSIZE/2;
	Dinvertrectframe(&dia_rect);
	dia_rect.left = scr->hthumbpos - THUMBSIZE/2;
	dia_rect.right = scr->hthumbpos + THUMBSIZE/2;
	Dinvertrectframe(&dia_rect);
	return(0);
}

void Dredrawscroll(INTBIG sc)
{
	RECTAREA r;
	INTBIG i;
	DSCROLL *scr;

	scr = &dia_it.scroll[sc];
	r = scr->userrect;
	r.left++;        r.right--;
	r.top++;         r.bottom--;
	Dintdrawrect(&r, 255, 255, 255);
	for(i=scr->firstline; i<scr->scrolllistlen; i++)
	{
		if (i-scr->firstline >= scr->linesinfolder) break;
		Ddrawmsg(sc, scr->scrolllist[i], i-scr->firstline);
	}
}

/*
 * Routine to set the vertical scroll bar
 */
void Dsetvscroll(INTBIG sc)
{
	RECTAREA r;
	INTBIG f;
	DSCROLL *scr;

	/* first redraw the border */
	Ddrawvertslider(sc);

	/* get the area of the slider without arrows */
	scr = &dia_it.scroll[sc];
	r = scr->userrect;
	r.top += 16;
	r.bottom -= 16;
	r.left = r.right;
	r.right += 13;

	/* if there is nothing to scroll, clear this area */
	if (scr->scrolllistlen <= scr->linesinfolder)
	{
		Dintdrawrect(&r, 255, 255, 255);
		return;
	}

	/* gray the scroll area */
	Dgrayrect(&r);

	/* compute position of vertical thumb area */
	f = scr->firstline;   f *= r.bottom - r.top-THUMBSIZE;
	f /= scr->scrolllistlen - scr->linesinfolder;
	scr->vthumbpos = r.top + THUMBSIZE/2 + f;
	if (scr->vthumbpos > r.bottom-THUMBSIZE/2) scr->vthumbpos = r.bottom-THUMBSIZE/2;

	/* draw the thumb */
	r.top = scr->vthumbpos - THUMBSIZE/2;
	r.bottom = scr->vthumbpos + THUMBSIZE/2;
	Dintdrawrect(&r, 255, 255, 255);
	Ddrawrectframe(&r, 1, 0);
}

/*
 * Routine to set the horizontal scroll bar
 */
void Dsethscroll(INTBIG sc)
{
	RECTAREA r;
	INTBIG f;
	DSCROLL *scr;

	/* get the area of the slider without arrows */
	scr = &dia_it.scroll[sc];
	r = scr->userrect;
	r.left += 16;
	r.right -= 16;
	r.top = r.bottom;
	r.bottom += 13;

	/* gray the scroll area */
	Dgrayrect(&r);

	/* compute position of vertical thumb area */
	f = scr->horizfactor;   f *= (INTBIG)(r.right-r.left-THUMBSIZE);   f /= 100;
	scr->hthumbpos = r.left + THUMBSIZE/2 + f;
	if (scr->hthumbpos > r.right-THUMBSIZE/2) scr->hthumbpos = r.right-THUMBSIZE/2;

	/* draw the thumb */
	r.left = scr->hthumbpos - THUMBSIZE/2;
	r.right = scr->hthumbpos + THUMBSIZE/2;
	Dintdrawrect(&r, 255, 255, 255);
	Ddrawrectframe(&r, 1, 0);
}

void Dinvertentry(INTBIG sc, INTBIG on)
{
	RECTAREA r;
	DSCROLL *scr;

	scr = &dia_it.scroll[sc];
	if (scr->which-scr->firstline >= 0 && scr->which-scr->firstline < scr->linesinfolder)
	{
		r.left = scr->userrect.left+1;
		r.right = scr->userrect.right-1;
		r.top = scr->userrect.top + scr->lineheight*(scr->which-scr->firstline)+1;
		r.bottom = r.top + scr->lineheight;
		if (r.bottom >= scr->userrect.bottom) r.bottom = scr->userrect.bottom-1;
		Dinvertrect(&r);
	}
}

/*
 * routine to determine which item falls under the coordinates (x, y)
 */
INTSML Dwhichitem(INTBIG x, INTBIG y)
{
	INTBIG i;
	RECTAREA r;

	for(i=0; i<dia_it.dlgresaddr->items; i++)
	{
		r = dia_it.dlgresaddr->list[i].r;
		if (x >= r.left && x <= r.right && y >= r.top && y <= r.bottom) return(i+1);
	}
	return(0);
}

void Dinsertstr(char *insmsg)
{
	INTBIG i, j, newcurspos;
	char *oldmsg, *newmsg, *pt;

	if (dia_it.curitem < 0) return;
	Dhighlight(0);
	oldmsg = (char *)dia_it.dlgresaddr->list[dia_it.curitem].data;

	/* allocate space for the new message and fill it */
	newmsg = (char *)emalloc(strlen(oldmsg)+strlen(insmsg)+dia_it.editend-dia_it.editstart+1, el_tempcluster);
	if (newmsg == 0) return;
	j = 0;
	for(i=0; i<dia_it.editstart; i++) newmsg[j++] = oldmsg[i];
	for(i=0; insmsg[i] != 0; i++) newmsg[j++] = insmsg[i];
	newcurspos = j;
	for(i=dia_it.editend; oldmsg[i] != 0; i++) newmsg[j++] = oldmsg[i];
	newmsg[j] = 0;
	dia_it.editstart = dia_it.editend = newcurspos;

	/* replace the message */
	efree((char *)dia_it.dlgresaddr->list[dia_it.curitem].data);
	dia_it.dlgresaddr->list[dia_it.curitem].data = (INTBIG)newmsg;

	/* make sure the cursor is visible */
	dia_it.firstch = Dneweditbase();
	if (dia_it.opaqueitem == dia_it.curitem)
	{
		allocstring(&oldmsg, newmsg, el_tempcluster);
		for(pt = oldmsg; *pt != 0; pt++)
			*pt = '\245';		/* bullet */
		Dstufftext(oldmsg, &dia_it.dlgresaddr->list[dia_it.curitem].r);
		efree(oldmsg);
	} else
	{
		Dstufftext(newmsg, &dia_it.dlgresaddr->list[dia_it.curitem].r);
	}

	/* set new highlighting */
	Dhighlight(1);
}

INTBIG Dneweditbase(void)
{
	INTBIG wid, prevpos, basepos, y, firstoff, len, newfirst;
	char *newmsg, *msg;
	RECTAREA r;

	newfirst = dia_it.firstch;
	newmsg = (char *)dia_it.dlgresaddr->list[dia_it.curitem].data;
	r = dia_it.dlgresaddr->list[dia_it.curitem].r;
	if (r.bottom-r.top < dia_lineheight*2)
	{
		/* if there is only 1 line in the edit field, shift horizontally */
		if (dia_it.editend < newfirst)
		{
			newfirst = dia_it.editend-1;
			if (newfirst < 0) newfirst = 0;
		}
		while (dia_it.editend > newfirst)
		{
			wid = Dgettextsize(&newmsg[newfirst], dia_it.editend-newfirst);
			if (wid <= r.right-r.left-2) break;
			newfirst++;
		}
	} else
	{
		/* multiple lines in the edit field, shift vertically */
		while (dia_it.editend < newfirst)
		{
			msg = newmsg;
			prevpos = 0;
			basepos = 0;
			while (*msg != 0)
			{
				for(len = strlen(msg); len>0; len--)
				{
					wid = Dgettextsize(msg, len);
					if (wid < r.right-r.left-2) break;
				}
				if (len == 0) break;
				basepos += len;
				if (basepos == newfirst) break;
				prevpos += len;
				msg += len;
			}
			newfirst = prevpos;
		}
		if (dia_it.editend > newfirst)
		{
			y = r.top + dia_lineheight;
			msg = &newmsg[newfirst];
			firstoff = 0;
			basepos = 0;
			while (*msg != 0)
			{
				for(len = strlen(msg); len>0; len--)
				{
					wid = Dgettextsize(msg, len);
					if (wid < r.right-r.left-2) break;
				}
				if (len == 0) break;
				if (dia_it.editend <= newfirst + basepos + len) break;
				if (firstoff == 0) firstoff = len;
				msg += len;
				basepos += len;
				y += dia_lineheight;
				if (y > r.bottom) { newfirst += firstoff;   break; }
			}
		}
	}
	return(newfirst);
}

void DTextCopy(void)
{
	char *msg;

	if (dia_it.curitem < 0) return;
	msg = (char *)dia_it.dlgresaddr->list[dia_it.curitem].data;
	setcutbuffer(msg);
}

void DTextCut(void)
{
	char *msg;

	if (dia_it.curitem < 0) return;
	msg = (char *)dia_it.dlgresaddr->list[dia_it.curitem].data;
	setcutbuffer(msg);
	DiaSetText(dia_it.curitem+1, "");
}

char DTextPaste(void)
{
	char *msg, lastchar;
	INTBIG len;

	if (dia_it.curitem < 0) return(0);
	msg = getcutbuffer();
	len = strlen(msg);
	if (len <= 0) lastchar = 0; else
	{
		lastchar = msg[len-1];
		msg[len-1] = 0;
		DiaSetText(dia_it.curitem+1, msg);
		msg[len-1] = lastchar;
	}
	return(lastchar);
}

void Dhighlight(INTBIG on)
{
	INTBIG i, sx, ex, sline, eline, line, dummy, es, ee;
	char *msg;
	RECTAREA r, itemrect;

	if (dia_it.curitem < 0) return;
	itemrect = dia_it.dlgresaddr->list[dia_it.curitem].r;
	msg = (char *)dia_it.dlgresaddr->list[dia_it.curitem].data;
	es = dia_it.editstart;   ee = dia_it.editend;
	if (es > ee) { i = es;   es = ee;   ee = i; }
	Dtextlocation(msg, es, &itemrect, &sx, &sline);
	Dtextlocation(msg, ee, &itemrect, &ex, &eline);
	for(i=sline; i<=eline; i++)
	{
		r.top = itemrect.top + i * dia_lineheight;
		if (r.top >= itemrect.bottom) break;
		r.bottom = r.top + dia_lineheight;
		if (r.bottom > itemrect.bottom) r.bottom = itemrect.bottom;

		r.left = itemrect.left;
		if (i == sline)
		{
			Dtextlocation(msg, es+1, &itemrect, &dummy, &line);
			if (line != sline && sline != eline) continue;
			r.left += sx;
		}
		r.right = itemrect.right;
		if (i == eline)
		{
			if (ex == 0 && sline != eline) continue;
			r.right = itemrect.left + ex + 1;
		}

		Dinvertrect(&r);
	}
}

/*
 * routine to determine where the cursor is in the edit item "r" given character
 * "len" of "msg".  The horizontal offset is placed in "wid" and the line number
 * in "line".
 */
void Dtextlocation(char *msg, INTBIG position, RECTAREA *r, INTBIG *x, INTBIG *line)
{
	INTBIG basepos, len, wid;

	/* if beyond the end, any hit is the end */
	*line = *x = 0;
	if (dia_it.firstch >= strlen(msg)) return;

	/* set initial message and offset in corner of edit box */
	msg = &msg[dia_it.firstch];
	basepos = dia_it.firstch;
	while (*msg != 0)
	{
		for(len = strlen(msg); len>0; len--)
		{
			wid = Dgettextsize(msg, len);
			if (wid < r->right-r->left-2) break;
		}
		if (len <= 0) { *x = 0;   break; }

		if (position - basepos <= len)
		{
			*x = Dgettextsize(msg, position - basepos);
			break;
		}
		msg += len;
		basepos += len;
		(*line)++;
	}
}

INTBIG Dgeteditpos(RECTAREA *r, INTBIG x, INTBIG y, char *msg)
{
	INTBIG pos, i, basepos, lastpos, len, wid;

	/* if beyond the end, any hit is the end */
	if (dia_it.firstch > strlen(msg)) return(strlen(msg));

	/* set initial message and offset in corner of edit box */
	msg = &msg[dia_it.firstch];
	basepos = dia_it.firstch;
	while (*msg != 0)
	{
		for(len = strlen(msg); len>0; len--)
		{
			wid = Dgettextsize(msg, len);
			if (wid < r->right-r->left-2) break;
		}
		if (len <= 0) break;

		if (y > r->top && y <= r->top+dia_lineheight)
		{
			lastpos = 0;
			for(i=0; i<len; i++)
			{
				pos = Dgettextsize(msg, i+1);
				if (r->left+(lastpos+pos)/2 > x) break;
				lastpos = pos;
			}
			return(basepos+i);
		}

		msg += len;
		basepos += len;
		y -= dia_lineheight;
	}
	return(basepos);
}

void Ddoneedit(void)
{
	INTBIG was;

	if (dia_it.curitem < 0) return;
	Dhighlight(0);
	was = dia_it.curitem;
	dia_it.curitem = -1;
	if (dia_it.firstch == 0) return;
	dia_it.firstch = 0;
	Dstufftext((char *)dia_it.dlgresaddr->list[was-1].data, &dia_it.dlgresaddr->list[was].r);
}

void Ddrawitem(INTBIG type, RECTAREA *r, char *msg, INTBIG dim)
{
	INTBIG j, save;
	POPUPDATA *pd;
	RECTAREA myrect;
	INTBIG xv[3], yv[3];

	if ((type&ITEMTYPE) == POPUP)
	{
		/* popup menu item */
		pd = (POPUPDATA *)msg;
		if (pd->count <= 0) return;
		Dstuffmessage(pd->namelist[pd->current], r, dim);
		Dinsetrect(r, -1);
		Ddrawrectframe(r, 1, 0);
		Ddrawline(r->left+3, r->bottom, r->right, r->bottom);
		Ddrawline(r->right, r->top+3, r->right, r->bottom);
		Dinsetrect(r, 1);
		xv[0] = r->right - 18;   yv[0] = r->top + 4;
		xv[1] = r->right - 12;   yv[1] = r->top + 10;
		xv[2] = r->right - 6;    yv[2] = r->top + 4;
		save = r->left;   r->left = r->right - 19;
		Dintdrawrect(r, 255, 255, 255);
		r->left = save;
		Ddrawpolygon(xv, yv, 3, 1);
	} else if ((type&ITEMTYPE) == BUTTON || (type&ITEMTYPE) == DEFBUTTON)
	{
		/* button */
		j = Dgettextsize(msg, strlen(msg));
		Ddrawroundrectframe(r, 12, dim);
		Ddrawtext(msg, strlen(msg), (r->left+r->right-j)/2,
			(r->top+r->bottom+dia_lineheight)/2, dim);
	} else if ((type&ITEMTYPE) == CHECK)
	{
		/* check box */
		myrect = *r;
		myrect.right = myrect.left + 12;
		myrect.top = (myrect.top + myrect.bottom) / 2 - 6;
		myrect.bottom = myrect.top + 12;
		Ddrawrectframe(&myrect, 1, dim);
		Ddrawtext(msg, strlen(msg), myrect.right+4, r->top+dia_lineheight, dim);
	} else if ((type&ITEMTYPE) == RADIO)
	{
		/* radio button */
		myrect = *r;
		myrect.right = myrect.left + 12;
		myrect.top = (myrect.top + myrect.bottom) / 2 - 6;
		myrect.bottom = myrect.top + 12;
		Ddrawcircle(&myrect, dim);
		Ddrawtext(msg, strlen(msg), myrect.right+4, myrect.top+dia_lineheight-2, dim);
	} else if ((type&ITEMTYPE) == MESSAGE)
	{
		/* message */
		Dstuffmessage(msg, r, dim);
	} else if ((type&ITEMTYPE) == EDITTEXT)
	{
		/* edit text */
		if (dim == 0) Dstufftext(msg, r);
		Deditbox(r, 1, dim);
	} else if ((type&ITEMTYPE) == SCROLL)
	{
		/* scrollable list */
		Ddrawrectframe(r, 1, dim);
	} else if ((type&ITEMTYPE) == ICON)
	{
		/* 32x32 icon */
		Dputicon(r->left, r->top, msg);
	} else if ((type&ITEMTYPE) == DIVIDELINE)
	{
		/* dividing line */
		DiaDrawRect(0, r, 255, 255, 255);
	}
}

void Ddrawpopupentries(void)
{
	INTBIG i, y;
	INTBIG xv[3], yv[3];

	Dintdrawrect(&dia_rect, 255, 255, 255);
	Ddrawrectframe(&dia_rect, 1, 0);
	for(i=dia_sta; i<dia_cnt; i++)
	{
		if (i == dia_sta && dia_sta != 0)
		{
			y = dia_rect.top+(i-dia_sta+1)*dia_lineheight+1;
			xv[0] = dia_rect.left + 18;   yv[0] = y - 4;
			xv[1] = dia_rect.left + 12;   yv[1] = y - 10;
			xv[2] = dia_rect.left + 6;    yv[2] = y - 4;
			Ddrawpolygon(xv, yv, 3, 1);
			continue;
		}
		if (i == dia_cnt-1 && dia_cnt != dia_pd->count)
		{
			y = dia_rect.top+(i-dia_sta+1)*dia_lineheight+1;
			xv[0] = dia_rect.left + 18;   yv[0] = y - 10;
			xv[1] = dia_rect.left + 12;   yv[1] = y - 4;
			xv[2] = dia_rect.left + 6;    yv[2] = y - 10;
			Ddrawpolygon(xv, yv, 3, 1);
			continue;
		}
		Ddrawtext(dia_pd->namelist[i], strlen(dia_pd->namelist[i]), dia_rect.left+1,
			dia_rect.top+(i-dia_sta+1)*dia_lineheight+1, 0);
	}
}

void Dstufftext(char *msg, RECTAREA *r)
{
	INTBIG len, wid, vpos;

	Dintdrawrect(r, 255, 255, 255);

	/* display the message in multiple lines */
	vpos = r->top+dia_lineheight;
	if (dia_it.firstch > strlen(msg)) return;
	msg = &msg[dia_it.firstch];
	while (*msg != 0)
	{
		for(len = strlen(msg); len>0; len--)
		{
			wid = Dgettextsize(msg, len);
			if (wid < r->right-r->left-2) break;
		}
		if (len <= 0) break;
		Ddrawtext(msg, len, r->left+1, vpos, 0);
		msg += len;
		vpos += dia_lineheight;
		if (vpos > r->bottom) break;
	}
}

void Dstuffmessage(char *msg, RECTAREA *r, INTBIG dim)
{
	INTBIG len, wid, truelen, vpos, mostch, i;
	char *pt;

	for(pt = msg; *pt != 0; pt++) if (strncmp(pt, "(tm)", 4) == 0)
	{
		(void)strcpy(pt, "\252");		/* "tm" character */
		(void)strcpy(&pt[1], &pt[4]);
		break;
	}
	Dintdrawrect(r, 255, 255, 255);

	/* see how much fits */
	truelen = strlen(msg);
	for(len = truelen; len > 0; len--)
	{
		wid = Dgettextsize(msg, len);
		if (wid <= r->right-r->left-2) break;
	}
	if (len <= 0) return;

	/* if it all fits or must fit (because there is just 1 line), draw it */
	vpos = r->top+dia_lineheight;
	if (len == truelen || dia_lineheight*2 > r->bottom - r->top)
	{
		if (len != truelen)
		{
			/* single line doesn't fit: put ellipsis at end */
			(void)initinfstr();
			for(i=0; i<len-3; i++) (void)addtoinfstr(msg[i]);
			(void)addstringtoinfstr("...");
			msg = returninfstr();
		}
		Ddrawtext(msg, len, r->left+1, vpos, dim);
		return;
	}

	/* write the message in multiple lines */
	while (truelen > 0)
	{
		mostch = 0;
		for(len = truelen; len > 0; len--)
		{
			wid = Dgettextsize(msg, len);
			if (wid > r->right-r->left-2) continue;
			if (mostch == 0) mostch = len;
			if (msg[len] == 0 || msg[len] == ' ') break;
		}
		if (len <= 0) len = mostch;
		Ddrawtext(msg, len, r->left+1, vpos, dim);
		if (msg[len] == ' ') len++;
		msg += len;
		truelen -= len;
		vpos += dia_lineheight;
		if (vpos > r->bottom) break;
	}
}

void Deditbox(RECTAREA *r, INTBIG draw, INTBIG dim)
{
	Dinsetrect(r, -3);
	Ddrawrectframe(r, draw, dim);
	Dinsetrect(r, 3);
}

/*
 * routine to draw line "msg" in position "which" of scroll area "sc"
 */
void Ddrawmsg(INTBIG sc, char *msg, INTBIG which)
{
	INTBIG wid, len, firstch, offset;
	INTBIG skip;
	DSCROLL *scr;

	Dsettextsmall(sc);
	scr = &dia_it.scroll[sc];
	firstch = offset = 0;
	if (scr->horizfactor != 0)
	{
		skip = scr->userrect.right - scr->userrect.left - 6;
		skip *= (INTBIG)scr->horizfactor;   skip /= 10;
		wid = Dgettextsize(msg, strlen(msg));
		if (skip >= wid) return;
		for(firstch = 0; firstch < strlen(msg); firstch++)
		{
			wid = Dgettextsize(msg, firstch);
			if (wid >= skip) break;
		}
		offset = wid - skip;
	}

	for(len = strlen(&msg[firstch]); len>0; len--)
	{
		wid = Dgettextsize(&msg[firstch], len);
		if (wid+offset < scr->userrect.right-scr->userrect.left-6) break;
	}
	if (len <= 0) return;
	Ddrawtext(&msg[firstch], len, scr->userrect.left+2+offset,
		scr->userrect.top + scr->lineheight*(which+1), 0);
	Dsettextsmall(-1);
}

void Dinsetrect(RECTAREA *r, INTBIG amt)
{
	r->left += amt;   r->right -= amt;
	r->top += amt;    r->bottom -= amt;
}

void Ddrawvertslider(INTBIG sc)
{
	RECTAREA r;
	DSCROLL *scr;

	scr = &dia_it.scroll[sc];
	r = scr->userrect;
	r.left = r.right-1;   r.right += 14;
	Ddrawrectframe(&r, 1, 0);
	Ddrawline(r.left, r.top+15, r.right-1, r.top+15);
	Ddrawarrow(sc, UPARROW, 0);
	Ddrawline(r.left, r.bottom-16, r.right-1, r.bottom-16);
	Ddrawarrow(sc, DOWNARROW, 0);
}

void Ddrawhorizslider(INTBIG sc)
{
	RECTAREA r;
	DSCROLL *scr;

	scr = &dia_it.scroll[sc];
	r = scr->userrect;
	r.top = r.bottom-1;   r.bottom += 14;
	Ddrawrectframe(&r, 1, 0);
	Ddrawline(r.left+15, r.top, r.left+15, r.bottom-1);
	Ddrawarrow(sc, LEFTARROW, 0);
	Ddrawline(r.right-16, r.top, r.right-16, r.bottom-1);
	Ddrawarrow(sc, RIGHTARROW, 0);
	Dsethscroll(sc);
}

void Ddrawarrow(INTBIG sc, INTBIG which, INTBIG filled)
{
	INTBIG i, left, top, bottom, right;
	INTBIG xv[ARROWLEN], yv[ARROWLEN];
	DSCROLL *scr;

	scr = &dia_it.scroll[sc];
	top = scr->userrect.top;
	bottom = scr->userrect.bottom-1;
	right = scr->userrect.right-1;
	left = scr->userrect.left;
	for(i=0; i<ARROWLEN; i++)
	{
		switch (which)
		{
			case UPARROW:
				xv[i] = right+dia_arrowx[i];
				yv[i] = top+dia_arrowy[i];
				break;
			case DOWNARROW:
				xv[i] = right+dia_arrowx[i];
				yv[i] = bottom-dia_arrowy[i];
				break;
			case LEFTARROW:
				xv[i] = left+dia_arrowy[i];
				yv[i] = bottom+dia_arrowx[i];
				break;
			case RIGHTARROW:
				xv[i] = right-dia_arrowy[i];
				yv[i] = bottom+dia_arrowx[i];
				break;
		}
	}
	Ddrawpolygon(xv, yv, ARROWLEN, filled);
}

typedef void (*USERTYPE)(RECTAREA*);

void Dredrawdialogwindow(void)
{
	INTBIG i, itemtype, dim;
	char *line;
	RECTAREA r;
	USERTYPE routine;

	if (dia_active <= 0) return;
	Dclear();

	/* do not handle the first macintosh update event */
	if (dia_firstupdate != 0)
	{
		dia_firstupdate = 0;
		return;
	}

	/* redraw the window frame */
	Dsettextsmall(-1);

	for(i=0; i<dia_it.dlgresaddr->items; i++)
	{
		/* draw the item */
		itemtype = dia_it.dlgresaddr->list[i].type;
		r = dia_it.dlgresaddr->list[i].r;
		if ((itemtype&ITEMTYPE) == USERDRAWN)
		{
			routine = (USERTYPE)dia_it.dlgresaddr->list[i].data;
			if (routine != 0) (*routine)(&r);
			continue;
		}
		if ((itemtype&ITEMTYPE) == MESSAGE || (itemtype&ITEMTYPE) == EDITTEXT || (itemtype&ITEMTYPE) == POPUP)
			line = (char *)dia_it.dlgresaddr->list[i].data; else
				line = dia_it.dlgresaddr->list[i].msg;
		if ((itemtype&ITEMTYPE) == CHECK || (itemtype&ITEMTYPE) == RADIO)
			DiaSetControl(i+1, dia_it.dlgresaddr->list[i].data);
		if ((itemtype&INACTIVE) != 0) dim = 1; else dim = 0;
		Ddrawitem(itemtype, &r, line, dim);

		/* highlight the default button */
		if (i == dia_it.defaultbutton-1 &&
			(itemtype == BUTTON || itemtype == DEFBUTTON)) Dhighlightrect(&r);
	}

	/* turn on anything being edited */
	if (dia_it.curitem >= 0) Dhighlight(1);
	for(i=0; i<dia_it.scrollcount; i++)
	{
		Dredrawscroll(i);
		Dinvertentry(i, 1);      /* restores previous inverted entry */
		Dsetvscroll(i);
		if ((dia_it.scroll[i].flags&SCHORIZBAR) != 0)
		{
			Ddrawhorizslider(i);
			Dsethscroll(i);
		}
	}
}

/****************************** DIALOG PRIMITIVE GRAPHICS ******************************/

/*
 * routine to prepare a dialog in location "r", draggable if "movable" is nonzero
 */
void Dnewdialogwindow(RECTAREA *r, char *movable)
{
	FontInfo fi;

#ifdef	USETK
	setdefaultcursortype(NORMALCURSOR);

	if (movable != 0)
	{
		dia_it.tkwindow = (void *)gra_maketkwindow(3, r->left, r->right, r->top, r->bottom, gra_messageEventProc);
		if (dia_it.tkwindow == NULL) return;
		gra_namewindow(dia_it.tkwindow, movable);
	} else
	{
		dia_it.tkwindow = (void *)gra_maketkwindow(4, r->left, r->right, r->top, r->bottom, gra_messageEventProc);
		if (dia_it.tkwindow == NULL) return;
	}
	dia_it.theDialog = (WindowPtr)TkMacGetDrawablePort(Tk_WindowId(dia_it.tkwindow));
#else
	WindowPtr frontmost;
	INTBIG type;
	Str255 line;

	setdefaultcursortype(NORMALCURSOR);

	if (movable != 0)
	{
		strcpy((char *)&line[1], movable);
		line[0] = strlen((char *)&line[1]);
		type = noGrowDocProc;
	} else
	{
		line[0] = 0;
		type = dBoxProc;
	}

	gra_frontnonfloatingwindow(&frontmost);
	if (frontmost != 0) gra_activatewindow(frontmost, 0);

	dia_it.theDialog = (GrafPtr)NewCWindow(0L, (Rect *)r, line, 0, type, (WindowPtr)(-1L), 0, 0L);
	if (dia_it.theDialog == 0) return;

	ShowHide(dia_it.theDialog, 1);
	gra_activatewindow(dia_it.theDialog, 1);
#endif

	SetPort(dia_it.theDialog);

	/* setup for small scroll text */
	TextFont(DSFONT);
	TextSize(10);
	GetFontInfo(&fi);
	dia_slineheight = fi.ascent + fi.descent + fi.leading;
	dia_slineoffset = fi.descent;

	/* setup for all other text */
	TextFont(DFONT);
	TextSize(12);
	GetFontInfo(&fi);
	dia_lineheight = fi.ascent + fi.descent + fi.leading;
	dia_lineoffset = fi.descent;
	dia_curlineoffset = dia_lineoffset;
}

/*
 * routine to terminate the current dialog
 */
void Ddonedialogwindow(void)
{
	Rect r;

#ifdef	USETK
	if (dia_it.dlgresaddr->movable != 0)
	{
		r = (*((WindowPeek)dia_it.theDialog)->strucRgn)->rgnBBox;
		dia_it.dlgresaddr->windowRect.left = r.left+1;
		dia_it.dlgresaddr->windowRect.right = r.right-2;
		dia_it.dlgresaddr->windowRect.top = r.top+1;
		dia_it.dlgresaddr->windowRect.bottom = r.bottom-1-MENUSIZE;
	}
	(void)Tk_DestroyWindow(dia_it.tkwindow);
#else
	if (dia_it.dlgresaddr->movable != 0)
	{
		r = (*((WindowPeek)dia_it.theDialog)->strucRgn)->rgnBBox;
		dia_it.dlgresaddr->windowRect.left = r.left+1;
		dia_it.dlgresaddr->windowRect.right = r.right-2;
		dia_it.dlgresaddr->windowRect.top = r.top+MENUSIZE;
		dia_it.dlgresaddr->windowRect.bottom = r.bottom-2;
	}
	gra_disposeoswindow(dia_it.theDialog, dia_it.tkwindow);
#endif
}

/*
 * routine to erase the dialog window
 */
void Dclear(void)
{
#ifdef USETK
	gra_raisewindow(dia_it.tkwindow);
#else
	gra_selectoswindow(dia_it.theDialog);
#endif
	SetPort(dia_it.theDialog);
}

/*
 * routine to force the dialog window to be current (on window systems
 * where this is relevant)
 */
void Dforcedialog(void)
{
#ifdef USETK
	gra_raisewindow(dia_it.tkwindow);
#else
	gra_selectoswindow(dia_it.theDialog);
#endif
	SetPort(dia_it.theDialog);
}

/*
 * routine to wait for the next action from the dialog window.  Returns:
 *  -1  Nothing happened
 *   0  Key typed, character in "chr"
 *   1  Mouse button pushed, coordinates in (x,y), modifiers in "chr"
 *   2  Mouse button released, coordinates in (x,y)
 *   4  Mouse motion while button down, coordinates in (x,y)
 *   5  Double-click of mouse button, coordinates in (x,y), modifiers in "chr"
 */
INTBIG Dwaitforaction(INTBIG *x, INTBIG *y, INTBIG *chr, UINTBIG *time)
{
	INTSML sx, sy, but;

	gra_handlingdialog = 1;
	if (ttydataready() != 0)
	{
		*chr = getnxtchar();
		*time = ticktime();
		gra_handlingdialog = 0;
		return(0);
	}

	/* get button */
	waitforbutton(&sx, &sy, &but);
	gra_handlingdialog = 0;
	*x = sx;   *y = sy;
	if (but < 0) return(-1);
	*time = ticktime();
	if (doublebutton(but) != 0) return(5);
	return(1);
}

/*
 * routine to track mouse movements while the button remains down, calling
 * the routine "eachdown" for each advance.  The routine is given the
 * mouse coordinates as parameters.  The initial mouse position is (fx,fy).
 */
void Dtrackcursor(INTBIG fx, INTBIG fy, INTSML (*eachdown)(INTBIG, INTBIG))
{
	trackcursor(0, us_nullup, us_nullvoid, eachdown, us_nullchar, us_nullvoid, TRACKNORMAL);
}

/*
 * routine to move the rectangle "sr" to "dr"
 */
void Dshiftbits(RECTAREA *sr, RECTAREA *dr)
{
	SetPort(dia_it.theDialog);
	CopyBits(&dia_it.theDialog->portBits, &dia_it.theDialog->portBits,
		(Rect *)sr, (Rect *)dr, srcCopy, 0);
}

/*
 * routine to handle popup menu "pd" in rectangle "r".  Returns negative if
 * the function is not possible.  Returns the entry in "pd" that is selected
 * (returns pd->current if no change is made).
 */
INTBIG Dhandlepopup(RECTAREA *r, POPUPDATA *pd)
{
	INTBIG ret;
	INTBIG i, j;
	Str255 name;
	MenuHandle menu;
	Point p;

	SetPort(dia_it.theDialog);
	p.h = r->left;   p.v = r->top;
	LocalToGlobal(&p);
	menu = NewMenu(2048, "\p");
	for(i=0; i<pd->count; i++)
	{
		for(j=0; pd->namelist[i][j] != 0; j++) name[j+1] = pd->namelist[i][j];
		name[0] = j;
		AppendMenu(menu, name);
	}
	InsertMenu(menu, -1);
	ret = PopUpMenuSelect(menu, p.v, p.h, pd->current+1);
	DeleteMenu(2048);
	DisposeMenu(menu);
	if (HiWord(ret) == 0) return(pd->current);
	return(LoWord(ret) - 1);
}

/*
 * routine to draw the 32x32 bitmap in "data" at (x,y)
 */
void Dputicon(INTBIG x, INTBIG y, char *data)
{
	BitMap b;

	b.baseAddr = data;
	b.rowBytes = 4;
	b.bounds.left = x;   b.bounds.top = y;
	b.bounds.right = x+32;   b.bounds.bottom = y+32;
	CopyBits(&b, &dia_it.theDialog->portBits, &b.bounds, &b.bounds, srcCopy, 0L);
}

/*
 * routine to draw a line from (xf,yf) to (xt,yt) in the dialog
 */
void Ddrawline(INTBIG xf, INTBIG yf, INTBIG xt, INTBIG yt)
{
	SetPort(dia_it.theDialog);
	MoveTo(xf, yf);
	LineTo(xt, yt);
}

/*
 * routine to draw a filled rectangle at "rect" in the dialog with color (r,g,b).
 */
void Dintdrawrect(RECTAREA *rect, INTBIG r, INTBIG g, INTBIG b)
{
	RGBColor color, oldcolor;

	SetPort(dia_it.theDialog);
	color.red = r << 8;
	color.green = g << 8;
	color.blue = b << 8;
	GetForeColor(&oldcolor);
	RGBForeColor(&color);
	FillRect((Rect *)rect, &qd.black);
	RGBForeColor(&oldcolor);
}

/*
 * routine to draw a gray rectangle at "r" in the dialog
 */
void Dgrayrect(RECTAREA *r)
{
	SetPort(dia_it.theDialog);
	FillRect((Rect *)r, &qd.ltGray);
}

/*
 * routines to invert a rectangle at "r" in the dialog
 */
void Dinvertrect(RECTAREA *r)
{
	SetPort(dia_it.theDialog);
	InvertRect((Rect *)r);
}

/*
 * routine to invert a rounded rectangle at "r" in the dialog
 */
void Dinvertroundrect(RECTAREA *r)
{
	SetPort(dia_it.theDialog);
	InvertRoundRect((Rect *)r, 10, 10);
}

/*
 * routine to draw the outline of rectangle "r" in the dialog.  Erases if "on" is zero
 */
void Ddrawrectframe(RECTAREA *r, INTBIG on, INTBIG dim)
{
	SetPort(dia_it.theDialog);
	if (on == 0) PenMode(patBic);
	FrameRect((Rect *)r);
	PenMode(patCopy);
	if (dim != 0)
	{
		PenMode(patBic);
		PenPat(&qd.gray);
		PaintRect((Rect *)r);
		PenMode(patCopy);
		PenPat(&qd.black);
	}
}

/*
 * routine to highlight the outline of rectangle "r" in the dialog
 */
void Dhighlightrect(RECTAREA *r)
{
	Dinsetrect(r, -4);
	SetPort(dia_it.theDialog);
	PenSize(3, 3);
	FrameRoundRect((Rect *)r, 16, 16);
	PenSize(1, 1);
}

/*
 * routine to draw the outline of rounded rectangle "r" in the dialog
 */
void Ddrawroundrectframe(RECTAREA *r, INTBIG arc, INTBIG dim)
{
	SetPort(dia_it.theDialog);
	FrameRoundRect((Rect *)r, arc, arc);
	if (dim != 0)
	{
		PenMode(patBic);
		PenPat(&qd.gray);
		PaintRect((Rect *)r);
		PenMode(patCopy);
		PenPat(&qd.black);
	}
}

/*
 * routine to invert the outline of rectangle "r" in the dialog
 */
void Dinvertrectframe(RECTAREA *r)
{
	SetPort(dia_it.theDialog);
	PenMode(patXor);
	FrameRect((Rect *)r);
	PenMode(patCopy);
}

/*
 * routine to draw the polygon in "count" points (xv,yv) and fill it if "filled" is nonzero
 */
void Ddrawpolygon(INTBIG *xv, INTBIG *yv, INTBIG count, INTBIG filled)
{
	PolyHandle ph;
	INTBIG i, l;

	SetPort(dia_it.theDialog);
	ph = OpenPoly();
	for(i=0; i<count; i++)
	{
		if (i == 0) l = count-1; else l = i-1;
		MoveTo(xv[l], yv[l]);
		LineTo(xv[i], yv[i]);
	}
	ClosePoly();
	if (filled != 0) FillPoly(ph, &qd.black); else
	{
		FillPoly(ph, &qd.white);
		FramePoly(ph);
	}
	KillPoly(ph);
}

/*
 * routine to draw a circle outline in rectangle "r" in the dialog
 */
void Ddrawcircle(RECTAREA *r, INTBIG dim)
{
	SetPort(dia_it.theDialog);
	FrameOval((Rect *)r);
	if (dim != 0)
	{
		PenMode(patBic);
		PenPat(&qd.gray);
		PaintRect((Rect *)r);
		PenMode(patCopy);
		PenPat(&qd.black);
	}
}

/*
 * routine to draw a filled circle in rectangle "r" in the dialog
 */
void Ddrawdisc(RECTAREA *r)
{
	SetPort(dia_it.theDialog);
	FillOval((Rect *)r, &qd.black);
}

/*
 * routine to determine the width of the "len" characters at "msg"
 */
INTBIG Dgettextsize(char *msg, INTBIG len)
{
	INTBIG wid;

	if (len == 0) return(0);
	SetPort(dia_it.theDialog);
	wid = TextWidth(msg, 0, len);
	return(wid);
}

/*
 * Routine to set the current text size to the large or small scroll text.
 */
void Dsettextsmall(INTBIG sc)
{
	INTBIG smallf;
	DSCROLL *scr;

	if (sc < 0) smallf = 0; else
	{
		scr = &dia_it.scroll[sc];
		smallf = scr->flags & SCSMALLFONT;
	}
	if (smallf != 0)
	{
		dia_curlineoffset = scr->lineoffset;
		TextFont(DSFONT);
		TextSize(10);
	} else
	{
		dia_curlineoffset = dia_lineoffset;
		TextFont(DFONT);
		TextSize(12);
	}
}

/*
 * routine to draw "len" characters of "msg" at (x,y)
 */
void Ddrawtext(char *msg, INTBIG len, INTBIG x, INTBIG y, INTBIG dim)
{
	Rect r;

	SetPort(dia_it.theDialog);
	MoveTo(x, y-dia_curlineoffset-1);
	DrawText(msg, 0, len);
	if (dim != 0)
	{
		r.left = x;     r.right = x + TextWidth(msg, 0, len);
		r.bottom = y;   r.top = y - dia_lineheight;
		PenPat(&qd.gray);
		PenMode(patBic);
		PaintRect(&r);
		PenMode(patCopy);
		PenPat(&qd.black);
	}
}

char *gra_makepstring(char *string)
{
	static char pstring[200];

	strcpy(&pstring[1], string);
	pstring[0] = strlen(string);
	return(pstring);
}

/******************** TCL/TK ********************/

#ifdef	USETK

/*
 * This is a hack to take over menu selection from TK.
 * The module "tkMacMenus.c" has this routine for handling
 * a limited set of pulldown menus.  By yanking that routine
 * from the TK build, this routine provides full menu access
 * to Electric.
 */
void TkMacHandleMenuSelect(long mResult, int optionKeyPressed)
{
	short theItem, theMenu;
	Str255 name;
	GrafPtr savePort;

	if (mResult == 0)
	{
		TkMacHandleTearoffMenu();
		return;
	}

	theItem = LoWord(mResult);
	theMenu = HiWord(mResult);
	if (theMenu == appleMENU)
	{
		GetPort(&savePort);
		GetMenuItemText(tkAppleMenu, theItem, name);
		(void)OpenDeskAcc(name);
		SetPort(savePort);
		return;
	}
	TkMacDispatchMenuEvent(theMenu, theItem);
	TkMacClearMenubarActive();
	HiliteMenu(0);
}

void gra_addeventtoqueue(INTBIG state, INTSML x, INTSML y)
{
	MYEVENTQUEUE *next;

	next = gra_eventqueuetail + 1;
	if (next >= &gra_eventqueue[EVENTQUEUESIZE]) next = gra_eventqueue;
	if (next == gra_eventqueuehead)
	{
		/* queue is full */
		SysBeep(100);
		SysBeep(100);
		return;
	}

	gra_eventqueuetail->inputstate = state;
	gra_eventqueuetail->cursorx = x;
	gra_eventqueuetail->cursory = y;
	gra_eventqueuetail = next;
}

void gra_clearwindowframe(WINDOWFRAME *whichwf)
{
	REGISTER WINDOWPART *w, *neww, *nextw;
	REGISTER NODEPROTO *np;
	REGISTER WINDOWFRAME *lastwf, *wf;

	/* save highlighting and turn it off */
	us_pushhighlight();
	us_clearhighlightcount();

	startobjectchange((INTBIG)us_aid, VAID);

	/* kill all editor windows on this frame */
	neww = NOWINDOWPART;
	for(w = el_topwindowpart; w != NOWINDOWPART; w = nextw)
	{
		nextw = w->nextwindowpart;
		if (w->frame != whichwf)
		{
			neww = w;
			continue;
		}

		/* kill this window */
		killwindowpart(w);
	}
	endobjectchange((INTBIG)us_aid, VAID);

	(void)setvalkey((INTBIG)us_aid, VAID, us_current_window, (INTBIG)neww, VWINDOWPART|VDONTSAVE);
	if (neww != NOWINDOWPART) np = neww->curnodeproto; else np = NONODEPROTO;
	(void)setval((INTBIG)el_curlib, VLIBRARY, "curnodeproto", (INTBIG)np, VNODEPROTO);

	/* restore highlighting */
	(void)us_pophighlight(0);

	/* remove the window frame object */
	lastwf = NOWINDOWFRAME;
	for(wf = el_firstwindowframe; wf != NOWINDOWFRAME; wf = wf->nextwindowframe)
	{
		if (wf == whichwf)
		{
			if (lastwf == NOWINDOWFRAME) el_firstwindowframe = wf->nextwindowframe; else
				lastwf->nextwindowframe = wf->nextwindowframe;
			DisposeGWorld(wf->window);
			efree((char *)wf);
			break;
		}
		lastwf = wf;
	}
}

void gra_raisewindow(Tk_Window win)
{
	char *argv[2];
	extern Tk_Window gra_tktopwindow;
	int ret;

	argv[0] = "raise";
	argv[1] = Tk_PathName(win);
	ret = Tk_RaiseCmd((ClientData)gra_tktopwindow, tcl_interp, 2, argv);
	if (ret != TCL_OK)
	{
		ttyputerr(_("ERROR from Tk_RaiseCmd: %s"), tcl_interp->result);
		Tcl_ResetResult(tcl_interp);
	}
}

void gra_namewindow(Tk_Window win, char *name)
{
	char *argv[4];
	extern Tk_Window gra_tktopwindow;
	int ret;

	argv[0] = "wmcmd";
	argv[1] = "title";
	argv[2] = Tk_PathName(win);
	argv[3] = name;
	ret = Tk_WmCmd((ClientData)gra_tktopwindow, tcl_interp, 4, argv);
	if (ret != TCL_OK)
	{
		ttyputerr(_("ERROR from Tk_WmCmd: %s"), tcl_interp->result);
		Tcl_ResetResult(tcl_interp);
	}
}

void gra_messageEventProc(ClientData clientData, XEvent *eventPtr)
{
	INTBIG which, key, inputstate, theTime;
	INTSML cntlCode, oldpos, x, y;
	static XWindowAttributes xwa;
	static INTSML update = 1;
	WindowPtr win;
	WINDOWFRAME *wf;
	Rect r;
	Point p;
	ControlHandle theControl;

	inputstate = NOEVENT;
	win = (WindowPtr)TkMacGetDrawablePort(eventPtr->xany.window);
	switch (eventPtr->type)
	{
		case KeyPress:
			which = ((XKeyEvent *)eventPtr)->keycode & charCodeMask;
			if (el_curwindowframe != NOWINDOWFRAME)
			{
				SetPort((WindowPtr)el_curwindowframe->realwindow);
				x = eventPtr->xkey.x;
				y = el_curwindowframe->revy - eventPtr->xkey.y;
			}
			if ((eventPtr->xbutton.state&Mod1Mask) != 0 && which == '.')
			{
				el_pleasestop = 2;
				break;
			}
			if ((eventPtr->xbutton.state&Mod1Mask) != 0)
			{
				key = MenuKey((char)which);
				gra_highmenu = HiWord(key);
				gra_lowmenu = -LoWord(key);
				inputstate = MENUEVENT;
				if (gra_highmenu != 0) break;
			}
			if (gra_messagesinfront != 0)
			{
				TESetSelect(32767, 32767, gra_TEH);
				TEKey(which, gra_TEH);
				gra_showselect();
				break;
			}
			inputstate = which & CHARREAD;
			us_state |= DIDINPUT;
			gra_addeventtoqueue(inputstate, x, y);
			break;

		case ButtonPress:
			p.h = eventPtr->xbutton.x;	p.v = eventPtr->xbutton.y;
			for(wf = el_firstwindowframe; wf != NOWINDOWFRAME; wf = wf->nextwindowframe)
				if ((WindowPtr)wf->realwindow == win) break;

			if (gra_messagesinfront != 0)
			{
				cntlCode = FindControl(p, win, &theControl);
				if (cntlCode == kControlIndicatorPart)
				{
					oldpos = GetControlValue(theControl);
					TrackControl(theControl, p, 0L);
					if (theControl == gra_vScroll) gra_adjustvtext(); else
						gra_adjusthtext(oldpos);
					break;
				}
				if (cntlCode == kControlUpButtonPart || cntlCode == kControlDownButtonPart ||
					cntlCode == kControlPageUpPart || cntlCode == kControlPageDownPart)
				{
					if (theControl == gra_vScroll)
						TrackControl(theControl, p, gra_scrollvprocUPP); else
							TrackControl(theControl, p, gra_scrollhprocUPP);
					break;
				}
				TEClick(p, (eventPtr->xbutton.state&ShiftMask) != 0, gra_TEH);
				break;
			}

			/* ignore clicks in the messages area at the bottom */
			if (wf != NOWINDOWFRAME && eventPtr->xbutton.y >= win->portRect.bottom - SBARHEIGHT)
				break;

			/* compute event information */
			inputstate = ISBUTTON;
			if ((eventPtr->xbutton.state&(ShiftMask|LockMask)) != 0) inputstate |= SHIFTDOWN;
			if ((eventPtr->xbutton.state&ControlMask) != 0) inputstate |= CONTROLDOWN;
			if ((eventPtr->xbutton.state&Mod1Mask) != 0) inputstate |= COMMANDDOWN;
			if ((eventPtr->xbutton.state&Mod2Mask) != 0) inputstate |= OPTIONDOWN;
			x = eventPtr->xbutton.x;
			if (wf == NOWINDOWFRAME) y = eventPtr->xbutton.y; else
			{
				y = wf->revy - eventPtr->xbutton.y;
			}
			theTime = LMGetTicks();
			if (theTime - gra_doubleclick < gra_lastclick &&
				(gra_inputstate & (SHIFTDOWN|COMMANDDOWN|OPTIONDOWN|CONTROLDOWN)) == 0 &&
				abs(x-gra_lstcurx) < 5 && abs(y-gra_lstcury) < 5)
			{
				inputstate |= DOUBLECLICK;
				gra_lastclick = theTime - gra_doubleclick - 1;
			} else gra_lastclick = theTime;
			gra_lstcurx = x;   gra_lstcury = y;
			us_state |= DIDINPUT;
			gra_addeventtoqueue(inputstate, x, y);
			break;

		case ButtonRelease:
			inputstate = ISBUTTON|BUTTONUP;
			if ((eventPtr->xbutton.state&(ShiftMask|LockMask)) != 0) inputstate |= SHIFTDOWN;
			if ((eventPtr->xbutton.state&ControlMask) != 0) inputstate |= CONTROLDOWN;
			if ((eventPtr->xbutton.state&Mod1Mask) != 0) inputstate |= COMMANDDOWN;
			if ((eventPtr->xbutton.state&Mod2Mask) != 0) inputstate |= OPTIONDOWN;
			x = eventPtr->xbutton.x;
			if (el_curwindowframe == NOWINDOWFRAME) y = eventPtr->xbutton.y; else
				y = el_curwindowframe->revy - eventPtr->xbutton.y;
			gra_addeventtoqueue(inputstate, x, y);
			break;

		case MotionNotify:
			for(wf = el_firstwindowframe; wf != NOWINDOWFRAME; wf = wf->nextwindowframe)
				if ((WindowPtr)wf->realwindow == win) break;
			inputstate = MOTION;
			if ((eventPtr->xmotion.state&Button1Mask) == 0) inputstate |= BUTTONUP;
			x = eventPtr->xmotion.x;
			if (wf == NOWINDOWFRAME) y = eventPtr->xmotion.y; else
				y = wf->revy - eventPtr->xmotion.y;
			gra_addeventtoqueue(inputstate, x, y);
			break;

		case ConfigureNotify:
			gra_mygrowwindow(win, win->portRect.right-win->portRect.left,
				win->portRect.bottom-win->portRect.top, &win->portRect);
			break;

		case FocusIn:
			if (win == gra_messageswindow && gra_messageswindow != 0)
			{
				gra_messagesinfront = 1;
				ShowControl(gra_vScroll);
				ShowControl(gra_hScroll);
			}
			gra_setcurrentwindowframe();
			break;

		case FocusOut:
			if (win == gra_messageswindow && gra_messageswindow != 0)
			{
				gra_messagesinfront = 0;
				HideControl(gra_vScroll);
				HideControl(gra_hScroll);
			}
			break;

		case DestroyNotify:
			if (win == gra_messageswindow && gra_messageswindow != 0)
			{
				/* messages window deleted */
				gra_messagesinfront = 0;
				gra_messageswindow = 0;
				break;
			}

			/* determine which editor window was hit */
			for(wf = el_firstwindowframe; wf != NOWINDOWFRAME; wf = wf->nextwindowframe)
				if ((WindowPtr)wf->realwindow == win) break;
			if (wf != NOWINDOWFRAME) gra_clearwindowframe(wf);
			break;

		case Expose:
			if (eventPtr->xexpose.count == 0)
			{
				SetPort(win);
				if (win == gra_messageswindow && gra_messageswindow != 0)
				{
					EraseRect(&win->portRect);
					DrawGrowIcon(win);
					DrawControls(win);
					TEUpdate(&win->portRect, gra_TEH);
				} else
				{
					for(wf = el_firstwindowframe; wf != NOWINDOWFRAME; wf = wf->nextwindowframe)
						if ((WindowPtr)wf->realwindow == win) break;
					if (wf != NOWINDOWFRAME)
					{
						DrawGrowIcon(win);
						r = (*wf->realwindow->visRgn)->rgnBBox;
						gra_setrect(wf, r.left, r.right, r.top, r.bottom);
						us_redostatus(wf);
					} else
					{
						/* probably a dialog, just process it */
						Dredrawdialogwindow();
					}
				}
			}
			break;
	}
}
#endif
