// 
//   Copyright (C) 2005, 2006, 2007, 2008 Free Software Foundation, Inc.
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 3 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

#ifndef __RC_H__
#define __RC_H__

// This is generated by autoconf
#ifdef HAVE_CONFIG_H
# include "gnashconfig.h"
#endif

#include "tu_config.h"

#include <string>
#include <vector>
#include <iostream>
#include <sys/shm.h>		// for key_t
#include <boost/cstdint.hpp>

namespace gnash {
  
class DSOEXPORT RcInitFile 
{
public:

    // Return the default instance of RC file,
    static RcInitFile& getDefaultInstance();

    bool loadFiles();

    bool parseFile(const std::string& filespec);
    
    /// Writes a valid gnashrc file. If the file already exists,
    /// is is overwritten.
    /// @param filespec the file to write
    /// @return whether the file was successfully written.
    bool updateFile(const std::string& filespec);
    
    /// Writes a gnashrc file to the file specified in the
    /// GNASHRC environment variable OR to the user's home
    /// directory.
    /// @return whether the file was successfully written.
    bool updateFile();
    
    bool useSplashScreen() const { return _splashScreen; }
    void useSplashScreen(bool value);

    bool useActionDump() const { return _actionDump; }
    void useActionDump(bool value);

    bool useParserDump() const { return _parserDump; }
    void useParserDump(bool value);

    bool useWriteLog() const { return _writeLog; }
    void useWriteLog(bool value);

    int getTimerDelay() const { return _delay; }
    void setTimerDelay(int x) { _delay = x; }

    bool showASCodingErrors() const { return _verboseASCodingErrors; }
    void showASCodingErrors(bool value);

    bool showMalformedSWFErrors() const { return _verboseMalformedSWF; }
    void showMalformedSWFErrors(bool value);

    int getMovieLibraryLimit() const { return _movieLibraryLimit; }
    void setMovieLibraryLimit(int value) { _movieLibraryLimit = value; }

    bool enableExtensions() const { return _extensionsEnabled; }

    /// Return true if user is willing to start the gui in "stop" mode
    //
    /// defaults to false.
    ///
    bool startStopped() const { return _startStopped; }
    void startStopped(bool value) { _startStopped = value; }

    bool insecureSSL() const { return _insecureSSL; }
    void insecureSSL(bool value) { _insecureSSL = value; }
    
    int verbosityLevel() const { return _verbosity; }
    void verbosityLevel(int value) { _verbosity = value; }
    
    void setDebugLog(std::string &x) { _log = x; }
    const std::string& getDebugLog() const { return _log; }

    void setDocumentRoot(std::string &x) { _wwwroot = x; }
    std::string getDocumentRoot() { return _wwwroot; }
    
    bool useDebugger() const { return _debugger; }
    void useDebugger(bool value) { _debugger = value; }

    bool useSound() const { return _sound; }
    void useSound(bool value) { _sound = value; }

    // strk: I'd drop this, and allow an -f switch to select
    //       the gnashrc file to use instead
    bool usePluginSound() const { return _pluginSound; }
    void usePluginSound(bool value) { _pluginSound = value; }

    bool useLocalDomain() const { return _localdomainOnly; }
    void useLocalDomain(bool value);
 
    /// Whether to restrict access to the local host   
    bool useLocalHost() const { return _localhostOnly; }

    /// Set whether to restrict access to the local host
    void useLocalHost(bool value);

    typedef std::vector<std::string> PathList;

    /// Get the current RcInitFile whitelist of domains to allow
    /// @return a std::vector of strings containing allowed domains 
    const PathList& getWhiteList() const { return _whitelist; }

    /// Sets the RcInitFile whitelist of domains to allow
    /// @param list a std::vector of strings containing domains without protocol
    void setWhitelist (std::vector<std::string>& list) { _whitelist = list; }

    /// Get the current RcInitFile blacklist of domains to block
    /// @return a std::vector of strings containing blocked domains    
    const PathList& getBlackList() const { return _blacklist; }
    
    /// Sets the RcInitFile blacklist of domains to block
    /// @param list a std::vector of strings containing domains without protocol
    void setBlacklist (std::vector<std::string>& list) { _blacklist = list; }

    /// Return the list of directories to be used as the 'local' sandbox
    //
    /// Local sendbox is the set of resources on the filesystem we want to
    /// give the current movie access to.
    ///
    const PathList& getLocalSandboxPath() const { return _localSandboxPath; }

    /// Add a directory to the local sandbox list
    void addLocalSandboxPath(const std::string& dir)
    {
        _localSandboxPath.push_back(dir);
    }

    /// Sets a list of sandbox paths. Gnash will only allow movies access
    /// to files in these paths. The path of the movie playing is automatically
    /// added.
    /// @param list a std::vector of strings containing paths to allow
    void setLocalSandboxPath(const PathList& path)
    {
        _localSandboxPath = path;
    }

    const std::string& getFlashVersionString() const { return _flashVersionString; }
    void setFlashVersionString(std::string& value) { _flashVersionString = value; }

    const std::string& getFlashSystemOS() const { return _flashSystemOS; }
    void setFlashSystemOS(std::string& value) { _flashSystemOS = value; }

    const std::string& getFlashSystemManufacturer() const { return _flashSystemManufacturer; }
    void setFlashSystemManufacturer(std::string& value) { _flashSystemManufacturer = value; }
    
    const std::string& getGstAudioSink() const { return _gstaudiosink; }
    void setGstAudioSink(const std::string& value) { _gstaudiosink = value; }

    int getRetries() const { return _retries; }
    void setRetries(int x) { _retries = x; }

    /// Return the number of seconds of inactivity before timing out streams downloads
    double getStreamsTimeout() const { return _streamsTimeout; }
    /// Set the number of seconds of inactivity before timing out streams downloads
    void setStreamsTimeout(double x) { _streamsTimeout = x; }

    /// Get the URL opener command format
    //
    /// The %u label will need to be substituted by the actual url
    /// properly escaped.
    ///
    const std::string &getURLOpenerFormat() const
    {
        return _urlOpenerFormat;
    }
    
    void setURLOpenerFormat(const std::string& value)
    {
        _urlOpenerFormat = value;
    }

    // Get the location of the sandbox for .sol files
    const std::string &getSOLSafeDir() const { return _solsandbox; }

    // Set the location of the sandbox for .sol files
    void setSOLSafeDir(std::string &x) { _solsandbox = x; }

    bool getSOLLocalDomain() const { return _sollocaldomain; }
    void setSOLLocalDomain(bool x) { _sollocaldomain = x; }
    bool getSOLReadOnly() const { return _solreadonly; }
    void setSOLReadOnly(bool x) { _solreadonly = x; }
    bool getLocalConnection() const { return _lcdisabled; }
    void setLocalConnection(bool x) { _lcdisabled = x; }
    // Enable tracing all LocalConnection traffic
    bool getLCTrace() const { return _lctrace; }
    void setLCTrace(bool x) { _lctrace = x; }

    // 
    key_t getLCShmKey() const { return static_cast<key_t>(_lcshmkey); }
    void setLCShmKey(bool x) { _lcshmkey = x; }
    
    void dump();    

private:

    /// The timer delay
    boost::uint32_t  _delay;

    /// Max number of movie clips to store in the library      
    boost::uint32_t  _movieLibraryLimit;   

    /// Enable debugging of this class
    bool _debug;

    /// Enable the Flash movie debugger
    bool _debugger;

    // Level of debugging output
    boost::uint32_t  _verbosity;

    /// Command format to use to open urls
    //
    /// The %u label will need to be substituted by the url
    /// (properly escaped)
    ///
    std::string  _urlOpenerFormat;

    /// String to pass as $version in Actionscript
    std::string  _flashVersionString;
    
    /// String representing the first GStreamer audio output pipeline to try
    std::string _gstaudiosink;

    /// String to pass as System.capabilities.os
    /// in Actionscript. If empty, leaves detection
    /// to System.cpp (default).
    std::string  _flashSystemOS;       

    /// String to pass as
    /// System.capabilities.manufacturer
    /// in Actionscript
    std::string  _flashSystemManufacturer;

    /// Enable dumping actionscript classes
    bool _actionDump;

    /// Enable dumping parser data
    bool _parserDump;

    /// Enable ActionScript errors verbosity
    bool _verboseASCodingErrors;

    /// Enable Malformed SWF verbosity
    bool _verboseMalformedSWF;

    /// Display a splash screen when loading a movie
    bool _splashScreen;

    /// Only access network resources in the local domain
    bool _localdomainOnly;
    
    /// Only access network resources on the local host
    bool _localhostOnly;
    
    /// Allowed domains
    PathList _whitelist;
    
    /// Blocked Domains 
    PathList _blacklist;
    
    /// The name of the debug log
    std::string _log;
    
    /// Enable writing the debug log to disk
    bool _writeLog;
    
    /// The root path for the streaming server        
    std::string _wwwroot;
    
    /// the number of retries for a thread 
    int _retries;
    
    /// Enable the sound handler at startup         
    bool _sound;
    
    /// Enable sound for the plugin
    bool _pluginSound;		

    /// Enable scanning plugin path for extensions
    bool _extensionsEnabled;	

    /// Start the gui in "stop" mode
    bool _startStopped;		

    /// Allow SSL connections without verifying the certificate
    bool _insecureSSL;		

    /// The number of seconds of inactivity triggering download timeout
    double _streamsTimeout;

    /// Local sandbox: the set of resources on the filesystem we want to
    /// give the current movie access to.
    PathList _localSandboxPath;

    // SOL Sandbox: This is the only dir .sol (Shared Object) files can be written in,
    // or read from.
    std::string _solsandbox;

    /// Whether SOL files can be written
    bool _solreadonly;
    bool _sollocaldomain;
    
    // Disable local connection
    bool _lcdisabled;
    
    /// Trace local connection activity (in log)
    bool _lctrace;
    
    /// Shared memory segment key (can be set for
    /// compatibility with other players.
    boost::uint32_t _lcshmkey;

    // A function only for writing path lists to an outstream.
    void writeList(PathList& list, std::ostream& o);

protected:
    
    /// Construct only by getDefaultInstance()
    RcInitFile();

    /// Never destroy (TODO: add a destroyDefaultInstance)
    ~RcInitFile();

    /// Performs substitution on a path string (~) 
    void expandPath(std::string& path);

    /// \brief
    /// If variable matches pattern (case-insensitive)
    /// set *var according to value
    //
    /// @return true if variable matches patter, false otherwise
    ///
    static bool extractSetting(bool *var, const char *pattern,
                        std::string &variable, std::string &value);

    /// \brief
    /// If variable matches pattern (case-insensitive)
    /// set *num according to value
    //
    /// @return true if variable matches patter, false otherwise
    ///
    static bool extractNumber(boost::uint32_t *num, const char *pattern,
                        std::string &variable, std::string &value);

    /// \brief
    /// If variable matches pattern (case-insensitive)
    /// set out according to value
    //
    /// @return true if variable matches patter, false otherwise
    ///
    static bool extractDouble(double& out, const char *pattern,
                        std::string &variable, std::string &value);

    void parseList(std::vector<std::string>& list, std::string &action,
			std::string &listname, std::string &items);

};

//extern DSOEXPORT RcInitFile rcfile;

// End of gnash namespace 
}

// __RC_H__
#endif


// local Variables:
// mode: C++
// indent-tabs-mode: t
// End:
