/* -*- c++ -*- */
/*
 * Copyright 2003 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <GrFileSource.h>
#include <cstdio>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdexcept>

// should be handled via configure
#ifdef O_LARGEFILE
#define	OUR_O_LARGEFILE	O_LARGEFILE
#else
#define	OUR_O_LARGEFILE 0
#endif

GrFileSource::GrFileSource (size_t itemsize, double sampling_freq,
			    const char *filename, bool repeat)
  : GrSource (itemsize, 1), d_itemsize (itemsize), d_fp (0), d_repeat (repeat)
{
  setSamplingFrequency (sampling_freq);
  
  int fd;
  if ((fd = open (filename, O_RDONLY | OUR_O_LARGEFILE)) < 0){
    perror (filename);
    throw std::runtime_error ("can't open file");
  }

  if ((d_fp = fdopen (fd, "rb")) == NULL){
    perror (filename);
    throw std::runtime_error ("can't open file");
  }
}

GrFileSource::~GrFileSource ()
{
  fclose ((FILE *) d_fp);
}

  
int 
GrFileSource::work2 (VrSampleRange output, void *ao[])
{
  char *o = ((char **) ao)[0];
  int i;
  int size = output.size;

  sync (output.index);
  
  while (size) {
    i = fread(o, d_itemsize, size, (FILE *) d_fp);
    
    size -= i;
    o += i * d_itemsize;

    if (size == 0)		// done
      break;

    if (i > 0)			// short read, try again
      continue;

    // We got a zero from fread.  This is either EOF or error.  In
    // any event, if we're in repeat mode, seek back to the beginning
    // of the file and try again, else break

    if (!d_repeat)
      break;

    if (fseek ((FILE *) d_fp, 0, SEEK_SET) == -1) {
      fprintf(stderr, "[%s] fseek failed\n", __FILE__);
      exit(-1);
    }
  }

  if (size > 0){	// EOF or error
    cerr << "end of file, throwing\n";
    throw std::runtime_error ("end of file");
  }

  return output.size;
}
