#include "TVAudioReceiver.h"

#include <VrAR5000Source.h>
#include <VrComplexFIRfilter.h>
#include <VrQuadratureDemod.h>
#include <VrRealFIRfilter.h>
#include <VrAudioSink.h>

#include <qlayout.h>
#include <qlabel.h>
#include <qpushbutton.h>
#include <qlcdnumber.h>
#include <qtimer.h>

#include "TVTuner.h"

const float freqOffset = 0.0;

const int inputRate = 33000000;
const float inputFreq = 10720000;

const int CFIRdecimate = 825;
const int chanTaps = 800;
const float chanGain = 2.0;

const float demodGain = 2200;

const int RFIRdecimate = 5;
const int ifTaps = 20;
const float ifGain = 1.0;

const int quadRate = inputRate / CFIRdecimate;
const int audioRate = quadRate / RFIRdecimate;

TVAudioReceiver::TVAudioReceiver(VrAR5000Source<char> *source,
				 QWidget *parent, const char *name) :
  QWidget(parent, name), 
  source(source),
  volume(1)
{
  chanFilter = new VrComplexFIRfilter<char>(CFIRdecimate, chanTaps, 
					    inputFreq + freqOffset,
					    chanGain);
  demod = new VrQuadratureDemod<float>(demodGain * volume);
  ifFilter = new VrRealFIRfilter<float,short>(RFIRdecimate, audioRate/2,
					      ifTaps, ifGain);
  audio = new VrAudioSink<short>();

  CONNECT(audio, ifFilter, audioRate, 16);
  CONNECT(ifFilter, demod, quadRate, 32);
  CONNECT(demod, chanFilter, quadRate, 64);
  CONNECT(chanFilter, source, inputRate, 8);

  audio->setup();

  
  QBoxLayout *layout = new QHBoxLayout(this);
  
  QBoxLayout *vLayout = new QVBoxLayout();
  layout->addLayout(vLayout, 1);

  QLabel *vLabel = new QLabel("-  Volume  +", this);
  vLabel->setAlignment(AlignCenter);
  vLabel->setMinimumSize(vLabel->sizeHint() + QSize(20,0));
  vLayout->addWidget(vLabel, 1);

  QBoxLayout *vButtons = new QHBoxLayout();
  vLayout->addLayout(vButtons, 1);
  
  QButton *volDown = new QPushButton(this);
  volDown->setAutoRepeat(true);
  vButtons->addWidget(volDown, 1);

  QButton *volUp = new QPushButton(this);
  volUp->setAutoRepeat(true);
  vButtons->addWidget(volUp, 1);

  QBoxLayout *cLayout = new QVBoxLayout();
  layout->addLayout(cLayout, 1);
  
  QLabel *cLabel = new QLabel("-  Channel  +", this);
  cLabel->setAlignment(AlignCenter);
  cLabel->setMinimumSize(cLabel->sizeHint() + QSize(20,0));
  cLayout->addWidget(cLabel, 1);

  QBoxLayout *cButtons = new QHBoxLayout();
  cLayout->addLayout(cButtons, 1);
  
  QButton *chanDown = new QPushButton(this);
  chanDown->setAutoRepeat(true);
  cButtons->addWidget(chanDown, 1);

  QButton *chanUp = new QPushButton(this);
  chanUp->setAutoRepeat(true);
  cButtons->addWidget(chanUp, 1);
  
  const QColorGroup &c = this->colorGroup();
  QColorGroup colors(green, black, c.light(), 
                     c.dark(), c.mid(), green, c.base());
  QPalette palette(colors, colors, colors);

  display = new QLCDNumber(2, this);
  display->setMinimumSize(100, 60);
  display->setPalette(palette);
  display->setFrameStyle(QFrame::Panel | QFrame::Sunken);  
  display->setSegmentStyle(QLCDNumber::Flat);
  display->setLineWidth(3);
  layout->addWidget(display);

  layout->freeze();

  tuner = new TVTuner(this);
  tuner->setWrap(true);

  QTimer *timer = new QTimer(this);

  connect(tuner, SIGNAL(channelChanged(int)), SLOT(setChannel(int)));
  connect(chanUp, SIGNAL(clicked()), tuner, SLOT(nextChannel()));
  connect(chanDown, SIGNAL(clicked()), tuner, SLOT(prevChannel()));
  connect(volUp, SIGNAL(clicked()), SLOT(incVolume()));
  connect(volDown, SIGNAL(clicked()), SLOT(decVolume()));
  connect(timer, SIGNAL(timeout()), SLOT(run()));

  setChannel(tuner->channel());

  timer->start(10);

}

TVAudioReceiver::~TVAudioReceiver()
{
  delete audio;
  delete ifFilter;
  delete demod;
  delete chanFilter;
}

void
TVAudioReceiver::setChannel(int channel)
{
  tuner->setChannel(channel);
  source->setRxFrequency(tuner->audioFrequency() + freqOffset);
  display->display(tuner->channel());
}

void
TVAudioReceiver::incVolume()
{
  if (volume < 10)
    demod->setGain(++volume * demodGain);
}

void
TVAudioReceiver::decVolume()
{
  if (volume > 0)
    demod->setGain(--volume * demodGain);
}

void
TVAudioReceiver::run()
{
  // audio->start(0.1);
}

