/* -*- c++ -*- */
/*
 * Copyright 2004 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <usrp1_source_base.h>
#include <gr_io_signature.h>
#include <usrp_standard.h>
#include <assert.h>

static const int OUTPUT_MULTIPLE_BYTES = 4 * 1024;

usrp1_source_base::usrp1_source_base (const std::string &name,
				      gr_io_signature_sptr output_signature,
				      int which_board,
				      unsigned int decim_rate,
				      int mode
				      ) throw (std::runtime_error)
  : gr_sync_block (name,
		   gr_make_io_signature (0, 0, 0),
		   output_signature),
    d_noverruns (0)
{
  d_usrp = usrp_standard_rx::make (which_board,  decim_rate, mode);
  if (d_usrp == 0)
    throw std::runtime_error ("can't open usrp1");

  // All calls to d_usrp->read must be multiples of 512 bytes.
  // We jack this up to 4k to reduce overhead.

  set_output_multiple (OUTPUT_MULTIPLE_BYTES / output_signature->sizeof_stream_item (0));
}

usrp1_source_base::~usrp1_source_base ()
{
  delete d_usrp;
}

int
usrp1_source_base::work (int noutput_items,
			 gr_vector_const_void_star &input_items,
			 gr_vector_void_star &output_items)
{
  static const int BUFSIZE = 4 * OUTPUT_MULTIPLE_BYTES;
  unsigned char buf[BUFSIZE];
  int output_index = 0;
  int output_items_produced;
  int bytes_read;
  bool overrun;

  while (output_index < noutput_items){
    int nbytes = ninput_bytes_reqd_for_noutput_items (noutput_items - output_index);
    nbytes = std::min (nbytes, BUFSIZE);

    int result_nbytes = d_usrp->read (buf, nbytes, &overrun);
    if (overrun){
      fprintf (stderr, "usrp1_source: overrun\n");
      d_noverruns++;
    }

    if (result_nbytes < 0)	// We've got a problem.  Usually board unplugged or powered down.
      return -1;		// Indicate we're done.

    if (result_nbytes != nbytes){	// not really an error, but unexpected
      fprintf (stderr, "usrp1_source: short read.  Expected %d, got %d\n",
	       nbytes, result_nbytes);
    }

    copy_from_usrp_buffer (output_items,
			   output_index,
			   noutput_items - output_index,   // output_items_available
			   output_items_produced,	   // [out]
			   buf,				   // usrp_buffer
			   result_nbytes,		   // usrp_buffer_length
			   bytes_read);			   // [out]

    assert (output_index + output_items_produced <= noutput_items);
    assert (bytes_read == result_nbytes);

    output_index += output_items_produced;
  }

  return noutput_items;
}


bool
usrp1_source_base::set_decim_rate (unsigned int rate)
{
  return d_usrp->set_decim_rate (rate);
}

bool
usrp1_source_base::set_rx_freq (int channel, double freq)
{
  return d_usrp->set_rx_freq (channel, freq);
}

long
usrp1_source_base::adc_freq () const
{
  return d_usrp->adc_freq ();
}

unsigned int
usrp1_source_base::decim_rate () const
{
  return d_usrp->decim_rate ();
}

double
usrp1_source_base::rx_freq (int channel) const
{
  return d_usrp->rx_freq (channel);
}

bool
usrp1_source_base::set_fpga_mode (int mode)
{
  return d_usrp->set_fpga_mode (mode);
}

void
usrp1_source_base::set_verbose (bool verbose)
{  
  d_usrp->set_verbose (verbose);
}

bool
usrp1_source_base::write_aux_dac (int slot, int which_dac, int value)
{
  return d_usrp->write_aux_dac (slot, which_dac, value);
}

int
usrp1_source_base::read_aux_adc (int slot, int which_adc)
{
  return d_usrp->read_aux_adc (slot, which_adc);
}

bool
usrp1_source_base::write_eeprom (int i2c_addr, int eeprom_offset, const std::string buf)
{
  return d_usrp->write_eeprom (i2c_addr, eeprom_offset, buf);
}

std::string
usrp1_source_base::read_eeprom (int i2c_addr, int eeprom_offset, int len)
{
  return d_usrp->read_eeprom (i2c_addr, eeprom_offset, len);
}

// internal routines...

bool
usrp1_source_base::_write_fpga_reg (int regno, int value)
{
  return d_usrp->_write_fpga_reg (regno, value);
}

int
usrp1_source_base::_read_fpga_reg (int regno)
{
  return d_usrp->_read_fpga_reg (regno);
}
