/* -*- Mode: Java; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*- 
 * $Id: LocalField.java,v 1.2 2000/09/09 19:51:48 metlov Exp $
 *
 * This file is part of the Java Expressions Library (JEL).
 *   For more information about JEL visit :
 *    http://galaxy.fzu.cz/JEL/
 *
 * (c) 1998 -- 2000 by Konstantin Metlov(metlov@fzu.cz);
 *
 * JEL is Distributed under the terms of GNU General Public License.
 *    This code comes with ABSOLUTELY NO WARRANTY.
 *  For license details see COPYING file in this directory.
 */

package gnu.jel.reflect;

import gnu.jel.debug.Debug;

/**
 * Represents a field local to the class being compiled.
 */
public class LocalField implements Member {
  protected int modifiers;
  private String name;
  private Class type;
  private int typeID;
  private Object constValue;

  /**
   * Constructs a new local field.
   * @param modifiers field modifiers, a sum of one or more of <TT>PUBLIC</TT>,
   *                <TT>PRIVATE</TT>,<TT>PROTECTED</TT>, <TT>STATIC</TT>,
   *                <TT>FINAL</TT>,<TT>VOLATILE</TT>, <TT>TRANSIENT</TT> 
   *                constants defined in java.lang.reflect.Modifier
   * @param type is a class representing the type of this field.
   * @param name is the name of this field.
   * @param constValue is the value of this field if it is static final,
   *                   <TT>null</TT> otherwise.
   */
  public LocalField(int modifiers, Class type, String name, Object constValue){
	this.type=type;
	this.typeID=gnu.jel.TypesStack.typeID(type);
	this.name=name;

    if (Debug.enabled)
      Debug.assert((constValue==null) || ((modifiers & 0x0018) ==0x0018));

	this.modifiers=(modifiers & 0x0000FFFF) | 0x00010000; // marks as field
    this.constValue=constValue;
  };

  public Class getDeclaringClass() {
  	return null; // means local field
  };

  public String getName() {
	return name;
  };

  public int getModifiers() {
	return modifiers;
  };
    
  public Class getType() {
	return type;
  };

  public int getTypeID() {
	return typeID;
  };

  public Class[] getParameterTypes() {
	return new Class[0];
  };

  public Object eval(Object obj,Object[] params) throws Exception {
      throw new Exception(); // Can't eval
  };

  public boolean equals(Object o) {
    if (!(o instanceof LocalField)) return false;
    LocalField obj=(LocalField)o;

    Class dc=getDeclaringClass();
    Class odc=obj.getDeclaringClass();
    return name.equals(obj.name) && 
      ((dc==odc) || ((dc!=null) && dc.equals(odc)));
  };

  /**
   * Generates the code for loading this field to stack.
   * <P>For field/variable invoke means to get value. No type checks are
   * performed.
   * @param cf classfile to append instructions to.
   */
  public void code(gnu.jel.ClassFile cf) {
    if ((modifiers & 0x0008)>0)
      cf.code(0xb2); //   |   getstatic
    else
      cf.code(0xb4); //   |   getfield
    
    cf.codeI(cf.getIndex(this,12));
  };
   
};
