/*  Motti -- a strategy game
    Copyright (C) 1999 Free Software Foundation

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
*/

#include <config.h>

#include "curses_if.h"
#include "map.h"
#include "occupy.h"

#ifdef HAVE_NCURSES_H
#include <ncurses.h>
#else
#include <curses.h>
#endif

#include <stdlib.h>

enum colors {
  COL_SEA = 1,	/* Sea.  */
  COL_ME,	/* Player in turn.  */
  COL_OTHER,	/* Other players.  */
  /* TODO: change invade routines to support these two.  */
  COL_ENC,	/* Last encircled and occupied cells.  */
  COL_O_ENC,	/* Last encircled cells in other players' control.  */
  COL_CAP,	/* Color of capital.  */
  COL_O_CAP,	/* Color of other players' capitals.  */
  COL_OCC,	/* Color of occupied areas.  */
  COL_O_OCC	/* Color of other players' occupied areas.  */
};

enum refresh_amount {NONE, CURSOR, STATUSLINE, REFRESH_MAP, ALL};

static void kill_curses (void);
static chtype choose_color (int);
static chtype first_char (int);
static chtype second_char (map_val);
static void draw_loc (const int);
static void draw_map (void);
static void draw_statusline (void);
static enum refresh_amount move_curs (int, int);
static enum refresh_amount resolve_update (Action);
static enum refresh_amount read_char (void);
static void game_loop (void);

static bool use_color = FALSE;
static WINDOW *map_pad, *status_line;

static int map_x, map_y;
static int map_max_x, map_max_y;
static int curs_x, curs_y;

extern void
curses_game_init ()
{
  const chtype turn_text[] = {'t', 'u', 'r', 'n', '\0'};
  const chtype cross_text[] = {'c', 'r', 'o', 's', 's', 'e', 's', '\0'};

  atexit (kill_curses);
  initscr ();
  cbreak ();
  noecho ();
  nonl ();
  intrflush (stdscr, FALSE);

  use_color = TRUE;
  if (use_color = has_colors ())
    {
      start_color ();
      /* Hard-coded color values here.  */
      init_pair (COL_SEA, COLOR_BLUE, COLOR_BLACK);
      init_pair (COL_ME, COLOR_YELLOW, COLOR_BLACK);
      init_pair (COL_OTHER, COLOR_WHITE, COLOR_BLACK);
      /* These two are yet unimplemented.  */
      init_pair (COL_ENC, COLOR_RED, COLOR_BLACK);
      init_pair (COL_O_ENC, COLOR_MAGENTA, COLOR_BLACK);
      init_pair (COL_CAP, COLOR_CYAN, COLOR_BLUE);
      init_pair (COL_O_CAP, COLOR_WHITE, COLOR_BLUE);
      init_pair (COL_OCC, COLOR_CYAN, COLOR_MAGENTA);
      init_pair (COL_O_OCC, COLOR_WHITE, COLOR_RED);
    }
  map_pad = newpad (game_map.height, game_map.width*2);
  status_line = newwin (1, COLS, 0, 0);
  keypad (map_pad, TRUE);
  mvwaddchnstr (status_line, 0, (COLS - 8), turn_text, 4);
  mvwaddchnstr (status_line, 0, (COLS - 24), cross_text, 7);

  game_loop ();
}

static void
kill_curses ()
{
  endwin ();
}

static chtype
choose_color (loc)
     int loc;
{
  if (game_map.map[loc] == SEA_VAL)
    return COLOR_PAIR (COL_SEA);
  if ((game_map.map[loc] & MASK_PLAYER) == game_map.turn)
    {
      if (game_map.map[loc] & MASK_CAPITAL)
	return COLOR_PAIR (COL_CAP);
      else if (game_map.map[loc] & MASK_OCCUPIED)
	return COLOR_PAIR (COL_OCC);
      else
	return COLOR_PAIR (COL_ME);
    }
  else
    {
      if (game_map.map[loc] & MASK_CAPITAL)
	return COLOR_PAIR (COL_O_CAP);
      else if (game_map.map[loc] & MASK_OCCUPIED)
	return COLOR_PAIR (COL_O_OCC);
      else
	return COLOR_PAIR (COL_OTHER);
    }
}

static chtype
first_char (loc)
     int loc;
{
  if (game_map.map[loc] & MASK_CAPITAL)
    {
      if (game_map.map[loc] & MASK_CROSS)
	return 'X';
      else
	return '!';
    }
  else if (game_map.map[loc] & MASK_CROSS)
    return 'x';
  return ' ' | A_BOLD;
}

static chtype
second_char (val)
     map_val val;
{
  int pl, occ;
  occ = val & MASK_OCCUPIED;
  pl = val & MASK_PLAYER;

  if (pl == SEA_VAL)
    return '.';
  return 'a'-1+pl - (occ ? 32 : 0);
}

static void
draw_loc (i)
     const int i;
{
  chtype color = 0;

  if (use_color)
    color = choose_color (i);
  if (game_map.map[i] & MASK_OCCUPIED)
    color |= A_BOLD;
  waddch (map_pad, first_char (i) | color);
  waddch (map_pad, second_char (game_map.map[i]) | color);
}

/* TODO: maybe some optimization here?  */
static void
draw_map ()
{
  register int i;

  wmove (map_pad, 0, 0);
  for (i = 0 ; i < game_map.size; i++)
    draw_loc (i);
}

static void
draw_statusline ()
{
  char update;
  static char last_mode;
  update = need_update (&last_mode);

  if (update & MODE_ATT)
    {
      chtype att[] = {'a' | A_NORMAL, 't', 't', 'a', 'c', 'k', '\0'};
      chtype noact[] = {' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ',
			' ', '\0'};
      if (game_map.def_mode == MODE_ATT)
	attrset (A_STANDOUT);
      if (game_map.modes & MODE_ATT)
	{
	  mvwaddchnstr (status_line, 0, 0, att, 6);
	  attrset (A_NORMAL);
	  mvwaddch (status_line, 0, 8, ('6'-game_map.n_att) | A_BOLD);
	}
      else
	mvwaddchstr (status_line, 0, 0, noact);
    }

  if (update & MODE_DEF)
    {
      chtype def[] = {'d' | A_NORMAL, 'e', 'f', 'e', 'n', 'd', '\0'};
      chtype noact[] = {' ', ' ', ' ', ' ', ' ', ' ', '\0'};
      if (game_map.def_mode == MODE_DEF)
	attrset (A_STANDOUT);
      mvwaddchnstr (status_line, 0, 16, (game_map.modes & MODE_DEF) ?
		    def : noact, 6);
      attrset (A_NORMAL);
    }

  if (update & MODE_GUE)
    {
      chtype gue[] = {'g' | A_NORMAL, 'u', 'e', 'r', 'i', 'l', 'l',
		      'a', '\0'};
      chtype noact[] = {' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', '\0'};
      if (game_map.def_mode == MODE_GUE)
	attrset (A_STANDOUT);
      mvwaddchnstr (status_line, 0, 32, (game_map.modes & MODE_GUE) ?
		    gue : noact, 8);
      attrset (A_NORMAL);
    }

  mvwaddch (status_line, 0, COLS-16, ('0'+game_map.n_cross) | A_BOLD);
  if (update & UPDATE_TURN)
    mvwaddch (status_line, 0, COLS-1, (second_char (game_map.turn |
						    MASK_OCCUPIED) |
				       A_BOLD));
}

/* TODO: a routine to scroll wrapped maps.  Low priority, to do when I
   feel like it.  */
static enum refresh_amount
move_curs (x, y)
     int x, y;
{
  enum refresh_amount refresh = CURSOR;

  if (x)
    { 
      /* Suicidal programming: use unsigned, if 0-1, result is greater
	 than game_map.[width|height].  */
      unsigned int new_x = curs_x+x;
      if (new_x < game_map.width)
	{
	  int disp_x = new_x-map_x;
	  curs_x = new_x;
	  if (disp_x < 0)
	    {
	      map_x -= map_x >= 4 ? 4 : map_x;
	      refresh = ALL;
	    }
	  else if (disp_x >= COLS/2)
	    {
	      register int dist_edge = map_max_x-map_x;
	      map_x += dist_edge >= 4 ? 4 : dist_edge;
	      refresh = ALL;
	    }
	}
    }
  if (y)
    {
      unsigned int new_y = curs_y+y;
      if (new_y < game_map.height)
	{
	  int disp_y = new_y-map_y;
	  curs_y = new_y;
	  if (disp_y < 0)
	    {
	      map_y -= map_y >= 4 ? 4 : map_y;
	      refresh = ALL;
	    }
	  else if (disp_y >= LINES-1)
	    {
	      register int dist_edge = map_max_y-map_y;
	      map_y += abs (dist_edge) >= 4 ? 4 : dist_edge;
	      refresh = ALL;
	    }
	}
    }
  return refresh;
}

static enum refresh_amount
resolve_update (act)
     Action act;
{
  if (act.type == 0)
    return NONE;
  if (!(act.type & (EVENT_ENCIRCLEMENT|EVENT_NEWTURN)))
    {
      if (act.count == 0)
	return STATUSLINE;
      while (act.count-- > 0)
	{
	  wmove (map_pad, act.loc[act.count].y,
		 act.loc[act.count].x+act.loc[act.count].x);
	  draw_loc (parse_real_coord (act.loc[act.count]));
	}
      return REFRESH_MAP;
    }
  return ALL;
}

/* TODO: User-defineable keys.  */
static enum refresh_amount
read_char ()
{
  int key;

  while (1)
    {
      while ((key = wgetch (map_pad)) == ERR);
      switch (key)
	{
	  Action act;
	  Coord cross_loc;
	case KEY_A1:
	  return move_curs (-1, -1);
	case KEY_UP:
	  return move_curs (0, -1);
	case KEY_A3:
	  return move_curs (1, -1);
	case KEY_RIGHT:
	  return move_curs (1, 0);
	case KEY_C3:
	  return move_curs (1, 1);
	case KEY_DOWN:
	  return move_curs (0, 1);
	case KEY_C1:
	  return move_curs (-1, 1);
	case KEY_LEFT:
	  return move_curs (-1, 0);
	case KEY_B2:
	case ' ':
	  cross_loc.x = curs_x % game_map.width;
	  cross_loc.y = curs_y % game_map.height;
	  if (toggle_cross (cross_loc))
	    {
	      Coord curs_coord;
	      curs_coord.x = curs_x;
	      curs_coord.y = curs_y;
	      wmove (map_pad, curs_y, curs_x+curs_x);
	      draw_loc (parse_coord (curs_coord));
	      return REFRESH_MAP;
	    }
	  else
	    return NONE;
	case 'z':
	case KEY_F(1):
	  action (MODE_ATT, &act);
	  return resolve_update (act);
	case 'x':
	case KEY_F(2):
	  action (MODE_DEF, &act);
	  return resolve_update (act);
	case 'c':
	case KEY_F(3):
	  action (MODE_GUE, &act);
	  return resolve_update (act);
	case 12:	/* C-l pressed.  */
	  return ALL;
	}
    }
}

static void
game_loop ()
{
  enum refresh_amount refresh = ALL;
  int map_physical_right, map_physical_bottom;

  /* No need to worry about these having negative values.  */
  map_max_x = game_map.width - (COLS/2-1);
  map_max_y = game_map.height - (LINES-1);
  map_physical_right = COLS/2-1 < game_map.width ? COLS :
    (game_map.width+game_map.width);
  map_physical_bottom = LINES-1 < game_map.height ? LINES :
    (game_map.height+game_map.height);

  refresh ();			/* Bogus refresh.  */
  while (1)
    {
      switch (refresh)
	{
	case ALL:
	  draw_map ();
	case REFRESH_MAP:
	  pnoutrefresh (map_pad, map_y, map_x*2, 1, 0,
			map_physical_bottom, map_physical_right);
	case STATUSLINE:
	  draw_statusline ();
	  wnoutrefresh (status_line);
	case CURSOR:
	  move (curs_y-map_y+1, (curs_x-map_x)*2);
	  refresh ();
	  doupdate ();
	}
      refresh = read_char ();
    }
}
