## Copyright (C) 2008, 2009 Bill Denney
##
## This file is part of Octave.
##
## Octave is free software; you can redistribute it and/or modify it
## under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 3 of the License, or (at
## your option) any later version.
##
## Octave is distributed in the hope that it will be useful, but
## WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
## General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with Octave; see the file COPYING.  If not, see
## <http://www.gnu.org/licenses/>.

## -*- texinfo -*-
## @deftypefn {Function File} {@var{msgstr} =} nargoutchk (@var{minargs}, @var{maxargs}, @var{nargs})
## @deftypefnx {Function File} {@var{msgstr} =} nargoutchk (@var{minargs}, @var{maxargs}, @var{nargs}, "string")
## @deftypefnx {Function File} {@var{msgstruct} =} nargoutchk (@var{minargs}, @var{maxargs}, @var{nargs}, "struct")
## Return an appropriate error message string (or structure) if the
## number of outputs requested is invalid.
##
## This is useful for checking to see that the number of output
## arguments supplied to a function is within an acceptable range.
## @seealso{nargchk, error, nargout, nargin}
## @end deftypefn

## Author: Bill Denney <bill@denney.ws>

function msg = nargoutchk (mina, maxa, narg, outtype)

  if (nargin < 3 || nargin > 4)
    print_usage ();
  elseif (mina > maxa)
    error ("nargoutchk: minargs must be <= maxargs");
  elseif (nargin == 3)
    outtype = "string";
  elseif (! any (strcmpi (outtype, {"string" "struct"})))
    error ("nargoutchk: output type must be either string or struct");
  elseif (! (isscalar (mina) && isscalar (maxa) && isscalar (narg)))
    error ("nargoutchk: mina, maxa, and narg must be scalars");
  endif

  msg = struct ("message", "", "identifier", "");
  if (narg < mina)
    msg.message = "not enough output arguments";
    msg.identifier = "Octave:nargoutchk:not-enough-outputs";
  elseif (narg > maxa)
    msg.message = "too many output arguments";
    msg.identifier = "Octave:nargoutchk:too-many-outputs";
  endif

  if (strcmpi (outtype, "string"))
    msg = msg.message;
  else
    if (isempty (msg.message))
      msg = struct ([]);
    endif
    ## FIXME: remove the error below if error is modified to accept
    ## struct inputs
    error ("nargoutchk: error does not yet support struct inputs");
  endif

endfunction

## Tests
%!shared stmin, stmax
%!  stmin = struct ("message", "not enough output arguments",
%!                  "identifier", "Octave:nargoutchk:not-enough-outputs");
%!  stmax = struct ("message", "too many output arguments",
%!                  "identifier", "Octave:nargoutchk:too-many-outputs");
%!assert (nargoutchk (0, 1, 0), "")
%!assert (nargoutchk (0, 1, 1), "")
%!assert (nargoutchk (1, 1, 0), "not enough output arguments")
%!assert (nargoutchk (0, 1, 2), "too many output arguments")
%!assert (nargoutchk (0, 1, 2, "string"), "too many output arguments")
## Struct outputs
#%!assert (nargoutchk (0, 1, 0, "struct"), struct([]))
#%!assert (nargoutchk (0, 1, 1, "struct"), struct([]))
#%!assert (nargoutchk (1, 1, 0, "struct"), stmin)
#%!assert (nargoutchk (0, 1, 2, "struct"), stmax)
