--
-- This file is part of TALER
-- Copyright (C) 2014--2025 Taler Systems SA
--
-- TALER is free software; you can redistribute it and/or modify it under the
-- terms of the GNU General Public License as published by the Free Software
-- Foundation; either version 3, or (at your option) any later version.
--
-- TALER is distributed in the hope that it will be useful, but WITHOUT ANY
-- WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
-- A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License along with
-- TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
--

CREATE OR REPLACE FUNCTION random_bytea(
  bytea_length INT
)
RETURNS BYTEA
  AS $body$
  SELECT decode(string_agg(lpad(to_hex(width_bucket(random(), 0, 1, 256)-1),2,'0') ,''), 'hex')
    FROM generate_series(1, $1);
  $body$
LANGUAGE 'sql'
VOLATILE;


CREATE FUNCTION create_table_wire_targets(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(wire_target_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',wire_target_h_payto BYTEA PRIMARY KEY CHECK (LENGTH(wire_target_h_payto)=32)'
      ',payto_uri TEXT NOT NULL'
      ',access_token BYTEA CHECK(LENGTH(access_token)=32)'
      ' DEFAULT random_bytea(32)'
      ',target_pub BYTEA CHECK(LENGTH(target_pub)=32) DEFAULT NULL'
      ',h_normalized_payto BYTEA CHECK(LENGTH(h_normalized_payto)=32) DEFAULT NULL'
      ',aml_program_lock_timeout INT8 DEFAULT NULL'
    ') %s ;'
    ,'wire_targets'
    ,'PARTITION BY HASH (wire_target_h_payto)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'All senders and recipients of money via the exchange'
    ,'wire_targets'
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Full payto URI. Can identify a regular bank account, or also be a URI identifying a reserve-account (for P2P payments)'
    ,'payto_uri'
    ,'wire_targets'
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Unsalted hash of (full) payto_uri'
    ,'wire_target_h_payto'
    ,'wire_targets'
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'high-entropy random value that is used as a bearer token used to authenticate access to the KYC SPA and its state (without requiring a signature)'
    ,'access_token'
    ,'wire_targets'
    ,NULL
  );
  PERFORM comment_partitioned_column(
     'Public key of a merchant instance or reserve to authenticate access; NULL if KYC is not allowed for the account (if there was no incoming KYC wire transfer yet); updated, thus NOT available to the auditor'
    ,'target_pub'
    ,'wire_targets'
    ,NULL
  );
  PERFORM comment_partitioned_column(
     'hash over the normalized payto URI for this account; used for KYC operations; NULL if not available (due to DB migration not initializing this value)'
    ,'h_normalized_payto'
    ,'wire_targets'
    ,NULL
  );
  PERFORM comment_partitioned_column(
     'If non-NULL, an AML program should be running and it holds a lock on this account, thus other AML programs should not be started concurrently. Given the possibility of crashes, the lock automatically expires at the time value given in this column. At that time, the lock can be considered stale.'
    ,'aml_program_lock_timeout'
    ,'wire_targets'
    ,NULL
  );
END $$;


CREATE FUNCTION constrain_table_wire_targets(
  IN partition_suffix TEXT
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'wire_targets';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_wire_target_serial_id_key'
    ' UNIQUE (wire_target_serial_id)'
  );
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_wire_target_access_token_unique'
    ' UNIQUE (access_token)'
  );
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_normalized_h_payto_index '
    'ON ' || table_name || ' '
    '(h_normalized_payto);'
  );
END
$$;


INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('wire_targets'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('wire_targets'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE);
