/*
  This file is part of TALER
  Copyright (C) 2014-2022 Taler Systems SA

  TALER is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as
  published by the Free Software Foundation; either version 3, or
  (at your option) any later version.

  TALER is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public
  License along with TALER; see the file COPYING.  If not, see
  <http://www.gnu.org/licenses/>
*/
/**
 * @file testing/testing_api_cmd_recoup_refresh.c
 * @brief Implement the /recoup-refresh test command.
 * @author Marcello Stanisci
 */
#include "platform.h"
#include "taler_json_lib.h"
#include <gnunet/gnunet_curl_lib.h>
#include "taler_testing_lib.h"


/**
 * State for a "pay back" CMD.
 */
struct RecoupRefreshState
{
  /**
   * Expected HTTP status code.
   */
  unsigned int expected_response_code;

  /**
   * Command that offers a reserve private key,
   * plus a coin to be paid back.
   */
  const char *coin_reference;

  /**
   * Entry in the old coin's history generated by this operation.
   */
  struct TALER_EXCHANGE_CoinHistoryEntry che_old;

  /**
   * Entry in the recouped coin's history generated by this operation.
   */
  struct TALER_EXCHANGE_CoinHistoryEntry che_new;

  /**
   * Public key of the refunded coin.
   */
  struct TALER_CoinSpendPublicKeyP coin_pub_old;

  /**
   * Public key of the refunded coin.
   */
  struct TALER_CoinSpendPublicKeyP coin_pub_new;

  /**
   * Amount to be recouped.
   */
  struct TALER_Amount amount;

  /**
   * The interpreter state.
   */
  struct TALER_TESTING_Interpreter *is;

  /**
   * Handle to the ongoing operation.
   */
  struct TALER_EXCHANGE_RecoupRefreshHandle *ph;

  /**
   * NULL if coin was not refreshed, otherwise reference
   * to the melt operation underlying @a coin_reference.
   */
  const char *melt_reference;

};


/**
 * Check the result of the recoup_refresh request: checks whether
 * the HTTP response code is good, and that the coin that
 * was paid back belonged to the right old coin.
 *
 * @param cls closure
 * @param rrr response details
 */
static void
recoup_refresh_cb (void *cls,
                   const struct TALER_EXCHANGE_RecoupRefreshResponse *rrr)
{
  struct RecoupRefreshState *rrs = cls;
  const struct TALER_EXCHANGE_HttpResponse *hr = &rrr->hr;
  struct TALER_TESTING_Interpreter *is = rrs->is;
  char *cref;
  unsigned int idx;

  rrs->ph = NULL;
  if (rrs->expected_response_code != hr->http_status)
  {
    TALER_TESTING_unexpected_status (is,
                                     hr->http_status,
                                     rrs->expected_response_code);
    return;
  }

  if (GNUNET_OK !=
      TALER_TESTING_parse_coin_reference (
        rrs->coin_reference,
        &cref,
        &idx))
  {
    TALER_TESTING_interpreter_fail (is);
    return;
  }
  (void) idx; /* do NOT use! We ignore 'idx', must be 0 for melt! */

  GNUNET_free (cref);
  switch (hr->http_status)
  {
  case MHD_HTTP_OK:
    /* check old_coin_pub */
    {
      const struct TALER_TESTING_Command *melt_cmd;
      const struct TALER_CoinSpendPrivateKeyP *dirty_priv;
      struct TALER_CoinSpendPublicKeyP oc;

      melt_cmd = TALER_TESTING_interpreter_lookup_command (is,
                                                           rrs->melt_reference);
      if (NULL == melt_cmd)
      {
        GNUNET_break (0);
        TALER_TESTING_interpreter_fail (is);
        return;
      }
      if (GNUNET_OK !=
          TALER_TESTING_get_trait_coin_priv (melt_cmd,
                                             0,
                                             &dirty_priv))
      {
        GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                    "Coin %u not found in command %s\n",
                    0,
                    rrs->melt_reference);
        GNUNET_break (0);
        TALER_TESTING_interpreter_fail (is);
        return;
      }
      GNUNET_CRYPTO_eddsa_key_get_public (&dirty_priv->eddsa_priv,
                                          &oc.eddsa_pub);
      if (0 != GNUNET_memcmp (&oc,
                              &rrr->details.ok.old_coin_pub))
      {
        GNUNET_break (0);
        TALER_TESTING_interpreter_fail (is);
        return;
      }
    }
    break;
  case MHD_HTTP_NOT_FOUND:
    break;
  case MHD_HTTP_CONFLICT:
    break;
  default:
    GNUNET_log (GNUNET_ERROR_TYPE_WARNING,
                "Unmanaged HTTP status code %u/%d.\n",
                hr->http_status,
                (int) hr->ec);
    break;
  }
  TALER_TESTING_interpreter_next (is);
}


/**
 * Run the command.
 *
 * @param cls closure.
 * @param cmd the command to execute.
 * @param is the interpreter state.
 */
static void
recoup_refresh_run (void *cls,
                    const struct TALER_TESTING_Command *cmd,
                    struct TALER_TESTING_Interpreter *is)
{
  struct RecoupRefreshState *rrs = cls;
  const struct TALER_TESTING_Command *coin_cmd;
  const struct TALER_TESTING_Command *melt_cmd;
  const struct TALER_CoinSpendPrivateKeyP *coin_priv;
  const struct TALER_CoinSpendPrivateKeyP *coin_priv_old;
  const struct TALER_EXCHANGE_DenomPublicKey *denom_pub;
  const struct TALER_DenominationSignature *coin_sig;
  const struct TALER_RefreshMasterSecretP *rplanchet;
  const struct TALER_PlanchetMasterSecretP *planchet;
  const struct TALER_ExchangeBlindingValues *ewv;
  char *cref;
  unsigned int idx;
  struct TALER_DenominationHashP h_denom_pub;

  rrs->is = is;
  if (GNUNET_OK !=
      TALER_TESTING_parse_coin_reference (
        rrs->coin_reference,
        &cref,
        &idx))
  {
    TALER_TESTING_interpreter_fail (is);
    return;
  }

  coin_cmd = TALER_TESTING_interpreter_lookup_command (is,
                                                       cref);
  GNUNET_free (cref);
  if (NULL == coin_cmd)
  {
    GNUNET_break (0);
    TALER_TESTING_interpreter_fail (is);
    return;
  }
  melt_cmd = TALER_TESTING_interpreter_lookup_command (is,
                                                       rrs->melt_reference);
  if (NULL == melt_cmd)
  {
    GNUNET_break (0);
    TALER_TESTING_interpreter_fail (is);
    return;
  }
  if (GNUNET_OK !=
      TALER_TESTING_get_trait_coin_priv (coin_cmd,
                                         idx,
                                         &coin_priv))
  {
    GNUNET_break (0);
    TALER_TESTING_interpreter_fail (is);
    return;
  }
  if (GNUNET_OK !=
      TALER_TESTING_get_trait_coin_priv (melt_cmd,
                                         0,
                                         &coin_priv_old))
  {
    GNUNET_break (0);
    TALER_TESTING_interpreter_fail (is);
    return;
  }
  GNUNET_CRYPTO_eddsa_key_get_public (
    &coin_priv->eddsa_priv,
    &rrs->coin_pub_new.eddsa_pub);
  GNUNET_CRYPTO_eddsa_key_get_public (
    &coin_priv_old->eddsa_priv,
    &rrs->coin_pub_old.eddsa_pub);

  if (GNUNET_OK !=
      TALER_TESTING_get_trait_exchange_blinding_values (melt_cmd,
                                                        idx,
                                                        &ewv))
  {
    GNUNET_break (0);
    TALER_TESTING_interpreter_fail (is);
    return;
  }
  if (GNUNET_OK !=
      TALER_TESTING_get_trait_planchet_secrets (coin_cmd,
                                                idx,
                                                &planchet))
  {
    GNUNET_break (0);
    TALER_TESTING_interpreter_fail (is);
    return;
  }
  if (GNUNET_OK !=
      TALER_TESTING_get_trait_refresh_secret (melt_cmd,
                                              &rplanchet))
  {
    GNUNET_break (0);
    TALER_TESTING_interpreter_fail (is);
    return;
  }
  if (GNUNET_OK !=
      TALER_TESTING_get_trait_denom_pub (coin_cmd,
                                         idx,
                                         &denom_pub))
  {
    GNUNET_break (0);
    TALER_TESTING_interpreter_fail (is);
    return;
  }
  if (GNUNET_OK !=
      TALER_TESTING_get_trait_denom_sig (coin_cmd,
                                         idx,
                                         &coin_sig))
  {
    GNUNET_break (0);
    TALER_TESTING_interpreter_fail (is);
    return;
  }
  GNUNET_log (GNUNET_ERROR_TYPE_INFO,
              "Trying to recoup_refresh denomination '%s'\n",
              TALER_B2S (&denom_pub->h_key));
  rrs->che_old.type
    = TALER_EXCHANGE_CTT_OLD_COIN_RECOUP;
  rrs->che_old.amount
    = rrs->amount;
  rrs->che_old.details.old_coin_recoup.new_coin_pub
    = rrs->coin_pub_new;
  rrs->che_new.type
    = TALER_EXCHANGE_CTT_RECOUP_REFRESH;
  rrs->che_new.amount
    = rrs->amount;
  rrs->che_new.details.recoup_refresh.old_coin_pub
    = rrs->coin_pub_old;
  TALER_planchet_blinding_secret_create (
    planchet,
    ewv,
    &rrs->che_new.details.recoup_refresh.coin_bks);
  TALER_denom_pub_hash (&denom_pub->key,
                        &h_denom_pub);
  TALER_wallet_recoup_refresh_sign (
    &h_denom_pub,
    &rrs->che_new.details.recoup_refresh.coin_bks,
    coin_priv,
    &rrs->che_new.details.recoup_refresh.coin_sig);
  rrs->ph = TALER_EXCHANGE_recoup_refresh (
    TALER_TESTING_interpreter_get_context (is),
    TALER_TESTING_get_exchange_url (is),
    TALER_TESTING_get_keys (is),
    denom_pub,
    coin_sig,
    ewv,
    rplanchet,
    planchet,
    idx,
    &recoup_refresh_cb,
    rrs);
  GNUNET_assert (NULL != rrs->ph);
}


/**
 * Cleanup the "recoup_refresh" CMD state, and possibly cancel
 * a pending operation thereof.
 *
 * @param cls closure.
 * @param cmd the command which is being cleaned up.
 */
static void
recoup_refresh_cleanup (void *cls,
                        const struct TALER_TESTING_Command *cmd)
{
  struct RecoupRefreshState *rrs = cls;
  if (NULL != rrs->ph)
  {
    TALER_EXCHANGE_recoup_refresh_cancel (rrs->ph);
    rrs->ph = NULL;
  }
  GNUNET_free (rrs);
}


/**
 * Offer internal data from a "recoup-refresh" CMD state to other
 * commands.
 *
 * @param cls closure
 * @param[out] ret result (could be anything)
 * @param trait name of the trait
 * @param index index number of the object to offer.
 * @return #GNUNET_OK on success
 */
static enum GNUNET_GenericReturnValue
recoup_refresh_traits (void *cls,
                       const void **ret,
                       const char *trait,
                       unsigned int index)
{
  struct RecoupRefreshState *rrs = cls;
  struct TALER_TESTING_Trait traits[] = {
    TALER_TESTING_make_trait_coin_history (0,
                                           &rrs->che_old),
    TALER_TESTING_make_trait_coin_pub (0,
                                       &rrs->coin_pub_old),
    TALER_TESTING_make_trait_coin_history (1,
                                           &rrs->che_new),
    TALER_TESTING_make_trait_coin_pub (1,
                                       &rrs->coin_pub_new),
    TALER_TESTING_trait_end ()
  };

  return TALER_TESTING_get_trait (traits,
                                  ret,
                                  trait,
                                  index);
}


struct TALER_TESTING_Command
TALER_TESTING_cmd_recoup_refresh (const char *label,
                                  unsigned int expected_response_code,
                                  const char *coin_reference,
                                  const char *melt_reference,
                                  const char *amount)
{
  struct RecoupRefreshState *rrs;

  rrs = GNUNET_new (struct RecoupRefreshState);
  rrs->expected_response_code = expected_response_code;
  rrs->coin_reference = coin_reference;
  rrs->melt_reference = melt_reference;
  if (GNUNET_OK !=
      TALER_string_to_amount (amount,
                              &rrs->amount))
  {
    GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                "Failed to parse amount `%s' at %s\n",
                amount,
                label);
    GNUNET_assert (0);
  }
  {
    struct TALER_TESTING_Command cmd = {
      .cls = rrs,
      .label = label,
      .run = &recoup_refresh_run,
      .cleanup = &recoup_refresh_cleanup,
      .traits = &recoup_refresh_traits
    };

    return cmd;
  }
}
