/*
 This file is part of GNU Taler
 (C) 2021-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 *
 * @author Sebastian Javier Marchano (sebasjm)
 */

import {
  AccessToken,
  Duration,
  HttpStatusCode,
  MerchantAuthMethod,
  opEmptySuccess,
  opFixedSuccess,
  TalerMerchantApi,
} from "@gnu-taler/taler-util";
import {
  ButtonBetterBulma,
  LocalNotificationBannerBulma,
  useChallengeHandler,
  useLocalNotificationBetter,
  useTranslationContext,
} from "@gnu-taler/web-util/browser";
import { Fragment, h, VNode } from "preact";
import { useState } from "preact/hooks";
import {
  FormErrors,
  FormProvider,
} from "../../../components/form/FormProvider.js";
import { Input } from "../../../components/form/Input.js";
import { DefaultInstanceFormFields } from "../../../components/instance/DefaultInstanceFormFields.js";
import { SolveMFAChallenges } from "../../../components/SolveMFA.js";
import { useSessionContext } from "../../../context/session.js";
import { usePreference } from "../../../hooks/preference.js";
import {
  EMAIL_REGEX,
  INSTANCE_ID_REGEX,
  PHONE_JUST_NUMBERS_REGEX,
} from "../../../utils/constants.js";
import { undefinedIfEmpty } from "../../../utils/table.js";
import { FOREVER_REFRESHABLE_TOKEN } from "../../login/index.js";

const TALER_SCREEN_ID = 25;

export type Entity = TalerMerchantApi.InstanceConfigurationMessage & {
  auth_token?: string;
  // default_pay_delay: Duration;
  // default_wire_transfer_delay: Duration;
  password: string;
  repeat: string;
};

export interface Props {
  onConfirm: () => void;
  onBack?: () => void;
  forceId?: string;
}

const twoHours = Duration.fromSpec({ hours: 2 });
const oneDay = Duration.fromSpec({ days: 1 });
const twoDays = Duration.fromSpec({ days: 2 });

function with_defaults(id: string | undefined, config: TalerMerchantApi.MerchantVersionResponse): Partial<Entity> {
  return {
    id,
    use_stefan: true,
    default_pay_delay: config.default_pay_delay ?? Duration.toTalerProtocolDuration(twoHours),
    default_refund_delay: config.default_refund_delay ?? Duration.toTalerProtocolDuration(oneDay),
    default_wire_transfer_delay: config.default_wire_transfer_delay ?? Duration.toTalerProtocolDuration(twoDays),
  };
}

export function CreatePage({ onConfirm, onBack, forceId }: Props): VNode {
  const { i18n } = useTranslationContext();
  const { state: session, lib, logIn, config } = useSessionContext();
  const [value, valueHandler] = useState(with_defaults(forceId, config));
  const [notification, safeFunctionHandler] = useLocalNotificationBetter();
  const mfa = useChallengeHandler();

  const errors = undefinedIfEmpty<FormErrors<Entity>>({
    id: !value.id
      ? i18n.str`Required`
      : !INSTANCE_ID_REGEX.test(value.id)
        ? i18n.str`Invalid`
        : undefined,
    name: !value.name ? i18n.str`Required` : undefined,
    email: !value.email
      ? undefined
      : !EMAIL_REGEX.test(value.email)
        ? i18n.str`Doesn't have the pattern of an email`
        : undefined,
    phone_number: !value.phone_number
      ? undefined
      : !value.phone_number.startsWith("+")
        ? i18n.str`Should start with +`
        : !PHONE_JUST_NUMBERS_REGEX.test(value.phone_number)
          ? i18n.str`A phone number consists of numbers only`
          : undefined,
    default_pay_delay: !value.default_pay_delay
      ? i18n.str`Required`
      : value.default_pay_delay.d_us === "forever"
        ? i18n.str`Invalid value`
        : undefined,
    default_refund_delay: !value.default_refund_delay
      ? i18n.str`Required`
      : value.default_refund_delay.d_us === "forever"
        ? i18n.str`Invalid value`
        : undefined,
    default_wire_transfer_delay: !value.default_wire_transfer_delay
      ? i18n.str`Required`
      : value.default_wire_transfer_delay.d_us === "forever"
        ? i18n.str`Invalid value`
        : undefined,
    address: undefinedIfEmpty({
      address_lines:
        value.address?.address_lines && value.address?.address_lines.length > 7
          ? i18n.str`Max 7 lines`
          : undefined,
    }),
    jurisdiction: undefinedIfEmpty({
      address_lines:
        value.address?.address_lines && value.address?.address_lines.length > 7
          ? i18n.str`Max 7 lines`
          : undefined,
    }),
    password: !value.password ? i18n.str`Required` : undefined,
    repeat: !value.repeat
      ? i18n.str`Required`
      : value.repeat !== value.password
        ? i18n.str`Doesn't match`
        : undefined,
  });

  const hasErrors = errors !== undefined;

  const data: TalerMerchantApi.InstanceConfigurationMessage = {
    ...(value as TalerMerchantApi.InstanceConfigurationMessage),
    auth: {
      method: MerchantAuthMethod.TOKEN,
      password: value.password!,
    },
  };
  const create = safeFunctionHandler(
    async (
      token: AccessToken | undefined,
      data: TalerMerchantApi.InstanceConfigurationMessage,
      challengeIds: string[],
    ) => {
      if (!data.address) data.address = {};
      if (!data.jurisdiction) data.jurisdiction = {};
      const instanceResp = await lib.instance.createInstance(token, data, {
        challengeIds,
      });
      if (instanceResp.type === "fail") return instanceResp;
      if (data.auth.password) {
        const api =
          data.id === "admin"
            ? lib.instance
            : lib.subInstanceApi(data.id).instance;
        const tokenResp = await api.createAccessToken(
          data.id,
          data.auth.password,
          FOREVER_REFRESHABLE_TOKEN(i18n.str`Instance created`),
        );
        if (tokenResp.type === "fail") return tokenResp;
        return opFixedSuccess(tokenResp.body);
      }
      return opEmptySuccess();
    },
    hasErrors ? undefined : [session.token, data, []],
  );
  create.onSuccess = (success, oldtoken, data) => {
    if (success) {
      logIn(data.id, success.access_token);
    }
    onConfirm();
  };
  create.onFail = (fail) => {
    switch (fail.case) {
      case HttpStatusCode.Accepted:
        mfa.onChallengeRequired(fail.body);
        return undefined;
      case HttpStatusCode.Unauthorized:
        return i18n.str`Unauthorized.`;
      case HttpStatusCode.Conflict:
        return i18n.str`Conflict.`;
      case HttpStatusCode.NotFound:
        return i18n.str`Not found.`;
    }
  };
  const retry = create.lambda((ids: string[]) => [
    create.args![0],
    create.args![1],
    ids,
  ]);

  if (mfa.pendingChallenge) {
    return (
      <SolveMFAChallenges
        currentChallenge={mfa.pendingChallenge}
        onCompleted={retry}
        onCancel={mfa.doCancelChallenge}
      />
    );
  }

  return (
    <div>
      <LocalNotificationBannerBulma notification={notification} />
      <section class="section is-main-section">
        <div class="columns">
          <div class="column" />
          <div class="column is-four-fifths">
            <FormProvider<Entity>
              errors={errors}
              object={value}
              valueHandler={valueHandler}
            >
              <DefaultInstanceFormFields
                readonlyId={!!forceId}
                showId={!forceId}
              />
              <Input<Entity>
                name="password"
                label={i18n.str`New password`}
                tooltip={i18n.str`Next password to be used`}
                inputType="password"
              />
              <Input<Entity>
                name="repeat"
                label={i18n.str`Repeat password`}
                tooltip={i18n.str`Confirm the same password`}
                inputType="password"
              />
              <div class="buttons is-right mt-5">
                {onBack && (
                  <button class="button" type="button" onClick={onBack}>
                    <i18n.Translate>Cancel</i18n.Translate>
                  </button>
                )}
                <ButtonBetterBulma
                  onClick={create}
                  type="submit"
                  data-tooltip={
                    hasErrors
                      ? i18n.str`Please complete the marked fields and choose authorization method`
                      : i18n.str`Confirm operation`
                  }
                >
                  <i18n.Translate>Confirm</i18n.Translate>
                </ButtonBetterBulma>
              </div>
            </FormProvider>
          </div>
          <div class="column" />
        </div>
      </section>
    </div>
  );
}
