/*
 This file is part of GNU Taler
 (C) 2022 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 *
 * @author Sebastian Javier Marchano (sebasjm)
 */

import {
  AbsoluteTime,
  AmountString,
  ProviderPaymentType,
  TalerPreciseTimestamp,
} from "@gnu-taler/taler-util";
import * as tests from "@gnu-taler/web-util/testing";
import { addDays } from "date-fns";
import {
  ShowRecoveryInfo,
  BackupView as TestedComponent,
} from "./BackupPage.js";

export default {
  title: "backup",
};

export const LotOfProviders = tests.createExample(TestedComponent, {
  providers: [
    {
      active: true,
      name: "sync.demo",
      syncProviderBaseUrl: "http://sync.taler:9967/",
      lastSuccessfulBackupTimestamp:
        TalerPreciseTimestamp.fromSeconds(1625063925),
      paymentProposalIds: [
        "43Q5WWRJPNS4SE9YKS54H9THDS94089EDGXW9EHBPN6E7M184XEG",
      ],
      paymentStatus: {
        type: ProviderPaymentType.Paid,
        paidUntil: AbsoluteTime.fromMilliseconds(1656599921000),
      },
      terms: {
        annualFee: "ARS:1" as AmountString,
        storageLimitInMegabytes: 16,
        supportedProtocolVersion: "0.0",
      },
    },
    {
      active: true,
      name: "sync.demo",
      syncProviderBaseUrl: "http://sync.taler:9967/",
      lastSuccessfulBackupTimestamp:
        TalerPreciseTimestamp.fromSeconds(1625063925),
      paymentProposalIds: [
        "43Q5WWRJPNS4SE9YKS54H9THDS94089EDGXW9EHBPN6E7M184XEG",
      ],
      paymentStatus: {
        type: ProviderPaymentType.Paid,
        paidUntil: AbsoluteTime.fromMilliseconds(
          addDays(new Date(), 13).getTime(),
        ),
      },
      terms: {
        annualFee: "ARS:1" as AmountString,
        storageLimitInMegabytes: 16,
        supportedProtocolVersion: "0.0",
      },
    },
    {
      active: false,
      name: "sync.demo",
      syncProviderBaseUrl: "http://sync.demo.taler.net/",
      paymentProposalIds: [],
      paymentStatus: {
        type: ProviderPaymentType.Pending,
        talerUri: "taler://",
      },
      terms: {
        annualFee: "KUDOS:0.1" as AmountString,
        storageLimitInMegabytes: 16,
        supportedProtocolVersion: "0.0",
      },
    },
    {
      active: false,
      name: "sync.demo",
      syncProviderBaseUrl: "http://sync.demo.taler.net/",
      paymentProposalIds: [],
      paymentStatus: {
        type: ProviderPaymentType.InsufficientBalance,
        amount: "KUDOS:10" as AmountString,
      },
      terms: {
        annualFee: "KUDOS:0.1" as AmountString,
        storageLimitInMegabytes: 16,
        supportedProtocolVersion: "0.0",
      },
    },
    {
      active: false,
      name: "sync.demo",
      syncProviderBaseUrl: "http://sync.demo.taler.net/",
      paymentProposalIds: [],
      paymentStatus: {
        type: ProviderPaymentType.TermsChanged,
        newTerms: {
          annualFee: "USD:2" as AmountString,
          storageLimitInMegabytes: 8,
          supportedProtocolVersion: "2",
        },
        oldTerms: {
          annualFee: "USD:1" as AmountString,
          storageLimitInMegabytes: 16,
          supportedProtocolVersion: "1",
        },
        paidUntil: AbsoluteTime.never(),
      },
      terms: {
        annualFee: "KUDOS:0.1" as AmountString,
        storageLimitInMegabytes: 16,
        supportedProtocolVersion: "0.0",
      },
    },
    {
      active: false,
      name: "sync.demo",
      syncProviderBaseUrl: "http://sync.demo.taler.net/",
      paymentProposalIds: [],
      paymentStatus: {
        type: ProviderPaymentType.Unpaid,
      },
      terms: {
        annualFee: "KUDOS:0.1" as AmountString,
        storageLimitInMegabytes: 16,
        supportedProtocolVersion: "0.0",
      },
    },
    {
      active: false,
      name: "sync.demo",
      syncProviderBaseUrl: "http://sync.demo.taler.net/",
      paymentProposalIds: [],
      paymentStatus: {
        type: ProviderPaymentType.Unpaid,
      },
      terms: {
        annualFee: "KUDOS:0.1" as AmountString,
        storageLimitInMegabytes: 16,
        supportedProtocolVersion: "0.0",
      },
    },
  ],
});

export const OneProvider = tests.createExample(TestedComponent, {
  providers: [
    {
      active: true,
      name: "sync.demo",
      syncProviderBaseUrl: "http://sync.taler:9967/",
      lastSuccessfulBackupTimestamp:
        TalerPreciseTimestamp.fromSeconds(1625063925),
      paymentProposalIds: [
        "43Q5WWRJPNS4SE9YKS54H9THDS94089EDGXW9EHBPN6E7M184XEG",
      ],
      paymentStatus: {
        type: ProviderPaymentType.Paid,
        paidUntil: AbsoluteTime.fromMilliseconds(1656599921000),
      },
      terms: {
        annualFee: "ARS:1" as AmountString,
        storageLimitInMegabytes: 16,
        supportedProtocolVersion: "0.0",
      },
    },
  ],
});

export const Empty = tests.createExample(TestedComponent, {
  providers: [],
});

export const Recovery = tests.createExample(ShowRecoveryInfo, {
  info: "taler://recovery/ASLDKJASLKDJASD",
});
