import { TalerFormAttributes } from "@gnu-taler/taler-util";
import {
  DoubleColumnFormDesign,
  FormMetadata,
  InternationalizationAPI,
} from "../../index.browser.js";
import { countryNationalityList } from "../../utils/select-ui-lists.js";
import { intervalToDuration, isFuture, isValid, parse } from "date-fns";

export const form_vqf_902_9_officer = (
  i18n: InternationalizationAPI,
): FormMetadata => ({
  label: i18n.str`Declaration of identity of the beneficial owner (A)`,
  id: "vqf_902_9_officer",
  version: 1,
  config: VQF_902_9_officer(i18n),
});

export function VQF_902_9_officer(
  i18n: InternationalizationAPI,
): DoubleColumnFormDesign {
  return {
    type: "double-column",
    title: i18n.str`Declaration of identity of the beneficial owner`,
    sections: [
      {
        title: "Identity of the contracting partner",
        fields: [
          {
            id: TalerFormAttributes.IDENTITY_CONTRACTING_PARTNER,
            label: i18n.str`Contracting partner`,
            type: "textArea",
            required: true,
          },
        ],
      },
      {
        title: i18n.str`Beneficial owner details`,
        fields: [
          {
            id: TalerFormAttributes.IDENTITY_LIST,
            label: i18n.str`Beneficial owner(s)`,
            help: i18n.str`The person(s) listed below is/are the beneficial owner(s) of the assets involved in the business relationship. If the contracting partner is also the sole beneficial owner of the assets, the contracting partner's detail must be set out below`,
            type: "array",
            validator(persons) {
              if (!persons || persons.length < 1) {
                return i18n.str`Can't be empty`;
              }
              return undefined;
            },
            labelFieldId: TalerFormAttributes.FULL_NAME,
            fields: [
              {
                id: TalerFormAttributes.FULL_NAME,
                label: i18n.str`Full name`,
                type: "text",
                required: true,
              },
              {
                id: TalerFormAttributes.DOMICILE_ADDRESS,
                label: i18n.str`Domicile address`,
                type: "textArea",
                required: true,
              },
              {
                id: TalerFormAttributes.DATE_OF_BIRTH,
                label: i18n.str`Date of birth`,
                type: "isoDateText",
                placeholder: "dd/MM/yyyy",
                pattern: "dd/MM/yyyy",
                required: true,
                validator(text, form) {
                  //FIXME: why returning in this format even if pattern is in another?
                  const time = parse(text, "yyyy-MM-dd", new Date());
                  if (!isValid(time)) {
                    return i18n.str`invalid format`;
                  }
                  if (isFuture(time)) {
                    return i18n.str`it can't be in the future`;
                  }
                  const { years } = intervalToDuration({
                    start: time,
                    end: new Date(),
                  });
                      if (years && years > 120) {
                    return i18n.str`it can't be greater than 120 years`;
                  }
                  return undefined;
                },
              },
              {
                id: TalerFormAttributes.NATIONALITY,
                label: i18n.str`Nationality`,
                type: "selectOne",
                choices: countryNationalityList(i18n),
                preferredChoiceVals: ["CH"],
                required: true,
              },
            ],
            required: false,
          },
        ],
      },
      {
        title: i18n.str`Signed Declaration`,
        description: i18n.str`Signed declaration by the customer`,
        fields: [
          {
            type: "caption",
            label: i18n.str`The uploaded document must contain the customer's signature on the beneficial owner declaration.`,
          },
          {
            id: TalerFormAttributes.ATTACHMENT_SIGNED_DOCUMENT,
            label: i18n.str`Signed Document`,
            type: "file",
            accept: "application/pdf",
            required: true,
          },
        ],
      },
    ],
  };
}
