/*
 * xlog - GTK+ logging program for amateur radio operators
 * Copyright (C) 2001-2003 Joop Stakenborg <pa4tu@amsat.org>
 *
 * This program is free oftware; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * callbacks_qsoframe.c - callbacks for the QSO frame, where all the entry
 * widgets are
 */

#include <time.h>
#include <math.h>
#include <ctype.h>
#ifdef __FreeBSD__
  #include <machine/param.h>
  #include <sys/types.h>
#endif
#include <sys/ipc.h>
#include <sys/shm.h>
#include <errno.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "callbacks_qsoframe.h"
#include "interface.h"
#include "support.h"
#include "dxcc.h"
#include "types.h"
#include "utils.h"
#include "types.h"
#include "log.h"
#include "wwl.h"

#if WANT_HAMLIB
#include "hamlib-utils.h"
#endif

/* defines and variables for shared memory support */
#define KEY 6146
#define SHMSIZE  40
void *shareCall;

extern GtkWidget *mainwindow, *b4dialog;
extern gint callid;
extern statetype state;
extern GList *logwindowlist;
extern preferencestype preferences;
extern GList *logwindowlist;

/* QSO FRAME */
/* get current date and fill in the dateentry */
void
on_datebutton_clicked (GtkButton * button, gpointer user_data)
{
  GtkWidget *entry;
  gchar *nowdate;

  nowdate = getdate ();
  entry = lookup_widget (mainwindow, "dateentry");
  gtk_entry_set_text (GTK_ENTRY (entry), nowdate);
  g_free (nowdate);
}

/* get current time and fill in the gmtentry */
void
on_gmtbutton_clicked (GtkButton * button, gpointer user_data)
{
  GtkWidget *entry;
  gchar *nowtime;

  entry = lookup_widget (mainwindow, "gmtentry");
  nowtime = gettime ();
  gtk_entry_set_text (GTK_ENTRY (entry), nowtime);
  g_free (nowtime);
}

void
on_endbutton_clicked (GtkButton * button, gpointer user_data)
{
  GtkWidget *entry;
  gchar *nowtime;

  entry = lookup_widget (mainwindow, "endentry");
  nowtime = gettime ();
  gtk_entry_set_text (GTK_ENTRY (entry), nowtime);
  g_free (nowtime);
}

/* convert callsign to uppercase */
void
on_callentry_insert_text (GtkEditable * editable, gchar * new_text,
			  gint new_text_length, gint * position,
			  gpointer user_data)
{
  gint i;
  gchar *result = g_new (gchar, new_text_length);

  for (i = 0; i < new_text_length; i++)
    result[i] = islower (new_text[i]) ? toupper (new_text[i]) : new_text[i];

  /* insert the new callsign and block changed and insert_text, so we don't call this twice */
  gtk_signal_handler_block_by_func (GTK_OBJECT (editable),
				    GTK_SIGNAL_FUNC
				    (on_callentry_insert_text), user_data);
  gtk_signal_handler_block_by_func (GTK_OBJECT (editable),
				    GTK_SIGNAL_FUNC (on_callentry_changed),
				    user_data);
  gtk_editable_insert_text (editable, result, new_text_length, position);
  gtk_signal_handler_unblock_by_func (GTK_OBJECT (editable),
				      GTK_SIGNAL_FUNC (on_callentry_changed),
				      user_data);
  gtk_signal_handler_unblock_by_func (GTK_OBJECT (editable),
				      GTK_SIGNAL_FUNC
				      (on_callentry_insert_text), user_data);
  gtk_signal_emit_stop_by_name (GTK_OBJECT (editable), "insert_text");
  g_free (result);
}

static void
updateb4dialog_and_typeaheadfind (gchar * callsign)
{
  GtkWidget *b4clist, *remarksvbox, *remarksentry, *namehbox, *nameentry,
    *qthhbox, *qthentry, *locatorhbox, *locatorentry;
  guint i, j, row;
  logtype *logwindow;
  gchar *logcallsign, **b4 = NULL, *temp;
  gboolean found = FALSE;

  b4clist = lookup_widget (b4dialog, "b4clist");
  if (GTK_WIDGET_VISIBLE (b4dialog))
    gtk_clist_clear (GTK_CLIST (b4clist));

  if (GTK_WIDGET_VISIBLE (b4dialog))
  {
    b4 = g_new0 (gchar *, 19);
    for (j = 0; j < 19; j++)
      b4[j] = g_new0 (gchar, 100);
  }

  if (GTK_WIDGET_VISIBLE (b4dialog) || preferences.typeaheadfind == 1)
  {
    for (i = 0; i < g_list_length (logwindowlist); i++)
    { /* in all logs */
      logwindow = g_list_nth_data (logwindowlist, i);
      for (row = 0; row < logwindow->qsos; row++)
	{ /* get callsign and compare to current callsign entry */
	  gtk_clist_get_text (GTK_CLIST (logwindow->clist),
			      row, CALL, &logcallsign);
          if (preferences.typeaheadfind == 1 && !found && 
            !g_strcasecmp (callsign, logcallsign))
            { /* type ahead find */
              namehbox = lookup_widget(mainwindow, "namehbox");
              if (GTK_WIDGET_VISIBLE (namehbox))
              {
                nameentry = lookup_widget(mainwindow, "nameentry");
                gtk_clist_get_text (GTK_CLIST (logwindow->clist), row, NAME, &temp);
                gtk_entry_set_text (GTK_ENTRY (nameentry), temp);
              }
              qthhbox = lookup_widget(mainwindow, "qthhbox");
              if (GTK_WIDGET_VISIBLE (qthhbox))
              {
                qthentry = lookup_widget(mainwindow, "qthentry");
                gtk_clist_get_text (GTK_CLIST (logwindow->clist), row, QTH, &temp);
                gtk_entry_set_text (GTK_ENTRY (qthentry), temp);
              }
              locatorhbox = lookup_widget(mainwindow, "locatorhbox");
              if (GTK_WIDGET_VISIBLE (locatorhbox))
              {
                locatorentry = lookup_widget(mainwindow, "locatorentry");
                gtk_clist_get_text (GTK_CLIST (logwindow->clist), row, LOCATOR, &temp);
                gtk_entry_set_text (GTK_ENTRY (locatorentry), temp);
              }
              remarksvbox = lookup_widget(mainwindow, "remarksvbox");
              if (!g_strcasecmp (preferences.defaultremarks, "?") && GTK_WIDGET_VISIBLE (remarksvbox))
              {
                remarksentry = lookup_widget(mainwindow, "remarksentry");
                gtk_clist_get_text (GTK_CLIST (logwindow->clist), row, REMARKS, &temp);
                gtk_entry_set_text (GTK_ENTRY (remarksentry), temp);
              }
              found = TRUE; /* lookup only once */
            }
 	  if (GTK_WIDGET_VISIBLE (b4dialog)
            && !g_strncasecmp (callsign, logcallsign, strlen (callsign)))
	    { /* worked b4 */
	      b4[0] = g_strdup (logwindow->logname);
                     for (j = 0; j < 18; j++)
                     {
                       gtk_clist_get_text (GTK_CLIST (logwindow->clist), row,
				  j, &temp);
                       b4[j + 1] = g_strdup (temp);
                     }
	      gtk_clist_append (GTK_CLIST (b4clist), b4);
	    }
	}
    }
  }

  if (GTK_WIDGET_VISIBLE (b4dialog))
  {
    for (j = 0; j < 19; j++)
      g_free (b4[j]);
    g_free (b4);
  }
}

void
on_callentry_changed (GtkEditable * editable, gpointer user_data)
{
  gchar *call, *errorstr;

  call = gtk_editable_get_chars (GTK_EDITABLE (editable), 0, -1);

  updatedxccframe (call);
  updateb4dialog_and_typeaheadfind (call);

  /* twpsk support */
  if (state.shmid == -1)	/* if no IPC ID yet, create one */
    {
      if ((state.shmid = shmget (KEY, SHMSIZE, IPC_CREAT | 0600)) < 0)
	{
	  errorstr =
	    g_strdup_printf (_("shmget failed: %s"), g_strerror (errno));
	  update_statusbar (errorstr);
	  g_free (errorstr);
	}
      if ((shareCall = (shmat (state.shmid, NULL, 0))) == (gchar *) - 1)
	{
	  errorstr =
	    g_strdup_printf (_("shmat failed: %s"), g_strerror (errno));
	  update_statusbar (errorstr);
	  g_free (errorstr);
	  state.shmid = -1;
	}
    }
  if (state.shmid != -1)
    strncpy (shareCall, call, SHMSIZE);	/* put call in shm */
  g_free (call);
}


/* catch return key when band-optionmenu has focus and show the menu */
gboolean
on_bandoptionmenu_key_press_event (GtkWidget * widget,
				   GdkEventKey * event, gpointer user_data)
{
  GtkWidget *bandoptionmenu, *menu;

  switch (event->keyval)
    {
    case GDK_Return:
      bandoptionmenu = lookup_widget (mainwindow, "bandoptionmenu");
      menu = gtk_option_menu_get_menu (GTK_OPTION_MENU (bandoptionmenu));
      gtk_menu_popup (GTK_MENU (menu), NULL, NULL, NULL, NULL, 0, 0);
      return TRUE;
      break;
    }
  return FALSE;
}

/* catch return key when mode-optionmenu has focus and show the menu */
gboolean
on_modeoptionmenu_key_press_event (GtkWidget * widget,
				   GdkEventKey * event, gpointer user_data)
{
  GtkWidget *modeoptionmenu, *menu;
  switch (event->keyval)
    {
    case GDK_Return:
      modeoptionmenu = lookup_widget (mainwindow, "modeoptionmenu");
      menu = gtk_option_menu_get_menu (GTK_OPTION_MENU (modeoptionmenu));
      gtk_menu_popup (GTK_MENU (menu), NULL, NULL, NULL, NULL, 0, 0);
      return TRUE;
      break;
    }
  return FALSE;
}

#if WANT_HAMLIB
void
on_mhzbutton_clicked (GtkButton * button, gpointer user_data)
{
  GtkWidget *entry;
  GString *digits = g_string_new ("");;

  entry = lookup_widget (mainwindow, "bandentry");
  if (g_strcasecmp (preferences.defaultmhz, "?"))
    gtk_entry_set_text (GTK_ENTRY (entry), preferences.defaultmhz);
  else
    {
      if (state.rigfrequency != 0)
	{
	  if (preferences.round == 0)
	    g_string_sprintf (digits, "%Ld", state.rigfrequency);
	  else
	    g_string_sprintf (digits, "%Ld",
			      (long long) rintl (state.rigfrequency /
						 pow (10,
						      preferences.round)));
	  g_string_insert_c (digits, (digits->len) - 6 + preferences.round,
			     '.');
	  g_strstrip (digits->str);
	  g_strdelimit (digits->str, " ", '0');
	}
      else
	{
	  g_string_sprintf (digits, "UNKNOWN");
	}
      gtk_entry_set_text (GTK_ENTRY (entry), digits->str);
      g_string_free (digits, TRUE);
    }
}

void
on_modebutton_clicked (GtkButton * button, gpointer user_data)
{
  GtkWidget *entry;
  gchar *mode;

  entry = lookup_widget (mainwindow, "modeentry");
  if (g_strcasecmp (preferences.defaultmode, "?"))
    gtk_entry_set_text (GTK_ENTRY (entry), preferences.defaultmode);
  else
    {
      mode = getrigmode (state.rigmode);
      gtk_entry_set_text (GTK_ENTRY (entry), mode);
      g_free (mode);
    }
}

void
on_rstbutton_clicked (GtkButton * button, gpointer user_data)
{
  GtkWidget *entry;

  entry = lookup_widget (mainwindow, "rstentry");
  if (g_strcasecmp (preferences.defaulttxrst, "?"))
    gtk_entry_set_text (GTK_ENTRY (entry), preferences.defaulttxrst);
  else if (state.rigrst != 0)
    gtk_entry_set_text (GTK_ENTRY (entry), state.rigrst);
  else
    gtk_entry_set_text (GTK_ENTRY (entry), "UNKNOWN");
}

void
on_powerbutton_clicked (GtkButton * button, gpointer user_data)
{
  GtkWidget *entry;
  gchar *rigpower;

  entry = lookup_widget (mainwindow, "powerentry");
  if (g_strcasecmp (preferences.defaultpower, "?"))
    gtk_entry_set_text (GTK_ENTRY (entry), preferences.defaultpower);
  else if (state.rigpower != 0)
    {
      rigpower = g_strdup_printf ("%d", state.rigpower * 256 / 100000);
      gtk_entry_set_text (GTK_ENTRY (entry), rigpower);
      g_free (rigpower);
    }
  else
    gtk_entry_set_text (GTK_ENTRY (entry), "UNKNOWN");
}
#endif

/* right click with mouse button */
gboolean
on_qsoframeeventbox_button_press_event (GtkWidget * widget,
					GdkEventButton * event,
					gpointer user_data)
{
  if (event->button != 1)
    {
      gtk_menu_popup (GTK_MENU (create_qsoframemenu ()), NULL, NULL, NULL,
		      NULL, 1, 0);
      return TRUE;
    }
  return FALSE;
}

/* POP-UP MENU: clear all the fields in the qso frame */
void
on_clearframe_activate (GtkMenuItem * menuitem, gpointer user_data)
{
  GtkWidget *dateentry, *gmtentry, *callentry, *rstentry, *myrstentry,
    *remarksentry, *modeentry, *bandentry, *incheckbutton, *outcheckbutton,
    *endentry, *powerentry, *nameentry, *qthentry, *locatorentry,
    *unknownentry1, *unknownentry2, *modeoptionmenu, *bandoptionmenu,
    *qsoframe, *locatorframe, *datebutton;
  gchar *frametext;
  logtype *logwindow;
  gint i;

  dateentry = lookup_widget (mainwindow, "dateentry");
  gmtentry = lookup_widget (mainwindow, "gmtentry");
  endentry = lookup_widget (mainwindow, "endentry");
  callentry = lookup_widget (mainwindow, "callentry");
  bandentry = lookup_widget (mainwindow, "bandentry");
  modeentry = lookup_widget (mainwindow, "modeentry");
  rstentry = lookup_widget (mainwindow, "rstentry");
  myrstentry = lookup_widget (mainwindow, "myrstentry");
  remarksentry = lookup_widget (mainwindow, "remarksentry");
  modeoptionmenu = lookup_widget (mainwindow, "modeoptionmenu");
  bandoptionmenu = lookup_widget (mainwindow, "bandoptionmenu");
  incheckbutton = lookup_widget (mainwindow, "incheckbutton");
  outcheckbutton = lookup_widget (mainwindow, "outcheckbutton");
  powerentry = lookup_widget (mainwindow, "powerentry");
  nameentry = lookup_widget (mainwindow, "nameentry");
  qthentry = lookup_widget (mainwindow, "qthentry");
  locatorentry = lookup_widget (mainwindow, "locatorentry");
  unknownentry1 = lookup_widget (mainwindow, "unknownentry1");
  unknownentry2 = lookup_widget (mainwindow, "unknownentry2");

  for (i = 0; i < g_list_length (logwindowlist); i++)
    {
      logwindow = g_list_nth_data (logwindowlist, i);
      while (GTK_CLIST (logwindow->clist)->selection)
	gtk_clist_unselect_all (GTK_CLIST (logwindow->clist));
    }
  datebutton = lookup_widget (mainwindow, "datebutton");
  gtk_widget_grab_focus (datebutton);

  gtk_editable_delete_text (GTK_EDITABLE (dateentry), 0, -1);
  gtk_editable_delete_text (GTK_EDITABLE (gmtentry), 0, -1);
  gtk_editable_delete_text (GTK_EDITABLE (endentry), 0, -1);
  gtk_editable_delete_text (GTK_EDITABLE (callentry), 0, -1);
  gtk_editable_delete_text (GTK_EDITABLE (modeentry), 0, -1);
  gtk_editable_delete_text (GTK_EDITABLE (bandentry), 0, -1);
  gtk_editable_delete_text (GTK_EDITABLE (rstentry), 0, -1);
  gtk_editable_delete_text (GTK_EDITABLE (myrstentry), 0, -1);
  gtk_editable_delete_text (GTK_EDITABLE (remarksentry), 0, -1);
  gtk_option_menu_set_history (GTK_OPTION_MENU (modeoptionmenu), 0);
  gtk_option_menu_set_history (GTK_OPTION_MENU (bandoptionmenu), 0);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (outcheckbutton), FALSE);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (incheckbutton), FALSE);
  gtk_editable_delete_text (GTK_EDITABLE (powerentry), 0, -1);
  gtk_editable_delete_text (GTK_EDITABLE (nameentry), 0, -1);
  gtk_editable_delete_text (GTK_EDITABLE (qthentry), 0, -1);
  gtk_editable_delete_text (GTK_EDITABLE (locatorentry), 0, -1);
  gtk_editable_delete_text (GTK_EDITABLE (unknownentry1), 0, -1);
  gtk_editable_delete_text (GTK_EDITABLE (unknownentry2), 0, -1);

  qsoframe = lookup_widget (mainwindow, "qsoframe");
  frametext = g_strdup_printf (_("New QSO"));
  gtk_frame_set_label (GTK_FRAME (qsoframe), frametext);
  g_free (frametext);

  locatorframe = lookup_widget (mainwindow, "locatorframe");
  if (GTK_WIDGET_VISIBLE (locatorframe))
    {
      frametext = g_strdup_printf (_("Locator"));
      gtk_frame_set_label (GTK_FRAME (locatorframe), frametext);
      g_free (frametext);
    }
}

void
on_clickall_activate (GtkMenuItem * menuitem, gpointer user_data)
{
  GtkWidget *datebutton, *gmtbutton, *endbutton, *endhbox,
    *myrstentry, *unknown1hbox, *unknown2hbox, *unknownentry1, *unknownentry2,
    *remarksvbox, *remarksentry;
#if WANT_HAMLIB
  GtkWidget *mhzbutton, *modebutton, *rstbutton, *powerbutton, *powerhbox;
#else
  GtkWidget *bandentry, *modeentry, *rstentry, *powerhbox, *powerentry;
#endif

  datebutton = lookup_widget (mainwindow, "datebutton");
  gmtbutton = lookup_widget (mainwindow, "gmtbutton");
  gtk_signal_emit_by_name (GTK_OBJECT (datebutton), "clicked");
  gtk_signal_emit_by_name (GTK_OBJECT (gmtbutton), "clicked");
  endhbox = lookup_widget (mainwindow, "endhbox");
  if (GTK_WIDGET_VISIBLE (endhbox))
    {
      endbutton = lookup_widget (mainwindow, "endbutton");
      gtk_signal_emit_by_name (GTK_OBJECT (endbutton), "clicked");
    }
#if WANT_HAMLIB
  if (preferences.hamlib > 0)
    {
      mhzbutton = lookup_widget (mainwindow, "mhzbutton");
      modebutton = lookup_widget (mainwindow, "modebutton");
      rstbutton = lookup_widget (mainwindow, "rstbutton");
      gtk_signal_emit_by_name (GTK_OBJECT (mhzbutton), "clicked");
      gtk_signal_emit_by_name (GTK_OBJECT (modebutton), "clicked");
      gtk_signal_emit_by_name (GTK_OBJECT (rstbutton), "clicked");
      powerhbox = lookup_widget (mainwindow, "powerhbox");
      if (GTK_WIDGET_VISIBLE (powerhbox))
	{
	  powerbutton = lookup_widget (mainwindow, "powerbutton");
	  gtk_signal_emit_by_name (GTK_OBJECT (powerbutton), "clicked");
	}
    }
#else
  /* set defaults */
  if (g_strcasecmp (preferences.defaultmhz, "?"))
  {
    bandentry = lookup_widget (mainwindow, "bandentry");
    gtk_entry_set_text (GTK_ENTRY (bandentry), preferences.defaultmhz);
  }
  if (g_strcasecmp (preferences.defaultmode, "?"))
  {
    modeentry = lookup_widget (mainwindow, "modeentry");
    gtk_entry_set_text (GTK_ENTRY (modeentry), preferences.defaultmode);
  }
  if (g_strcasecmp (preferences.defaulttxrst, "?"))
  {
    rstentry = lookup_widget (mainwindow, "rstentry");
    gtk_entry_set_text (GTK_ENTRY (rstentry), preferences.defaulttxrst);
  }
  powerhbox = lookup_widget (mainwindow, "powerhbox");
  if (GTK_WIDGET_VISIBLE (powerhbox) 
    && g_strcasecmp (preferences.defaultpower, "?"))
  {
      powerentry = lookup_widget (mainwindow, "powerentry");
      gtk_entry_set_text (GTK_ENTRY (powerentry), preferences.defaultpower);
  }
#endif

  /* set defaults */
  if (g_strcasecmp (preferences.defaultrxrst, "?"))
  {
    myrstentry = lookup_widget (mainwindow, "myrstentry");
    gtk_entry_set_text (GTK_ENTRY (myrstentry), preferences.defaultrxrst);
  }
  unknown1hbox = lookup_widget (mainwindow, "unknown1hbox");
  if (GTK_WIDGET_VISIBLE (unknown1hbox)
    && g_strcasecmp (preferences.defaultfreefield1, "?"))
  {
    unknownentry1 = lookup_widget (mainwindow, "unknownentry1");
    gtk_entry_set_text (GTK_ENTRY (unknownentry1), 
      preferences.defaultfreefield1);
  }
  unknown2hbox = lookup_widget (mainwindow, "unknown2hbox");
  if (GTK_WIDGET_VISIBLE (unknown2hbox)
    && g_strcasecmp (preferences.defaultfreefield2, "?"))
  {
    unknownentry2 = lookup_widget (mainwindow, "unknownentry2");
    gtk_entry_set_text (GTK_ENTRY (unknownentry2),
      preferences.defaultfreefield2);
  }
  remarksvbox = lookup_widget (mainwindow, "remarksvbox");
  if (GTK_WIDGET_VISIBLE (remarksvbox)
    && g_strcasecmp (preferences.defaultremarks, "?"))
  {
    remarksentry = lookup_widget (mainwindow, "remarksentry");
    gtk_entry_set_text (GTK_ENTRY (remarksentry),
      preferences.defaultremarks);
  }
}

/* check locator and calculate distance and azimuth */
void
on_locatorentry_changed (GtkEditable * editable, gpointer user_data)
{
  gchar *locator;

  locator = gtk_editable_get_chars (GTK_EDITABLE (editable), 0, -1);
  updatelocatorframe (locator);
  g_free (locator);
}
