/* GnomeTalk: Main interface module
 * (C) 1997 the Free Software Foundation
 *
 * Author: Federico Mena          (quartic@gimp.org)
 * Further hacks: Miguel de Icaza (miguel@kernel.org)
 */

#include <config.h>
#include <unistd.h>
#include <gdk/gdkkeysyms.h>
#include <gnome.h>
#include <zvt/zvtterm.h>
#include "global.h"
#include "talk.h"
#include "protocol.h"

#define TALK_WIDTH      80
#define TALK_HEIGHT     10
#define TALK_SCROLLBACK 100


static int sock_input_id;

char his_erase, his_kill, his_win_erase;


static void
tty_display (ZvtTerm *tty, char ch)
{
	int i;
	
	switch (ch) {
		case '\n':
			zvt_term_feed (tty, "\r\n", 2);
			break;

		case TALK_ERASE:
			zvt_term_feed (tty, "\b \b", 3);
			break;

		case TALK_KILL: /* ^U */
			for (i = tty->vx->vt.cursorx; i > 0; i--)
				zvt_term_feed (tty, "\b \b", 3);
			break;

		case TALK_WIN_ERASE:
			zvt_term_feed (tty, "\033[2J", 4);
			break;

		default:
			zvt_term_feed (tty, &ch, 1);
			break;
	}
}


static char
translate(char c)
{
	if (c == his_erase)
		return TALK_ERASE;
	else if (c == his_kill)
		return TALK_KILL;
	else if (c == his_win_erase)
		return TALK_WIN_ERASE;
	else
		return c;
}

gint
talk_quit(GtkWidget *widget, gpointer data)
{
	gdk_input_remove(sock_input_id);
	gtk_main_quit();
	
	return TRUE;
}

static void
read_from_socket(gpointer data, gint fd, GdkInputCondition condition)
{
	int  n;
	char ch;
	
	if (!(condition & GDK_INPUT_READ))
		return;

	n = read(fd, &ch, 1);

	if (n <= 0) {
                talk_quit(NULL, NULL);
		return;
	}

	tty_display (ZVT_TERM (data), translate(ch));
}


static gint
key_press(GtkWidget *widget, GdkEventKey *event, gpointer data)
{
	int length = event->length;
	int char_code  = event->keyval;
	int keyval     = event->keyval;
	char ch;

	if (length > 0) {
		switch (keyval) {
		case GDK_Return:
			ch = '\n';
			break;
			
		case GDK_Delete:
		case GDK_BackSpace:
			ch = TALK_ERASE;
			break;
			
		default:
			ch = char_code;
			break;
		}
		
		write(talk_socket, &ch, 1);
		tty_display(ZVT_TERM (widget), ch);
		
		return TRUE;
	}
	
	return FALSE;
}


static void
about_cb (void)
{
        GtkWidget *about;

        const gchar *authors[] = {
		"Federico Mena (federico@gimp.org)",
		"Miguel de Icaza (miguel@kernel.org)",
		NULL
	};

        about = gnome_about_new (_("GNOME Talk"), VERSION,
				 "(C) 1998 the Free Software Foundation",
				 authors,
				 _("The GNOME talk program."),
				 NULL);
        gtk_widget_show (about);
}

GnomeUIInfo talkmenu[] = {
	{ GNOME_APP_UI_ITEM, N_("_Terminate"), N_("Terminates this talk session"), GTK_SIGNAL_FUNC (talk_quit), NULL, NULL,
	  GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_EXIT, 'n', GDK_CONTROL_MASK, NULL },
	GNOMEUIINFO_END
};

GnomeUIInfo helpmenu[] = {
	{ GNOME_APP_UI_ITEM, N_("_About"), N_("Info about this program"), about_cb, NULL, NULL, 
	  GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_ABOUT },
	GNOMEUIINFO_END
};

GnomeUIInfo mainmenu[] = {
    GNOMEUIINFO_SUBTREE(N_("_Talk"), talkmenu),
    GNOMEUIINFO_SUBTREE(N_("_Help"), helpmenu),
    GNOMEUIINFO_END
};

void
create_talk_interface(void)
{
	GtkWidget *window;
	GtkWidget *vbox;
	GtkWidget *hbox;
	GtkWidget *his;
	GtkWidget *ours;
	GtkWidget *scrollbar;
	char      *str;

	talk_report_shutdown ();

	str = g_copy_strings(_("Talking to "), his_name, NULL);
	g_free(str);

	window = gnome_app_new (_("Talk"), str);
	gnome_app_create_menus (GNOME_APP (window), mainmenu);
	gtk_signal_connect(GTK_OBJECT(window), "delete_event",
			   GTK_SIGNAL_FUNC(talk_quit), NULL);
	gtk_window_set_title(GTK_WINDOW(window), _("Talk"));

	vbox = gtk_vbox_new(FALSE, 0);
	gnome_app_set_contents (GNOME_APP (window), vbox);
	gtk_widget_show(vbox);

	/* His */

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 2);
	gtk_widget_show(hbox);
	
	his = zvt_term_new ();
	zvt_term_set_size(ZVT_TERM(his), 80, 11);
	gtk_box_pack_start(GTK_BOX(hbox), his, TRUE, TRUE, 0);
	gtk_widget_set_sensitive (his, FALSE);
	gtk_widget_show(his);

	sock_input_id = gdk_input_add(talk_socket,
				      GDK_INPUT_READ,
				      read_from_socket,
				      his);

	scrollbar = gtk_vscrollbar_new(GTK_ADJUSTMENT (ZVT_TERM (his)->adjustment));
	gtk_box_pack_start(GTK_BOX(hbox), scrollbar, FALSE, FALSE, 0);
	gtk_widget_show(scrollbar);

	
	/* Ours */

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 2);
	gtk_widget_show(hbox);

	ours = zvt_term_new ();
	zvt_term_set_size(ZVT_TERM(ours), 80, 11);
	gtk_box_pack_start(GTK_BOX(hbox), ours, TRUE, TRUE, 0);
	gtk_signal_connect(GTK_OBJECT(ours), "key_press_event",
			   (GtkSignalFunc) key_press,
			   NULL);

	gtk_widget_grab_focus (ours);
	gtk_widget_show(ours);

	scrollbar = gtk_vscrollbar_new(GTK_ADJUSTMENT (ZVT_TERM (ours)->adjustment));
	gtk_box_pack_start(GTK_BOX(hbox), scrollbar, FALSE, FALSE, 0);
	gtk_widget_show(scrollbar);

	gtk_widget_show(window);
}
