/***************************************************************************
                          glowclient.cpp  -  description
                             -------------------
    begin                : Thu Sep 6 2001
    copyright            : (C) 2001 by Henning Burchardt
    email                : h_burchardt@gmx.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <cmath>
#include <qpixmap.h>
#include <qbitmap.h>
#include <qpainter.h>
#include <qapplication.h>
#include <qlayout.h>
#include <qfontmetrics.h>
#include <klocale.h>
#include <kpixmapeffect.h>
#include <kpixmap.h>
#include <kconfig.h>
#include <kwin/options.h>
#include <kwin/workspace.h>
#include "bitmaps.h"
#include "glowclient.h"
#include "glowbutton.h"

#define SIDE_MARGIN 4
#define BOTTOM_MARGIN 2
#define RESIZE_HANDLE_WIDTH 20
#define RESIZE_HANDLE_HEIGHT 7
#define TITLE_MARGIN 2
#define TITLE_SPACING 1
#define TITLE_BAR_EXTENSION_SIZE 20

using namespace KWinInternal;

//-----------------------------------------------------------------------------

GlowClientConfig::GlowClientConfig()
{
}

void GlowClientConfig::load()
{
	KConfig conf("kwinglowrc");
	conf.setGroup("General");

	QColor defaultButtonColor = Qt::white;
	QColor defaultCloseButtonColor = Qt::red;

	QString stickyButtonGlowType =
		conf.readEntry("stickyButtonGlowType", "TitleBar");
	if(stickyButtonGlowType=="TitleBar")
		stickyButtonGlowColor = options->color(Options::TitleBar);
	else if(stickyButtonGlowType=="TitleBlend")
		stickyButtonGlowColor = options->color(Options::TitleBlend);
	else // if( stickyButtonGlowType=="Custom" )
		stickyButtonGlowColor = conf.readColorEntry(
			"stickyButtonGlowColor", &defaultButtonColor);

	QString helpButtonGlowType =
		conf.readEntry("helpButtonGlowType", "TitleBar");
	if(helpButtonGlowType=="TitleBar")
		helpButtonGlowColor = options->color(Options::TitleBar);
	else if(helpButtonGlowType=="TitleBlend")
		helpButtonGlowColor = options->color(Options::TitleBlend);
	else // if( helpButtonGlowType=="Custom" )
		helpButtonGlowColor = conf.readColorEntry(
			"helpButtonGlowColor", &defaultButtonColor);

	QString iconifyButtonGlowType =
		conf.readEntry("iconifyButtonGlowType", "TitleBar");
	if(iconifyButtonGlowType=="TitleBar")
		iconifyButtonGlowColor = options->color(Options::TitleBar);
	else if(iconifyButtonGlowType=="TitleBlend")
		iconifyButtonGlowColor = options->color(Options::TitleBlend);
	else // if( iconifyButtonGlowType=="Custom" )
		iconifyButtonGlowColor = conf.readColorEntry(
			"iconifyButtonGlowColor", &defaultButtonColor);

	QString maximizeButtonGlowType =
		conf.readEntry("maximizeButtonGlowType", "TitleBar");
	if(maximizeButtonGlowType=="TitleBar")
		maximizeButtonGlowColor = options->color(Options::TitleBar);
	else if(maximizeButtonGlowType=="TitleBlend")
		maximizeButtonGlowColor = options->color(Options::TitleBlend);
	else // if( maximizeButtonGlowType=="Custom" )
		maximizeButtonGlowColor = conf.readColorEntry(
			"maximizeButtonGlowColor", &defaultButtonColor);

	QString closeButtonGlowType =
		conf.readEntry("closeButtonGlowType", "Custom");
	if(closeButtonGlowType=="TitleBar")
		closeButtonGlowColor = options->color(Options::TitleBar);
	else if(closeButtonGlowType=="TitleBlend")
		closeButtonGlowColor = options->color(Options::TitleBlend);
	else // if( closeButtonGlowType=="Custom" )
		closeButtonGlowColor = conf.readColorEntry(
			"closeButtonGlowColor", &defaultCloseButtonColor);

	showResizeHandle = conf.readBoolEntry("showResizeHandle", true);
	titlebarGradientType = conf.readNumEntry("titlebarGradientType",
		KPixmapEffect::DiagonalGradient);
}

//-----------------------------------------------------------------------------

GlowClientGlobals *GlowClientGlobals::m_instance = 0;

GlowClientGlobals *GlowClientGlobals::instance()
{
	if( ! m_instance )
		m_instance = new GlowClientGlobals();
	return m_instance;
}

GlowClientGlobals::~GlowClientGlobals()
{
	deletePixmaps();
	m_instance = 0;
}

QString GlowClientGlobals::getPixmapName(
	int type, bool isActive, bool isLeft, bool isSmall)
{
	QString s = getPixmapTypeName(type);
	s += "|";
	s += isActive ? "Active" : "NotActive";
	s += "|";
	s += isLeft ? "PosLeft" : "PosRight";
	s += "|";
	s += isSmall ? "SizeSmall" : "SizeNormal";
	return s;
}

GlowClientGlobals::GlowClientGlobals()
	: QObject()
{
	buttonFactory = new GlowButtonFactory();
	readConfig();
	createPixmaps();
}

void GlowClientGlobals::readConfig()
{
	config = new GlowClientConfig();
	config->load();
}

std::vector<int> GlowClientGlobals::getPixmapTypes()
{
	std::vector<int> pixmapTypes;
	pixmapTypes.push_back(StickyOn);
	pixmapTypes.push_back(StickyOff);
	pixmapTypes.push_back(Help);
	pixmapTypes.push_back(Iconify);
	pixmapTypes.push_back(MaximizeOn);
	pixmapTypes.push_back(MaximizeOff);
	pixmapTypes.push_back(Close);

	return pixmapTypes;
}

void GlowClientGlobals::reset()
{
	deletePixmaps();
	delete config;
	readConfig();
	createPixmaps();
}

void GlowClientGlobals::createPixmaps()
{
	std::vector<int> types = getPixmapTypes();
	for( int i=0; i<types.size(); i++ )
	{
		createPixmap(types[i], false, false, false);
		createPixmap(types[i], false, false, true);
		createPixmap(types[i], false, true, false);
		createPixmap(types[i], false, true, true);
		createPixmap(types[i], true, false, false);
		createPixmap(types[i], true, false, true);
		createPixmap(types[i], true, true, false);
		createPixmap(types[i], true, true, true);
	}
}

void GlowClientGlobals::deletePixmaps()
{
	PixmapCache::clear();
}

QString GlowClientGlobals::getPixmapTypeName(int type)
{
	if( type == StickyOn )
		return "StickyOn";
	else if( type == StickyOff )
		return "StickyOff";
	else if( type == Help )
		return "Help";
	else if( type == Iconify )
		return "Iconify";
	else if( type == MaximizeOn )
		return "MaximizeOn";
	else if( type == MaximizeOff )
		return "MaximizeOff";
	else if( type == Close )
		return "Close";
	else
		return QString::null;
}

void GlowClientGlobals::createPixmap(
	int type, bool isActive, bool isLeft, bool isSmall)
{
	int size;
	if( isSmall )
		size = SMALL_BITMAP_SIZE;
	else
		size = DEFAULT_BITMAP_SIZE;
	QColorGroup g;
	if( isLeft )
		g = options->colorGroup(Options::TitleBar, isActive);
	else
		g = options->colorGroup(Options::ButtonBg, isActive);
	QColor c;
	if( qGray(g.background().rgb()) <= 127 ) // background is dark
		c = Qt::white;
	else // background is light
		c = Qt::black;

	QPixmap pm(size, size);
	pm.fill(c);

	if( type == StickyOn )
	{
		if( isSmall )
			pm.setMask(QBitmap(size, size, stickyon_small_bits, true));
		else
			pm.setMask(QBitmap(size, size, stickyon_bits, true));
		QPixmap *glowPm = buttonFactory->createGlowButtonPixmap(
			QSize(size,size), config->stickyButtonGlowColor, g, pm);
		PixmapCache::insert(
			getPixmapName(type, isActive, isLeft, isSmall), glowPm);
	}
	else if( type==StickyOff )
	{
		if( isSmall )
			pm.setMask(QBitmap(size, size, stickyoff_small_bits, true));
		else
			pm.setMask(QBitmap(size, size, stickyoff_bits, true));
		QPixmap *glowPm = buttonFactory->createGlowButtonPixmap(
			QSize(size,size), config->stickyButtonGlowColor, g, pm);
		PixmapCache::insert(
			getPixmapName(type, isActive, isLeft, isSmall), glowPm);
	}
	else if( type==Help )
	{
		if( isSmall )
			pm.setMask(QBitmap(size, size, help_small_bits, true));
		else
			pm.setMask(QBitmap(size, size, help_bits, true));
		QPixmap *glowPm = buttonFactory->createGlowButtonPixmap(
			QSize(size,size), config->helpButtonGlowColor, g, pm);
		PixmapCache::insert(
			getPixmapName(type, isActive, isLeft, isSmall), glowPm);
	}
	else if( type==Iconify )
	{
		if( isSmall )
			pm.setMask(QBitmap(size, size, minimize_small_bits, true));
		else
			pm.setMask(QBitmap(size, size, minimize_bits, true));
		QPixmap *glowPm = buttonFactory->createGlowButtonPixmap(
			QSize(size,size), config->iconifyButtonGlowColor, g, pm);
		PixmapCache::insert(
			getPixmapName(type, isActive, isLeft, isSmall), glowPm);
	}
	else if( type==MaximizeOn )
	{
		if( isSmall )
			pm.setMask(QBitmap(size, size, maximizeon_small_bits, true));
		else
			pm.setMask(QBitmap(size, size, maximizeon_bits, true));
		QPixmap *glowPm = buttonFactory->createGlowButtonPixmap(
			QSize(size,size), config->maximizeButtonGlowColor, g, pm);
		PixmapCache::insert(
			getPixmapName(type, isActive, isLeft, isSmall), glowPm);
	}
	else if( type==MaximizeOff )
	{
		if( isSmall )
			pm.setMask(QBitmap(size, size, maximizeoff_small_bits, true));
		else
			pm.setMask(QBitmap(size, size, maximizeoff_bits, true));
		QPixmap *glowPm = buttonFactory->createGlowButtonPixmap(
			QSize(size,size), config->maximizeButtonGlowColor, g, pm);
		PixmapCache::insert(
			getPixmapName(type, isActive, isLeft, isSmall), glowPm);
	}
	else if( type==Close )
	{
		if( isSmall )
			pm.setMask(QBitmap(size, size, close_small_bits, true));
		else
			pm.setMask(QBitmap(size, size, close_bits, true));
		QPixmap *glowPm = buttonFactory->createGlowButtonPixmap(
			QSize(size,size), config->closeButtonGlowColor, g, pm);
		PixmapCache::insert(
			getPixmapName(type, isActive, isLeft, isSmall), glowPm);
	}
}

//-----------------------------------------------------------------------------

GlowClient::GlowClient(KWinInternal::Workspace *ws, WId w,
	QWidget *parent, const char* name )
	: KWinInternal::Client(ws, w, parent, name),
		m_stickyButton(0), m_helpButton(0), m_minimizeButton(0),
		m_maximizeButton(0), m_closeButton(0),
		m_mainLayout(0), m_leftButtonLayout(0), m_rightButtonLayout(0)
{
	createButtons();
	resetLayout();
	repaint();
}

GlowClient::~GlowClient()
{
}

void GlowClient::resizeEvent( QResizeEvent *e )
{
	Client::resizeEvent(e);
	doShape();
	repaint(false);
}

void GlowClient::paintEvent( QPaintEvent *e )
{
	Client::paintEvent(e);

	GlowClientConfig *conf = GlowClientGlobals::instance()->config;
	QRect r_this = rect();
	QRect r_title = m_titleSpacer->geometry();
	QRect r_wrapper = windowWrapper()->geometry();
	QColorGroup titleCg =
		options->colorGroup(Options::TitleBar, isActive());
	QColorGroup titleBlendCg =
		options->colorGroup(Options::TitleBlend, isActive());
	QColorGroup cg = colorGroup();
	QColor titleColor = options->color(Options::TitleBar, isActive());
	QColor titleBlendColor = options->color(Options::TitleBlend, isActive());
	QPainter painter;
	QPointArray pArray, pArray2;

	// pixmap for title bar
	QSize tBSize(width(), r_title.height());
	KPixmap gradientPixmap(tBSize-QSize(3,3));
	KPixmapEffect::gradient(gradientPixmap, titleColor, titleBlendColor,
		(KPixmapEffect::GradientType) conf->titlebarGradientType);
	QPixmap titleBuffer(tBSize);
	painter.begin(&titleBuffer);
	painter.drawPixmap(2, 2, gradientPixmap);
	// draw caption
	painter.setFont(options->font(isActive()));
	painter.setPen(options->color(Options::Font, isActive()));
	painter.drawText(r_title.x(), 0,
		r_title.width(), r_title.height(),
		Qt::AlignLeft | Qt::AlignVCenter, caption() );
	// draw borders
	pArray = QPointArray(7);
	pArray.setPoint(0, 0, tBSize.height()-1);
	pArray.setPoint(1, 0, 0);
	pArray.setPoint(2, tBSize.width()-1, 0);
	pArray.setPoint(3, tBSize.width()-1, tBSize.height()/2-1);
	pArray.setPoint(4, r_title.x()+r_title.width()-1+tBSize.height()/2,
		tBSize.height()/2-1);
	pArray.setPoint(5, r_title.x()+r_title.width()-1, tBSize.height()-1);
	pArray.setPoint(6, 0, tBSize.height()-1);
	painter.setPen(titleCg.mid());
	painter.drawPolyline(pArray, 3, 4);
	painter.setPen(Qt::black);
	painter.drawPolyline(pArray, 0, 4);
	painter.setPen(titleCg.light());
	pArray2 = QPointArray(3);
	pArray2.setPoint(0, 1, tBSize.height()-2);
	pArray2.setPoint(1, 1, 1);
	pArray2.setPoint(2, tBSize.width()-2, 1);
	painter.drawPolyline(pArray2);
	painter.end();

	// set mask
	QBitmap titleBitmap(tBSize);
	titleBitmap.fill(Qt::color0);
	painter.begin(&titleBitmap);
	painter.setPen(Qt::color1);
	painter.setBrush(Qt::color1);
	painter.drawPolygon(pArray);
	painter.end();
	titleBuffer.setMask(titleBitmap);

	painter.begin(this);
	painter.drawPixmap(0, 0, titleBuffer);
	painter.setPen(Qt::black);
	painter.drawLine(0,tBSize.height(),0,r_this.height()-1);
	painter.drawLine(0,r_this.height()-1,r_this.width()-1,r_this.height()-1);
	painter.drawLine(r_this.width()-1,r_this.height()-1,
		r_this.width()-1,tBSize.height()/2);
	painter.end();

	// paint resize handle if necessary
	if( GlowClientGlobals::instance()->config->showResizeHandle
		&& !isTool() && isResizable() )
	{
		int rh_w = RESIZE_HANDLE_WIDTH;
		int rh_h = RESIZE_HANDLE_HEIGHT;
		QSize rh_s(rh_w, rh_h);

		QPixmap rh_pm = DrawUtils::drawRoundRectButton(rh_s,
			options->colorGroup(Options::ButtonBg, isActive()));
		painter.begin(this);
		painter.drawPixmap(0,height()-rh_pm.height(),rh_pm);
		painter.drawPixmap(width()-rh_pm.width(),height()-rh_pm.height(),
			rh_pm);
		painter.end();
	}
}

void GlowClient::showEvent( QShowEvent *e )
{
	Client::showEvent(e);
	doShape();
	repaint(false);
}

void GlowClient::mouseDoubleClickEvent( QMouseEvent *e )
{
	if( m_titleSpacer->geometry().contains(e->pos()) )
		workspace()->performWindowOperation(
			this, options->operationTitlebarDblClick());
}

void GlowClient::captionChange(const QString&)
{
	repaint(false);
}

void GlowClient::activeChange(bool)
{
	updateButtonPixmaps();
	repaint(false);
}

void GlowClient::iconChange()
{
	// we have no (t yet an) icon button, so do nothing
}

void GlowClient::stickyChange(bool on)
{
	if(on) {
		m_stickyButton->setPixmapName(
			GlowClientGlobals::instance()->getPixmapName(
				GlowClientGlobals::StickyOn, isActive(),
				isLeft(m_stickyButton), isTool()));
		m_stickyButton->setTipText(i18n("Un-Sticky"));
	}
	else {
		m_stickyButton->setPixmapName(
			GlowClientGlobals::instance()->getPixmapName(
				GlowClientGlobals::StickyOff, isActive(),
				isLeft(m_stickyButton), isTool()));
		m_stickyButton->setTipText(i18n("Sticky"));
	}
}

void GlowClient::maximizeChange(bool on)
{
        if(on) {
		m_maximizeButton->setPixmapName(
			GlowClientGlobals::instance()->getPixmapName(
				GlowClientGlobals::MaximizeOn, isActive(),
				isLeft(m_maximizeButton), isTool()));
		m_maximizeButton->setTipText(i18n("Restore"));
	}
	else {
		m_maximizeButton->setPixmapName(
			GlowClientGlobals::instance()->getPixmapName(
				GlowClientGlobals::MaximizeOff, isActive(),
				isLeft(m_maximizeButton), isTool()));
		m_maximizeButton->setTipText(i18n("Maximize"));
	}
}

Client::MousePosition GlowClient::mousePosition(const QPoint &pos) const
{
	Client::MousePosition m = Nowhere;
	int rh_w = RESIZE_HANDLE_WIDTH;
	int rh_h = RESIZE_HANDLE_HEIGHT;

	if( GlowClientGlobals::instance()->config->showResizeHandle && !isTool() )
	{
		if( pos.y() >= height()-rh_h )
		{
			if( pos.x() < rh_w )
				m = BottomLeft;
			else if( pos.x() > width()-rh_w )
				m = BottomRight;
			else m = Bottom;
		}
		else
			m = Client::mousePosition(pos);
	}
	else
		m = Client::mousePosition(pos);
	return m;
}

void GlowClient::createButtons()
{
	GlowClientGlobals *globals = GlowClientGlobals::instance();
	GlowButtonFactory *factory = globals->buttonFactory;
	int s = isTool() ? SMALL_BITMAP_SIZE : DEFAULT_BITMAP_SIZE;
	QSize size(s,s);

	m_stickyButton = factory->createGlowButton(
		this, "StickyButton", i18n("Sticky"));
	m_stickyButton->setFixedSize(size);
	QObject::connect(m_stickyButton, SIGNAL(clicked()),
		this, SLOT(toggleSticky()));
	m_buttonList.insert(m_buttonList.end(), m_stickyButton);

	m_helpButton = factory->createGlowButton(
		this, "HelpButton", i18n("Help"));
	m_helpButton->setFixedSize(size);
	QObject::connect(m_helpButton, SIGNAL(clicked()),
		this, SLOT(contextHelp()));
	m_buttonList.insert(m_buttonList.end(), m_helpButton);

	m_minimizeButton = factory->createGlowButton(
		this, "IconifyButton", i18n("Minimize"));
	m_minimizeButton->setFixedSize(size);
	QObject::connect(m_minimizeButton, SIGNAL(clicked()),
		this, SLOT(iconify()));
	m_buttonList.insert(m_buttonList.end(), m_minimizeButton);

	m_maximizeButton=factory->createGlowButton(
		this, "MaximizeButton", i18n("Maximize"));
	m_maximizeButton->setFixedSize(size);
	QObject::connect(m_maximizeButton, SIGNAL(clicked(int)),
		this, SLOT(slotMaximize(int)));
	m_buttonList.insert(m_buttonList.end(), m_maximizeButton);

	m_closeButton = factory->createGlowButton(
		this, "CloseButton", i18n("Close"));
	m_closeButton->setFixedSize(size);
	QObject::connect(m_closeButton, SIGNAL(clicked()),
		this, SLOT(closeWindow()));
	m_buttonList.insert(m_buttonList.end(), m_closeButton);
}

void GlowClient::resetLayout()
{
	const unsigned int sideMargin = SIDE_MARGIN;
	const unsigned int bottomMargin = BOTTOM_MARGIN;

	if( m_mainLayout )
		delete m_mainLayout;
	m_mainLayout = new QVBoxLayout(this, 0, 0);

	// update button positions and colors
	updateButtonPositions();
	updateButtonPixmaps();

	QHBoxLayout *topLayout = new QHBoxLayout(m_mainLayout);
	topLayout->setMargin(0);
	topLayout->setSpacing(TITLE_SPACING);
	topLayout->addSpacing(sideMargin);
	QVBoxLayout *outerLeftLayout = new QVBoxLayout(topLayout);
	outerLeftLayout->addSpacing(TITLE_MARGIN);
	outerLeftLayout->addItem(m_leftButtonLayout);
	outerLeftLayout->addSpacing(1);
	topLayout->addSpacing(sideMargin);

	m_titleSpacer = new QSpacerItem(0, 0,
		QSizePolicy::Expanding, QSizePolicy::Expanding);
	topLayout->addItem(m_titleSpacer);

	topLayout->addSpacing(sideMargin);
	QVBoxLayout *outerRightLayout = new QVBoxLayout(topLayout);
	outerRightLayout->addSpacing(TITLE_MARGIN);
	outerRightLayout->addItem(m_rightButtonLayout);
	outerRightLayout->addSpacing(1);
	topLayout->addSpacing(sideMargin);

	QHBoxLayout *midLayout = new QHBoxLayout(m_mainLayout, 0);
	midLayout->addSpacing(sideMargin);
	midLayout->addWidget(windowWrapper());
	midLayout->addSpacing(sideMargin);

	if( GlowClientGlobals::instance()->config->showResizeHandle
		&& !isTool() && isResizable() )
		m_mainLayout->addSpacing(RESIZE_HANDLE_HEIGHT);
	else
		m_mainLayout->addSpacing(bottomMargin);
	m_mainLayout->setStretchFactor(topLayout, 0);
	m_mainLayout->setStretchFactor(midLayout, 1);
}

void GlowClient::updateButtonPositions()
{
	QString buttons = options->titleButtonsLeft() + "|"
		+ options->titleButtonsRight();
	bool leftButtons=true;

	// hide all buttons
	for( unsigned int i=0; i<m_buttonList.size(); i++ )
		m_buttonList[i]->hide();

	m_leftButtonList.clear();
	m_rightButtonList.clear();

	// reset left and right button layout
	if(m_leftButtonLayout)
		delete m_leftButtonLayout;
	m_leftButtonLayout = new QHBoxLayout();
	m_leftButtonLayout->setMargin(0);
	m_leftButtonLayout->setSpacing(TITLE_SPACING);
	if(m_rightButtonLayout)
		delete m_rightButtonLayout;
	m_rightButtonLayout = new QHBoxLayout();
	m_rightButtonLayout->setMargin(0);
	m_rightButtonLayout->setSpacing(TITLE_SPACING);

	for( unsigned int i=0; i<buttons.length(); i++ )
	{
		char c = buttons[i].latin1();
		GlowButton *button = 0;
		if( c=='S' ) // sticky
			button = m_stickyButton;
		else if( c=='H' && providesContextHelp() ) // help
			button = m_helpButton;
		else if( c=='I' && isMinimizable() ) // iconify
			button = m_minimizeButton;
		else if( c=='A' && isMaximizable() ) // maximize
			button = m_maximizeButton;
		else if( c=='X' ) // close
			button= m_closeButton;
		else if( c=='_' ) // spacer item
		{
			if(leftButtons)
				m_leftButtonLayout->addSpacing(4);
			else
				m_rightButtonLayout->addSpacing(4);
		}
		else if( c=='|' )
			leftButtons = false;

		if(button)
		{
			button->show(); // show visible buttons
			if(leftButtons)
			{
				m_leftButtonList.insert(m_leftButtonList.end(), button);
				m_leftButtonLayout->addWidget(button);
			}
			else
			{
				m_rightButtonList.insert(m_rightButtonList.end(), button);
				m_rightButtonLayout->addWidget(button);
			}
		}
	}
}

void GlowClient::updateButtonPixmaps()
{
	GlowClientGlobals *globals = GlowClientGlobals::instance();

	if( isSticky() )
		m_stickyButton->setPixmapName(globals->getPixmapName(
			GlowClientGlobals::StickyOn, isActive(),
			isLeft(m_stickyButton), isTool()));
	else
		m_stickyButton->setPixmapName(globals->getPixmapName(
			GlowClientGlobals::StickyOff, isActive(),
			isLeft(m_stickyButton), isTool()));

	m_helpButton->setPixmapName(globals->getPixmapName(
		GlowClientGlobals::Help, isActive(),
		isLeft(m_helpButton), isTool()));

	m_minimizeButton->setPixmapName(globals->getPixmapName(
		GlowClientGlobals::Iconify, isActive(),
		isLeft(m_minimizeButton), isTool()));

	if( isMaximized() )
		m_maximizeButton->setPixmapName(globals->getPixmapName(
			GlowClientGlobals::MaximizeOn, isActive(),
			isLeft(m_maximizeButton), isTool()));
	else
		m_maximizeButton->setPixmapName(globals->getPixmapName(
			GlowClientGlobals::MaximizeOff, isActive(),
			isLeft(m_maximizeButton), isTool()));

	m_closeButton->setPixmapName(globals->getPixmapName(
		GlowClientGlobals::Close, isActive(),
		isLeft(m_closeButton), isTool()));
}

void GlowClient::doShape()
{
	QRect r_title = m_titleSpacer->geometry();

	QRegion mask(rect());
	// edges

	mask -= QRegion(width()-1,0,1,1);
	mask -= QRegion(0,height()-1,1,1);
	mask -= QRegion(width()-1,height()-1,1,1);
	setMask(mask);
}

bool GlowClient::isLeft(GlowButton *button)
{
	for( unsigned int i=0; i<m_leftButtonList.size(); i++ )
		if( m_leftButtonList[i] == button )
			return true;
	return false;
}

bool GlowClient::isRight(GlowButton *button)
{
	for( unsigned int i=0; i<m_rightButtonList.size(); i++ )
		if( m_rightButtonList[i] == button )
			return true;
	return false;
}

void GlowClient::slotMaximize(int button)
{
	if(button == QMouseEvent::RightButton)
		maximize(MaximizeHorizontal);
	else if(button == QMouseEvent::MidButton)
		maximize(MaximizeVertical);
	else // if(button == QMouseEvent::LeftButton)
		maximize(MaximizeFull);
}

extern "C"
{
	Client * allocate(Workspace * ws, WId w)
	{
		return new GlowClient(ws, w);
	}

	void init()
	{
		GlowClientGlobals::instance();
	}

	void reset()
	{
		GlowClientGlobals::instance()->reset();
		Workspace::self()->slotResetAllClientsDelayed();
	}

	void deinit()
	{
		delete GlowClientGlobals::instance();
	}
}

#include "glowclient.moc"
