/* ****************************************************************************
  This file is part of KBabel

  Copyright (C) 1999-2000 by Matthias Kiefer
                            <matthias.kiefer@gmx.de>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

**************************************************************************** */
#ifndef CATALOGITEM_H
#define CATALOGITEM_H

#include <qstringlist.h>
#include <qlist.h>

class QTextStream;
class FlexLexer;
class EditCommand;

/**
* This class represents an entry in a catalog.
* It contains the comment, the Msgid and the Msgstr.
* It defines some functions to query the state of the entry
* (fuzzy, untranslated, cformat).
*
* @short Class, representing an entry in a catalog
* @author Matthias Kiefer <matthias.kiefer@gmx.de>
*/
class CatalogItem
{

public:
    enum IOStatus{Ok,RecoveredParseError,ParseError,Obsolete};

	/**
	 * flags, what errors this item has
	 * @see checkArgs
	 * @see checkAccel
	 * @see checkEquation
	 * @see checkContext
     * @see checkSingularPlural
	 */
	enum Error{None=0, Syntax=1, Args=2, Accel=4, Equation=8, Context=16
        , SingularPlural=32};

    /** Constructor */
    CatalogItem();
    /** Constructs a catalogitem from a string*/
    CatalogItem(QString item);
    /** Destructor */
    ~CatalogItem();

    CatalogItem(const char *msgid, const char *msgstr,
		const char *comment);

    /**
    * returns true, if the translation is fuzzy, this means
    * if the string ", fuzzy" is contained in the comment
    */
    bool isFuzzy() const;
    /**
    * returns true, if the translation in c-format, this means
    * if the string ", c-format" is contained in the comment
    */
    bool isCformat() const;
    /**
    * returns true, if the Msgid is untranslated, this means the
    * Msgstr is empty
    */
    bool isUntranslated() const;

    /**
     * @return true if this item has gettext plural forms
     */
    bool isPluralForm() const;

    bool isValid() const {return _valid;}

	void setSyntaxError(bool);

    /** returns the number of lines, the entry will need in a file */
    int totalLines() const;

    /**
    * removes the string ", fuzzy" from the comment
    * @param doIt if false, the comment is not changed, just the
    * commands for doing it are calculated
    */
    QPtrList<EditCommand> removeFuzzy(bool doIt=true);
 
    /**
    * adds the string ", fuzzy" to the comment
    * @param doIt if false, the comment is not changed, just the
    * commands for doing it are calculated
    */
    QPtrList<EditCommand> addFuzzy(bool doIt=true);


    /** cleares the item */
    void clear();


    /** returns the comment of this entry */
    QString comment() const;
    /** returns the msgid of the entry */
    QString msgid() const;
    /** returns the msgstr of the entry */
    QString msgstr() const;

    /**
     * @param plural If this is true and the item is a gettext plural
     * form item, it returns the msgid_plural as list
     * @return the msgid or the msgid_plural as list
     */
    QStringList msgidAsList(bool plural=false) const;
    /**
     * @param pluralNr If this item is a gettext plural form item,
     * it returns the plural form number pluralNr, else always the msgstr
     * is returned
     * @return the msgstr as list
     */
    QStringList msgstrAsList(int pluralNr=0) const;
    QStringList commentAsList() const;

    void setComment(QString com){_comment=com;}
    void setMsgid(QString msg){_msgid=msg;}
    void setMsgstr(QString msg){_msgstr=msg;}

    void processCommand(EditCommand* cmd, bool undo=false);


    /** returns the complete entry as a string */
    QString asString();

    /** @return a list of tags in the msgid */
    const QStringList& tagList();

	/** 
	* checks, if number and type of arguments (%1, %s,...)
	* are the same in msgid and msgstr
	* @return false, if an error has been found
	*/
	bool checkArgs(const QRegExp& comment);
	
	/**
	* checks if msgid and msgstr have the same number of occurences 
	* of accelMarker 
	* @return false, if an error has been found
	*/
	bool checkAccelerator(QChar accelMarker,const QRegExp& comment);
	
	/**
	* when the msgid is of the form abc=yxz it checks if abc is 
	* exactly the same in msgstr. Useful for checking KDE's desktop.po
	* files.
	* @return false, if an error has been found
	*/
	bool checkEquation();
	
	/**
	 * checks if the context of a message is translated.
	 * This is a patch to gettext which is used by KDE. It adds one line
	 * in front of the actual text, which starts with _: and then contains
	 * a small description. The msgstr must not translate this.
	 * @param reg a regular expression, that determines what is 
	 * the context information
	 */
	bool checkForContext(const QRegExp& reg);

    /**
     * checks if the singular and plural forms are translated correctly,
     * especially if there is more than one line
	 * @param reg a regular expression, that determines how a string with 
     * singular and plural form is determined
     * @param neededLines how many lines a string with singular-plural form
     * must have
     */
    bool checkSingularPlural( const QRegExp& reg, const int neededLines );
	
	/**
	 * @return the errors of this item. @see CatalogItem::Error
	 */
	int errors() const { return _error; }

	/**
	 * makes some sanity checks and set status accordingly
	 * @return the new status of this item
	 * @see CatalogItem::Error
	 * @param accelMarker a char, that marks the keyboard accelerators
	 * @param contextInfo a regular expression, that determines what is 
	 * the context information
	 * @param singularPlural a regular expression, that determines what is 
	 * string with singular and plural form
     * @param neededLines how many lines a string with singular-plural form
     * must have
	 */
	int checkErrors(QChar accelMarker, const QRegExp& contextInfo
            , const QRegExp& singularPlural, const int neededLines);
	
    /**
    * reads an entry from the stream
    * @see QTextStream
    */
    IOStatus read(QTextStream& stream);
    
    /**
    * Reads an entry from the stream without trying hard to resolve parsing problems.
    * It does not handle obsolete entries correctly.
    * However, it is about 50% faster than @ref @read. Suggested use is to use @ref @read after
    * a ParseError. As an input it requires a FlexLexer class returning token IDs from
    * "pofiles.h". 
    * @param lexer	instance of FlexLexer to get input from
    * @param codec	codec to be used for encoding the input into Unicode
    * @param storeText	do not store unnecesary text - it stores comments, whole header and first parts of msgid/msgstr
    */
    IOStatus fastRead(FlexLexer *lexer, QTextCodec *codec, bool storeText = true);
    
    /**
    * writes an entry to the stream
    * @see QTextStream
    */
    void write(QTextStream& stream) const;

    void operator=(const CatalogItem& rhs) {
	_comment = rhs._comment;
	_msgid = rhs._msgid;
	_msgstr = rhs._msgstr;
	_valid = rhs._valid;
    _error = rhs._error;
    _gettextPluralForm = rhs._gettextPluralForm;
    }

private:
   QString _comment;
   QStringList _msgid;
   QStringList _msgstr;

   QStringList *_tagList;

   bool _valid;
   int _error;

   /** flag, if this entry is a plural forms entry */
   bool _gettextPluralForm;
};

#endif // CATALOGITEM_H
