/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Pan - A Newsreader for Gtk+
 * Copyright (C) 2002  Charles Kerr <charles@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <config.h>

#include <string.h>

#include <glib.h>

#include <pan/base/debug.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/pan-i18n.h>

#include <pan/filters/filter-phrase.h>

const gchar * FILTER_PHRASE_CLASS_ID = "PanObject::Filter::FilterPhrase";

/************
*************  PROTECTED
************/

static Filter *
filter_phrase_dup (const Filter * f_old)
{
	Filter * f_new = filter_phrase_new ();
	FilterPhrase * fp_old = FILTER_PHRASE(f_old);
	FilterPhrase * fp_new = FILTER_PHRASE(f_new);
	filter_class_dup (f_old, f_new);
	filter_phrase_set (fp_new, fp_old->match_type, fp_old->key_type, fp_old->key);
	return f_new;
}

static char*
filter_phrase_to_string (const Filter * filter)
{
	const char * key_str = "";
	const char * match_str = "";
	const gboolean negate = filter->negate;
	const FilterPhrase * phrase = FILTER_PHRASE(filter);

	switch (phrase->key_type) {
		case PHRASE_KEY_SUBJECT:     key_str = _("Subject"); break;
		case PHRASE_KEY_AUTHOR:      key_str = _("Author"); break;
		case PHRASE_KEY_MESSAGE_ID:  key_str = _("Message-ID"); break;
		case PHRASE_KEY_REFERENCES:  key_str = _("References"); break;
		case PHRASE_KEY_XREF:        key_str = _("Xref"); break;
		default:                     g_warning ("Unrecognized key_type %d", phrase->key_type);
	}

	if (!negate) {
		switch (phrase->match_type) {
			case PHRASE_MATCH_CONTAINS:     match_str = _("contains"); break;
			case PHRASE_MATCH_IS:           match_str = _("is"); break;
			case PHRASE_MATCH_STARTS_WITH:  match_str = _("starts with"); break;
			case PHRASE_MATCH_ENDS_WITH:    match_str = _("ends with"); break;
			case PHRASE_MATCH_REGEX:        match_str = _("matches regular expression"); break;
		        default:                        g_warning ("Unrecognized matck_type %d", phrase->match_type);
		}
	} else {
		switch (phrase->match_type) {
			case PHRASE_MATCH_CONTAINS:     match_str = _("does not contain"); break;
			case PHRASE_MATCH_IS:           match_str = _("is not"); break;
			case PHRASE_MATCH_STARTS_WITH:  match_str = _("does not start with"); break;
			case PHRASE_MATCH_ENDS_WITH:    match_str = _("does not end with"); break;
			case PHRASE_MATCH_REGEX:        match_str = _("does not match regular expression"); break;
		        default:                        g_warning ("Unrecognized matck_type %d", phrase->match_type);
		}
	}

	return g_strdup_printf ("%s %s \"%s\"", key_str, match_str, phrase->key);
}

static void
filter_phrase_test_articles (Filter          * filter,
                             const Article  ** articles,
                             int               article_qty,
                             gboolean        * does_match)
{
	int i;
	FilterPhrase * phrase = FILTER_PHRASE(filter);

	for (i=0; i<article_qty; ++i)
	{
		char buf[512];
		const Article * a = articles[i];
		const char * compare_str = "";

		switch (phrase->key_type)
		{
			case PHRASE_KEY_SUBJECT:     compare_str = article_get_subject (a); break;
			case PHRASE_KEY_AUTHOR:      compare_str = article_get_author_str (a, buf, sizeof(buf)); break;
			case PHRASE_KEY_MESSAGE_ID:  compare_str = article_get_message_id (a); break;
			case PHRASE_KEY_REFERENCES:  compare_str = a->references; break;
			case PHRASE_KEY_XREF:        compare_str = a->xref; break;
		}

		if (compare_str == NULL)
		{
			does_match[i] = FALSE;
		}
		else if (phrase->match_type == PHRASE_MATCH_REGEX)
		{
			does_match[i] = !regexec (&phrase->key_regex, compare_str, 0, NULL, 0);
		}
		else
		{
			char * compare_lower = compare_str == NULL ? NULL : g_utf8_strdown (compare_str, -1);

			if (phrase->match_type == PHRASE_MATCH_IS)
			{
				does_match[i] = !pan_strcmp (compare_lower, phrase->key_lower);
			}
			else
			{
				const char * pch = strstr (compare_lower, phrase->key_lower);

				if (phrase->match_type == PHRASE_MATCH_CONTAINS)
				{
					does_match[i] = pch != NULL;
				}
				else if (phrase->match_type == PHRASE_MATCH_STARTS_WITH)
				{
					does_match[i] = pch == compare_lower;
				}
				else if (phrase->match_type == PHRASE_MATCH_ENDS_WITH)
				{
					const int compare_len = strlen (compare_lower);
					does_match[i] = pch == compare_lower + compare_len - phrase->key_len;
				}
			}

			g_free (compare_lower);
		}
	}
}

static void
filter_phrase_destructor (PanObject * o)
{
	filter_phrase_set (FILTER_PHRASE(o), PHRASE_MATCH_IS, PHRASE_KEY_SUBJECT, NULL);
	filter_destructor (o);
}

static void
filter_phrase_constructor (FilterPhrase * f)
{
	debug_enter ("filter_phase_constructor");

	filter_constructor ((Filter*)f,
	                    filter_phrase_destructor,
	                    filter_phrase_test_articles,
	                    filter_phrase_to_string,
	                    filter_phrase_dup,
	                    FILTER_PHRASE_CLASS_ID);

	f->match_type = PHRASE_MATCH_IS;
	f->key_type = PHRASE_KEY_SUBJECT;
	f->key = NULL;
	f->key_lower = NULL;
	f->key_len = 0;

	debug_exit ("filter_phase_constructor");
}

/************
*************  PUBLIC
************/

Filter*
filter_phrase_new (void)
{
	FilterPhrase * f;
	debug_enter ("filter_phrase_new");

	f = g_new0 (FilterPhrase, 1);
	filter_phrase_constructor (f);

	debug_exit ("filter_phrase_new");
	return FILTER(f);
}

void
filter_phrase_set (FilterPhrase        * filter,
                   PhraseMatchType       match_type,
                   PhraseKeyType         key_type,
                   const char          * key)
{
	debug_enter ("filter_phrase_set");

	/* sanity clause */
	g_return_if_fail (filter!=NULL);

	/* clear out old phrase */
	replace_gstr (&filter->key, NULL);
	replace_gstr (&filter->key_lower, NULL);
	if (filter->match_type == PHRASE_MATCH_REGEX)
		regfree (&filter->key_regex);

	/* repopulate the filter */
	filter->match_type = match_type;
	filter->key_type = key_type;
	if (key != NULL) {
		filter->key = g_strdup (key);
		filter->key_lower = g_utf8_strdown (key, -1);
		filter->key_len = strlen(key);
		if (filter->match_type == PHRASE_MATCH_REGEX)
			regcomp (&filter->key_regex, key, REG_EXTENDED|REG_ICASE);
	}

	debug_exit ("filter_phrase_set");
}
