/*
 * xlog - GTK+ logging program for amateur radio operators
 * Copyright (C) 2001-2003 Joop Stakenborg <pa4tu@amsat.org>
 *
 * This program is free oftware; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* callbacks_printdialog.c
 *
 * This is where the callbacks for the print dialog resides. 
 * For callbacks related to the widgets in the main window, see the files
 * called callbacks_<widgetname>.[c|h].
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <gtk/gtk.h>

#include "interface.h"
#include "support.h"
#include "types.h"
#include "log.h"
#include "utils.h"
#include "callbacks_printdialog.h"

#if WANT_HAMLIB
#	include <hamlib/rig.h>
#endif

extern GtkWidget *mainwindow;
extern GtkWidget *mainnotebook;
extern gchar *xlogdir;
extern preferencestype preferences;
extern GtkWidget *printdialog;
extern GList *logwindowlist;

#if WANT_HAMLIB
extern gint hamlibtimer;
extern RIG *myrig;
#endif

/* PRINT DIALOG */
static void run_command (gchar* system, gboolean wait);

static gboolean
latex_finished (gint pid)
{
  gchar *dvipsstring = NULL;

  if (waitpid (pid, NULL, WNOHANG | WUNTRACED))
  {
      if (preferences.printdestination == 0 && preferences.printstyle == 0)
	dvipsstring =
	  g_strdup_printf
	  ("nice -n19 dvips -t landscape log.dvi");
      else if (preferences.printdestination == 1
	       && preferences.printstyle == 0)
	dvipsstring =
	  g_strdup_printf
	  ("nice -n19 dvips -t landscape -o log.ps log.dvi");
      else if (preferences.printdestination == 0
	       && preferences.printstyle == 1)
	dvipsstring =
	  g_strdup_printf
	  ("nice -n19 dvips log.dvi");
      else if (preferences.printdestination == 1
	       && preferences.printstyle == 1)
	dvipsstring =
	  g_strdup_printf
	  ("nice -n19 dvips -o log.ps log.dvi");
    run_command(dvipsstring, FALSE);
    g_free(dvipsstring);
    return FALSE;
  }
  return TRUE;
}

static void
run_command (gchar* sys, gboolean wait)
{
  pid_t load_pid;
  gchar **cmdline = NULL, *temp;

  cmdline = g_strsplit(sys, " ", -1);

  chdir(xlogdir);
  load_pid = fork ();
  if (load_pid == 0) 
    {
#if WANT_HAMLIB
      gdk_input_remove (hamlibtimer);
      rig_close ((RIG *) myrig);
      rig_cleanup ((RIG *) myrig);
# endif
      execvp (cmdline[0], cmdline);
      _exit (1);
    } 
  else if (load_pid == -1) 
    {
      temp = g_strdup_printf (_("Could not fork child process"));
      update_statusbar (temp);
      g_free(temp);
      g_strfreev (cmdline);
      return;
    }
  g_strfreev (cmdline);
  if (wait) gtk_timeout_add (100, (GtkFunction) latex_finished, (gpointer) load_pid);
}

void
on_printokbutton_clicked (GtkButton * button, gpointer user_data)
{
  GtkWidget *printallradiobutton, *printrangeradiobutton, *printlowentry,
    *printhighentry, *sizecomboentry, *pointcomboentry,
    *conteststyleradiobutton, *normalstyleradiobutton,
    *filedestinationradiobutton, *printerdestinationradiobutton, *endhbox,
    *qslhbox, *powerhbox, *namehbox, *qthhbox, *locatorhbox, *unknown1hbox,
    *unknown2hbox, *remarksvbox;
  gchar *temp, *texpoint, *texorientation, *texsize, *texfile, *latexstring =
    NULL, **item;
  FILE *fp;
  gint i, j, k = 0, printhigh, printlow;
  gboolean printok = FALSE;
  logtype *logwindow;

  printallradiobutton = lookup_widget (printdialog, "printallradiobutton");
  printrangeradiobutton =
    lookup_widget (printdialog, "printrangeradiobutton");
  printlowentry = lookup_widget (printdialog, "printlowentry");
  printhighentry = lookup_widget (printdialog, "printhighentry");
  sizecomboentry = lookup_widget (printdialog, "sizecomboentry");
  pointcomboentry = lookup_widget (printdialog, "pointcomboentry");
  filedestinationradiobutton = lookup_widget (printdialog,
					      "filedestinationradiobutton");
  printerdestinationradiobutton = lookup_widget (printdialog,
						 "printerdestinationradiobutton");
  normalstyleradiobutton =
    lookup_widget (printdialog, "normalstyleradiobutton");
  conteststyleradiobutton =
    lookup_widget (printdialog, "conteststyleradiobutton");

  /* save settings of the print dialog */
  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (printallradiobutton)))
    preferences.printall = 0;
  else
    preferences.printall = 1;
  temp = gtk_editable_get_chars (GTK_EDITABLE (printlowentry), 0, -1);
  printlow = atoi (temp);
  temp = gtk_editable_get_chars (GTK_EDITABLE (printhighentry), 0, -1);
  printhigh = atoi (temp);
  temp = gtk_editable_get_chars (GTK_EDITABLE (sizecomboentry), 0, -1);
  if (g_strcasecmp (temp, "a4") == 0)
    preferences.printsize = 0;
  else
    preferences.printsize = 1;
  temp = gtk_editable_get_chars (GTK_EDITABLE (pointcomboentry), 0, -1);
  if (g_strcasecmp (temp, "10") == 0)
    preferences.printpoint = 0;
  else if (g_strcasecmp (temp, "11") == 0)
    preferences.printpoint = 1;
  else if (g_strcasecmp (temp, "12") == 0)
    preferences.printpoint = 2;
  if (gtk_toggle_button_get_active
      (GTK_TOGGLE_BUTTON (printerdestinationradiobutton)))
    preferences.printdestination = 0;
  else
    preferences.printdestination = 1;
  if (gtk_toggle_button_get_active
      (GTK_TOGGLE_BUTTON (normalstyleradiobutton)))
    preferences.printstyle = 0;
  else
    preferences.printstyle = 1;

  /* tex strings that may differ */
  if (preferences.printsize == 0)
    texsize = g_strdup ("a4paper");
  else
    texsize = g_strdup ("letterpaper");
  if (preferences.printstyle == 0)
    texorientation = g_strdup ("landscape");
  else
    texorientation = g_strdup ("portrait");
  if (preferences.printpoint == 0)
    texpoint = g_strdup ("10pt");
  else if (preferences.printpoint == 1)
    texpoint = g_strdup ("11pt");
  else
    texpoint = g_strdup ("12pt");

  logwindow = g_list_nth_data (logwindowlist,
			       gtk_notebook_get_current_page (GTK_NOTEBOOK
							      (mainnotebook)));
  gtk_clist_freeze (GTK_CLIST (logwindow->clist));

  item = g_new0 (gchar *, QSO_FIELDS);
  texfile = g_strconcat (xlogdir, G_DIR_SEPARATOR_S, "log.tex", NULL);
  fp = fopen (texfile, "w");
  if (fp)
    {				/* widgets which may be visible */
      endhbox = lookup_widget (mainwindow, "endhbox");
      qslhbox = lookup_widget (mainwindow, "qslhbox");
      powerhbox = lookup_widget (mainwindow, "powerhbox");
      namehbox = lookup_widget (mainwindow, "namehbox");
      qthhbox = lookup_widget (mainwindow, "qthhbox");
      locatorhbox = lookup_widget (mainwindow, "locatorhbox");
      unknown1hbox = lookup_widget (mainwindow, "unknown1hbox");
      unknown2hbox = lookup_widget (mainwindow, "unknown2hbox");
      remarksvbox = lookup_widget (mainwindow, "remarksvbox");

      fprintf (fp, "\\documentclass[%s,%s,%s]{article}\n", texsize,
	       texorientation, texpoint);
      fprintf (fp, "\\pagestyle{empty}\n");
      fprintf (fp,
	       "\\usepackage[%s,left=15mm,right=15mm,top=10mm,bottom=10mm]{geometry}\n",
	       texsize);
      fprintf (fp, "\\begin{document}\n");

      /* TABLE */
      if (preferences.printstyle == 0) /* normal style */
	{
	  fprintf (fp, "\\begin{tabular}{||l|l|l"); /* nr, date, gmt */
	  if (GTK_WIDGET_VISIBLE (endhbox))
	    fprintf (fp, "|l");	/* gmtend */
	  fprintf (fp, "|l|l|l|l|l"); /* call band mode rst myrst */
	  if (GTK_WIDGET_VISIBLE (qslhbox))
	    fprintf (fp, "|c|c"); /* qslin, qslout */
	  if (GTK_WIDGET_VISIBLE (powerhbox))
	    fprintf (fp, "|l");	/* power */
	  if (GTK_WIDGET_VISIBLE (namehbox))
	    fprintf (fp, "|l");	/* name */
	  if (GTK_WIDGET_VISIBLE (qthhbox))
	    fprintf (fp, "|l");	/* qth */
	  if (GTK_WIDGET_VISIBLE (locatorhbox))
	    fprintf (fp, "|l");	/* locator */
	  if (GTK_WIDGET_VISIBLE (unknown1hbox))
	    fprintf (fp, "|l");	/* unknown1 */
	  if (GTK_WIDGET_VISIBLE (unknown2hbox))
	    fprintf (fp, "|l");	/* unknown2 */
	  if (GTK_WIDGET_VISIBLE (remarksvbox))
	    fprintf (fp, "|l");	/*remarks */
	  fprintf (fp, "||}\n");
	}
      else /* contest style */
	fprintf (fp, "\\begin{tabular}{||l|l|l|l|l|l|l|l||l|l||}\n");
      fprintf (fp, "\\hline\n");

      /* HEADER */
      if (preferences.printstyle == 0)
	{
	  fprintf (fp, "\\textbf{Nr} &\\textbf{Date} &\\textbf{GMT}");
	  if (GTK_WIDGET_VISIBLE (endhbox))
	    fprintf (fp, "&\\textbf{END}");
	  fprintf (fp,
		   "&\\textbf{Callsign} &\\textbf{MHz} &\\textbf{Mode} &\\textbf{TX} &\\textbf{RX}");
	  if (GTK_WIDGET_VISIBLE (qslhbox))
	    fprintf (fp, "&\\textbf{Out} &\\textbf{In}");
	  if (GTK_WIDGET_VISIBLE (powerhbox))
	    fprintf (fp, "&\\textbf{Pwr}");
	  if (GTK_WIDGET_VISIBLE (namehbox))
	    fprintf (fp, "&\\textbf{Name}");
	  if (GTK_WIDGET_VISIBLE (qthhbox))
	    fprintf (fp, "&\\textbf{QTH}");
	  if (GTK_WIDGET_VISIBLE (locatorhbox))
	    fprintf (fp, "&\\textbf{Locator}");
	  if (GTK_WIDGET_VISIBLE (unknown1hbox))
	    {
	      fprintf (fp, "&\\textbf{");
	      fprintf (fp, preferences.freefield1);
	      fprintf (fp, "}");
	    }
	  if (GTK_WIDGET_VISIBLE (unknown2hbox))
	    {
	      fprintf (fp, "&\\textbf{");
	      fprintf (fp, preferences.freefield2);
	      fprintf (fp, "}");
	    }
	  if (GTK_WIDGET_VISIBLE (remarksvbox))
	    fprintf (fp, "&\\textbf{Remarks}");
	  fprintf (fp, "\\\\\n");
	}
      else
	fprintf (fp,
		 "\\textbf{Nr} &\\textbf{Date} &\\textbf{GMT} &\\textbf{Callsign} &\\textbf{MHz} &\\textbf{Mode} &\\textbf{TX} &\\textbf{RX} &\\textbf{Multiplier} &\\textbf{Points}\\\\\n");
      fprintf (fp, "\\hline\n");
      fprintf (fp, "\\hline\n");


      if (preferences.printall == 0)
	{
	  printlow = 1;
	  printhigh = logwindow->qsos;
	}


      /* QSO s */
      if (GTK_CLIST (logwindow->clist)->rows > 0)
	{
	  for (i = GTK_CLIST (logwindow->clist)->rows - printlow;
	       i >= GTK_CLIST (logwindow->clist)->rows - printhigh; i--)
	    {
	      for (j = 0; j < QSO_FIELDS; j++)
		gtk_clist_get_text (GTK_CLIST (logwindow->clist), i, j,
				    &item[j]);
	      if (preferences.printstyle == 0)
		{
		  fprintf (fp, "%s&%s&%s", item[NR], item[DATE], item[GMT]);
		  if (GTK_WIDGET_VISIBLE (endhbox))
		    fprintf (fp, "&%s", item[GMTEND]);
		  fprintf (fp, "&\\textbf{%s}&%s&%s&%s&%s",
			   item[CALL], item[BAND], item[MODE], item[RST],
			   item[MYRST]);
		  if (GTK_WIDGET_VISIBLE (qslhbox))
		    fprintf (fp, "&%s&%s", item[QSLOUT], item[QSLIN]);
		  if (GTK_WIDGET_VISIBLE (powerhbox))
		    fprintf (fp, "&%s", item[POWER]);
		  if (GTK_WIDGET_VISIBLE (namehbox))
		    fprintf (fp, "&%s", item[NAME]);
		  if (GTK_WIDGET_VISIBLE (qthhbox))
		    fprintf (fp, "&%s", item[QTH]);
		  if (GTK_WIDGET_VISIBLE (locatorhbox))
		    fprintf (fp, "&%s", item[LOCATOR]);
		  if (GTK_WIDGET_VISIBLE (unknown1hbox))
		    fprintf (fp, "&%s", item[U1]);
		  if (GTK_WIDGET_VISIBLE (unknown2hbox))
		    fprintf (fp, "&%s", item[U2]);
		  if (GTK_WIDGET_VISIBLE (remarksvbox))
		    {		/* escape # for latex, TODO: &, _, %, $, \, {, }, ^ */
		      if (strstr (item[REMARKS], "#"))
                      {
			temp = strreplace (item[REMARKS], "#", "\\#");
                        fprintf (fp, "&%s", temp);
                      }
                      else
                        fprintf (fp, "&%s", item[REMARKS]);
		    }
		  fprintf (fp, "\\\\\n");
		}
	      else
		fprintf (fp, "%d&%s&%s&\\textbf{%s}&%s&%s&%s&%s&~&~\\\\\n",
			 k + 1, item[DATE], item[GMT], item[CALL], item[BAND],
			 item[MODE], item[RST], item[MYRST]);
	      k++;
	      if ((k % 30 == 0 && preferences.printstyle == 0
		   && k != printhigh - printlow + 1) || (k % 50 == 0
							 && preferences.
							 printstyle == 1
							 && k !=
							 printhigh -
							 printlow + 1))
		{
		  fprintf (fp, "\\hline\n");
		  fprintf (fp, "\\end{tabular}\n");
		  fprintf (fp, "\\begin{center}\n");
		  fprintf (fp, "\\emph{Printed by xlog version %s}\n",
			   VERSION);
		  fprintf (fp, "\\end{center}\n");
		  fprintf (fp, "\\newpage\n");
		  if (preferences.printstyle == 0)
		    fprintf (fp,
			     "\\begin{tabular}{||l|l|l|l|l|l|l|l||l|c|c||}\n");
		  else
		    fprintf (fp,
			     "\\begin{tabular}{||l|l|l|l|l|l|l|l||l|l||}\n");
		  fprintf (fp, "\\hline\n");
		  if (preferences.printstyle == 0)
		    {
		      fprintf (fp,
			       "\\textbf{Nr} &\\textbf{Date} &\\textbf{GMT}");
		      if (GTK_WIDGET_VISIBLE (endhbox))
			fprintf (fp, "&\\textbf{END}");
		      fprintf (fp,
			       "&\\textbf{Callsign} &\\textbf{MHz} &\\textbf{Mode} &\\textbf{TX} &\\textbf{RX}");
		      if (GTK_WIDGET_VISIBLE (qslhbox))
			fprintf (fp, "&\\textbf{Out} &\\textbf{In} ");
		      if (GTK_WIDGET_VISIBLE (powerhbox))
			fprintf (fp, "&\\textbf{Pwr}");
		      if (GTK_WIDGET_VISIBLE (namehbox))
			fprintf (fp, "&\\textbf{Name}");
		      if (GTK_WIDGET_VISIBLE (qthhbox))
			fprintf (fp, "&\\textbf{QTH}");
		      if (GTK_WIDGET_VISIBLE (locatorhbox))
			fprintf (fp, "&\\textbf{Locator}");
		      if (GTK_WIDGET_VISIBLE (unknown1hbox))
			{
			  fprintf (fp, "&\\textbf{");
			  fprintf (fp, preferences.freefield1);
			  fprintf (fp, "}");
			}
		      if (GTK_WIDGET_VISIBLE (unknown2hbox))
			{
			  fprintf (fp, "&\\textbf{");
			  fprintf (fp, preferences.freefield2);
			  fprintf (fp, "}");
			}
		      if (GTK_WIDGET_VISIBLE (remarksvbox))
			fprintf (fp, "&\\textbf{Remarks}");
		      fprintf (fp, "\\\\\n");
		    }
		  else
		    fprintf (fp,
			     "\\textbf{Nr} &\\textbf{Date} &\\textbf{GMT} &\\textbf{Callsign} &\\textbf{MHz} &\\textbf{Mode} &\\textbf{TX} &\\textbf{RX} &\\textbf{Multiplier} &\\textbf{Points}\\\\\n");
		  fprintf (fp, "\\hline\n");
		  fprintf (fp, "\\hline\n");
		}
	    }
	}

      fprintf (fp, "\\hline\n");
      fprintf (fp, "\\end{tabular}\n");
      fprintf (fp, "\\begin{center}\n");
      fprintf (fp, "\\emph{Printed by xlog version %s}\n", VERSION);
      fprintf (fp, "\\end{center}\n");
      fprintf (fp, "\\end{document}\n");
      fclose (fp);
    }

  gtk_clist_thaw (GTK_CLIST (logwindow->clist));
  g_free (item);

  if ((printok = latexavailable ()) == FALSE)
    {
      update_statusbar (_
			("Latex is not available, install the tetex package"));
    }

  if (printok)
    {
      latexstring =
	  g_strdup_printf
	  ("nice -n19 latex -interaction=nonstopmode log.tex");
      if (preferences.printdestination == 0)
        temp = g_strdup_printf (_("Printing in the background"));
      else
        temp = g_strdup_printf (_("Printing in the background to %s%slog.ps"),
          xlogdir, G_DIR_SEPARATOR_S);
      run_command (latexstring, TRUE);
      update_statusbar (temp);
      g_free (latexstring);
    }

  g_free (texsize);
  g_free (texorientation);
  g_free (texpoint);
  g_free (texfile);
  g_free (temp);

  gtk_widget_set_sensitive (mainwindow, 1);
  gtk_widget_destroy (gtk_widget_get_toplevel (GTK_WIDGET (button)));
}

/* cancel clicked, nothing printed */
void
on_printcancelbutton_clicked (GtkButton * button, gpointer user_data)
{
  gtk_widget_set_sensitive (mainwindow, 1);
  gtk_widget_destroy (gtk_widget_get_toplevel (GTK_WIDGET (button)));
}

/* print dialog is deleted */
gboolean
on_printdialog_delete_event (GtkWidget * widget, GdkEvent * event,
			     gpointer user_data)
{
  gtk_widget_set_sensitive (mainwindow, 1);
  return FALSE;
}

/* set sensitivity of range widgets in print dialog */
void
on_printradiobutton_toggled (GtkToggleButton * togglebutton,
			     gpointer user_data)
{
  GtkWidget *printlowentry, *printrangelabel, *printhighentry;

  printlowentry = lookup_widget (printdialog, "printlowentry");
  printhighentry = lookup_widget (printdialog, "printhighentry");
  printrangelabel = lookup_widget (printdialog, "printrangelabel");
  switch (atoi (user_data))
    {
    case 1:
      gtk_widget_set_sensitive (printlowentry, 0);
      gtk_widget_set_sensitive (printhighentry, 0);
      gtk_widget_set_sensitive (printrangelabel, 0);
      break;
    case 2:
      gtk_widget_set_sensitive (printlowentry, 1);
      gtk_widget_set_sensitive (printhighentry, 1);
      gtk_widget_set_sensitive (printrangelabel, 1);
      break;
    }
}


/* end of file */
