/* srspeech.c
 *
 * Copyright 2001, 2002 Sun Microsystems, Inc.,
 * Copyright 2001, 2002 BAUM Retec, A.G.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <string.h>
#include <stdio.h>
#include <stdlib.h>

#include "SRMessages.h"
#include "srspeech.h"
#include "spgs.h"

#undef	SRSSPEECH_DEBUG 
#undef	SRSSPEECH_CB_DEBUG 
#undef	SRSSPEECH_MARKER_DEBUG
/*_________________________________< GLOBALS >________________________________*/

GHashTable	*srs_voices_hash_table;
GQueue      	*srs_queue;
TTSEngine  	*current_engine;
SRSText    	*CurrText;
gboolean    	is_speaking;
static int 	cb_support_tranzition = 1;
/*_________________________________</GLOBALS >________________________________*/

/* ___________________________< SRSVoice METHODS>_____________________________*/

SRSVoice * 
srs_voice_new ()
{
    SRSVoice *srs_voice;
    
    srs_voice = g_new0 (SRSVoice, 1);

    srs_voice->cb_support = -1;
    	
    return srs_voice;
}

void 
srs_voice_free (SRSVoice *srs_voice)
{
    if (srs_voice)
    {
	/* free the fields */
	g_free (srs_voice->tts_engine_name);
	g_free (srs_voice->tts_voice_name);
	g_free (srs_voice->voice_id);
	
	g_free (srs_voice);
	srs_voice = NULL;
    }
}

gboolean
srs_voice_is_eqal(SRSVoice *voice1, SRSVoice *voice2)
{
  gboolean rv = FALSE;



  if (voice1 && voice2 &&
      voice1->tts_voice_name && voice2->tts_voice_name
      /* voice1->tts_engine_name && voice2->tts_engine_name && */
      /* voice1->voice_id && voice2->voice_id */
  )
  {
    rv = (strcmp(voice1->tts_voice_name, voice2->tts_voice_name) == 0) &&
         /* (strcmp(voice1->tts_engine_name, voice2->tts_engine_name) == 0) && */
         /* (g_strcasecmp(voice1->voice_id, voice2->voice_id) == 0) && */
         (voice1->rate  == voice2->rate) &&
	       (voice1->pitch == voice2->pitch) &&
	       (voice1->volume == voice2->volume);
  }

  return rv;

}

SRSVoice *
srs_voice_copy (SRSVoice *orig_srs_voice)
{
    SRSVoice* 	srs_voice;
	
    srs_voice = g_malloc0 (sizeof (SRSVoice) );
    memcpy (srs_voice, 
	    orig_srs_voice,
	    sizeof (SRSVoice) );

    srs_voice->voice_id        = g_strdup (orig_srs_voice->voice_id);	
    srs_voice->tts_engine_name = g_strdup (orig_srs_voice->tts_engine_name);
    srs_voice->tts_voice_name  = g_strdup (orig_srs_voice->tts_voice_name);

    return srs_voice;
}

SRSVoice *
srs_voice_dup (SRSVoice *orig_srs_voice)
{
    return (SRSVoice *)orig_srs_voice;
}

gboolean
srs_voice_find (SRSVoice **srs_voice)
{
    SRSVoice 	*voice_found 	= NULL;
    gboolean   	rv          	= FALSE;
	

    if (*srs_voice && (*srs_voice)->voice_id)
    {
	 /* search voice in the container first*/
	voice_found = srs_get_voice ((*srs_voice)->voice_id);

	if (voice_found)
	{
	    /*voice found*/
	    srs_voice_free (*srs_voice);
	    *srs_voice = (SRSVoice *) srs_voice_dup (voice_found);
	     /* fprintf (stderr, "Find voice ID:%x, ENG:%x, V:%x\n", 
	    		(*srs_voice)->voice_id, 
			(*srs_voice)->tts_engine_name, 
			(*srs_voice)->tts_voice_name); 
	    */
    	    rv = TRUE;
	}
    }
    return rv;
}


void 
srs_voice_set_id (SRSVoice 	*srs_voice, 
		  gchar     	*id)
{
    if (srs_voice->voice_id) 
	g_free (srs_voice->voice_id);	
    srs_voice->voice_id = g_strdup (id);
}

void 
srs_voice_set_tts_engine_name (SRSVoice 	*srs_voice,
			       gchar     	*tts_engine_name)
{
    if (srs_voice->tts_engine_name) 
	g_free (srs_voice->tts_engine_name);	
    srs_voice->tts_engine_name = g_strdup (tts_engine_name);
    /* 
    fprintf (stderr, "EN:%d\n", srs_voice->tts_engine_name); 
    */
}

void 
srs_voice_set_tts_voice_name (SRSVoice	*srs_voice, 
			      gchar     *tts_voice_name)
{
    if (srs_voice->tts_voice_name) 
	g_free (srs_voice->tts_voice_name);	
    srs_voice->tts_voice_name = g_strdup (tts_voice_name);
    /* 
    fprintf (stderr, "VN:%s\n", srs_voice->tts_voice_name); 
    */
}

void 
srs_voice_set_priority (SRSVoice	*srs_voice, 
			gchar     	*priority)
{
    srs_voice->priority = atoi(priority);
}

void 
srs_voice_set_preempt (SRSVoice	*srs_voice, 
		       gchar	*preempt)
{
    if ( (g_strcasecmp(preempt, "yes")  == 0) ||
         (g_strcasecmp(preempt, "true") == 0) ||
         (g_strcasecmp(preempt, "1")    == 0) )
    {
	srs_voice->preempt = TRUE;
    }
    else
    {	
	srs_voice->preempt = FALSE;
    }
}

void 
srs_voice_set_rate (SRSVoice	*srs_voice,
		    gchar     	*rate)
{
    if (!srs_voice) 
	return; 

    if (rate)
    {
	if ( (strncasecmp ("+", rate, 1) != 0 ) &&
	     (strncasecmp ("-", rate, 1) != 0 ) )
	    srs_voice->rate = 0;
	srs_voice->rate += atoi(rate);
    }

#ifdef SRSSPEECH_PARAM_DEBUG
    fprintf (stderr, "\nsrspeech : Voice %s has rate = %d.",
	    srs_voice->voice_id, 
	    srs_voice->rate);
#endif		
}

void 
srs_voice_set_pitch (SRSVoice 	*srs_voice, 
		     gchar     	*pitch)
{
    if (!srs_voice) 
	return; 
	
    if (pitch)
    {
	if ( (strncasecmp ("+", pitch, 1) != 0 ) &&
	     (strncasecmp ("-", pitch, 1) != 0 ) )
	    srs_voice->pitch = 0;
	srs_voice->pitch += atoi(pitch);
    }
	
#ifdef SRSSPEECH_PARAM_DEBUG
    fprintf (stderr, "\nsrspeech : Voice %s has pitch = %d.",
	    srs_voice->voice_id, 
	    srs_voice->pitch);
#endif		
}

void 
srs_voice_set_volume (SRSVoice *srs_voice,
		      gchar     *volume)
{
    if (!srs_voice) 
	return; 
	
    if (volume)
    {
	if ( (strncasecmp ("+", volume, 1) != 0 ) &&
	     (strncasecmp ("-", volume, 1) != 0 ) )
	    srs_voice->volume = 0;
	srs_voice->volume += atoi(volume);
    }
	
#ifdef SRSSPEECH_PARAM_DEBUG
    fprintf (stderr, "\nsrspeech : Voice %s has volume = %d.",
	    srs_voice->voice_id, 
	    srs_voice->volume);
#endif		
}
/*____________________________</SRSVoice METHODS >____________________________*/

/*__________________________<SRSVoices CONTAINER >____________________________*/
void 
srs_voices_init (void)
{
    srs_voices_hash_table = g_hash_table_new (g_str_hash,
				    	      g_str_equal);
}

void 
srs_voices_flush (gpointer 	key, 
		  gpointer 	value, 
		  gpointer 	user_data)
{
    if (key)
    {
        g_free (key);
        key = NULL;
    }
    if (value)
    {
        g_free (value);
        value = NULL;
    }
}

void 
srs_voices_terminate (void)
{
    if (!srs_voices_hash_table) 
	return;
	
    g_hash_table_foreach (srs_voices_hash_table, 
			  srs_voices_flush, 
			  NULL);
    g_hash_table_destroy (srs_voices_hash_table);
    srs_voices_hash_table = NULL;
}

SRSVoice *
srs_get_voice (gchar *voice_name)
{
    SRSVoice *rv = NULL;

    rv = (SRSVoice *) g_hash_table_lookup (srs_voices_hash_table, 
					   voice_name);
    return rv;
}


void 
srs_add_voice (SRSVoice *srs_voice)
{
    SRSVoice *srv = NULL;

    if (srs_voice && srs_voice->voice_id)
    {
	/* search voice in the container first */
	srv = srs_get_voice (srs_voice->voice_id);
	
	if (!srv)
	{
	    /* voice not found, create new copy of the voice */
	    srv = srs_voice_copy (srs_voice);
			
	    /* add copy to the other SRS voices (srs_voices_hash_table owns it) */
	    g_hash_table_insert (srs_voices_hash_table, 
				 srv->voice_id, 
				 srv);
	}		
    }
}
/*__________________________</SRSVoices CONTAINER >____________________________*/

/*_____________________________< SRSText METHODS >_____________________________*/
SRSText *
srs_text_new (void)
{
    SRSText *srs_text;
    
    srs_text = g_new0 (SRSText, 1);
	
    return srs_text;
}

void 
srs_text_free (SRSText *srs_text)
{
    if (srs_text)
    {
	if (srs_text->voice)  
	    g_free(srs_text->voice);
	if (srs_text->marker) 
	    g_free(srs_text->marker);
	if (srs_text->spelling)   
    	    g_free(srs_text->spelling);
	if (srs_text->language)   
	    g_free(srs_text->language);
	if (srs_text->text)   
	    g_free(srs_text->text);
	
	g_free (srs_text);
	srs_text = NULL;	
    }
}

SRSText *
srs_text_copy (SRSText *orig_srs_text)
{
    SRSText *srs_text;
	
    srs_text = g_malloc0 (sizeof (SRSText) );
    memcpy (srs_text,
            orig_srs_text, 
	    sizeof (SRSText) );
	
    srs_text->voice  = g_strdup (orig_srs_text->voice);	
    srs_text->marker = g_strdup (orig_srs_text->marker);
    srs_text->text   = g_strdup (orig_srs_text->text);
	
    return srs_text;
}

void 
srs_text_set_voice (SRSText	*srs_text, 
		    gchar	*voice)
{
    if (srs_text->voice) 
	g_free (srs_text->voice);
    srs_text->voice = g_strdup (voice);	
}

void 
srs_text_set_marker (SRSText 	*srs_text,
		     gchar    	*marker)
{
    if (srs_text->marker) 
	g_free (srs_text->marker);
    srs_text->marker = g_strdup (marker);	
}

void 
srs_text_set_language (SRSText 	*srs_text, 
		       gchar    *language)
{
    if (srs_text->language) 
	g_free (srs_text->language);
    srs_text->language = g_strdup (language);
#ifdef SRSSPEECH_DEBUG
    fprintf (stderr, "\nsrspeech : Language is %s.", 
	    srs_text->language);
#endif		
}

void 
srs_text_set_spelling (SRSText	*srs_text,
		       gchar	*spelling)
{
    if (srs_text->spelling) 
	g_free (srs_text->spelling);
    srs_text->spelling = g_strdup (spelling);
#ifdef SRSSPEECH_DEBUG
    fprintf (stderr, "\nsrspeech : Spelling mode is %s.", 
	    srs_text->spelling);
#endif		
}

void 
srs_text_set_text (SRSText	*srs_text,
		    gchar	*text)
{
    if (srs_text->text) 
	g_free (srs_text->text);	
    srs_text->text = g_strdup (text);
}

void 
srs_text_add_text (SRSText 	*srs_text,
		   gchar    	*text)
{
    gchar* ts = NULL;
	
    if (text)
    {
	if (srs_text->text)
	{	
	    /* subsequent text */
	    ts = srs_text->text;
	    srs_text->text = g_strconcat (srs_text->text,
	    			          text,
					  NULL);
	    g_free (ts);
	    ts = NULL;
	}
	else
	{
    	    /* 1'st text */
	    srs_text->text = g_strdup (text);
	}
    }	
}
/*_____________________________</SRSText METHODS >_____________________________*/

/*__________________________________< API >____________________________________*/
void
tts_callback (TTSEvent 	event, 
	      gpointer 	event_data, 
	      gpointer 	user_data)
{
#ifdef SRSSPEECH_CB_DEBUG
    static int 	cb_start_n = 0;
    static int 	cb_end_n = 0;
    static int 	cb_n = 0;

    cb_n++;
#endif
    
    switch (event)
    {
	case TTS_EV_SPEECH_STARTED:
#ifdef SRSSPEECH_CB_DEBUG
	    cb_start_n++;
	    fprintf (stderr, "\nstart event #%4d, event #%4d",
				cb_start_n,
				cb_n);
#endif
/*	    is_speaking = TRUE;*/
	    break;
	
        case TTS_EV_END_OF_SPEECH:
#ifdef SRSSPEECH_CB_DEBUG
	    cb_end_n++;
	    fprintf (stderr, "\nend   event #%4d, event #%4d",
				cb_end_n,
				cb_n);
#endif
	    if (CurrText && CurrText->text && is_speaking) /* !!! TBR !!! - CHECK WHY THIS CAN HAPPEN */
    	    {
		is_speaking = FALSE;

    		 /* free current item*/
        	srs_text_free(CurrText);
		CurrText = NULL;

    		 /* get next item(s) from queue */
    		 if (!g_queue_is_empty (srs_queue) )
    		 {
        	     CurrText = g_queue_pop_head (srs_queue);
		    
		    /* OPTIMIZATION FOR GS - concatenate texts with same 
		    voices */
		    srs_speak_optimization ();
        	    
		    srs_speak_to_engine (CurrText);	
#ifdef SRSSPEECH_CB_DEBUG		    
		    fprintf (stderr, "\ntext to say (callback) %s",CurrText->text);
#endif
    		} /*end if (!g_queue_is_empty (srs_queue) )*/
    	   }/* end if (CurrText && CurrText->text && is_speaking)*/
    	   break;
	case TTS_EV_WORD_MARKER:
    	    /* noting to do yet */
#ifdef SRSSPEECH_MARKER_DEBUG
    	    sru_message ("Word marker\n");
#endif
    	    break;
	default:
    	    sru_warning ("Unknown TTS event");
    	    break;
    }
}


int
srs_initialize (void)
{	
    srs_queue = g_queue_new();	
    srs_voices_init();
    current_engine = g_malloc0 (sizeof(TTSEngine));
    current_engine->callback = tts_callback;

    /* initialize gnome speech */
    return gs_init (current_engine);
}

void 
srs_terminate (void)
{	
    gs_terminate ();
    srs_voices_terminate();
    g_queue_free (srs_queue);
}


void 
srs_speak_to_engine (SRSText *srs_text)
{
    SRSVoice* srv = NULL;
    int cb_support_temp = 1;
    	
    /* get SRSVoice from the SRSText */
    if (srs_text && srs_text->voice)
    {
	srv = srs_get_voice (srs_text->voice);
	if (srv)
	{
	    /* get the TTSEngine from SRSVoice and select it as the current engine	*/
	    /* tts_select_engine (srv); */ 
	    /* !!! TBI !!! when we support multiple engines. 
	    Right now we support only gnome-speech */
			
	    /* consider preemptive speech item */
	    if (srv->preempt)
	    {
		if (current_engine && current_engine->shut_up)
		{
		    current_engine->shut_up();
		}
	    }
	    if (srv->cb_support == 1)
		cb_support_temp = 1;
	    else
		cb_support_temp = 0;
	}					
	if (current_engine && current_engine->speak )
	{			
	    /* !!! TBI !!! add callback data here, at least marker */
#ifdef SRSSPEECH_CB_DEBUG
	    fprintf (stderr, "\n\tsend to GS: %s", srs_text->text); 
#endif
	    current_engine->speak (srv, srs_text);
	    if (cb_support_tranzition)
		is_speaking = TRUE;
	    else
		is_speaking = FALSE;
	}			
	cb_support_tranzition = cb_support_temp;					
    } 	
}

/*
gboolean
tts_is_speaking ()
{
  return FALSE;
}
*/

int 
srs_speak_optimization (void)
{
    SRSText    	*tt;
    SRSVoice    *v1, *v2;
    static gboolean busy = FALSE;
    
    if (busy)
    {
	fprintf (stderr, "\nBUSY");
	return FALSE;
    }
    busy = TRUE;

    /* OPTIMIZATION FOR GS - concatenate texts with same 
    voices */
    while (!g_queue_is_empty (srs_queue))
    {
	tt = g_queue_peek_head (srs_queue);

	v1 = srs_get_voice (CurrText->voice);
	v2 = srs_get_voice (tt->voice);

	if (srs_voice_is_eqal (v1, v2))
	{
    	    /* fprintf (stderr, "same voice\n"); */
            /* same voice, concatenate text */
            tt = g_queue_pop_head (srs_queue);
/* FIXME: rd@baum.ro - you should provide spaces between the chuncks */
	    srs_text_add_text (CurrText, g_strdup(" ")); 
			    
	    srs_text_add_text (CurrText, tt->text);
	    srs_text_free(tt);
	}
	else
	{
	    /* different voice, exit loop  */
	    /* fprintf (stderr, "different voices\n"); */
	    break;
	}
    } /* while (!g_queue_is_empty (srs_queue)) */
    busy = FALSE;
    return TRUE;
}

int
srs_speak (SRSText *srs_text)
{
    int rv = FALSE;
    /* !!! TBR !!! if current_engine is not speaking ?*/
		
    if (!cb_support_tranzition)
	is_speaking = FALSE;

    if (is_speaking)
    {
	    /* speaking, add to queue */
#ifdef SRSSPEECH_DEBUG	    
	    fprintf (stderr, "\n\tin queue %s", srs_text->text); 
#endif
	    g_queue_push_tail (srs_queue, srs_text);
    }
    else
    {
#ifdef SRSSPEECH_DEBUG    
	     fprintf (stderr, "\n\tdirectly %s", srs_text->text); 
#endif
	    /* not speaking, speak directly */
/*	   
	    if (!busy_queue)
	    { 
	    busy_queue = TRUE;
    	    if (!g_queue_is_empty (srs_queue) )
    	    {
            	CurrText = g_queue_pop_head (srs_queue);
		    
		    srs_speak_optimization ();
        	    
	    }
	    }
	    busy_queue = FALSE;	    
	    
	    if (CurrText)
		srs_text_add_text (CurrText, srs_text);
	    else
*/		CurrText = srs_text;
    	    
	    srs_speak_to_engine (CurrText);	

#ifdef SRSSPEECH_DEBUG
	    fprintf (stderr, "\nCRT text directly %s",CurrText->text);
#endif
	    rv = TRUE;
    }
    return rv;
}


int 
srs_shut_up ()
{
    int rv = FALSE;
    SRSText  *st;

#ifdef SRSSPEECH_CB_DEBUG
    fprintf (stderr, "\n\tSHUT_UP"); 						
#endif    
    while (!g_queue_is_empty (srs_queue))
    {
	st = g_queue_pop_head (srs_queue);
	srs_text_free (st);
	CurrText = NULL;
    }

    if (current_engine && current_engine->shut_up)
    {
	current_engine->shut_up();
	rv = TRUE;
    }
    is_speaking = FALSE;
	
    return rv;
}


int
srs_pause ()
{
    if (current_engine && current_engine->pause)
    {
	current_engine->pause();  
    }
    is_speaking = FALSE;

    return 0;
}

int 
srs_resume ()
{
    if (current_engine && current_engine->resume)
    {
	/* NOTE: GS doesn't currently support that */    
	current_engine->resume(); 
	is_speaking = TRUE;
    }

    return 0;
}
/*__________________________________</API >____________________________________*/
