// SPDX-License-Identifier: GPL-2.0-or-later
/*
 *
 *  BlueZ - Bluetooth protocol stack for Linux
 *
 *  Copyright (C) 2014-2015  Intel Corporation. All rights reserved.
 *
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <stdint.h>
#include <unistd.h>
#include <errno.h>
#include <stdbool.h>

#include <glib.h>

#include "lib/bluetooth.h"
#include "lib/hci.h"
#include "lib/mgmt.h"

#include "monitor/bt.h"
#include "emulator/bthost.h"
#include "emulator/hciemu.h"

#include "src/shared/tester.h"
#include "src/shared/mgmt.h"
#include "src/shared/hci.h"
#include "src/shared/util.h"

struct test_data {
	struct mgmt *mgmt;
	uint16_t mgmt_index;
	struct hciemu *hciemu;
	enum hciemu_type hciemu_type;
	const void *test_data;
	unsigned int remove_id;
	struct bt_hci *hci;
	uint32_t current_settings;
};

static void mgmt_debug(const char *str, void *user_data)
{
	const char *prefix = user_data;

	tester_print("%s%s", prefix, str);
}

static void read_info_callback(uint8_t status, uint16_t length,
					const void *param, void *user_data)
{
	struct test_data *data = tester_get_data();
	const struct mgmt_rp_read_info *rp = param;
	char addr[18];
	uint16_t manufacturer;
	uint32_t supported_settings, current_settings;

	tester_print("Read Info callback");
	tester_print("  Status: 0x%02x", status);

	if (status || !param) {
		tester_pre_setup_failed();
		return;
	}

	ba2str(&rp->bdaddr, addr);
	manufacturer = btohs(rp->manufacturer);
	supported_settings = btohl(rp->supported_settings);
	current_settings = btohl(rp->current_settings);

	tester_print("  Address: %s", addr);
	tester_print("  Version: 0x%02x", rp->version);
	tester_print("  Manufacturer: 0x%04x", manufacturer);
	tester_print("  Supported settings: 0x%08x", supported_settings);
	tester_print("  Current settings: 0x%08x", current_settings);
	tester_print("  Class: 0x%02x%02x%02x",
			rp->dev_class[2], rp->dev_class[1], rp->dev_class[0]);
	tester_print("  Name: %s", rp->name);
	tester_print("  Short name: %s", rp->short_name);

	data->current_settings = current_settings;

	if (strcmp(hciemu_get_address(data->hciemu), addr)) {
		tester_pre_setup_failed();
		return;
	}

	tester_pre_setup_complete();
}

static void index_added_callback(uint16_t index, uint16_t length,
					const void *param, void *user_data)
{
	struct test_data *data = tester_get_data();

	tester_print("Index Added callback");
	tester_print("  Index: 0x%04x", index);

	if (data->mgmt_index != MGMT_INDEX_NONE)
		return;

	data->mgmt_index = index;

	mgmt_send(data->mgmt, MGMT_OP_READ_INFO, data->mgmt_index, 0, NULL,
					read_info_callback, NULL, NULL);
}

static void index_removed_callback(uint16_t index, uint16_t length,
					const void *param, void *user_data)
{
	struct test_data *data = tester_get_data();

	tester_print("Index Removed callback");
	tester_print("  Index: 0x%04x", index);

	if (index != data->mgmt_index)
		return;

	if (data->remove_id) {
		mgmt_unregister(data->mgmt, data->remove_id);
		data->remove_id = 0;
		tester_test_passed();
		return;
	}

	mgmt_unregister_index(data->mgmt, data->mgmt_index);

	mgmt_unref(data->mgmt);
	data->mgmt = NULL;

	tester_post_teardown_complete();
}

static void read_index_list_callback(uint8_t status, uint16_t length,
					const void *param, void *user_data)
{
	struct test_data *data = tester_get_data();

	tester_print("Read Index List callback");
	tester_print("  Status: 0x%02x", status);

	if (status || !param) {
		tester_pre_setup_failed();
		return;
	}

	mgmt_register(data->mgmt, MGMT_EV_INDEX_ADDED, MGMT_INDEX_NONE,
					index_added_callback, NULL, NULL);

	data->hciemu = hciemu_new(data->hciemu_type);
	if (!data->hciemu) {
		tester_warn("Failed to setup HCI emulation");
		tester_pre_setup_failed();
	}

	tester_print("New hciemu instance created");
}

static void test_pre_setup(const void *test_data)
{
	struct test_data *data = tester_get_data();

	data->mgmt = mgmt_new_default();
	if (!data->mgmt) {
		tester_warn("Failed to setup management interface");
		tester_pre_setup_failed();
		return;
	}

	if (tester_use_debug())
		mgmt_set_debug(data->mgmt, mgmt_debug, "mgmt: ", NULL);

	mgmt_send(data->mgmt, MGMT_OP_READ_INDEX_LIST, MGMT_INDEX_NONE, 0, NULL,
					read_index_list_callback, NULL, NULL);
}

static void test_post_teardown(const void *test_data)
{
	struct test_data *data = tester_get_data();

	mgmt_register(data->mgmt, MGMT_EV_INDEX_REMOVED, data->mgmt_index,
					index_removed_callback,
					NULL, NULL);

	hciemu_unref(data->hciemu);
	data->hciemu = NULL;
}

static void test_data_free(void *test_data)
{
	struct test_data *data = test_data;

	free(data);
}

static void setup_powered_client_callback(uint8_t status, uint16_t length,
					const void *param, void *user_data)
{
	if (status != MGMT_STATUS_SUCCESS) {
		tester_setup_failed();
		return;
	}

	tester_print("Controller powered on");

	tester_setup_complete();
}

static void setup_powered(const void *test_data)
{
	struct test_data *data = tester_get_data();
	unsigned char param[] = { 0x01 };

	tester_print("Powering on controller");

	mgmt_send(data->mgmt, MGMT_OP_SET_POWERED, data->mgmt_index,
			sizeof(param), param, setup_powered_client_callback,
			NULL, NULL);
}

static void toggle_powered(const void *test_data);

static void toggle_powered_client_callback(uint8_t status, uint16_t length,
					const void *param, void *user_data)
{
	bool power = PTR_TO_INT(user_data);

	if (status != MGMT_STATUS_SUCCESS) {
		tester_setup_failed();
		return;
	}

	tester_print("Controller powered %s", power ? "on" : "off");

	if (power)
		toggle_powered(false);
	else
		tester_setup_complete();
}

static void toggle_powered(const void *test_data)
{
	struct test_data *data = tester_get_data();
	bool power = PTR_TO_INT(test_data);
	unsigned char param[1];

	param[0] = power ? 0x01 : 0x00;

	tester_print("Powering %s controller", power ? "on" : "off");

	mgmt_send(data->mgmt, MGMT_OP_SET_POWERED, data->mgmt_index,
			sizeof(param), param, toggle_powered_client_callback,
			INT_TO_PTR(power), NULL);
}

static void test_open_success(const void *test_data)
{
	struct test_data *data = tester_get_data();
	struct bt_hci *hci;

	data->remove_id = mgmt_register(data->mgmt, MGMT_EV_INDEX_REMOVED,
					data->mgmt_index,
					index_removed_callback,
					NULL, NULL);

	hci = bt_hci_new_user_channel(data->mgmt_index);
	if (hci) {
		bt_hci_unref(hci);
		return;
	}

	mgmt_unregister(data->mgmt, data->remove_id);
	data->remove_id = 0;

	tester_test_failed();
}

static void test_open_failed(const void *test_data)
{
	struct test_data *data = tester_get_data();
	struct bt_hci *hci;

	hci = bt_hci_new_user_channel(data->mgmt_index);
	if (!hci) {
		tester_test_passed();
		return;
	}

	bt_hci_unref(hci);
	tester_test_failed();
}

static void close_read_info_callback(uint8_t status, uint16_t length,
					const void *param, void *user_data)
{
	const struct mgmt_rp_read_info *rp = param;
	uint32_t current_settings;

	tester_print("Read Info callback");
	tester_print("  Status: 0x%02x", status);

	if (status || !param) {
		tester_test_failed();
		return;
	}

	current_settings = btohl(rp->current_settings);
	if (current_settings & MGMT_SETTING_POWERED) {
		tester_print("Controller is powered");
		tester_test_failed();
		return;
	}

	tester_test_passed();
}

static void setup_channel_open(const void *test_data)
{
	struct test_data *data = tester_get_data();

	/* Check power off */
	if (data->current_settings & MGMT_SETTING_POWERED) {
		tester_print("Controller is powered");
		tester_setup_failed();
		return;
	}

	/* Open Channel */
	data->hci = bt_hci_new_user_channel(data->mgmt_index);
	if (!data->hci) {
		mgmt_unregister(data->mgmt, data->remove_id);
		data->remove_id = 0;

		tester_setup_failed();
		return;
	}

	tester_print("User Channel Opened");

	tester_setup_complete();
}

static void test_close_success(const void *test_data)
{
	struct test_data *data = tester_get_data();

	tester_print("Close User Channel");
	bt_hci_unref(data->hci);

	/* Check if power is off */
	mgmt_send(data->mgmt, MGMT_OP_READ_INFO, data->mgmt_index, 0, NULL,
					close_read_info_callback, NULL, NULL);
}

#define test_user(name, data, setup, func) \
	do { \
		struct test_data *user; \
		user = malloc(sizeof(struct test_data)); \
		if (!user) \
			break; \
		user->hciemu_type = HCIEMU_TYPE_BREDR; \
		user->mgmt_index = MGMT_INDEX_NONE; \
		user->test_data = data; \
		user->remove_id = 0; \
		tester_add_full(name, data, \
				test_pre_setup, setup, func, NULL, \
				test_post_teardown, 2, user, test_data_free); \
	} while (0)

int main(int argc, char *argv[])
{
	tester_init(&argc, &argv);

	test_user("User channel open - Success", NULL,
					NULL, test_open_success);
	test_user("User channel open - Failed", NULL,
					setup_powered, test_open_failed);
	test_user("User channel open - Power Toggle Success", INT_TO_PTR(true),
					toggle_powered, test_open_success);
	test_user("User channel close - Success", NULL,
					setup_channel_open, test_close_success);


	return tester_run();
}
