/***
 *** SVGATextMode, an SVGA textmode setting program
 *** Written by Koen Gadeyne (kmg@barco.be)
 ***
 *** Version: 0.2
 ***
 *** Edited on 132x64 screen. Hence the LONG lines. For best viewing conditions, use this program to make itself more readable :-)
 ***/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <math.h>
#include <sys/types.h>
#include <linux/vt.h>
#include <sys/ioctl.h>
#include "vga_prg.h"
#include "configfile.h"
#include "messages.h"

#ifndef TRUE
#define TRUE (1)
#endif
#ifndef FALSE
#define FALSE (-1)
#endif


#define MAX_ATTRIBS     4
#define ATTRIB_LEN      10
#define MAX_CLOCKDEVIATION 3.0


/*
 * global variables
 */

char *CommandName;
int debug_messages=FALSE;


void usage()
{
     PMESSAGE(("usage: %s [options] textmodelabel\n
     Options: -n  Don't program VGA hardware
              -d  print debugging information
              -h  print usage information\n
     Textmodelabel: an existing label from the config file %s\n",CommandName, CONFIGFILE))
}

/****************************************************************************************************************************/
int main (int argc, char* argv[])
{
  char* req_label;
  char* arg_str;
  char tempstr[1024]="";
  char extclockpath[1024]="";
  int extclock=FALSE;
  const char *str_chipsets[NUM_CHIPSETS] = CHIPSET_STRINGS;
  const char *str_options[NUM_OPTIONS] = OPTION_STRINGS;
  const char *str_attributes[NUM_ATTRIBS] = ATTRIB_STRINGS;

  int chipset = FALSE;
#ifdef ALLOW_CLOCKDIV2
  float clocks[MAX_CLOCKS*2];
#else
  float clocks[MAX_CLOCKS];
#endif
  int num_clocks=0;
  int et4_hibit=1;       /* "hibit" status ET4000 */
  int attribval=-1;
  int program_hardware=TRUE;
  char c;
    
  float clockfreq,realclock,horfreq;
  int activepix, start_hsync, stop_hsync, totalpix;
  int activelines, start_vsync, stop_vsync, totallines;
  char attrib[MAX_ATTRIBS][ATTRIB_LEN];
  
  int textlines;
  int blanklines;
  int charsperline;

  struct vt_sizes *p_my_vt_size;      /* this struct passes the new screen size on to the kernel */
  
  int i=0, result=0;

  int font_height=16, font_width=8;      /* height/width of character cell */
  int h_polarity = -1, v_polarity = 1; /* default sync polarities: May not be OK */
  
  FILE *param_file;
    

 /*
  * command-line argument parsing
  */

  CommandName = argv[0];

  while ((c = getopt (argc, argv, "ndh")) != EOF)
    switch (c)
    {
      case 'n': program_hardware=FALSE;
                break;
      case 'd': debug_messages=TRUE;
                break;
      case 'h': usage();
                exit(0);
                break;
      case '?': usage();
                PERROR(("Bad option '%s'\n",argv[optind-1]));
                exit(-1);
                break;
      default: PERROR(("getopt returned unknown token '%c'.",c));
    }
  if (optind < argc)
     req_label = argv[optind];
  else
     {
       usage();
       PERROR(("No textmode label on commandline."));
     }

 /*
  * open parameter file
  */
  param_file = open_param_file(CONFIGFILE);

 /*
  * get chipset definition 
  */
  sscanf(findlabel(param_file, "ChipSet"), "%*s %s", tempstr);
  chipset = findoption(tempstr, str_chipsets, NUM_CHIPSETS, "chip set");
  
 /*
  * For ET4000 only: get "high-bit" option for clock selection.
  */
  if (chipset == CS_ET4000)
  {
     sscanf(findlabel(param_file, "Option"), "%*s %s", tempstr);
     et4_hibit = findoption(tempstr, str_options, NUM_OPTIONS, "ET4000 hibit");
  }
  

 /*
  * get pixel clocks (must be on ONE line!)
  */
  switch (chipset)
  {
    case CS_CIRRUS: break;   /* no clocks for Cirrus */
    default:
        arg_str = strtok(findlabel(param_file, "Clocks")," ");
        num_clocks = 0;
        if (!strcasecmp(arg_str=strtok(NULL," "), "external"))
        {
          if ((arg_str=strtok(NULL," ")) == NULL) PERROR(("External clock selection: pathname of clock program not defined"));
          strcpy(extclockpath, arg_str);
          PDEBUG(("Using external clock program '%s'", extclockpath));
          extclock = TRUE;
        }
        else
        {
          do clocks[num_clocks++] = getfloat(arg_str, "'Clocks' definition: clock frequency in configuration file", MIN_CLOCK, MAX_CLOCK);
            while ((arg_str=strtok(NULL," ")) != NULL);
          if ((num_clocks <4) || (num_clocks >32))
             PWARNING(("Unusual number of clocks (%d) in config file '%s'",num_clocks,CONFIGFILE));
          for (i=0; i<num_clocks; i++) sprintf(&tempstr[i*7], "%7.2f ", clocks[i]);
          PDEBUG(("Clocks: found %d clocks: %s",num_clocks, tempstr));
        }
  }

 /*
  * check for correct amount of clocks, if possible for the specified chip set. This could be incorrect...
  */
  if ((chipset == CS_PVGA1) && (num_clocks != 8) && (num_clocks != 4))
    PERROR(("PVGA1 chipset must have 4 or 8 clocks in 'clocks' line"));
  if (((chipset == CS_WDC90C0X) || (chipset == CS_WDC90C1X) || (chipset == CS_WDC90C2X)) && (num_clocks != 9))
    PERROR(("WDC90C0X, 1X and 2X chipsets have 8 pixel clocks PLUS the MClock."));
  if ((chipset == CS_WDC90C3X) && (num_clocks != 17))
    PERROR(("WDC90C3X chipsets have 16 pixel clocks PLUS the MClock."));
         

 /*
  * find requested text mode line
  */
  arg_str = findlabel(param_file, req_label);
  result = sscanf(arg_str, "%*s %f  %d %d %d %d  %d %d %d %d   %s %s %s %s\n",
              &clockfreq,
              &activepix, &start_hsync, &stop_hsync, &totalpix,
              &activelines, &start_vsync, &stop_vsync, &totallines,
              attrib[0], attrib[1], attrib[2], attrib[3]);
  if (result<10)
     PERROR(("Badly formed textmode config string at label '%s' in config file '%s'",req_label,CONFIGFILE));
     
 /*
  * Do some SEVERE error checking on the text mode string timings!
  * the ranges are somewhat randomly chosen. Need to study REAL hardware limits for this...
  */
  
  check_int_range(activepix, 16, 4096, "active pixels");
  check_int_range(start_hsync, activepix, 4096, "start of H-sync");
  check_int_range(stop_hsync, start_hsync+1, 4096, "end of H-sync");
  check_int_range(totalpix, 16, 4096, "total pixels");

  check_int_range(activelines, 16, 4096, "active lines");
  check_int_range(start_vsync, activelines, 4096, "start of V-sync");
  check_int_range(stop_vsync, start_vsync+1, 4096, "end of V-sync");
  check_int_range(totallines, 16, 4096, "total lines");

 /*
  *  parse any attribute strings
  */
  for (i=0; i<(result-9); i++)   /* supposing "result" is still the result from the sscanf ... */
  {
     attribval = findoption(attrib[i], str_attributes, NUM_ATTRIBS, "attribute");
     switch(attribval)
     {
        case ATTR_FONT: i++;  /* get next attribute, which SHOULD be the font SIZE */
                        if (i >= (result-9))
                           PERROR(("'font' attribute must be followed by a font size (e.g. 8x16)"));
                        if (sscanf(attrib[i], "%dx%d", &font_width, &font_height) < 2)
                          PERROR(("Illegal font size specification '%s'", attrib[i]));
                        check_int_range(font_width, 8, 9, "font width");
                        check_int_range(font_height, 1, 32, "font height");
                        break;
        case ATTR_PHSYNC: h_polarity = 1;
                          break;
        case ATTR_NHSYNC: h_polarity = -1;
                          break;
        case ATTR_PVSYNC: v_polarity = 1;
                          break;
        case ATTR_NVSYNC: v_polarity = -1;
                          break;
        default: PERROR(("Unknown attribute '%s' in config line '%s'.", attrib[i], req_label));
     }
  }
     
  PDEBUG(("Font size will be %dx%d", font_width, font_height));
  PDEBUG(("sync polarities: Hsync: %c, Vsync: %c", (h_polarity<0) ? '-' : '+', (v_polarity<0) ? '-' : '+'));
  
 /*
  * look for a suitable clock for the specified chip set
  */
  if (extclock == FALSE) 
  { 
    GetClock(chipset, clocks, num_clocks, clockfreq, &realclock);
    if( fabs(realclock-clockfreq)> MAX_CLOCKDEVIATION )
      PERROR(("The closest available clock %f differs too much from specified clock %f",realclock,clockfreq));
  }
  else realclock=clockfreq; /* suppose the external clock program can do ANY clock */

 /*
  * calculate some screen parameters
  */
  textlines = activelines / font_height;
  activelines = font_height * textlines; /* make activelines integer multiple of font_height */
  blanklines = totallines - activelines;
  charsperline = (activepix / 8) & 0xFFFFFFFE; /* must be multiple of 2 in VGA byte-mode adressing */  

  if (program_hardware == TRUE)
  {
      get_VGA_io_perm();
      get_VGA_io_perm_CHIPSET(chipset);
      Renounce_SUID;
      
     /*
      * now get to the REAL hardware stuff !
      */
      /*  WAIT_VERT_BLK; SCREEN_OFF; */
      unlock(chipset);

      if (extclock == TRUE)
      {
         sprintf(tempstr,"%s %d", extclockpath, (int) (clockfreq*1000));
         PDEBUG(("Executing external command '%s'", tempstr));
         SYNCRESET_SEQ;
         result=system(tempstr);
         ENDRESET_SEQ;
         if (result !=0) PERROR(("'system' returned error code %d", result));
      }       
      else
        SetClock(chipset, clocks, num_clocks, clockfreq, &realclock, et4_hibit);

      special(chipset); /* change chipset-specific things, if needed */

      set_MAX_SCANLINE (font_height);

      set_VERT_TOTAL (totallines);
      set_VDISPL_END (activelines);
      set_VRETRACE (start_vsync, stop_vsync);
      set_VBLANK (start_vsync, stop_vsync);

      set_HOR_TOTAL(totalpix/8);
      set_HOR_DISPL_END(charsperline);
      set_HSYNC(start_hsync/8, stop_hsync/8);
      set_HBLANK((start_hsync-1)/8, stop_hsync/8);
      set_LOG_SCREEN_WIDTH(charsperline);

      set_CURSOR(font_height-2, font_height-1); /* cursor start/end */

      set_SYNC_POLARITY(h_polarity, v_polarity);

      set_textmode();  /* just in case some jerk set us in graphics mode. Or if he's in X-windows: surprise surprise ! */

      if (set_charwidth(font_width)) PERROR(("Illegal character width: %d",font_width));
      
      /* SCREEN_ON;  */

     /*
      * resize the internal kernel VT parameters, so they comply with the new ones
      * This sometimes gives a kernel error. Still don't know why. Once in that situation, it's ALWAYS that way...
      * But in that case the only remedy is rebooting, or returning to the previous mode.
      */
      p_my_vt_size = (struct vt_sizes *) malloc(sizeof(struct vt_sizes));
      p_my_vt_size->v_rows = textlines;
      p_my_vt_size->v_cols = charsperline;
      p_my_vt_size->v_scrollsize = 0; /* kernel tries to get as many scroll-back lines as possible by itself (?) */
      if (ioctl(0, VT_RESIZE, p_my_vt_size)) {
          perror("VT_RESIZE");
          PERROR((""));
      }
  }
  
 /*
  * show the user what he has just done
  */
  horfreq = (realclock*1000)/( ((int)(totalpix / 8)) * font_width);
  PDEBUG(("Refresh will be %3.2fkHz/%3.1fHz",horfreq,(horfreq*1000)/totallines));
  printf("Chipset = '%s', Textmode clock = %3.2f MHz, %dx%d chars, CharCell = %dx%d. Refresh = %3.2fkHz/%3.1fHz.\n",
          str_chipsets[chipset],realclock,charsperline,textlines,font_width,font_height,horfreq,(horfreq*1000)/totallines);

  return(0);
}

