/***
 *** VGA chip programming functions for SVGATextMode
 *** Written by Koen Gadeyne (kmg@barco.be)
 ***
 *** version : 0.2
 ***/

#include <stdio.h>
#include <stdarg.h>
#include <unistd.h>
#include <asm/io.h>
#include <math.h>
#include "vga_prg.h"
#include "messages.h"  

/*
 * Some functions to make life easier (?!)
 */

void outbit_SEQ (int index, int bitno, int data)
{
  int bitmask = (1 << bitno);
  int onebit = ((data == 0) ? 0 : bitmask);
  Outb_SEQ (index, (Inb_SEQ(index) & ~bitmask) | onebit);
}

void outbit_CRTC (int index, int bitno, int data)
{
  int bitmask = (1 << bitno);
  int onebit = ((data == 0) ? 0 : bitmask);
  Outb_CRTC (index, (Inb_CRTC(index) & ~bitmask) | onebit);
}

void outbit_GR_CTL(int index, int bitno, int data)
{ 
  int bitmask = (1 << bitno);
  int onebit = ((data == 0) ? 0 : bitmask);
  Outb_GR_CTL(index, (Inb_GR_CTL(index) & ~bitmask) | onebit);
}

void outb_ATR_CTL (int index, int data)
{
  inb(ATR_CTL_INDEX_DATA_SWITCH);
  outb (index & 0x1f, ATR_CTL_INDEX);
  outb ( data, ATR_CTL_DATA_W);
  inb(ATR_CTL_INDEX_DATA_SWITCH);
  outb ( (index & 0x1f) | 0x20, ATR_CTL_INDEX);
  outb ( data, ATR_CTL_DATA_W);
}

int inb_ATR_CTL (int index)
{
  int res;

  inb(ATR_CTL_INDEX_DATA_SWITCH);
  outb (index & 0x1f, ATR_CTL_INDEX);
  res=inb(ATR_CTL_DATA_R);
  inb(ATR_CTL_INDEX_DATA_SWITCH);
  outb ( (index & 0x1f) | 0x20, ATR_CTL_INDEX);
  inb(ATR_CTL_DATA_R);
  return res;
}

void outb_VGA_mem(int register_set, int data)
{
   switch(register_set)
  {
    case REGSET_MISC  : outb(data, VGA_MISC_W); break;
    default: PERROR(("outb_VGA_indexed: unknown register set %d",register_set));
  }
}

int inb_VGA_mem(int register_set)
{
   switch(register_set)
  {
    case REGSET_MISC  : return(inb(VGA_MISC_R)); break;
    default: PERROR(("inb_VGA_indexed: unknown register set %d",register_set));
  }
}

void outb_VGA_indexed(int register_set, int reg_index, int data)
{
   switch(register_set)
  {
    case REGSET_CRTC  : Outb_CRTC(reg_index,data); break;
    case REGSET_SEQ   : Outb_SEQ(reg_index,data); break;
    case REGSET_ATRCTL: outb_ATR_CTL(reg_index,data); break;
    case REGSET_GRCTL : Outb_GR_CTL(reg_index,data); break;
    default: PERROR(("outb_VGA_indexed: unknown register set %d",register_set));
  }
}

int inb_VGA_indexed(int register_set, int reg_index)
{
   switch(register_set)
  {
    case REGSET_CRTC  : return(Inb_CRTC(reg_index)); break;
    case REGSET_SEQ   : return(Inb_SEQ(reg_index)); break;
    case REGSET_ATRCTL: return(inb_ATR_CTL(reg_index)); break;
    case REGSET_GRCTL : return(Inb_GR_CTL(reg_index)); break;
    default: PERROR(("inb_VGA_indexed: unknown register set %d",register_set));
  }
}

/*****************************************************************************************************************************/
void get_VGA_io_perm()
{
  PDEBUG(("Getting VGA IO permissions"));
  if (ioperm(0x3b4, 0x3df - 0x3b4 + 1, 1)) PERROR(("Cannot get VGA I/O permissions: must be superuser or setuid root!"));
}

void get_VGA_io_perm_CHIPSET(int chipset)
{
  int err=0;
  PDEBUG(("Getting special VGA IO permissions for chipset #%d", chipset));
  switch(chipset)
  {
    case CS_ATI:
    case CS_ATIMACH32:  err = ioperm(ATI_EXTREG, 2, 1);
                        break;
    default: PDEBUG(("No special VGA IO permissions needed for chipset #%d", chipset));
  }
  if (err) PERROR(("Cannot get special VGA I/O permissions for chipset %d: must be superuser or setuid root!", chipset));
}

/*****************************************************************************************************************************/

void clock_check(int result)
{
  if (result < 0)
  {
    switch(result)
    {
      case CLKSEL_DONOTHING:     PWARNING(("Clock selection: Warning: Clock not changed"));
                                 break;
      case CLKSEL_ILLEGAL_NUM:   PERROR(("Clock selection: illegal clock number."));
                                 break;
      case CLKSEL_NULLCLOCK:     PERROR(("Clock selection: 0 MHz clock selected! It would lock up the video card. Aborting..."));
                                 break;
      case CLKSEL_OUT_OF_BOUNDS: PERROR(("Clock selection: Requested clock frequency out of bounds"));
                                 break;
      default: PERROR(("Clock selection: unknown error"));
    }
  }
}

/*****************************************************************************************************************************/

void TVGAClockSelect(int chipset, int num_clocks, int no)
{
    PDEBUG(("TVGAClockSelect: clock no %d (of total %d) for chipset %d.", no, num_clocks, chipset));
    SET_CLOCKBITS_0_1(no);
    Outb_SEQ(0x0B, 0); Inb_SEQ(0x0B);  /* Select "new mode regs" for CS2 and TVGA9000 CS3. */
#if 0
    Outb_SEQ(0x0D, Inb_SEQ(0x0D) & 0xF9);  /* set clock dividers to no division */
#else
    outbit_SEQ(0x0D, 1, 0);  /* set clock dividers (div 2) to no division */
    outbit_SEQ(0x0D, 2, 0);  /* set clock dividers (div 4 ?) to no division */
#endif    
    outbit_SEQ(0x0D, 0, no & 0x04);  /* bit 2 of clock no */
    if (chipset == CS_TVGA9000)
    {
       outbit_SEQ(0x0D, 6, no & 0x08);  /* bit 3 of clock no */
    }
    if ((chipset == CS_TVGA8900) && (num_clocks > 7))
    {
       Outb_SEQ(0x0B, 0);    /* Switch to Old Mode for CS3 */
       outbit_SEQ(0x0E, 4, no & 0x08);      /* bit 3 of clock no */
    }
}

void s3ClockSelect(int no)
  /* clock number is supposed to be in the allowable range by now. error checking should have been done before */
{
   SET_CLOCKBITS_0_1(3);    /* select clock #3 to allow extra clocks to be used */
   Outb_CRTC(0x42, no);
}

void ATIClockSelect(int chipset, int no)
{
   unsigned char b9, be;
   if (chipset==CS_ATIMACH32) no = no ^ 0x04; /* bit 2 inverted for MACH32 and up */
   b9 = ATI_GETEXTREG(0xB9);
   be = ATI_GETEXTREG(0xBE);
   ATI_PUTEXTREG(0xBE, (be & 0xEF) | ((no << 2) & 0x10));  /* clock bit 2 */
   ATI_PUTEXTREG(0xB9, (b9 & 0xFD) | ((no >> 2) & 0x02));  /* clock bit 3 */
   ATI_PUTEXTREG(0xB8, (no << 2) & 0xC0);
   SET_CLOCKBITS_0_1(no);    /* clock bits no 0 and 1 , must be set LAST */
 }

void WDCClockSelect(int chipset, int num_clocks, int no, int swap_hibit)
/* almost literal copy of the WDC driver clock selection code from XFREE, (C) Thomas Roell */
{
  int MClkIndex = 8;
  if (num_clocks > 8) MClkIndex = num_clocks - 1;

  if ((chipset==CS_WDC90C1X) || (chipset==CS_WDC90C2X) || (chipset==CS_WDC90C3X))
  {
    if (swap_hibit!=0) swap_hibit = 0; 
    else swap_hibit = 1;
  }
         
  if ((no == MClkIndex) && (chipset != CS_PVGA1))
  {
    /*
     * On all of the chipsets after PVGA1, you can feed MCLK as VCLK.
     */
    PDEBUG(("WDCClockSelect: Selecting MCLOCK as VCLOCK"));
    outbit_CRTC(0x2E, 4, 1);
  }
  else
  {
    /*
     * Disable feeding MCLK to VCLK
     */
    if (chipset != CS_PVGA1) outbit_CRTC(0x2E, 4, 0);

    SET_CLOCKBITS_0_1(no);
    
  if ((chipset != CS_PVGA1) && (chipset != CS_WDC90C0X))
    outbit_GR_CTL(0x0C, 2, (swap_hibit == 0) ? (no & 0x04) : (no & 0x04)^0x04); /* bit 2 of clock no */ 
  if (chipset==CS_WDC90C3X)
    outbit_SEQ(0x12, 2, (no & 0x08) ^ 0x08); /* inverse clock bit 3 */
  }
}

void ET4000ClockSelect(int no, int hibit)
{
   /* get access to extended registers */
   outb(3, 0x3bf);
   outb(0xa0, inb(VGA_MISC_R) & 0x01 ? 0x3d8 : 0x3b8);
   /* program clock */
   SET_CLOCKBITS_0_1(no);
   outbit_CRTC(0x34, 1, no & 0x04);                                   /* bit 2 of clock no */
#if 1
   outbit_SEQ(7, 6, (hibit == 0) ? (no & 0x08) : (no & 0x08)^0x08);   /* bit 3 of clock no */
#else   
   /* the data book for the W32i says the following bits should be used for clock selection, but it doesn't work ... 
      anyone know why ? */
   outbit_CRTC(0x31, 6, no & 0x08);    /* bit 3 of clock no */
   outbit_CRTC(0x31, 7, no & 0x10);    /* bit 4 of clock no */ 
#endif
}

int CirrusClockSelect(float freq, float *closestfreq, int setit)
{
  float realclock;
  float fraction;
  int dmin,dmax,nmin,nmax;
  int nom,denom,closestnom,closestdenom;
  float delta;

  if(freq<CIRRUS_MIN_CLOCK || freq>CIRRUS_MAX_CLOCK) 
  {
     return(CLKSEL_OUT_OF_BOUNDS);
  }
         
  fraction=freq/CIRRUS_CLOCK_REF;
  
  dmin=1/fraction+0.5;
  dmax=127/fraction+0.5;
  
  if(dmin<=0) dmin=1;
  if(dmax>=124) dmax=124;
  
  nmin=fraction*dmin+0.5;
  nmax=fraction*dmax+0.5;
  
  if(nmin>=0) nmin=1;
  if(nmax>127) nmax=127;

  delta=1e10;
  closestnom=closestdenom=-1;
  for(nom=nmax;nom>=nmin;nom--)
  {
    if(nom/fraction+0.5 > 62)
    {  /* denom = 4-fold */
      denom=nom/(fraction*4)+0.5; denom*=4;
    }
    else
    {
      denom=nom/(fraction*2)+0.5; denom*=2;
    }
    if(denom<=0) denom=2;
    PDEBUG(("CIRRUS: %f",fabs(nom*CIRRUS_CLOCK_REF/denom-freq)));
    if(fabs(nom*CIRRUS_CLOCK_REF/denom-freq)<delta)
    {
      closestnom=nom;
      closestdenom=denom;
      delta=fabs(nom*CIRRUS_CLOCK_REF/denom-freq);
    }
  }
  
  realclock=closestnom*CIRRUS_CLOCK_REF/closestdenom;
  PDEBUG(("CIRRUS: nom = %d ; denom = %d ; realclock = %f",closestnom,closestdenom,realclock));
  if(setit)
  {
    int sr,sr1;
    
    sr=closestnom;
    sr1=closestdenom>62 ? closestdenom/2 +1 : closestdenom & 0xfe;

/* clock in kHz is (numer * (CIRRUS_REF_KHZ / (denom & 0x3E)) >> (denom & 1) */
#define CLOCKVAL(n, d) \
   ((((n) & 0x7F) * CIRRUS_REF_KHZ / ((d) & 0x3E)) >> ((d) & 1))

  
  PDEBUG(("%d %d   %d",sr,sr1,CLOCKVAL(sr,sr1)));
    /* Use VCLK3 for these extended clocks */
    outb (inb (0x3cc) | 0xc, 0x3c2);
    /* Set SRE and SR1E */
    Outb_SEQ (0x0e, (Inb_SEQ (0xe) & 0x80) | (sr & 0x7f));
    Outb_SEQ (0x1e, (Inb_SEQ (0x1e) & 0xc0) | (sr1 & 0x3f));
  }
  *closestfreq = realclock;
  return(0);  
}

int findclosestclock(float *gclocks, int num_gclocks, float req_clock, float *closest_clock)
{
   /* returns closest clock NUMBER when one is found, error code otherwise */
   /* should also take the standard VGA possibility to divide the dot clock by 2 !!! ==> more clocks */
   /* and the maybe an option to disable that for VGA chips that do NOT allow this ! */

   int i, closest=0;

#ifdef ALLOW_CLOCKDIV2
   for (i=0; i<(num_gclocks-1); i++) gclocks[i+num_gclocks] = gclocks[i]/2;
#endif

   /* find closest clock frequency */
#ifdef ALLOW_CLOCKDIV2
   for (i=0 ; i < num_gclocks*2 ; i ++)
#else
   for (i=0 ; i < num_gclocks ; i ++)
#endif
   {
      if ( fabs(gclocks[i] - req_clock) < fabs(gclocks[closest] - req_clock) ) { closest = i; }
   }
   *closest_clock = gclocks[closest];
   PDEBUG(("findclosestclock: closest clock nr %d = %f MHz.",closest, *closest_clock));
   if (closest < 0)
   {
      return(CLKSEL_DONOTHING);
   }
#ifdef ALLOW_CLOCKDIV2
   if (closest > num_gclocks*2)
#else
   if (closest > num_gclocks)
#endif
   {
      return(CLKSEL_ILLEGAL_NUM);
   }
   if (gclocks[closest] == 0)
   {
      return(CLKSEL_NULLCLOCK);
   }
   return(closest);
}


int GetClock(int chipset, float* clocks, int num_clocks, float freq, float *closestfreq)
{
   int result;
   
   switch(chipset)
   {
     case CS_CIRRUS: result = CirrusClockSelect(freq, closestfreq, 0);
                     break;
     default: result = findclosestclock(clocks,num_clocks,freq,closestfreq);
   }
   clock_check(result);
   return(result);
}


void SetClock(int chipset, float* clocks, int num_clocks, float freq, float *closestfreq, int swap_hibit)
{
   int result, divby2=0;
   
   result = GetClock(chipset, clocks, num_clocks, freq, closestfreq);
  /* clock number is supposed to be in the allowable range by now.
     error checking should have been done before.
     No error checking will be done in clock-setting routine! */
#ifdef ALLOW_CLOCKDIV2
   if (result > num_clocks-1) 
   {
      divby2 = 1;
      result -= num_clocks; /* make clock selection routine pick the real clock (before division by 2) */
   }
#endif
   PDEBUG(("Setting Clock to %f MHz", *closestfreq));
   SYNCRESET_SEQ;
   switch(chipset)
   {
    case CS_CIRRUS : CirrusClockSelect(freq, closestfreq, 1);
                     break;
    case CS_S3     : s3ClockSelect(result);
                     break;
    case CS_ET4000 : ET4000ClockSelect(result,swap_hibit);
                     break;
    case CS_TVGA8900 :
    case CS_TVGA9000 : TVGAClockSelect(chipset, num_clocks, result);
                       break;
    case CS_PVGA1:
    case CS_WDC90C0X:
    case CS_WDC90C1X:
    case CS_WDC90C2X:
    case CS_WDC90C3X: WDCClockSelect(chipset, num_clocks, result, swap_hibit);
                      break;
    case CS_ATI:
    case CS_ATIMACH32: ATIClockSelect(chipset, result);
                       break;
    default: PERROR(("SetClock: unknown chip set #%d", chipset));
   }
#ifdef ALLOW_CLOCKDIV2
   if (chipset != CS_CIRRUS)
   /* Cirrus goes bananas when this register is programmed */
   {
      outbit_SEQ(1,3,divby2);
      if (divby2) PDEBUG(("Clock (%f) needed 'division by 2' feature.",*closestfreq));
   }
#endif
   ENDRESET_SEQ;
   usleep(150000); /* let PLL clock synthesizer stabilize */
   clock_check(result);
}

/*****************************************************************************************************************************/

void unlock(int chipset)
{
   /* unlock ALL locked registers for specified chipset. A bit rough, but simplest */
   PDEBUG(("Unlocking chipset %d",chipset));
   outbit_CRTC (0x11, 7, 0); /* CRTC index 0x00..0x07 */
   switch(chipset)
   {
    case CS_CIRRUS :
       Outb_SEQ (0x6, 0x12);	/* unlock cirrus special */
       break;
    case CS_S3     : 
       Outb_CRTC(0x39, 0xa5); /* system extension regs (CRTC index 0x50..0x5E) */
       Outb_CRTC(0x38, 0x48); /* S3 register set (CRTC index 0x30..0x3C) */
       outbit_CRTC(0x35, 4, 0); /* VERT timing regs (CRTC index 6,7(bit0,2,3,5,7),9,10,11(bits0..3),15,16 ) */
       outbit_CRTC(0x35, 5, 0); /* HOR timing regs (CRTC index 0..5, 17(bit2) ) */
      /*  outbit_CRTC(0x40, 0, 1); */ /* enhanced register access, only for access to accelerator commands. Does NOT seem to work on my 805 */
       break;
    case CS_ET4000 : outb(0x03, 0x3BF); outb(0xA0, 0x3D8); /* ET4000W32i key */
                     break;
    case CS_TVGA9000 :
    case CS_TVGA8900 : 
#if 0  /* this does not work properly on 8900CL */   
                       Outb_SEQ(0x0B, 0);      /* Select "old mode" by writing to SEQ index 0x0B */
                       Outb_SEQ(0x0E, 0x82);   /* unlock conf. reg */
#endif
                       break;
    case CS_PVGA1:
    case CS_WDC90C0X:
    case CS_WDC90C1X:
    case CS_WDC90C2X:
    case CS_WDC90C3X: break;
    case CS_ATI:
    case CS_ATIMACH32: ATI_PUTEXTREG(0xB4, ATI_GETEXTREG(0xB4) & 0x03);
                       ATI_PUTEXTREG(0xB8, ATI_GETEXTREG(0xb8) & 0xC0);
                       ATI_PUTEXTREG(0xB9, ATI_GETEXTREG(0xb9) & 0x7F);
                       ATI_PUTEXTREG(0xBE, ATI_GETEXTREG(0xbe) | 0x01);
                       break;
    default: PERROR(("UNLOCK VGA: unknown chipset #%d",chipset));
   }
}
/*****************************************************************************************************************************/

void special(int chipset)
/* chipset specific settings, like memory speed and the likes */
{
   PDEBUG(("Setting chipset-specific special registers"));
   switch(chipset)
   {
    case CS_CIRRUS : Outb_SEQ(31,37);  /* set DRAM timing (34=62 Mhz) */
                     break;
    default: PDEBUG(("SPECIAL VGA chip settings: no special settings for chipset #%d",chipset));
   }
}
/*****************************************************************************************************************************/

/*
 * Some more general functions (less low-level than separate register access)
 */
 
void set_VERT_TOTAL (int vt)
{
  vt = vt - 2; /* must program actual value - 2 */
  Outb_CRTC (0x6, vt & 0xff);       /* bits 0..7 */
  outbit_CRTC(0x7, 0, vt & 0x100);  /* bit 8 */ 
  outbit_CRTC(0x7, 5, vt & 0x200);  /* bit 9 */
}

void set_MAX_SCANLINE (int msl)
{
  msl = msl - 1;
  Outb_CRTC (0x9, (Inb_CRTC (0x9) & 0xe0) | (msl & 0x1f));
}

void set_VRETRACE (int start, int end)
{
  Outb_CRTC (0x10, start & 0xff);                           /* start bits 0..7 */
  outbit_CRTC (0x7, 2, start & 0x100);                      /* start bit 8 */
  outbit_CRTC (0x7, 7, start & 0x200);                      /* start bit 9 */
  Outb_CRTC (0x11, (Inb_CRTC (0x11) & 0xf0) | (end & 0xf)); /* end */
}

void set_VDISPL_END (int vde)
{
  vde = vde - 1;
  Outb_CRTC (0x12, vde & 0xff);      /* bits 0..7 */
  outbit_CRTC(0x7, 1, vde & 0x100);  /* bit 8 */
  outbit_CRTC(0x7, 6, vde & 0x200);  /* bit 9 */
}

void set_VBLANK (int start, int end)
{
  start = start - 1;
  Outb_CRTC (0x15, start & 0xff);                       /* start bits 0..7 */
  outbit_CRTC(0x7, 3, start & 0x100);                   /* start bit 8 */
  outbit_CRTC(0x9, 5, start & 0x200);                   /* start bit 9 */
  Outb_CRTC (0x16, (start & 0xFF) - 1 + (end - start)); /* end */
}

void set_CURSOR (int start, int end)
{
  Outb_CRTC (0x0A,  (Inb_CRTC(0x0a) & 0xe0) | (start & 0x1f) );
  Outb_CRTC (0x0B,  (Inb_CRTC(0x0b) & 0xe0) | (end   & 0x1f) );
}

void set_HOR_TOTAL (int htot)
{
  Outb_CRTC(0, htot - 5);
}

void set_HOR_DISPL_END (int hend)
{
  PDEBUG(("set_HOR_DISPL_END: active chars = %d", hend));
  Outb_CRTC(1, hend - 1);
}

void set_HSYNC (int start, int end)
{
  Outb_CRTC(4, start);
  Outb_CRTC(5, (Inb_CRTC(5) & 0xe0) | (end & 0x1f));
}

void set_HBLANK (int start, int end)
{
  Outb_CRTC(2, start);                                 /* start */
  Outb_CRTC(3, (Inb_CRTC(3) & 0xe0) | (end & 0x1f));   /* end bits 0..4 */
  outbit_CRTC(5, 7, end & 0x20);                       /* end bit 5 */  
}

void set_SYNC_POLARITY(int hpol, int vpol)
{
   outb((inb(VGA_MISC_R) & 0xBF) | ((hpol < 0) ? 0x40 : 0x00) , VGA_MISC_W);
   outb((inb(VGA_MISC_R) & 0x7F) | ((vpol < 0) ? 0x80 : 0x00) , VGA_MISC_W);
}

void set_LOG_SCREEN_WIDTH(int width)
{
   Outb_CRTC(0x13, width/2);
}

void set_textmode()
{
   Outb_GR_CTL(6,Inb_GR_CTL(6) & 0xFE);
   outb_ATR_CTL(16,inb_ATR_CTL(16) & 0xFE);
}

void set_graphmode()
{
   Outb_GR_CTL(6,Inb_GR_CTL(6) | 0x01);
   outb_ATR_CTL(16,inb_ATR_CTL(16) | 0x01);
}

int set_charwidth(int width)
{
   switch(width)
   {
      case 8: outbit_SEQ(1, 0, 1);
              SET_PIXELPAN(0);
              break;
      case 9: outbit_SEQ(1, 0, 0);
              SET_PIXELPAN(8);
              break;
      default: return(1);
   }
   PDEBUG(("Set_CharWidth: SEQ (1) = 0x%x",Inb_SEQ(1)));
   return(0);
}

