/***
 *** SVGA clock programming functions for SVGATextMode
 *** (c) 1995 Koen Gadeyne (kmg@barco.be)
 ***
 *** This was separated from the rest of the vga programming functions (in vga_prg.c),
 *** because it is not needed by ALL modules, and it only burdens them. 
 ***
 ***/

#include <stdio.h>
#include <unistd.h>
#ifndef DOS
#include <asm/io.h>
#else
#include <dos.h>
#include <conio.h>
#define outb(data,port) outp(port,data)
#define inb(port) inp(port)
#define ioperm(x,y,z) (0)
#define usleep(x) delay(x/1000) /* Note lack of resolution */
#endif
#include <math.h>
#include "vga_prg.h"
#include "setclock.h"
#include "messages.h"  
#include "XFREE/xfree_compat.h"
#include "XFREE/common_hw/xf86_HWlib.h"


/*****************************************************************************************************************************/

void clock_check(int result)
{
  if (result < 0)
  {
    switch(result)
    {
      case CLKSEL_DONOTHING:     PWARNING(("Clock selection: Warning: Clock not changed"));
                                 break;
      case CLKSEL_ILLEGAL_NUM:   PERROR(("Clock selection: illegal clock number."));
                                 break;
      case CLKSEL_NULLCLOCK:     PERROR(("Clock selection: 0 MHz clock selected! It would lock up the video card. Aborting..."));
                                 break;
      case CLKSEL_OUT_OF_BOUNDS: PERROR(("Clock selection: Requested clock frequency out of bounds"));
                                 break;
      default: PERROR(("Clock selection: unknown error"));
    }
  }
}

/*****************************************************************************************************************************/

void TVGAClockSelect(int chipset, int num_clocks, int no)
{
    PDEBUG(("TVGAClockSelect: clock no %d (of total %d) for chipset %d.", no, num_clocks, chipset));
    SET_CLOCKBITS_0_1(no);
    Outb_SEQ(0x0B, 0); Inb_SEQ(0x0B);  /* Select "new mode regs" for CS2 and TVGA9000 CS3. */
#if 0
    Outb_SEQ(0x0D, Inb_SEQ(0x0D) & 0xF9);  /* set clock dividers to no division */
#else
    Outbit_SEQ(0x0D, 1, 0);  /* set clock dividers (div 2) to no division */
    Outbit_SEQ(0x0D, 2, 0);  /* set clock dividers (div 4 ?) to no division */
#endif    
    Outbit_SEQ(0x0D, 0, no & 0x04);  /* bit 2 of clock no */
    if (chipset == CS_TVGA9000)
    {
       Outbit_SEQ(0x0D, 6, no & 0x08);  /* bit 3 of clock no */
    }
    if ((chipset == CS_TVGA8900) && (num_clocks > 7))
    {
       Outb_SEQ(0x0B, 0);    /* Switch to Old Mode for CS3 */
       Outbit_SEQ(0x0E, 4, no & 0x08);      /* bit 3 of clock no */
    }
}



void s3ClockSelect(int no)
  /* clock number is supposed to be in the allowable range by now. error checking should have been done before */
{
   SET_CLOCKBITS_0_1(3);    /* select clock #3 to allow extra clocks to be used */
   Outb_CRTC(0x42, no & 0x0f);
}



void ATIClockSelect(int chipset, int no)
{
   if (chipset==CS_ATIMACH32) no = no ^ 0x04; /* bit 2 inverted for MACH32 and up */
   ATI_PUTEXTREG(0xBE, (ATI_GETEXTREG(0xBE) & 0xEF) | ((no << 2) & 0x10));  /* clock bit 2 */
   ATI_PUTEXTREG(0xB9, (ATI_GETEXTREG(0xB9) & 0xFD) | ((no >> 2) & 0x02));  /* clock bit 3 */
   ATI_PUTEXTREG(0xB8, (ATI_GETEXTREG(0xB8) & 0x3F) | ((no << 2) & 0xC0));  /* clock divider */
   SET_CLOCKBITS_0_1(no);    /* clock bits no 0 and 1 , must be set LAST */
 }



void WDCClockSelect(int chipset, int num_clocks, int no, int optmask)
/* almost literal copy of the WDC driver clock selection code from XFREE, (C) Thomas Roell */
{
  int MClkIndex = 8;

  if (num_clocks > 8) MClkIndex = num_clocks - 1;

  if ((no == MClkIndex) && (chipset != CS_PVGA1))
  {
    /*
     * On all of the chipsets after PVGA1, you can feed MCLK as VCLK.
     */
    PDEBUG(("WDCClockSelect: Selecting MCLOCK as VCLOCK"));
    Outbit_CRTC(0x2E, 4, 1);
  }
  else
  {
    /*
     * Disable feeding MCLK to VCLK
     */
    if (chipset != CS_PVGA1) Outbit_CRTC(0x2E, 4, 0);

    SET_CLOCKBITS_0_1(no);
    
  if ((chipset != CS_PVGA1) && (chipset != CS_WDC90C0X))
    Outbit_GR_CTL(0x0C, 2, (optmask & OPT_SWAP_HIBIT) ? (no & 0x04)^0x04 : (no & 0x04)); /* bit 2 of clock no */ 
  if (chipset==CS_WDC90C3X)
    Outbit_SEQ(0x12, 2, (no & 0x08) ^ 0x08); /* inverse clock bit 3 */
  }
}



void ET4000ClockSelect(int no, int optmask)
{
   int hibit = 1;
   if (optmask & OPT_HIBIT_HIGH) hibit = 1;
   if (optmask & OPT_HIBIT_LOW) hibit = 0;
   PDEBUG(("ET4000 Clock selection: hibit option = hibit_%s", (hibit==0) ? "low" : "high"));
   /* program clock */
   if (optmask & OPT_LEGEND)
   {
     PDEBUG(("Using Sigma Legend clock selection"));
     outb((inb(VGA_MISC_R) & 0xF3) | ((no & 0x10) >> 1) | (no & 0x04), VGA_MISC_W);
     Outb_CRTC(0x34, 0x00);
     Outb_CRTC(0x34, 0x02);
     Outb_CRTC(0x34, (no & 0x08) >> 2);
     SET_CLOCKBITS_0_1(no);
   }
   else
   {
     SET_CLOCKBITS_0_1(no);
     Outbit_CRTC(0x34, 1, no & 0x04);                                   /* bit 2 of clock no */
     if (!(optmask & OPT_ET4000_ALTCLK))
     {
       Outbit_SEQ(7, 6, (hibit == 0) ? (no & 0x08) : (no & 0x08)^0x08);   /* bit 3 of clock no */
     }
     else
     {
       PDEBUG(("ET4000 clock selection: Using alternate clock selection mechanism"));
       Outbit_SEQ(7, 6, 0); /* just to make sure */
       Outbit_CRTC(0x31, 6, (hibit == 0) ? (no & 0x08) : (no & 0x08)^0x08);   /* bit 3 of clock no */
       Outbit_CRTC(0x31, 7, no & 0x10);    /* bit 4 of clock no */
     }
   }
}



int CirrusClockSelect(float freq, float *closestfreq, int setit, int optmask)
{
  float realclock;
  float fraction;
  int dmin,dmax,nmin,nmax;
  int nom,denom,closestnom,closestdenom;
  float delta;

  if(freq<CIRRUS_MIN_CLOCK || freq>CIRRUS_MAX_CLOCK) 
  {
     return(CLKSEL_OUT_OF_BOUNDS);
  }
         
  fraction=freq/CIRRUS_CLOCK_REF;
  
  dmin=1/fraction+0.5;
  dmax=127/fraction+0.5;
  
  if(dmin<=0) dmin=1;
  if(dmax>=124) dmax=124;
  
  nmin=fraction*dmin+0.5;
  nmax=fraction*dmax+0.5;
  
  if(nmin>=0) nmin=1;
  if(nmax>127) nmax=127;

  delta=1e10;
  closestnom=closestdenom=-1;
  for(nom=nmax;nom>=nmin;nom--)
  {
    if(nom/fraction+0.5 > 62)
    {  /* denom = 4-fold */
      denom=nom/(fraction*4)+0.5; denom*=4;
    }
    else
    {
      denom=nom/(fraction*2)+0.5; denom*=2;
    }
    if(denom<=0) denom=2;
/*    PDEBUG(("CIRRUS: %f",fabs(nom*CIRRUS_CLOCK_REF/denom-freq))); */
    if(fabs(nom*CIRRUS_CLOCK_REF/denom-freq)<delta)
    {
      closestnom=nom;
      closestdenom=denom;
      delta=fabs(nom*CIRRUS_CLOCK_REF/denom-freq);
    }
  }
  
  realclock=closestnom*CIRRUS_CLOCK_REF/closestdenom;
  PDEBUG(("CIRRUS: nom = %d ; denom = %d ; realclock = %f",closestnom,closestdenom,realclock));
  if(setit)
  {
    int sr,sr1;
    
    sr=closestnom;
    sr1=closestdenom>62 ? closestdenom/2 +1 : closestdenom & 0xfe;

    /* clock in kHz is (numer * (CIRRUS_REF_KHZ / (denom & 0x3E)) >> (denom & 1) */
    #define CLOCKVAL(n, d) ((((n) & 0x7F) * CIRRUS_REF_KHZ / ((d) & 0x3E)) >> ((d) & 1))
  
    PDEBUG(("Programmed clock parameters: %d, %d (%d kHz)",sr,sr1,CLOCKVAL(sr,sr1)));
    /* Use VCLK3 for these extended clocks */
    SET_CLOCKBITS_0_1(3);
    /* Set SRE and SR1E */
    Outb_SEQ (0x0e, (Inb_SEQ (0x0e) & 0x80) | (sr & 0x7f));
    Outb_SEQ (0x1e, (Inb_SEQ (0x1e) & 0xc0) | (sr1 & 0x3f));
    
    /* set DRAM speed (MCLK) */
    if (optmask & OPT_XFAST_DRAM) Outb_SEQ(0x1f,0x25);  /* 66 MHz */
    if (optmask & OPT_FAST_DRAM)  Outb_SEQ(0x1f,0x22);   /* 62 MHz */
    if (optmask & OPT_MED_DRAM)   Outb_SEQ(0x1f,0x1f);   /* 56 MHz */
    if (optmask & OPT_SLOW_DRAM)  Outb_SEQ(0x1f,0x1c);   /* 50 MHz */
  }
  *closestfreq = realclock;
  return(0);  
}



void Video7ClockSelect(int no)
/* from XFREE3.0 server */
{
  if (no < 2) SET_CLOCKBITS_0_1(no);
  else SET_CLOCKBITS_0_1(2);
  outw(0x3C4, 0xA4 | (no << 10));
}



int findclosestclock(float *gclocks, int num_gclocks, float req_clock, float *closest_clock)
{
   /* returns closest clock NUMBER when one is found, error code otherwise */
   /* should also take the standard VGA possibility to divide the dot clock by 2 !!! ==> more clocks */
   /* and maybe an option to disable that for VGA chips that do NOT allow this ! */

   int i, closest=0;

#ifdef ALLOW_CLOCKDIV2
   for (i=0; i<(num_gclocks-1); i++) gclocks[i+num_gclocks] = gclocks[i]/2;
#endif

   /* find closest clock frequency */
#ifdef ALLOW_CLOCKDIV2
   for (i=0 ; i < num_gclocks*2 ; i ++)
#else
   for (i=0 ; i < num_gclocks ; i ++)
#endif
   {
      if ( fabs(gclocks[i] - req_clock) < fabs(gclocks[closest] - req_clock) ) { closest = i; }
   }
   *closest_clock = gclocks[closest];
   PDEBUG(("findclosestclock: closest clock nr %d = %f MHz.",closest, *closest_clock));
   if (closest < 0)
   {
      return(CLKSEL_DONOTHING);
   }
#ifdef ALLOW_CLOCKDIV2
   if (closest > num_gclocks*2)
#else
   if (closest > num_gclocks)
#endif
   {
      return(CLKSEL_ILLEGAL_NUM);
   }
   if (gclocks[closest] == 0)
   {
      return(CLKSEL_NULLCLOCK);
   }
   return(closest);
}



int GetClock(int chipset, float* clocks, int num_clocks, float freq, float *closestfreq, int optmask)
{
   int result;
   
   switch(chipset)
   {
     case CS_CIRRUS: result = CirrusClockSelect(freq, closestfreq, 0, optmask);
                     break;
     default: result = findclosestclock(clocks,num_clocks,freq,closestfreq);
   }
   clock_check(result);
   return(result);
}



void SetClock(int chipset, float* clocks, int num_clocks, float freq, float *closestfreq, int optmask)
{
   int result, divby2=0;
   
   result = GetClock(chipset, clocks, num_clocks, freq, closestfreq, optmask);
  /* clock number is supposed to be in the allowable range by now.
     error checking should have been done before.
     No error checking will be done in clock-setting routine! */
#ifdef ALLOW_CLOCKDIV2
   if (result > num_clocks-1) 
   {
      divby2 = 1;
      result -= num_clocks; /* make clock selection routine pick the real clock (before division by 2) */
   }
#endif
   PDEBUG(("Setting Clock to %f MHz", *closestfreq));
   SYNCRESET_SEQ;
   switch(chipset)
   {
    case CS_VGA : SET_CLOCKBITS_0_1(result);
                  break;
    case CS_CIRRUS : CirrusClockSelect(freq, closestfreq, 1, optmask);
                     break;
    case CS_S3     : s3ClockSelect(result);
                     break;
    case CS_ET4000 : ET4000ClockSelect(result,optmask);
                     break;
    case CS_TVGA8900 :
    case CS_TVGA9000 : TVGAClockSelect(chipset, num_clocks, result);
                       break;
    case CS_PVGA1:
    case CS_WDC90C0X:
    case CS_WDC90C1X:
    case CS_WDC90C2X:
    case CS_WDC90C3X: WDCClockSelect(chipset, num_clocks, result, optmask);
                      break;
    case CS_ATI:
    case CS_ATIMACH32: ATIClockSelect(chipset, result);
                       break;
    case CS_VIDEO7: Video7ClockSelect(result);
                    break;
    default: PERROR(("SetClock: unknown chip set #%d", chipset));
   }
#ifdef ALLOW_CLOCKDIV2
   if (chipset != CS_CIRRUS)
   /* Cirrus goes bananas when this register is programmed */
   {
      Outbit_SEQ(1,3,divby2);
      if (divby2) PDEBUG(("Clock (%f) needed 'division by 2' feature.",*closestfreq));
   }
#endif
   ENDRESET_SEQ;
   usleep(50000); /* let PLL clock synthesizer stabilize */
   clock_check(result);
}

/*****************************************************************************************************************************/

void set_s3_clockchip_clock(int clockchip, long freq, int optmask)   /* "freq" is in Hz ! */
{
  int result=TRUE;
  static int vgaIOBase;
  
  PDEBUG(("Setting clock through clock chip #%d. Freq = %ld Hz. Optionmask = %d", clockchip, freq, optmask));
  vgaIOBase = GET_VGA_BASE;
  switch(clockchip)
  {
    case CLKCHIP_ICD2061A: /* setting exactly 120 MHz doesn't work all the time */
                           if (freq > 119900000) PERROR(("ICD2061 clock frequency (%1.2f MHz) too high. Max = 119.9 MHz.", freq/1000000));
                           AltICD2061SetClock(freq, 2);
                           AltICD2061SetClock(freq, 2);
                           AltICD2061SetClock(freq, 2);
                           s3ClockSelect(2); /* select the clock */
                           break;
    case CLKCHIP_ICS2595:  result = ICS2595SetClock((long)freq/1000);
                           result = ICS2595SetClock((long)freq/1000);
                           result = ICS2595SetClock((long)freq/1000);
                           break;
    case CLKCHIP_S3_SDAC:
    case CLKCHIP_S3GENDAC: (void) S3gendacSetClock((long)freq/1000, 2); /* can't fail */
                           s3ClockSelect(2); /* select the clock */
                           break;
    case CLKCHIP_SC11412:  result = SC11412SetClock((long)freq/1000);
                           s3ClockSelect(2); /* select the clock */
                           break;
    case CLKCHIP_TI3025:   if (freq < 20000000) PERROR(("Ti3025SetClock: Min. clock is 20 MHz."));
                           (void) Ti3025SetClock(freq, 2); /* can't fail */
                           s3ClockSelect(2); /* select the clock */
                           break;
    default: PERROR(("Unknown clock chip #%d.", clockchip));
  }
  if (result == FALSE) PWARNING(("ClockChip: error while programming clock chip"));
  usleep(50000);
}

 
