/* git.c -- the main git file. Performs most of the initializations + commands
   expansion / execution. */

/* Copyright (C) 1993, 1994, 1995 Free Software Foundation, Inc.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

/* Written by Tudor Hulubei and Andrei Pitis.  */


#include <stdio.h>

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#else /* !HAVE_STDLIB_H */
#include "ansi_stdlib.h"
#endif /* !HAVE_STDLIB_H */

#include <sys/types.h>
#include "file.h"
#include <fcntl.h>
#include <ctype.h>
#include <signal.h>
#include <pwd.h>
#include <grp.h>
#include <limits.h>
#include <sys/wait.h>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif /* HAVE_UNISTD_H */

#include "stdc.h"
#include "xstring.h"
#include "xmalloc.h"
#include "xio.h"
#include "xtimer.h"
#include "tty.h"
#include "window.h"
#include "inputline.h"
#include "status.h"
#include "panel.h"
#include "config.h"
#include "signals.h"
#include "history.h"
#include "tilde.h"
#include "misc.h"


extern int FrameDisplay;
extern int suspend_allowed;
extern int signals_status;
extern int UserHeartAttack;


#define MAX_STATIC_SIZE 50


#ifdef HAVE_LINUX
extern int LinuxConsole;
#endif /* HAVE_LINUX */


#ifdef HAVE_LINUX
int AnsiColorSequences = ON;
#else	/* !HAVE_LINUX */
int AnsiColorSequences = OFF;
#endif	/* !HAVE_LINUX */


/* These are the only possible current_mode values. Used while resuming
   from suspended mode in order to correctly refresh the display. */
#define GIT_SCREEN_MODE		0
#define GIT_TERMINAL_MODE	1


int  pid;
char *home;
char *program;
int gitclock_xtimer;
int  SCREEN_X, SCREEN_Y;
int current_mode = GIT_SCREEN_MODE;
char *stdout_log_name = NULL;
char *stderr_log_name = NULL;
char cSection[]  = "[GIT-Color]";
char bwSection[] = "[GIT-Monochrome]";
#ifdef HAVE_GCC
char title[] = " "PRODUCT" "VERSION;
#else
char title[] = " GNU Interactive Tools 4.3.5";
#endif /* !HAVE_GCC */
char login[] = "User:";
char tty[] = "tty:";
#ifdef HAVE_GCC
char exit_msg[] = "*** (EXIT) Exit "PRODUCT" ? \
(ENTER to confirm, ^G to cancel) ***";
#else
char exit_msg[] = "*** (EXIT) Exit GNU Interactive Tools ? \
(ENTER to confirm, ^G to cancel) ***";
#endif /* !HAVE_GCC */
char PS1[4] = " $ ";
char *screen = NULL;
panel *left_panel, *right_panel, *current_panel, *other_panel, *temp_panel;

static char *TempDirectory       = "";
static char *NormalModeHelp      = "";
static char *CommandLineModeHelp = "";


#define BUILTIN_OPERATIONS			 64

#define BUILTIN_copy_files			-1
#define BUILTIN_rename_move_files		-2
#define BUILTIN_make_directory			-3
#define BUILTIN_delete_files			-4
#define BUILTIN_exit				-5
#define BUILTIN_previous_history_element	-6
#define BUILTIN_show_terminal			-7
#define BUILTIN_refresh				-8
#define BUILTIN_goto_root_directory		-9
#define BUILTIN_switch_panels			-10
#define BUILTIN_next_history_element		-11
#define BUILTIN_panel_display_next_mode		-12
#define BUILTIN_panel_display_owner_group	-13
#define BUILTIN_panel_display_date_time		-14
#define BUILTIN_panel_display_size		-15
#define BUILTIN_panel_display_mode		-16
#define BUILTIN_panel_display_full_name		-17
#define BUILTIN_panel_sort_next_method		-18
#define BUILTIN_panel_sort_by_name		-19
#define BUILTIN_panel_sort_by_extension		-20
#define BUILTIN_panel_sort_by_size		-21
#define BUILTIN_panel_sort_by_date		-22
#define BUILTIN_panel_sort_by_mode		-23
#define BUILTIN_panel_sort_by_owner_id		-24
#define BUILTIN_panel_sort_by_group_id		-25
#define BUILTIN_panel_sort_by_owner_name	-26
#define BUILTIN_panel_sort_by_group_name	-27
#define BUILTIN_select_file			-28
#define BUILTIN_file_to_input_line		-29
#define BUILTIN_beginning_of_panel		-30
#define BUILTIN_end_of_panel			-31
#define BUILTIN_scroll_down			-32
#define BUILTIN_scroll_up			-33
#define BUILTIN_previous_line			-34
#define BUILTIN_next_line			-35
#define BUILTIN_other_panel			-36
#define BUILTIN_change_directory		-37
#define BUILTIN_hard_refresh			-38
#define BUILTIN_select_files_matching_pattern	-39
#define BUILTIN_unselect_files_matching_pattern	-40
#define BUILTIN_conform_current_directory	-41
#define BUILTIN_conform_other_directory		-42
#define BUILTIN_other_path_to_input_line	-43
#define BUILTIN_selected_files_to_input_line	-44
#define BUILTIN_backward_char			-45
#define BUILTIN_forward_char			-46
#define BUILTIN_backward_word			-47
#define BUILTIN_forward_word			-48
#define BUILTIN_beginning_of_line		-49
#define BUILTIN_end_of_line			-50
#define BUILTIN_delete_char			-51
#define BUILTIN_backward_delete_char		-52
#define BUILTIN_backward_kill_word		-53
#define BUILTIN_kill_line			-54
#define BUILTIN_kill_to_beginning_of_line	-55
#define BUILTIN_kill_to_end_of_line		-56
#define BUILTIN_just_one_space			-57
#define BUILTIN_delete_horizontal_space		-58
#define BUILTIN_action				-59
#define BUILTIN_set_mark			-60
#define BUILTIN_kill_region			-61
#define BUILTIN_kill_ring_save			-62
#define BUILTIN_yank				-63
#define BUILTIN_exchange_point_and_mark		-64


#define MAX_BUILTIN_NAME		  	 35


char built_in[BUILTIN_OPERATIONS][MAX_BUILTIN_NAME] =
{
    "copy-files",
    "rename-move-files",
    "make-directory",
    "delete-files",
    "exit",
    "previous-history-element",
    "show-terminal",
    "refresh",
    "goto-root-directory",
    "switch-panels",
    "next-history-element",
    "panel-display-next-mode",
    "panel-display-owner-group",
    "panel-display-date-time",
    "panel-display-size",
    "panel-display-mode",
    "panel-display-full-name",
    "panel-sort-next-method",
    "panel-sort-by-name",
    "panel-sort-by-extension",
    "panel-sort-by-size",
    "panel-sort-by-date",
    "panel-sort-by-mode",
    "panel-sort-by-owner-id",
    "panel-sort-by-group-id",
    "panel-sort-by-owner-name",	
    "panel-sort-by-group-name",
    "select-file",
    "file-to-input-line",
    "beginning-of-panel",
    "end-of-panel",
    "scroll-down",
    "scroll-up",
    "previous-line",
    "next-line",
    "other-panel",
    "change-directory",
    "hard-refresh",
    "select-files-matching-pattern",
    "unselect-files-matching-pattern",
    "conform-current-directory",
    "conform-other-directory",
    "other-path-to-input-line",
    "selected-files-to-input-line",
    "backward-char",
    "forward-char",
    "backward-word",
    "forward-word",
    "beginning-of-line",
    "end-of-line",
    "delete-char",
    "backward-delete-char",
    "backward-kill-word",
    "kill-line",
    "kill-to-beginning-of-line",
    "kill-to-end-of-line",
    "just-one-space",
    "delete-horizontal-space",
    "action",
    "set-mark",
    "kill-region",
    "kill-ring-save",
    "yank",
    "exchange-point-and-mark",
};


typedef struct
{
    char *OpName;
    char *OpCommand;
    char *OpNewDir;
    char  OpSaveScreen;
    char  OpPause;
    char  OpHide;
    char  OpBuiltin;
} ConfigKey;


#define MAX_KEYS        2048      /* enough ?   :-) */
#define KEYSDATA_FIELDS	   8


#define TITLE_FIELDS	8
 
static char *TitleFields[TITLE_FIELDS] =
{
    "TitleForeground",
    "TitleBackground",
    "TitleBrightness",
    "UserName",
    "TtyName",
    "ClockForeground",
    "ClockBackground",
    "ClockBrightness"
};

#ifdef HAVE_LINUX
static int TitleColors[TITLE_FIELDS] = 
{
    CYAN, BLUE, ON, YELLOW, YELLOW, BLACK, CYAN, OFF
};
#else	/* !HAVE_LINUX */
static int TitleColors[TITLE_FIELDS] = 
{
    WHITE, BLACK, ON, WHITE, WHITE, BLACK, WHITE, OFF
};
#endif	/* !HAVE_LINUX */

#define TitleForeground	TitleColors[0]
#define TitleBackground	TitleColors[1]
#define TitleBrightness	TitleColors[2]
#define UserName 	TitleColors[3]
#define TtyName 	TitleColors[4]
#define ClockForeground TitleColors[5]
#define ClockBackground TitleColors[6]
#define ClockBrightness TitleColors[7]


int start(cmd, hide)
    char *cmd;
    int hide;
{
    int child_exit_code;
    FILE *stdout_log, *stderr_log;

    if (hide)
    {
        close(1);
        close(2);
        stdout_log = fopen(stdout_log_name, "w");
	stderr_log = fopen(stderr_log_name, "w");
	
	xtimer(XT_OFF);
	restore_signals();
	signals_dfl();
        child_exit_code = system(cmd);
	signals(signals_status);
	ignore_signals();
	xtimer(XT_ON);

	fclose(stderr_log);
        fclose(stdout_log);
        open(tty_name, O_RDWR);
        open(tty_name, O_RDWR);
    }
    else
    {
        tty_set_mode(TTY_CANONIC);
        tty_put_screen(screen);

	xtimer(XT_OFF);
	restore_signals();
	signals_dfl();
        child_exit_code = system(cmd);
        signals(signals_status);
	ignore_signals();
	xtimer(XT_ON);

	xwrite(1, "\n\n", 2);
        tty_set_mode(TTY_NONCANONIC);
    }

    return child_exit_code;
}


void removelog()
{
    if (stdout_log_name) unlink(stdout_log_name);
    if (stderr_log_name) unlink(stderr_log_name);
}


void clean_up()
{
    tty_exit();

    /* It is better not to do this here.  It can lead to an endless loop
       if xmalloc fails in write_history because xmalloc will call fatal
       and fatal will call clean_up again...  */
#if 0
    if (il)
        il_end();
#endif

    status_end();
    removelog();
}


void fatal(postmsg)
    char *postmsg;
{
    clean_up();
    fprintf(stderr, "%s: fatal error: %s.\n", program, postmsg);
    exit(1);
}


void settitle()
{
    int len;
    char *buf;
    tty_status status;
    window *title_win = window_init(0, 0, 1, SCREEN_X);

    tty_save(&status);
    tty_cursor(OFF);

    tty_bright(TitleBrightness);
    tty_foreground(TitleForeground);
    tty_background(TitleBackground);

    window_cursormove_notify(title_win, 0, 0);
    window_write(title, strlen(title));

    buf = xmalloc(SCREEN_X + 1);

    len = (sizeof(login) - 1) + 1 + login_name_len + 2 +
    	  (sizeof(tty)   - 1) + 1 + tty_name_len;

    memset(buf, ' ', len = SCREEN_X - strlen(title) - len - 2 - 6 - 1);
    window_cursormove_notify(title_win, 0, strlen(title));
    window_write(buf, len);

    free(buf);

    window_cursormove_notify(title_win, 0, strlen(title) + len);
    window_write(login, sizeof(login) - 1);
    window_putch(' ');
    tty_foreground(UserName);
    window_write(login_name, login_name_len);
    window_putch(' ');
    window_putch(' ');
    tty_foreground(TitleForeground);
    window_write(tty, sizeof(tty) - 1);
    window_putch(' ');
    tty_foreground(TtyName);
    window_write(tty_name, tty_name_len);

    tty_foreground(TitleForeground);
    window_putch(' ');
    window_putch(' ');
    window_cursormove_notify(title_win, 0, SCREEN_X - 1);
    window_putch(' ');
    
    tty_restore(&status);
}


void gitclock_validate()
{
    xtimer_set_flags(gitclock_xtimer, XT_YES);
}


void gitclock_invalidate()
{
    xtimer_set_flags(gitclock_xtimer, XT_NO);
}


void gitclock(xtimerp, signum)
    xtimer_struct *xtimerp;
    int signum;
{
    int hour;
    char buf[16];
    time_t __time;
    struct tm *__tm;
    tty_status status;

    /* Get the broken-down time representation.  */
    __time = time(NULL);
    __tm = localtime(&__time);

    tty_save(&status);
    tty_cursor(OFF);
    if ((hour = __tm->tm_hour % 12) == 0) hour = 12;
    sprintf(buf, "%2d:%02d%c", hour, __tm->tm_min,
	    (__tm->tm_hour < 12) ? 'a' : 'p');
    tty_cursormove_notify(0, SCREEN_X - 7);
    tty_bright(ClockBrightness);
    tty_foreground(ClockForeground);
    tty_background(ClockBackground);
    tty_write(buf, strlen(buf));
    tty_restore(&status);

    /* Don't update the input_line point if not called within interrupt.  */
    if (xtimerp)
	il_update_point();
}


void dispatch_input_line_commands(key)
    int key;
{
    switch (key)
    {
	case BUILTIN_backward_char:
	    il_backward_char();
	    break;

	case BUILTIN_forward_char:
	    il_forward_char();
	    break;

	case BUILTIN_backward_word:
	    il_backward_word();
	    break;

	case BUILTIN_forward_word:
	    il_forward_word();
	    break;

	case BUILTIN_beginning_of_line:
	    il_beginning_of_line();
	    break;

	case BUILTIN_end_of_line:
	    il_end_of_line();
	    break;

	case BUILTIN_delete_char:
	    il_delete_char();
	    break;

	case BUILTIN_backward_delete_char:
	    il_backward_delete_char();
	    break;

	case BUILTIN_backward_kill_word:
	    il_backward_kill_word();
	    break;

	case BUILTIN_kill_line:
	    il_kill_line(IL_STORE);
	    break;

	case BUILTIN_kill_to_beginning_of_line:
	    il_kill_to_beginning_of_line();
	    break;

	case BUILTIN_kill_to_end_of_line:
	    il_kill_to_end_of_line();
	    break;

	case BUILTIN_just_one_space:
	    il_just_one_space();
	    break;

	case BUILTIN_delete_horizontal_space:
	    il_delete_horizontal_space();
	    break;

        case BUILTIN_set_mark:
	    il_set_mark();
	    break;

	case BUILTIN_kill_region:
	    il_kill_region();
	    break;

	case BUILTIN_kill_ring_save:
	    il_kill_ring_save();
	    break;

	case BUILTIN_yank:
	    il_yank();
	    break;

	case BUILTIN_exchange_point_and_mark:
	    il_exchange_point_and_mark();
	    break;

	default:
	    il_insert_char(key);
	    break;
    }

    il_update();
}


/* WARNING: dest *must* be a pointer to a NULL pointer or a pointer to a
   pointer allocated with xmalloc. In the first case, il_wait_for_input()
   will return a string allocated with xmalloc(). In the second case, it
   will reallocate the pointer as needed using xrealloc. You should free
   this pointer yourself.  */

char *il_wait_for_input(static_text, dest, default_string)
    char *static_text;
    char **dest;
    char *default_string;
{
    int key;
    input_line *saved_il;
    struct key_struct *ks;
    ConfigKey *ConfigKeyInfo;

    saved_il = il_save();
    il_reset_line();

    if (static_text)
	il_set_static_text(static_text);

    if (default_string)
	il_insert_text(default_string);

    il_update();
    il_update_point();

    while (1)
    {
	ks  = tty_get_key(NULL);
	key = ks->key_seq[0];

	ConfigKeyInfo = (ConfigKey *)ks->aux_data;
	if (ConfigKeyInfo && ConfigKeyInfo->OpBuiltin)
          key = - 1 - (ConfigKeyInfo->OpName - built_in[0]) / MAX_BUILTIN_NAME;

	switch (key)
	{
	    case BUILTIN_action:
		il_get_contents(dest);
		goto done;

	    case key_INTERRUPT:
		il_kill_line(IL_DONT_STORE);
		goto done;

	    default:
		if (key)
		    dispatch_input_line_commands(key);
		break;
	}
	il_update_point();
    }

  done:

    il_restore(saved_il);
    return (key == BUILTIN_action) ? *dest : NULL;
}


/* This function is a mess. Don't try to understand what it does ... :-(  */

int getOpCommand(CommandName, src, dest, p, l)
    char *CommandName, *src, **dest;
    panel *p, *l;
{
    panel *t;
    struct group *grp;
    struct passwd *pwd;
    static int busy = 0;
    char *answer = NULL;
    char *question = NULL;
    int retval, uid, gid, len;
    int entry, oldtmplen, tmplen;
    char *ptr, *tmp = NULL, *d, *flag;


    d = *dest = xmalloc(len = (strlen(src) + 1));

    while (*src)
    {
        if (*src != '%')
	    *d++ = *src++;
        else
        {
	    t = islower(*++src) ? p : l;

            switch (*src)
            {
                case 's':

		    if (busy) { busy = 0; goto bad_command; }
		    if (*++src != '{') goto bad_command;
		    if ((ptr = strchr(++src, ',')) == NULL) goto bad_command;
		    *ptr = 0;
		    retval = getOpCommand(NULL, src, &answer, p, l);

		    if (retval != 1)
		        if (retval == 0)
		            goto bad_command;
		        else
		            goto strings_dont_match;

		    question = xmalloc(16 + strlen(CommandName) +
		    			    strlen(answer) + 1);
		    sprintf(question, "(%s) %s", CommandName, answer);
		    free(answer);
		    answer =  NULL;
		    *ptr++ = ',';
		    if ((src = strchr(ptr, '}')) == NULL) goto bad_command;
		    *src = 0;

		    if (strlen(question) > MAX_STATIC_SIZE)
		        question[MAX_STATIC_SIZE] = 0;

		    retval = getOpCommand(NULL, ptr, &answer, p, l);

		    if (retval != 1)
		    {
			*src = '}';
			free(question);
			question = NULL;
			if (retval == 0)
			    goto bad_command;
			goto strings_dont_match;
		    }

		    flag = il_wait_for_input(question, &tmp, answer);
		    
		    free(question);
		    free(answer);
		    question = answer = NULL;

		    if (flag == NULL)
		    {
			*src = '}';
			goto strings_dont_match;
		    }

		    *src = '}';
		    break;

                case 'f':
                case 'F':

		    if (panel_get_current_file_type(t) != FILE_ENTRY)
			goto strings_dont_match;
		  get_file_name:
		    ptr = panel_get_current_file_name(t);
		    tmp = xmalloc(1 + strlen(ptr) + 1 + 1);
		    sprintf(tmp, "\"%s\"", ptr);
		    break;

                case 'd':
                case 'D':

		    if (panel_get_current_file_type(t) != DIR_ENTRY)
			goto strings_dont_match;
		    goto get_file_name;

                case 'l':
                case 'L':

		    if (panel_get_current_file_type(t) != SYMLINK_ENTRY)
			goto strings_dont_match;
		    goto get_file_name;

                case 't':
                case 'T':

		    if (panel_get_current_file_type(t) != FIFO_ENTRY)
			goto strings_dont_match;
		    goto get_file_name;

                case 'z':
                case 'Z':

		    if (panel_get_current_file_type(t) != SOCKET_ENTRY)
			goto strings_dont_match;
		    goto get_file_name;

                case 'a':
                case 'A':

		    goto get_file_name;

                case 'm':
                case 'M':

		    tmp = xmalloc(16);
		    sprintf(tmp, "%o", panel_get_current_file_mode(t) & 07777);
		    break;

                case 'o':
                case 'O':

		    uid = panel_get_current_file_uid(t);
		    pwd = getpwuid(uid);

		    if (pwd)
			tmp = xstrdup(pwd->pw_name);
		    else
		    {
			tmp = xmalloc(16);
		        sprintf(tmp, "%o", uid);
		    }

		    break;

                case 'g':
                case 'G':

		    gid = panel_get_current_file_gid(t);
		    grp = getgrgid(gid);

		    if (grp)
		        tmp = xstrdup(grp->gr_name);
		    else
		    {
			tmp = xmalloc(16);
		        sprintf(tmp, "%o", gid);
		    }

		    break;

                case 'p':
                case 'P':

		    tmp = xmalloc(1 + strlen(t->path) + 1 + 1);
		    sprintf(tmp, "\"%s\"", t->path);
		    break;

                case 'b':
                case 'B':

		    ptr = strrchr(t->path, '/');
		    ptr = (*++ptr) ? ptr : "/";
		    tmp = xmalloc(1 + strlen(ptr) + 1 + 1);
		    sprintf(tmp, "\"%s\"", ptr);
		    break;

		case 'i':
		case 'I':

		    tmp = NULL;
		    tmplen = 0;

		    panel_init_iterator(t);

		    while ((entry = panel_get_next(t)) != -1)
		    {
			oldtmplen = tmplen;
			tmplen += 1 + strlen(t->dir_entry[entry].name) + 1 + 1;
			tmp = xrealloc(tmp, tmplen + 1);
			tmp[oldtmplen] = '"';
			strcpy(tmp + oldtmplen + 1, t->dir_entry[entry].name);
			tmp[tmplen - 2] = '"';
			tmp[tmplen - 1] = ' ';
			tmp[tmplen    ] = 0;

			/* this should *NOT* be here because is *VERY* ugly */

			if (t->dir_entry[entry].selected)
			{
			    t->dir_entry[entry].selected = 0;
			    t->selected_files--;
			}
		    }

		    break;

                default :

		    goto bad_command;
            }
	    src++;
	    *d = 0;
	    if (tmp)
	    {
		*dest = xrealloc(*dest, len += strlen(tmp));
		strcat(*dest, tmp);
		d = *dest + strlen(*dest);
		free(tmp);
		tmp = NULL;
	    }
        }
    }

    *d = 0;
    return 1;

  bad_command:

    free(*dest);
    *dest = NULL;
    return 0;

  strings_dont_match:

    if (tmp)
	free(tmp);

    *dest = NULL;
    return -1;
}


void refresh_after_suspend(mode)
    int mode;
{
    char *cmdln = NULL;
    char PWD[MAX_STATIC_SIZE + 1];

    if (mode == GIT_SCREEN_MODE)
    {
	/* switch back to noncanonical mode. */
	tty_set_mode(TTY_NONCANONIC);

	settitle();
	gitclock(NULL, 0);

	panel_no_optimizations(current_panel);
	panel_no_optimizations(other_panel);
	panel_action(current_panel, act_REFRESH, other_panel,   (void *)-1, 1);
	panel_action(other_panel,   act_REFRESH, current_panel, (void *)-1, 1);

	panel_set_focus(current_panel, ON, other_panel);

	status(NULL, 0, 0, 1, MSG_OK, MSG_CENTERED);

	il_get_contents(&cmdln);
	il_reset_line();
	il_set_static_text(strcat(panel_get_path(current_panel,
						 PWD,
						 MAX_STATIC_SIZE),
				  PS1));
	il_insert_text(cmdln);
	il_update();
	il_update_point();

	signals(SIG_ON);
    }
    else
    {
	/* switch back to noncanonical mode. */
	tty_set_mode(TTY_NONCANONIC);

	panel_no_optimizations(current_panel);
	panel_no_optimizations(other_panel);
	tty_put_screen(screen);
	status(CommandLineModeHelp, 0, 0, 0, MSG_OK, MSG_CENTERED);
	il_update();
	il_update_point();
    }
}


void add_to_environment(variable, alternate_variable, value)
    char *variable, *alternate_variable, *value;
{
    int result;
    char *alternate_value, *environment_string;

    if (getenv(variable) == NULL)
    {
	if (alternate_variable && (alternate_value=getenv(alternate_variable)))
	{
	    environment_string = xmalloc(strlen(variable) + 1 +
					 strlen(alternate_value) + 1);
	    sprintf(environment_string, "%s=%s", variable, alternate_value);
#ifdef HAVE_PUTENV
	    result = putenv(environment_string);
#else
	    result = setenv(variable, alternate_value, 1);
#endif /* !HAVE_PUTENV */
	}
	else
	{
	    environment_string = xmalloc(strlen(variable) + 1 +
					 strlen(value) + 1);
	    sprintf(environment_string, "%s=%s", variable, value);
#ifdef HAVE_PUTENV
	    result = putenv(environment_string);
#else
	    result = setenv(variable, value, 1);
#endif /* !HAVE_PUTENV */
	}

	if (result == -1)
            fprintf(stderr, "%s: warning: cannot add '%s' to environment\n",
		    program, variable);
    }
}


int main(argc, argv)
    int argc;
    char *argv[];
{
    FILE *stderr_log;
    struct key_struct *ks;
    ConfigKey *ConfigKeyInfo;
    unsigned char key_seq[80];
    char *StartupLeftPanelPath;
    char *StartupRightPanelPath;
    char PWD[MAX_STATIC_SIZE + 1];
    char *contents[KEYSDATA_FIELDS - 2];
    int child_exit_code, repeat_count, need_convertion;
    int entry, key, app_end = 0, wait = 0, need_clrscr = 0;
    int panel_no = 0, action_status, i, j, retval, len, ptrlen;
    char *data = NULL, *cmdln = NULL, *input = NULL, *ptr, *srcptr;


    /* Job control stuff. */
    signal(SIGTSTP, suspend);
    signal(SIGCONT, resume);

    signal(SIGSEGV, fatal_signal);
    signal(SIGHUP,  fatal_signal);

    signals(SIG_OFF);
    ignore_signals();

    xtimer(XT_OFF);

#ifdef HAVE_GCC
    printf(PRODUCT" "VERSION" (%s), %s %s\n",
	   HOST, __TIME__, __DATE__);
#else
    printf("GNU Interactive Tools (%s)\n",
	   HOST);
#endif /* !HAVE_GCC */

    printf("GIT is free software; you can redistribute it and/or modify it under the\n");
    printf("terms of the GNU General Public License as published by the Free Software\n");
    printf("Foundation; either version 2, or (at your option) any later version.\n");
    printf("Copyright (C) 1993, 1994, 1995 Free Software Foundation, Inc.\n");
    printf("Written by Tudor Hulubei and Andrei Pitis, students at PUB, Romania\n");

    program = argv[0];
    pid     = getpid();

    home = getenv("HOME");
    if (home == NULL) home = ".";

    get_tty_name();
    get_login_name();

    if (system(NULL) == 0)
	fprintf(stderr, "%s: warning: cannot find a command processor\n",
		program);

    add_to_environment("GIT_EDITOR",   "EDITOR", "vi");
    add_to_environment("GIT_SHELL",    "SHELL",  "/bin/sh");
    add_to_environment("GIT_RMAIL",    NULL,     "emacs -f rmail");
    add_to_environment("GIT_COMPRESS", NULL,     "gzip -9");
    add_to_environment("GIT_VMSTAT",   NULL,     "free");

    tty_get_capabilities();
    tty_kbdinit(TTY_RESTRICTED_INPUT);

    configuration_check();

    tty_get_exit_colors();

    use_section("[Setup]");

    configuration_getvarinfo("TempDirectory", &data, 1, DO_SEEK);
    TempDirectory = data ? tilde_expand(data) : "/tmp";

    AnsiColorSequences = get_flag_var("AnsiColorSequences", OFF);


    use_section("[GIT-Setup]");

    NormalModeHelp      = get_string_var("NormalModeHelp",      "");
    CommandLineModeHelp = get_string_var("CommandLineModeHelp", "");

    configuration_getvarinfo("StartupLeftPanelPath", &data, 1, DO_SEEK);
    StartupLeftPanelPath = data ? tilde_expand(data) : ".";

    configuration_getvarinfo("StartupRightPanelPath", &data, 1, DO_SEEK);
    StartupRightPanelPath = data ? tilde_expand(data) : ".";


    use_section(AnsiColorSequences ? cSection : bwSection);

    get_colorset_var(TitleColors, TitleFields, TITLE_FIELDS);


    use_section("[GIT-Keys]");

    for (i = 0; i < MAX_KEYS; i++)
    {
        configuration_getvarinfo((char *)key_seq, contents,
        			 KEYSDATA_FIELDS - 2, NO_SEEK);

	if (*key_seq == 0)
	    break;

	if (*key_seq != '^')
	{
	    char *key_seq_ptr = tty_get_symbol_key_seq((char *)key_seq);

	    if (!(need_convertion = key_seq_ptr == NULL))
		strcpy(key_seq, key_seq_ptr);
	}
	else
	    need_convertion = 1;

	ConfigKeyInfo = (ConfigKey *)xmalloc(sizeof(ConfigKey));

	if (contents[0])
            ConfigKeyInfo->OpName = xstrdup(contents[0]);
        else
        {
            ConfigKeyInfo->OpName = NULL;
            continue;
        }

	if (contents[1])
            ConfigKeyInfo->OpCommand = xstrdup(contents[1]);
        else
        {
            ConfigKeyInfo->OpCommand = NULL;
            goto insert;
        }

        if (contents[2])
            ConfigKeyInfo->OpNewDir = xstrdup(contents[2]);
        else
	    ConfigKeyInfo->OpNewDir = NULL;

	if (contents[3])
	    ConfigKeyInfo->OpSaveScreen = ((tolower(contents[3][0])=='y')?1:0);
        else
	    ConfigKeyInfo->OpSaveScreen = 1;

	if (contents[4])
	    ConfigKeyInfo->OpPause = ((tolower(contents[4][0]) == 'y') ? 1:0);
        else
	    ConfigKeyInfo->OpPause = 0;

	if (contents[5])
	    ConfigKeyInfo->OpHide = ((tolower(contents[5][0]) == 'y') ? 1:0);
        else
	    ConfigKeyInfo->OpHide = 0;

      insert:

        ConfigKeyInfo->OpBuiltin = 0;

        for (j = 0; j < BUILTIN_OPERATIONS; j++)
            if (strcmp(ConfigKeyInfo->OpName, built_in[j]) == 0)
            {
                free(ConfigKeyInfo->OpName);
                ConfigKeyInfo->OpName = built_in[j];
                ConfigKeyInfo->OpBuiltin = 1;
                break;
            }

	if (ConfigKeyInfo->OpBuiltin || ConfigKeyInfo->OpCommand)
	    if (!need_convertion || tty_key_convert(key_seq))
	        tty_key_list_insert(key_seq, (void *)ConfigKeyInfo);
    }

    if (i == MAX_KEYS)
        fprintf(stderr,	"%s: too many key sequences; only %d are allowed.\n",
		program, MAX_KEYS);

    tty_get_size(&SCREEN_X, &SCREEN_Y);
    tty_startup();

#ifndef HAVE_LONG_FILE_NAMES
    fprintf(stderr, "%s: warning: your system doesn't support long file names.",
    	    program);
#endif /* !HAVE_LONG_FILE_NAMES */

    stdout_log_name = xmalloc(32 + strlen(TempDirectory) + 1);
    stderr_log_name = xmalloc(32 + strlen(TempDirectory) + 1);
    sprintf(stdout_log_name, "%s/git.1.%d", TempDirectory, pid);
    sprintf(stderr_log_name, "%s/git.2.%d", TempDirectory, pid);

#ifdef HAVE_LINUX
    if (LinuxConsole == ON)
	screen = xmalloc(2 + SCREEN_X * SCREEN_Y);
#endif	/* HAVE_LINUX */

    status_init(SCREEN_X, SCREEN_Y - 1, NormalModeHelp);

    if (getuid() == 0)
	PS1[1] = '#';

    il_init(SCREEN_X, SCREEN_Y - 2);

    left_panel  = panel_init(SCREEN_Y - 3, SCREEN_X >> 1, 0, 1,
			     StartupLeftPanelPath,  &UserHeartAttack);
    right_panel = panel_init(SCREEN_Y - 3, SCREEN_X >> 1, SCREEN_X >> 1, 1,
			     StartupRightPanelPath, &UserHeartAttack);

    configuration_end();

    tty_get_screen(screen);
    tty_set_mode(TTY_NONCANONIC);

    /* Setup the clock and the tty hooks.  */
    gitclock_xtimer = xtimer_register(gitclock);
    tty_enter_idle_hook = gitclock_validate;
    tty_exit_idle_hook  = gitclock_invalidate;

    if (FrameDisplay == OFF)
    {
	tty_defaults();
	tty_clrscr();
    }

restart:

    signals(SIG_OFF);

    if (wait)
    {
	tty_get_key(NULL);
	wait = 0;
    }

    if (need_clrscr)
    {
	tty_defaults();
	tty_clrscr();
	need_clrscr = 0;
    }

    settitle();
    gitclock(NULL, 0);

    current_panel = panel_no ? right_panel : left_panel;
    other_panel   = panel_no ?  left_panel : right_panel;

    panel_action(current_panel, act_REFRESH, other_panel,   (void *)-1, 1);
    panel_action(other_panel,   act_REFRESH, current_panel, (void *)-1, 1);

    panel_set_focus(current_panel, ON, other_panel);

    status(NULL, 0, 0, 1, MSG_OK, MSG_CENTERED);

    il_set_static_text(strcat(panel_get_path(current_panel,
					     PWD,
					     MAX_STATIC_SIZE),
			      PS1));
    signals(SIG_ON);
    xtimer(XT_ON);
    gitclock(NULL, 0);

    il_update();
    il_update_point();

    while(!app_end)
    {
	UserHeartAttack = 0;
	suspend_allowed = ON;
	ks  = tty_get_key(&repeat_count);
	key = ks->key_seq[0];
	suspend_allowed = OFF;

	signals(SIG_OFF);

	ConfigKeyInfo = (ConfigKey *)ks->aux_data;

	if (ConfigKeyInfo)
	 if (ConfigKeyInfo->OpBuiltin)
	    key = - 1 - (ConfigKeyInfo->OpName-built_in[0]) / MAX_BUILTIN_NAME;
	 else
	 {
	    if (ConfigKeyInfo->OpName)
	    {
		panel_no_optimizations(current_panel);
		panel_no_optimizations(other_panel);

		if (ConfigKeyInfo->OpCommand)
		{
		    char *cmd = NULL;

		    if ((retval = getOpCommand(ConfigKeyInfo->OpName,
					       ConfigKeyInfo->OpCommand, &cmd,
					       current_panel, other_panel)))
		    {
			if (retval == 1)
			{
			    int msglen = 32 + strlen(ConfigKeyInfo->OpName) +
			    		 strlen(cmd) + 1;
			    char *msg  = xmalloc(msglen);
			    
			    sprintf(msg, "%s: %s",
				    ConfigKeyInfo->OpName, cmd);
			    status(msg, 0, 0, 0, MSG_WARNING, MSG_CENTERED);
			    free(msg);

			    child_exit_code = start(cmd,
			    			    ConfigKeyInfo->OpHide);

			    free(cmd);

			    if (ConfigKeyInfo->OpHide)
			    {
				stderr_log = fopen(stderr_log_name, "r");

				if (child_exit_code)
				    tty_beep();

				if (stderr_log == NULL)
				   status("*** Can't open stderr log file ***",
					  1, 1, 1, MSG_ERROR, MSG_CENTERED);
				else
				{
				    char *buf = xmalloc(SCREEN_X + 1);

				    while (fgets(buf, SCREEN_X+1, stderr_log))
				        if (status(buf, 1, 0, 0, MSG_ERROR,
				        	   MSG_LEFTALIGNED) !=
					    key_ENTER)
					    break;

				    free(buf);
				}

				fclose(stderr_log);
			    }
			    else
			    {
				if (ConfigKeyInfo->OpSaveScreen)
				    tty_get_screen(screen);

				tty_touch();

				if (FrameDisplay == OFF)
				    need_clrscr = 1;
				if (ConfigKeyInfo->OpPause)
				    wait = 1;
			    }

			    if (child_exit_code == 0 &&
				ConfigKeyInfo->OpNewDir)
			    {
				char *expanded_dir =
					tilde_expand(ConfigKeyInfo->OpNewDir);

				panel_action(current_panel, act_CHDIR,
					     other_panel,
					     expanded_dir, 1);
				
				free(expanded_dir);
			    }

			    goto restart;
			}
			else
			    continue;
		    }
		    else
		    {
			char *msg = xmalloc(80 + strlen(ks->key_seq) + 1);
			sprintf(msg, "Bad configured command for key %s !",
				ks->key_seq);
			status(msg, 1, 1, 1, MSG_ERROR, MSG_CENTERED);
			free(msg);
			continue;
		    }
		}
	     }
	}

        signals(SIG_ON);

        switch (key)
        {
	    case key_INTERRUPT:
		il_kill_line(IL_DONT_STORE);
		il_update();
		break;

            case BUILTIN_other_panel:

 		signals(SIG_OFF);
		if ((repeat_count & 1) == 0)
		    break;
		panel_set_focus(current_panel, OFF, NULL);
		temp_panel = current_panel;
		current_panel = other_panel;
		other_panel = temp_panel;
		panel_no = !panel_no;
		panel_set_focus(current_panel, ON, other_panel);

	    restore:

		/* Every case statement on which we use the input line
		   should end here in order to restore it.  */

		il_set_static_text(strcat(panel_get_path(current_panel,
							 PWD,
							 MAX_STATIC_SIZE),
					  PS1));
		il_update();
		break;

            case BUILTIN_previous_line:

		signals(SIG_OFF);
		panel_action(current_panel, act_UP, other_panel,
			     NULL, repeat_count);
		break;

            case BUILTIN_next_line:

		signals(SIG_OFF);
		panel_action(current_panel, act_DOWN, other_panel,
			     NULL, repeat_count);
		break;

            case BUILTIN_action:

		action_status = 0;
		il_get_contents(&cmdln);

		switch (*cmdln)
		{
		    case '+':
		        action_status = panel_action(current_panel,
						     act_SELECT_ALL,
						     other_panel, NULL, 1);
			break;

		    case '-':
			action_status = panel_action(current_panel,
						     act_UNSELECT_ALL,
						     other_panel, NULL, 1);
			break;

		    case '*':
			action_status = panel_action(current_panel,
						     act_TOGGLE,
						     other_panel, NULL, 1);
			break;

		    case 0:
			signals(SIG_OFF);
			action_status = panel_action(current_panel, act_ENTER,
						     other_panel, screen, 1);
			break;

		    default:
			{
			    char *output_string;
			    if (history_expand(cmdln, &output_string) >= 0)
			    {
				il_kill_line(IL_DONT_STORE);
				il_insert_text(output_string);
				start(output_string, 0);
				panel_no_optimizations(current_panel);
				panel_no_optimizations(other_panel);
				tty_touch();
				action_status = -1; 
			    }
			    else
			    {
				status(output_string, 1, 1, 1,
				       MSG_ERROR, MSG_CENTERED);
			    }
			}
			break;
		}

		if (action_status == 1)
		{
		    il_set_static_text(strcat(panel_get_path(current_panel,
							     PWD,
							     MAX_STATIC_SIZE),
					      PS1));
		    il_kill_line(IL_DONT_STORE);
		    il_update();
		}
		else
		    if (action_status == -1)
		    {
			tty_get_screen(screen);
			il_history(IL_RECORD);
			il_kill_line(IL_DONT_STORE);
			if (FrameDisplay == OFF)
			    need_clrscr = 1;
			goto restart;
		    }
		break;

            case BUILTIN_select_file:

		signals(SIG_OFF);

		for (i = 0; i < repeat_count; i++)
		    panel_action(current_panel, act_SELECT,
				 other_panel, NULL, 1);
		break;

            case BUILTIN_scroll_down:

		signals(SIG_OFF);

		for (i = 0; i < repeat_count; i++)
		    panel_action(current_panel, act_PGUP, other_panel, NULL,1);
		
		break;

            case BUILTIN_scroll_up:

		signals(SIG_OFF);

		for (i = 0; i < repeat_count; i++)
		    panel_action(current_panel, act_PGDOWN,
				 other_panel, NULL, 1);
		break;

            case BUILTIN_beginning_of_panel:

		signals(SIG_OFF);
		panel_action(current_panel, act_HOME, other_panel, NULL, 1);
		break;

            case BUILTIN_end_of_panel:

		signals(SIG_OFF);
		panel_action(current_panel, act_END, other_panel, NULL, 1);
		break;

            case BUILTIN_goto_root_directory:  

		signals(SIG_OFF);
		panel_action(current_panel, act_CHDIR, other_panel, "/", 1);
		goto restore;

	    case BUILTIN_hard_refresh:
	    
	        tty_touch();

            case BUILTIN_refresh:

		panel_no_optimizations(current_panel);
		panel_no_optimizations(other_panel);

		if (FrameDisplay == OFF)
		    need_clrscr = 1;
		goto restart;

            case BUILTIN_show_terminal:

		if ((repeat_count & 1) == 0)
		    break;

		tty_put_screen(screen);
		status(CommandLineModeHelp, 0, 0, 0, MSG_OK, MSG_CENTERED);
		gitclock(NULL, 0);
		il_update();
		il_update_point();

		while (1)
		{
		    current_mode = GIT_TERMINAL_MODE;
		    suspend_allowed = ON;
		    ks  = tty_get_key(&repeat_count);
		    key = ks->key_seq[0];
		    suspend_allowed = OFF;

		    ConfigKeyInfo = (ConfigKey *)ks->aux_data;
		    if (ConfigKeyInfo && ConfigKeyInfo->OpBuiltin)
		        key = - 1 - (ConfigKeyInfo->OpName - built_in[0]) /
			             MAX_BUILTIN_NAME;

		    if (key == BUILTIN_show_terminal && (repeat_count & 1))
		        break;

		    switch (key)
		    {
			case key_INTERRUPT:
			    il_kill_line(IL_DONT_STORE);
			    il_update();
			    break;

			case BUILTIN_action:

			    il_get_contents(&cmdln);

			    if (*cmdln)
			    {
				char *output_string;

			        if (history_expand(cmdln, &output_string) < 0)
			        {
				    status(output_string, 1, 1, 1,
					   MSG_ERROR, MSG_CENTERED);
				    break;
				}

				tty_put_screen(screen);
				il_kill_line(IL_DONT_STORE);
				il_insert_text(output_string);
				start(output_string, 0);
				gitclock(NULL, 0);
				tty_get_screen(screen);
				il_history(IL_RECORD);
				status(CommandLineModeHelp, 0, 0, 0,
				       MSG_OK, MSG_CENTERED);
				il_kill_line(IL_DONT_STORE);
				il_update();
			    }
			    break;

			case BUILTIN_previous_history_element:
			case BUILTIN_previous_line:

			    for (i = 0; i < repeat_count; i++)
			        il_history(IL_PREVIOUS);

			    break;

			case BUILTIN_next_history_element:
			case BUILTIN_next_line:

			    for (i = 0; i < repeat_count; i++)
			        il_history(IL_NEXT);

			    break;

			case BUILTIN_refresh:

			    panel_no_optimizations(current_panel);
			    panel_no_optimizations(other_panel);
			    tty_put_screen(screen);
			    status(CommandLineModeHelp, 0, 0, 0,
			    	   MSG_OK, MSG_CENTERED);
			    gitclock(NULL, 0);
			    il_update();
			    break;

			case BUILTIN_exit:

			    if (status(exit_msg, 1, 0, 1,
			    	       MSG_ERROR, MSG_CENTERED) == key_ENTER)
			    {
				app_end = 1;
				goto end_show_terminal;
			    }
			    status(CommandLineModeHelp, 0, 0, 0,
			    	   MSG_OK, MSG_CENTERED);
			    break;

			default:

			    if (key)
			        dispatch_input_line_commands(key);
			    break;
		    }

		    il_update_point();
		}

	      end_show_terminal:

		panel_no_optimizations(current_panel);
		panel_no_optimizations(other_panel);
		tty_touch();
		status(NULL, 0, 0, 1, MSG_OK, MSG_CENTERED);

		current_mode = GIT_SCREEN_MODE;

		if (app_end)
		    continue;
		else
		{
		    if (FrameDisplay == OFF)
			need_clrscr = 1;
		    goto restart;
		}

            case BUILTIN_copy_files:

		panel_action(current_panel, act_COPY, other_panel, NULL, 1);
		goto restore;

            case BUILTIN_rename_move_files:

		panel_action(current_panel, act_MOVE, other_panel, NULL, 1);
		goto restore;

            case BUILTIN_make_directory:

		signals(SIG_OFF);
		panel_action(current_panel, act_MKDIR, other_panel, NULL, 1);
		goto restore;

            case BUILTIN_delete_files:

		panel_action(current_panel, act_DELETE, other_panel, NULL, 1);
		break;

	    case BUILTIN_panel_display_next_mode:

	    case BUILTIN_panel_display_owner_group:
	    case BUILTIN_panel_display_date_time:
	    case BUILTIN_panel_display_size:
	    case BUILTIN_panel_display_mode:
	    case BUILTIN_panel_display_full_name:

		signals(SIG_OFF);
		panel_action(current_panel,
			     act_DISPLAY_NEXT_MODE -
			     (key - BUILTIN_panel_display_next_mode),
			     NULL, NULL, 1);
		break;

	    case BUILTIN_panel_sort_next_method:

	    case BUILTIN_panel_sort_by_name:
	    case BUILTIN_panel_sort_by_extension:
	    case BUILTIN_panel_sort_by_size:
	    case BUILTIN_panel_sort_by_date:
	    case BUILTIN_panel_sort_by_mode:
	    case BUILTIN_panel_sort_by_owner_id:
	    case BUILTIN_panel_sort_by_group_id:
	    case BUILTIN_panel_sort_by_owner_name:
	    case BUILTIN_panel_sort_by_group_name:

		signals(SIG_OFF);
		panel_action(current_panel,
			     act_SORT_NEXT_METHOD -
			     (key - BUILTIN_panel_sort_next_method),
			     NULL, NULL, 1);
		break;

            case BUILTIN_exit:

		signals(SIG_OFF);
		if (status(exit_msg, 1, 0, 1, MSG_ERROR, MSG_CENTERED) ==
		    key_ENTER)
		    app_end = 1;
		il_update_point();
		continue;

            case BUILTIN_file_to_input_line:

		signals(SIG_OFF);
		len = il_get_contents(&cmdln);

		if (len && isalnum(cmdln[len - 1]))
		    break;

		srcptr = panel_get_current_file_name(current_panel);
		ptr = xmalloc(ptrlen = 1 + strlen(srcptr) + 1 + 1 + 1);

	      copy_to_cmdln:

		sprintf(ptr, "\"%s\" ", srcptr);
		cmdln = xrealloc(cmdln, len + ptrlen);

		for (i = 0; ptr[i]; i++)
		    if (!is_print(ptr[i]))
		    {
			tty_beep();
			ptr[i] = 0;
			break;
		    }

		strcpy(&cmdln[len], ptr);
		il_insert_text(ptr);
		free(ptr);
		il_update();
		break;

            case BUILTIN_other_path_to_input_line:

		signals(SIG_OFF);
		len = il_get_contents(&cmdln);

		if (len && isalnum(cmdln[len - 1]))
		    break;

		ptr = xmalloc(ptrlen = 1 + other_panel->pathlen + 1 + 1 + 1);
		srcptr = other_panel->path;

		goto copy_to_cmdln;

            case BUILTIN_selected_files_to_input_line:

		signals(SIG_OFF);
		len = il_get_contents(&cmdln);

		if (len && isalnum(cmdln[len - 1]))
		    break;

		panel_init_iterator(current_panel);

		while ((entry = panel_get_next(current_panel)) != -1)
		{
		    srcptr = current_panel->dir_entry[entry].name;
		    ptr = xmalloc(ptrlen = 1 + strlen(srcptr) + 1 + 1 + 1);
		    sprintf(ptr, "\"%s\" ", srcptr);
		    cmdln = xrealloc(cmdln, len + ptrlen);

		    for (i = 0; ptr[i]; i++)
			if (!is_print(ptr[i]))
			{
			    tty_beep();
			    ptr[i] = 0;
			    break;
			}

		    strcpy(&cmdln[len], ptr);
		    il_insert_text(ptr);
		    free(ptr);
		}

		il_update();
		break;

	    case BUILTIN_previous_history_element:

		signals(SIG_OFF);

		for (i = 0; i < repeat_count; i++)
		    il_history(IL_PREVIOUS);

		break;

	    case BUILTIN_next_history_element:

		signals(SIG_OFF);

		for (i = 0; i < repeat_count; i++)
		    il_history(IL_NEXT);

		break;
		
	    case BUILTIN_switch_panels:

		if ((repeat_count & 1) == 0)
		    break;

		signals(SIG_OFF);
		panel_no_optimizations(current_panel);
		panel_no_optimizations(other_panel);
		panel_action(current_panel, act_SWITCH, other_panel, NULL, 1);
		panel_action(other_panel, act_REFRESH,
			     current_panel, (void *)-1, 1);
		panel_action(current_panel, act_REFRESH,
			     other_panel, (void *)-1, 1);
		break;

	    case BUILTIN_change_directory:

		signals(SIG_OFF);

		if (il_wait_for_input("(CHDIR) Change to directory: ",
				      &input, NULL))
		{
		    char *expanded_input;

		    if (input[0] == 0)
			goto restore;

		    panel_action(current_panel, act_CHDIR, other_panel,
		    		 expanded_input = tilde_expand(input), 1);

		    free(expanded_input);
		    free(input);
		    input = NULL;
		}

		goto restore;


	    case BUILTIN_select_files_matching_pattern:

		signals(SIG_OFF);

		if (il_wait_for_input("(SELECT) Pattern: ",
				      &input, NULL))
		{
		    if (input[0] == 0)
			goto restore;

		    panel_action(current_panel, act_PATTERN_SELECT,
		    		 other_panel, input, 1);

		    free(input);
		    input = NULL;
		}

		goto restore;

	    case BUILTIN_unselect_files_matching_pattern:

		signals(SIG_OFF);

		if (il_wait_for_input("(UNSELECT) Pattern: ",
				      &input, NULL))
		{
		    if (input[0] == 0)
			goto restore;

		    panel_action(current_panel, act_PATTERN_UNSELECT,
		    		 other_panel, input, 1);

		    free(input);
		    input = NULL;
		}

		goto restore;

	    case BUILTIN_conform_current_directory:

		signals(SIG_OFF);

		panel_action(current_panel, act_CHDIR, other_panel,
			     other_panel->path, 1);

		goto restore;

	    case BUILTIN_conform_other_directory:

		signals(SIG_OFF);

		panel_action(other_panel, act_CHDIR, current_panel,
			     current_panel->path, 1);

		goto restore;
	    
            default:

		if (key)
		    dispatch_input_line_commands(key);
		break;
	}

	signals(SIG_ON);
        il_update_point();
    }

    signals(SIG_OFF);
    panel_end(left_panel);
    panel_end(right_panel);
    tty_set_mode(TTY_CANONIC);

    if (il)
        il_end();

    status_end();
    removelog();

#ifdef HAVE_LINUX
    if (LinuxConsole == ON)
	tty_put_screen(screen);
    else
    {
	tty_defaults();
	tty_clrscr();
    }
#else	/* !HAVE_LINUX */
    tty_defaults();
    tty_clrscr();
#endif	/* !HAVE_LINUX */

    return 0;
}
