/*  "@(#) autolog.c      0.2
    Originally ported by David Dickson" 
    Modified by Michael C. Mitchell, 15Oct94
    Modified by Kyle Bateman, 27Nov94

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include    <stdio.h>
#include    <signal.h>
#include    <string.h>
#include    <sys/types.h>
#include    <sys/stat.h>
#include    <utmp.h>
#include    <time.h>

#define     GRACE       15  /* grace time (sec) for user reply */
#define     TIMEMAX     5   /* maximum idle time (minutes) */
#define		DOMAIL		1	/* notify user by mail */
#define		CIDLE		1	/* consider idle time only */
#define     KWAIT       20  /* time to wait after kill (in sec.) */
#define     WARNING     1   /* a warning message */
#define     LOGOFF      2   /* a log-off message */
#define     NOLOGOFF    3   /* a log-off failure message */
#define		NOTIFY		4	/* hard logoff notification */
#define     STRLEN      64
#define     LINELEN     256
#define     MAXCONF     512 /* maximum lines in config file */

struct utmp *utmpp;         /* pointer to utmp file entry */
int debug = 0;
int nokill = 0;
int listall = 0;
char *confname = "/etc/autolog.conf";
char *logfname = "/usr/adm/logoutlog";
int g_timemax = TIMEMAX;
int g_grace = GRACE;
int g_domail = DOMAIL;
int g_cidle = CIDLE;

typedef struct conf
    {
    char    *name;          /* username */
    char    *line;          /* tty line */
    int     idle;           /* minutes of idle time */
    int     grace;          /* minutes of grace time */
    int     domail;         /* notify by mail */
    int     cidle;          /* consider idle time only */
    } conf_el;
conf_el conf_arr[MAXCONF];
int conf_idx = 0;

main(int argc, char *argv[])
    {
    int i;
    for (i = 1; i < argc; i++)
        if (argv[i][0] == '-')
            switch(argv[i][1])
                {
                case 'a':
                    listall = 1;
                    break;
                case 'd':
                    debug = 1;
                    break;
                case 'n':
                    nokill = 1;
                    break;
                case 'h':
                    g_cidle = 0;
                    break;
                case 'l':
                    logfname = argv[++i];
                    break;
                case 'f':
                    confname = argv[++i];
                    break;
                case 't':
                    g_timemax = atoi(argv[++i]);
                    break;
                case 'g':
                    g_grace = atoi(argv[++i]);
                    break;
                case 'm':
                    g_domail = (argv[++i][0] == 'y');
                    break;
                default:
                    fprintf(stderr, "autologout: illegal switch: %s\n", argv[i]);
                }
        else
            fprintf(stderr, "autologout: illegal parameter: %s\n", argv[i]);
    eat_confile();
    if (!debug)
      	if (fork())             /* the parent process */
        	exit(0);            /* exits */
    /* the child processes all utmp file entries: */
    while ((utmpp = getutent()) != (struct utmp *) NULL)
        check_idle();
    exit(0);                    /* done, so bye */
    }

eat_confile()
    {
    FILE *f;
    char *s;
    char iline[LINELEN], tmpname[STRLEN], tmpline[STRLEN], tmpmail[STRLEN], tmpih;
    int tmpidle, tmpgrace;
    
    if (!(f=fopen(confname, "r")) )
        if (debug)
	        fprintf(stderr, "autologout: Can't find file: %s\n", confname);
    while (fgets(iline, LINELEN, f))
        {
        if (*iline == '#' || isspace(*iline))
        	continue;
        sscanf(iline,"%s %s %d %d %s %c\n",tmpname,tmpline,&tmpidle,&tmpgrace,tmpmail,&tmpih);
        if (debug)
            printf("N:%-20s L:%-20s I:%3d G:%3d\n",tmpname,tmpline,tmpidle,tmpgrace);
        conf_arr[conf_idx].name=strcpy((char *)malloc(strlen(tmpname)+1),tmpname);
        conf_arr[conf_idx].line=strcpy((char *)malloc(strlen(tmpline)+1),tmpline);
        conf_arr[conf_idx].idle =tmpidle;
        conf_arr[conf_idx].grace=tmpgrace;
        conf_arr[conf_idx].domail=(*tmpmail=='y' || *tmpmail=='Y');
        conf_arr[conf_idx].cidle=(tmpih!='h' && tmpih!='H');
        conf_idx++;
        }
    fclose(f);
    }

/* return true if strg matches the regex in pattern */
pat_match(char *pattern,char *strg)
	{
	re_comp(pattern);
	return(re_exec(strg));
	}

check_idle()    /* select utmp entries older than TIMEMAX */
    {
    char dev[STRLEN], name[STRLEN], prname[STRLEN];
    struct stat status;
    time_t idle, pres_time, start, time(), stime;
    int i, timemax = g_timemax, grace = g_grace, domail = g_domail, cidle = g_cidle;

    if (utmpp->ut_type != USER_PROCESS) /* if not user process */
    	{
    	if (listall)
    		printf("Non-user process: N:%-8s P:%5d Login:%s",utmpp->ut_user,utmpp->ut_pid,ctime(&utmpp->ut_time));
        return(0);                      /* skip the utmp entry */
        }
    sprintf(dev,"/dev/%s",utmpp->ut_line);/* append /dev/ to base name */
    if (stat(dev, &status))               /* if can't get status for port */
        bailout("Can't get status of user's terminal", 1);

    if (debug)
        printf("\nChecking: %-8s on %-12s P:%-6d Login: %s",utmpp->ut_user,dev,utmpp->ut_pid,ctime(&utmpp->ut_time));
    /* idle time = current time less last access time: */
    time(&pres_time);      						/* get current time */
    idle = (pres_time - status.st_atime) / 60;	/* total idle minutes */
    stime = (pres_time - utmpp->ut_time) / 60;	/* total session minutes */
    strncpy(name, utmpp->ut_user, UT_NAMESIZE);	/* get user name */
    name[UT_NAMESIZE] = '\0';           /* null terminate user name string */
    
    for(i = 0; i < conf_idx; i++)
    	if (pat_match(conf_arr[i].name,name) && pat_match(conf_arr[i].line,utmpp->ut_line))
    		{
    		if (debug)
    			printf("Matched record #%2d N:%-8s L:%-9s P:%-6d Sess_time:%3d:%02d\n",i,name,utmpp->ut_line,utmpp->ut_pid,stime/60,stime%60);
    		if (!conf_arr[i].idle)		/* if user exempt */
    			{
	    		if (debug)
    				printf("User exempt\n");
    			return(0);
    			}
    		timemax = conf_arr[i].idle;
    		grace = conf_arr[i].grace;
    		domail = conf_arr[i].domail;
    		cidle = conf_arr[i].cidle;
    		break;
    		}
    		
	sprintf(prname,"/proc/%d",utmpp->ut_pid);
    if (stat(prname, &status))				/* is process active? */
        {
        if (debug)
	   		printf("Process %d not found in process table\n",utmpp->ut_pid);
	   	return(0);
        }
    if (cidle)								/* if considering idle time */
    	{
		if (debug)
   			printf("PID: %-8d subject to logout   Idle time: %4d (%2d allowed)\n",utmpp->ut_pid,idle,timemax);
	    if (idle < timemax)	                 	/* if user was recently active */
    	    return(0);      	                /* let it live */
    	}
    else
    	{
		if (debug)
   			printf("PID: %-8d subject to logout   Total time: %4d (%2d allowed)\n",utmpp->ut_pid,stime,timemax);
	    if (stime < timemax)                 	/* if user still under limit */
    	    return(0);      	                /* let it live */
    	}
	if (nokill)
		{
    	printf("Would kill user N:%s L:%s\n",name,utmpp->ut_line);
    	return(1);
    	}
    if (debug)
	   	printf("Warning user N:%s L:%s  Sleep %d sec\n",name,utmpp->ut_line,grace);
    mesg(cidle?WARNING:NOTIFY, name, dev, cidle?idle:stime, 0, grace); /* send warning to user */
    if (stat(dev, &status))
        bailout("Can't get status of user's terminal", 2);
    start = status.st_atime;                /* start time for countdown */
    sleep(grace);
    if (stat(dev, &status))
        bailout("Can't get status of user's terminal", 3);
    if (!cidle || start >= status.st_atime)	/* user still idle */
        {
	    if (debug)
		   	printf("Killing user N:%s L:%s  Sleep %d sec\n",name,utmpp->ut_line,KWAIT);
        if (killit(utmpp->ut_pid))
	        {
	        mesg(LOGOFF, name, dev, (int)idle, domail, 0); /* okay */
	        return(1);
	        }
	   	mesg(NOLOGOFF, name, dev, (int)idle, 1, 0);	/* couldn't kill */
	   	}
    return(0);
    }

mesg(flag, name, dev, idle, domail, grace) /* mail to user and log message */
int     flag, idle;             /* flag indicates message type */
char    *name, *dev;
int domail, grace;
    {
    char    mbuf[256];          /* message buffer */
    time_t  tvec;
    FILE    *fp, *log = 0, *mprog;
    struct stat status;

    time(&tvec);                /* tvec = current time */
    if (stat(logfname, &status) >= 0)			/* if logfile exists */
	    log = fopen(logfname, "a");	/* append to it */
    if (flag == NOTIFY)    
        {                       /* process warning message */
        if (!(fp = fopen(dev, "w")) )
            bailout("Can't open user's terminal", 5);
        fprintf(fp,"%s: You've been on for %d min.\07\n", name, idle);
        fprintf(fp,"You'll be logged off in %d sec. so finish up:",grace);
        fclose(fp);
        if (log)
	        fprintf(log, "** NOTIFY  ** %s %s (%d min idle time) %s",name, dev+5, idle, ctime(&tvec)+3);
        }
    if (flag == WARNING)    
        {                       /* process warning message */
        if (!(fp = fopen(dev, "w")) )
            bailout("Can't open user's terminal", 5);
        fprintf(fp,"%s: You've been idle for %d min.\07\n", name, idle);
        fprintf(fp,"You'll be logged off in %d sec. unless you hit return:",grace);
        fclose(fp);
        if (log)
	        fprintf(log, "** WARNING ** %s %s (%d min idle time) %s",name, dev+5, idle, ctime(&tvec)+3);
        }
    if (flag == LOGOFF) 
        {                       /* process log-off message */
        if (log)
	        fprintf(log, "** LOGOFF  ** %s %s (%d min idle time) %s",name, dev + 5, idle, ctime(&tvec) + 3);
	    if (domail)
	    	{
	        sprintf(mbuf, "/bin/mail %s", name);
	        /* open pipe to mail program for writing */
	        if (!(mprog = popen(mbuf, "w")) )
	            bailout("Can't use /bin/mail program", 6);
	        fprintf(mprog, "Subject: Excess Idle Time\nLogged off - excess idle time - %s %s\ntty = %s\n",name, ctime(&tvec), dev + 5);
	        fclose(mprog);
	        }
        }
    if (flag == NOLOGOFF) 
        {                       /* send mail to root if can't kill */
        fprintf(log, "** LOGOFF-FAIL ** %s (pid = %d) %s (%d min idle time) %s",name, utmpp->ut_pid, dev+5, idle, ctime(&tvec)+3);
        if (domail)
        	{
	        sprintf(mbuf, "/bin/mail root");
	        if ((mprog = popen(mbuf, "w")) == (FILE *) NULL)
	            bailout("Can't use /bin/mail program", 7);
	        fprintf(mprog, "Subject: Can't logoff %s\nCan't Log off - %s %s\ntty = %s\n", name, name, ctime(&tvec), dev+5);
	        fclose(mprog);
	        }
        }
    fclose(log);
    return(0);
    }

killit(pid)	    	/* terminate process using SIGHUP, then SIGKILL */
int pid;
    {
    kill(pid, SIGHUP);          /* first send "hangup" signal */
    sleep(KWAIT);
    if (!kill(pid, 0)) 
        {                       /* SIGHUP might be ignored */
        kill(pid, SIGKILL);     /* then send sure "kill" signal */
        sleep(KWAIT);
        if (!kill(pid, 0))
            return(0);          /* failure--refuses to die! */
        else
            return(1);          /* successful kill with SIGKILL */
        }
    else
        return(1);              /* successful kill with SIGHUP */
    }

bailout(message, status)        /* display error message and exit */
int     status;                 /* exit status */
char    *message;               /* pointer to the error message */
    {
    fprintf(stderr, "autologout: %s\n", message);
    exit(status);
    }
