/* This file contains system-specific functions for ESIX.
 * The program pfdisk.c calls these routines.
 * Note that ESIX can't use the generic Sys.V/386 version of
 * this file because it uses ioctl calls to access the
 * primary boot sector.  Other systems provide a device which
 * maps onto the whole disk (starting with the boot sector).
 */
#include <stdio.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/vtoc.h>

#define extern
#include "sysdep.h"
#undef extern

int usage(prog)	/* print a usage message */
char	*prog;	/* program name */
{
  fprintf(stderr,"Usage: %s dev\n\t%s\n", prog,
	  "where 'dev' is the device name, i.e. /dev/rdsk/0s0");
}

void getGeometry(dev, c, h, s)
char	*dev;		/* device name */
unsigned *c,*h,*s;	/* cyls, heads, sectors */
{
  int devfd, retval;
  struct disk_parms dp;
  
  devfd = open(dev, O_RDONLY, 0);
  if (devfd < 0) {
    fprintf(stderr,"%s: can't open for reading\n", dev);
    return;
  }
  retval = ioctl(devfd, V_GETPARMS, &dp);
  close(devfd);
  if (retval < 0) {
    fprintf(stderr,"%s: can't get disk parameters\n", dev);
    return;
  }
  if (dp.dp_type != DPT_WINI) {
    fprintf(stderr,"%s: not a Winchester Disk\n", dev);
    return;
  }
  *c = dp.dp_cyls;
  *h = dp.dp_heads;
  *s = dp.dp_sectors;
}

int getFile(name, buf, len)	/* read file into buffer */
char	*name, *buf;
int	len;
{	/* (open, read, close) */
  int devfd, retval;
  
  devfd = open(name, O_RDONLY, 0);
  if (devfd < 0) {
    fprintf(stderr,"%s: can't open for reading\n", name);
    return(devfd);
  }
  retval = read(devfd, buf, len);
  if (retval < 0)
    fprintf(stderr,"%s: read failed\n", name);
  close(devfd);
  return(retval);
}

int putFile(name, buf, len)	/* write buffer to file */
char	*name, *buf;
int	len;
{	/* (open, write, close) */
  int devfd, retval;
  
  devfd = open(name, O_WRONLY|O_CREAT, 0666);
  if (devfd < 0) {
    fprintf(stderr,"%s: can't open for writing\n", name);
    return(devfd);
  }
  retval = write(devfd, buf, len);
  if (retval < 0)
    fprintf(stderr,"%s: write failed\n", name);
  close(devfd);
  return(retval);
}

int getBBlk(name, buf)	/* read Boot Block into buffer */
char	*name, *buf;
{	/* (open, read, close) */
  int devfd, retval;
  struct absio abs;
  
  devfd = open(name, O_RDONLY, 0);
  if (devfd < 0) {
    fprintf(stderr,"%s: can't open for reading\n", name);
    return(devfd);
  }
  abs.abs_sec = 0;	/* the primary boot sector */
  abs.abs_buf = buf;
  retval = ioctl(devfd, V_RDABS, &abs);
  if (retval < 0)
    fprintf(stderr,"%s: read failed\n", name);
  close(devfd);
  return(retval);
}

int putBBlk(name, buf)	/* write buffer to Boot Block */
char	*name, *buf;
{	/* (open, write, close) */
  int devfd, retval;
  struct absio abs;
  
  devfd = open(name, O_WRONLY, 0);
  if (devfd < 0) {
    fprintf(stderr,"%s: can't open for writing\n", name);
    return(devfd);
  }
  abs.abs_sec = 0;	/* the primary boot sector */
  abs.abs_buf = buf;
  retval = ioctl(devfd, V_WRABS, &abs);
  if (retval < 0)
    fprintf(stderr,"%s: write failed\n", name);
  close(devfd);
  return(retval);
}
