/*
   procinfo.c

   Displays general info from /proc.

   copyright (c) 1994 svm@kozmix.hacktic.nl

   This software is released under the GNU Public Licence. See the
   file `COPYING' for details. Since you're probably running Linux I'm
   sure your hard disk is already infested with copies of this file,
   but if not, mail me and I'll send you one.

   $Id: procinfo.c,v 1.16 1994/05/21 06:46:52 svm Exp svm $
 */

#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <termcap.h>
#include <signal.h>
#include <getopt.h>
#include <unistd.h>
#include <termios.h>
#include <sys/ioctl.h>

#define VERSION		"0.4 (1994-05-21)"
#define PROC_DIR	"/proc/"	/* Needs the trailing slash. */

#define ISSTR(s)	(!strcmp(s, type))
#define VAL		(atol(strtok(NULL, " ")))
#define MAX(a,b)	((a)>(b)?(a):(b))
#define DIFF(x)		(show_diff?(new.x)-(old.x):(new.x))

#define CDRV		0
#define BDRV		1
#ifndef MAX_CHRDEV
#define MAX_CHRDEV	32
#endif
#ifndef MAX_BLKDEV
#define MAX_BLKDEV	32
#endif
#define MAX_DEV		MAX(MAX_CHRDEV, MAX_BLKDEV)

typedef signed long ul;

struct info
{
    ul  uptime;
    ul  m_to, m_us, m_fr, m_sh, m_bu;
    ul  s_to, s_us, s_fr;
    ul  cpu_user, cpu_nice, cpu_sys, cpu_idle;
    ul  disk[4];
    ul  pgin, pgout, swin, swout;
    ul  intr[16];
    ul  old_intr;
    ul  ctxt;
};

struct info new, old;

void init_terminal_data (void);
char *my_tgets (char *te);

char *cd;			/* clear to eos */
char *ce;			/* clear to eol */
char *cl;			/* clear screen */
char *cm;			/* move cursor */
char *ho;			/* home */
char *se;			/* standout off */
char *so;			/* standout on */
char *ve;			/* cursor on */
char *vi;			/* cursor off */
int co;				/* columns */
int li;				/* lines */

int fs = 0, redrawn = 0;
int show_moddev = 0;
int show_all = 0;
int show_diff = 0;
int irq_array = 0;

static char booted[40];

FILE *
myfopen (char *name)
{
    FILE *fp;

    if ((fp = fopen (name, "r")) == NULL)
    {
	fprintf (stdout, "can't open file %s: %s\n", name, strerror (errno));
	exit (1);
    }
    return fp;
}

/* Note: we're using a static char array, so use this only once per printf. */
char *
hms (long t)
{
    int h, m, s;
    static char buf[16];

    h = (int) (t / 360000);
    t = t - (long) (h * 360000);
    m = (int) (t / 6000);
    t = t - (long) (m * 6000);
    s = (int) (t / 100);
    t = t - (long) (s * 100);
    sprintf (buf, "%3d:%02d:%02d.%02d", h, m, s, (int) t);
    return buf;
}

/* Note: we're using a static char array, so use this only once per printf. */
char *
perc (long i, long t)
{
    int v;
    static char buf[16];

    v = (int) (i < 1000000 ?
	       ((1000 * i + t / 2) / t) :
	       ((i + t / 2000) / (t / 1000)));
    sprintf (buf, "%3d.%d%%", v / 10, v % 10);
    return buf;
}

void
bye (int i)
{
    printf ("%s%s%s", ve, se, tgoto (cm, 0, li + 1));
    exit (0);
}

void
tstp (int i)
{
   /* Restore cursor & attributes, then do the actual suspend. Should be fun
      to try this unaltered on BSD-based systems. :-) */
    printf ("%s%s%s", ve, se, tgoto (cm, 0, li - 1));
    fflush (stdout);
    raise (SIGTSTP);
}

void
cont (int i)
{
    signal (SIGTSTP, tstp);
    signal (SIGCONT, cont);
    printf ("%s%s", cl, vi);
    fflush (stdout);
}

/* This function stolen from top(1) (kmem-version). */
void
window_init (int i)
{
    struct winsize ws;
    struct sigaction sa;

    co = li = 0;
    if (ioctl (1, TIOCGWINSZ, &ws) >= 0)
    {
	co = ws.ws_col;
	li = ws.ws_row;
    }
    if (co == 0)
	co = tgetnum ("co");
    if (li == 0)
	li = tgetnum ("li");
    li -= 2;
    if (fs)
	redrawn = 1;
    sa.sa_handler = window_init;
    sa.sa_flags = 0;
    sigemptyset (&sa.sa_mask);
    sigaction (SIGWINCH, &sa, NULL);
}

int
main (int ac, char **av)
{
    FILE *loadavgfp, *meminfofp, *modulesfp, *statfp, *uptimefp, *versionfp,
       *devicesfp, *filesystemsfp;
    char line[1024];
    char version[1024];
    int sl = 5;
    int getoptopt;
    char outbuf[4096];

    while ((getoptopt = getopt (ac, av, "fn:madhv")) != EOF)
    {
	switch (getoptopt)
	{
	case 'n':
	    sl = atoi (optarg);
	   /* PLUMMET */
	case 'f':
	    fs = 1;
	    break;
	case 'm':
	    show_moddev = 1;
	    break;
	case 'a':
	    show_all = 1;
	    break;
	case 'd':
	    show_diff = 1;
	    break;
	case 'v':
	    printf ("This is procinfo version " VERSION "\n");
	    exit (0);
	case 'h':
	default:
	    printf ("procinfo version %s\n"
		    "usage: %s [-fmavdh] [-nN]\n"
		    "\n"
		    "\t-f\trun full screen\n"
		    "\t-nN\tpause N second between updates (implies -f)\n"
		    "\t-m\tdisplay module and device info\n"
		    "\t-a\tdisplay all info\n"
		    "\t-d\tshow differences rather than totals\n"
		    "\t-v\tprint version info\n"
		    "\t-h\tprint this help\n",
		    VERSION, av[0]);
	    exit (1);
	}
    }
    if (sl == 0)
	nice (-20);

    setvbuf (stdout, outbuf, _IOFBF, sizeof (outbuf));
    init_terminal_data ();
    window_init (0);

    cd = my_tgets ("cd");
    ce = my_tgets ("ce");
    cl = my_tgets ("cl");
    cm = my_tgets ("cm");
    ho = my_tgets ("ho");
    se = my_tgets ("se");
    so = my_tgets ("so");
    ve = my_tgets ("ve");
    vi = my_tgets ("vi");
    signal (SIGHUP, bye);
    signal (SIGINT, bye);
    signal (SIGQUIT, bye);
    signal (SIGTERM, bye);
    signal (SIGTSTP, tstp);
    signal (SIGCONT, cont);

    versionfp = myfopen (PROC_DIR "version");
    fgets (version, sizeof (version), versionfp);
    version[strlen (version) - 1] = '\0';
    fclose (versionfp);

    uptimefp = myfopen (PROC_DIR "uptime");
    loadavgfp = myfopen (PROC_DIR "loadavg");
    meminfofp = myfopen (PROC_DIR "meminfo");
    statfp = myfopen (PROC_DIR "stat");
    modulesfp = myfopen (PROC_DIR "modules");
   /* These two may be missing, so check for NULL later. */
    devicesfp = fopen (PROC_DIR "devices", "r");
    filesystemsfp = fopen (PROC_DIR "filesystems", "r");

   /* See what the intr line in /proc/stat says. */
    while (fgets (line, sizeof (line), statfp))
    {
	if (!strncmp ("intr", line, 4))
	{
	   /* If this line has a space somewhere after line[5], it's a
	      new-style intr. */
	    if (strchr (&line[5], ' '))
		irq_array = 1;
	    continue;
	}
       /* While we're at it, fill in booted. */
	else if (!strncmp ("btime", line, 5))
	{
	    time_t btime;

	    btime = (time_t) atol(&line[6]);
	    strftime (booted, sizeof (booted), "%c", localtime (&btime));
	    continue;
	}
    }

    if (fs)
	printf ("%s%s", cl, vi);

    while (42)
    {
	int i;
	char loadavg[32];
	
	if (redrawn)
	{
	    redrawn = 0;
	    fputs (cl, stdout);
	}
	if (fs)
	    printf ("%s%s%*s%s\n\n", ho, so, -co, version, se);
	else
	    printf ("%s\n\n", version);

	fseek (uptimefp, 0L, SEEK_SET);
	fgets (line, sizeof (line), uptimefp);
	new.uptime = (ul) (atof (strtok (line, " ")) * 100.0);
	fgets (line, sizeof (line), uptimefp);	/* For libc 4.4; more below. */

	fseek (loadavgfp, 0L, SEEK_SET);
	fgets (line, sizeof (line), loadavgfp);
	strcpy (loadavg, line);
	loadavg[strlen (loadavg) - 1] = '\0';
	fgets (line, sizeof (line), loadavgfp);

	if (!show_moddev || show_all)
	{
	    fseek (meminfofp, 0L, SEEK_SET);
	    fputs ("Memory:        Total        Used        Free      "
		   "Shared     Buffers\n", stdout);
	    fgets (line, sizeof (line), meminfofp);
	    fgets (line, sizeof (line), meminfofp);
	    strtok (line, " ");
	    new.m_to = VAL / 1024;
	    new.m_us = VAL / 1024;
	    new.m_fr = VAL / 1024;
	    new.m_sh = VAL / 1024;
	    new.m_bu = VAL / 1024;
	    printf ("Mem:    %12ld%12ld%12ld%12ld%12ld\n",
		    DIFF (m_to), DIFF (m_us), DIFF (m_fr), DIFF (m_sh),
		    DIFF (m_bu));
	    fgets (line, sizeof (line), meminfofp);
	    strtok (line, " ");
	    new.s_to = VAL / 1024;
	    new.s_us = VAL / 1024;
	    new.s_fr = VAL / 1024;
	    printf ("Swap:   %12ld%12ld%12ld\n\n",
		    DIFF (s_to), DIFF (s_us), DIFF (s_fr));
	    fgets (line, sizeof (line), meminfofp);
	}

	fseek (statfp, 0L, SEEK_SET);
	while (fgets (line, sizeof (line), statfp))
	{
	    char *type = strtok (line, " ");

	    if (ISSTR ("cpu"))
	    {
		new.cpu_user = VAL;
		new.cpu_nice = VAL;
		new.cpu_sys = VAL;
		new.cpu_idle = VAL;
	    }
	    else if (ISSTR ("disk"))
	    {
		new.disk[0] = VAL;
		new.disk[1] = VAL;
		new.disk[2] = VAL;
		new.disk[3] = VAL;
	    }
	    else if (ISSTR ("page"))
	    {
		new.pgin = VAL;
		new.pgout = VAL;
	    }
	    else if (ISSTR ("swap"))
	    {
		new.swin = VAL;
		new.swout = VAL;
	    }
	    else if (ISSTR ("intr"))
	    {
		if (irq_array)
		{
		    VAL;	/* First value is total of all interrupts,
				   for compatibility with rpc.rstatd. We
				   ignore it. */
		    for (i = 0; i < 16; i++)
			new.intr[i] = VAL;
		}
		else
		    new.old_intr = VAL;
	    }
	    else if (ISSTR ("ctxt"))
		new.ctxt = VAL;
#if 0
	    else if (ISSTR ("btime"))
		new.btime = (time_t) VAL;
#endif
	}

	printf ("Bootup: %s\tLoad average: %s\n\n", booted, loadavg);

	if (!show_moddev || show_all)
	{
	    long elapsed;

	    elapsed = new.uptime;

	    if (irq_array)
	    {
		if (fs && old.uptime)
		    elapsed = DIFF (intr[0]);
	    }
	    else
	    {
		if (fs && old.uptime)
		   /* This won't be exact... */
		    elapsed = 100 * sl;
	    }

	    printf ("user  : %s %s\t",
		    hms (DIFF (cpu_user)), perc (DIFF (cpu_user), elapsed));
	    printf ("page in : %8ld\t", DIFF (pgin));
	    if (new.disk[0])
		printf ("disk 1: %8ld\n", DIFF (disk[0]));
	    else
		putchar ('\n');

	    printf ("nice  : %s %s\t",
		    hms (DIFF (cpu_nice)), perc (DIFF (cpu_nice), elapsed));
	    printf ("page out: %8ld\t", DIFF (pgout));
	    if (new.disk[1])
		printf ("disk 2: %8ld\n", DIFF (disk[1]));
	    else
		putchar ('\n');

	    printf ("system: %s %s\t",
		    hms (DIFF (cpu_sys)), perc (DIFF (cpu_sys), elapsed));
	    printf ("swap in : %8ld\t", DIFF (swin));
	    if (new.disk[2])
		printf ("disk 3: %8ld\n", DIFF (disk[2]));
	    else
		putchar ('\n');

	    printf ("idle  : %s %s\t",
		    hms (DIFF (cpu_idle)), perc (DIFF (cpu_idle), elapsed));
	    printf ("swap out: %8ld\t", DIFF (swout));
	    if (new.disk[3])
		printf ("disk 4: %8ld\n", DIFF (disk[3]));
	    else
		putchar ('\n');

	    printf ("uptime: %s\t\tcontext : %8ld", hms (new.uptime),
		    DIFF (ctxt));

	    if (irq_array)
	    {
		fputs ("\n\n", stdout);
		for (i = 0; i < 4; i++)
		    printf ("irq %2d: %9ld  irq %2d: %9ld  "
			    "irq %2d: %9ld  irq %2d: %9ld\n",
			    i, DIFF (intr[i]), i + 4, DIFF (intr[i + 4]),
			    i + 8, DIFF (intr[i + 8]), i + 12,
			    DIFF (intr[i + 12]));
	    }
	    else
		printf ("\tinterrupts: %8ld\n", DIFF (old_intr));
	}

	if (show_moddev || show_all)
	{
	    int barf = 0;

	    fseek (modulesfp, 0L, SEEK_SET);
	    if (show_all)
		putchar ('\n');
	    printf ("Modules:      size\n");
	    while (fgets (line, sizeof (line), modulesfp))
	    {
		char *mod;
		long pg;
		char *status;

		barf += 20;
		mod = strtok (line, " ");
		pg = atol (strtok (NULL, " ")) * 4;
		status = strtok (NULL, ")");
		printf ("%-12s%c %4ld  ",
			mod, status ? status[2] : ' ', pg);
	    }
	    if (co && (barf % co))
		printf ("%s\n", fs ? ce : "");

	    if (devicesfp)
	    {
		int maj[2][MAX_DEV];
		char *n[2][MAX_DEV];
		int count[2] = {0, 0};
		int which = 0, i, j;

		memset (n, 0, sizeof (n));
		fseek (devicesfp, 0L, SEEK_SET);
		printf ("%s\nCharacter Devices:                    "
			"Block Devices:\n",
			fs ? ce : "");
		while (fgets (line, sizeof (line), devicesfp))
		{
		    switch (line[0])
		    {
		    case 'C':
			which = CDRV;
			break;
		    case 'B':
			which = BDRV;
			break;
		    case '\n':
			break;
		    default:
			maj[which][count[which]] =
			    atoi (strtok (line, " "));
			n[which][count[which]] =
			    strdup (strtok (NULL, "\n"));
			count[which]++;
			break;
		    }
		}

		j = (1 + MAX (count[0], count[1])) / 2;
		for (i = 0; i < j; i++)
		{
		    if (n[CDRV][i])
		    {
			printf ("%2d %-16s", maj[CDRV][i], n[CDRV][i]);
			free (n[CDRV][i]);
		    }
		    else
			fputs ("                    ", stdout);
		    if (n[CDRV][i + j])
		    {
			printf ("%2d %-16s",
				maj[CDRV][i + j], n[CDRV][i + j]);
			free (n[CDRV][i + j]);
		    }
		    else
			fputs ("                    ", stdout);
		    if (n[BDRV][i])
		    {
			printf ("%2d %-16s", maj[BDRV][i], n[BDRV][i]);
			free (n[BDRV][i]);
		    }
		    else
			fputs ("                    ", stdout);
		    if (n[BDRV][i + j])
		    {
			printf ("%2d %-16s",
				maj[BDRV][i + j], n[BDRV][i + j]);
			free (n[BDRV][i + j]);
		    }
		    printf ("%s\n", fs ? ce : "");
		    if (i >= count[CDRV] && i >= count[BDRV])
			break;
		}
	    }			/* devicesfp */

	    if (filesystemsfp)
	    {
		barf = 0;
		fseek (filesystemsfp, 0L, SEEK_SET);
		printf ("\nFile Systems:\n");
		while (fgets (line, sizeof (line), filesystemsfp))
		{
		    char *fs;
		    char tmp[21];

		    barf += 20;
		    fs = strchr (line, '\t');
		    fs = strtok (fs + 1, "\n");
		    if (line[0] == 'n')
		    {
			sprintf (tmp, "[%s]", fs);
			printf ("%-20s", tmp);
		    }
		    else
			printf ("%-20s", fs);
		}
		if (co && (barf % co))
		    printf ("%s\n", fs ? ce : "");
	    }			/* filesystemsfp */
	}			/* show_moddev || show_all */

	if (fs)
	{
	    fputs (cd, stdout);
	    fflush (stdout);
	    if (sl)
		sleep (sl);
	}
	else
	{
	    putchar ('\n');
	    exit (0);
	}
	memcpy (&old, &new, sizeof (struct info));
    }				/* 42 */
}
