/* screen.c - "smart" display handling */

/* Written 1994,1995 by Werner Almesberger */


#include <stdarg.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <signal.h>
#include <assert.h>
#include <sys/types.h>

#include "common.h"
#include "cursed.h"
#include "screen.h"


#define MAX_LINE 1024


static int active = 0;
static char *r_chr,*v_chr,*r_attr,*v_attr;
static int rows,cols;
static int cx,cy;
static int attrib;


static void clear_r(void)
{
    memset(r_chr,' ',(size_t) (rows*cols));
    memset(r_attr,0,(size_t) (rows*cols));
}


static void up_r(void)
{
    initscr();
    noecho();
    raw();
    /* nonl(); */
    clear();
}


static void down_r(int end)
{
    if (end) move(rows-1,0);
    else {
	clear();
	move(0,0);
    }
    /* nl(); */
    noraw();
    echo();
    endwin();
}


void scr_up(void)
{
    rows = LINES;
    cols = COLS;
    r_chr = alloc((size_t) (rows*cols));
    r_attr = alloc((size_t) (rows*cols));
    v_chr = alloc((size_t) (rows*cols));
    v_attr = alloc((size_t) (rows*cols));
    clear_r();
    memset(v_chr,' ',(size_t) (rows*cols));
    memset(v_attr,0,(size_t) (rows*cols));
}


void scr_begin(void)
{
    up_r();
    scr_up();
    cx = cy = 0;
    attrib = 0;
    active = 1;
}


void scr_end(void)
{
    if (!active) return;
    down_r(0);
    free(r_chr);
    free(r_attr);
    free(v_chr);
    free(v_attr);
    active = 0;
}


void scr_clear(void)
{
    cx = cy = 0;
    memset(v_chr,' ',(size_t) (rows*cols));
    memset(v_attr,0,(size_t) (rows*cols));
}


void scr_clrtoeol(void)
{
    if (cx < cols) {
	memset(v_chr+cols*cy+cx,' ',(size_t) (cols-cx));
	memset(v_attr+cols*cy+cx,0,(size_t) (cols-cx));
    }
}


void scr_bold(int on)
{
    attrib = !!on;
}


void scr_move(int y,int x)
{
    cx = x;
    cy = y;
}


void scr_puts(const char *str)
{
    int len;

    len = strlen(str);
    if (cols-cx < len)
	if ((len = cols-cx) < 0) return;
    memcpy(v_chr+cols*cy+cx,str,(size_t) len);
    memset(v_attr+cols*cy+cx,attrib,(size_t) len);
    cx += len;
}


void scr_printf(const char *fmt,...)
{
    char buffer[MAX_LINE+1];
    int len;
    va_list ap;

    va_start(ap,fmt);
    len = vsprintf(buffer,fmt,ap);
    va_end(ap);
    assert(len <= MAX_LINE);
    if (cols-cx < len)
	if ((len = cols-cx) < 0) return;
    memcpy(v_chr+cols*cy+cx,buffer,(size_t) len);
    memset(v_attr+cols*cy+cx,attrib,(size_t) len);
    cx += len;
}


void scr_sync(void)
{
    int i,j;
    int bold;

    bold = 0;
    for (i = 0; i < rows; i++)
	if (memcmp(r_chr+cols*i,v_chr+cols*i,(size_t) cols) ||
	  memcmp(r_attr+cols*i,v_attr+cols*i,(size_t) cols)) {
	    move(i,0);
	    for (j = 0; j < cols; j++) {
		if (v_attr[cols*i+j] != bold) {
		    if (bold) standend();
		    else standout();
		    bold = !bold;
		}
#ifdef USES_CURSED
		putchar(v_chr[cols*i+j]);
#else
		printw("%c",v_chr[cols*i+j]);
#endif
	    }
	}
    if (bold) standend();
    memcpy(r_chr,v_chr,(size_t) (rows*cols));
    memcpy(r_attr,v_attr,(size_t) (rows*cols));
    move(cy,cx);
    refresh();
}


int scr_cols(void)
{
    return cols;
}


int scr_rows(void)
{
    return rows;
}


void scr_redraw(void)
{
    standend();
    clear_r();
    clear();
    scr_sync();
}


void scr_resize(void)
{
    char *o_chr,*o_attr;
    int o_rows,o_cols;
    int i,j;

    free(r_chr);
    free(r_attr);
    o_chr = v_chr;
    o_attr = v_attr;
    o_rows = rows;
    o_cols = cols;
    scr_up();
    clear();
    for (i = 0; i < rows && i < o_rows; i++)
	for (j = 0; j < cols && j < o_cols; j++) {
	    v_chr[i*cols+j] = o_chr[i*o_cols+j];
	    v_attr[i*cols+j] = o_attr[i*o_cols+j];
	}
    free(o_chr);
    free(o_attr);
}


void scr_suspend(int sig)
{
    down_r(sig);
    if (sig) kill(getpid(),SIGSTOP);
    else {
	fprintf(stderr,"Return with \"exit\"\n");
	system("exec $SHELL");
    }
    up_r();
    scr_redraw();
}


void scr_save(SCR_CTX *ctx)
{
    ctx->v_chr = alloc((size_t) (rows*cols));
    memcpy(ctx->v_chr,v_chr,(size_t) (rows*cols));
    ctx->v_attr = alloc((size_t) (rows*cols));
    memcpy(ctx->v_attr,v_attr,(size_t) (rows*cols));
    ctx->rows = rows;
    ctx->cols = cols;
    ctx->cx = cx;
    ctx->cy = cy;
    ctx->attrib = attrib;
}


void scr_restore(SCR_CTX ctx)
{
    free(v_chr);
    v_chr = ctx.v_chr;
    free(v_attr);
    v_attr = ctx.v_attr;
    rows = ctx.rows;
    cols = ctx.cols;
    cx = ctx.cx;
    cy = ctx.cy;
    attrib = ctx.attrib;
    scr_resize();
}
