/*
 * @(#)net.c	1.6 91/09/06	- communications system calls
 */

#include <stdio.h>
#include <sys/types.h>
#include <sys/ptrace.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <netinet/in.h>

#include "defs.h"

static Xlat domains[] = {
	PF_UNIX,	"PF_UNIX",
	PF_INET,	"PF_INET",
#ifndef linux
	PF_IMPLINK,	"PF_IMPLINK",
#endif
	0,		NULL,
};
static Xlat socktypes[] = {
	SOCK_STREAM,	"STREAM",
	SOCK_DGRAM,	"DGRAM",
	SOCK_RAW,	"RAW",
	SOCK_SEQPACKET,	"SEQ_PKT",
	SOCK_RDM,	"RDM",
	0,		NULL,
};
static Xlat protocols[] = {
	IPPROTO_IP,	"IPPROTO_IP",
	IPPROTO_ICMP,	"IPPROTO_ICMP",
	IPPROTO_GGP,	"IPPROTO_GGP",
	IPPROTO_TCP,	"IPPROTO_TCP",
	IPPROTO_EGP,	"IPPROTO_EGP",
	IPPROTO_PUP,	"IPPROTO_PUP",
	IPPROTO_UDP,	"IPPROTO_UDP",
	IPPROTO_IDP,	"IPPROTO_IDP",
#ifndef linux
	IPPROTO_IGMP,	"IPPROTO_IGMP",
	IPPROTO_HELLO,	"IPPROTO_HELLO",
	IPPROTO_ND,	"IPPROTO_ND",
#endif
	IPPROTO_RAW,	"IPPROTO_RAW",
	IPPROTO_MAX,	"IPPROTO_MAX",
	0,		NULL,
};
static Xlat msg_flags[] = {
	MSG_OOB,	"OOB",
#ifndef linux
	MSG_DONTROUTE,	"DONTROUTE",
#else
	MSG_PEEK,	"PEEK",
#endif
	0, 		NULL,
};

static Xlat sockoptions[] = {
	SO_DEBUG,	"DEBUG",
	SO_REUSEADDR,	"REUSEADDR",
	SO_KEEPALIVE,	"KEEPALIVE",
	SO_DONTROUTE,	"DONTROUTE",
	SO_BROADCAST,	"BROADCAST",
#ifndef linux
	SO_ACCEPTCONN,	"ACCEPTCONN",
	SO_USELOOPBACK,	"USELOOPBACK",
#endif
	SO_LINGER,	"LINGER",
	SO_OOBINLINE,	"OOBINLINE",
	0,		NULL,
};

void
printsock(pid, addr)
{
	struct sockaddr sa;
	struct sockaddr_in *sin = (struct sockaddr_in *)&sa;
	struct sockaddr_un sau;

	umove(pid, addr, sizeof sa, (char *)&sa);
	switch (sa.sa_family) {
	case AF_UNIX:
		umove(pid, addr, sizeof sau, (char *)&sau);
		fprintf(outf, "AF_UNIX(%s)", sau.sun_path);
		break;
	case AF_INET:
		fprintf(outf, "AF_INET(%u, %s)",
			ntohs(sin->sin_port), inet_ntoa(sin->sin_addr));
		break;
	default:
		fprintf(outf, "PF %u addr %*s", sa.sa_family,
				sizeof sa.sa_data, sa.sa_data);
		break;
	}
}

#ifndef linux
static void
printmsghdr(pid, addr)
{
	struct msghdr msg;

	umove(pid, addr, sizeof msg, (char *)&msg);
	fprintf(outf, "[name %s len %u iovec %#x iovlen %u acc %#x acclen %u]",
		msg.msg_name, msg.msg_namelen,
		msg.msg_iov, msg.msg_iovlen,
		msg.msg_accrights, msg.msg_accrightslen);
}
#endif

int
sys_socket(tcp)
struct tcb *tcp;
{
	char *pf = xlookup(domains, tcp->u_args[0]);
	char *st = xlookup(socktypes, tcp->u_args[1]);

	if (entering(tcp)) {
		fprintf(outf, "%s", pf==NULL?"PF_???":pf);
		fprintf(outf, ", %s", st==NULL?"SOCK_???":st);
		if (tcp->u_args[0] == PF_INET) {
			char *pr = xlookup(protocols, tcp->u_args[2]);
			fprintf(outf, ", %s", pr==NULL?"IPPROTO_???":pr);
		} else {
			fprintf(outf, ", %u", tcp->u_args[2]);
		}
	}
	return 0;
}

int
sys_bind(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		fprintf(outf, "%u, ", tcp->u_args[0]);
		printsock(tcp->pid, tcp->u_args[1]);
		fprintf(outf, ", %u", tcp->u_args[2]);
	}
	return 0;
}

int
sys_connect(tcp)
struct tcb *tcp;
{
	return sys_bind(tcp);
}

int
sys_listen(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		fprintf(outf, "%u, %u", tcp->u_args[0], tcp->u_args[1]);
	}
	return 0;
}

int
sys_accept(tcp)
struct tcb *tcp;
{
	int len;

	if (entering(tcp)) {
		fprintf(outf, "%u, ", tcp->u_args[0]);
	} else {
		umove(tcp->pid, tcp->u_args[2], sizeof len, (char *)&len);
		if (syserror(tcp)) {
			fprintf(outf, "%#x, %u", tcp->u_args[1], len);
		} else {
			printsock(tcp->pid, tcp->u_args[1]);
			fprintf(outf, ", %u", len);
		}
	}
	return 0;
}

int
sys_send(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		fprintf(outf, "%u, ", tcp->u_args[0]);
		printstr(tcp->pid, tcp->u_args[1], tcp->u_args[2]);
		/* flags */
		fprintf(outf, ", ");
		if (printflags(msg_flags, tcp->u_args[3]) == 0)
			fprintf(outf, "0");
	}
	return 0;
}

int
sys_sendto(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		fprintf(outf, "%u, ", tcp->u_args[0]);
		printstr(tcp->pid, tcp->u_args[1], tcp->u_args[2]);
		/* flags */
		fprintf(outf, ", ");
		if (printflags(msg_flags, tcp->u_args[3]) == 0)
			fprintf(outf, "0");

		/* to address */
		fprintf(outf, ", ");
		printsock(tcp->pid, tcp->u_args[4]);
		/* to length */
		fprintf(outf, ", %u", tcp->u_args[5]);
	}
	return 0;
}

#ifndef linux
int
sys_sendmsg(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		fprintf(outf, "%u, ", tcp->u_args[0]);
		printmsghdr(tcp->pid, tcp->u_args[1]);
		/* flags */
		fprintf(outf, ", ");
		if (printflags(msg_flags, tcp->u_args[2]) == 0)
			fprintf(outf, "0");
	}
	return 0;
}
#endif

int
sys_recv(tcp)
struct tcb *tcp;
{
	int fromlen;

	if (entering(tcp)) {
		fprintf(outf, "%u, ", tcp->u_args[0]);
	} else {
		printstr(tcp->pid, tcp->u_args[1], tcp->u_rval);

		/* flags */
		fprintf(outf, ", ");
		if (printflags(msg_flags, tcp->u_args[3]) == 0)
			fprintf(outf, "0");
	}
	return 0;
}

int
sys_recvfrom(tcp)
struct tcb *tcp;
{
	int fromlen;

	if (entering(tcp)) {
		fprintf(outf, "%u, ", tcp->u_args[0]);
	} else {
		printstr(tcp->pid, tcp->u_args[1], tcp->u_rval);

		/* flags */
		fprintf(outf, ", ");
		if (printflags(msg_flags, tcp->u_args[3]) == 0)
			fprintf(outf, "0");

		/* from address */
		fprintf(outf, ", ");
		printsock(tcp->pid, tcp->u_args[4]);
		/* from length */
		umove(tcp->pid, tcp->u_args[5], sizeof fromlen, (char *)&fromlen);
		fprintf(outf, ", %u", fromlen);
	}
	return 0;
}

#ifndef linux
int
sys_recvmsg(tcp)
struct tcb *tcp;
{
	int fromlen;

	if (entering(tcp)) {
		fprintf(outf, "%u, ", tcp->u_args[0]);
	} else {
		printmsghdr(tcp->pid, tcp->u_args[1]);
		/* flags */
		fprintf(outf, ", ");
		if (printflags(msg_flags, tcp->u_args[2]) == 0)
			fprintf(outf, "0");
	}
	return 0;
}
#endif

int
sys_shutdown(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		fprintf(outf, "%u, ", tcp->u_args[0]);
		switch (tcp->u_args[1]) {
		case 0:
			fprintf(outf, "%s", "RECEIVE");
			break;
		case 1:
			fprintf(outf, "%s", "SEND");
			break;
		case 2:
			fprintf(outf, "%s", "SEND/RECEIVE");
			break;
		default:
			fprintf(outf, "???(%u)", tcp->u_args[1]);
			break;
		}
	}
	return 0;
}

int
sys_getsockname(tcp)
struct tcb *tcp;
{
	return sys_accept(tcp);
}

int
sys_getpeername(tcp)
struct tcb *tcp;
{
	return sys_accept(tcp);
}

int
sys_pipe(tcp)
struct tcb *tcp;
{
#ifndef linux
	if (exiting(tcp)) {
		fprintf(outf, "[%u,%u]", tcp->u_rval, getrval2(tcp->pid));
	}
#else
	int fds[2];

	if (exiting(tcp)) {
		if (syserror(tcp)) {
			fprintf(outf, "%#x", tcp->u_args[0]);
			return 0;
		}
		if (umove(tcp->pid, tcp->u_args[0], sizeof fds, (char *) fds) < 0)
			fprintf(outf, "[...]");
		else
			fprintf(outf, "[%u,%u]", fds[0], fds[1]);
	}
#endif
	return 0;
}

int
sys_socketpair(tcp)
struct tcb *tcp;
{
	int fds[2];
	char *pf = xlookup(domains, tcp->u_args[0]);
	char *st = xlookup(socktypes, tcp->u_args[1]);

	if (entering(tcp)) {
		fprintf(outf, "%s", pf==NULL?"PF_???":pf);
		fprintf(outf, ", %s", st==NULL?"SOCK_???":st);
		if (tcp->u_args[0] == PF_INET) {
			char *pr = xlookup(protocols, tcp->u_args[2]);
			fprintf(outf, ", %s", pr==NULL?"IPPROTO_???":pr);
		} else {
			fprintf(outf, ", %u", tcp->u_args[2]);
		}
	} else {
#ifdef linux
		if (syserror(tcp)) {
			fprintf(outf, "%#x", tcp->u_args[0]);
			return 0;
		}
		if (umove(tcp->pid, tcp->u_args[0], sizeof fds, (char *) fds) < 0)
			fprintf(outf, "[...]");
		else
			fprintf(outf, ", [%u,%u]", fds[0], fds[1]);
#else
		fprintf(outf, ", [%u,%u]", tcp->u_rval, getrval2(tcp->pid));
#endif
	}
	return 0;
}

int
sys_getsockopt(tcp)
struct tcb *tcp;
{
	int	optval;
	char	*so;

	if (entering(tcp)) {
		fprintf(outf, "%u, ", tcp->u_args[0]);
		if (tcp->u_args[1] == SOL_SOCKET) {
			so = xlookup(sockoptions, tcp->u_args[2]);
			fprintf(outf, "SOL_SOCKET, %s, ", so?so:"SO_???");
		} else {
			/* XXX - should know socket family here */
			char *pr = xlookup(protocols, tcp->u_args[1]);
			fprintf(outf, ", %s", pr==NULL?"IPPROTO_???":pr);
		}
		fprintf(outf, "%u, ", tcp->u_args[2]);
	} else {
		umove(tcp->pid, tcp->u_args[3], sizeof optval, (char *)&optval);
		fprintf(outf, "%u, ", optval);
	}
	if (entering(tcp)) {
		fprintf(outf, "%u", tcp->u_args[4]);
	}
	return 0;
}

int
sys_setsockopt(tcp)
struct tcb *tcp;
{
	int	optval;
	char	*so;

	if (entering(tcp)) {
		fprintf(outf, "%u, ", tcp->u_args[0]);
		if (tcp->u_args[1] == SOL_SOCKET) {
			so = xlookup(sockoptions, tcp->u_args[2]);
			fprintf(outf, "SOL_SOCKET, %s, ", so?so:"SO_???");
		} else {
			/* XXX - should know socket family here */
			char *pr = xlookup(protocols, tcp->u_args[1]);
			fprintf(outf, ", %s", pr==NULL?"IPPROTO_???":pr);
			fprintf(outf, "%u, ", tcp->u_args[2]);
		}
		umove(tcp->pid, tcp->u_args[3], sizeof optval, (char *)&optval);
		fprintf(outf, "%u, ", optval);
		fprintf(outf, "%u", tcp->u_args[4]);
	}
	return 0;
}
