/*
 * @(#)signal.c	1.5 91/09/05
 *
 */

#include <stdio.h>
#include <sys/types.h>
#include <sys/ptrace.h>
#include <sys/signal.h>

#include "defs.h"

char *signals[] = {
#ifdef linux
	"SIGNULL",
	"SIGHUP",
	"SIGINT",
	"SIGQUIT",
	"SIGILL",
	"SIGTRAP",
	"SIGABRT",
	"SIGUNUSED",
	"SIGFPE",
	"SIGKILL",
	"SIGUSR1",
	"SIGSEGV",
	"SIGUSR2",
	"SIGPIPE",
	"SIGALRM",
	"SIGTERM",
	"SIGSTKFLT",
	"SIGCHLD",
	"SIGCONT",
	"SIGSTOP",
	"SIGTSTP",
	"SIGTTIN",
	"SIGTTOU",
	"SIGIO",
	"SIGXCPU",
	"SIGXFSZ",
	"SIGVTALRM",
	"SIGPROF",
	"SIGWINCH",
#else
     "SIGNULL",
     "SIGHUP",   /*    1    hangup */
     "SIGINT",   /*    2    interrupt */
     "SIGQUIT",  /*    3*   quit */
     "SIGILL",   /*    4*   illegal instruction */
     "SIGTRAP",  /*    5*   trace trap */
     "SIGIOT",   /*    6*   IOT trap (not generated on Suns) */
     "SIGEMT",   /*    7*   EMT trap (A-line or F-line op code) */
     "SIGFPE",   /*    8*   arithmetic exception */
     "SIGKILL",  /*    9    kill (cannot be caught, blocked, or ignored) */
     "SIGBUS",   /*    10*  bus error */
     "SIGSEGV",  /*    11*  segmentation violation */
     "SIGSYS",   /*    12*  bad argument to system call */
     "SIGPIPE",  /*    13   write on a pipe with no one to read it */
     "SIGALRM",  /*    14   alarm clock */
     "SIGTERM",  /*    15   software termination signal */
     "SIGURG",   /*    16@  urgent condition present on socket */
     "SIGSTOP",  /*    17+  stop (cannot be caught, blocked, or ignored) */
     "SIGTSTP",  /*    18+  stop signal generated from keyboard */
     "SIGCONT",  /*    19@  continue after stop (cannot be blocked) */
     "SIGCHLD",  /*    20@  child status has changed */
     "SIGTTIN",  /*    21+  background read attempted from control terminal */
     "SIGTTOU",  /*    22+  background write attempted to control terminal */
     "SIGIO",    /*    23@  I/O is possible on a descriptor (see fcntl(2)) */
     "SIGXCPU",  /*    24   cpu time limit exceeded (see setrlimit(2)) */
     "SIGXFSZ",  /*    25   file size limit exceeded (see setrlimit(2)) */
     "SIGVTALRM", /*   26   virtual time alarm (see setitimer(2)) */
     "SIGPROF",  /*    27   profiling timer alarm (see setitimer(2)) */
     "SIGWINCH", /*    28@  window changed (see win(4S)) */
     "SIGLOST",  /*    29*  resource lost (see lockd(8C)) */
     "SIGUSR1",  /*    30   user-defined signal 1 */
     "SIGUSR2"   /*    31   user-defined signal 2 */
#endif
};

static Xlat sigvec_flags[] = {
#ifndef linux
	SV_ONSTACK,	"ONSTACK",
	SV_INTERRUPT,	"INTERRUPT",
	SV_RESETHAND,	"RESETHAND",
#else
	SA_INTERRUPT,	"INTERRUPT",
	SA_NOMASK,	"NOMASK",
	SA_ONESHOT,	"ONESHOT",
#endif
	SA_NOCLDSTOP,	"NOCLDSTOP",
	0,		NULL,
};

static void
printsigmask(mask)
sigset_t mask;
{
	int i;
	char *format;

	format = "%s";
	fprintf(outf, "[", mask);
	for (i = 0; i < NSIG; i++) {
		if (mask & sigmask(i)) {
			fprintf(outf, format, signals[i]);
			format = " %s";
		}
	}
	fprintf(outf, "]", mask);
}

#ifndef linux
int
sys_sigvec(tcp)
struct tcb *tcp;
{
	struct sigvec sv;
	int addr;

	if (entering(tcp)) {
		fprintf(outf, "%s, ", signals[tcp->u_args[0]]);
		addr = tcp->u_args[1];
	} else {
		addr = tcp->u_args[2];
	}
	if (addr == 0 || umove(tcp->pid, addr, sizeof sv, (char *)&sv) < 0) {
		fprintf(outf, "(struct sigvec *)0");
	} else {
		switch (sv.sv_handler) {
		case SIG_ERR:
			fprintf(outf, "SIG_ERR");
			break;
		case SIG_DFL:
			fprintf(outf, "SIG_DFL");
			break;
		case SIG_IGN:
			fprintf(outf, "SIG_IGN");
			break;
		default:
			fprintf(outf, "{%#x, ", sv.sv_handler);
			printsigmask(sv.sv_mask);
			fprintf(outf, ", ");
			if (!printflags(sigvec_flags, sv.sv_flags))
				putc('0', outf);
			putc('}', outf);
		}
	}
	if (entering(tcp))
		fprintf(outf, ", ");
	return 0;
}

int
sys_sigblock(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		printsigmask(tcp->u_args[0]);
		if ((tcp->u_args[0] & sigmask(SIGTRAP)) && brutal) {
			pokearg(tcp->pid, 0, tcp->u_args[0] & ~sigmask(SIGTRAP));
		}
	}
	return 0;
}

int
sys_sigsetmask(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		if (tcp->u_args[0])
			printsigmask(tcp->u_args[0]);
		else
			fprintf(outf, "0");
	}
	return 0;
}

int
sys_sigpause(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		if (tcp->u_args[0])
			printsigmask(tcp->u_args[0]);
		else
			fprintf(outf, "0");
	}
	return 0;
}

int
sys_sigstack(tcp)
struct tcb *tcp;
{
	struct sigstack ss;
	int addr;

	if (entering(tcp)) {
		addr = tcp->u_args[0];
	} else {
		addr = tcp->u_args[1];
	}
	if (addr == 0 || umove(tcp->pid, addr, sizeof ss, (char *)&ss)) {
		fprintf(outf, "(struct sigstack *)0");
	} else {
		fprintf(outf, "{sp: %#x,", ss.ss_sp);
		fprintf(outf, "%s}", ss.ss_onstack?"ON":"OFF");
	}
	if (entering(tcp))
		fprintf(outf, ", ");
	return 0;
}

int
sys_sigcleanup(tcp)
struct tcb *tcp;
{
	return 0;
}
#else

int
sys_sigaction(tcp)
struct tcb *tcp;
{
	struct sigaction sv;
	int addr;

	if (entering(tcp)) {
		fprintf(outf, "%s, ", signals[tcp->u_args[0]]);
		addr = tcp->u_args[1];
	} else {
		addr = tcp->u_args[2];
	}
	if (addr == 0 || umove(tcp->pid, addr, sizeof sv, (char *)&sv) < 0) {
		fprintf(outf, "(struct sigaction *)0");
	} else {
		switch ((int) sv.sa_handler) {
		case SIG_ERR:
			fprintf(outf, "SIG_ERR");
			break;
		case SIG_DFL:
			fprintf(outf, "SIG_DFL");
			break;
		case SIG_IGN:
			fprintf(outf, "SIG_IGN");
			break;
		default:
			fprintf(outf, "{%#x, ", sv.sa_handler);
			printsigmask(sv.sa_mask);
			fprintf(outf, ", ");
			if (!printflags(sigvec_flags, sv.sa_flags))
				putc('0', outf);
			putc('}', outf);
		}
	}
	if (entering(tcp))
		fprintf(outf, ", ");
	return 0;
}

int
sys_signal(tcp)
struct tcb *tcp;
{
	struct sigaction sv;
	int addr;

	if (entering(tcp)) {
		fprintf(outf, "%s, ", signals[tcp->u_args[0]]);
		switch (tcp->u_args[1]) {
		case SIG_ERR:
			fprintf(outf, "SIG_ERR");
			break;
		case SIG_DFL:
			fprintf(outf, "SIG_DFL");
			break;
		case SIG_IGN:
			fprintf(outf, "SIG_IGN");
			break;
		default:
			fprintf(outf, "%#x", tcp->u_args[1]);
		}
	}
	return 0;
}

int
sys_ssetmask(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		if (tcp->u_args[0])
			printsigmask(tcp->u_args[0]);
		else
			fprintf(outf, "0");
	}
	return RVAL_HEX;
}

int
sys_sgetmask(tcp)
struct tcb *tcp;
{
	if (exiting(tcp)) {
		if (tcp->u_rval)
			printsigmask(tcp->u_rval);
	}
	return RVAL_HEX;
}

int
sys_sigsuspend(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		/* first two are not really arguments, but print them anyway */
		fprintf(outf, "%d, %#x, ", tcp->u_args[0], tcp->u_args[1]);
		if (tcp->u_args[2])
			printsigmask(tcp->u_args[2]);
		else
			fprintf(outf, "0");
	}
	return 0;
}

#endif

int
sys_kill(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		fprintf(outf, "%d, %s",
				tcp->u_args[0], signals[tcp->u_args[1]]);
	}
	return 0;
}

int
sys_killpg(tcp)
struct tcb *tcp;
{
	return sys_kill(tcp);
}

int
sys_sigpending(tcp)
struct tcb *tcp;
{
	sigset_t sigset;

	if (exiting(tcp)) {
		if (syserror(tcp) ||
			umove(tcp->pid, tcp->u_args[0],
				sizeof sigset, (char *)&sigset) < 0)
			fprintf(outf, "%#x", tcp->u_args[0]);
		else
			printsigmask(sigset);
	}
	return 0;
}
