/*
**                      Softwarepraktikum iMaze
**                              1993/94
**                Joerg Czeranski    Hans-Ulrich Kiel
**
** Datei: xv_tasten.c
**
** Kommentar:
**  Wertet Events von der Tastatur aus und setzt definierte Signale
*/


#ifdef JOYSTICK
#ifdef linux

#include <fcntl.h>
#include <linux/joystick.h>

/* Pfusch! Fehler in den System-Header-Dateien! */
#define _WAITFLAGS_H

#ifndef DEFAULT_JOYSTICK_DEVICE
#define DEFAULT_JOYSTICK_DEVICE "/dev/js0"
#endif

#else

#undef JOYSTICK

#endif
#endif

#include <xview/xview.h>
#include <xview/canvas.h>
#include <xview/panel.h>

#include "global.h"
#include "signale.h"
#include "ereignisse.h"
#include "einaus.h"
#include "grafik.h"
#include "xv_einaus.h"

static char sccsid[] = "@(#)xv_tasten.c	1.19 7/14/94";


#define GEDRUECKT 1       /* Bit 0 wird gesetzt, wenn Taste gedrueckt;
                             und geloescht, wenn diese losgelassen */
#define NICHT_ABGEFRAGT 2 /* Bit 1 bleibt gesetzt, bis ein Tastendruck
                             registriert wurde */

/* die Cursor-Tasten haben eine Repeat-Funktion, alle anderen nicht */
#define HAT_REPEAT(signalnr) \
	((signalnr) == VORSIGNAL || (signalnr) == ZURUECKSIGNAL || \
	(signalnr) == LINKSSIGNAL || (signalnr) == RECHTSSIGNAL)

static char tasten_aktiv[SIGNALANZ]; /* aktueller Tastenzustand
                                        (GEDRUECKT / NICHT_ABGEFRAGT) */

#ifdef JOYSTICK
static char joystick_aktiv[SIGNALANZ]; /* aktueller Joystickzustand
                                          (GEDRUECKT / NICHT_ABGEFRAGT) */

static int joystick;          /* Deskriptor fuer Joystick-Device */
static int joystick_abfragen; /* Flag: Joystick benutzen? */
static int joystick_gelesen;  /* Flag: joystick_xmax, ... initialisiert? */
static int joystick_xmin;     /* minimaler x-Wert der Joystick-Position */
static int joystick_xmax;     /* maximaler x-Wert der Joystick-Position */
static int joystick_ymin;     /* minimaler y-Wert der Joystick-Position */
static int joystick_ymax;     /* maximaler y-Wert der Joystick-Position */
#endif


#ifdef JOYSTICK
/*
** fehler_text
**  ermittelt den zur Systemvariable errno passenden Fehlertext
**
** Rueckgabewert:
**  Zeiger in das Feld sys_errlist oder in eine lokale Variable, die
**  den Fehler als Nummer enthaelt (falls kein Standardtext vorhanden ist)
**
** Seiteneffekte:
**  der zuletzt zurueckgegebene Zeiger wird evtl. ungueltig
*/
static char *fehler_text()
{
	extern int errno;           /* Systemvariable: letzter aufgetretener
	                               Fehlercode oder 0 */
	extern char *sys_errlist[]; /* Liste der Standard-Fehlertexte */
	extern int sys_nerr;        /* Anzahl der Eintraege in sys_errlist */

	/* falls errno kein gueltiger Index fuer sys_errlist ist, nur die
	   Nummer als String zurueckgeben */
	if (errno < 0 || errno >= sys_nerr)
	{
		static char text[20]; /* Puffer fuer Fehlernummer */

		sprintf(text, "Error #%d", errno);

		/* Zeiger auf Puffer zurueckgeben; Achtung: dieser Zeiger
		   ist immer derselbe */
		return text;
	}

	/* Standard-Fehlertext zurueckgeben */
	return sys_errlist[errno];
}
#endif


/*
** aktiv_setzen
**  setzt oder loescht das "gedrueckt"-Flag fuer ein Signal
**
** Parameter:
**  aktiv: Feld mit Flags fuer die Signale
**  signalnr: Nummer des Signals
**  gedrueckt: 1, falls Taste gedrueckt wurde; 0, falls losgelassen
**
** Seiteneffekte:
**  aktiv[signalnr] wird veraendert
*/
static void aktiv_setzen(aktiv, signalnr, gedrueckt)
char aktiv[SIGNALANZ];
int signalnr, gedrueckt;
{
	/* signalnr ungueltig? */
	if (signalnr < 0 || signalnr >= SIGNALANZ)
		return;

	/* Taste wurde losgelassen */
	if (!gedrueckt)
		aktiv[signalnr] &= ~GEDRUECKT;
	/* Taste ist niedergedrueckt (neu oder durch repeat);
	   falls Taste nicht zuvor schon gedrueckt war, als
	   gedrueckt und nicht abgefragt markieren */
	else if (!(aktiv[signalnr] & GEDRUECKT))
			aktiv[signalnr] = GEDRUECKT | NICHT_ABGEFRAGT;
}


/*
** event_routine
**  verarbeitet die Tastatur-Events und ordnet sie den internen Signalen zu
**
** Parameter:
**  window: Fenster fuer den der Event bestimmt ist
**  event: Beschreibung des Events
**
** Seiteneffekte:
**  aktiv wird veraendert
*/
static void event_routine(window, event)
Xv_Window window;
Event *event;
{
	int signalnr;  /* Index auf Tabelle mit Signalen */
	short action;  /* Typ bzw. ASCII-Code des Events */

	switch (action = event_action(event))
	{
		case 3: /* Control-C */
			if (event_is_down(event))
				disconnect_abfrage();
			return;
		case 17: /* Control-Q */
			signalnr = WEITERSIGNAL;
			break;
		case 19: /* Control-S */
			signalnr = STOPSIGNAL;
			break;
		case ' ': /* Leertaste */
		case SHIFT_LEFT:
		case SHIFT_RIGHT:
		case SHIFT_LEFTCTRL:
		case SHIFT_RIGHTCTRL:
		case SHIFT_ALT:
			signalnr = SCHUSSSIGNAL;
			break;
		case ACTION_GO_COLUMN_BACKWARD: /* Cursor hoch */
			signalnr = VORSIGNAL;
			break;
		case ACTION_GO_COLUMN_FORWARD: /* Cursor runter */
		case 31758:                    /* 5 auf dem Ziffernblock */
			signalnr = ZURUECKSIGNAL;
			break;
		case ACTION_GO_CHAR_BACKWARD: /* Cursor links */
			signalnr = LINKSSIGNAL;
			break;
		case ACTION_GO_CHAR_FORWARD: /* Cursor rechts */
			signalnr = RECHTSSIGNAL;
			break;
		case '0': /* Tasten 0 bis 9 */
			signalnr = AKTIONSSIGNALE + 9;
			break;
		case '1':
		case '2':
		case '3':
		case '4':
		case '5':
		case '6':
		case '7':
		case '8':
		case '9':
			signalnr = AKTIONSSIGNALE + action - '1';
			break;
		case KBD_DONE: /* Fenster verliert Fokus (erhaelt keine
			   Tastatur-Eingaben mehr) */
			/* dann werden alle Tasten wie losgelassen behandelt */
			for (signalnr = 0; signalnr < SIGNALANZ; signalnr++)
				tasten_aktiv[signalnr] &= ~GEDRUECKT;
			return;
		default:
			return;
	}

	/* Tastenflags setzen */
	aktiv_setzen(tasten_aktiv, signalnr, event_is_down(event));
}


/*
** frame_init
**  sucht in einem Fenster nach Unterfenstern vom Typ CANVAS und
**  initialisiert darin die Event-Abfrage
**
** Parameter:
**  frame: Fenster, in dem gesucht werden soll
*/
static void frame_init(frame)
Frame frame;
{
	int i;
	Window subwindow;

	for (i = 1; (subwindow = xv_get(frame,
		FRAME_NTH_SUBWINDOW, i)) != 0; i++)
		if ((Xv_pkg *)xv_get(subwindow, XV_TYPE) == CANVAS)
		{
			/* KBD_DONE: Fenster verliert Fokus
			   WIN_ASCII_EVENT: Tasten werden gedrueckt
			   WIN_UP_EVENT: Taste losgelassen (wird evtl.
			   nicht unterstuetzt) */
			xv_set(canvas_paint_window(subwindow),
				WIN_CONSUME_EVENTS,
					KBD_DONE, WIN_ASCII_EVENTS, WIN_UP_EVENTS, 0,
				WIN_EVENT_PROC, event_routine,
				0);
		}
}


/* bis hier lokaler Teil                       */
/***********************************************/
/* ab hier globaler Teil                       */


/*
** eingabe_abfragen
**  setzt in einem Feld aller Signale diejenigen, die seit der letzten
**  Abfrage aufgetreten sind. Die Tasten werden als abgefragt markiert
**
** Paramter:
**  signale: Feld mit allen Signalen
**           (0 = nicht aufgetreten; 1 = aufgetreten)
**
** Seiteneffekte:
**  aktiv und signale wird veraendert
*/
void eingabe_abfragen(signale)
char signale[SIGNALANZ];
{
	int signalnr; /* Index auf Tabelle mit Signalen */

#ifdef JOYSTICK
	/* Joystick benutzen? */
	if (joystick_abfragen)
	{
		struct JS_DATA_TYPE joystick_daten; /* Position des Joysticks */
		int laenge;                         /* Laenge der gelesenen Daten */

		/* Joystick abfragen */
		laenge = read(joystick, &joystick_daten, JS_RETURN);

		/* Fehler? */
		if (laenge < 0 && errno != EWOULDBLOCK && errno != EAGAIN)
		{
			static char *meldung[] = { "iMaze - Joystick Error", "",
				"Can't read from joystick device:", NULL, NULL };

			meldung[3] = fehler_text();
			uebler_fehler(meldung, NULL);
		}

		/* stimmt die Laenge der gelesenen Daten? */
		if (laenge == JS_RETURN)
		{
			/* xmin, ... schon initialisiert? */
			if (joystick_gelesen)
			{
				joystick_xmin = min(joystick_daten.x, joystick_xmin);
				joystick_xmax = max(joystick_daten.x, joystick_xmax);
				joystick_ymin = min(joystick_daten.y, joystick_ymin);
				joystick_ymax = max(joystick_daten.y, joystick_ymax);
			}
			else
			{
				joystick_xmin = joystick_xmax = joystick_daten.x;
				joystick_ymin = joystick_ymax = joystick_daten.y;

				joystick_gelesen = 1;
			}

			/* Joystick-Buttons abfragen */
			aktiv_setzen(joystick_aktiv, SCHUSSSIGNAL,
				joystick_daten.buttons != 0);

			/* x-Richtung abfragen */
			aktiv_setzen(joystick_aktiv, LINKSSIGNAL,
				3 * joystick_daten.x < 2 * joystick_xmin + joystick_xmax);
			aktiv_setzen(joystick_aktiv, RECHTSSIGNAL,
				3 * joystick_daten.x > joystick_xmin + 2 * joystick_xmax);

			/* y-Richtung abfragen */
			aktiv_setzen(joystick_aktiv, VORSIGNAL,
				3 * joystick_daten.y < 2 * joystick_ymin + joystick_ymax);
			aktiv_setzen(joystick_aktiv, ZURUECKSIGNAL,
				3 * joystick_daten.y > joystick_ymin + 2 * joystick_ymax);
		}
	}
#endif

	for (signalnr = 0; signalnr < SIGNALANZ; signalnr++)
	{
		/* falls REPEAT-Funktion vorgesehen (Cursor-Tasten),
		   Signal setzen, wenn GEDRUECKT oder NICHT_ABGEFRAGT,
		   sonst nur wenn NICHT_ABGEFRAGT */
		if (HAT_REPEAT(signalnr))
			signale[signalnr] = tasten_aktiv[signalnr] != 0;
		else
			signale[signalnr] =
				(tasten_aktiv[signalnr] & NICHT_ABGEFRAGT) != 0;

		/* alle Tasten als ABGEFRAGT markieren */
		tasten_aktiv[signalnr] &= ~NICHT_ABGEFRAGT;

#ifdef JOYSTICK
		/* Joystick benuzten? */
		if (joystick_abfragen)
		{
			/* falls REPEAT-Funktion vorgesehen (Joystick-Bewegung),
			   Signal setzen, wenn GEDRUECKT oder NICHT_ABGEFRAGT,
			   sonst nur wenn NICHT_ABGEFRAGT */
			if (HAT_REPEAT(signalnr))
				signale[signalnr] |= joystick_aktiv[signalnr] != 0;
			else
				signale[signalnr] |=
					(joystick_aktiv[signalnr] & NICHT_ABGEFRAGT) != 0;

			/* alle Joystick-Daten als ABGEFRAGT markieren */
			joystick_aktiv[signalnr] &= ~NICHT_ABGEFRAGT;
		}
#endif
	}
}


/*
** xv_eingabe_init
**  sucht nach Unterfenstern vom Typ CANVAS und initialisiert darin
**  die Event-Abfrage
**
** Parameter:
**  hauptframe: Fenster, von dem alle anderen Fenster abstammen
**  panel: Panel, in dem Buttons ergaenzt werden duerfen
**
** Rueckgabewert:
**  0 fuer Erfolg, 1 fuer Fehler
*/
int xv_eingabe_init(hauptframe, panel, argc, argv)
Frame hauptframe;
Panel panel;
int *argc;
char **argv;
{
	int i;          /* Index */
	Frame subframe; /* ein Fenster */

#ifdef JOYSTICK
	char *joystick_device; /* Name des Joystick-Devices */
	int ueberspringen;     /* Anzahl der ausgewerteten
	                          Kommandozeilenparameter */
#endif

	/* im Hauptfenster suchen */
	frame_init(hauptframe);

	/* in allen weiteren Fenstern suchen */
	for (i = 1; (subframe = xv_get(hauptframe,
		FRAME_NTH_SUBFRAME, i)) != 0; i++)
		frame_init(subframe);

#ifdef JOYSTICK
	/* initialisieren */
	joystick_abfragen = 0;
	joystick_device = DEFAULT_JOYSTICK_DEVICE;

	/* Kommandozeilenparameter auswerten */
	for (ueberspringen = 1; ueberspringen < *argc &&
		argv[ueberspringen][0] == '-'; ueberspringen++)
	{
		for (i = 1; argv[ueberspringen][i] != 0 &&
			strchr("j", argv[ueberspringen][i]) != NULL; i++);

		if (strchr("J", argv[ueberspringen][i]) != NULL)
		{
			if (argv[ueberspringen][i + 1] == 0 &&
				ueberspringen == *argc - 1)
				break;
		}
		else if (argv[ueberspringen][i] != 0 || i < 2)
			break;

		/* argv[ueberspingen] faengt mit "-" an und enthaelt
		   nur bekannte Optionen; Optionen auswerten */
		for (i = 1; i > 0 && argv[ueberspringen][i] != 0;)
			switch (argv[ueberspringen][i++])
			{
				case 'J': /* Joystick benutzen mit alternativem Device */
					if (argv[ueberspringen][i] == 0)
						ueberspringen++, i = 0;
					joystick_device = argv[ueberspringen] + i;
					i = 0;

				case 'j': /* Joystick benutzen */
					joystick_abfragen = 1;
			}
	}

	/* "--" ignorieren */
	if (ueberspringen < *argc && !strcmp(argv[ueberspringen], "--"))
		ueberspringen++;

	/* mehr als argv[0] ausgewertet? */
	if (ueberspringen > 1)
	{
		/* Aufrufparameter verschieben, ausgewertete ueberschreiben */
		ueberspringen--;
		*argc -= ueberspringen;
		for (i = 1; i < *argc; i++)
			argv[i] = argv[i + ueberspringen];
	}

	/* Joystick nicht benutzen? */
	if (!joystick_abfragen)
		return 0;

	/* Joystick-Device oeffnen Erfolg oder Abbruch */
	for (;;)
	{
		static char *meldung[] = { "iMaze - Joystick Error",
			"", NULL, NULL, NULL };

		/* Joystick-Device oeffnen */
		if ((joystick = open(joystick_device, O_RDONLY | O_NDELAY, 0)) >= 0)
			break;

		/* Fehlertext erzeugen */
		speicher_belegen((void **)&meldung[2],
			strlen(joystick_device) + 20);
		sprintf(meldung[2], "Can't open %s:", joystick_device);
		meldung[3] = fehler_text();

		/* sonstiger Fehler? */
		if (errno != EBUSY)
			if (rueckfrage(meldung, "No Joystick", NULL))
			{
				joystick_abfragen = 0;
				speicher_freigeben((void **)&meldung[2]);

				return 0;
			}
			else
			{
				speicher_freigeben((void **)&meldung[2]);

				return 1;
			}
		/* Device gerade in Benutzung? */
		else if (!rueckfrage(meldung, "Try again", "No Joystick"))
		{
			joystick_abfragen = 0;
			speicher_freigeben((void **)&meldung[2]);

			return 0;
		}

		speicher_freigeben((void **)&meldung[2]);
	}

	/* Joystick-Device in nicht blockierenden Modus schalten */
	if (fcntl(joystick, F_SETFL, O_NONBLOCK))
	{
		static char *meldung[] = { "iMaze - Joystick Error", "",
			"Can't switch joystick device to non-blocking mode:",
			NULL, NULL };

		meldung[3] = fehler_text();
		uebler_fehler(meldung, NULL);
	}

	/* xmin, xmax, ... noch nicht initialisiert */
	joystick_gelesen = 0;
#endif

	return 0;
}


/*
** xv_eingabe_ende
**  Dummy-Routine fuer das Schliessen von Eingabekanaelen etc.
*/
void xv_eingabe_ende()
{
#ifdef JOYSTICK
	/* Joystick schliessen */
	if (joystick_abfragen && close(joystick))
	{
		static char *meldung[] = { "iMaze - Joystick Error", "",
			"Can't close joystick device:", NULL, NULL };

		meldung[3] = fehler_text();
		uebler_fehler(meldung, NULL);
	}

	/* Joystick nicht mehr abfragen */
	joystick_abfragen = 0;
#endif
}
